/* GNU polyxmass - the massist's program.
   -------------------------------------- 
   Copyright (C) 2000,2001,2002,2003,2004 Filippo Rusconi

   http://www.polyxmass.org

   This file is part of the "GNU polyxmass" project.
   
   The "GNU polyxmass" project is an official GNU project package (see
   www.gnu.org) released ---in its entirety--- under the GNU General
   Public License and was started at the Centre National de la
   Recherche Scientifique (FRANCE), that granted me the formal
   authorization to publish it under this Free Software License.

   This software is free software; you can redistribute it and/or
   modify it under the terms of the GNU  General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This software is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.
   
   You should have received a copy of the GNU  General Public
   License along with this software; if not, write to the
   Free Software Foundation, Inc., 51 Franklin St, Fifth Floor,
   Boston, MA 02110-1301, USA.
*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "libpolyxmass-prop.h"



/* NEW'ING FUNCTIONS, DUPLICATING FUNCTIONS ...
 */
PxmProp*
libpolyxmass_prop_new (void)
{
  PxmProp *prop = g_malloc0 (sizeof (PxmProp));
  
  /* Initialize accessory function pointers with default 
   * implementations.
   */
  prop->custom_dup = libpolyxmass_prop_default_dup;
  prop->custom_cmp = libpolyxmass_prop_default_cmp;
  prop->custom_free = libpolyxmass_prop_default_free;
  
  return prop;
}


PxmProp*
libpolyxmass_prop_dup (PxmProp *prop, PxmHowDup how_dup)
{
  g_assert (prop != NULL);
  
  g_assert (prop->custom_dup != NULL);

  return prop->custom_dup (prop, how_dup);
}


PxmProp* 
libpolyxmass_prop_both_strings_new (gchar *name, gchar *data)
{
  PxmProp *prop = NULL;
  g_assert (name != NULL);
  g_assert (data != NULL);

  prop = g_malloc0 (sizeof (PxmProp));

  prop->name = g_strdup (name);
  g_assert (prop->name != NULL);

  prop->data = g_strdup (data);
  g_assert (prop->data != NULL);

  /* Since both the name and data are strings we can set
     the default pointers which handle the data member as strings:
   */
  prop->custom_dup = libpolyxmass_prop_default_dup;
  prop->custom_cmp = libpolyxmass_prop_default_cmp;
  prop->custom_free = libpolyxmass_prop_default_free;

  return prop;
}


PxmProp*
libpolyxmass_prop_default_dup (PxmProp *prop, PxmHowDup how_dup)
{
  /* In this default duplication function, we consider  
   * that the data member of prop is a string (as is the name member).
   */
  PxmProp *prop_new = NULL;
  
  g_assert (prop != NULL);
  
  prop_new = libpolyxmass_prop_new ();
  
  /* Set the custom accessory function pointers.
   */
  /* Take the opportunity to check that prop has
   * non-NULL custom func pointers. If NULL, error condition.
   */
  g_assert (prop->custom_dup != NULL);
  prop_new->custom_dup = prop->custom_dup;
  
  g_assert (prop->custom_cmp != NULL);
  prop_new->custom_cmp = prop->custom_cmp;
  
  g_assert (prop->custom_free != NULL);
  prop_new->custom_free = prop->custom_free;
    
  /* Set the name of the pxm_prop.
   */
  prop_new->name = g_strdup (prop->name);
  g_assert (prop_new->name != NULL);
  
  /* Set the data of the pxm_prop. We can allocate the data
   * using, as above, the default mechanism, g_strdup () 
   * on a string.
   */
  prop_new->data = g_strdup ((gchar*) prop->data);
  g_assert (prop_new->data != NULL);

  return prop_new;
}


GPtrArray * 
libpolyxmass_prop_GPA_dup (GPtrArray *GPA, gint how_dup)
{
  GPtrArray *new_GPA = NULL;

  PxmProp *prop = NULL;
  PxmProp *new_prop = NULL;

  gint iter = 0;

  g_assert (GPA != NULL);

  new_GPA = g_ptr_array_new ();
  g_assert (new_GPA != NULL);

  for (iter = 0; iter < GPA->len; iter++)
    {
      prop = g_ptr_array_index (GPA, iter);
      g_assert (prop != NULL);

      new_prop = libpolyxmass_prop_dup (prop, how_dup);
      
      g_ptr_array_add (new_GPA, new_prop);
    }

  g_assert (new_GPA != NULL);
  
  return new_GPA;
}






/* DATA MODIFYING FUNCTIONS
 */
gchar* 
libpolyxmass_prop_set_name (PxmProp *prop, gchar *name)
{
  g_assert (prop != NULL && name != NULL);

  /* If name already allocated, free it before.
   */
  if (prop->name != NULL)
    g_free (prop->name);

  prop->name = g_strdup (name);

  return prop->name;
}


/* LOCATING FUNCTIONS
 */
PxmProp*
libpolyxmass_prop_find_prop (GPtrArray *GPA,
			 gint *idx,
			 gint (*libpolyxmass_prop_find_cmp_func) (),
			 gchar *name,
			 gpointer data,
			 PxmHowCmp how_cmp)
{
  PxmProp *prop = NULL;
  gint what;
  gint iter = 0;
  
  g_assert (GPA != NULL);

  g_assert (name != NULL && strlen (name) > 0);

  /* Depending on the parameters passed we can establish what
   * is the criterion chosen by the user to find the PxmProp object 
   * in the GPtrArray (name is by necessity a criterion).
   */
  if (data == NULL)
    what = PROP_NAME;
  else
    what = PROP_BOTH;
  
  /* Start the iteration in the GPtrArray at index idx
   * passed as param if it is non NULL and less than the highest
   * index of the GPtrArray
   */
  if (idx != NULL && *idx < GPA->len)
    iter = *idx;
  else if (idx != NULL && *idx >= GPA->len)
    return NULL;
  else
    iter = 0;

  /* Iterate through the items in the array to check if we find an item
   * that already exists that matches the name we are searching.
   */
  for (; iter < GPA->len; iter++)
    {
      /* If we find the item we are looking for, put its address in 
       * prop.
       */
      prop = g_ptr_array_index (GPA, iter);

      if (0 == strcmp (prop->name, name))
	{
	  /* Found an item in array by the same name as the one passed
	   * as parameter, continue the search process.
	   */
	  if (what == PROP_NAME)
	    {
	      /* The name is the only criterion for the search, so
	       * we did the job!
	       * Put the currently iterated
	       * item index in the idx param (if non NULL)
	       * so that this function can be called further to 
	       * continue searching in the array if more than one item
	       * by the current name is in there (recursivity allowed).
	       */
	      if (idx != NULL)
		*idx = iter;
	      return prop;
	    }

	  else
	    {
	      /* Not only the name must be identical to the one
	       * passed as param, but also the data member. This
	       * means we'll have to check if the user provided a function
	       * to perform the comparison.
	       */
	      if (libpolyxmass_prop_find_cmp_func != NULL)
		{
		  if (0 == 
		      libpolyxmass_prop_find_cmp_func ((gpointer) prop->data,
						       (gpointer) data,
						       how_cmp))
		    {
		      /* The iterated pxm_prop instance in the
		       * GPtrArray is identical to the params passed to
		       * this function. Put the currently iterated
		       * item index in the idx param 
		       * (if non NULL) so that this function can be 
		       * called further to continue serching through it.
		       */
		      if (idx != NULL)
			*idx = iter;
		      return prop;
		    }
		}
	      else 
		/* (libpolyxmass_prop_data_cmp_func == NULL)
		 */
		{
		  /* No comparison function was given as param, so 
		   * consider that the data member of the pxm_prop 
		   * instance is mere string and use the standard 
		   * string comparison  function.
		   */
		  if (0 == strcmp (prop->data, data))
		    {
		      /* The iterated pxm_prop instance in the
		       * GPtrArray is identical to the params passed to
		       * this function. Put the currently iterated
		       * item index in the idx param 
		       * (if non NULL) so that this function can be 
		       * called further to continue serching through it.
		       */
		      if (idx != NULL)
			*idx = iter;
		      return prop;
		    }
		}
	    }
	}
      /* end of if (0 == strcmp (prop->name, name))
       */

      /* the currently iterated item has not the same name 
       * as the one passed as param, continue the iteration.
       */
      continue;
    }

  /* if we are here, that means that we did not find any pxm_prop that
   * corresponded to what we were searching for.
   */
  return NULL;
}





/* COMPARISON FUNCTIONS
 */
gint
libpolyxmass_prop_cmp (PxmProp *prop1, PxmProp *prop2, PxmHowCmp how_cmp)
{
  /* Two prop instances must be compared. We are going to use the
     prop's custom comparison function (prop->custom_cmp) to perform
     the comparison. However, we first have to make sure that the two
     prop objects actually have AT LEAST the same name. Otherwise, we
     won't be able to assert that the custom_cmp pointers are
     identical.

     Also, it only makes sense to go into a full comparison process if
     the name of prop1 and prop2 is identical, as this is the minimum
     identity requirement for a comparison !
  */

  g_assert (prop1 != NULL);
  g_assert (prop2 != NULL);
  
  if (0 != strcmp (prop1->name, prop2->name))
    return 1;
  
  /* We are dealing with two prop objects that at least have their
     name identical, so we are expected to have identical custom
     comparison function pointers !!! Thus we can try to assert that.
   */

  g_assert (prop1->custom_cmp != NULL);
  g_assert (prop2->custom_cmp != NULL);

  g_assert (prop1->custom_cmp == prop2->custom_cmp);
  
  return prop1->custom_cmp (prop1, prop2, how_cmp);
}



gint
libpolyxmass_prop_default_cmp (PxmProp *prop1,
			       PxmProp *prop2,
			       PxmHowCmp how_cmp)
{
  /* It is systematically considered that the data member of PxmProp
     is a string, thus compared like the name member of the PxmProp:
     using strcmp (). The returned value is not 0 if differences are *
     encountered and greater than zero if value differ.
   */
  gint result = 0;

  g_assert (prop1 != NULL && prop2 != NULL);

  g_assert (prop1->name != NULL && prop2->name != NULL ); 

  if (0 != strcmp (prop1->name, prop2->name))
    result++;
   
  g_assert (prop1->data != NULL && prop2->data != NULL );

  if (0 != strcmp ((gchar*) prop1->data, (gchar*) prop2->data))
    result++;
  
  return result;
}



/* RENDERING FUNCTIONS (FORMATTER FUNCTION)
 */
gchar*
libpolyxmass_prop_format_xml_prop_default (PxmProp *prop,
				  gchar *indent,
				  gint offset,
				  gpointer user_data)
{
  /* This default xml formatter, considers that the data member of the
     prop parameter is a string. It is thus considered simple xml
     #PCDATA exactly as is the name of the same prop instance.
   */
  GString *gs = NULL;
  
  gchar *lead = NULL;
  gchar *help = NULL;

  gint new_offset = 0;

  
  g_assert (prop != NULL);
  g_assert (prop->name != NULL && prop->data != NULL);
  
  gs = g_string_new ("");
  g_assert (gs != NULL);

  /* We are willing to create a SINGLE <prop> node that should look 
     like this:
     <prop>
       <name>WHATEVER_NAME</name>
       <data>WHATEVER_STRING</data>
     </prop>
  */

  new_offset = offset;

  /* Open the <prop> element.
   */
  lead = libpolyxmass_globals_format_string_lead (indent, new_offset);

  g_string_append_printf (gs, "%s<prop>\n", lead);

  g_free (lead);
  new_offset++;
  lead = libpolyxmass_globals_format_string_lead (indent, new_offset);
  
  /* The prop name member.
   */
  g_string_append_printf (gs, "%s<name>%s</name>\n",
		     lead, prop->name);
  
  /* The prop data member, which is a string in _this_ plugin.
   */
  g_string_append_printf (gs, "%s<data>%s</data>\n",
		     lead, (gchar*) prop->data);
  
  g_free (lead);
  new_offset--;
  lead = libpolyxmass_globals_format_string_lead (indent, new_offset);
  
  /* Close the prop element.
   */
  g_string_append_printf (gs, "%s</prop>\n", lead);

  g_free (lead);
  
  help = gs->str;
  
  g_string_free (gs, FALSE);
  
  return help;
}


gchar*
libpolyxmass_prop_format_txt_prop_default (PxmProp *prop,
					   gpointer user_data)
{
  /* This default txt formatter, considers that the data member of the
     prop parameter is a string. It is thus considered simple
     characters, exactly as is the name of the same prop instance.
   */
  GString *gs = NULL;
  
  gchar *help = NULL;

  g_assert (prop != NULL);
  g_assert (prop->name != NULL && prop->data != NULL);
  
  gs = g_string_new ("");
  g_assert (gs != NULL);


  g_string_append_printf (gs, "prop:\n"
                               "-----\n");

  /* The prop name member.
   */
  g_string_append_printf (gs, "name: '%s'\n", prop->name);
  
  /* The prop data member, which is a string in _this_ plugin.
   */
  g_string_append_printf (gs, "data: '%s'\n", (gchar*) prop->data);
  
  help = gs->str;
  
  g_string_free (gs, FALSE);
  
  return help;
}


PxmProp*
libpolyxmass_prop_render_xml_prop_default (xmlDocPtr xml_doc,
				       xmlNodePtr xml_node,
				       gpointer user_data)
{
  /* This is the default xml renderer, that is the code that will
     change xml data into a fully characterized PxmProp dynamically
     allocated into memory. In this default renderer, the <data>
     sub-element of the <prop> element is of #PCDATA and there is only
     one <data> element.
   
     The xml node we are in is structured this way:
   
     <prop>
       <name>MODIF</name>
       <data>acetylation</data>
       </prop>
  */
  /* The p_xml_node that is passed to the function should point to the
     xml element <prop> which is the entry node to the xml data
     describing the prop object.
   
     That is, p_xml_node->name == "prop", as exemplified below:
   
     <prop>
      ^
      |
      +------- here we are right now.
       <name>MODIF</name>
       
     Which means that p_xml_node->name == "prop" and that following
     the call to p_xml_node->children we should be pointing to the
     <name> child node of the <prop> node, thus:
     p_xml_node->xmlChildrenNode->content == "MODIFICATION".
  */
  PxmProp *prop = NULL;
  

  g_assert (xml_node != NULL);
  g_assert (xml_doc != NULL);

  g_assert (0 == strcmp ((gchar *) xml_node->name, "prop"));
  
  /* Allocate the PxmProp, so that we'll be able to fill it 
   * it with xml-based data.
   */
  prop = libpolyxmass_prop_new ();
  
  /* Go to the <name> node, which is the first <prop>'s child:
   */
  xml_node = xml_node->children;

  /* From a rigourous XML parsing point of view, the blanks found in
   * the XML document are considered to be nodes, and we have to detect
   * these and take proper action: go next sibling (next blank) as long
   * as blanks are encountered.
   */
  while (TRUE == xmlIsBlankNode(xml_node))
    xml_node = xml_node->next;

  g_assert (0 == strcmp ((gchar *) xml_node->name, "name"));

  /* Get the name of the property so that we can start filling the
   * prop instance.
   */
  prop->name = 
    (gchar *) xmlNodeListGetString (xml_doc, xml_node->xmlChildrenNode, 1);
  g_assert (prop->name != NULL);
  
  /* Go to next child of the prop element, which, according
   * to DTD should be one <data> element. To go to the <data> element
   * just go next:
   */
  xml_node = xml_node->next;
  while (TRUE == xmlIsBlankNode(xml_node))
    xml_node = xml_node->next;
  
  /* In this default xml renderer the single <data> element 
   * contains #PCDATA
   * which means that we just allocate a string:
   */
  g_assert (0 == strcmp ((gchar *) xml_node->name, "data"));

  prop->data = 
    (gpointer) (gchar *) xmlNodeListGetString (xml_doc, xml_node->xmlChildrenNode, 1);
  g_assert (prop->data != NULL);
  
  return prop;
}


/* FREE'ING FUNCTIONS
 */
gint 
libpolyxmass_prop_free (PxmProp *prop)
{
  g_assert (prop != NULL);
  
  g_assert (prop->custom_free != NULL);

  return prop->custom_free (prop);
}


gint 
libpolyxmass_prop_default_free (PxmProp *prop)
{
  g_assert (prop != NULL);

  /* The name member
   */
  if (prop->name != NULL)
    g_free (prop->name);
    
  /* The data member
   */
  if (prop->data != NULL)
    g_free (prop->data);
  
  /* Finally free the object itself.
   */
  g_free (prop);

  return 1;
}


gint
libpolyxmass_prop_free_but_not_data (PxmProp *prop) 
{
  g_assert (prop != NULL);

  /* Only free the name and the object
   */
  if (prop->name != NULL)
    g_free (prop->name);
  
  g_free (prop);

  return 1;
}



gint 
libpolyxmass_prop_GPA_free (GPtrArray *GPA)
{
  gint count = 0;
  gint len = 0;
  
  PxmProp *prop = NULL;

  g_assert (GPA != NULL);

  while (GPA->len > 0)
    {
      /* I'm not allowed to first remove the PxmProp from the array,
       * because under some rare circumstances (when dealing with
       * chembridge PxmProp instances, for example) it is important that
       * the PxmProp object still be present in the array. See code
       * in PxmProp_chembridge_free_full () to better understand.
       * So, with respect of other array freeing functions, the
       * difference here is that the PxmProp object is first freed, 
       * and AFTER this, its reference in the array is removed.
       */
      prop =  g_ptr_array_index (GPA, 0);
      g_assert (prop != NULL);

      len = GPA->len;
      libpolyxmass_prop_free (prop);
      count++;
      
      /* As I said above, some PxmProp instances upon freeing 
       * remove themselves automatically from the array in which
       * they are located when freeing is asked for them. This
       * is typically the case for the chembridge_pxm_prop instance
       * that removes itself from the array. Thus the length
       * of the array is decreased by one. This is what we can test:
       * if the length of the array is the same after the call
       * to libpolyxmass_prop_free () that means that the PxmProp 
       * instance did not self remove herself, so we can dot. 
       * And vice versa.
       */
      if (len == GPA->len)
	g_ptr_array_remove_index (GPA, 0);
    }

  g_ptr_array_free (GPA, TRUE);
  
  return count;
}




