/***************************************************************************
 *                                                                         *
 *                         Powersave Daemon                                *
 *                                                                         *
 *          Copyright (C) 2004,2005 SUSE Linux Products GmbH               *
 *                                                                         *
 *               Author(s): Holger Macht <hmacht@suse.de>                  *
 *                                                                         *
 * This program is free software; you can redistribute it and/or modify it *
 * under the terms of the GNU General Public License as published by the   *
 * Free Software Foundation; either version 2 of the License, or (at you   *
 * option) any later version.                                              *
 *                                                                         *
 * This program is distributed in the hope that it will be useful, but     *
 * WITHOUT ANY WARRANTY; without even the implied warranty of              *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU       *
 * General Public License for more details.                                *
 *                                                                         *
 * You should have received a copy of the GNU General Public License along *
 * with this program; if not, write to the Free Software Foundation, Inc., *
 * 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA                  *
 *                                                                         *
 ***************************************************************************/

#include "config.h"
#include "device_management.h"
#include "pm_interface.h"
#include "globals.h"

using namespace std;

DeviceManagement::DeviceManagement()
{
}

void DeviceManagement::init()
{
	list<string> device_udis;

	/*** wlan devices ***/
	WlanDevice::getDevices(device_udis);

	for (list< string >::iterator it = device_udis.begin();
	     it != device_udis.end(); ++it) {
		addWlanDevice(*it);
	}
	device_udis.clear();

	/*** usb devices ***/
	UsbDevice::getDevices(device_udis);

	for (list< string >::iterator it = device_udis.begin();
	     it != device_udis.end(); ++it) {
		addUsbDevice(*it);
	}
	device_udis.clear();

	/*** sound devices ***/
	SoundDevice::getDevices(device_udis);
	
	for (list< string >::iterator it = device_udis.begin();
	     it != device_udis.end(); ++it) {
		addSoundDevice(*it);
	}
	device_udis.clear();

	
	/*** lan devices ***/
	LanDevice::getDevices(device_udis);
	
	for (list< string >::iterator it = device_udis.begin();
	     it != device_udis.end(); ++it) {
		addLanDevice(*it);
	}
}

DeviceManagement::~DeviceManagement()
{
	for (list< Device* >::iterator it = _devices.begin();
	     it != _devices.end(); ++it) {
		
		(*it)->resume();
		delete (*it);
	}
}

bool DeviceManagement::suspend(DPM_DEVICE_CLASS device_class, int device_num, bool force)
{
	if (!Powersave::Globals::config_obj->current_scheme->DPM_ENABLED) {
		return false;
	}

	int num = 0;

	for (list< Device* >::iterator it = _devices.begin();
	     it != _devices.end(); ++it) {

		// continue if this is the wrong device class
		if ((*it)->deviceClass() != device_class) {
			continue;
		}

		//suspend all devices off this class
		if (device_num < 0) {
			(*it)->suspend(DPM_STATE_D3, force);
		}
		// only suspend one device
		else if (device_num == num) {
			(*it)->suspend(DPM_STATE_D3, force);
			break;
		}

		num++;
	}

	if (num == 0) {
		return false;
	}

	return true;
}

bool DeviceManagement::update(std::list< DPM_DEVICE_CLASS > rpm_classes, bool force)
{
	if (!Powersave::Globals::config_obj->current_scheme->DPM_ENABLED) {
		return false;
	}

	for (list< Device* >::iterator it = _devices.begin();
	     it != _devices.end(); ++it) {
		
		list< DPM_DEVICE_CLASS >::iterator i = find(rpm_classes.begin(),
							    rpm_classes.end(),
							    (*it)->deviceClass());

		DPM_POWER_STATE state = (*it)->updateState();
		if (i != rpm_classes.end()) {
			if (state == DPM_STATE_D0) {
				(*it)->suspend();
			}
			else if ( state == DPM_STATE_UNKNOWN ) {
				_devices.erase(it);
				it = _devices.begin();
			}
		}		
		else {
			if (state > DPM_STATE_D0) {
				(*it)->resume();
			}
			else if (state == DPM_STATE_UNKNOWN) {
				_devices.erase(it);
				it = _devices.begin();
			}
		}
	}

	return true;
}
bool DeviceManagement::resume(DPM_DEVICE_CLASS device_class, int device_num)
{
	if (!Powersave::Globals::config_obj->current_scheme->DPM_ENABLED) {
		return false;
	}

	int num = 0;

	for (list< Device* >::iterator it = _devices.begin();
	     it != _devices.end(); ++it) {

		if ((*it)->deviceClass() != device_class) {
			continue;
		}

		/* resume all devices off this class */
		if (device_num < 0) {
			(*it)->resume();
		}
		/* only resume one device */
		else if (device_num == num) {
			(*it)->resume();
			break;
		}
		
		num++;
	}

	if (num == 0) {
		return false;
	}

	return true;
}

int DeviceManagement::getDevices(DPM_DEVICE_CLASS device_class,
				 list< string > &udis,
				 list< DPM_POWER_STATE > &power_states)
{
	for (list< Device* >::iterator it = _devices.begin();
	     it != _devices.end(); ++it) {

		if ((*it)->deviceClass() == device_class) {
			DPM_POWER_STATE state = (*it)->updateState();
			if (state == DPM_STATE_UNKNOWN) {
				_devices.erase(it);
				it = _devices.begin();
			}
			else {
				udis.push_back((*it)->udi());
				power_states.push_back((*it)->updateState());
			}
		}
	}

	return udis.size();
}

bool DeviceManagement::checkDeviceAdded(char *udi)
{
	if (UsbDevice::isDevice(udi)) {
		addUsbDevice(udi);
	}
	else if (WlanDevice::isDevice(udi)) {
		addWlanDevice(udi);
	}
	else if (LanDevice::isDevice(udi)) {
		addLanDevice(udi);
	}
	else if (SoundDevice::isDevice(udi)) {
		addSoundDevice(udi);
	}
	else {
		return false;
	}

	return true;
}

bool DeviceManagement::checkDeviceRemoved(char *udi)
{
	for (list< Device* >::iterator it = _devices.begin();
	     it != _devices.end(); ++it) {
		
		if ((*it)->udi() == udi) {
			_devices.erase(it);
			pDebug(DBG_INFO, "Removed device '%s'", udi);
			return true;
		}
	}

	return false;
}

void DeviceManagement::addWlanDevice(const string &udi)
{
	WlanDevice* dev = new WlanDevice(udi);
	_devices.push_back(dev);
	pDebug(DBG_INFO,"added wlan device %s\n", udi.c_str());
}

void DeviceManagement::addUsbDevice(const string &udi)
{
	UsbDevice* dev = new UsbDevice(udi);
	_devices.push_back(dev);
	pDebug(DBG_INFO,"added usb device %s\n", udi.c_str());
}

void DeviceManagement::addSoundDevice(const string &udi)
{
	SoundDevice* dev = new SoundDevice(udi);
	_devices.push_back(dev);
	pDebug(DBG_INFO,"added sound device %s\n", udi.c_str());
}

void DeviceManagement::addLanDevice(const string &udi)
{
	LanDevice* dev = new LanDevice(udi);
	_devices.push_back(dev);
	pDebug(DBG_INFO,"added aln device %s\n", udi.c_str());
}

