/*
 *  $Id: xml.c,v 1.24 2001/10/01 22:17:51 davej Exp $
 *	This file is part of Powertweak Linux.
 *	(C) 2000 Dave Jones, Arjan van de Ven.
 *
 * 	Licensed under the terms of the GNU GPL License version 2.
 *
 *	This file contains the XML handling code for the MSR tuning 
 *   backend
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <assert.h>

#include <powertweak.h>
#include <parser.h>		/* libxml */
#include <libxmlcompat.h>

#include <sys/types.h>		/* for readdir() */
#include <dirent.h>		/* for readdir() */

#include "x86.h"
#include "../cpu.h"


static void MSR_change_value(struct tweak *tweak, value_t value, int immediate)
{
	struct private_CPU_data *pvt;
	unsigned long long current;
	unsigned long long dest;

	if (tweak == NULL)
		return;
	pvt = (struct private_CPU_data *) tweak->PrivateData;

	if (pvt == NULL)
		return;

	assert (!(pvt->value.strVal));	/* DJ: Will this _ever_ trigger? */

	set_value_int (pvt->value, get_value_int (value));

	if (!immediate)
		return;

	if ((tweak->Type==TYPE_RADIO_OPTION)&&(get_value_int(value)==0))
		return;		

	/* read current MSR value. */
	if (rdmsr(pvt->CPU_number, pvt->MSR_Register, &current) != -1) {

		switch (tweak->Type) {
		case TYPE_CHECKBOX:
			if (get_value_int(pvt->value) == 1) {
				dest = (current & ~pvt->MSR_Mask);
				dest = (dest | pvt->MSR_OnMask);
			} else {
				dest = (current & ~pvt->MSR_Mask);
				dest = (dest | pvt->MSR_OffMask);
			}
			break;

		case TYPE_SPINBOX:
			dest = get_value_int(pvt->value);
			dest = (current & ~pvt->MSR_Mask) | dest;
			break;

		case TYPE_RADIO_OPTION:
			if (get_value_int(pvt->value) == 1) {
				dest = (current & ~pvt->MSR_Mask);
				dest = (dest | pvt->MSR_OnMask);
			}
			break;

		default:
			printf("Unknown widget type in MSR backend: %d\n", tweak->Type);
			break;
		}

		/* write the MSR back. */
		wrmsr(pvt->CPU_number, pvt->MSR_Register, &dest);
	}
}


static value_t MSR_get_value(struct tweak *tweak)
{
	struct private_CPU_data *pvt;

	assert(tweak != NULL);

	pvt = (struct private_CPU_data *) tweak->PrivateData;
	assert(pvt != NULL);

	return pvt->value;
}


static int MSR_get_value_raw(struct tweak *tweak)
{
	struct private_CPU_data *pvt;
	unsigned long long value;

	pvt = (struct private_CPU_data *) tweak->PrivateData;

	if (rdmsr(pvt->CPU_number, pvt->MSR_Register, &value)==-1)
		return FALSE;
	else {
		/* Fill tweak/privatedata with values from the MSR here. */
		if (tweak->Type == TYPE_CHECKBOX) {
			if ((value & pvt->MSR_OnMask) == pvt->MSR_OnMask) {
				value = 1;
			} else {
				value = 0;
			}
		}

		if (tweak->Type == TYPE_SPINBOX)
			value = value & pvt->MSR_Mask;

		set_value_int (pvt->value, value);

		if ((unsigned int) value > tweak->MaxValue) {
			printf ("CPU: Value for \"%s\" (%lld) exceeded max (%d), increasing.\n",
				tweak->WidgetText, value, tweak->MaxValue);
			tweak->MaxValue = value;
		}
	}
	return TRUE;
}


/* 
I am sorry for the use of macro's here, but they make the function
below so much more readable....
*/
#define match_record(text,variable) \
	do {  if (strcasecmp((char*)cur->name, (text))==0) \
		(variable) = (char *) xmlNodeListGetString(doc, cur->xmlChildrenNode, 1);} while (0)

#define match_record3(text,variable,func) \
	do { char *CHPTR; \
	     CHPTR = (char*)xmlNodeListGetString(doc,cur->xmlChildrenNode,1); \
	     if ( (strcasecmp((char*)cur->name, (text))==0) && \
		 (CHPTR!=NULL) ) \
		(variable) = func(CHPTR); free(CHPTR);} while (0)




static void do_one_xml_record(struct cpu_identity *id, xmlDocPtr doc, xmlNodePtr cur, struct tweak *tweak)
{
	struct private_CPU_data *pvt;
	int bitnr;
	char *Frame = NULL;
	char *Menu1 = "Hardware", *Menu2 = "CPU", *Tab = "Tweaks", *Group = NULL;
	int stepping=-1,stepoper=OPERATOR_GET;

	pvt = (struct private_CPU_data *) tweak->PrivateData;

	/* First we decode the <MSR> tag */
	assert(cur != NULL);	/* cannot happen.. in theory */

	if (strcasecmp((char*)cur->name, ("MSR"))==0) {
		char *registerstr=NULL;
		registerstr = xmlGetProp(cur, "register");
		assert(registerstr != NULL);
		pvt->MSR_Register = strtoul(registerstr, NULL, 0);
		free(registerstr);
		registerstr = xmlGetProp(cur, "bit");

		if (registerstr != NULL) {
			bitnr = strtoul(registerstr, NULL, 0);
			pvt->MSR_Mask = 1 << bitnr;
			pvt->MSR_OnMask = 1 << bitnr;
			pvt->MSR_OffMask = 0;
			free(registerstr);
		}
		/* If we had no bit= we will use the <BITMASK>
		 * stuff defined within the <MSR> below. */

	} else {
		/* We didn't begin with an <MSR> tag. Fuck knows whats
		 * going on. Abort and hope for the best. */
		CPU_tweak_destructor(tweak);
		free(tweak);
		return;	
	}

	/* Then iterate over all XML elements and fill the structure */
	cur = cur->xmlChildrenNode;
	while (cur != NULL) {
		assert(cur->name != NULL);

		match_record("FRAME", Frame);
		match_record("GROUP", Group);
		match_record("WIDGETTEXT", tweak->WidgetText);
		match_record("DESCRIPTION", tweak->Description);
		match_record3("TYPE", tweak->Type, string_widget_to_int);

		if (!strcasecmp (cur->name, "CONFIGNAME")) {
			char *XMLstr;
			tweak->ConfigName = malloc (CONFIGNAME_MAXSIZE);
			if (tweak->ConfigName == NULL) {
				if (tweak->Destroy)
					tweak->Destroy(tweak);
				free (tweak);
				return;
			}
			XMLstr = (char *) xmlNodeListGetString (doc, cur->xmlChildrenNode, 1);
			snprintf (tweak->ConfigName, CONFIGNAME_MAXSIZE,
				"CPU%d_%s", id->CPU_number, XMLstr);
			if (XMLstr!=NULL)
				free (XMLstr);
		}

		if (strcasecmp(cur->name,"STEPPING")==0) {
			char *Compare;
			Compare = xmlGetProp(cur, "operator");
			if (Compare!=NULL) {
				stepoper = Operator2Operator(Compare);
				free(Compare);
			}
			match_record3("STEPPING",stepping,atoi);
		}

		if (tweak->Type == TYPE_CHECKBOX ||
			tweak->Type == TYPE_INFO_BOOL) {
			tweak->MinValue = 0;
			tweak->MaxValue = 1;
		} else {
			match_record3("LOW", tweak->MinValue, atoi);
    		match_record3("HIGH", tweak->MaxValue, atoi);
		}

		match_record3("BITMASK", pvt->MSR_Mask, bitstring_to_long);
		match_record3("ONBITS", pvt->MSR_OnMask, bitstring_to_long);
		match_record3("OFFBITS", pvt->MSR_OffMask, bitstring_to_long);

		cur = cur->next;
	}


	if (stepping>=0) {
		if (!operand_is_valid(id->stepping,stepoper,stepping)) {
			CPU_tweak_destructor(tweak);
			free(tweak);
		}
	}

	if (MSR_get_value_raw(tweak) != TRUE) {
		CPU_tweak_destructor(tweak);
		free(tweak);
	}

	Make_CPU_menuname(id);

	switch (tweak->Type) {
		case TYPE_COMBOELEM:
			RegisterTweak(tweak, "mmmtc", Menu1, Menu2, strdup(id->CPUName), Tab, Group);
			printf("COMBO <this is not supposed to work>!\n");
			break;

		case TYPE_RADIO_OPTION:
			if (Frame != NULL) 
				RegisterTweak(tweak, "mmmtfr", Menu1, Menu2, strdup (id->CPUName), Tab, Frame,Group);
			else
				RegisterTweak(tweak, "mmmtr", Menu1, Menu2, strdup (id->CPUName), Tab, Group);
			break;

		default:
			if (Frame != NULL)
				RegisterTweak(tweak, "mmmtf", Menu1, Menu2, strdup (id->CPUName), Tab, Frame);
			else
				RegisterTweak(tweak, "mmmt", Menu1, Menu2, strdup (id->CPUName), Tab);
			break;
	}

	if (Group != NULL)
		free(Group);
	if (Frame != NULL)
		free(Frame);
}

/* 
parse_file() enumerates over the entries found in a xml file, and calls
the do_one_tweak() handling function on them. Only <MSRENTRY> records
are actually processed.
*/

static void parse_file(struct cpu_identity *id, xmlDocPtr doc, xmlNodePtr cur)
{
	struct tweak *tweak = NULL;
	int family = 0, model = 0, modeloper = OPERATOR_EQ;
	char *vendorstr = NULL, *familystr=NULL, *modelstr=NULL;

	assert(cur != NULL);	/* isn't supposed to be possible */

	if ((cur->name != NULL) && (strcasecmp((char *) cur->name, "MSRENTRY") == 0)) {
		char *oper;
		vendorstr = xmlGetProp(cur, "vendor");
		assert(vendorstr != NULL);
		familystr = xmlGetProp(cur, "family");
		assert(familystr != NULL);
		modelstr = xmlGetProp(cur, "model");
		assert(modelstr != NULL);
		family = strtoul(familystr, NULL, 0);
		model = strtoul(modelstr, NULL, 0);
		oper = xmlGetProp(cur,"modeloper");
		if (oper!=NULL) {
			modeloper = Operator2Operator(oper);
			free(oper);
		}
	}

	cur = cur->xmlChildrenNode;

	while (cur != NULL) {
		if ((TranslateVendor(vendorstr) == id->vendor) &&
		    (family == id->family) &&
			operand_is_valid (id->model, modeloper, model))
		{
			tweak = alloc_CPU_tweak(id->CPU_number, TYPE_NODE_ONLY);

			tweak->ChangeValue = &MSR_change_value;
			tweak->GetValue = &MSR_get_value;
			tweak->GetValueRaw = &MSR_get_value_raw;
			tweak->IsValid = &generic_is_valid;

			do_one_xml_record(id, doc, cur, tweak);
		}
		cur = cur->next;
	}

	if (familystr!=NULL)
		free(familystr);
	if (modelstr!=NULL)
		free(modelstr);
	if (vendorstr!=NULL)
		free(vendorstr);
}

/*
load_MSR_xmlfile() takes care of loading and registering all tweaks
that are present in the file denoted by "Filename".
*/

void load_MSR_xmlfile(struct cpu_identity *id, char *Filename)
{
	xmlDocPtr doc;
	xmlNodePtr cur;

	if (Filename == NULL)
		return;

	doc = xmlParseFile(Filename);
	if (doc == NULL) {
		printf("Severe XML error: doc == NULL!!\n");
		printf("Probable cause: file %s not found.\n", Filename);
		return;
	}

	cur = xmlDocGetRootElement(doc);
	if (cur == NULL) {
		printf("Severe XML error: cur == NULL");
		xmlFreeDoc(doc);
		return;
	}

	parse_file(id, doc, cur);

	cur = NULL;
	xmlFreeDoc(doc);
}
