
/***************************************************************************
                          pppstatus.c  -  description
                             -------------------
    begin                : Mar 13 1999
    copyright            : (C) 2000 by Gabriel Montenegro
    email                : johnpetrucci@users.sourceforge.net
 
 ***************************************************************************
 *                                                                         *
 *  This program is free software; you can redistribute it and/or modify   *
 *  it under the terms of the GNU General Public License as published by   *
 *  the Free Software Foundation; either version 2 of the License, or      *
 *  (at your option) any later version.                                    *
 *                                                                         *
 *  This program is distributed in the hope that it will be useful,        *
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of         *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the          *
 *  GNU General Public License for more details.                           *
 *                                                                         *
 *  You should have received a copy of the GNU General Public License      *
 *  along with this program; if not, write to the Free Software            *
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307,  *
 *  USA.                                                                   *
 *                                                                         *
 ***************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <unistd.h>
#include <string.h>
#include <time.h>
#include <ctype.h>
#include <curses.h>

#include <sys/wait.h>
#include <sys/stat.h>
#include <sys/param.h>
#include <sys/ioctl.h>

#include <arpa/inet.h>

#include <net/ppp_defs.h>
#include <net/if.h> 
#include <net/if_ppp.h>
#include <netinet/in.h>
#include "pppstatus.h"

/*-----------------------------------------------------------------------*
 *  IP_ADDRESS - Gets local IP address                                   *
 *-----------------------------------------------------------------------*/

void ip_address(int display)
{
   struct ifreq ifr;
   struct sockaddr_in *skt;
   
   strcpy(ifr.ifr_name, interface);
   
   if(ioctl(ppp_s, SIOCGIFADDR, &ifr) < 0);

   skt = (struct sockaddr_in*)&ifr.ifr_addr;
   
   if(skt->sin_addr.s_addr) {
     stats.ip_addr_rtrn =  inet_ntoa(skt->sin_addr);
     if(strcmp( stats.ip_addr_rtrn, "10.64.64.64") == 0) { /* JGH 10 Aug 2001 */
	stats.ip_addr_rtrn = "Demand"; /* JGH 10 Aug 2001 */
     }
   }
   else
    stats.ip_addr_rtrn = "Unknown";

   if(display)
     {
	chcolor(colors.data[0], colors.data[1]);
	mvprintw(16, 21, "%s", stats.ip_addr_rtrn);
	refresh();
     }
}

/*-----------------------------------------------------------------------*
 *  CHECK_EMAIL - Check for new emails                                   *
 *-----------------------------------------------------------------------*/

void check_email(void)
{
   FILE *email_box;
   int rtrn = 0;
   char location[128], buf[128];
   
   sprintf(location, MAIL_PATH "%s", stats.user_box);
   
   if(!stat(location, &st))
     rtrn = st.st_size; 
   stats.email = 0;
   chcolor(colors.data[0], colors.data[1]);
   if(rtrn){
      if((email_box = fopen(location, "r"))) 
	{
	   while(!feof(email_box))
	     {
		fgets(buf, 128, email_box);
		if(!strncmp(buf, "From ", 5))
		  stats.email++; 
	     }
	   if(stats.email < 10)
	     mvprintw(17, 65, "0%d         ", stats.email);
	   else
	     mvprintw(17, 65, "%d           ", stats.email);
	   fclose(email_box);
	}
      else
	mvprintw(17, 65, "Can't check");
   }
   else
     mvprintw(17, 65, "00         ");
   refresh();
}

/*-----------------------------------------------------------------------*
 *  CENTER   -   Put a string in center                                  *
 *-----------------------------------------------------------------------*/

void center(short int line, char *strg, ...)
{
   short int where;
   char *final;
   va_list trans;
   
   final = malloc(80);
   va_start(trans, strg);
   (void)vsnprintf(final, 80, strg, trans);
   va_end(trans);
   where = (int)(40 - (strlen(final) / 2));
   if(line != 0 && line != 24) {
      move(line, 20); 
      hline(' ', 50); /* was 32 JGH Aug 9 2001 */ 
   }
   mvprintw(line, where, "%s",final);
   free(final);
}

/*-----------------------------------------------------------------------*
 *  GET_STATISTICS - Gets RX/TX Packets, Bytes and Errors                *
 *-----------------------------------------------------------------------*/
int get_stat(void)
{
   struct ppp_stats ppp_cur;
   static int ppp_opened = 0;
   if (!ppp_opened) { /* If not opened yet... Open the PPP device */
      memset(&ppp_cur, 0, sizeof(ppp_cur));
      if ((ppp_s = socket(AF_INET, SOCK_DGRAM, 0)) < 0)
	return -1;
      ppp_opened = 1;
   }
   get_ppp_stats(&ppp_cur); /* Update the information */
   stats.tx_packets = ppp_cur.p.ppp_opackets;
   stats.rx_packets = ppp_cur.p.ppp_ipackets;   
   stats.rx_bytes = ppp_cur.p.ppp_ibytes;
   stats.tx_bytes = ppp_cur.p.ppp_obytes;
   stats.rx_errors = ppp_cur.p.ppp_ierrors;
   stats.tx_errors = ppp_cur.p.ppp_oerrors;
   return(0);
   
}

void get_ppp_stats(struct ppp_stats *cur) {
   
   struct ifpppstatsreq req;
   memset(&req, 0, sizeof(req));  
   req.stats_ptr = (char *)&req.stats;
  
   strcpy(req.ifr__name, interface);
   if (ioctl(ppp_s, SIOCGPPPSTATS, &req) >= 0)
      *cur = req.stats;
}

/*-----------------------------------------------------------------------*
 * LOG - Put in a file the connection specs                              *
 *-----------------------------------------------------------------------*/
int log(int quit)
{
   FILE *log_file;
   time_t off;
   char buf[256], logfile[128];
   short int exists = 0;
   short int warning = FALSE;
   struct tm tme;

   off = time(NULL);
   tme = *localtime(&off);
   sprintf(logfile, "%s/pppstatus.%s.%s-%02d.log", home, interface, mth[tme.tm_mon], (tme.tm_year + 1900));
   
   if(!(log_file = fopen(logfile, "a+"))) {
      close_ppps("PPPStatus can not write the information of connection on log file"
		 "\n\nPPPStatus v" VERSION " - " AUTHOR "\n\n");
   }   
   rewind(log_file);
   while(fgets(buf, sizeof(buf), log_file))
     if(strstr(buf, "PPPStatus v " VERSION))
       exists = 1;
   if(!exists)
     fprintf(log_file, "PPPStatus v " VERSION " LOG file\n\n"
	     "=====\n"
	     " %s\n"
	     "=====\n\n", mth[tme.tm_mon]);
   
   if(!stats.online_hour && !stats.online_min && !stats.online_sec)
      return fclose(log_file);
   
   fprintf(log_file, ". : .\n\n");
   fprintf(log_file, "Connection Started at : %s", ctime(&stats.connect_time));
   if(quit)
     fprintf(log_file, "PPPStatus Exited at   : %s", ctime(&off));
   else
     fprintf(log_file, "Connection Finished at: %s", ctime(&off));
   if(stats.online_days)
     fprintf(log_file, "Connected Time: %d day(s) %d hour(s) %d minute(s) and %d second(s)\n\n", stats.online_days, stats.online_hour, stats.online_min, stats.online_sec);
   else
     fprintf(log_file, "Connected Time:  %d hour(s) %d minute(s) and %d second(s)\n\n", stats.online_hour, stats.online_min, stats.online_sec);
   if(check_costs)
     fprintf(log_file, "You will pay for this connection %s %.2f\n", costs.monetary_sign,  costs.total[0]);
   fprintf(log_file, "Transmition Information\n");
   fprintf(log_file, "TX Packets: %ld    ", stats.tx_packets);
   fprintf(log_file, "TX Bytes: %.0f    ", stats.tx_bytes);
   fprintf(log_file, "TX Errors: %ld\n", stats.tx_errors);
   fprintf(log_file, "Total Receive Information\n");
   fprintf(log_file, "RX Packets: %ld    ", stats.rx_packets);
   fprintf(log_file, "RX Bytes: %.0f    ", stats.rx_bytes);
   fprintf(log_file, "RX Errors: %ld\n\n", stats.rx_errors);
   if(quit)
     {
	fprintf(log_file, "(*) PPPStatus still with connection active (user quits PPPStatus).\n\n");
	fprintf(stdout, "Saving logfile... DONE!\n\n");
     }
   else {

      if(curmon != tme.tm_mon && curyear != tme.tm_year)
	warning = 3;
      else if (curmon != tme.tm_mon)
	warning = 2;
      else if (curyear != tme.tm_year)
	warning = 1;
      
      stc( (warning == 2 || warning == 3) ? FALSE : TRUE,
	  (warning == 1 || warning == 3) ? FALSE : TRUE);
      check_costs_file();
   }
   
 return fclose(log_file);
} 

/*-----------------------------------------------------------------------*
 * GET_TIME - Get the online time                                        *
 *-----------------------------------------------------------------------*/
void get_time(void)
{
   long x;
   stats.current_time = time(NULL); /* Get the current time */
   if(!stat(var_run_pid, &st)) 
     stats.connect_time = st.st_mtime; /* Get the time that PPP run */
   x = stats.current_time - stats.connect_time; /* Online Time */
   stats.online_sec = x % 60;
   stats.online_min = (x / 60) % 60; 
   stats.online_hour = (x / 3600) % 24;
   stats.online_days = (x / (3600 * 24)) % 365;
}

/*-----------------------------------------------------------------------*
 *  IS_ONLINE - Check if the connection is active                        *
 *-----------------------------------------------------------------------*/
int is_online(short int force, short int only_check)
{
   static int one_time;
   int rtrn = 0;
   char chk[128];
   FILE *routefile;
   
   if(!stat(var_run_pid, &st))
     {
	if(st.st_size)
	  {
	     if(!force)
	       if(online)
		 return 1;
	     
	     if((routefile = fopen(PATH_PROC_NET_ROUTE, "r")) == NULL)
	       close_ppps("Error! Unable to open %s\n"
			  "Check in your Kernel Configuration if /proc filesystem"
			  " is supported\n\n", PATH_PROC_NET_ROUTE);
	     
	     
	     while(fgets(chk, 128, routefile))
	       if(strstr(chk, interface)){
		  online = rtrn = 1;
		  break;
	       }
	     fclose(routefile);	     
	  }
	else
	  rtrn = 0;
     }
   else
     {
	if(one_time && disconnected && check_costs)
	  {
	     chcolor(colors.labels[0], colors.labels[1]);
	     mvprintw(check?16:17, 41, "You will pay (%s):", costs.monetary_sign);
	     refresh();
	     chcolor(colors.data[0], colors.data[1]);
	     mvprintw(check?16:17, 65, "%.2f  ", costs.total[0]);
	     refresh();
	     one_time = FALSE;
	  }
     }
   
   /* LED POWER ON/OFF */
   if(!only_check) {
      if(rtrn) {
	 if(!con || (force && con)){
	    chcolor(colors.power_led[0],colors.power_led[1]); move(12, 10); 
	    hline(CHAR,3);
	    con = 1; discon = 0; one_time = TRUE; online = 1;
	    refresh();
	 }
      }
      else{
	 if(!discon || (force && discon)) {
	    chcolor(colors.power_led[2], colors.power_led[3]); move(12, 10); 
	    hline(CHAR, 3);
	    con = 0; discon = 1; online = 0; disconnected = 1;
	    refresh();
	    log(0);
	 }
      }
   }
   return rtrn;
}

   
/*-----------------------------------------------------------------------*
 * UPDATE_INFO - Puts TX/RX Packets, Bytes, Erros on the screen          * 
 *-----------------------------------------------------------------------*/
void update_info(void)
{
   chcolor(colors.data[0] ,colors.data[1]);
   mvprintw(19, 65, "%ld   ", stats.online_days);
   mvprintw(19, 27, "%d:%02d:%02d   ", stats.online_hour, stats.online_min, stats.online_sec);
   mvprintw(20, 27, "%ld  ", stats.rx_packets);
   mvprintw(20, 65, "%ld  ", stats.tx_packets);
   mvprintw(22, 27, "%ld  ", stats.rx_errors);
   mvprintw(22, 65, "%ld  ", stats.tx_errors);
   
   switch(data_type[0])
     {
      case 'b' : 
	mvprintw(21, 27, "%.0f", stats.rx_bytes);
	mvprintw(21, 65, "%.0f", stats.tx_bytes);
	break;
      case 'k' :
	mvprintw(21, 27, "%.0f", stats.rx_bytes / 1024);
	mvprintw(21, 65, "%.0f", stats.tx_bytes / 1024);
	break;
      case 'm' :
	mvprintw(21, 27, "%.3f", (stats.rx_bytes / 1048570));
	mvprintw(21, 65, "%.3f", (stats.tx_bytes / 1048570));
	break;
     }	
}

/*-------------------------------------------------------------------------*
 * CLEAR_INFO - Clear TX/RX Packets, Bytes, Erros...                       *
 *-------------------------------------------------------------------------*/
void clear_info(void)
{
   chcolor(colors.background ,0);
   mvprintw(19, 65, "    ");
   mvprintw(19, 27, "              ");
   mvprintw(20, 27, "              ");
   mvprintw(20, 65, "            ");
   mvprintw(22, 27, "              ");
   mvprintw(22, 65, "            ");
   mvprintw(21, 27, "              ");
   mvprintw(21, 65, "            ");
   mvprintw(16, 21, "               ");
   mvprintw(17, 27, "0.00 KB/s     ");
   mvprintw(17, 65, "            ");
   mvprintw(16, 65, "            ");
   refresh();
}

/*-----------------------------------------------------------------------*
 * DO_EXIT                                                               *
 *-----------------------------------------------------------------------*/
void do_exit(int x, char *file, short int line)
{
   fprintf(stderr, "\n\nPPPStatus-v%s - Gabriel Montenegro", VERSION);
   if(!x)
     fprintf(stderr, "\n%s: Have no value in line %d\n\n", file, line);
   else
     fprintf(stderr, "\n%s contains an invalid value on line %d.\n\n", file, line);
   fprintf(stdout, "Exiting...\n\n");
   exit(0);
}

/*-----------------------------------------------------------------------*
 *  STC - Save Total Costs in a file                                     *
 *-----------------------------------------------------------------------*/

int stc(short int cm, short int cy)
{
   FILE *tf;
   char tmp[256];
   struct tm tme;
   long now;
   
   now = time(NULL);
   tme = *localtime(&now);
   sprintf(tmp, "%s/%s.%s-%02d", home, TOTAL_COST_FILE, cm ? mth[tme.tm_mon] : mth[tme.tm_mon-1], cy ? (tme.tm_year + 1900) : (tme.tm_year + 1899));
   if(!(tf = fopen(tmp, "w")))
     return 0;
   fprintf(tf, "# This file contains the total price to pay to your Telecompany in %s.\n"
	   "# I hope that this price below isn't too expensive for you :) hehe\n\n"
	   "total cost (%s) = %.2f\n", cm ? mth[tme.tm_mon]:mth[tme.tm_mon-1], costs.monetary_sign, costs.total[0] + costs.total[1]);
   fclose(tf);
   
   if(!cm)
     curmon = tme.tm_mon;
   if(!cy)
     curyear = tme.tm_year;
   
   return 1;
}

/*-----------------------------------------------------------------------*
 *   CHECK_COSTS_FILE                                                    *
 *-----------------------------------------------------------------------*/

void check_costs_file(void)
{
   FILE *tf;
   char tmp[258], get[128], real[256];
   struct tm tme;
   struct stat st;
   long curt = time(NULL);
   tme = *localtime(&curt);

   sprintf(tmp, "%s/%s.%s-%04d", home, TOTAL_COST_FILE, mth[tme.tm_mon], tme.tm_year + 1900);
   if(stat(tmp, &st))
     {
	if(!(tf = fopen(tmp, "w")))
	  {
	     endwin();
	     fprintf(stderr, "PPPStatus couldn't create %s file.\n"
		     "Check if your disk device isn't full.\n\n"
		     "Exiting...\n\n", tmp);
	     exit(0);
	  }
	else
	  {
	     costs.total[1] = 0;
	     fprintf(tf, "total cost = 0.00\n");
	     fclose(tf);
	  }
     }
   
   tf = fopen(tmp, "r");
   
   while(!feof(tf))
     {
	fgets(tmp, 256, tf);
	strcpy(real, opt_analiser(tmp));
	strcpy(get, get_option(real));
	if(!strncmp(real, "total cost", 10))
	  {
	     costs.total[1] = atof(get);
	  }
     }
}
   
/*-----------------------------------------------------------------------*
 *  COSTS_CALC                                                           *
 *-----------------------------------------------------------------------*/

void costs_calc(void)
{
   long ct = 0, curt;
   static int fst = TRUE;
   int warning = FALSE;
   struct tm tme;
   struct stat st; 
   curt = time(NULL);
   tme = *localtime(&curt);
   	    
   if(fst)
     {
	curmon = tme.tm_mon;
	curyear = tme.tm_year;
	fst = FALSE;
     }
   
   if(curmon != tme.tm_mon && curyear != tme.tm_year)
     warning = 3;
   else if (curmon != tme.tm_mon)
     warning = 2;
   else if (curyear != tme.tm_year)
     warning = 1;
   
   if(warning)
     {
	stc( (warning == 2 || warning == 3) ? FALSE : TRUE, 
	    (warning == 1 || warning == 3) ? FALSE : TRUE);
	check_costs_file();
     }
   
   costs.total[0] = 0;
   
   if(!stat(var_run_pid, &st))
     ct = st.st_mtime;
   
   costs.next_check = stats.current_time + (costs.minutes * 60);
   
   while(!(ct >= curt))
     {
	ct = ct + (costs.minutes * 60);
	tme = *localtime(&ct);
	if(tme.tm_wday > 0 && tme.tm_wday < 6) /* if is a work day */
	  costs.total[0] = costs.total[0] + costs.costs[0][tme.tm_hour];
	else if(tme.tm_wday == 6) /* if is saturday */
	  costs.total[0] = costs.total[0] + costs.costs[1][tme.tm_hour];
	else if(tme.tm_wday == 0) /* if is sunday */ 
	  costs.total[0] = costs.total[0] + costs.costs[2][tme.tm_hour];
     }
   
   ct = st.st_mtime;
   tme = *localtime(&ct);
   /* remove the first pulse cost */
   if(tme.tm_wday > 0 && tme.tm_wday < 6)
     costs.total[0] = costs.total[0] - costs.costs[0][tme.tm_hour];
   else if(tme.tm_wday == 6)
     costs.total[0] = costs.total[0] - costs.costs[1][tme.tm_hour];
   else if(tme.tm_wday == 0)
     costs.total[0] = costs.total[0] - costs.costs[2][tme.tm_hour]; 
   /* - */
   
   /* if there is a cost when connected, add this value to the total */
   if(costs.poc)
     costs.total[0] = costs.total[0] + costs.poc_cost;
   
}



/*-----------------------------------------------------------------------*
 *  GET_COSTS                                                            *
 *-----------------------------------------------------------------------*/

void get_costs(void)
{
   FILE *cfgfile;
   char cfg[128], tmp[256], real[256], get[256], cmp[15];
   short int c = 0, n = 0, x = 0, line = 0;
   char d[3][4] = { "wd", "sat", "sun" };
      
   sprintf(cfg, "%s/%s", home, costs.costs_file); 
   if((cfgfile = fopen(cfg, "r")) == NULL)
     {
	fprintf(stderr, "\nPPPStatus couldn't read costs file (%s)\n"
                        "Exiting...\n\n", cfg);
	exit(0);
     }
   while(!feof(cfgfile))
     {
	fgets(tmp, 256, cfgfile);
	line++;
	strcpy(real, opt_analiser(tmp) );
	strcpy(get, get_option(real) );
	
	if(strstr(real, "monetary_sign"))
	  {
	     if(get[0] != '\0')
	       snprintf(costs.monetary_sign, sizeof(costs.monetary_sign) , "%s", get);
	     else
	       do_exit(0, costs.costs_file, line);
	     c++;
	  }
	else if(strstr(real, "pulse_charge_time"))
	  {
	     if(get[0] != '\0')
	       costs.minutes = atof(get);
	     else
	       do_exit(0, costs.costs_file, line);
	     c++;
	  }
	else if(strstr(real, "pulse_on_connect"))
	  {
	     if(get[0] != '\0')
	       {
		  if(!strcmp(get, "yes"))
		    costs.poc = 1;
		  else if(!strcmp(get, "no"))
		    costs.poc = 0;
		  else
		    do_exit(1, costs.costs_file , line);
		  c++;
	       }
	     else
	       do_exit(0, costs.costs_file, line);
	  }
	else if(strstr(real, "on_connect_cost"))
	  {
	     if(get[0] != '\0')
	       costs.poc_cost = atof(get);
	     else
	       do_exit(0, costs.costs_file, line);
	     c++;
	  }
	else
	  {
	     for(x = 0; x <= 2; x++)
	       {
		  for(n = 0; n <= 23; n++)
		    {
		       sprintf(cmp, "%02d:00 %s", n, d[x]);
		       if(strstr(real, cmp))
			 {
			    if(get[0] != '\0')
			      costs.costs[x][n] = atof(get);
			    else
			      do_exit(0, costs.costs_file, line);
			    c++;
			 }
		    }
	       }
	  }	
     }
   if(c <= 76)
     {
	fprintf(stderr, "\n\nAll cfg fields in %s file weren't filled\n\n"
		"Exiting...%d\n\n", costs.costs_file,c);
	exit(0);
     }
}


/*-----------------------------------------------------------------------*
 * OPT_ANALISER - For cfgs files                                         *
 *-----------------------------------------------------------------------*/
char *opt_analiser(char *in)
{
   char real[256], temp[256], *rt;
   short int a, value = 0;
   bzero(temp, strlen(in));
   sprintf(temp, "%s", in);

   bzero(real, strlen(real));
   for(a = 0; a < strlen(temp)-1; a++){
      switch(temp[a])
	{
	 case '\t':
	   real[a] = '\0';
	   break;
	 case '#':
	   value++;
	   real[a] = '\0';
	   break;
	 default:
	   if(!value)
	     real[a] = temp[a];
	}
   }
   real[a] = '\0'; 
   rt = malloc(strlen(real));
   rt = (char *)real;
   return rt;
   free(rt);
}

/*-----------------------------------------------------------------------*
 *  GET_OPTION                                                           *
 *-----------------------------------------------------------------------*/
char *get_option(char *in)
{
   short int b = 0, a;
   char real[256], get[256], *rt;
   
   sprintf(real, "%s", in);
   for(a = 0; a <= strlen(real)-1; a++)
     if(real[a] == '=') {
	while(real[a+1] != '\0')
	  {
	     if(real[a+1] != ' ')
	       get[b] = real[a+1];
	     else
	       b--;
	     b++; a++;
	  }
	get[b] = '\0';
	break;
     }
   
   rt = malloc(strlen(get));
   rt = (char *)get;
   return rt;
   free(rt);
}

/*-----------------------------------------------------------------------*
 *  LOAD_OPTIONS - Open and read options in a file                       *
 *-----------------------------------------------------------------------*/
int load_options(void)
{
   FILE *optfile;
   char cfgfile[128], temp[256] , real[256], get[256];
   int a, color, light;
   short int line = 0;
   
   sprintf(cfgfile, "%s/" RC_FILENAME, home);

   optfile = fopen(cfgfile, "r");

   while(!feof(optfile))
     {
	fgets(temp, 256, optfile);
	line++;
	strcpy(real, opt_analiser(temp)); 
	strcpy(get, get_option(real));
	
	if(strstr(real, "costs_file"))
	  {
	     if(get[0] != '\0')
	       sprintf(costs.costs_file, "%s", get);
	     else
	       do_exit(0, RC_FILENAME, line);
	     check_costs = 1;
	  } 
	else if(strstr(real, "interface"))
	  {
	     if(get[0] != '\0')
	       {
		  interface = malloc(sizeof(char));
		  for(a = 0; a <= 100; a++)
		    {
		       sprintf(temp, "ppp%i", a);
		       if(strstr(temp, get))
			 {
			    sprintf(interface, "%s", get);
			    sprintf(var_run_pid, "/var/run/%s.pid", interface);
			    break;
			 }			    
		    }
		  if(strcmp(interface, get))
		    do_exit(1, RC_FILENAME, line);
                  free(interface);
	       }
	     else
	       do_exit(0, RC_FILENAME, line);
	  }
	else if(strstr(real, "video"))
	  {
	     if(get[0] != '\0') {
		if(!strcmp(get, "vga"))
		  VGA = 1;
		else if(!strcmp(get, "mono"))
		  VGA = 0;
		else
		  do_exit(1, RC_FILENAME, line);
	     }
	     else
	       do_exit(0, RC_FILENAME, line);
	  }
	else if(strstr(real, "speed"))
	  {
	     if(get[0] != '\0')
	       {
		  if(!strcmp(get, "14.4") || !strcmp(get, "28.8")  ||
		     !strcmp(get, "33.6") || !strcmp(get, "56")    ||
		     !strcmp(get, "64")   || !strcmp(get, "115.2") ||
		     !strcmp(get, "128")  || !strcmp(get, "256")   ||
		     !strcmp(get, "768")  || !strcmp(get, "1540"))
		    {
		       SPEED[0] = atof(get);
		       if(!strcmp(get, "256"))
			 SPEED[1] = 64;
		       else if(!strcmp(get, "768"))
			 SPEED[1] = 128;
		       else if(!strcmp(get, "1540"))
			 SPEED[1] = 256;
		       else
			 SPEED[1] = SPEED[0];
		    }
		  else
		    do_exit(1, RC_FILENAME, line);
	       }
	     else
	       do_exit(0, RC_FILENAME, line);
	  }
	else if(strstr(real, "user_email"))
	  {
	     if(get[0] != '\0') {
		strcpy(stats.user_box, get);
		check = 1;
	     }
	     else
	       do_exit(0, RC_FILENAME, line);
	  }
	else if(strstr(real, "bytes_info"))
	  {
	     if(get[0] != '\0') {
		if(!strcmp(get, "bytes") || !strcmp(get, "kbytes")
		   || !strcmp(get, "mbytes") )
		  sprintf(data_type, "%s", get);
		else
		  do_exit(1, RC_FILENAME, line);
	     }
	     else
	       do_exit(0, RC_FILENAME, line);
	  }
	else if(strstr(real, "ingoing") || strstr(real, "outgoing") ||
		strstr(real, "intersection") || strstr(real, "background") ||
		strstr(real, "border") || strstr(real, "data") ||
		strstr(real, "label") || strstr(real, "label") ||
		strstr(real, "version") || strstr(real, "power_led_on") || 
		strstr(real, "power_led_off") ) 
	  {
	     color = light = 0; 
	     if(strstr(real, "black")) 
	       color = 0;
	     else if(strstr(real, "red"))
	       color = 1;
	     else if(strstr(real, "green"))
	       color = 2;
	     else if(strstr(real, "yellow"))
	       color = 3;
	     else if(strstr(real, "blue"))
	       color = 4;
	     else if(strstr(real, "magenta"))
	       color = 5;
	     else if(strstr(real, "cyan"))
	       color = 6;
	     else if(strstr(real, "white"))
	       color = 7;
	     
	     if(strstr(real, "light"))
	       light = 1;
	     
	     if(strstr(real, "ingoing")) {
		colors.ingoing[0] = color; colors.ingoing[1] = light; 
	     }
	     if(strstr(real, "outgoing")) {
		colors.outgoing[0] = color; colors.outgoing[1] = light; 
	     }
	     if(strstr(real, "intersection")) {
		colors.intersection[0] = color; colors.intersection[1] = light; 
	     }
	     if(strstr(real, "background")) 
	       colors.background = color;
	     if(strstr(real, "data")) {
		colors.data[0] = color; colors.data[1] = light; 
	     }
	     if(strstr(real, "border")) {
		colors.border[0] = color; colors.border[1] = light; 
	     }
	     if(strstr(real, "labels")) {
		colors.labels[0] = color; colors.labels[1] = light; 
	     }
	     if(strstr(real, "power_led_on")) {
		colors.power_led[0] = color; colors.power_led[1] = light; 
	     }
	     if(strstr(real, "power_led_off")) {
		colors.power_led[2] = color; colors.power_led[3] = light; 
	     }
	     if(strstr(real, "version")) {
		colors.version[0] = color; colors.version[1] = light;
	     }
	  }
     }
   return fclose(optfile);
}

/*-----------------------------------------------------------------------*
 *  LED_ON/OFF - Show the leds on the screen                             *
 *-----------------------------------------------------------------------*/
void led_on(unsigned int who)
{
   if(who == LED_RX){
      chcolor(colors.ingoing[0], colors.ingoing[1]);
      move(12, 37); hline(CHAR, 2);
   }
   else if(who == LED_TX){
      chcolor(colors.outgoing[0], colors.outgoing[1]); 
      move(12, 41); hline(CHAR, 2);
   }
}


void led_off(unsigned int who)
{
   if(who == LED_RX){
      chcolor(colors.labels[0], colors.labels[1]);
      mvprintw(12, 37, "RX");
   }
   else if(who == LED_TX){
      chcolor(colors.labels[0], colors.labels[1]);
      mvprintw(12, 41, "TX");
   }
}


/*-----------------------------------------------------------------------*
 *  UPDATE_STAT - Update The Graphical Statistic                         *
 *-----------------------------------------------------------------------*/
void update_stat(int reload)
{
   int x, y;
   long bps[2];
   float makegraph_in = 0, makegraph_out = 0, ks;
   static short int stalled = FALSE; 
   /* Putting in the last hline the new statistic */
   
   if(!reload)
     {
	if(!stats.rx_bytes_comp)
	  stats.rx_bytes_comp = stats.rx_bytes;
	if(!stats.tx_bytes_comp)
	  stats.tx_bytes_comp = stats.tx_bytes;
	
	bps[0] = (stats.rx_bytes - stats.rx_bytes_comp);
	bps[1] = (stats.tx_bytes - stats.tx_bytes_comp);
	
	/* ks variable gets the speed in k/s of the connection */
	ks = ((float)(bps[0] + bps[1])/1024);

	if(ks >= 1) 
	  center(14, "%.2f KBytes/s", ks);
	else
	  {
	     if(stalled && !ks)
	       center(14, "Stalled");
	     else {
		center(14, "%.0f Bytes/s", ks * 1024);
		if(!ks)
		  stalled = TRUE;
		else
		  stalled = FALSE;
	     }
	  }
	
	if((ks > stats.top_speed) || reload)
	  {
	     mvprintw(17, 27, "%.2f KB/s", ks);
	     if(!reload)
	       stats.top_speed = ks;
	  }
	
	stats.rx_bytes_comp = stats.rx_bytes;
	stats.tx_bytes_comp = stats.tx_bytes;
   
	/* makegraph variable gets the amplitude of line bar */
	
	for(y = 0; y != 10; y++)
	  {
	     if( ( (float)bps[0] > (float)((SPEED[0]/8) * 102.4) * y ) &&
		( (float)bps[0] <= (float)((SPEED[0]/8) * 102.4) * (y+1)) )
	       makegraph_in = y + 1;
	     else if((float)bps[0] > (float)((SPEED[0]/8) * 1024))
	       makegraph_in = 10;
	     
	     if( ( (float)bps[1] > (float)((SPEED[1]/8) * 102.4) * y ) &&
		( (float)bps[1] <= (float)((SPEED[1]/8) * 102.4) * (y+1)) )
	       makegraph_out = y + 1;
	     else if((float)bps[1] > (float)((SPEED[1]/8) * 1024))
	       makegraph_out = 10;
	     
	  }
	
	/* clear the last column */
	
	for(x = 0; x != 10; x++) 
	  graph[x].hline[75] = 0;
	
	/* start to fill the blocks of the new column */
	for(x = 0; x < makegraph_in; x++)
	  graph[x].hline[75] = 1;
	
	for(x = 0; x < makegraph_out; x++)
	  {
	     if(graph[x].hline[75] == 1)
	       graph[x].hline[75] = 3;
	     else
	       graph[x].hline[75] = 2;
	  }
     }
   /* Hnn... now I have to print the new statistic on the screen... */
   
   for(y = 0; y != 10; y++) {
      for(x = 0; x != 76; x++) { 
	 /* Move the graphic to the left) */
	 if(!reload)
	   if(x)
	     graph[y].hline[x-1] = graph[y].hline[x];
	 
	 /* Print the graphic on screen */
	 if(graph[y].hline[x] == 3) /* intersection color */
	   { 
	      chcolor(colors.intersection[0], colors.intersection[1]);
	      move((y+10)-(2*y), x+2);
	      hline(CHAR, 1);
	   }
	 else if(graph[y].hline[x] == 1) /* ingoing color */
	   {
	      chcolor(colors.ingoing[0], colors.ingoing[1]);
	      move((y+10)-(2*y), x+2);
	      hline(CHAR, 1);
	   }
	 else if(graph[y].hline[x] == 2) /* outgoing color */
	   {
	      chcolor(colors.outgoing[0], colors.outgoing[1]);
	      move((y+10)-(2*y), x+2);
	      hline(CHAR, 1);
	   }
	 else
	   {
	      chcolor(colors.background, 0);
	      mvaddstr((y+10)-(2*y), x+2, " ");
	   }
      }
   }
}   

/*--------------------------------------------------------------------------*
 * SHOW_USAGE - Print the right usage on the screen                         *
 *--------------------------------------------------------------------------*/
void show_usage(char *name)
{
   fprintf(stderr, "\nPPPStatus v" VERSION " - " AUTHOR "\n"
	   "Usage:  %s [-i] [-v] [-s] ([--bytes] [--packets] [--errors] [--time]\n"
	   "                           [--ipaddress] [--costs])\n\n" 
	   " -i\t-Interface-\t\tppp0, ppp1, ...\n"
	   " -v\t-  Video  -\t\tvga, mono.\n"
	   " -s\t-  Speed  -\t\t14.4, 28.8, 33.6, 56, 64, 115.2, 128, 256,\n"
           "                                                \t\t768, 1540.\n\n"
	   " --bytes     -  Information of TX and RX bytes\n"
	   " --packets   -       \"      \"  \"   \"  \"  packets\n"
	   " --errors    -       \"      \"  \"   \"  \"  errors\n"
	   " --time      -  Online time information\n"
	   " --ipaddress -  Local IP Address information\n"
	   " --costs     -  Total price to pay\n\n"
	   "\n\t\tPlease, send bug reports to Gabriel Montenegro\n    "
	   "\t\t      <johnpetrucci@users.sourceforge.net>\n\n", name);
}

/*--------------------------------------------------------------------------*
 *  CHCOLOR  -  Change the Color                                            *
 *--------------------------------------------------------------------------*/
void chcolor(int x, int y) {
   
   init_pair(1, COLOR_BLACK, colors.background);
   init_pair(2, COLOR_RED, colors.background);
   init_pair(3, COLOR_GREEN, colors.background);
   init_pair(4, COLOR_YELLOW, colors.background);
   init_pair(5, COLOR_BLUE, colors.background);
   init_pair(6, COLOR_MAGENTA, colors.background);
   init_pair(7, COLOR_CYAN, colors.background);
   init_pair(8, COLOR_WHITE, colors.background);
   
   if(!y)
     attrset(COLOR_PAIR(x+1));
   if(y)
     attrset(COLOR_PAIR(x+1)|A_BOLD);
}

/*---------------------------------------------------------------------------*
 * CLOSE_PPPS                                                                *
 *---------------------------------------------------------------------------*/

void close_ppps(char *errmsg, ...)
{
   va_list trans;
   char *final;
   final = malloc(strlen(errmsg));
   va_start(trans, errmsg);
   (void)vsnprintf(final, strlen(errmsg), errmsg, trans);
   va_end(trans);
   endwin();
   curs_set(1);
   system("clear");
   fprintf(stderr, "%s", final);
   free(final);
   exit(0);
}

/*---------------------------------------------------------------------------*
 *  DIRC - Directory Creator                                                 *
 *---------------------------------------------------------------------------*/

int dirc(void)
{
   short int s;
   FILE *optfile;
   char dest[128];
   sprintf(dest, "%s/%s", home, RC_FILENAME);
   if(!stat(home, &st) && !stat(dest, &st));
   else
     {
	fprintf(stderr, "\nPPPStatus will now create the directory of configuration files and/or\n"
		"run with default configurations...\n\n"
		"\tDirectory   = %s\n"
		"\tConfig file = %s/%s\n\n", home, home, RC_FILENAME);
	for(s = 10; s != 0; s--)
	  {
	     fprintf(stderr, "\r%d secs left ", s);
	     sleep(1);
	  }
	fprintf(stderr, "\r");
	sprintf(dest, "%s/%s", home, RC_FILENAME);
	mkdir(home, 0700);
	if(!(optfile = fopen(dest, "w")))
	  {
	     fprintf(stderr, "\nPPPStatus couldn't create %s file.\n"
		     "Check if your disk device isn't full.\n\n"
		     "Exiting...\n\n", dest);
	     exit(0);
	  }
	else {
	   fprintf(optfile, "# PPPStatus v" VERSION " Configuration file - Gabriel Montenegro\n"
		   "# Please, leave interface, speed, high_contrast and video untouched (lowcase)\n\n"
		   "interface = ppp0           # ppp0, ppp1, ...\n"
		   "speed = 33.6               # 14.4, 28.8, 33.6, 56, 64, 115.2, 128, 256, 768, 1540 (kbits/s)\n"
		   "video = vga                # vga, mono\n"
		   "bytes_info = mbytes        # bytes, kbytes or mbytes\n"
		   "#user_email = root         # Check user email\n\n"
		   "# Costs Configuration\n"
		   "#costs_file = costs        # The filename of your costs cfg\n"
		   "                           # Leave it commented if you don't\n"
		   "                           # want to use this feature\n"
		   "                           # The costs file path are:\n"
		   "                           # ~/.pppstatus/ (for normal users)\n"
		   "                           # /var/log/ (if you ran runonboot or\n"
		   "                           # if you ran pppstatus with adm user (root))\n\n"
		   "# PPPStatus Theme\n"
		   "# The colors are: black, white, red, green, yellow, blue and magenta.\n"
		   "# Bright color: light\n\n"
		   "ingoing = light green\n"
		   "outgoing = yellow\n"
		   "intersection = green\n\n"
		   "# Other colors definitions\n"
		   "background = black\n"
		   "data = cyan\n"
		   "border = white\n"
		   "labels = white\n"
		   "version = white light\n"
		   "power_led_on = light green\n"
		   "power_led_off = red\n"
		   "# End Of Configuration File");
	   fclose(optfile);
	}
     }  
return 1;
}


/*---------------------------------------------------------------------------*
 *  FACE - Draw the face of the program                                      *
 *---------------------------------------------------------------------------*/
void face(void)
{
   int x, y;
   
   chcolor(colors.background, colors.background);
   for(x = 0; x <= 24; x++)
     for(y = 0; y <= 80; y++)
     {
	mvprintw(x, y, " "); 
     }
   refresh();
   chcolor(colors.border[0], colors.border[1]);
   border(0, 0, 0, 0, 0, 0, 0, 0);
   center(24, ".Author : Gabriel Montenegro <johnpetrucci@users.sourceforge.net>.");
   refresh();
   chcolor(colors.labels[0], colors.labels[1]);
   mvprintw(16, 6,"IP Address:");
   if(check_costs)
     {
	if(!check) x = 17; else x = 16;
	mvprintw(x, 41, "Total Costs (%s):", costs.monetary_sign);
     }
   if(check)
     mvprintw(17, 41, "New email(s):");
   mvprintw(19, 41,"Online Days:");
   mvprintw(19, 6, "Online Time:");
   mvprintw(20, 6, "Received Packets:");
   mvprintw(22, 6, "Errors on Receiving:");
   mvprintw(20, 41,"Transmited Packets:");
   
   switch(data_type[0])
     {
      case 'b' :
	mvprintw(21, 6, "Bytes Received:");
	mvprintw(21,41, "Bytes Transmited:");
	break;
      case 'k' :
	mvprintw(21, 6, "KBytes Received:");
	mvprintw(21,41, "KBytes Transmited:");
	break;
      case 'm' :
	mvprintw(21, 6, "MBytes Received:");
	mvprintw(21,41, "MBytes Transmited:");
	break;
     }
   
   mvprintw(22, 41,"Errors on Transmission:");
   mvprintw(12, 37,"RX");
   mvprintw(12, 41,"TX");
   mvprintw(18, 6,"Active Interface:");
   mvprintw(18, 41, "Graphic Using Speed:");
   mvprintw(13, 9, "ON/OFF"); 
   mvprintw(17, 6, "Top Speed:"); 
   refresh();
   chcolor(colors.version[0], colors.version[1]);
   center(00, ".PPPStatus v%s.", VERSION);
   refresh();
   chcolor(colors.data[0], colors.data[1]);
   mvprintw(18, 27, "%s", interface);
   mvprintw(18, 65, "%.1f Kb/s", SPEED[0]);
   mvprintw(17, 27, "%.2f KB/s", stats.top_speed);
   refresh();
   if(check_costs)
     {
	check_costs_file();
	if(is_online(0, 0));
	else
	  {
	     chcolor(colors.data[0], colors.data[1]);
	     mvprintw(check?16:17, 65, "%.2f", costs.total[1]);
	     refresh();
	  }
     }       
}

/*--------------------------------------------------------------------------*
 *  M A I N - The head of the program                                       *
 *--------------------------------------------------------------------------*/
int main(int argc, char *argv[]) 
{  
   short int x, y, key_pressed, kill = 0, totalcosts = FALSE;
   char *temp, chk[6];
   long now, old[2];
   int reload = FALSE;
   
   strncpy(home, (temp = getenv("HOME")) != NULL ? temp : "(null)", sizeof(home));
   if(home[1] == '\0' || !strcmp(home, "/root"))
     strcpy(home, ROOTandRUNONBOOTcfgPATH);
   else
     strcat(home, "/.pppstatus");
   /* Cfg functions */
   dirc();
   load_options();
   if(check_costs)
     get_costs();
   /* -------------- */
   for(x = 1; x < argc; x++) {
      char *a = argv[x];
      if(*a == '-')
	switch(a[1]) {
	 case 'i' : 
	   if(!(argv[x+1]) || (strlen(argv[x+1]) > 6) || strncmp(argv[x+1], "ppp", 3) || argv[x+1] == "ppp")
	     {
		show_usage(argv[0]);
		exit(0);
	     }
	   for(y = 0; y <= 100; y++)
	     {
		sprintf(chk, "ppp%d", y);
		if(!strcmp(chk, argv[x+1]))
		  {
		     interface = argv[x+1];
		     y = 'k'; /* 107 dec */
		     break;
		  }
	     }
	   if(y != 'k')
	     {
		show_usage(argv[0]);
		exit(0);
	     }
	   strcpy(var_run_pid, "/var/run/");
	   strcat(var_run_pid, interface);
	   strcat(var_run_pid, ".pid");
	   break;
	 case 'v' :
	   if(!(argv[x+1]))
	     {
		show_usage(argv[0]);
		exit(0);
	     }
	   if(!strcmp(argv[x+1], "mono")) {
	      VGA = 0;
	      break;
	   }
	   if(!strcmp(argv[x+1], "vga")) {
	      VGA = 1;
	      break;
	   }
	   else
	     {
		show_usage(argv[0]);
		exit(0);
	     }
	   break;
	 case 's' :
	   if(!(argv[x+1]))
	     {
		show_usage(argv[0]);
		exit(0);
	     }
	   if(!strcmp(argv[x+1], "14.4") || !strcmp(argv[x+1], "28.8")  ||
	      !strcmp(argv[x+1], "33.6") || !strcmp(argv[x+1], "56")    ||
	      !strcmp(argv[x+1], "64")   || !strcmp(argv[x+1], "115.2") ||
	      !strcmp(argv[x+1], "128")  || !strcmp(argv[x+1], "256")   ||
	      !strcmp(argv[x+1], "768")  || !strcmp(argv[x+1], "1540"))
	     {
		SPEED[0] = atof(argv[x+1]);
		if(!strcmp(argv[x+1], "256"))
		  SPEED[1] = 64;
		else if(!strcmp(argv[x+1], "768"))
		  SPEED[1] = 128;
		else if(!strcmp(argv[x+1], "1540"))
		  SPEED[1] = 256;
		else
		  SPEED[1] = SPEED[0];
	      break;
	     }
	   else
	     {
		show_usage(argv[0]);
		exit(0);
	     }
	 case '-' :
	   if(!strcmp(argv[x], "--bytes") || !strcmp(argv[x], "--packets") ||
	      !strcmp(argv[x], "--errors") || !strcmp(argv[x], "--time") ||
	      !strcmp(argv[x], "--ipaddress") || !strcmp(argv[x], "--costs"))
	     {
		if(is_online(1, 1))
		  {
		     get_stat();
		     fprintf(stdout, "\nPPPStatus v%s - ", VERSION );
		     if(!strcmp(argv[x], "--bytes")) {
			fprintf(stdout, "%s Bytes information\n\n", interface);
			fprintf(stdout, "Bytes sent: %.0f\n"
				"Bytes received: %.0f\n\n", stats.tx_bytes,
				stats.rx_bytes);
		     }
		     else if(!strcmp(argv[x], "--packets"))
		       {
			  fprintf(stdout, "%s Packets Information \n\n", interface);
			  fprintf(stdout, "Packets sent: %ld\n"
				  "Packets received: %ld\n\n", stats.tx_packets,
				  stats.rx_packets);
		       }
		     else if(!strcmp(argv[x], "--errors"))
		       {
			  fprintf(stdout, "%s Errors information\n\n", interface);
			  fprintf(stdout, "Errors on sending: %ld\n"
				  "Errors on receiving: %ld\n\n", stats.tx_errors,
				  stats.rx_errors);
		       }
		     else if(!strcmp(argv[x], "--ipaddress"))
		       {
			  fprintf(stdout, "%s Ip Address: ", interface);
			  ip_address(0);
			  fprintf(stdout, "%s\n\n", stats.ip_addr_rtrn);
		       }
		     else if(!strcmp(argv[x], "--costs"))
		       {
			  if(!check_costs)
			    {
			       fprintf(stderr, "You have not configured the costs file. Please, "
				       "edit the costs file that\ncomes with PPPStatus tarball "
				       "and configure either in pppstatus.cfg file\nthe "
				       "item \"costs_file\".\n\n");
			       exit(0);
			    }
			  else
			    {
			       costs_calc();
			       check_costs_file();
			       fprintf(stdout, "%s - Costs Calculator\n\n"
				       "Costs of the current connection: %s %.02f\n"
				       "Total Costs: %s %.02f\n\n", 
				       interface, costs.monetary_sign, 
				       costs.total[0], costs.monetary_sign,
				       costs.total[0] + costs.total[1]);
			    }
		       }
		     else
		       {
			  get_time();
			  fprintf(stdout, "%s Online Time\n\n", interface);
			  fprintf(stdout, "%d day%s%d hour%s%d minute%s%d second%s\n\n",
				  stats.online_days, (stats.online_days != 1)?"s, ":", ",
				  stats.online_hour, (stats.online_hour > 1)?"s ":" ",
				  stats.online_min,  (stats.online_min  > 1)?"s ":" ",
				  stats.online_sec,  (stats.online_sec  > 1)?"s ":" " );
		       }
		  }
		else
		  fprintf(stderr, "\nPPPStatus hasn't detected connection "
			  "in %s interface\n\n", interface); 
	     }
	   else
	     show_usage(argv[0]);	   
	   kill = 1;
	}
   }
   if(kill)
     exit(0);
   /* Initialize some info variables */
   stats.online_days = 0;
   stats.online_hour = 0; stats.online_min = 0; stats.online_sec = 0;
   stats.tx_packets = 0; stats.tx_bytes = 0; stats.tx_errors = 0;
   stats.rx_packets = 0; stats.rx_bytes = 0; stats.rx_errors = 0;
   stats.tx_bytes_comp = 0; stats.tx_bytes_comp = 0;
   fprintf(stderr, "Running PPPStatus v%s...", VERSION);
/*   printf("\e(U");  Thanks kspoon  commented out JGH Aug 8 2001 */
   initscr();
   intrflush(stdscr, FALSE);
   if(VGA)
     start_color();
   init_color(1, COLOR_RED, COLOR_GREEN, COLOR_BLUE);
   curs_set(0); /* Toggle the cursor off */ 
   nonl(); 
   noecho(); 
   cbreak(); 
   nodelay(stdscr, TRUE);
   face(); 
   update_info();
   if(check)
     check_email();
   if(check_costs)
     check_costs_file();
   now = time(NULL);
   old[0] = old[1] = time(NULL) - 3;
   while(1){ 
      if(is_online(0, 0)) {
	 if(disconnected) {
	    for(y = 0; y != 10; y++)
	      for(x = 0; x != 75; x++)
		graph[y].hline[x] = 0;
	    if(check_costs)
	      check_costs_file();
	    stats.rx_bytes = stats.tx_bytes = stats.tx_packets = stats.rx_packets = stats.rx_errors = stats.tx_errors =stats.rx_bytes_comp = stats.tx_bytes_comp = stats.rx_packets_comp = stats.tx_packets_comp = 0;
	    stats.top_speed = stats.online_days = stats.online_hour = stats.online_min = stats.online_sec = 0;
	    if(check)
	      now = old[0] = old[1] = time(NULL);
	    clear_info();
	    update_stat(0);
	    update_info();
	 }       
	 get_stat();
	 /* In case of a new RX Packet... the RX Led Turn on 
	  * Else... Turn off */
	 if(stats.rx_packets != stats.rx_packets_comp)
	   led_on(LED_RX);
	 else
	   led_off(LED_RX);
	 /* In case of a new TX Packet... the TX Led Turn on
	  * Else... Turn off */
	 if(stats.tx_packets != stats.tx_packets_comp)
	   led_on(LED_TX);
	 else
	   led_off(LED_TX);
	 
	 stats.rx_packets_comp = stats.rx_packets;
	 stats.tx_packets_comp = stats.tx_packets;
	 
	 get_time();
	 update_info();
	 
	 if((!sec_value && first) || disconnected)
	   {
	      if(check_costs)
		costs_calc();
	      sec_value = stats.online_sec;
	      min_value = stats.online_min;
	      ip_address(1);
	      update_stat(0);
	      first = 0; disconnected = 0;
	   }
	 
	 if(check_costs && (costs.next_check <= stats.current_time)) 
	   costs_calc();

	 now = time(NULL);
	 if(check)
	   if(now - 5 == old[0]) /* every 5 secs email is checked */
	   {
	      old[0] = now;
	      check_email();
	   }
	 
	 if(((now - 3 >= old[1]) || reload) && check_costs)
	   {
	      reload = FALSE;
	      old[1] = now;
	      
	      check ? (x = 16) : (x = 17);
	      
	      chcolor(colors.labels[0], colors.labels[1]);
	      if(totalcosts == TRUE)
		mvprintw(check?16:17, 41, "Total Costs (%s): ", costs.monetary_sign);
	      else
		mvprintw(check?16:17, 41, "Costs (%s):       ", costs.monetary_sign);
	      refresh();
	      
	      chcolor(colors.data[0], colors.data[1]);
	      if(totalcosts == TRUE)
		mvprintw(check?16:17, 65, "%.2f", costs.total[0] + costs.total[1]);
	      else
		mvprintw(check?16:17, 65, "%.2f   ", costs.total[0]);
	      
	      if(totalcosts == FALSE)
		totalcosts = TRUE;
	      else
		totalcosts = FALSE;
	   }
	 
	 if(stats.online_sec != sec_value)
	   {
	      update_stat(0);
	      sec_value = stats.online_sec;
	   }
	 usleep(10L); 
      }
      else
	usleep(100000L);
      if(check)
	{
	   now = time(NULL);
	   if(now - 5 >= old[0]) /* Every 5 secs email is checked */ 
	     {
		old[0] = now;
		check_email();
	     }
	}
      key_pressed = getch();
      if(key_pressed != ERR && tolower(key_pressed) == 'q')
	break;
      else if(key_pressed != ERR && tolower(key_pressed) == 'r' && online)
	{
	   face();
	   is_online(1, 0);
	   ip_address(1);
	   if(check) /* Added test to stop crash.  JGH 25 Aug 2001 */
	     {
	       check_email();
	     }
	   update_stat(1);
	   reload = TRUE;
	}
   }
   endwin();
   system("clear");
   curs_set(1);
   if(!disconnected)
     log(1);
   fprintf(stdout, "PPPStatus v" VERSION " - " AUTHOR "\n\n");

   return 0; 
}
