/* 
 * Prospect: a developer's system profiler.
 * GDB-based disassembler.
 *
 * COPYRIGHT (C) 2001-2004 Hewlett-Packard Company
 *
 * Author: Bob Montgomery, HP
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/* $Id: dass.c,v 1.3 2004/01/09 20:29:27 type2 Exp $ */

/* Use gdb's disassembly capability from a program.
 * Bob Montgomery, HP
 * 20 Nov 2001
 */

/* Issues with talking to gdb:
 * gdb's prompt is "(gdb) ".
 * It does not end with a newline, so fgets will hang if you try
 * to read the current prompt, waiting for gdb to write something
 * else, while gdb waits for a command.
 * That's why I'm not using fgets, but instead use line_or_prompt
 * to read to an EOL or a prompt, whichever arrives first.
 */ 

#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <fcntl.h>
#include <string.h>
#include <signal.h>

#include "dass.h"

#define DASS_MAGIC 0xda55
#define DASS_MAXLINELEN 1024
#define DASS_MAXLINES 1024

static char *dass_dbg = "/usr/bin/gdb";
static char *dass_prompt = "(gdb) ";

struct dass_id_str {
    int cmds[2];
    int resp[2];
    FILE *dass_cp;
    FILE *dass_rp;
    char *filename;
    int magic;
    pid_t pid;
};

typedef struct dass_id_str dass_id;

static void skip_to_prompt(FILE *fp, char *prompt);
static int line_or_prompt(FILE *fp, char *prompt, char *line, int len);


void *
dass_open(const char *filename)
{
    dass_id *handle;

    /* verify that the debugger exists */
    if (access(dass_dbg, R_OK | X_OK | F_OK) != 0) {
        return NULL;
    }

    /* verify that the target file exists */
    if (access(filename, R_OK | F_OK) != 0) {
        return NULL;
    }
        
    handle = malloc(sizeof(dass_id));
    if (handle == NULL) {
        errno = ENOMEM;
        return NULL;
    }
    handle->magic = DASS_MAGIC;
    if (pipe(handle->cmds) != 0) {
        perror("pipe failed");
        return NULL;
    }
    if (pipe(handle->resp) != 0) {
        perror("pipe failed");
        return NULL;
    }

    if ((handle->pid = fork()) == 0) {
        /* child needs to read cmds, and write resp */
        char *args[8];
        char *envp[8];
        args[0] = dass_dbg;
        args[1] = "--nx";
        args[2] = "--nw";
        args[3] = "--quiet";
        args[4] = (char *)filename;
        args[5] = NULL;
        envp[0] = NULL;
        dup2(handle->cmds[0], 0);
        dup2(handle->resp[1], 1);
        fcntl(0, F_SETFD, 0);
        fcntl(1, F_SETFD, 0);
        close(handle->cmds[0]);
        close(handle->cmds[1]);
        close(handle->resp[0]);
        close(handle->resp[1]);
        execve(dass_dbg, args, envp);
        /* shouldn't be here */
        perror("disassembler debugger execve failed");
        exit(1);
    } 
    /* Parent closes unused ends of pipes, or else read won't
     * get EOF when the child closes the same end.
     */

    close(handle->cmds[0]); /* won't be reading commands */
    close(handle->resp[1]); /* won't be writing responses */

    /* Use stream IO on the pipes */
    if ((handle->dass_cp = fdopen(handle->cmds[1], "w")) == NULL) {
        perror("fdopen failed");
        return NULL;
    }
    /* Set line buffering for writing commands */
    setvbuf(handle->dass_cp, (char *)NULL, _IOLBF, 0);
    if ((handle->dass_rp = fdopen(handle->resp[0], "r")) == NULL) {
        perror("fdopen failed");
        return NULL;
    }
    /* FIXME: Should I verify that gdb is up and responding here? */ 
    skip_to_prompt(handle->dass_rp, dass_prompt);
    fprintf(handle->dass_cp, "set width 0\n");
    skip_to_prompt(handle->dass_rp, dass_prompt);
    return (void *)handle;
}

int 
dass_close(void *vhandle)
{
    dass_id *handle = (dass_id *)vhandle;
    int status = 0;

    if (!handle || (handle->magic != DASS_MAGIC))
        return -1;

    fprintf(handle->dass_cp, "quit\n");
#ifdef CAREFULLY_WAIT_FOR_GDB_TO_FINISH_INSTEAD_OF_CLOSING_ITS_PIPE
    {
        char buf[DASS_MAXLINELEN];
        while (fgets(buf, DASS_MAXLINELEN, handle->dass_rp) != NULL) 
            ;
    }
#endif
    fclose(handle->dass_rp);
    fclose(handle->dass_cp);
    if (waitpid(handle->pid, &status, 0) < 0) {
        perror("waitpid failed");
    }
    free(handle);
    /* This is probably too careful.  Why not just return 0 here? */
    if (WIFEXITED(status)) {
        return WEXITSTATUS(status);
    } else if (WIFSIGNALED(status)) {
        if (WTERMSIG(status) != SIGPIPE)
            return WTERMSIG(status);
        else
            return 0;
    } else {
        return status;
    }
}

char **
dass(void *vhandle, char *bp, char *ep)
{
    char *lines[DASS_MAXLINES + 1];
    char **rv;
    char buf[DASS_MAXLINELEN + 1];
    char *bufp;
    int ii, linecnt, done;
    dass_id *handle = (dass_id *)vhandle;

    if (!handle || (handle->magic != DASS_MAGIC))
        return NULL;

    /* don't use %p for pointers, it prints 0x0 as "(nil)" */
    fprintf(handle->dass_cp, "disas 0x%lx 0x%lx\n", (long)bp, (long)ep);
    done = 0;
    linecnt = 0;
    while (!done && (linecnt < DASS_MAXLINES)) {
        if (line_or_prompt(handle->dass_rp, dass_prompt, 
                           buf, DASS_MAXLINELEN) <= 0) {
            done++;
        } else if (strncmp(buf,"0x", 2) == 0) {
            /* a line of disassembly, remove the newline */
            bufp = strstr(buf, "\n");
            if (bufp)
                *bufp = '\0';
            lines[linecnt] = (char *)malloc(strlen(buf) + 1);
            strcpy(lines[linecnt], buf);
            linecnt++;
        }
    }
    if (!done) {
        skip_to_prompt(handle->dass_rp, dass_prompt);
    }
    rv = NULL;
    lines[linecnt++] = NULL;
    rv = (char **)malloc(linecnt * sizeof(char *));
    if (rv == NULL) {
        errno = ENOMEM;
        return NULL;
    }
        
    for (ii=0; ii<linecnt; ii++) {
        rv[ii] = lines[ii];
    }
    return rv;
}

void
dass_free(char **lines)
{
    char **ptr;
    if (lines) {
        ptr = lines;
        while (*ptr != NULL) {
            free(*ptr);
            ptr++;
        }
        free(lines);
    }
}

static
void
skip_to_prompt(FILE *fp, char *prompt)
{
    int cc;
    int state = 0;
    int final = strlen(prompt);
    while (state < final) {
        if ((cc = fgetc(fp)) == EOF) 
            state=final;
        else if (cc == prompt[state]) 
            state++;
        else
            state=0;
    }
}

/* Returns:
 * -1: EOF, 0: prompt reached, or 1: newline reached
 */
static
int
line_or_prompt(FILE *fp, char *prompt, char *line, int len)
{
    int cc;
    int state = 0;
    int final = strlen(prompt);
    int rv = 0;
    while ((state < final) && (rv == 0)) {
        if ((cc = fgetc(fp)) == EOF)
            rv = -1;
        else if (cc == '\n') {
            if (--len > 0) {
                *line++ = cc;
            }
            rv = 1;
        }
        else if (cc == prompt[state]) {
            if (--len > 0) {
                *line++ = cc;
            }
            state++;
        } else {
            if (--len > 0) {
                *line++ = cc;
            }
            state=0;
        }
    }
    *line = '\0';
    return rv;
}
