#ifndef _DTREE_INCLUDED
#define _DTREE_INCLUDED

/* 
 * Prospect: a developer's system profiler.
 * Digital Tree ADT.
 *
 * COPYRIGHT (C) 2001-2004 Hewlett-Packard Company
 *
 * Authors: Doug Baskins, HP
 *          Alex Tsariounov, HP
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/* $Id: dtree.h,v 1.3 2004/01/09 20:29:27 type2 Exp $ */

/*
 * Sparce arrays implemented as a Digital Tree, or Trie.
 * See Knuth, Volume 3, pp. 492-507, 2nd Edition, 1998, for reference.
 * Note: Trie and Digital Tree are synonymous.
 *
 * This ADT is a quaternary Trie.  All references go 16 levels every
 * access.  Not the most efficient, however, it serves our needs now.
 * The leaf node stores a pointer that may be used to access your
 * sturcture, or may be cast to an unsigned long for simple storage.
 * Branches are how to get to flowers which is where the data is.
 */

/*
 *******************************************************************************
 * Define shortcuts to the naming of dtree entry points.
 *
 * All return a Pointer to associated Value and NULL in if none,
 * except dtree_del().  Note that not all of these are currently implemented.
 * You can of course use the regular functions, however, life will be
 * easier if you use these defines as an interface definition, which was
 * their intended purpose.  Pay attention to the address-of operators: this
 * makes is easy to set up a dtree by first declaring a void* then using DTI
 * to put the initial value in it.  Dtree handles all the memory management.
 *
 * For example:
 *
 *    unsigned int *ui_ptr;
 *    struct big_struct *bs_ptr;
 *    void *ui_tree=NULL, *bs_tree=NULL;
 *
 *    ui_ptr = DTI(ui_tree, 1993);
 *    if (*ui_ptr==NULL)
 *        *ui_ptr = 42;
 *    else
 *        *ui_ptr += 42;
 *
 *    bs_ptr = DTI(bs_tree, 451);
 *    if (*bs_ptr==NULL) {
 *        bs_ptr=malloc(sizeof(big_struct));
 *        bs_ptr->bs_member = 42;
 *    }
 *    else 
 *        bs_ptr->bs_member += 42;
 *
 * The common idiom for retrieving all members from the tree is:
 *
 *    unsigned long idx;
 *    void **P;
 *   
 *    for (idx=0L; P  = DTF(ui_tree, idx);
 *                 P != NULL:
 *                 P  = DTN(ui_tree, idx))
 *    {
 *        printf("Index %ld holds: %ld\n", idx, (unsigned long) *P);
 *    }
 *
 * This will not print out emply slots.  The mechanism for larger structures
 * and pointers is the same.
 *    
 *******************************************************************************
 */

#define	DTG(Pdt, Idx) (dtree_get (  Pdt,    Idx))  /* Search or lookup        */
#define	DTI(Pdt, Idx) (dtree_ins (&(Pdt),   Idx))  /* Insert Index            */
#define	DTD(Pdt, Idx) (dtree_del (&(Pdt),   Idx))  /* Delete Index, 1:success */

#define	DTF(Pdt, Idx) (dtree_first( Pdt,  &(Idx))) /* Get minimum Index       */
#define	DTL(Pdt, Idx) (dtree_last ( Pdt,  &(Idx))) /* Get maximum Index       */
#define	DTP(Pdt, Idx) (dtree_prev ( Pdt,  &(Idx))) /* Get predecessor Index   */
#define	DTN(Pdt, Idx) (dtree_next ( Pdt,  &(Idx))) /* Get successor Index     */


/****************************************************************/
/*      Define dtree prototypes for use by USER programs        */
/****************************************************************/

/* May be cast to (unsigned long *) etc. */
void **dtree_get(void *Pdt, unsigned long Index); 
void **dtree_ins(void **PPdt, unsigned long Index);
		                       
int  dtree_del(void **PPdt, unsigned long Index); /* If none, return 0 */
int dtree_free(void **PPdt);                      /* Free whole tree */

void **dtree_first(void *Pdt, unsigned long *PIndex);
void  **dtree_last(void *Pdt, unsigned long *PIndex);
void  **dtree_prev(void *Pdt, unsigned long *PIndex);
void  **dtree_next(void *Pdt, unsigned long *PIndex);


#endif /* _DTREE_INCLUDED */
