/* 
 * Prospect: a developer's system profiler.
 *
 * COPYRIGHT (C) 2001-2004 Hewlett-Packard Company
 *
 * Author: Alex Tsariounov, HP
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/* $Id: malloc_debug.c,v 1.4 2004/01/09 20:29:28 type2 Exp $ */

/*
********************************************************************************
**
**                          PROSPECT PROJECT
**                         MALLOC DEBUG MODULE
**
********************************************************************************
*/

/* Comment: this is interesting, but if you can use Electric Fence, that's
 *          much better.
 */

#ifndef __LINT__
static const char gRCSid[] = "@(#) $Id: malloc_debug.c,v 1.4 2004/01/09 20:29:28 type2 Exp $";
#endif

/*
 *  System Header Files
 */
#include <sys/param.h>
#include <stdio.h>
#include <stdlib.h>

/*
 * Malloc Debug Header Files
 */
#include "../dtree/dtree.h"
#include "malloc_debug.h"

#ifdef MALLOC_DEBUG

FILE *gpFileErr=NULL;

/* Number of bytes to make the fence (bytes between mallocs) */
static const size_t cFENCE = (size_t) 2;

/* Malloc Debug Global Variables */
static void *gMallocHead = (void*) NULL;

void *
DebugMalloc(size_t Size, int Line, char *pFileName)
{
   void    *buf;
   void   **p;
   size_t  *psize, ii;
   char    *puchar;

   if (!gpFileErr) gpFileErr=stderr;

   buf = malloc(Size + cFENCE);

   puchar = (char*)buf + Size;

   for (ii = 0; ii < cFENCE; ii++)
   {
      *puchar++ = 0xaa;
   }

   p = DTI(gMallocHead, (unsigned long)buf);

   psize = (size_t*)p; 

   *psize = Size;

   if (gpFileErr)
   {
      fprintf (
                 gpFileErr, 
                 "0x%lx = malloc(%3d), line = %4d, file = %s\n", 
                 (unsigned long)buf,
                 (unsigned int)Size, 
                 Line, 
                 pFileName
              );
      fflush((FILE *)0);
   }

   return(buf);

} /* DebugMalloc() */


void *
DebugCalloc( size_t  Size1, 
             size_t  Size2, 
             int    Line, 
             char   *pFileName
           )
{
   void     *buf;
   void    **p;
   size_t   *psize, size, ii;
   char     *puchar;

   size = Size1 * Size2;

   buf = calloc(1, size + cFENCE);

   puchar = (char *)buf + size;

   for (ii = 0; ii < cFENCE; ii++)
   {
      *puchar++ = 0xaa;
   }

   p = DTI(gMallocHead, (unsigned long)buf);

   psize = (size_t*)p; 

   *psize = size;

   if (gpFileErr)
   {
      fprintf (
                 gpFileErr, 
                 "0x%lx = calloc(%3d,%d), line = %4d, file = %s\n", 
                 (unsigned long)buf,
                 (unsigned int)Size1, 
                 (unsigned int)Size2, 
                 Line, 
                 pFileName
              );

      fflush((FILE *)0);
   }

   return(buf);

} /* DebugCalloc() */


void
DebugFree(void *Buf, int Line, char *pFileName)
{
   void   **p;
   size_t  *psize;
   char    *puchar;
   size_t  size, ii, flag;

   p = DTI(gMallocHead, (unsigned long)Buf);

   psize = (size_t *)p; 

   size = *psize;

   if (gpFileErr)
   {
      fprintf (
                 gpFileErr,
                 "0x%lx free() of %4d bytes, line = %4d, file = %s", 
                 (unsigned long)Buf, 
                 (unsigned int)*psize,
                 Line, 
                 pFileName
              );
   }

   puchar = (char  *)Buf + size;

   for (flag = ii = 0; ii < cFENCE; ii++)
   {
      if (puchar[ii] != (char)0xaa) { flag++; }
   }
   if (flag)
   {
      /* force output even if not tracing */
      if (gpFileErr == NULL) { gpFileErr = stderr; }

      fprintf (
                 gpFileErr,
                 " 0x%lx is Corrupt!!, size = %d, #bad = %d",
                 (unsigned long)Buf,
                 (unsigned int)size,
                 (unsigned int)flag
              );

      for (ii = 0; ii < cFENCE; ii++)
      {
         fprintf (
                    gpFileErr,
                    "%02x", 
                    puchar[ii]
                 );
      }
   }

   if (gpFileErr)
   {
      fprintf(gpFileErr, "\n");
      fflush((FILE *)0);
   }

   *psize = 0;

   if (flag == 0) 
      free(Buf);

   return;

} /* DebugFree() */

void
DebugFreeAll(void)
{
   unsigned long  Index;
   size_t *PValue;

   /* Run through list and extract all values */
   if (gMallocHead)
       for (Index = 0L, PValue  = (size_t *) DTF(gMallocHead, Index);
            PValue != (size_t *) NULL;
            PValue  = (size_t *) DTN(gMallocHead, Index)
           )
        {
            if (PValue != 0)
            {
                /* Free memory if not already freed.
                 * Note that we store the _size_ of the chunk at the 
                 * pointer - not the pointer.  The Index is the pointer.
                 */
                DebugFree((void *)Index, __LINE__, __FILE__);
            }
        }

   return;

} /* DebugFreeAll() */

#endif  /* MALLOC_DEBUG */
