/****************************************************************************
** jabcon.cpp - core control class for Psi
** Copyright (C) 2001, 2002  Justin Karneges
**
** This program is free software; you can redistribute it and/or
** modify it under the terms of the GNU General Public License
** as published by the Free Software Foundation; either version 2
** of the License, or (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307,USA.
**
****************************************************************************/

#include<qapplication.h>
#include<qstring.h>
#include<qsettings.h>
#include<qstylesheet.h>

#include<qfile.h>
#include<qmessagebox.h>
#include<qtimer.h>
#include<qsound.h>
#include<qprocess.h>
#include<qdesktopwidget.h>

#include"jabcon.h"
#include"common.h"

#include"eventdlg.h"
#include"historydlg.h"
#include"statusdlg.h"
#include"optionsdlg.h"
#include"servicesdlg.h"
#include"infodlg.h"
#include"info.h"
#include"searchdlg.h"
#include"offermaindlg.h"

#include"anim.h"


jabcon::jabcon(QObject *par)
:QObject(par)
{
	pdb(DEBUG_JABCON, QString("%1 v%2\n By Justin Karneges\n    infiniti@affinix.com\n\n").arg(PROG_NAME).arg(PROG_VERSION));

	lastStatusString = "";
	mainwin = 0;

	// load the profile
	pro.reset();
	pro.fromFile(pathToProfileConfig(activeProfile));

	// if no accounts, add a blank account
	if(pro.acc.isEmpty()) {
		pro.acc.append(UserAccount());
	}

	// set the "global" profile data
	mwgeom = pro.mwgeom;
	lastStatusString = pro.lastStatusString;
	useSound = pro.useSound;

	// set the preferences
	option = pro.prefs;
}

bool jabcon::init()
{
	// first thing, try to load the iconset
	if(!loadPsiIconSet(option.iconset)) {
		option.iconset = "stellar";
		if(!loadPsiIconSet(option.iconset)) {
			QMessageBox::critical(0, tr("Error"), tr("Unable to load iconset!  Please make sure Psi is properly installed."));
			return FALSE;
		}
	}

	// create a new session
	s = new JabSession(&jsm);
	jsm.add(s);
	debug_jabber = s->serv();

	// setup the main window
	mainwin = new MainWin(option.alwaysOnTop, 0, "psimain");
	mainwin->userlist = &s->userlist;
	mainwin->setUseDock(option.useDock);

	// if the coordinates are out of the desktop bounds, reset to the top left
	QRect r = QApplication::desktop()->screenGeometry();
	if(mwgeom.x() >= r.right() || mwgeom.y() >= r.bottom()) {
		mwgeom.setX(r.left() + 32);
		mwgeom.setY(r.top() + 32);
	}
	mainwin->move(mwgeom.x(), mwgeom.y());
	mainwin->resize(mwgeom.width(), mwgeom.height());

	cvp = mainwin->cvlist->createProfile("justin@orbit.tenpura.org", "Orbit", TRUE);

	mainwin->setUsingSSL(FALSE);

	fileserv = new FileServer;
	//fileserv->setServer("127.0.0.1", 12346);
	//FileServerItem *i = fileserv->addFile("/opt/kde", "", "");
	//if(i) {
	//	printf("offering: [%s]\n", i->url.latin1());
	//}

	// jsm
	connect(&jsm, SIGNAL(accountSettingsChanged()), SLOT(slotApplyAccounts()));
	connect(&jsm, SIGNAL(jab_connected(JabSession *)),                                           SLOT(jab_connected(JabSession *)));
	connect(&jsm, SIGNAL(jab_disconnected(JabSession *)),                                        SLOT(jab_disconnected(JabSession *)));
	connect(&jsm, SIGNAL(jab_statusUpdate(JabSession *, JabUpdate *)),                           SLOT(jab_statusUpdate(JabSession *, JabUpdate *)) );
	connect(&jsm, SIGNAL(jab_error(JabSession *, JabError *)),                                   SLOT(jab_error(JabSession *, JabError *)) );
	connect(&jsm, SIGNAL(jab_messageReceived(JabSession *, const JabMessage &)),                 SLOT(jab_messageReceived(JabSession *, const JabMessage &)) );
	connect(&jsm, SIGNAL(jab_resourceAvailable(JabSession *, const Jid &, const JabResource &)), SLOT(jab_resourceAvailable(JabSession *, const Jid &, const JabResource &)));
	connect(&jsm, SIGNAL(jab_resourceUnavailable(JabSession *, const Jid &)),                    SLOT(jab_resourceUnavailable(JabSession *, const Jid &)));
	connect(&jsm, SIGNAL(jab_contactChanged(JabSession *, JabRosterEntry *)),                    SLOT(jab_contactChanged(JabSession *, JabRosterEntry *)) );
	connect(&jsm, SIGNAL(jab_contactNew(JabSession *, JabRosterEntry *)),                        SLOT(jab_contactNew(JabSession *, JabRosterEntry *)) );
	connect(&jsm, SIGNAL(jab_contactRemove(JabSession *, JabRosterEntry *)),                     SLOT(jab_contactRemove(JabSession *, JabRosterEntry *)) );
	connect(&jsm, SIGNAL(jab_authRequest(JabSession *, const Jid &)),                            SLOT(jab_authRequest(JabSession *, const Jid &)) );
	connect(&jsm, SIGNAL(jab_authGrant(JabSession *, const Jid &)),                              SLOT(jab_authGrant(JabSession *, const Jid &)) );
	connect(&jsm, SIGNAL(jab_authRemove(JabSession *, const Jid &)),                             SLOT(jab_authRemove(JabSession *, const Jid &)) );

	// jabber
	/*connect(serv, SIGNAL(connected()),                         SLOT(servConnected()));
	connect(serv, SIGNAL(disconnected()),                      SLOT(servDisconnected()));
	connect(serv, SIGNAL(statusUpdate(JabUpdate *)),           SLOT(statusUpdate(JabUpdate *)) );
	connect(serv, SIGNAL(error(JabError *)),                   SLOT(servError(JabError *)) );
	connect(serv, SIGNAL(messageReceived(const JabMessage &)), SLOT(messageReceived(const JabMessage &)) );

	connect(serv, SIGNAL(resourceAvailable(const Jid &, const JabResource &)), SLOT(resourceAvailable(const Jid &, const JabResource &)));
	connect(serv, SIGNAL(resourceUnavailable(const Jid &)), SLOT(resourceUnavailable(const Jid &)));

	connect(serv, SIGNAL(contactChanged(JabRosterEntry *)),    SLOT(contactChanged(JabRosterEntry *)) );
	connect(serv, SIGNAL(contactNew(JabRosterEntry *)),        SLOT(contactNew(JabRosterEntry *)) );
	connect(serv, SIGNAL(contactRemove(JabRosterEntry *)),     SLOT(contactRemove(JabRosterEntry *)) );

	connect(serv, SIGNAL(authRequest(const Jid &)),            SLOT(authRequest(const Jid &)) );
	connect(serv, SIGNAL(authGrant(const Jid &)),              SLOT(authGrant(const Jid &)) );
	connect(serv, SIGNAL(authRemove(const Jid &)),             SLOT(authRemove(const Jid &)) );*/

	// actions
	connect(this, SIGNAL(emitOptionsUpdate()), mainwin->cvlist, SLOT(optionsUpdate()) );
	connect(this, SIGNAL(emitLocalUpdate(const JabRosterEntry &)), cvp, SLOT(localUpdate(const JabRosterEntry &)));
	connect(cvp, SIGNAL(actionDefault(ContactViewItem *)),SLOT(actionDefault(ContactViewItem *)) );
	connect(cvp, SIGNAL(actionRecvEvent(ContactViewItem *)),SLOT(actionRecvEvent(ContactViewItem *)) );
	connect(cvp, SIGNAL(actionSendMessage(ContactViewItem *)),SLOT(actionSendMessage(ContactViewItem *)) );
	connect(cvp, SIGNAL(actionSendMessage(const QStringList &)),SLOT(actionSendMessage(const QStringList &)) );
	connect(cvp, SIGNAL(actionSendUrl(const QString &)),SLOT(actionSendUrl(const QString &)) );
	connect(cvp, SIGNAL(actionAuthorize(ContactViewItem *)),SLOT(actionAuthorize(ContactViewItem *)) );
	connect(cvp, SIGNAL(actionAdd(ContactViewItem *)),SLOT(actionAdd(ContactViewItem *)) );
	connect(cvp, SIGNAL(actionRemove(const QString &)),SLOT(actionRemove(const QString &)) );
	connect(cvp, SIGNAL(actionRename(ContactViewItem *, const QString &)),SLOT(actionRename(ContactViewItem *, const QString &)) );
	connect(cvp, SIGNAL(actionGroupChange(const QString &, const QString &)),SLOT(actionGroupChange(const QString &, const QString &)) );
	connect(cvp, SIGNAL(actionGroupRename(const QString &, const QString &)),SLOT(actionGroupRename(const QString &, const QString &)) );
	connect(cvp, SIGNAL(actionHistory(const QString &)),SLOT(actionHistory(const QString &)) );
	connect(cvp, SIGNAL(actionStatusShow(const QString &)),SLOT(actionStatusShow(const QString &)) );
	connect(cvp, SIGNAL(actionOpenChat(const QString &)), SLOT(actionOpenChat(const QString &)) );
	connect(cvp, SIGNAL(actionAgentSetStatus(const QString &, int)), SLOT(agentSetStatus(const QString &, int)));
	connect(cvp, SIGNAL(actionInfo(const QString &)), SLOT(actionInfo(const QString &)));
	connect(cvp, SIGNAL(actionOfferFile(const QString &)), SLOT(actionOfferFile(const QString &)));
	connect(mainwin, SIGNAL(doOptions()), SLOT(doOptions()) );
	connect(mainwin, SIGNAL(doManageServices()), SLOT(doManageServices()) );
	connect(mainwin, SIGNAL(doManageAccounts()), SLOT(doManageAccounts()) );
	connect(mainwin, SIGNAL(doFileSharing()), SLOT(doFileSharing()) );
	connect(mainwin, SIGNAL(accountInfo()), SLOT(openAccountInfo()) );
	connect(mainwin, SIGNAL(blankMessage()), SLOT(composeBlankMessage()) );
	connect(mainwin, SIGNAL(statusChanged(int)),SLOT(statusMenuChanged(int)) );
	connect(mainwin, SIGNAL(closeProgram()), SLOT(closeProgram()));
	connect(mainwin, SIGNAL(changeProfile()), SLOT(changeProfile()));
	connect(mainwin, SIGNAL(addUser()), SLOT(askAddUser()));
	connect(mainwin, SIGNAL(recvNextEvent()), SLOT(recvNextEvent()));
	connect(this, SIGNAL(emitLocalUpdate(const JabRosterEntry &)), mainwin, SLOT(localUpdate(const JabRosterEntry &)));
	connect(this, SIGNAL(emitOptionsUpdate()), mainwin, SLOT(optionsUpdate()) );

	s->setContactProfile(cvp);
	s->setAccount(pro.acc.first());

	mainwin->cvlist->setShowOffline(s->acc()->tog_offline);
	mainwin->cvlist->setShowAway(s->acc()->tog_away);
	mainwin->cvlist->setShowAgents(s->acc()->tog_agents);

	mainwin->show();

	JabUpdate ju = s->serv()->getJU();
	jab_statusUpdate(s, &ju);

	emitLocalUpdate(*s->serv()->userEntry());

	idle.start();
	connect(&idle, SIGNAL(secondsIdle(int)), s, SLOT(secondsIdle(int)));

	// empty?
	if(s->acc()->host.isEmpty()) {
		jsm.modify(s);
	}
	else {
		if(s->acc()->opt_auto)
			statusMenuChanged(STATUS_ONLINE);
	}

	return TRUE;
}

jabcon::~jabcon()
{
	// serv not created yet?  don't bother destructing
	if(!s->serv())
		return;

	delete fileserv;

	//isDisconnecting = TRUE;
	///serv->disc();
	s->acc()->roster = *s->serv()->getCurrentRoster();
	s->acc()->olr_string = s->serv()->getOLR();

	mwgeom.setX(mainwin->x());
	mwgeom.setY(mainwin->y());
	mwgeom.setWidth(mainwin->width());
	mwgeom.setHeight(mainwin->height());
	s->acc()->tog_offline = mainwin->cvlist->isShowOffline();
	s->acc()->tog_away = mainwin->cvlist->isShowAway();
	s->acc()->tog_agents = mainwin->cvlist->isShowAgents();

	delete mainwin;

	// delete all toplevel windows except log window (and QDesktopWidget..)
	QWidgetList *list = QApplication::topLevelWidgets();
	QWidgetListIt it(*list);
	for(QWidget *w; (w = it.current()); ++it) {
		if(w->isA("LogWindow") && (LogWindow *)w == debug_window)
			continue;
		if(w->isA("QDesktopWidget") || w->isA("QWidget"))
			continue;

		w->deleteLater();
	}
	delete list;

	pro.mwgeom = mwgeom;
	pro.lastStatusString = lastStatusString;
	pro.useSound = useSound;

	pro.acc.first() = *s->acc();

	pro.prefs = option;
	pro.toFile(pathToProfileConfig(activeProfile));

	// delete sessions
	jsm.clear();

	unloadPsiIconSet();

	debug_jabber = 0;
}

void jabcon::jab_connected(JabSession *)
{
	mainwin->setUsingSSL(s->serv()->isSSLEnabled());
}

void jabcon::jab_disconnected(JabSession *)
{
}

void jabcon::jab_statusUpdate(JabSession *, JabUpdate *x)
{
	// make a status string
	QString str = "";
	if(x->queuePending > 0)
		str.append(QString("[%1] - ").arg(x->queuePending));
	str.append(x->str);

	// show status in the gui
	mainwin->setInfo(str);

	pdb(DEBUG_JABCON, QString("[jabcon: status update: %1]\n").arg(str));
}

void jabcon::jab_error(JabSession *, JabError *err)
{
	if(err->type == JABERR_CONNECT) {
		QMessageBox::critical(0, CAP(tr("Error")), QString(tr("There was an error communicating with the Jabber server.\nReason: %1")).arg(err->msg));
		//delayedAskLogin();
	}
	else if(err->type == JABERR_AUTH) {
		QMessageBox::critical(0, CAP(tr("Error")), QString(tr("Authorization failed.\nReason: %1")).arg(err->msg));
		//delayedAskLogin();
	}
	else if(err->type == JABERR_CREATE) {
		QMessageBox::critical(0, CAP(tr("Error")), QString(tr("Failed to create the new account.\nThe server gave this reason: \"%1\"")).arg(err->msg));
		//delayedAskLogin();
	}
	else if(err->type == JABERR_DISCONNECT) {
		QMessageBox::critical(0, CAP(tr("Error")), tr("You have been disconnected from the Jabber server."));
	}
	else {
		QMessageBox::critical(0, CAP(tr("Error")), QString(tr("An unknown error occurred.\nType: %1\nReason: %2")).arg(err->type).arg(err->msg));
	}
}

void jabcon::jab_messageReceived(JabSession *, const JabMessage &_msg)
{
	// skip headlines?
	if(_msg.type == JABMESSAGE_HEADLINE && option.ignoreHeadline)
		return;

	Message msg;
	msg.from = _msg.from.full();
	if(_msg.type == JABMESSAGE_CHAT)
		msg.type = MESSAGE_CHAT;
	else if(_msg.type == JABMESSAGE_ERROR)
		msg.type = MESSAGE_ERROR;
	else if(_msg.type == JABMESSAGE_HEADLINE)
		msg.type = MESSAGE_HEADLINE;
	else
		msg.type = MESSAGE_NORM;
	msg.text = _msg.body;
	msg.subject = _msg.subject;
	msg.url = _msg.url;
	msg.url_desc = _msg.url_desc;
	msg.timeStamp = _msg.timeStamp;
	msg.late = _msg.late;
	msg.originLocal = FALSE;

	pdb(DEBUG_JABCON, QString("[jabcon: message received from %1]\n").arg(msg.from));


	if(option.incomingAs == 1)
		msg.type = MESSAGE_NORM;
	else if(option.incomingAs == 2)
		msg.type = MESSAGE_CHAT;

	if(msg.type == MESSAGE_CHAT) {
		ChatDlg *c = ChatDlg::find(msg.from);
		if(c && !c->isHidden()) {
			c->incomingMessage(msg);
			doOnEvent(option.onevent[eChat2]);

			// since it never goes into the queue in this case, we need to log here
			if(s->acc()->opt_log)
				logMessage(msg);
			return;
		}
	}

	//msg.unread = TRUE;
	if(msg.type == MESSAGE_ERROR) {
        	msg.text = QString(tr("<big>[Error Message]</big><br>%1").arg(plain2rich(msg.text)));
	}

	insertMessage(msg);
}

void jabcon::jab_resourceAvailable(JabSession *s, const Jid &j, const JabResource &r)
{
	QString dstr = QString("[jabcon: resource available: jid=[%1], name=[%2], status=[%3] ]\n").arg(j.full()).arg(r.name).arg(status2txt(r.status));
	pdb(DEBUG_JABCON, dstr);

	JabRosterEntry *me = s->serv()->userEntry();
	if(jidcmp(me->jid, j.s())) {
		int oldStatus = s->localStatus;

		s->localStatus = me->local()->status;
		s->localStatusString = me->local()->statusString;

		pdb(DEBUG_JABCON, QString("presence: [%1]  total=%2\n").arg(status2txt(s->localStatus)).arg(me->res.count()));

		mainwin->decorateButton(s->localStatus);
		emitLocalUpdate(*me);

		// coming from offline to online
		if(oldStatus == STATUS_OFFLINE && s->localStatus != STATUS_OFFLINE) {
			/*if(doNewUser) {
				QMessageBox::information(0, CAP(tr("Success")), tr("New account was created and logged in successfully.  Enjoy."));
				doNewUser = FALSE;

				// allow the new user to set account info
				openAccountInfo();
			}*/

			QTimer::singleShot(15000, this, SLOT(enableOnEventOnline()));

			JT_VCard *j = new JT_VCard(s->serv()->ioUser());
			connect(j, SIGNAL(finished(JabTask *)), SLOT(slotCheckVCard(JabTask *)));
			j->get(s->acc()->jid());
			j->go();
		}
	}
	else {
		UserListItem *item = s->userlist.findAnyByJid(j.s());
		if(!item)
			return;

		// if the contact was offline before this, then do things
		bool doThings = FALSE;
		if(item->state == STATUS_OFFLINE && s->onEventOnlineOk)
			doThings = TRUE;

		//item->state = r.status;
		//item->statusString = r.statusString;

		// add/update resource
		JabResource *r2 = item->res.find(r.name);
		if(!r2) {
			r2 = new JabResource(r);
			item->res.append(r2);
		}
		else {
			*r2 = r;
		}

		// contact view should reflect the priority status
		item->state = item->res.priority()->status;
		item->statusString = item->res.priority()->statusString;

		// update gui
		cvp->updateEntry(*item);
		//mainwin->cvlist->updateEntry(item->jid, jidnick(item->jid, item->nick), item->group, item->sub, item->state, item->statusString, item->inList, item->isTransport, item->res);

		if(doThings) {
			doOnEvent(option.onevent[eOnline]);

			if(option.rosterAnim)
				cvp->animateNick(item->jid);
		}

		// tell the world
		emitContact(item);
	}
}

void jabcon::jab_resourceUnavailable(JabSession *s, const Jid &j)
{
	QString dstr = QString("[jabcon: resource unavailable: jid=[%1], name=[%2] ]\n").arg(j.s()).arg(j.resource());
	pdb(DEBUG_JABCON, dstr);

	JabRosterEntry *me = s->serv()->userEntry();
	if(jidcmp(me->jid, j.s())) {
		// only go offline if the local resource goes offline
		if(!me->local()) {
			s->localStatus = STATUS_OFFLINE;
			s->localStatusString = "";

			pdb(DEBUG_JABCON, QString("presence: [%1]  total=%2\n").arg(status2txt(s->localStatus)).arg(me->res.count()));

			mainwin->setUsingSSL(FALSE);

			mainwin->decorateButton(s->localStatus);
			emitLocalUpdate(*me);

			// restore original account and delete contacts if guestMode
			/*if(guestMode) {
				pdb(DEBUG_JABCON, "Return from GUESTMODE\n");
				acc = oldacc;
				guestMode = FALSE;

				s->serv()->reset();
				cvp->clear();
			}*/
		}
		else
			emitLocalUpdate(*me);
	}
	else {
		JabRosterEntry *entry = s->serv()->findByJid(j.s());
		if(!entry)
			return;
		UserListItem *item = s->userlist.findAnyByJid(j.s());
		if(!item)
			return;

		// if the contact was not offline before this, then sound
		if(item->state != STATUS_OFFLINE && !s->isDisconnecting)
			doOnEvent(option.onevent[eOffline]);

		// still available?  use next highest resource
		if(entry->isAvailable()) {
			JabResource *r = entry->res.priority();
			item->state = r->status;
			item->statusString = r->statusString;
		}
		// not available
		else {
			item->state = STATUS_OFFLINE;
			item->statusString = entry->unavailableStatusString;
		}

		// remove resource
		JabResource *r = item->res.find(j.resource());
		if(r)
			item->res.remove(r);

		// update gui
		cvp->updateEntry(*item);
		//mainwin->cvlist->updateEntry(item->jid, jidnick(item->jid, item->nick), item->group, item->sub, item->state, item->statusString, item->inList, item->isTransport, item->res);

		// tell the world
		emitContact(item);
	}
}

void jabcon::jab_contactChanged(JabSession *, JabRosterEntry *entry)
{
	QString dstr = QString("[jabcon: roster changed: %1]\n").arg(entry->jid);
	pdb(DEBUG_JABCON, dstr);

	// this should never fail...
	UserListItem *item = s->userlist.findAnyByJid(cleanJid(entry->jid));
	if(!item)
		return;

	// copy into userlist
	item->nick = entry->nick;
	if(entry->groups.isEmpty())
		item->group = "";
	else
		item->group = entry->groups[0];
	item->sub = entry->sub;

	QString nick = item->nick.isEmpty() ? item->jid: item->nick;

	// update gui
	cvp->updateEntry(*item);
	//mainwin->cvlist->updateEntry(item->jid, nick, item->group, item->sub, item->state, item->statusString, item->inList, item->isTransport, item->res);

	// tell the world
	emitContact(item);
}

void jabcon::jab_contactNew(JabSession *, JabRosterEntry *entry)
{
	// we probably already have the entry in our list somewhere,
	//  because of the authorization transaction
	UserListItem *item = s->userlist.findAnyByJid(cleanJid(entry->jid));

	if(item) {
		item->inList = TRUE;
		cvp->updateEntry(*item);
		//mainwin->cvlist->updateEntry(item->jid, jidnick(item->jid, item->nick), item->group, item->sub, item->state, item->statusString, item->inList, item->isTransport, item->res);
	}
	else {
		// add to the userlist
		UserListItem *item = new UserListItem;
		item->jid = cleanJid(entry->jid);
		item->nick = entry->nick;
		if(entry->groups.isEmpty())
			item->group = "";
		else
			item->group = entry->groups[0];
		item->sub = entry->sub;

		item->state = STATUS_OFFLINE;
		item->isTransport = jidIsAgent(item->jid);

		// a new entry that is "push"?  [pushinfo]
		/*if(entry->push) {
			VCard info;
			if(readUserInfo(entry->jid, &info) && !info.field[vNickname].isEmpty()) {
				item->nick = info.field[vNickname];
			}
			else {
				if(localStatus != STATUS_OFFLINE)
					serv->getVCard(item->jid);
			}
		}*/

		s->userlist.add(item);

		// add to the contact gui
		cvp->updateEntry(*item);
		//mainwin->cvlist->updateEntry(item->jid, jidnick(item->jid, item->nick), item->group, item->sub, item->state, item->statusString, item->inList, item->isTransport, item->res);

		// search for unread messages and queue them
		//queueUnread(item);
	}

	pdb(DEBUG_JABCON, QString("[jabcon: %1 has been added to the internal contact list]\n").arg(entry->jid));
}

void jabcon::jab_contactRemove(JabSession *, JabRosterEntry *entry)
{
	UserListItem *item = s->userlist.findAnyByJid(cleanJid(entry->jid));
	if(!item)
		return;

	// remove all events from the queue (and tell mainwin)
	events.removeAll(item->jid);
	int next_type = 0;
	int next_amount = events.QPtrList<EventItem>::count();
	if(next_amount > 0)
		next_type = events.peekNext()->msg.type;
	mainwin->updateReadNext(next_type, next_amount);

	// remove from the contact gui
	cvp->removeEntry(item->jid);

	// remove from the userlist
	s->userlist.remove(item->jid);

	pdb(DEBUG_JABCON, QString("[jabcon: %1 has been removed from the internal contact list]\n").arg(entry->jid));
}

void jabcon::jab_authRequest(JabSession *, const Jid &from)
{
	QString body(tr("<big>[System Message]</big><br>This user wants to subscribe to your presence.  Click the button labelled \"Add/Auth\" to authorize the subscription.  This will also add the person to your contact list if it is not already there."));

	Message tmp;
	tmp.from = from.full();
	tmp.type = MESSAGE_AUTHREQ;
	tmp.text = body;
	tmp.timeStamp = QDateTime::currentDateTime();
	tmp.late = FALSE;
	tmp.originLocal = FALSE;
	insertMessage(tmp);

	pdb(DEBUG_JABCON, QString("[jabcon: %1 wants to add you to their list]\n").arg(from.full()));
}

void jabcon::jab_authGrant(JabSession *, const Jid &from)
{
	QString body(tr("<big>[System Message]</big><br>You are now authorized."));

	Message tmp;
	tmp.from = from.full();
	tmp.type = MESSAGE_SYS;
	tmp.text = body;
	tmp.timeStamp = QDateTime::currentDateTime();
	tmp.late = FALSE;
	tmp.originLocal = FALSE;
	insertMessage(tmp);

	pdb(DEBUG_JABCON, QString("[jabcon: %1 has authorized you]\n").arg(from.full()));
}

void jabcon::jab_authRemove(JabSession *, const Jid &from)
{
	// only send a sysmessage if it's done by the other person
	//  (because we get this signal when we remove contacts too.  I think...)
	if(s->userlist.findAnyByJid(from.s())) {
		QString body(tr("<big>[System Message]</big><br>Your authorization has been removed!"));

		Message tmp;
		tmp.from = from.full();
		tmp.type = MESSAGE_SYS;
		tmp.text = body;
		tmp.timeStamp = QDateTime::currentDateTime();
		tmp.late = FALSE;
		tmp.originLocal = FALSE;
		insertMessage(tmp);
	}

	pdb(DEBUG_JABCON, QString("[jabcon: %1 has removed you from their contact list]\n").arg(from.full()));
}



/*void jabcon::servConnected()
{
	mainwin->setUsingSSL(serv->isSSLEnabled());
}

void jabcon::servDisconnected()
{
	isDisconnecting = TRUE;
}

void jabcon::statusUpdate(JabUpdate *x)
{
	// make a status string
	QString str = "";
	if(x->queuePending > 0)
		str.append(QString("[%1] - ").arg(x->queuePending));
	str.append(x->str);

	// show status in the gui
	mainwin->setInfo(str);

	pdb(DEBUG_JABCON, QString("[jabcon: status update: %1]\n").arg(str));
}
*/

void jabcon::insertMessage(const Message &msg, bool doLog)
{
	// sound
	if(msg.type == MESSAGE_CHAT) {
		// any pending chats from this person?
		bool found = FALSE;
		UserListItem *i = s->userlist.findAnyByJid(msg.from);
		if(i)
			found = events.hasChats(i->jid);

		doOnEvent(option.onevent[found ? eChat2: eChat1]);
	}
	else if(msg.type == MESSAGE_NORM)
		doOnEvent(option.onevent[eMessage]);
	else if(msg.type == MESSAGE_HEADLINE)
		doOnEvent(option.onevent[eMessage]);
	else
		doOnEvent(option.onevent[eSystem]);

	queueMessage(msg);

	if(doLog) {
		if(s->acc()->opt_log)
			logMessage(msg);
	}

	// cvlist alert
	cvp->showAlert(cleanJid(msg.from), msg.type);

	// update mainwin status
	mainwin->updateReadNext(events.peekNext()->msg.type, events.QPtrList<EventItem>::count());

	// messagebox open?
	EventDlg *e = EventDlg::find(msg.from);
	if(e) {
		UserListItem *item = s->userlist.findAnyByJid(msg.from);
		e->updateReadNext(events.peek(item->jid)->msg.type, events.count(item->jid));
	}

	if((msg.type != MESSAGE_CHAT && option.popupMsgs) ||
	   (msg.type == MESSAGE_CHAT && option.popupChats)) {
		UserListItem *i = s->userlist.findAnyByJid(msg.from);
		if(i) {
			readMessage(i);
		}
	}
	if(option.raise) {
//#ifdef Q_WS_WIN
//		win_raiseNoFocus(mainwin);
//#else
		mainwin->showNormal();
		mainwin->raise();
//#endif
	}
}

void jabcon::actionDefault(ContactViewItem *item)
{
        UserListItem *i = s->userlist.findAnyByJid(item->jid());
        if(!i)
                return;

        //c->erasePrompt();
        QString dstr; dstr.sprintf("[jabcon: default action: %s]\n", i->jid.latin1());
        pdb(DEBUG_JABCON, dstr);

        //c->reshowPrompt();

        // pending messages from this person?
        if(events.count(i->jid) > 0) {
                readMessage(i);
        }
        else {
		if(option.defaultAction == 0)
                	composeMessage(i, FALSE);
		else
			openChat(i);
        }
}

void jabcon::actionSendMessage(ContactViewItem *item)
{
        UserListItem *i = s->userlist.findAnyByJid(item->jid());
        if(!i)
                return;
        composeMessage(i, FALSE);
}

void jabcon::actionSendMessage(const QStringList &list)
{
        EventDlg *c = new EventDlg(list, s->localStatus);
        connect(c, SIGNAL(aSend(const Message &)), SLOT(sendMessage(const Message &)));
	connect(c, SIGNAL(aOpenURL(const QString &)), SLOT(actionOpenURL(const QString &)));
        connect(this, SIGNAL(emitLocalUpdate(const JabRosterEntry &)), c, SLOT(localUpdate(const JabRosterEntry &)));
        connect(this, SIGNAL(emitOptionsUpdate()), c, SLOT(optionsUpdate()));

        c->show();
}

void jabcon::actionSendUrl(const QString &jid)
{
        UserListItem *i = s->userlist.findAnyByJid(jid);
        if(!i)
                return;
        composeMessage(i, TRUE);
}

void jabcon::actionRecvEvent(ContactViewItem *item)
{
        UserListItem *i = s->userlist.findAnyByJid(item->jid());
        if(!i)
                return;

	if(events.count(i->jid) > 0)
        	readMessage(i);
}

void jabcon::actionRemove(const QString &jid)
{
        UserListItem *i = s->userlist.findAnyByJid(jid);
        if(!i)
                return;

        userRemove(i);
}

void jabcon::actionRename(ContactViewItem *item, const QString &newname)
{
        UserListItem *i = s->userlist.findAnyByJid(item->jid());
        if(!i) {
                return;
        }

        userRename(i, newname);
}

void jabcon::actionGroupChange(const QString &jid, const QString &groupname)
{
        UserListItem *i = s->userlist.findAnyByJid(jid);
        if(!i) {
                return;
        }
        if(!i->inList) {
                return;
        }

        userGroupChange(i, groupname);
}

void jabcon::actionGroupRename(const QString &oldname, const QString &groupname)
{
        // change all users in this group to use the new name
        UserListItem *i = s->userlist.start(USERLIST_CONTACTS);
        for(; i ; i = s->userlist.next()) {
                if(oldname == i->group)
                        userGroupChange(i, groupname);
        }
        s->userlist.end();
}

void jabcon::actionHistory(const QString &jid)
{
        UserListItem *i = s->userlist.findAnyByJid(jid);
        if(!i)
                return;

        // search for an already opened historydlg
        HistoryDlg *hist = HistoryDlg::find(i->jid);

        if(hist) {
                bringToFront(hist);
        }
        else {
                hist = new HistoryDlg(i, 0, "HistoryDlg");
                connect(hist, SIGNAL(openMessage(const Message &)), SLOT(actionHistoryBox(const Message &)));
                hist->show();
        }
}

// quick and dirty
void jabcon::actionHistoryBox(const Message &msg)
{
	UserListItem tmp;

	tmp.jid = msg.from;

	// create the messagebox
	EventDlg *e = new EventDlg(&tmp, s->localStatus, FALSE, FALSE);
	connect(e, SIGNAL(aReply(const QString &)), SLOT(composeMessage(const QString &)));
	connect(e, SIGNAL(aReadNext(const QString &)), SLOT(processReadNext(const QString &)));
	connect(e, SIGNAL(aAuth(const QString &)), SLOT(userAddAuth(const QString &)));
	connect(e, SIGNAL(aInfo(const QString &)), SLOT(actionInfo(const QString &)));
	connect(e, SIGNAL(aHistory(const QString &)), SLOT(actionHistory(const QString &)));
	connect(e, SIGNAL(aOpenURL(const QString &)), SLOT(actionOpenURL(const QString &)));
	connect(this, SIGNAL(emitContact(UserListItem *)), e, SLOT(updateContact(UserListItem *)));
	connect(this, SIGNAL(emitLocalUpdate(const JabRosterEntry &)), e, SLOT(localUpdate(const JabRosterEntry &)));
	connect(this, SIGNAL(emitOptionsUpdate()), e, SLOT(optionsUpdate()));

	// load the message
	//processReadNext(item);
	e->updateEvent(msg);

	e->show();
}

void jabcon::actionStatusShow(const QString &jid)
{
        UserListItem *i = s->userlist.findAnyByJid(jid);
        if(!i)
                return;

        StatusShowDlg *w = new StatusShowDlg(i);
        w->show();
}

void jabcon::actionAuthorize(ContactViewItem *item)
{
        userAuthorize(item->jid());
}

void jabcon::actionAdd(ContactViewItem *item)
{
        userAdd(item->jid());
}

void jabcon::actionOpenChat(const QString &jid)
{
	UserListItem *item = s->userlist.findAnyByJid(jid);
	if(!item)
		return;

	openChat(item);
}

void jabcon::actionInfo(const QString &jid)
{
	InfoDlg *w = InfoDlg::find(jid);
	if(w) {
		bringToFront(w);
	}
	else {
		Info *p = InfoBank::get(jid);
		VCard tmp;
		if(p)
			tmp = p->vcard;

		w = new InfoDlg(INFODLG_CONTACT, jid, tmp, s->localStatus, 0);
		connect(w, SIGNAL(signalGetVCard(const QString &, QString *)), SLOT(slotGetVCard(const QString &, QString *)));
		connect(w, SIGNAL(signalCancelTransaction(const QString &)), SLOT(slotCancelTransaction(const QString &)));
		connect(this, SIGNAL(emitLocalUpdate(const JabRosterEntry &)), w, SLOT(localUpdate(const JabRosterEntry &)));
		w->show();

		// automatically retrieve info if it doesn't exist
		if(!p)
			w->doRefresh();
	}
}


ChatDlg *jabcon::ensureChatDlg(const QString &jid)
{
	UserListItem *item = s->userlist.findAnyByJid(jid);
	if(!item)
		return 0;

	ChatDlg *c = ChatDlg::find(jid);
	if(!c) {
		// create the chatbox
		c = new ChatDlg(item, s->acc()->user, s->localStatus);
		connect(c, SIGNAL(aSend(const Message &)), SLOT(sendMessage(const Message &)));
		connect(c, SIGNAL(aInfo(const QString &)), SLOT(actionInfo(const QString &)));
		connect(c, SIGNAL(aHistory(const QString &)), SLOT(actionHistory(const QString &)));
		connect(c, SIGNAL(aOpenURL(const QString &)), SLOT(actionOpenURL(const QString &)));
		connect(this, SIGNAL(emitContact(UserListItem *)), c, SLOT(updateContact(UserListItem *)));
		connect(this, SIGNAL(emitLocalUpdate(const JabRosterEntry &)), c, SLOT(localUpdate(const JabRosterEntry &)));
		connect(this, SIGNAL(emitOptionsUpdate()), c, SLOT(optionsUpdate()));
	}

	return c;
}

void jabcon::openChat(UserListItem *item)
{
	ChatDlg *c = ensureChatDlg(item->jid);
	if(!c)
		return; // not quite sure how this would happen

	processChats(item);

	if(c->isHidden())
		c->show();

	bringToFront(c);
}

void jabcon::readMessage(UserListItem *item)
{
        Message *m = &events.peek(item->jid)->msg;
	if(m->type == MESSAGE_CHAT) {
		openChat(item);
	}
	else {
		// search for an already opened messagebox
		EventDlg *e = EventDlg::find(item->jid);

		if(e) {
			bringToFront(e);
		}
		else {
			// create the messagebox
			e = new EventDlg(item, s->localStatus, TRUE, FALSE);
			connect(e, SIGNAL(aReply(const QString &)), SLOT(composeMessage(const QString &)));
			connect(e, SIGNAL(aReadNext(const QString &)), SLOT(processReadNext(const QString &)));
			connect(e, SIGNAL(aAuth(const QString &)), SLOT(userAddAuth(const QString &)));
		        connect(e, SIGNAL(aInfo(const QString &)), SLOT(actionInfo(const QString &)));
			connect(e, SIGNAL(aHistory(const QString &)), SLOT(actionHistory(const QString &)));
			connect(e, SIGNAL(aOpenURL(const QString &)), SLOT(actionOpenURL(const QString &)));

			connect(this, SIGNAL(emitContact(UserListItem *)), e, SLOT(updateContact(UserListItem *)));
			connect(this, SIGNAL(emitLocalUpdate(const JabRosterEntry &)), e, SLOT(localUpdate(const JabRosterEntry &)));
			connect(this, SIGNAL(emitOptionsUpdate()), e, SLOT(optionsUpdate()));

			// load the message
			processReadNext(item);

			e->show();
			bringToFront(e);
		}
	}
}

void jabcon::composeMessage(const QString &jid)
{
        UserListItem *i = s->userlist.findAnyByJid(jid);
        if(!i) {
		QStringList tmp;
		tmp += jid;
		actionSendMessage(tmp);
                return;
	}
        composeMessage(i, FALSE);
}

void jabcon::composeMessage(UserListItem *item, bool urlMode)
{
        EventDlg *c = new EventDlg(item, s->localStatus, FALSE, TRUE, urlMode);
        connect(c, SIGNAL(aSend(const Message &)), SLOT(sendMessage(const Message &)));
        connect(c, SIGNAL(aInfo(const QString &)), SLOT(actionInfo(const QString &)));
        connect(c, SIGNAL(aHistory(const QString &)), SLOT(actionHistory(const QString &)));
	connect(c, SIGNAL(aOpenURL(const QString &)), SLOT(actionOpenURL(const QString &)));
        connect(this, SIGNAL(emitContact(UserListItem *)), c, SLOT(updateContact(UserListItem *)));
        connect(this, SIGNAL(emitLocalUpdate(const JabRosterEntry &)), c, SLOT(localUpdate(const JabRosterEntry &)));
        connect(this, SIGNAL(emitOptionsUpdate()), c, SLOT(optionsUpdate()));

        c->show();
}

void jabcon::composeBlankMessage()
{
	QStringList to;

        EventDlg *c = new EventDlg(to, s->localStatus);
        connect(c, SIGNAL(aSend(const Message &)), SLOT(sendMessage(const Message &)));
	connect(c, SIGNAL(aOpenURL(const QString &)), SLOT(actionOpenURL(const QString &)));
        connect(this, SIGNAL(emitLocalUpdate(const JabRosterEntry &)), c, SLOT(localUpdate(const JabRosterEntry &)));
        connect(this, SIGNAL(emitOptionsUpdate()), c, SLOT(optionsUpdate()));

        c->show();
}

void jabcon::sendMessage(const Message &msg)
{
	JabMessage tmp;
	tmp.to = msg.to;
	if(msg.type == MESSAGE_CHAT)
		tmp.type = JABMESSAGE_CHAT;
	else
		tmp.type = JABMESSAGE_NORM;
	tmp.body = msg.text;
	tmp.subject = msg.subject;
	tmp.url = msg.url;
	tmp.url_desc = msg.url_desc;

        s->serv()->sendMessage(tmp);

        QString dstr; dstr.sprintf("jabcon: message sent to %s.\n", msg.to.latin1());
        pdb(DEBUG_JABCON, dstr);

	if(s->acc()->opt_log) {
        	MessageHistory log(msg.to, HISTORY_WRITE);
        	log.writeEntry(msg);
	}

	doOnEvent(option.onevent[eSend]);

        // auto close an open messagebox (if non-chat)
	if(msg.type != MESSAGE_CHAT) {
        	EventDlg *e = EventDlg::find(msg.to);
        	if(e) {
                	e->closeAfterReply();
        	}
	}
}

void jabcon::processReadNext(const QString &jid)
{
        UserListItem *i = s->userlist.findAnyByJid(jid);
        if(!i)
                return;
        processReadNext(i);
}

void jabcon::processReadNext(UserListItem *item)
{
	EventDlg *e = EventDlg::find(item->jid);
	if(!e) {
		// this should NEVER happen
		return;
	}

	// look at the message
	Message *m = &events.peek(item->jid)->msg;

	// if it's a chat message, just open the chat window.  there is no need to do
	// further processing.  the chat window will remove it from the queue, de-alert
	// the cvlist, etc etc.
	if(m->type == MESSAGE_CHAT) {
		openChat(item);
		return;
	}

	// remove from queue
	EventItem *ei = events.dequeue(item->jid);
	//flagAsRead(ei->msg);


	// update the eventdlg
	e->updateEvent(ei->msg);
	delete ei;

	// update eventdlg's read-next
	int nexttype = 0;
	if(events.count(item->jid) > 0)
		nexttype = events.peek(item->jid)->msg.type;
	e->updateReadNext(nexttype, events.count(item->jid));

	// clear the cvlist alert
	cvp->clearAlert(item->jid);

	// mainwin status
	int next_type = 0;
	int next_amount = events.QPtrList<EventItem>::count();
	if(next_amount > 0)
		next_type = events.peekNext()->msg.type;
	mainwin->updateReadNext(next_type, next_amount);
}

void jabcon::processChats(const QString &jid)
{
        UserListItem *i = s->userlist.findAnyByJid(jid);
        if(!i)
                return;
        processChats(i);
}

void jabcon::processChats(UserListItem *item)
{
        ChatDlg *c = ChatDlg::find(item->jid);
        if(!c) {
                // this should NEVER happen
                return;
        }
	pdb(DEBUG_JABCON, "jabcon: processChats: processing chats.\n");

	// extract the chats and position list
	QPtrList<Message> chatList;
	QValueList<int> pos;
	events.extractChats(item->jid, &chatList, &pos);

	// dump the chats into the chat window, and remove the related cvlist alerts
	chatList.setAutoDelete(TRUE);
	QValueList<int>::Iterator it = pos.begin();
	int modifier = 0;
	for(Message *m = chatList.last(); m;) {
		// process the message
		//flagAsRead(*m);
        	c->incomingMessage(*m);

		// delete it
		chatList.remove();

		// deleting an item at the end means we need to move up to the new "last"
		m = chatList.last();

		// clear the alert
		int x = (*it) - modifier;
		//printf("deleting entry %d (pos=%d,mod=%d)\n", x, (*it), modifier);
		cvp->clearAlert(item->jid, x);
		++it;
		++modifier;
	}

	// if there is an associated eventdlg, make sure that gets updated also
	EventDlg *e = EventDlg::find(item->jid);
	if(e) {
		int nexttype = 0;
		if(events.count(item->jid) > 0)
			nexttype = events.peek(item->jid)->msg.type;

		e->updateReadNext(nexttype, events.count(item->jid));
	}

	// mainwin status
	int next_type = 0;
	int next_amount = events.QPtrList<EventItem>::count();
	if(next_amount > 0)
		next_type = events.peekNext()->msg.type;
	mainwin->updateReadNext(next_type, next_amount);
}

void jabcon::userRemove(UserListItem *i)
{
        if(!i->inList)
                return;

        s->serv()->unsubscribe(i->jid);
}

void jabcon::userRename(UserListItem *i, const QString &newname)
{
	if(i->nick == newname)
		return;

        // setting to jidname?  remove nick
        if(newname == i->jid)
                i->nick = "";
        else
                i->nick = newname;

        emitContact(i);

        if(!i->inList)
                return;

        s->serv()->setRoster(i->jid, i->nick, i->group);
}

void jabcon::userGroupChange(UserListItem *i, const QString &groupname)
{
        i->group = groupname;

        s->serv()->setRoster(i->jid, i->nick, i->group);
}


/****************************************************************************
  User functions
****************************************************************************/
void jabcon::userAdd(const QString &jid)
{
	// the contact should be in the local list someplace
	UserListItem *i = s->userlist.findAnyByJid(jid);
	if(!i)
		return;

	// put the user in our contact list (don't do this if it's already there)
	if(!s->serv()->findByJid(i->jid))
		s->serv()->setRoster(i->jid, i->nick, i->group);

	// request subscription
	s->serv()->subscribe(i->jid);
}

void jabcon::userAuthorize(const QString &jid)
{
	s->serv()->subscribed(jid);
}

void jabcon::userAddAuth(const QString &jid)
{
	// the contact should be in the local list someplace
	UserListItem *i = s->userlist.findAnyByJid(jid);
	if(!i)
		return;

	// authorize using authoritative jid (argument) instead of local jid (userlist)
	userAuthorize(jid);

	// only add if it's _not_ a transport.
	// this is a workaround for the AIM transport, which has status change
	//  problems if fully subscribed.
	if(!i->isTransport)
		userAdd(jid);
}



void jabcon::actionAdd(const QString &jid, const QString &nick, const QString &group)
{
        s->serv()->subscribe(jid);

        // skip setting the roster if it's already in the server contact list
        if(s->serv()->findByJid(jid))
                return;

        s->serv()->setRoster(jid, nick, group);
}

void jabcon::agentSetStatus(const QString &jid, int status)
{
        UserListItem *i = s->userlist.findServiceByJid(jid);
        if(!i)
                return;

        s->serv()->agentSetStatus(jid, status);
}

void jabcon::agentRemove(const QString &jid)
{
        UserListItem *i = s->userlist.findServiceByJid(jid);
        if(!i)
                return;

        s->serv()->unsubscribe(i->jid);
        //userRemove(i);        // FIXME: i say this whole jabcon class needs help
}


void jabcon::queueMessage(const Message &msg)
{
	UserListItem *item = s->userlist.findAnyByJid(msg.from);

	// if not in list, add a temporary entry
	if(!item) {
		// add to the userlist
		item = new UserListItem;
		item->jid = cleanJid(msg.from);
		item->nick = "";
		item->group = "";
		item->sub = "none";
		item->state = STATUS_OFFLINE;
		item->statusString = "";
		item->inList = FALSE;
		item->isTransport = jidIsAgent(item->jid);

		// treat it like a push  [pushinfo]
		/*VCard info;
		if(readUserInfo(item->jid, &info) && !info.field[vNickname].isEmpty())
			item->nick = info.field[vNickname];
		else {
			if(localStatus != STATUS_OFFLINE)
				serv->getVCard(item->jid);
		}*/

		s->userlist.add(item);
	}

	// make sure the contact is displayed
	QString nick = item->nick.isEmpty() ? item->jid: item->nick;
	cvp->updateEntry(*item);
	//mainwin->cvlist->updateEntry(item->jid, nick, item->group, item->sub, item->state, item->statusString, item->inList, item->isTransport, item->res);

	EventItem *ei = new EventItem;
	ei->jid = item->jid;
	ei->msg = msg;
	events.enqueue(ei);

	QString dstr; dstr.sprintf("jabcon: queuing a message from %s.\n", msg.from.latin1());
	pdb(DEBUG_JABCON, dstr);
}

void jabcon::logMessage(const Message &msg)
{
	Message m(msg);
	MessageHistory log(m.from, HISTORY_WRITE);

	// use a short message for AUTHREQ.   FIXME: this is not a good way to do it
	if(msg.type == MESSAGE_AUTHREQ)
		m.text = QString(tr("<big>[System Message]</big><br>This user wants to subscribe to your presence."));

	log.writeEntry(m);
}

void jabcon::queueUnread(UserListItem *item)
{
	MessageHistory log(item->jid, HISTORY_FLAG);
	Message *msg;
	bool inZone = FALSE;

	// collect unread received messages into a QPtrList, save in reverse order
	QPtrList<Message> unreadList;
	unreadList.setAutoDelete(TRUE);
	while(1) {
		msg = log.readEntry();
		if(!msg)
			break;

		if(!inZone) {
			// skip sent messages
			if(msg->originLocal) {
				delete msg;
				continue;
			}
		}

		if(!msg->unread) {
			delete msg;
			break;
		}

		unreadList.insert(0, msg);
		inZone = TRUE;
	}

	// empty?  bail
	if(unreadList.isEmpty())
		return;

	// now put them in the queue. don't log history
	QPtrListIterator<Message> it(unreadList);
	for(; (msg = it.current()); ++it)
		insertMessage(*msg, FALSE);

	unreadList.clear();
}

void jabcon::flagAsRead(const Message &msg)
{
	// messages must be read in sequence, so this message must be earliest.
	// scan backwards until we find the last unread message.

	MessageHistory log(msg.from, HISTORY_FLAG);
	Message *m;
	bool inZone = FALSE;
	int num = 0;

	while(1) {
		//printf("reading entry\n");
		m = log.readCurrent();
		if(!m)
			break;
		++num;
		//printf("read entry\n");

		if(!inZone) {
			// skip sent messages
			if(m->originLocal) {
				delete m;
				//printf("local message\n");
				log.stepForward();
				continue;
			}
		}

		if(!m->unread) {
			delete m;
			//printf("read message\n");
			break;
		}

		delete m;
		//printf("unread message, going forward\n");
		log.stepForward();
		inZone = TRUE;
	}

	//printf("num = %d\n", num);

	// no history...
	if(num == 0)
		return;

	// go back one message
	//printf("stepping back\n");
	log.stepBack();
	log.setFlagsCurrent("**-*");
}

// mainwin triggers this when the user alters the status button
void jabcon::statusMenuChanged(int x)
{
	// setting to some type of online status
	if(x != STATUS_OFFLINE) {
		StatusInfo info;
		info.type = x;
		bool ok = FALSE;

		if(x == STATUS_ONLINE && !option.askOnline) {
			info.str = "";
			ok = TRUE;
		}
		else {
			if(lastStatusString.isEmpty())
				info.str = tr("I am away from my computer.  Please leave a message.");
			else
				info.str = lastStatusString; // default to old away message

			if(StatusSetDlg::getStatus(&info)) {
				ok = TRUE;
				lastStatusString = info.str;
			}
		}

		if(ok)
			s->setStatus(info);
	}
	// setting to offline
	else {
		s->disc();
		mainwin->setUsingSSL(FALSE);
	}
}

/*void jabcon::conn()
{
	if(!s->serv()->isActive()) {
		s->isDisconnecting = FALSE;
		s->onEventOnlineOk = FALSE;

		s->serv()->setHost(s->acc()->host, s->acc()->port);
		s->serv()->setAccount(s->acc()->user, s->acc()->pass, s->acc()->resource);
		s->serv()->login(loginAs.type, loginAs.str, s->acc()->priority);
	}
}
*/

/*void jabcon::askLogin()
{
	pdb(DEBUG_JABCON, "askLogin\n");

        if(serv->isActive()) {
                QMessageBox::information(0, CAP(tr("Error")), tr("Please disconnect before changing the account."));
                return;
        }

        // cancel any pending connection
	isDisconnecting = TRUE;
        serv->disc();

        doNewUser = FALSE;

	LoginInfo info;
	info.user = acc.user;
	info.pass = acc.pass;
	info.host = acc.host;
	info.port = acc.port;
	info.resource = acc.resource;
	info.priority = acc.priority;
	info.savepw = opt_pass;
	info.autologin = opt_auto;
	info.use_ssl = opt_ssl;
	info.guest = FALSE;

        LoginDlg *login = new LoginDlg(&info, serv->isSSLSupported());
        int n = login->exec();

	// different user?
        if(info.user != acc.user) {
		// nuke the offline contact list (this will clear cvlist thru signal-chain)
                serv->reset();

		// there might be some "not in list" contacts left, so this gets them too
		cvp->clear();
	}

	opt_pass = info.savepw;
	opt_auto = info.autologin;
	opt_ssl = info.use_ssl;
	guestMode = info.guest;

	serv->setSSLEnabled(opt_ssl);

	// Login or Create
	if(n > 0) {
		if(guestMode) {
			pdb(DEBUG_JABCON, "Using GUESTMODE\n");
			oldacc = acc;
		}

		acc.user = info.user;
		acc.pass = info.pass;
		acc.host = info.host;
		acc.port = info.port;
		acc.resource = info.resource;
		acc.priority = info.priority;

		// Login
		if(n == 1) {
			statusMenuChanged(STATUS_ONLINE);
		}
		// Create
		else if(n == 2) {
			doNewUser = TRUE;
			createAndConn();
		}
	}
}

void jabcon::delayedAskLogin()
{
        QTimer *t = new QTimer(this);
        connect(t, SIGNAL(timeout()), SLOT(askLogin()));
        t->start(0, TRUE);
}
*/

void jabcon::askAddUser()
{
	if(s->localStatus == STATUS_OFFLINE) {
		QMessageBox::information(0, CAP(tr("Error")), tr("You must be connected to the server in order to do this."));
		return;
	}

	AddUserDlg *w = AddUserDlg::find();
	if(w)
		bringToFront(w);
	else {
		QStringList groups;

		// build groups list
		UserListItem *item;
		for(item = s->userlist.start(USERLIST_CONTACTS); item; item = s->userlist.next()) {
			if(item->group.isEmpty())
				continue;

			// weed out duplicates
			if(qstringlistmatch(groups, item->group) == -1)
				groups.append(item->group);
		}

		// build services list
		QStringList services, names;
		for(item = s->userlist.start(USERLIST_SERVICES); item; item = s->userlist.next()) {
			services += item->jid;
			if(item->nick.isEmpty())
				names += QString(item->jid);
			else
				names += QString("%1 (%2)").arg(item->nick).arg(item->jid);
		}

		AddUserDlg *w = new AddUserDlg(services, names, groups, s->localStatus);
		connect(w, SIGNAL(signalGetGateway(const QString &, QString *)), SLOT(slotGetIQGateway(const QString &, QString *)));
		connect(w, SIGNAL(signalSetGateway(const QString &, const QString &, QString *)), SLOT(slotSetIQGateway(const QString &, const QString &, QString *)));
		connect(w, SIGNAL(signalCancelTransaction(const QString &)), SLOT(slotCancelTransaction(const QString &)));
		connect(w, SIGNAL(add(const AddUserDlg::Info &)), SLOT(slotAddUser(const AddUserDlg::Info &)));
		connect(this, SIGNAL(emitLocalUpdate(const JabRosterEntry &)), w, SLOT(localUpdate(const JabRosterEntry &)));
		w->show();
	}
}

void jabcon::slotAddUser(const AddUserDlg::Info &info)
{
	actionAdd(info.jid, info.nick, info.group);
}

/*void jabcon::servError(JabError *err)
{
        if(err->type == JABERR_CONNECT) {
                QMessageBox::critical(0, CAP(tr("Error")), QString(tr("There was an error communicating with the Jabber server.\nReason: %1")).arg(err->msg));
                //delayedAskLogin();
        }
        else if(err->type == JABERR_AUTH) {
                QMessageBox::critical(0, CAP(tr("Error")), QString(tr("Authorization failed.\nReason: %1")).arg(err->msg));
                //delayedAskLogin();
        }
        else if(err->type == JABERR_CREATE) {
                QMessageBox::critical(0, CAP(tr("Error")), QString(tr("Failed to create the new account.\nThe server gave this reason: \"%1\"")).arg(err->msg));
                //delayedAskLogin();
        }
        else if(err->type == JABERR_DISCONNECT) {
                QMessageBox::critical(0, CAP(tr("Error")), tr("You have been disconnected from the Jabber server."));
        }
        else {
                QMessageBox::critical(0, CAP(tr("Error")), QString(tr("An unknown error occurred.\nType: %1\nReason: %2")).arg(err->type).arg(err->msg));
        }
}
*/

void jabcon::closeProgram()
{
	s->disc(TRUE);

	// quit
        quit(0);
}

void jabcon::changeProfile()
{
        if(s->serv()->isActive()) {
                QMessageBox::information(0, CAP(tr("Error")), tr("Please disconnect before changing the profile."));
                return;
        }

	quit(1);
}

void jabcon::doOptions()
{
	OptionsDlg *w = OptionsDlg::find();
	if(w)
		bringToFront(w);
	else {
		w = new OptionsDlg(option);
		connect(w, SIGNAL(applyOptions(const Options &)), SLOT(slotApplyOptions(const Options &)));
		w->show();
	}
}

void jabcon::slotApplyOptions(const Options &opt)
{
	QString oldiconset = option.iconset;
	option = opt;

	// change icon set
	if(option.iconset != oldiconset) {
		unloadPsiIconSet();
		if(!loadPsiIconSet(option.iconset)) {
			QMessageBox::critical(0, tr("Error"), QString(tr("Unable to load the \"%1\" iconset.")).arg(option.iconset));
			option.iconset = oldiconset;
			loadPsiIconSet(option.iconset);
		}
	}

	mainwin->setAlwaysOnTop(option.alwaysOnTop);
	mainwin->setUseDock(option.useDock);

	emitOptionsUpdate();

	// save just the options
	pro.prefs = option;
	pro.toFile(pathToProfileConfig(activeProfile));

	//QSettings settings;
	//settings.insertSearchPath(QSettings::Windows, "/Affinix");
	//settings.insertSearchPath(QSettings::Unix, g.pathHome);
	//configSaveOptions(settings);
}

void jabcon::slotApplyAccounts()
{
        pro.acc.first() = *s->acc();
        pro.toFile(pathToProfileConfig(activeProfile));
}

void jabcon::doManageServices()
{
	ServicesDlg *w = ServicesDlg::find();
	if(w)
		bringToFront(w);
	else {
		w = new ServicesDlg(s->localStatus, 0);
		connect(this, SIGNAL(emitLocalUpdate(const JabRosterEntry &)), w, SLOT(localUpdate(const JabRosterEntry &)));
		connect(w, SIGNAL(signalRefresh(QString *)), SLOT(slotGetServices(QString *)));
		connect(w, SIGNAL(signalSearch(const QString &, QString *)), SLOT(slotGetSearchForm(const QString &, QString *)));
		connect(w, SIGNAL(signalGetRegForm(const QString &, QString *)), SLOT(slotGetRegForm(const QString &, QString *)));
		connect(w, SIGNAL(signalCancelTransaction(const QString &)), SLOT(slotCancelTransaction(const QString &)));
		w->show();

		w->doRefresh();
        }
}

void jabcon::doManageAccounts()
{
	jsm.manage();
}

void jabcon::doFileSharing()
{
	OfferMainDlg *w = OfferMainDlg::find();
	if(w)
		bringToFront(w);
	else {
		w = new OfferMainDlg(0);
		connect(this, SIGNAL(emitNewOffer(FileServerItem *)), w, SLOT(slotNewOffer(FileServerItem *)));

		// fill the listing
		QPtrList<FileServerItem> list = fileserv->items();
		QPtrListIterator<FileServerItem> it(list);
		for(FileServerItem *fi; (fi = it.current()); ++it)
			emitNewOffer(fi);

		w->show();
	}
}

#include<qfiledialog.h>
void jabcon::actionOfferFile(const QString &jid)
{
	QString fname = QFileDialog::getOpenFileName(QDir::homeDirPath(), "(*.*)", 0, "", "Choose a file to offer");
	if(!fname.isEmpty())
		offerFile(fname, "A cool video", jid);
}

void jabcon::actionOpenURL(const QString &jid)
{
	openURL(jid);
}

void jabcon::offerFile(const QString &fname, const QString &desc, const QString &jid)
{
	FileServerItem *i = fileserv->addFile(fname, desc, jid);
	if(i) {
		printf("%s\n", QString("Offering: [%1]").arg(i->url).latin1());
		emitNewOffer(i);
	}
}

void jabcon::openAccountInfo()
{
	InfoDlg *w = InfoDlg::find(s->acc()->jid());
	if(w)
		bringToFront(w);
	else {
		/*Info *p = InfoBank::get(jid);
		VCard tmp;
		if(p)
			tmp = p->vcard;*/

		VCard tmp;
		w = new InfoDlg(INFODLG_USER, s->acc()->jid(), tmp, s->localStatus);
		connect(w, SIGNAL(signalGetVCard(const QString &, QString *)), SLOT(slotGetVCard(const QString &, QString *)));
		connect(w, SIGNAL(signalSetVCard(const VCard &, QString *)), SLOT(slotSetVCard(const VCard &, QString *)));
		connect(w, SIGNAL(signalCancelTransaction(const QString &)), SLOT(slotCancelTransaction(const QString &)));
		connect(this, SIGNAL(emitLocalUpdate(const JabRosterEntry &)), w, SLOT(localUpdate(const JabRosterEntry &)));
		w->show();

		w->doRefresh();
	}
}

void jabcon::slotGetServices(QString *id)
{
	JT_GetServices *j = new JT_GetServices(s->serv()->ioUser(), "");
	connect(j, SIGNAL(finished(JabTask *)), SLOT(slotJTDone(JabTask *)));
	*id = j->id();
	j->go();
}

void jabcon::slotJTDone(JabTask *t)
{
	//printf("slotJTDone\n");
	if(t->isA("JT_GetServices")) {
		JT_GetServices *j = (JT_GetServices *)t;
		if(t->success()) {
			slotGetServicesResponse(&j->services);
		}
		else {
			ServicesDlg *w = ServicesDlg::find();
			if(w)
				w->loadFail();
		}
	}
	else if(t->isA("JT_VCard")) {
		JT_VCard *j = (JT_VCard *)t;
		QString jid = j->jid;
		if(j->type == 0) {
			if(t->success()) {
				slotGetVCardResponse(jid, &j->vcard);
			}
			else {
				slotGetVCardResponse(jid, 0);
			}
		}
		else {
			slotSetVCardResponse(j->success());
		}
	}
	else if(t->isA("JT_RegForm")) {
		JT_RegForm *j = (JT_RegForm *)t;
		if(j->type == 0) {
			if(t->success()) {
				slotGetRegFormResponse(j->form);
			}
			else {
				ServicesDlg *w = ServicesDlg::find();
				if(w)
					w->loadFormFail();
			}
		}
		else {
			slotPutRegFormResponse(j->jid, j->success(), j->errorString());
		}
	}
	else if(t->isA("JT_Search")) {
		JT_Search *j = (JT_Search *)t;
		if(j->type == 0) {
			if(t->success())
				slotGetSearchFormResponse(j->jid, &j->form);
			else
				slotGetSearchFormResponse(j->jid, 0);
		}
		else {
			if(t->success())
				slotPutSearchFormResponse(j->jid, &j->roster);
			else
				slotPutSearchFormResponse(j->jid, 0);
		}
	}
	else if(t->isA("JT_Gateway")) {
		JT_Gateway *j = (JT_Gateway *)t;
		AddUserDlg *w = AddUserDlg::find();
		if(w) {
			if(j->type == 0)
				w->slotGetGatewayResponse(j->id(), j->success(), j->desc);
			else
				w->slotSetGatewayResponse(j->id(), j->success(), j->prompt);
		}
	}

	t->deleteLater();
}

void jabcon::slotGetServicesResponse(JabRoster *services)
{
	ServicesDlg *w = ServicesDlg::find();
	if(w)
		w->loadSuccess(services);
}

void jabcon::slotGetRegForm(const QString &service, QString *id)
{
	// do we already have this open?
	RegistrationDlg *r = RegistrationDlg::find(service);
	if(r) {
		// loaded successfully
		ServicesDlg *w = ServicesDlg::find();
		if(w)
			w->loadFormSuccess();

		bringToFront(r);
		return;
	}

	JT_RegForm *j = new JT_RegForm(s->serv()->ioUser());
	connect(j, SIGNAL(finished(JabTask *)), SLOT(slotJTDone(JabTask *)));
	j->get(service);
	*id = j->id();
	j->go();
}

void jabcon::slotGetRegFormResponse(const JabForm &form)
{
	ServicesDlg *w = ServicesDlg::find();
	if(w) {
		w->loadFormSuccess();

		RegistrationDlg *r = new RegistrationDlg(form);
		connect(this, SIGNAL(emitLocalUpdate(const JabRosterEntry &)), r, SLOT(localUpdate(const JabRosterEntry &)));
		connect(r, SIGNAL(signalSubmitForm(const JabForm &, QString *)), this, SLOT(slotPutRegForm(const JabForm &, QString *)));
		r->show();
	}
}

void jabcon::slotPutRegForm(const JabForm &form, QString *id)
{
	pdb(DEBUG_JABCON, "submitting form\n");

	JT_RegForm *j = new JT_RegForm(s->serv()->ioUser());
	connect(j, SIGNAL(finished(JabTask *)), SLOT(slotJTDone(JabTask *)));
	j->set(form);
	*id = j->id();
	j->go();
}

void jabcon::slotPutRegFormResponse(const QString &service, bool ok, const QString &err)
{
	pdb(DEBUG_JABCON, QString("Registration result from %1: %2\n").arg(service).arg(ok ? "Success" : "Failed"));

	RegistrationDlg *r = RegistrationDlg::find(service);
	if(r)
		r->putRegFormResponse(ok, err);
}

void jabcon::slotGetVCard(const QString &jid, QString *id)
{
	JT_VCard *j = new JT_VCard(s->serv()->ioUser());
	connect(j, SIGNAL(finished(JabTask *)), SLOT(slotJTDone(JabTask *)));
	j->get(jid);
	*id = j->id();
	j->go();
}

void jabcon::slotGetVCardResponse(const QString &jid, VCard *vcard)
{
	if(vcard) {
		Info *p = new Info;
		p->jid = jid;
		p->vcard = *vcard;
		//InfoBank::put(p);

		// rename the contact in list [pushinfo]
		//UserListItem *i = userlist.findAnyByJid(jid);
		//if(i && !info->field[vNickname].isEmpty())
		//	userRename(i, info->field[vNickname]);
	}

	InfoDlg *w = InfoDlg::find(jid);
	if(!w)
		return;

	if(vcard) {
		w->updateVCard(*vcard);
	}
	else
		w->error();
}

void jabcon::slotSetVCard(const VCard &vcard, QString *id)
{
	JT_VCard *j = new JT_VCard(s->serv()->ioUser());
	connect(j, SIGNAL(finished(JabTask *)), SLOT(slotJTDone(JabTask *)));
	VCard x = vcard;
	j->set(x);
	*id = j->id();
	j->go();
}

void jabcon::slotSetVCardResponse(bool x)
{
	InfoDlg *w = InfoDlg::find(s->acc()->jid());
	if(!w)
		return;

	w->result(x);
}

void jabcon::slotGetSearchForm(const QString &service, QString *id)
{
	// do we already have this open?
	SearchDlg *sd = SearchDlg::find(service);
	if(sd) {
		// loaded successfully
		ServicesDlg *w = ServicesDlg::find();
		if(w)
			w->loadFormSuccess();

		bringToFront(sd);
		return;
	}

	JT_Search *j = new JT_Search(s->serv()->ioUser());
	connect(j, SIGNAL(finished(JabTask *)), SLOT(slotJTDone(JabTask *)));
	j->get(service);
	*id = j->id();
	j->go();
}

void jabcon::slotGetSearchFormResponse(const QString &service, JabForm *p)
{
	ServicesDlg *w = ServicesDlg::find();
	if(w) {
		if(p) {
			w->loadFormSuccess();

			SearchDlg *sd = new SearchDlg(service, *p, s->localStatus);
			connect(this, SIGNAL(emitLocalUpdate(const JabRosterEntry &)), sd, SLOT(localUpdate(const JabRosterEntry &)));
			connect(sd, SIGNAL(signalSubmitForm(const JabForm &, QString *)), this, SLOT(slotPutSearchForm(const JabForm &, QString *)));
			connect(sd, SIGNAL(signalCancelTransaction(const QString &)), SLOT(slotCancelTransaction(const QString &)));
			connect(sd, SIGNAL(aInfo(const QString &)), SLOT(actionInfo(const QString &)));
			connect(sd, SIGNAL(aAdd(const QString &, const QString &, const QString &)), SLOT(actionAdd(const QString &, const QString &, const QString &)));
			sd->show();
		}
		else
			w->loadFormFail();
	}
}

void jabcon::slotPutSearchForm(const JabForm &form, QString *id)
{
	pdb(DEBUG_JABCON, "submitting search form\n");

	JT_Search *j = new JT_Search(s->serv()->ioUser());
	connect(j, SIGNAL(finished(JabTask *)), SLOT(slotJTDone(JabTask *)));
	j->set(form);
	*id = j->id();
	j->go();
}

void jabcon::slotPutSearchFormResponse(const QString &service, JabRoster *p)
{
	pdb(DEBUG_JABCON, QString("Search result from %1: %2\n").arg(service).arg(p ? "Success" : "Failed"));

	SearchDlg *s = SearchDlg::find(service);
	if(s)
		s->putSearchFormResponse(p);
}


void jabcon::slotGetIQGateway(const QString &service, QString *id)
{
	JT_Gateway *j = new JT_Gateway(s->serv()->ioUser());
	connect(j, SIGNAL(finished(JabTask *)), SLOT(slotJTDone(JabTask *)));
	j->get(service);
	*id = j->id();
	j->go();
}

void jabcon::slotSetIQGateway(const QString &service, const QString &prompt, QString *id)
{
	JT_Gateway *j = new JT_Gateway(s->serv()->ioUser());
	connect(j, SIGNAL(finished(JabTask *)), SLOT(slotJTDone(JabTask *)));
	j->set(service, prompt);
	*id = j->id();
	j->go();
}


void jabcon::slotCancelTransaction(const QString &id)
{
	pdb(DEBUG_JABCON, QString("Cancelling transaction: [%1]\n").arg(id));
	s->serv()->cancelTransaction(id);
}


void jabcon::slotCheckVCard(JabTask *t)
{
	JT_VCard *j = (JT_VCard *)t;
	if(! (j->success() && !j->vcard.isIncomplete()) )
		openAccountInfo();
}


void jabcon::doOnEvent(const QString &str)
{
	if(str.isEmpty())
		return;

	// no sound?
	if(!useSound)
		return;

	// no away sounds?
	if(option.noAwaySound && (s->localStatus == STATUS_AWAY || s->localStatus == STATUS_XA || s->localStatus == STATUS_DND))
		return;

#if defined(Q_WS_WIN) || defined(Q_WS_MAC)
	QSound::play(str);
#else
	if(!option.player.isEmpty()) {
		QStringList args;
		args += option.player;
		args += str;
		QProcess cmd(args);
		cmd.start();
	}
#endif
}

void jabcon::enableOnEventOnline()
{
	s->onEventOnlineOk = TRUE;
}

void jabcon::recvNextEvent()
{
	EventItem *ei = events.peekNext();
	if(!ei)
		return;
	UserListItem *i = s->userlist.findAnyByJid(ei->jid);
	if(!i)
		return;

	readMessage(i);
}


/****************************************************************************
  EventQueue
****************************************************************************/
EventQueue::EventQueue()
:QPtrList<EventItem>()
{
}

int EventQueue::count(const QString &jid)
{
	int total = 0;

	for(EventItem *i = last(); i; i = prev()) {
		if(i->jid == jid)
			++total;
	}

	return total;
}

void EventQueue::enqueue(EventItem *i)
{
	insert(0, i);
}

EventItem *EventQueue::dequeue(const QString &jid)
{
	if(isEmpty())
		return 0;

	for(EventItem *i = last(); i; i = prev()) {
		if(i->jid == jid) {
			remove();
			return i;
		}
	}
	return 0;
}

EventItem *EventQueue::peek(const QString &jid)
{
	if(isEmpty())
		return 0;

	for(EventItem *i = last(); i; i = prev()) {
		if(i->jid == jid) {
			return i;
		}
	}
	return 0;
}

EventItem *EventQueue::dequeueNext()
{
	if(isEmpty())
		return 0;

	EventItem *i = last();
	remove();

	return i;
}

EventItem *EventQueue::peekNext()
{
	return last();
}

bool EventQueue::hasChats(const QString &jid)
{
	if(isEmpty())
		return FALSE;

	for(EventItem *i = last(); i; i = prev()) {
		if(i->jid == jid) {
			if(i->msg.type == MESSAGE_CHAT)
				return TRUE;
		}
	}

	return FALSE;
}

// this function extracts all chats from the queue, and returns a list of queue positions
void EventQueue::extractChats(const QString &jid, QPtrList<Message> *list, QValueList<int> *pos)
{
	if(isEmpty())
		return;

	int at = 0;
	for(EventItem *i = last(); (i = current());) {
		if(i->jid == jid) {
			if(i->msg.type == MESSAGE_CHAT) {
				Message *m = new Message(i->msg);
				list->insert(0, m);
				pos->append(at);

				bool isLast = (i == getLast());
				remove();
				if(isLast) {
					++at;
					continue;
				}
			}
			++at;
		}
		i = prev();
	}
}

// this function removes all events associated with the input jid
void EventQueue::removeAll(const QString &jid)
{
	for(EventItem *i = first(); (i = current());) {
		if(i->jid == jid)
			remove();
		else
			i = next();
	}
}
