/****************************************************************************
** jabstream.h - handles a Jabber XML stream
** Copyright (C) 2001, 2002  Justin Karneges
**
** This program is free software; you can redistribute it and/or
** modify it under the terms of the GNU General Public License
** as published by the Free Software Foundation; either version 2
** of the License, or (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307,USA.
**
****************************************************************************/

#ifndef JABSTREAM_H
#define JABSTREAM_H

#include<qxml.h>
#include<qdom.h>
#include<qobject.h>
#include<qstring.h>
#include<qtimer.h>
#include<qptrstack.h>
#include<qptrqueue.h>
#include<qsocket.h>
#include"sslfilter.h"


#define JABSTREAM_ERR_DNS         0
#define JABSTREAM_ERR_CONNREFUSED 1
#define JABSTREAM_ERR_CONNTIMEOUT 2
#define JABSTREAM_ERR_DISC        3
#define JABSTREAM_ERR_SOCKET      4
#define JABSTREAM_ERR_HANDSHAKE   5


/****************************************************************************
  JabStream

  This class handles a Jabber XML stream.  It collects XML data into
  QDomElements and signals the chunks back to the caller.  There is no
  actual Jabber logic in this class.  It simply takes care of XML and
  a socket connection.

  HOWTO:

    JabStream stream;
    stream.connectToHost("jabber.org", 5222");

  The class communicates back to you via signals:

    connected()                       - connection success
    error(int)                        - connection/stream error
    packetReady(const QDomElement &)  - new XML chunk ready

****************************************************************************/

class JabXmlHandler : public QObject, public QXmlDefaultHandler
{
	Q_OBJECT
public:
	JabXmlHandler(QDomDocument *);

	// Xml functions (reimplemented)
	bool startDocument();
	bool startElement(const QString &, const QString &, const QString &, const QXmlAttributes &);
	bool endElement(const QString &, const QString &, const QString &);
	bool characters(const QString &);

	QString toLower(QString s);

signals:
	void packetReady(const QDomElement &);
	void handshake(bool, const QString &);

private:
	QString indent, characterData;
	int depth;
	QDomDocument *doc;
	QDomElement chunk, current;
};


class JabStream : public QObject
{
	Q_OBJECT
public:
	JabStream();
	~JabStream();

	void connectToHost(const QString &host, int port);
	void disc();
	bool isConnected() { return v_isHandShaken; }
	QString id() { return v_id; }

	void setNoop(int);

	bool isSSLSupported();
	bool isSSLEnabled() { return use_ssl; }
	void setSSLEnabled(bool);

	static QCString encodeXML(QString);
	static QCString elemToString(const QDomElement &);

signals:
	void connected();
	void error(int);
	void receivePacket(const QDomElement &);

public slots:
	void sendPacket(const QDomElement &);
	void sendString(const QCString &);

private slots:
	// QSocket slots
	void sock_connected();
	void sock_disconnected();
	void sock_readyRead();
	void sock_error(int);

	// SSLFilter
	void ssl_outgoingReady();
	void ssl_readyRead();
	void ssl_error();

	// delayed functions so we can "get out" of a QSocket slot before continuing
	void delayedProcessError();
	void delayedProcessReceived();
	void delayedProcessHandShake();

	// prevent NAT/connection timeouts
	void doNoop();

	// JabXmlHandler slots
	void packetReady(const QDomElement &);
	void handshake(bool, const QString &);

private:
	QSocket *sock;
	SSLFilter *ssl;
	QDomDocument *doc;
	QXmlInputSource *src;
	QXmlSimpleReader *reader;
	JabXmlHandler *handler;

	QTimer *t;
	bool v_isConnected, v_isHandShaken, first_time;
	QString host;
	int port;
	bool use_ssl;

	int errType;
	int noop_time;

	QString v_id;

	QPtrQueue<QDomElement> in;

	void processIncomingData(const QByteArray &);
};

#endif
