//
// Ispell-Pspell module using Ispell through a Pipe
//
// Copyright 2000      Kevin Atkinson under the LGPL
// Copyright 1995-1998 The LyX Team
// Copyright 1995      Matthias Ettrich
// Parts of this file were orignally part of the LyX under the LyX licence.
// However, Asger K. Alstrup Nielsen, the main author of the rewritten
// spellchecker code gave Kevin Atkinson the right to co-license it
// under the LGPL
//

#include <config.h>

#include <unistd.h>
#include <fcntl.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <signal.h>
#include <assert.h>
#include <sys/wait.h>
#include <sys/types.h>
#include <ctype.h>

#if TIME_WITH_SYS_TIME
# include <sys/time.h>
# include <time.h>
#else
# if HAVE_SYS_TIME_H
#  include <sys/time.h>
# else
#  include <time.h>
# endif
#endif

#ifdef HAVE_SYS_SELECT_H
#include <sys/select.h>
#endif

#include <pspell/manager_impl.hh>
#include <pspell/wordlist.h>
#include <pspell/convert.hh>
#include <pspell/error_messages.hh>
#include <pspell/error_impl.hh>

struct IspellPipeSugWordList : public PspellWordList 
{
  const char * data_;
  unsigned int size_;
  const PspellConvert * from_internal_;

  bool empty() const;
  int size() const;
  PspellStringEmulation * elements() const;
  const char * encoding() const;
};

class IspellPipeManager : public PspellManagerImplBase
{
private:
  FILE *in_, *out_;  /* streams to communicate with ispell */
  volatile pid_t * isp_pid;

  char buf_[1024];

  PspellString          word_;
  PspellString          temp_word_;
  IspellPipeSugWordList sugs_;
  bool                  cor_;

  PspellConfig * config_;

  const PspellConvert * to_internal_;
  
  IspellPipeManager(const IspellPipeManager &);
  IspellPipeManager & operator= (const IspellPipeManager &);

public:
  
  IspellPipeManager(PspellConfig * config, PspellManagerLtHandle h);
  ~IspellPipeManager();

  bool setup();
  
  PspellConfig & config();
 
  int check(const char * word, int size);
  
  bool add_to_personal(const char * word, int size);
  
  bool add_to_session (const char * word, int size);
  
  const PspellWordList * suggest(const char * word, int size);
  
  bool save_all_word_lists(); 

  // will always return false
  bool store_replacement(const char * mis, int mis_size,
			 const char * cor, int cor_size);

  // the following methods are currently unimplented
  const PspellWordList * master_word_list(); 
  
  const PspellWordList * personal_word_list();
  
  const PspellWordList * session_word_list();
  
  bool clear_session();
  
};

struct IspellPipeSugEmulation : public PspellStringEmulation
{
  PspellString word;
  const char * b;
  const char * e;
  int  words_left;
  const PspellConvert * from_internal_;
  IspellPipeSugEmulation(const char * s, int wl, const PspellConvert * c)
    : b(s), e(s), words_left(wl), from_internal_(c) {}
  const char * next();
  bool at_end() const;
  PspellStringEmulation * clone() const;
  void assign(const PspellStringEmulation * other);
};

struct IspPidNode {
  volatile pid_t  pid;
  IspPidNode    * next;
  IspPidNode(pid_t p) : pid(p) {}
};

IspPidNode * isp_pid_first = 0; // pid for the `ispell' process.

void sigchild_h(int);

///////////////////////////////////////////////////////////
//
// IspellPipeSugEmulation methods
//
//

bool IspellPipeSugEmulation::at_end() const {
  return words_left == 0;
}

const char * IspellPipeSugEmulation::next() {
  if (words_left == 0) return 0;
  --words_left;
  b = e;
  while (*e != ',' && *e != '\n' && *e != '\0') ++e;
  assert(!((*b == '\n' || *b == '\0') && words_left != 0));
  word.clear();
  from_internal_->convert_until(b, e, word);
  e += 2;
  return word.c_str();
}

PspellStringEmulation * IspellPipeSugEmulation::clone() const
{
  return new IspellPipeSugEmulation(*this);
}

void IspellPipeSugEmulation::assign(const PspellStringEmulation * other)
{
  *this = *(const IspellPipeSugEmulation *)(other);
}


///////////////////////////////////////////////////////////
//
// IspellPipeSugWordList methods
//
//

bool IspellPipeSugWordList::empty() const 
{
  return size_ == 0;
}

int IspellPipeSugWordList::size() const 
{
  return size_;
}

const char * IspellPipeSugWordList::encoding() const
{
  return from_internal_->out_code();
}

PspellStringEmulation * IspellPipeSugWordList::elements() const
{
  return new IspellPipeSugEmulation(data_, 
				    size_, 
				    from_internal_);
}

///////////////////////////////////////////////////////////
//
// IspellManager methods
//
//

IspellPipeManager::IspellPipeManager(PspellConfig * config,
				     PspellManagerLtHandle h)
  :  PspellManagerImplBase(h)
  , in_(0), out_(0), isp_pid(0)
{
  to_internal_         = 0;
  sugs_.from_internal_ = 0;

  config_ = config->clone();
}

IspellPipeManager::~IspellPipeManager()
{
  if (to_internal_ != 0)
    delete to_internal_;
  if (sugs_.from_internal_ != 0)
    delete sugs_.from_internal_;
  delete config_;
  if (out_ != 0) {
    fflush(out_);
    fclose(out_);
  }
  if (isp_pid != 0) {
    sigset_t sigset;
    sigemptyset(&sigset);
    sigaddset(&sigset, SIGCHLD);
    sigprocmask(SIG_BLOCK, &sigset, 0);

    IspPidNode * p = 0;
    IspPidNode * c = isp_pid_first;
    while (c->pid != *isp_pid) {
      p = c;
      c = c->next;
    }
    if (p == 0) {
      isp_pid_first = c->next;
    } else {
      p->next = c->next;
    }
    delete c;

    sigprocmask(SIG_UNBLOCK, &sigset, 0);
  }
}

bool IspellPipeManager::setup()
{
  error_.reset_error();

  PspellString encoding    = config_->retrieve("encoding");
  PspellString wl_encoding = config_->retrieve("master-flags");
  PspellCanHaveError * temp;
  
  temp = new_pspell_convert(*config_, 
			    encoding.c_str(), 
			    wl_encoding.c_str());
  if (temp->error_number() != 0) {
    error_.set_error(temp);
    return false;
  }
  to_internal_ = static_cast<PspellConvert *>(temp);
  
  temp = new_pspell_convert(*config_, 
			    wl_encoding.c_str(), 
			    encoding.c_str());
  if (temp->error_number() != 0) {
    error_.set_error(temp);
    return false;
  }
  sugs_.from_internal_ = static_cast<PspellConvert *>(temp);


  static char o_buf[BUFSIZ];  // jc: it could be smaller
  int pipein[2], pipeout[2];
  pid_t t_pid = -1;

  if(pipe(pipein) == -1 || pipe(pipeout) == -1) {
    error_.set_error(PERROR_CANT_CREATE_PIPE, 
		     "Can't create pipe for spellchecker!");
    return false;
  }

  if ((out_ = fdopen(pipein[1], "w")) == 0 ||
      (in_ = fdopen(pipeout[0], "r")) == 0) {
    error_.set_error(PERROR_CANT_CREATE_PIPE,
		     "Can't create stream for pipe for spellchecker!");
    return false;
  }

  setvbuf(out_, o_buf, _IOLBF, BUFSIZ);

  t_pid = fork();

  if(t_pid == -1) {
    error_.set_error(PERROR_CANT_CREATE_PIPE,
		     "Can't create child process for spellchecker!");
    return false;
  } 

  const char * argv[8];
  int argc = 0;

  argv[argc++] = "ispell";
  argv[argc++] = "-a";
  
  argv[argc++] = "-d";
  PspellString mwl = config_->retrieve("master");
  argv[argc++] = mwl.c_str();

  PspellString per;
  if (config_->have("personal")) {
    argv[argc++] = "-p";
    per = config_->retrieve("personal");
    argv[argc++] = per.c_str();
  }
  
  PspellString ign;
  if (config_->have("ignore")) {
    argv[argc++] = "-W";
    ign = config_->retrieve("ignore");
    argv[argc++] = ign.c_str();
  }
  
  if (config_->have("run-together")) {
    if (config_->retrieve_bool("run-together")) {
      // Consider run-together words as legal compounds
      argv[argc++] = "-C";
    } else {
      // Report run-together words with
      // missing blanks as errors
      argv[argc++] = "-B"; 
    }
  }
  
  argv[argc++] = 0;
  
  if(t_pid == 0) {   
    //
    // child process 
    //
    dup2(pipein[0], STDIN_FILENO);
    dup2(pipeout[1], STDOUT_FILENO);
    close(pipein[0]);
    close(pipein[1]);
    close(pipeout[0]);
    close(pipeout[1]);

    execvp(argv[0], const_cast<char * const *>(argv));

    // if we get to this point then something went wront
    
    write(STDOUT_FILENO, "ERROR\n", 6);

    _exit(0);

  } 
  {
    sigset_t sigset;
    sigemptyset(&sigset);
    sigaddset(&sigset, SIGCHLD);
    sigprocmask(SIG_BLOCK, &sigset, 0);
    IspPidNode * c = new IspPidNode(t_pid);
    isp_pid = &c->pid;
    c->next = isp_pid_first;
    isp_pid_first = c;
    sigprocmask(SIG_UNBLOCK, &sigset, 0);

    signal(SIGPIPE, SIG_IGN);
    signal(SIGCHLD, sigchild_h);

    /* Parent process: Read ispells identification message */
    char buf[2048];
    fd_set infds;
    struct timeval tv;
    int retval = 0;
    FD_ZERO(&infds);
    FD_SET(pipeout[0], &infds);
    tv.tv_sec = 15; // fifteen second timeout
    tv.tv_usec = 0;

    // Configure provides us with macros which are supposed to do
    // the right typecast.
    retval = select(SELECT_TYPE_ARG1 (pipeout[0]+1), 
		    SELECT_TYPE_ARG234 (&infds), 
		    0, 
		    0, 
		    SELECT_TYPE_ARG5 (&tv));

    if (retval > 0) {

      // Ok, do the reading. We don't have to FD_ISSET since
      // there is only one fd in infds.
      fgets(buf, 2048, in_);
      if (strcmp(buf, "ERROR\n") == 0) {
	error_.set_error(PERROR_CANT_CREATE_PIPE,
			 "Could not execute Ispell.");
      }
      fputs("!\n", out_); // Set terse mode (silently accept correct words)
		
    } else if (retval == 0) {

      error_.set_error(PERROR_CANT_CREATE_PIPE,
		       "Ispell process times out.");

      close(pipeout[0]); close(pipeout[1]);
      close(pipein[0]); close(pipein[1]);
      return false;
      
    } else {

      error_.set_error(PERROR_CANT_CREATE_PIPE,
		       "Select on Ispell process returned an error.");
      return false;
    }
  }
  return true;
}

#define CHECK_STATE                             \
 do {                                           \
  if (*isp_pid == -1) {                           \
    error_.set_error(process_died, "ispell");   \
    return -1;                                  \
  }                                             \
 } while (false)                                \


int IspellPipeManager::check(const char * w, int s)
{
  error_.reset_error();
  word_.clear();
  to_internal_->convert(w, s, word_);
  CHECK_STATE;
  fputs(word_.c_str(), out_);
  fputc('\n', out_);
  CHECK_STATE;
  fgets(buf_, 1024, in_);
  
  if (*buf_ == '\n') { // Word ok, nothing returned as ispell in terse mode
 
    cor_ = true;

  } else {

    cor_ = false;
    char * p = strchr(buf_ + 2, ' ');
    assert(p != 0);
    ++p;
    if (*p == '0' || *buf_ == '#') {
      sugs_.size_ = 0;
    } else {
      sugs_.size_ = strtol(p, 0, 10);
      sugs_.data_ = strchr(p+1, ':') + 2;
    }
    // wait for ispell to finish
    char tbuf[256];
    do {
      CHECK_STATE;
      fgets(tbuf, 256, in_);
    } while (*tbuf != '\n');

  }
  return cor_;
}

const PspellWordList * IspellPipeManager::suggest(const char * w, int s) 
{
  error_.reset_error();
  temp_word_.clear();
  to_internal_->convert(w, s, temp_word_);
  if (strcmp(word_.c_str(), temp_word_.c_str()) != 0)
    check(w, s);
  if (error_.error_number() != 0) 
    return 0;
  if (cor_) {
    sugs_.data_ = word_.c_str();
    sugs_.size_ = 1;
  }
  return &sugs_;
}

PspellConfig & IspellPipeManager::config() 
{
  return *config_;
}

#undef CHECK_STATE
#define CHECK_STATE                             \
 do {                                           \
  if (*isp_pid == 1) {                          \
    error_.set_error(process_died, "ispell");   \
    return false;                               \
  }                                             \
 } while (false)                                \

bool IspellPipeManager::save_all_word_lists() {
  error_.reset_error();
  CHECK_STATE;
  fputs("#\n", out_);
  CHECK_STATE;
  return true;
}

bool IspellPipeManager::add_to_personal (const char * w, int s)
{
  error_.reset_error();
  word_.clear();
  to_internal_->convert(w, s, word_);
  CHECK_STATE;
  putc('*', out_); // Insert word in personal dictionary
  fputs(word_.c_str(), out_);
  putc('\n', out_);
  CHECK_STATE;
  return true;
}

bool IspellPipeManager::add_to_session (const char * w, int s)
{
  error_.reset_error();
  word_.clear();
  to_internal_->convert(w, s, word_);
  CHECK_STATE;
  putc('@', out_); // Accept in this session
  fputs(word_.c_str(), out_);
  putc('\n', out_);
  CHECK_STATE;
  return true;
}

bool IspellPipeManager::store_replacement(char const *, int, 
					  char const *, int)
{
  error_.reset_error();
  return false;
}

const PspellWordList * IspellPipeManager::master_word_list() {
  error_.reset_error();
  error_.set_error(operation_not_supported, "get master_word_list");
  return 0;
}

const PspellWordList * IspellPipeManager::personal_word_list() {
  error_.reset_error();
  error_.set_error(operation_not_supported, "get personal_word_list");
  return 0;
}

const PspellWordList * IspellPipeManager::session_word_list() {
  error_.reset_error();
  error_.set_error(operation_not_supported, "get session_word_list");
  return 0;
}
  
bool IspellPipeManager::clear_session() {
  error_.reset_error();
  error_.set_error(operation_not_supported, "clear_session");
  return false;
}

void sigchild_h(int) {
  int status;
  int s;
  while (s = waitpid(0, &status, WNOHANG | WUNTRACED), s > 0) {
    IspPidNode * c = isp_pid_first;
    while (c != 0 && c->pid != s) c = c->next;
    if (c == 0) return;
    if (WIFEXITED(status) || WIFSIGNALED(status)) {
      c->pid = -1;
    } else if (WIFSTOPPED(status)) {
      kill(c->pid, SIGCONT);
    } else {
      abort();
    }
  }
}


extern "C"
PspellCanHaveError *
libpspell_ispell_LTX_new_pspell_manager_class(PspellConfig * config, 
					      PspellManagerLtHandle h) 
{
  IspellPipeManager * m = new  IspellPipeManager(config,h);
  if (!m->setup()) {

    PspellCanHaveErrorImpl * e = new PspellCanHaveErrorImpl();
    e->set_error(m->error_number(), m->error_message());
    delete m;
    return e;

  } else {

    return m;
  }

}
