#include <iomanip>
#include "catch.hpp"
#include "purify/MeasurementOperator.h"
#include "purify/directories.h"
#include "purify/kernels.h"
#include "purify/pfitsio.h"
#include "purify/utilities.h"
#include "purify/FFTOperator.h"
using namespace purify;
using namespace purify::notinstalled;

TEST_CASE("Measurement Operator [Kaiser Bessel Linear Interpolation]", "[KB_Interp]") {

  utilities::vis_params uv_vis;
  t_real max;
  t_real max_diff;
  t_real over_sample;
  t_real cellsize;
  std::string kernel;
  std::string vis_file = vla_filename("at166B.3C129.c0.vis");

  // Gridding example
  cellsize = 0.3;
  over_sample = 1.375;
  t_int J = 4;
  uv_vis = utilities::read_visibility(vis_file); // visibility data being read in
  t_int width = 1024;
  t_int height = 1024;

  uv_vis = utilities::uv_symmetry(
      uv_vis); // Enforce condjugate symmetry by reflecting measurements in uv coordinates

  kernel = "kb_interp";
  MeasurementOperator op(uv_vis, J, J, kernel, width, height, 5, over_sample, cellsize,
                         cellsize); // Generating gridding matrix

  Vector<t_complex> point_source = uv_vis.vis * 0;
  point_source.setOnes(); // Creating model visibilities for point source
  Image<t_complex> psf;
  psf = op.grid(point_source);
  max = psf.real().maxCoeff();
  psf = psf / max;
  pfitsio::write2d(psf.real(), output_filename("kb_interp_psf.fits"));

  Image<t_real> kb_img = op.grid(uv_vis.vis).real();
  max = kb_img.maxCoeff();
  kb_img = kb_img / max;
  pfitsio::write2d(kb_img.real(), output_filename("grid_image_real_kb_interp_4.fits"));
  pfitsio::write2d(op.S.real(), output_filename("scale_kb_interp_4.fits"));
}

TEST_CASE("Measurement Operator [Kaiser Bessel]", "[KB_Non-Interp]") {

  utilities::vis_params uv_vis;
  t_real max;
  t_real max_diff;
  t_real over_sample;
  t_real cellsize;
  std::string kernel;
  std::string vis_file = vla_filename("at166B.3C129.c0.vis");

  // Gridding example
  cellsize = 0.3;
  over_sample = 2;
  t_int J = 4;
  uv_vis = utilities::read_visibility(vis_file); // visibility data being read in
  t_int width = 1024;
  t_int height = 1024;

  uv_vis = utilities::uv_symmetry(
      uv_vis); // Enforce condjugate symmetry by reflecting measurements in uv coordinates

  kernel = "kb";
  MeasurementOperator op(uv_vis, J, J, kernel, width, height, 5, over_sample, cellsize,
                         cellsize); // Generating gridding matrix

  Vector<t_complex> point_source = uv_vis.vis * 0;
  point_source.setOnes(); // Creating model visibilities for point source
  Image<t_complex> psf;
  psf = op.grid(point_source);
  max = psf.real().maxCoeff();
  psf = psf / max;
  pfitsio::write2d(psf.real(), output_filename("kb_psf.fits"));

  Image<t_real> kb_img = op.grid(uv_vis.vis).real();
  max = kb_img.maxCoeff();
  kb_img = kb_img / max;
  pfitsio::write2d(kb_img.real(), output_filename("grid_image_real_kb_4.fits"));
  pfitsio::write2d(op.S.real(), output_filename("scale_kb_4.fits"));
}

TEST_CASE("Measurement Operator [Kernels]", "[Kernels]") {
  t_real x_values[] = {-3,
                       -2.99000000000000,
                       -2.98000000000000,
                       -2.97000000000000,
                       -2.96000000000000,
                       -2.95000000000000,
                       -2.94000000000000,
                       -2.93000000000000,
                       -2.92000000000000,
                       -2.91000000000000,
                       -2.90000000000000,
                       -2.89000000000000,
                       -2.88000000000000,
                       -2.87000000000000,
                       -2.86000000000000,
                       -2.85000000000000,
                       -2.84000000000000,
                       -2.83000000000000,
                       -2.82000000000000,
                       -2.81000000000000,
                       -2.80000000000000,
                       -2.79000000000000,
                       -2.78000000000000,
                       -2.77000000000000,
                       -2.76000000000000,
                       -2.75000000000000,
                       -2.74000000000000,
                       -2.73000000000000,
                       -2.72000000000000,
                       -2.71000000000000,
                       -2.70000000000000,
                       -2.69000000000000,
                       -2.68000000000000,
                       -2.67000000000000,
                       -2.66000000000000,
                       -2.65000000000000,
                       -2.64000000000000,
                       -2.63000000000000,
                       -2.62000000000000,
                       -2.61000000000000,
                       -2.60000000000000,
                       -2.59000000000000,
                       -2.58000000000000,
                       -2.57000000000000,
                       -2.56000000000000,
                       -2.55000000000000,
                       -2.54000000000000,
                       -2.53000000000000,
                       -2.52000000000000,
                       -2.51000000000000,
                       -2.50000000000000,
                       -2.49000000000000,
                       -2.48000000000000,
                       -2.47000000000000,
                       -2.46000000000000,
                       -2.45000000000000,
                       -2.44000000000000,
                       -2.43000000000000,
                       -2.42000000000000,
                       -2.41000000000000,
                       -2.40000000000000,
                       -2.39000000000000,
                       -2.38000000000000,
                       -2.37000000000000,
                       -2.36000000000000,
                       -2.35000000000000,
                       -2.34000000000000,
                       -2.33000000000000,
                       -2.32000000000000,
                       -2.31000000000000,
                       -2.30000000000000,
                       -2.29000000000000,
                       -2.28000000000000,
                       -2.27000000000000,
                       -2.26000000000000,
                       -2.25000000000000,
                       -2.24000000000000,
                       -2.23000000000000,
                       -2.22000000000000,
                       -2.21000000000000,
                       -2.20000000000000,
                       -2.19000000000000,
                       -2.18000000000000,
                       -2.17000000000000,
                       -2.16000000000000,
                       -2.15000000000000,
                       -2.14000000000000,
                       -2.13000000000000,
                       -2.12000000000000,
                       -2.11000000000000,
                       -2.10000000000000,
                       -2.09000000000000,
                       -2.08000000000000,
                       -2.07000000000000,
                       -2.06000000000000,
                       -2.05000000000000,
                       -2.04000000000000,
                       -2.03000000000000,
                       -2.02000000000000,
                       -2.01000000000000,
                       -2,
                       -1.99000000000000,
                       -1.98000000000000,
                       -1.97000000000000,
                       -1.96000000000000,
                       -1.95000000000000,
                       -1.94000000000000,
                       -1.93000000000000,
                       -1.92000000000000,
                       -1.91000000000000,
                       -1.90000000000000,
                       -1.89000000000000,
                       -1.88000000000000,
                       -1.87000000000000,
                       -1.86000000000000,
                       -1.85000000000000,
                       -1.84000000000000,
                       -1.83000000000000,
                       -1.82000000000000,
                       -1.81000000000000,
                       -1.80000000000000,
                       -1.79000000000000,
                       -1.78000000000000,
                       -1.77000000000000,
                       -1.76000000000000,
                       -1.75000000000000,
                       -1.74000000000000,
                       -1.73000000000000,
                       -1.72000000000000,
                       -1.71000000000000,
                       -1.70000000000000,
                       -1.69000000000000,
                       -1.68000000000000,
                       -1.67000000000000,
                       -1.66000000000000,
                       -1.65000000000000,
                       -1.64000000000000,
                       -1.63000000000000,
                       -1.62000000000000,
                       -1.61000000000000,
                       -1.60000000000000,
                       -1.59000000000000,
                       -1.58000000000000,
                       -1.57000000000000,
                       -1.56000000000000,
                       -1.55000000000000,
                       -1.54000000000000,
                       -1.53000000000000,
                       -1.52000000000000,
                       -1.51000000000000,
                       -1.50000000000000,
                       -1.49000000000000,
                       -1.48000000000000,
                       -1.47000000000000,
                       -1.46000000000000,
                       -1.45000000000000,
                       -1.44000000000000,
                       -1.43000000000000,
                       -1.42000000000000,
                       -1.41000000000000,
                       -1.40000000000000,
                       -1.39000000000000,
                       -1.38000000000000,
                       -1.37000000000000,
                       -1.36000000000000,
                       -1.35000000000000,
                       -1.34000000000000,
                       -1.33000000000000,
                       -1.32000000000000,
                       -1.31000000000000,
                       -1.30000000000000,
                       -1.29000000000000,
                       -1.28000000000000,
                       -1.27000000000000,
                       -1.26000000000000,
                       -1.25000000000000,
                       -1.24000000000000,
                       -1.23000000000000,
                       -1.22000000000000,
                       -1.21000000000000,
                       -1.20000000000000,
                       -1.19000000000000,
                       -1.18000000000000,
                       -1.17000000000000,
                       -1.16000000000000,
                       -1.15000000000000,
                       -1.14000000000000,
                       -1.13000000000000,
                       -1.12000000000000,
                       -1.11000000000000,
                       -1.10000000000000,
                       -1.09000000000000,
                       -1.08000000000000,
                       -1.07000000000000,
                       -1.06000000000000,
                       -1.05000000000000,
                       -1.04000000000000,
                       -1.03000000000000,
                       -1.02000000000000,
                       -1.01000000000000,
                       -1,
                       -0.990000000000000,
                       -0.980000000000000,
                       -0.970000000000000,
                       -0.960000000000000,
                       -0.950000000000000,
                       -0.940000000000000,
                       -0.930000000000000,
                       -0.920000000000000,
                       -0.910000000000000,
                       -0.900000000000000,
                       -0.890000000000000,
                       -0.880000000000000,
                       -0.870000000000000,
                       -0.860000000000000,
                       -0.850000000000000,
                       -0.840000000000000,
                       -0.830000000000000,
                       -0.820000000000000,
                       -0.810000000000000,
                       -0.800000000000000,
                       -0.790000000000000,
                       -0.780000000000000,
                       -0.770000000000000,
                       -0.760000000000000,
                       -0.750000000000000,
                       -0.740000000000000,
                       -0.730000000000000,
                       -0.720000000000000,
                       -0.710000000000000,
                       -0.700000000000000,
                       -0.690000000000000,
                       -0.680000000000000,
                       -0.670000000000000,
                       -0.660000000000000,
                       -0.650000000000000,
                       -0.640000000000000,
                       -0.630000000000000,
                       -0.620000000000000,
                       -0.610000000000000,
                       -0.600000000000000,
                       -0.590000000000000,
                       -0.580000000000000,
                       -0.570000000000000,
                       -0.560000000000000,
                       -0.550000000000000,
                       -0.540000000000000,
                       -0.530000000000000,
                       -0.520000000000000,
                       -0.510000000000000,
                       -0.500000000000000,
                       -0.490000000000000,
                       -0.480000000000000,
                       -0.470000000000000,
                       -0.460000000000000,
                       -0.450000000000000,
                       -0.440000000000000,
                       -0.430000000000000,
                       -0.420000000000000,
                       -0.410000000000000,
                       -0.400000000000000,
                       -0.390000000000000,
                       -0.380000000000000,
                       -0.370000000000000,
                       -0.360000000000000,
                       -0.350000000000000,
                       -0.340000000000000,
                       -0.330000000000000,
                       -0.320000000000000,
                       -0.310000000000000,
                       -0.300000000000000,
                       -0.290000000000000,
                       -0.280000000000000,
                       -0.270000000000000,
                       -0.260000000000000,
                       -0.250000000000000,
                       -0.240000000000000,
                       -0.230000000000000,
                       -0.220000000000000,
                       -0.210000000000000,
                       -0.200000000000000,
                       -0.190000000000000,
                       -0.180000000000000,
                       -0.170000000000000,
                       -0.160000000000000,
                       -0.150000000000000,
                       -0.140000000000000,
                       -0.130000000000000,
                       -0.120000000000000,
                       -0.110000000000000,
                       -0.100000000000000,
                       -0.0899999999999999,
                       -0.0800000000000001,
                       -0.0699999999999998,
                       -0.0600000000000001,
                       -0.0499999999999998,
                       -0.0400000000000000,
                       -0.0299999999999998,
                       -0.0200000000000000,
                       -0.00999999999999979,
                       0,
                       0.00999999999999979,
                       0.0200000000000000,
                       0.0299999999999998,
                       0.0400000000000000,
                       0.0499999999999998,
                       0.0600000000000001,
                       0.0699999999999998,
                       0.0800000000000001,
                       0.0899999999999999,
                       0.100000000000000,
                       0.110000000000000,
                       0.120000000000000,
                       0.130000000000000,
                       0.140000000000000,
                       0.150000000000000,
                       0.160000000000000,
                       0.170000000000000,
                       0.180000000000000,
                       0.190000000000000,
                       0.200000000000000,
                       0.210000000000000,
                       0.220000000000000,
                       0.230000000000000,
                       0.240000000000000,
                       0.250000000000000,
                       0.260000000000000,
                       0.270000000000000,
                       0.280000000000000,
                       0.290000000000000,
                       0.300000000000000,
                       0.310000000000000,
                       0.320000000000000,
                       0.330000000000000,
                       0.340000000000000,
                       0.350000000000000,
                       0.360000000000000,
                       0.370000000000000,
                       0.380000000000000,
                       0.390000000000000,
                       0.400000000000000,
                       0.410000000000000,
                       0.420000000000000,
                       0.430000000000000,
                       0.440000000000000,
                       0.450000000000000,
                       0.460000000000000,
                       0.470000000000000,
                       0.480000000000000,
                       0.490000000000000,
                       0.500000000000000,
                       0.510000000000000,
                       0.520000000000000,
                       0.530000000000000,
                       0.540000000000000,
                       0.550000000000000,
                       0.560000000000000,
                       0.570000000000000,
                       0.580000000000000,
                       0.590000000000000,
                       0.600000000000000,
                       0.610000000000000,
                       0.620000000000000,
                       0.630000000000000,
                       0.640000000000000,
                       0.650000000000000,
                       0.660000000000000,
                       0.670000000000000,
                       0.680000000000000,
                       0.690000000000000,
                       0.700000000000000,
                       0.710000000000000,
                       0.720000000000000,
                       0.730000000000000,
                       0.740000000000000,
                       0.750000000000000,
                       0.760000000000000,
                       0.770000000000000,
                       0.780000000000000,
                       0.790000000000000,
                       0.800000000000000,
                       0.810000000000000,
                       0.820000000000000,
                       0.830000000000000,
                       0.840000000000000,
                       0.850000000000000,
                       0.860000000000000,
                       0.870000000000000,
                       0.880000000000000,
                       0.890000000000000,
                       0.900000000000000,
                       0.910000000000000,
                       0.920000000000000,
                       0.930000000000000,
                       0.940000000000000,
                       0.950000000000000,
                       0.960000000000000,
                       0.970000000000000,
                       0.980000000000000,
                       0.990000000000000,
                       1,
                       1.01000000000000,
                       1.02000000000000,
                       1.03000000000000,
                       1.04000000000000,
                       1.05000000000000,
                       1.06000000000000,
                       1.07000000000000,
                       1.08000000000000,
                       1.09000000000000,
                       1.10000000000000,
                       1.11000000000000,
                       1.12000000000000,
                       1.13000000000000,
                       1.14000000000000,
                       1.15000000000000,
                       1.16000000000000,
                       1.17000000000000,
                       1.18000000000000,
                       1.19000000000000,
                       1.20000000000000,
                       1.21000000000000,
                       1.22000000000000,
                       1.23000000000000,
                       1.24000000000000,
                       1.25000000000000,
                       1.26000000000000,
                       1.27000000000000,
                       1.28000000000000,
                       1.29000000000000,
                       1.30000000000000,
                       1.31000000000000,
                       1.32000000000000,
                       1.33000000000000,
                       1.34000000000000,
                       1.35000000000000,
                       1.36000000000000,
                       1.37000000000000,
                       1.38000000000000,
                       1.39000000000000,
                       1.40000000000000,
                       1.41000000000000,
                       1.42000000000000,
                       1.43000000000000,
                       1.44000000000000,
                       1.45000000000000,
                       1.46000000000000,
                       1.47000000000000,
                       1.48000000000000,
                       1.49000000000000,
                       1.50000000000000,
                       1.51000000000000,
                       1.52000000000000,
                       1.53000000000000,
                       1.54000000000000,
                       1.55000000000000,
                       1.56000000000000,
                       1.57000000000000,
                       1.58000000000000,
                       1.59000000000000,
                       1.60000000000000,
                       1.61000000000000,
                       1.62000000000000,
                       1.63000000000000,
                       1.64000000000000,
                       1.65000000000000,
                       1.66000000000000,
                       1.67000000000000,
                       1.68000000000000,
                       1.69000000000000,
                       1.70000000000000,
                       1.71000000000000,
                       1.72000000000000,
                       1.73000000000000,
                       1.74000000000000,
                       1.75000000000000,
                       1.76000000000000,
                       1.77000000000000,
                       1.78000000000000,
                       1.79000000000000,
                       1.80000000000000,
                       1.81000000000000,
                       1.82000000000000,
                       1.83000000000000,
                       1.84000000000000,
                       1.85000000000000,
                       1.86000000000000,
                       1.87000000000000,
                       1.88000000000000,
                       1.89000000000000,
                       1.90000000000000,
                       1.91000000000000,
                       1.92000000000000,
                       1.93000000000000,
                       1.94000000000000,
                       1.95000000000000,
                       1.96000000000000,
                       1.97000000000000,
                       1.98000000000000,
                       1.99000000000000,
                       2,
                       2.01000000000000,
                       2.02000000000000,
                       2.03000000000000,
                       2.04000000000000,
                       2.05000000000000,
                       2.06000000000000,
                       2.07000000000000,
                       2.08000000000000,
                       2.09000000000000,
                       2.10000000000000,
                       2.11000000000000,
                       2.12000000000000,
                       2.13000000000000,
                       2.14000000000000,
                       2.15000000000000,
                       2.16000000000000,
                       2.17000000000000,
                       2.18000000000000,
                       2.19000000000000,
                       2.20000000000000,
                       2.21000000000000,
                       2.22000000000000,
                       2.23000000000000,
                       2.24000000000000,
                       2.25000000000000,
                       2.26000000000000,
                       2.27000000000000,
                       2.28000000000000,
                       2.29000000000000,
                       2.30000000000000,
                       2.31000000000000,
                       2.32000000000000,
                       2.33000000000000,
                       2.34000000000000,
                       2.35000000000000,
                       2.36000000000000,
                       2.37000000000000,
                       2.38000000000000,
                       2.39000000000000,
                       2.40000000000000,
                       2.41000000000000,
                       2.42000000000000,
                       2.43000000000000,
                       2.44000000000000,
                       2.45000000000000,
                       2.46000000000000,
                       2.47000000000000,
                       2.48000000000000,
                       2.49000000000000,
                       2.50000000000000,
                       2.51000000000000,
                       2.52000000000000,
                       2.53000000000000,
                       2.54000000000000,
                       2.55000000000000,
                       2.56000000000000,
                       2.57000000000000,
                       2.58000000000000,
                       2.59000000000000,
                       2.60000000000000,
                       2.61000000000000,
                       2.62000000000000,
                       2.63000000000000,
                       2.64000000000000,
                       2.65000000000000,
                       2.66000000000000,
                       2.67000000000000,
                       2.68000000000000,
                       2.69000000000000,
                       2.70000000000000,
                       2.71000000000000,
                       2.72000000000000,
                       2.73000000000000,
                       2.74000000000000,
                       2.75000000000000,
                       2.76000000000000,
                       2.77000000000000,
                       2.78000000000000,
                       2.79000000000000,
                       2.80000000000000,
                       2.81000000000000,
                       2.82000000000000,
                       2.83000000000000,
                       2.84000000000000,
                       2.85000000000000,
                       2.86000000000000,
                       2.87000000000000,
                       2.88000000000000,
                       2.89000000000000,
                       2.90000000000000,
                       2.91000000000000,
                       2.92000000000000,
                       2.93000000000000,
                       2.94000000000000,
                       2.95000000000000,
                       2.96000000000000,
                       2.97000000000000,
                       2.98000000000000,
                       2.99000000000000,
                       3};
  t_real pswf_values[] = {0,
                          2.86633764298305e-05,
                          6.10846400522510e-05,
                          9.74867114534197e-05,
                          0.000138099275436011,
                          0.000183158834715579,
                          0.000232908759839985,
                          0.000287599335260365,
                          0.000347487801482491,
                          0.000412838393227599,
                          0.000483922373532247,
                          0.000561018063717058,
                          0.000644410869154929,
                          0.000734393300769595,
                          0.000831264992196336,
                          0.000935332712537120,
                          0.00104691037464341,
                          0.00116631903886053,
                          0.00129388691216873,
                          0.00142994934265658,
                          0.00157484880926400,
                          0.00172893490673290,
                          0.00189256432570485,
                          0.00206610082790644,
                          0.00224991521636458,
                          0.00244438530059500,
                          0.00264989585670940,
                          0.00286683858238768,
                          0.00309561204666389,
                          0.00333662163447619,
                          0.00359027948593289,
                          0.00385700443024882,
                          0.00413722191430810,
                          0.00443136392581194,
                          0.00473986891097181,
                          0.00506318168671114,
                          0.00540175334734086,
                          0.00575604116567666,
                          0.00612650848856852,
                          0.00651362462681571,
                          0.00691786473944308,
                          0.00733970971231783,
                          0.00777964603108813,
                          0.00823816564842889,
                          0.00871576584558228,
                          0.00921294908818468,
                          0.00973022287637387,
                          0.0102680995891754,
                          0.0108270963231686,
                          0.0114077347254377,
                          0.0120105408208164,
                          0.0126360448334390,
                          0.0132847810026130,
                          0.0139572873930336,
                          0.0146541056993654,
                          0.0153757810452170,
                          0.0161228617765427,
                          0.0168958992495060,
                          0.0176954476128461,
                          0.0185220635847929,
                          0.0193763062245777,
                          0.0202587366985953,
                          0.0211699180412740,
                          0.0221104149107163,
                          0.0230807933391785,
                          0.0240816204784586,
                          0.0251134643402719,
                          0.0261768935316915,
                          0.0272724769857440,
                          0.0284007836872455,
                          0.0295623823939785,
                          0.0307578413533050,
                          0.0319877280143243,
                          0.0332526087356823,
                          0.0345530484891478,
                          0.0358896256250000,
                          0.0372628731910778,
                          0.0386733693229407,
                          0.0401216707418761,
                          0.0416083313712472,
                          0.0431339020245731,
                          0.0446989300905778,
                          0.0463039592153665,
                          0.0479495289818953,
                          0.0496361745868978,
                          0.0513644265154379,
                          0.0531348102132585,
                          0.0549478457571002,
                          0.0568040475231635,
                          0.0587039238538933,
                          0.0606479767232639,
                          0.0626367014007467,
                          0.0646705861141433,
                          0.0667501117114703,
                          0.0688757513220816,
                          0.0710479700172189,
                          0.0732672244701787,
                          0.0755339626162902,
                          0.0778486233128962,
                          0.0802116359995324,
                          0.0826234203585025,
                          0.0850843859760454,
                          0.0875949320042938,
                          0.0901554468242243,
                          0.0927663077097996,
                          0.0954278804935044,
                          0.0981405192334776,
                          0.100904565882443,
                          0.103720349958646,
                          0.106588188218989,
                          0.109508384334590,
                          0.112481228568947,
                          0.115506997458931,
                          0.118585953498792,
                          0.121718344827413,
                          0.124904404918983,
                          0.128144352277314,
                          0.131438390134002,
                          0.134786706150630,
                          0.138189472125222,
                          0.141646843703139,
                          0.145158960092630,
                          0.148725943785234,
                          0.152347900281221,
                          0.156024917820285,
                          0.159757067117674,
                          0.163544401105951,
                          0.167386954682584,
                          0.171284744463553,
                          0.175237768543162,
                          0.179246006260240,
                          0.183309417970919,
                          0.187427944828171,
                          0.191601508568284,
                          0.195830011304447,
                          0.200113335327629,
                          0.204451342914916,
                          0.208843876145476,
                          0.213290756724322,
                          0.217791785814026,
                          0.222346743874551,
                          0.226955390511359,
                          0.231617464331931,
                          0.236332682810872,
                          0.241100742163720,
                          0.245921317229620,
                          0.250794061362988,
                          0.255718606334302,
                          0.260694562240146,
                          0.265721517422636,
                          0.270799038398339,
                          0.275926669796807,
                          0.281103934308841,
                          0.286330332644570,
                          0.291605343501472,
                          0.296928423542418,
                          0.302299007383825,
                          0.307716507594023,
                          0.313180314701895,
                          0.318689797215877,
                          0.324244301653390,
                          0.329843152580757,
                          0.335485652663674,
                          0.341171082728286,
                          0.346898701832912,
                          0.352667747350463,
                          0.358477435061591,
                          0.364326959258596,
                          0.370215492860113,
                          0.376142187536609,
                          0.382106173846686,
                          0.388106561384206,
                          0.394142438936235,
                          0.400212874651809,
                          0.406316916221489,
                          0.412453591067706,
                          0.418621906545869,
                          0.424820850156189,
                          0.431049389766206,
                          0.437306473843954,
                          0.443591031701730,
                          0.449901973750396,
                          0.456238191764172,
                          0.462598559155827,
                          0.468981931262213,
                          0.475387145640055,
                          0.481813022371900,
                          0.488258364382151,
                          0.494721957763063,
                          0.501202572110616,
                          0.507698960870139,
                          0.514209861691576,
                          0.520733996794259,
                          0.527270073341071,
                          0.533816783821854,
                          0.540372806445917,
                          0.546936805543511,
                          0.553507431976100,
                          0.560083323555276,
                          0.566663105470158,
                          0.573245390723099,
                          0.579828780573518,
                          0.586411864989690,
                          0.592993223108302,
                          0.599571423701575,
                          0.606145025651764,
                          0.612712578432832,
                          0.619272622599090,
                          0.625823690280586,
                          0.632364305685044,
                          0.638892985606103,
                          0.645408239937661,
                          0.651908572194071,
                          0.658392480035969,
                          0.664858455801489,
                          0.671304987042621,
                          0.677730557066477,
                          0.684133645481194,
                          0.690512728746252,
                          0.696866280726914,
                          0.703192773252561,
                          0.709490676678633,
                          0.715758460451928,
                          0.721994593678982,
                          0.728197545697252,
                          0.734365786648846,
                          0.740497788056505,
                          0.746592023401566,
                          0.752646968703630,
                          0.758661103101640,
                          0.764632909436099,
                          0.770560874832130,
                          0.776443491283096,
                          0.782279256234493,
                          0.788066673167824,
                          0.793804252184162,
                          0.799490510587115,
                          0.805123973464901,
                          0.810703174271229,
                          0.816226655404721,
                          0.821692968786546,
                          0.827100676436013,
                          0.832448351043794,
                          0.837734576542518,
                          0.842957948674420,
                          0.848117075555774,
                          0.853210578237811,
                          0.858237091263836,
                          0.863195263222258,
                          0.868083757295261,
                          0.872901251802801,
                          0.877646440741690,
                          0.882318034319456,
                          0.886914759482714,
                          0.891435360439777,
                          0.895878599177233,
                          0.900243255970207,
                          0.904528129886074,
                          0.908732039281311,
                          0.912853822291283,
                          0.916892337312664,
                          0.920846463478258,
                          0.924715101123971,
                          0.928497172247691,
                          0.932191620959817,
                          0.935797413925227,
                          0.939313540796430,
                          0.942739014637679,
                          0.946072872339833,
                          0.949314175025725,
                          0.952462008445847,
                          0.955515483364125,
                          0.958473735933589,
                          0.961335928061722,
                          0.964101247765328,
                          0.966768909514690,
                          0.969338154566861,
                          0.971808251287910,
                          0.974178495463931,
                          0.976448210600678,
                          0.978616748211639,
                          0.980683488094415,
                          0.982647838595249,
                          0.984509236861565,
                          0.986267149082369,
                          0.987921070716417,
                          0.989470526707983,
                          0.990915071690158,
                          0.992254290175525,
                          0.993487796734152,
                          0.994615236158772,
                          0.995636283617083,
                          0.996550644791080,
                          0.997358056003338,
                          0.998058284330194,
                          0.998651127701744,
                          0.999136414988631,
                          0.999514006075548,
                          0.999783791921433,
                          0.999945694606328,
                          0.999999667364857,
                          0.999945694606328,
                          0.999783791921433,
                          0.999514006075548,
                          0.999136414988631,
                          0.998651127701744,
                          0.998058284330194,
                          0.997358056003338,
                          0.996550644791080,
                          0.995636283617083,
                          0.994615236158772,
                          0.993487796734152,
                          0.992254290175525,
                          0.990915071690158,
                          0.989470526707983,
                          0.987921070716417,
                          0.986267149082369,
                          0.984509236861565,
                          0.982647838595249,
                          0.980683488094415,
                          0.978616748211639,
                          0.976448210600678,
                          0.974178495463931,
                          0.971808251287910,
                          0.969338154566861,
                          0.966768909514690,
                          0.964101247765328,
                          0.961335928061722,
                          0.958473735933589,
                          0.955515483364125,
                          0.952462008445847,
                          0.949314175025725,
                          0.946072872339833,
                          0.942739014637679,
                          0.939313540796430,
                          0.935797413925227,
                          0.932191620959817,
                          0.928497172247691,
                          0.924715101123971,
                          0.920846463478258,
                          0.916892337312664,
                          0.912853822291283,
                          0.908732039281311,
                          0.904528129886074,
                          0.900243255970207,
                          0.895878599177233,
                          0.891435360439777,
                          0.886914759482714,
                          0.882318034319456,
                          0.877646440741690,
                          0.872901251802801,
                          0.868083757295261,
                          0.863195263222258,
                          0.858237091263836,
                          0.853210578237811,
                          0.848117075555774,
                          0.842957948674420,
                          0.837734576542518,
                          0.832448351043794,
                          0.827100676436013,
                          0.821692968786546,
                          0.816226655404721,
                          0.810703174271229,
                          0.805123973464901,
                          0.799490510587115,
                          0.793804252184162,
                          0.788066673167824,
                          0.782279256234493,
                          0.776443491283096,
                          0.770560874832130,
                          0.764632909436099,
                          0.758661103101640,
                          0.752646968703630,
                          0.746592023401566,
                          0.740497788056505,
                          0.734365786648846,
                          0.728197545697252,
                          0.721994593678982,
                          0.715758460451928,
                          0.709490676678633,
                          0.703192773252561,
                          0.696866280726914,
                          0.690512728746252,
                          0.684133645481194,
                          0.677730557066477,
                          0.671304987042621,
                          0.664858455801489,
                          0.658392480035969,
                          0.651908572194071,
                          0.645408239937661,
                          0.638892985606103,
                          0.632364305685044,
                          0.625823690280586,
                          0.619272622599090,
                          0.612712578432832,
                          0.606145025651764,
                          0.599571423701575,
                          0.592993223108302,
                          0.586411864989690,
                          0.579828780573518,
                          0.573245390723099,
                          0.566663105470158,
                          0.560083323555276,
                          0.553507431976100,
                          0.546936805543511,
                          0.540372806445917,
                          0.533816783821854,
                          0.527270073341071,
                          0.520733996794259,
                          0.514209861691576,
                          0.507698960870139,
                          0.501202572110616,
                          0.494721957763063,
                          0.488258364382151,
                          0.481813022371900,
                          0.475387145640055,
                          0.468981931262213,
                          0.462598559155827,
                          0.456238191764172,
                          0.449901973750396,
                          0.443591031701730,
                          0.437306473843954,
                          0.431049389766206,
                          0.424820850156189,
                          0.418621906545869,
                          0.412453591067706,
                          0.406316916221489,
                          0.400212874651809,
                          0.394142438936235,
                          0.388106561384206,
                          0.382106173846686,
                          0.376142187536609,
                          0.370215492860113,
                          0.364326959258596,
                          0.358477435061591,
                          0.352667747350463,
                          0.346898701832912,
                          0.341171082728286,
                          0.335485652663674,
                          0.329843152580757,
                          0.324244301653390,
                          0.318689797215877,
                          0.313180314701895,
                          0.307716507594023,
                          0.302299007383825,
                          0.296928423542418,
                          0.291605343501472,
                          0.286330332644570,
                          0.281103934308841,
                          0.275926669796807,
                          0.270799038398339,
                          0.265721517422636,
                          0.260694562240146,
                          0.255718606334302,
                          0.250794061362988,
                          0.245921317229620,
                          0.241100742163720,
                          0.236332682810872,
                          0.231617464331931,
                          0.226955390511359,
                          0.222346743874551,
                          0.217791785814026,
                          0.213290756724322,
                          0.208843876145476,
                          0.204451342914916,
                          0.200113335327629,
                          0.195830011304447,
                          0.191601508568284,
                          0.187427944828171,
                          0.183309417970919,
                          0.179246006260240,
                          0.175237768543162,
                          0.171284744463553,
                          0.167386954682584,
                          0.163544401105951,
                          0.159757067117674,
                          0.156024917820285,
                          0.152347900281221,
                          0.148725943785234,
                          0.145158960092630,
                          0.141646843703139,
                          0.138189472125222,
                          0.134786706150630,
                          0.131438390134002,
                          0.128144352277314,
                          0.124904404918983,
                          0.121718344827413,
                          0.118585953498792,
                          0.115506997458931,
                          0.112481228568947,
                          0.109508384334590,
                          0.106588188218989,
                          0.103720349958646,
                          0.100904565882443,
                          0.0981405192334776,
                          0.0954278804935044,
                          0.0927663077097996,
                          0.0901554468242243,
                          0.0875949320042938,
                          0.0850843859760454,
                          0.0826234203585025,
                          0.0802116359995324,
                          0.0778486233128962,
                          0.0755339626162902,
                          0.0732672244701787,
                          0.0710479700172189,
                          0.0688757513220816,
                          0.0667501117114703,
                          0.0646705861141433,
                          0.0626367014007467,
                          0.0606479767232639,
                          0.0587039238538933,
                          0.0568040475231635,
                          0.0549478457571002,
                          0.0531348102132585,
                          0.0513644265154379,
                          0.0496361745868978,
                          0.0479495289818953,
                          0.0463039592153665,
                          0.0446989300905778,
                          0.0431339020245731,
                          0.0416083313712472,
                          0.0401216707418761,
                          0.0386733693229407,
                          0.0372628731910778,
                          0.0358896256250000,
                          0.0345530484891478,
                          0.0332526087356823,
                          0.0319877280143243,
                          0.0307578413533050,
                          0.0295623823939785,
                          0.0284007836872455,
                          0.0272724769857440,
                          0.0261768935316915,
                          0.0251134643402719,
                          0.0240816204784586,
                          0.0230807933391785,
                          0.0221104149107163,
                          0.0211699180412740,
                          0.0202587366985953,
                          0.0193763062245777,
                          0.0185220635847929,
                          0.0176954476128461,
                          0.0168958992495060,
                          0.0161228617765427,
                          0.0153757810452170,
                          0.0146541056993654,
                          0.0139572873930336,
                          0.0132847810026130,
                          0.0126360448334390,
                          0.0120105408208164,
                          0.0114077347254377,
                          0.0108270963231686,
                          0.0102680995891754,
                          0.00973022287637387,
                          0.00921294908818468,
                          0.00871576584558228,
                          0.00823816564842889,
                          0.00777964603108813,
                          0.00733970971231783,
                          0.00691786473944308,
                          0.00651362462681571,
                          0.00612650848856852,
                          0.00575604116567666,
                          0.00540175334734086,
                          0.00506318168671114,
                          0.00473986891097181,
                          0.00443136392581194,
                          0.00413722191430810,
                          0.00385700443024882,
                          0.00359027948593289,
                          0.00333662163447619,
                          0.00309561204666389,
                          0.00286683858238768,
                          0.00264989585670940,
                          0.00244438530059500,
                          0.00224991521636458,
                          0.00206610082790644,
                          0.00189256432570485,
                          0.00172893490673290,
                          0.00157484880926400,
                          0.00142994934265658,
                          0.00129388691216873,
                          0.00116631903886053,
                          0.00104691037464341,
                          0.000935332712537120,
                          0.000831264992196336,
                          0.000734393300769595,
                          0.000644410869154929,
                          0.000561018063717058,
                          0.000483922373532247,
                          0.000412838393227599,
                          0.000347487801482491,
                          0.000287599335260365,
                          0.000232908759839985,
                          0.000183158834715579,
                          0.000138099275436011,
                          9.74867114534197e-05,
                          6.10846400522510e-05,
                          2.86633764298305e-05,
                          0};
  t_real kb_values[] = {7.43469443502669e-06,
                        1.00805758527258e-05,
                        1.31611100296223e-05,
                        1.67204973730424e-05,
                        2.08059363905017e-05,
                        2.54677549797957e-05,
                        3.07595444911051e-05,
                        3.67382965053836e-05,
                        4.34645422677190e-05,
                        5.10024947086680e-05,
                        5.94201929808341e-05,
                        6.87896494321026e-05,
                        7.91869989310889e-05,
                        9.06926504543629e-05,
                        0.000103391440839045,
                        0.000117372790598282,
                        0.000132730861691048,
                        0.000149564717131562,
                        0.000167978482317502,
                        0.000188081507949999,
                        0.000209988534412247,
                        0.000233819857467383,
                        0.000259701495130145,
                        0.000287765355560641,
                        0.000318149405822505,
                        0.000350997841341567,
                        0.000386461255895199,
                        0.000424696811956460,
                        0.000465868411211272,
                        0.000510146865061051,
                        0.000557710064917393,
                        0.000608743152089792,
                        0.000663438687061795,
                        0.000721996817945560,
                        0.000784625447899411,
                        0.000851540401287862,
                        0.000922965588358437,
                        0.000999133168204828,
                        0.00108028370978106,
                        0.00116666635072691,
                        0.00125853895376027,
                        0.00135616826038822,
                        0.00145983004168424,
                        0.00156980924587551,
                        0.00168640014248058,
                        0.00180990646273441,
                        0.00194064153603439,
                        0.00207892842213892,
                        0.00222510003884643,
                        0.00237949928488150,
                        0.00254247915771185,
                        0.00271440286601900,
                        0.00289564393654305,
                        0.00308658631502198,
                        0.00328762446094390,
                        0.00349916343583111,
                        0.00372161898477411,
                        0.00395541761093375,
                        0.00420099664273084,
                        0.00445880429344256,
                        0.00472929971292675,
                        0.00501295303119638,
                        0.00531024539356870,
                        0.00562166898711538,
                        0.00594772705814315,
                        0.00628893392043690,
                        0.00664581495400105,
                        0.00701890659403844,
                        0.00740875630991058,
                        0.00781592257382674,
                        0.00824097481901578,
                        0.00868449338713837,
                        0.00914706946470409,
                        0.00962930500826366,
                        0.0101318126581527,
                        0.0106552156405711,
                        0.0112001476577889,
                        0.0117672527662767,
                        0.0123571852425686,
                        0.0129706094366715,
                        0.0136081996128458,
                        0.0142706397775905,
                        0.0149586234946749,
                        0.0156728536870719,
                        0.0164140424256529,
                        0.0171829107045218,
                        0.0179801882028699,
                        0.0188066130332503,
                        0.0196629314761796,
                        0.0205498977009858,
                        0.0214682734728370,
                        0.0224188278458950,
                        0.0234023368425520,
                        0.0244195831187241,
                        0.0254713556151832,
                        0.0265584491949310,
                        0.0276816642666247,
                        0.0288418063940844,
                        0.0300396858919227,
                        0.0312761174073554,
                        0.0325519194882648,
                        0.0338679141376024,
                        0.0352249263542347,
                        0.0366237836603505,
                        0.0380653156155631,
                        0.0395503533178591,
                        0.0410797288915580,
                        0.0426542749624655,
                        0.0442748241204196,
                        0.0459422083694431,
                        0.0476572585657329,
                        0.0494208038437357,
                        0.0512336710305704,
                        0.0530966840490786,
                        0.0550106633098010,
                        0.0569764250921886,
                        0.0589947809153793,
                        0.0610665368988864,
                        0.0631924931135554,
                        0.0653734429231685,
                        0.0676101723170910,
                        0.0699034592343620,
                        0.0722540728796589,
                        0.0746627730315699,
                        0.0771303093436293,
                        0.0796574206385861,
                        0.0822448341963846,
                        0.0848932650363573,
                        0.0876034151941392,
                        0.0903759729938291,
                        0.0932116123159342,
                        0.0961109918616499,
                        0.0990747544140367,
                        0.102103526096669,
                        0.105197915630347,
                        0.108358513588461,
                        0.111585891651624,
                        0.114880601862191,
                        0.118243175879295,
                        0.121674124235024,
                        0.125173935592412,
                        0.128743076005881,
                        0.132381988184787,
                        0.136091090760777,
                        0.139870777559591,
                        0.143721416878019,
                        0.147643350766696,
                        0.151636894319412,
                        0.155702334969659,
                        0.159839931795092,
                        0.164049914830625,
                        0.168332484390842,
                        0.172687810402464,
                        0.177116031747545,
                        0.181617255618122,
                        0.186191556883019,
                        0.190838977467515,
                        0.195559525746569,
                        0.200353175952304,
                        0.205219867596454,
                        0.210159504908447,
                        0.215171956289834,
                        0.220257053785717,
                        0.225414592573868,
                        0.230644330472202,
                        0.235945987465250,
                        0.241319245250294,
                        0.246763746803798,
                        0.252279095968757,
                        0.257864857063589,
                        0.263520554513175,
                        0.269245672502630,
                        0.275039654654401,
                        0.280901903729233,
                        0.286831781351579,
                        0.292828607759967,
                        0.298891661582857,
                        0.305020179640481,
                        0.311213356773146,
                        0.317470345696486,
                        0.323790256884076,
                        0.330172158477869,
                        0.336615076226841,
                        0.343117993454229,
                        0.349679851053751,
                        0.356299547515114,
                        0.362975938979164,
                        0.369707839322947,
                        0.376494020274986,
                        0.383333211560984,
                        0.390224101080214,
                        0.397165335112778,
                        0.404155518557911,
                        0.411193215203480,
                        0.418276948026807,
                        0.425405199526899,
                        0.432576412088166,
                        0.439788988375669,
                        0.447041291761895,
                        0.454331646785080,
                        0.461658339638999,
                        0.469019618694186,
                        0.476413695050466,
                        0.483838743120690,
                        0.491292901245497,
                        0.498774272338951,
                        0.506280924564807,
                        0.513810892043182,
                        0.521362175587379,
                        0.528932743470522,
                        0.536520532221727,
                        0.544123447451410,
                        0.551739364705383,
                        0.559366130347294,
                        0.567001562469006,
                        0.574643451828413,
                        0.582289562814228,
                        0.589937634437199,
                        0.597585381347206,
                        0.605230494875692,
                        0.612870644102763,
                        0.620503476948407,
                        0.628126621287106,
                        0.635737686085219,
                        0.643334262560384,
                        0.650913925362262,
                        0.658474233773817,
                        0.666012732932407,
                        0.673526955069859,
                        0.681014420770724,
                        0.688472640247859,
                        0.695899114634501,
                        0.703291337291929,
                        0.710646795131823,
                        0.717962969952412,
                        0.725237339787475,
                        0.732467380267226,
                        0.739650565990148,
                        0.746784371904763,
                        0.753866274700369,
                        0.760893754205708,
                        0.767864294794583,
                        0.774775386797333,
                        0.781624527917188,
                        0.788409224650396,
                        0.795126993709089,
                        0.801775363445819,
                        0.808351875278672,
                        0.814854085115884,
                        0.821279564778882,
                        0.827625903422644,
                        0.833890708952285,
                        0.840071609434801,
                        0.846166254504825,
                        0.852172316763347,
                        0.858087493168273,
                        0.863909506415738,
                        0.869636106311106,
                        0.875265071128524,
                        0.880794208958001,
                        0.886221359038912,
                        0.891544393078866,
                        0.896761216556889,
                        0.901869770009860,
                        0.906868030301180,
                        0.911754011870646,
                        0.916525767964508,
                        0.921181391844709,
                        0.925719017976353,
                        0.930136823192383,
                        0.934433027834571,
                        0.938605896869844,
                        0.942653740981037,
                        0.946574917631207,
                        0.950367832100585,
                        0.954030938495344,
                        0.957562740727331,
                        0.960961793463961,
                        0.964226703047482,
                        0.967356128382838,
                        0.970348781793416,
                        0.973203429843927,
                        0.975918894129774,
                        0.978494052032215,
                        0.980927837438710,
                        0.983219241427842,
                        0.985367312918236,
                        0.987371159280924,
                        0.989229946914677,
                        0.990942901783765,
                        0.992509309917736,
                        0.993928517872787,
                        0.995199933154319,
                        0.996323024600329,
                        0.997297322725331,
                        0.998122420024500,
                        0.998797971237777,
                        0.999323693573725,
                        0.999699366892946,
                        0.999924833850901,
                        1,
                        0.999924833850901,
                        0.999699366892946,
                        0.999323693573725,
                        0.998797971237777,
                        0.998122420024500,
                        0.997297322725331,
                        0.996323024600329,
                        0.995199933154319,
                        0.993928517872787,
                        0.992509309917736,
                        0.990942901783765,
                        0.989229946914677,
                        0.987371159280924,
                        0.985367312918236,
                        0.983219241427842,
                        0.980927837438710,
                        0.978494052032215,
                        0.975918894129774,
                        0.973203429843927,
                        0.970348781793416,
                        0.967356128382838,
                        0.964226703047482,
                        0.960961793463961,
                        0.957562740727331,
                        0.954030938495344,
                        0.950367832100585,
                        0.946574917631207,
                        0.942653740981037,
                        0.938605896869844,
                        0.934433027834571,
                        0.930136823192383,
                        0.925719017976353,
                        0.921181391844709,
                        0.916525767964508,
                        0.911754011870646,
                        0.906868030301180,
                        0.901869770009860,
                        0.896761216556889,
                        0.891544393078866,
                        0.886221359038912,
                        0.880794208958001,
                        0.875265071128524,
                        0.869636106311106,
                        0.863909506415738,
                        0.858087493168273,
                        0.852172316763347,
                        0.846166254504825,
                        0.840071609434801,
                        0.833890708952285,
                        0.827625903422644,
                        0.821279564778882,
                        0.814854085115884,
                        0.808351875278672,
                        0.801775363445819,
                        0.795126993709089,
                        0.788409224650396,
                        0.781624527917188,
                        0.774775386797333,
                        0.767864294794583,
                        0.760893754205708,
                        0.753866274700369,
                        0.746784371904763,
                        0.739650565990148,
                        0.732467380267226,
                        0.725237339787475,
                        0.717962969952412,
                        0.710646795131823,
                        0.703291337291929,
                        0.695899114634501,
                        0.688472640247859,
                        0.681014420770724,
                        0.673526955069859,
                        0.666012732932407,
                        0.658474233773817,
                        0.650913925362262,
                        0.643334262560384,
                        0.635737686085219,
                        0.628126621287106,
                        0.620503476948407,
                        0.612870644102763,
                        0.605230494875692,
                        0.597585381347206,
                        0.589937634437199,
                        0.582289562814228,
                        0.574643451828413,
                        0.567001562469006,
                        0.559366130347294,
                        0.551739364705383,
                        0.544123447451410,
                        0.536520532221727,
                        0.528932743470522,
                        0.521362175587379,
                        0.513810892043182,
                        0.506280924564807,
                        0.498774272338951,
                        0.491292901245497,
                        0.483838743120690,
                        0.476413695050466,
                        0.469019618694186,
                        0.461658339638999,
                        0.454331646785080,
                        0.447041291761895,
                        0.439788988375669,
                        0.432576412088166,
                        0.425405199526899,
                        0.418276948026807,
                        0.411193215203480,
                        0.404155518557911,
                        0.397165335112778,
                        0.390224101080214,
                        0.383333211560984,
                        0.376494020274986,
                        0.369707839322947,
                        0.362975938979164,
                        0.356299547515114,
                        0.349679851053751,
                        0.343117993454229,
                        0.336615076226841,
                        0.330172158477869,
                        0.323790256884076,
                        0.317470345696486,
                        0.311213356773146,
                        0.305020179640481,
                        0.298891661582857,
                        0.292828607759967,
                        0.286831781351579,
                        0.280901903729233,
                        0.275039654654401,
                        0.269245672502630,
                        0.263520554513175,
                        0.257864857063589,
                        0.252279095968757,
                        0.246763746803798,
                        0.241319245250294,
                        0.235945987465250,
                        0.230644330472202,
                        0.225414592573868,
                        0.220257053785717,
                        0.215171956289834,
                        0.210159504908447,
                        0.205219867596454,
                        0.200353175952304,
                        0.195559525746569,
                        0.190838977467515,
                        0.186191556883019,
                        0.181617255618122,
                        0.177116031747545,
                        0.172687810402464,
                        0.168332484390842,
                        0.164049914830625,
                        0.159839931795092,
                        0.155702334969659,
                        0.151636894319412,
                        0.147643350766696,
                        0.143721416878019,
                        0.139870777559591,
                        0.136091090760777,
                        0.132381988184787,
                        0.128743076005881,
                        0.125173935592412,
                        0.121674124235024,
                        0.118243175879295,
                        0.114880601862191,
                        0.111585891651624,
                        0.108358513588461,
                        0.105197915630347,
                        0.102103526096669,
                        0.0990747544140367,
                        0.0961109918616499,
                        0.0932116123159342,
                        0.0903759729938291,
                        0.0876034151941392,
                        0.0848932650363573,
                        0.0822448341963846,
                        0.0796574206385861,
                        0.0771303093436293,
                        0.0746627730315699,
                        0.0722540728796589,
                        0.0699034592343620,
                        0.0676101723170910,
                        0.0653734429231685,
                        0.0631924931135554,
                        0.0610665368988864,
                        0.0589947809153793,
                        0.0569764250921886,
                        0.0550106633098010,
                        0.0530966840490786,
                        0.0512336710305704,
                        0.0494208038437357,
                        0.0476572585657329,
                        0.0459422083694431,
                        0.0442748241204196,
                        0.0426542749624655,
                        0.0410797288915580,
                        0.0395503533178591,
                        0.0380653156155631,
                        0.0366237836603505,
                        0.0352249263542347,
                        0.0338679141376024,
                        0.0325519194882648,
                        0.0312761174073554,
                        0.0300396858919227,
                        0.0288418063940844,
                        0.0276816642666247,
                        0.0265584491949310,
                        0.0254713556151832,
                        0.0244195831187241,
                        0.0234023368425520,
                        0.0224188278458950,
                        0.0214682734728370,
                        0.0205498977009858,
                        0.0196629314761796,
                        0.0188066130332503,
                        0.0179801882028699,
                        0.0171829107045218,
                        0.0164140424256529,
                        0.0156728536870719,
                        0.0149586234946749,
                        0.0142706397775905,
                        0.0136081996128458,
                        0.0129706094366715,
                        0.0123571852425686,
                        0.0117672527662767,
                        0.0112001476577889,
                        0.0106552156405711,
                        0.0101318126581527,
                        0.00962930500826366,
                        0.00914706946470409,
                        0.00868449338713837,
                        0.00824097481901578,
                        0.00781592257382674,
                        0.00740875630991058,
                        0.00701890659403844,
                        0.00664581495400105,
                        0.00628893392043690,
                        0.00594772705814315,
                        0.00562166898711538,
                        0.00531024539356870,
                        0.00501295303119638,
                        0.00472929971292675,
                        0.00445880429344256,
                        0.00420099664273084,
                        0.00395541761093375,
                        0.00372161898477411,
                        0.00349916343583111,
                        0.00328762446094390,
                        0.00308658631502198,
                        0.00289564393654305,
                        0.00271440286601900,
                        0.00254247915771185,
                        0.00237949928488150,
                        0.00222510003884643,
                        0.00207892842213892,
                        0.00194064153603439,
                        0.00180990646273441,
                        0.00168640014248058,
                        0.00156980924587551,
                        0.00145983004168424,
                        0.00135616826038822,
                        0.00125853895376027,
                        0.00116666635072691,
                        0.00108028370978106,
                        0.000999133168204828,
                        0.000922965588358437,
                        0.000851540401287862,
                        0.000784625447899411,
                        0.000721996817945560,
                        0.000663438687061795,
                        0.000608743152089792,
                        0.000557710064917393,
                        0.000510146865061051,
                        0.000465868411211272,
                        0.000424696811956460,
                        0.000386461255895199,
                        0.000350997841341567,
                        0.000318149405822505,
                        0.000287765355560641,
                        0.000259701495130145,
                        0.000233819857467383,
                        0.000209988534412247,
                        0.000188081507949999,
                        0.000167978482317502,
                        0.000149564717131562,
                        0.000132730861691048,
                        0.000117372790598282,
                        0.000103391440839045,
                        9.06926504543629e-05,
                        7.91869989310889e-05,
                        6.87896494321026e-05,
                        5.94201929808341e-05,
                        5.10024947086680e-05,
                        4.34645422677190e-05,
                        3.67382965053836e-05,
                        3.07595444911051e-05,
                        2.54677549797957e-05,
                        2.08059363905017e-05,
                        1.67204973730424e-05,
                        1.31611100296223e-05,
                        1.00805758527258e-05,
                        7.43469443502669e-06};
  t_real gauss_values[] = {0.000699879769063488,
                           0.000734550335813170,
                           0.000770813962412435,
                           0.000808737299262344,
                           0.000848389458032121,
                           0.000889842087032812,
                           0.000933169448195548,
                           0.000978448495656672,
                           0.00102575895595036,
                           0.00107518340980737,
                           0.00112680737555679,
                           0.00118071939412560,
                           0.00123701111562861,
                           0.00129577738753945,
                           0.00135711634443075,
                           0.00142112949926929,
                           0.00148792183624960,
                           0.00155760190514682,
                           0.00163028191716677,
                           0.00170607784226887,
                           0.00178510950793429,
                           0.00186750069934881,
                           0.00195337926096717,
                           0.00204287719942195,
                           0.00213613078773737,
                           0.00223328067080457,
                           0.00233447197207188,
                           0.00243985440139970,
                           0.00254958236402629,
                           0.00266381507058677,
                           0.00278271664812404,
                           0.00290645625202629,
                           0.00303520817882174,
                           0.00316915197975733,
                           0.00330847257508363,
                           0.00345336036896434,
                           0.00360401136492399,
                           0.00376062728174355,
                           0.00392341566970854,
                           0.00409259002711025,
                           0.00426836991689576,
                           0.00445098108335776,
                           0.00464065556875038,
                           0.00483763182971278,
                           0.00504215485337676,
                           0.00525447627303051,
                           0.00547485448320485,
                           0.00570355475404418,
                           0.00594084934481857,
                           0.00618701761642908,
                           0.00644234614275277,
                           0.00670712882066924,
                           0.00698166697860525,
                           0.00726626948342904,
                           0.00756125284552083,
                           0.00786694132184118,
                           0.00818366701681348,
                           0.00851176998083246,
                           0.00885159830620527,
                           0.00920350822032722,
                           0.00956786417588879,
                           0.00994503893790690,
                           0.0103354136673680,
                           0.0107393780012658,
                           0.0111573301288138,
                           0.0115896768636053,
                           0.0120368337114932,
                           0.0124992249339547,
                           0.0129772836067038,
                           0.0134714516733111,
                           0.0139821799935847,
                           0.0145099283864656,
                           0.0150551656671856,
                           0.0156183696784335,
                           0.0162000273152735,
                           0.0168006345435551,
                           0.0174206964115545,
                           0.0180607270545826,
                           0.0187212496922949,
                           0.0194027966184369,
                           0.0201059091827567,
                           0.0208311377648161,
                           0.0215790417394323,
                           0.0223501894334788,
                           0.0231451580737793,
                           0.0239645337258231,
                           0.0248089112230371,
                           0.0256788940863468,
                           0.0265750944337617,
                           0.0274981328797241,
                           0.0284486384239596,
                           0.0294272483295739,
                           0.0304346079901428,
                           0.0314713707855435,
                           0.0325381979262851,
                           0.0336357582860962,
                           0.0347647282225337,
                           0.0359257913853865,
                           0.0371196385126465,
                           0.0383469672138356,
                           0.0396084817404756,
                           0.0409048927435024,
                           0.0422369170174299,
                           0.0436052772310812,
                           0.0450107016447105,
                           0.0464539238133536,
                           0.0479356822762494,
                           0.0494567202321930,
                           0.0510177852006850,
                           0.0526196286687601,
                           0.0542630057233863,
                           0.0559486746693428,
                           0.0576773966324938,
                           0.0594499351483950,
                           0.0612670557361800,
                           0.0631295254576919,
                           0.0650381124618393,
                           0.0669935855141739,
                           0.0689967135117022,
                           0.0710482649829623,
                           0.0731490075734142,
                           0.0752997075162107,
                           0.0775011290884327,
                           0.0797540340528955,
                           0.0820591810856499,
                           0.0844173251893211,
                           0.0868292170924514,
                           0.0892956026350324,
                           0.0918172221404328,
                           0.0943948097739498,
                           0.0970290928882350,
                           0.0997207913558665,
                           0.102470616889362,
                           0.105279272348950,
                           0.108147451038438,
                           0.111075835989545,
                           0.114065099235078,
                           0.117115901071369,
                           0.120228889310397,
                           0.123404698522063,
                           0.126643949267090,
                           0.129947247321051,
                           0.133315182890056,
                           0.136748329818647,
                           0.140247244790477,
                           0.143812466522364,
                           0.147444514952346,
                           0.151143890422381,
                           0.154911072856348,
                           0.158746520934044,
                           0.162650671261893,
                           0.166623937541076,
                           0.170666709733863,
                           0.174779353228893,
                           0.178962208006213,
                           0.183215587802879,
                           0.187539779279958,
                           0.191935041191769,
                           0.196401603558241,
                           0.200939666841274,
                           0.205549401125988,
                           0.210230945307796,
                           0.214984406286217,
                           0.219809858166382,
                           0.224707341469186,
                           0.229676862351052,
                           0.234718391834299,
                           0.239831865049088,
                           0.245017180487957,
                           0.250274199273943,
                           0.255602744443307,
                           0.261002600243882,
                           0.266473511450057,
                           0.272015182695439,
                           0.277627277824195,
                           0.283309419262124,
                           0.289061187408468,
                           0.294882120049496,
                           0.300771711794874,
                           0.306729413537832,
                           0.312754631940148,
                           0.318846728942931,
                           0.325005021304212,
                           0.331228780164299,
                           0.337517230639884,
                           0.343869551447840,
                           0.350284874559649,
                           0.356762284887377,
                           0.363300820002107,
                           0.369899469885695,
                           0.376557176716720,
                           0.383272834691460,
                           0.390045289880703,
                           0.396873340123180,
                           0.403755734956380,
                           0.410691175585469,
                           0.417678314891011,
                           0.424715757476165,
                           0.431802059753978,
                           0.438935730075381,
                           0.446115228898436,
                           0.453338968999378,
                           0.460605315725911,
                           0.467912587293223,
                           0.475259055123106,
                           0.482642944226556,
                           0.490062433630159,
                           0.497515656846536,
                           0.505000702389078,
                           0.512515614331134,
                           0.520058392909812,
                           0.527626995174436,
                           0.535219335679730,
                           0.542833287223675,
                           0.550466681629998,
                           0.558117310575161,
                           0.565782926459664,
                           0.573461243323464,
                           0.581149937805200,
                           0.588846650144909,
                           0.596548985229844,
                           0.604254513682941,
                           0.611960772993456,
                           0.619665268689203,
                           0.627365475549802,
                           0.635058838860270,
                           0.642742775704245,
                           0.650414676296073,
                           0.658071905350936,
                           0.665711803492151,
                           0.673331688694706,
                           0.680928857764055,
                           0.688500587849150,
                           0.696044137988615,
                           0.703556750688933,
                           0.711035653533477,
                           0.718478060821137,
                           0.725881175233287,
                           0.733242189527750,
                           0.740558288258413,
                           0.747826649519071,
                           0.755044446710050,
                           0.762208850326119,
                           0.769317029764164,
                           0.776366155149043,
                           0.783353399176031,
                           0.790275938968212,
                           0.797130957947146,
                           0.803915647715110,
                           0.810627209947195,
                           0.817262858291488,
                           0.823819820275572,
                           0.830295339217536,
                           0.836686676139680,
                           0.842991111683067,
                           0.849205948021079,
                           0.855328510770093,
                           0.861356150895418,
                           0.867286246610587,
                           0.873116205268119,
                           0.878843465239859,
                           0.884465497784977,
                           0.889979808903757,
                           0.895383941175252,
                           0.900675475576940,
                           0.905852033284487,
                           0.910911277449748,
                           0.915850914955168,
                           0.920668698142714,
                           0.925362426515542,
                           0.929929948410593,
                           0.934369162640325,
                           0.938678020101856,
                           0.942854525351773,
                           0.946896738144925,
                           0.950802774935535,
                           0.954570810339018,
                           0.958199078552896,
                           0.961685874735289,
                           0.965029556339449,
                           0.968228544402904,
                           0.971281324789764,
                           0.974186449384856,
                           0.976942537238333,
                           0.979548275659527,
                           0.982002421258801,
                           0.984303800936268,
                           0.986451312816259,
                           0.988443927126510,
                           0.990280687021073,
                           0.991960709346049,
                           0.993483185347261,
                           0.994847381319100,
                           0.996052639193792,
                           0.997098377070437,
                           0.997984089683222,
                           0.998709348808275,
                           0.999273803608710,
                           0.999677180917468,
                           0.999919285457637,
                           1,
                           0.999919285457637,
                           0.999677180917468,
                           0.999273803608710,
                           0.998709348808275,
                           0.997984089683222,
                           0.997098377070437,
                           0.996052639193792,
                           0.994847381319100,
                           0.993483185347261,
                           0.991960709346049,
                           0.990280687021073,
                           0.988443927126510,
                           0.986451312816259,
                           0.984303800936268,
                           0.982002421258801,
                           0.979548275659527,
                           0.976942537238333,
                           0.974186449384856,
                           0.971281324789764,
                           0.968228544402904,
                           0.965029556339449,
                           0.961685874735289,
                           0.958199078552896,
                           0.954570810339018,
                           0.950802774935535,
                           0.946896738144925,
                           0.942854525351773,
                           0.938678020101856,
                           0.934369162640325,
                           0.929929948410593,
                           0.925362426515542,
                           0.920668698142714,
                           0.915850914955168,
                           0.910911277449748,
                           0.905852033284487,
                           0.900675475576940,
                           0.895383941175252,
                           0.889979808903757,
                           0.884465497784977,
                           0.878843465239859,
                           0.873116205268119,
                           0.867286246610587,
                           0.861356150895418,
                           0.855328510770093,
                           0.849205948021079,
                           0.842991111683067,
                           0.836686676139680,
                           0.830295339217536,
                           0.823819820275572,
                           0.817262858291488,
                           0.810627209947195,
                           0.803915647715110,
                           0.797130957947146,
                           0.790275938968212,
                           0.783353399176031,
                           0.776366155149043,
                           0.769317029764164,
                           0.762208850326119,
                           0.755044446710050,
                           0.747826649519071,
                           0.740558288258413,
                           0.733242189527750,
                           0.725881175233287,
                           0.718478060821137,
                           0.711035653533477,
                           0.703556750688933,
                           0.696044137988615,
                           0.688500587849150,
                           0.680928857764055,
                           0.673331688694706,
                           0.665711803492151,
                           0.658071905350936,
                           0.650414676296073,
                           0.642742775704245,
                           0.635058838860270,
                           0.627365475549802,
                           0.619665268689203,
                           0.611960772993456,
                           0.604254513682941,
                           0.596548985229844,
                           0.588846650144909,
                           0.581149937805200,
                           0.573461243323464,
                           0.565782926459664,
                           0.558117310575161,
                           0.550466681629998,
                           0.542833287223675,
                           0.535219335679730,
                           0.527626995174436,
                           0.520058392909812,
                           0.512515614331134,
                           0.505000702389078,
                           0.497515656846536,
                           0.490062433630159,
                           0.482642944226556,
                           0.475259055123106,
                           0.467912587293223,
                           0.460605315725911,
                           0.453338968999378,
                           0.446115228898436,
                           0.438935730075381,
                           0.431802059753978,
                           0.424715757476165,
                           0.417678314891011,
                           0.410691175585469,
                           0.403755734956380,
                           0.396873340123180,
                           0.390045289880703,
                           0.383272834691460,
                           0.376557176716720,
                           0.369899469885695,
                           0.363300820002107,
                           0.356762284887377,
                           0.350284874559649,
                           0.343869551447840,
                           0.337517230639884,
                           0.331228780164299,
                           0.325005021304212,
                           0.318846728942931,
                           0.312754631940148,
                           0.306729413537832,
                           0.300771711794874,
                           0.294882120049496,
                           0.289061187408468,
                           0.283309419262124,
                           0.277627277824195,
                           0.272015182695439,
                           0.266473511450057,
                           0.261002600243882,
                           0.255602744443307,
                           0.250274199273943,
                           0.245017180487957,
                           0.239831865049088,
                           0.234718391834299,
                           0.229676862351052,
                           0.224707341469186,
                           0.219809858166382,
                           0.214984406286217,
                           0.210230945307796,
                           0.205549401125988,
                           0.200939666841274,
                           0.196401603558241,
                           0.191935041191769,
                           0.187539779279958,
                           0.183215587802879,
                           0.178962208006213,
                           0.174779353228893,
                           0.170666709733863,
                           0.166623937541076,
                           0.162650671261893,
                           0.158746520934044,
                           0.154911072856348,
                           0.151143890422381,
                           0.147444514952346,
                           0.143812466522364,
                           0.140247244790477,
                           0.136748329818647,
                           0.133315182890056,
                           0.129947247321051,
                           0.126643949267090,
                           0.123404698522063,
                           0.120228889310397,
                           0.117115901071369,
                           0.114065099235078,
                           0.111075835989545,
                           0.108147451038438,
                           0.105279272348950,
                           0.102470616889362,
                           0.0997207913558665,
                           0.0970290928882350,
                           0.0943948097739498,
                           0.0918172221404328,
                           0.0892956026350324,
                           0.0868292170924514,
                           0.0844173251893211,
                           0.0820591810856499,
                           0.0797540340528955,
                           0.0775011290884327,
                           0.0752997075162107,
                           0.0731490075734142,
                           0.0710482649829623,
                           0.0689967135117022,
                           0.0669935855141739,
                           0.0650381124618393,
                           0.0631295254576919,
                           0.0612670557361800,
                           0.0594499351483950,
                           0.0576773966324938,
                           0.0559486746693428,
                           0.0542630057233863,
                           0.0526196286687601,
                           0.0510177852006850,
                           0.0494567202321930,
                           0.0479356822762494,
                           0.0464539238133536,
                           0.0450107016447105,
                           0.0436052772310812,
                           0.0422369170174299,
                           0.0409048927435024,
                           0.0396084817404756,
                           0.0383469672138356,
                           0.0371196385126465,
                           0.0359257913853865,
                           0.0347647282225337,
                           0.0336357582860962,
                           0.0325381979262851,
                           0.0314713707855435,
                           0.0304346079901428,
                           0.0294272483295739,
                           0.0284486384239596,
                           0.0274981328797241,
                           0.0265750944337617,
                           0.0256788940863468,
                           0.0248089112230371,
                           0.0239645337258231,
                           0.0231451580737793,
                           0.0223501894334788,
                           0.0215790417394323,
                           0.0208311377648161,
                           0.0201059091827567,
                           0.0194027966184369,
                           0.0187212496922949,
                           0.0180607270545826,
                           0.0174206964115545,
                           0.0168006345435551,
                           0.0162000273152735,
                           0.0156183696784335,
                           0.0150551656671856,
                           0.0145099283864656,
                           0.0139821799935847,
                           0.0134714516733111,
                           0.0129772836067038,
                           0.0124992249339547,
                           0.0120368337114932,
                           0.0115896768636053,
                           0.0111573301288138,
                           0.0107393780012658,
                           0.0103354136673680,
                           0.00994503893790690,
                           0.00956786417588879,
                           0.00920350822032722,
                           0.00885159830620527,
                           0.00851176998083246,
                           0.00818366701681348,
                           0.00786694132184118,
                           0.00756125284552083,
                           0.00726626948342904,
                           0.00698166697860525,
                           0.00670712882066924,
                           0.00644234614275277,
                           0.00618701761642908,
                           0.00594084934481857,
                           0.00570355475404418,
                           0.00547485448320485,
                           0.00525447627303051,
                           0.00504215485337676,
                           0.00483763182971278,
                           0.00464065556875038,
                           0.00445098108335776,
                           0.00426836991689576,
                           0.00409259002711025,
                           0.00392341566970854,
                           0.00376062728174355,
                           0.00360401136492399,
                           0.00345336036896434,
                           0.00330847257508363,
                           0.00316915197975733,
                           0.00303520817882174,
                           0.00290645625202629,
                           0.00278271664812404,
                           0.00266381507058677,
                           0.00254958236402629,
                           0.00243985440139970,
                           0.00233447197207188,
                           0.00223328067080457,
                           0.00213613078773737,
                           0.00204287719942195,
                           0.00195337926096717,
                           0.00186750069934881,
                           0.00178510950793429,
                           0.00170607784226887,
                           0.00163028191716677,
                           0.00155760190514682,
                           0.00148792183624960,
                           0.00142112949926929,
                           0.00135711634443075,
                           0.00129577738753945,
                           0.00123701111562861,
                           0.00118071939412560,
                           0.00112680737555679,
                           0.00107518340980737,
                           0.00102575895595036,
                           0.000978448495656672,
                           0.000933169448195548,
                           0.000889842087032812,
                           0.000848389458032121,
                           0.000808737299262344,
                           0.000770813962412435,
                           0.000734550335813170,
                           0.000699879769063488};

  t_int number_of_samples = sizeof(x_values) / sizeof(*x_values);
  for(t_int i = 0; i < number_of_samples; ++i) {
    // Test that pswf is the same as matlab calculations
    t_real pswf_difference = std::abs(kernels::pswf(x_values[i], 6) - pswf_values[i]);
    CHECK(pswf_difference < 1e-13);
    // Test that kaiser_bessel is the same as matlab calculations
    t_real kb_difference = std::abs(kernels::kaiser_bessel(x_values[i], 6) - kb_values[i]);
    CHECK(kb_difference < 1e-13);
    // Test that gaussian is the same as matlab calculations
    t_real gauss_difference = std::abs(kernels::gaussian(x_values[i], 6) - gauss_values[i]);
    CHECK(gauss_difference < 1e-13);
  }
}

TEST_CASE("Measurement Operator [Kernel Gridding Correction]", "[Gridding_Correction]") {
  t_real x_values[] = {-1,
                       -0.990000000000000,
                       -0.980000000000000,
                       -0.970000000000000,
                       -0.960000000000000,
                       -0.950000000000000,
                       -0.940000000000000,
                       -0.930000000000000,
                       -0.920000000000000,
                       -0.910000000000000,
                       -0.900000000000000,
                       -0.890000000000000,
                       -0.880000000000000,
                       -0.870000000000000,
                       -0.860000000000000,
                       -0.850000000000000,
                       -0.840000000000000,
                       -0.830000000000000,
                       -0.820000000000000,
                       -0.810000000000000,
                       -0.800000000000000,
                       -0.790000000000000,
                       -0.780000000000000,
                       -0.770000000000000,
                       -0.760000000000000,
                       -0.750000000000000,
                       -0.740000000000000,
                       -0.730000000000000,
                       -0.720000000000000,
                       -0.710000000000000,
                       -0.700000000000000,
                       -0.690000000000000,
                       -0.680000000000000,
                       -0.670000000000000,
                       -0.660000000000000,
                       -0.650000000000000,
                       -0.640000000000000,
                       -0.630000000000000,
                       -0.620000000000000,
                       -0.610000000000000,
                       -0.600000000000000,
                       -0.590000000000000,
                       -0.580000000000000,
                       -0.570000000000000,
                       -0.560000000000000,
                       -0.550000000000000,
                       -0.540000000000000,
                       -0.530000000000000,
                       -0.520000000000000,
                       -0.510000000000000,
                       -0.500000000000000,
                       -0.490000000000000,
                       -0.480000000000000,
                       -0.470000000000000,
                       -0.460000000000000,
                       -0.450000000000000,
                       -0.440000000000000,
                       -0.430000000000000,
                       -0.420000000000000,
                       -0.410000000000000,
                       -0.400000000000000,
                       -0.390000000000000,
                       -0.380000000000000,
                       -0.370000000000000,
                       -0.360000000000000,
                       -0.350000000000000,
                       -0.340000000000000,
                       -0.330000000000000,
                       -0.320000000000000,
                       -0.310000000000000,
                       -0.300000000000000,
                       -0.290000000000000,
                       -0.280000000000000,
                       -0.270000000000000,
                       -0.260000000000000,
                       -0.250000000000000,
                       -0.240000000000000,
                       -0.230000000000000,
                       -0.220000000000000,
                       -0.210000000000000,
                       -0.200000000000000,
                       -0.190000000000000,
                       -0.180000000000000,
                       -0.170000000000000,
                       -0.160000000000000,
                       -0.150000000000000,
                       -0.140000000000000,
                       -0.130000000000000,
                       -0.120000000000000,
                       -0.110000000000000,
                       -0.100000000000000,
                       -0.0900000000000000,
                       -0.0800000000000000,
                       -0.0700000000000000,
                       -0.0599999999999999,
                       -0.0499999999999999,
                       -0.0400000000000000,
                       -0.0300000000000000,
                       -0.0200000000000000,
                       -0.0100000000000000,
                       0,
                       0.0100000000000000,
                       0.0200000000000000,
                       0.0300000000000000,
                       0.0400000000000000,
                       0.0499999999999999,
                       0.0599999999999999,
                       0.0700000000000000,
                       0.0800000000000000,
                       0.0900000000000000,
                       0.100000000000000,
                       0.110000000000000,
                       0.120000000000000,
                       0.130000000000000,
                       0.140000000000000,
                       0.150000000000000,
                       0.160000000000000,
                       0.170000000000000,
                       0.180000000000000,
                       0.190000000000000,
                       0.200000000000000,
                       0.210000000000000,
                       0.220000000000000,
                       0.230000000000000,
                       0.240000000000000,
                       0.250000000000000,
                       0.260000000000000,
                       0.270000000000000,
                       0.280000000000000,
                       0.290000000000000,
                       0.300000000000000,
                       0.310000000000000,
                       0.320000000000000,
                       0.330000000000000,
                       0.340000000000000,
                       0.350000000000000,
                       0.360000000000000,
                       0.370000000000000,
                       0.380000000000000,
                       0.390000000000000,
                       0.400000000000000,
                       0.410000000000000,
                       0.420000000000000,
                       0.430000000000000,
                       0.440000000000000,
                       0.450000000000000,
                       0.460000000000000,
                       0.470000000000000,
                       0.480000000000000,
                       0.490000000000000,
                       0.500000000000000,
                       0.510000000000000,
                       0.520000000000000,
                       0.530000000000000,
                       0.540000000000000,
                       0.550000000000000,
                       0.560000000000000,
                       0.570000000000000,
                       0.580000000000000,
                       0.590000000000000,
                       0.600000000000000,
                       0.610000000000000,
                       0.620000000000000,
                       0.630000000000000,
                       0.640000000000000,
                       0.650000000000000,
                       0.660000000000000,
                       0.670000000000000,
                       0.680000000000000,
                       0.690000000000000,
                       0.700000000000000,
                       0.710000000000000,
                       0.720000000000000,
                       0.730000000000000,
                       0.740000000000000,
                       0.750000000000000,
                       0.760000000000000,
                       0.770000000000000,
                       0.780000000000000,
                       0.790000000000000,
                       0.800000000000000,
                       0.810000000000000,
                       0.820000000000000,
                       0.830000000000000,
                       0.840000000000000,
                       0.850000000000000,
                       0.860000000000000,
                       0.870000000000000,
                       0.880000000000000,
                       0.890000000000000,
                       0.900000000000000,
                       0.910000000000000,
                       0.920000000000000,
                       0.930000000000000,
                       0.940000000000000,
                       0.950000000000000,
                       0.960000000000000,
                       0.970000000000000,
                       0.980000000000000,
                       0.990000000000000,
                       1};
  t_real pswf_values[] = {0,
                          0,
                          0,
                          0,
                          0,
                          0,
                          0,
                          0,
                          0,
                          0,
                          0,
                          0,
                          0,
                          0,
                          0,
                          0,
                          0,
                          0,
                          0,
                          0,
                          0,
                          0,
                          0,
                          0,
                          0,
                          0,
                          0,
                          0,
                          0,
                          0,
                          0,
                          0,
                          0,
                          0,
                          0,
                          0,
                          0,
                          0,
                          0,
                          0,
                          0,
                          0,
                          0,
                          0,
                          0,
                          0,
                          0,
                          0,
                          0,
                          0,
                          0.00402855900000000,
                          0.00588153433939356,
                          0.00821952639228226,
                          0.0111158669430303,
                          0.0146478855232069,
                          0.0188962078206995,
                          0.0239439421424684,
                          0.0298757528075581,
                          0.0367768217498934,
                          0.0447317023790150,
                          0.0538230728460491,
                          0.0641303992346064,
                          0.0757285227611846,
                          0.0886862748494166,
                          0.103065146567479,
                          0.118917601418165,
                          0.136285759803160,
                          0.155200092140847,
                          0.175678099523451,
                          0.197723107257007,
                          0.221323193286154,
                          0.246450272914332,
                          0.273059360180902,
                          0.301088024737891,
                          0.330456061079659,
                          0.361065384531118,
                          0.392800165519523,
                          0.425527210380104,
                          0.459096593328513,
                          0.493342540336922,
                          0.528084561549678,
                          0.563128824651590,
                          0.598269757346345,
                          0.633291862907367,
                          0.667971728722788,
                          0.702080203962751,
                          0.735384719039146,
                          0.767651716486410,
                          0.798649160339165,
                          0.828149089079260,
                          0.855930175819319,
                          0.881780258617002,
                          0.905498803694023,
                          0.926899264872818,
                          0.945811303733580,
                          0.962082836813986,
                          0.975581878589836,
                          0.986198151942241,
                          0.993844441281575,
                          0.998457667397153,
                          0.999999667364857,
                          0.998457667397153,
                          0.993844441281575,
                          0.986198151942241,
                          0.975581878589836,
                          0.962082836813986,
                          0.945811303733580,
                          0.926899264872818,
                          0.905498803694023,
                          0.881780258617002,
                          0.855930175819319,
                          0.828149089079260,
                          0.798649160339165,
                          0.767651716486410,
                          0.735384719039146,
                          0.702080203962751,
                          0.667971728722788,
                          0.633291862907367,
                          0.598269757346345,
                          0.563128824651590,
                          0.528084561549678,
                          0.493342540336922,
                          0.459096593328513,
                          0.425527210380104,
                          0.392800165519523,
                          0.361065384531118,
                          0.330456061079659,
                          0.301088024737891,
                          0.273059360180902,
                          0.246450272914332,
                          0.221323193286154,
                          0.197723107257007,
                          0.175678099523451,
                          0.155200092140847,
                          0.136285759803160,
                          0.118917601418165,
                          0.103065146567479,
                          0.0886862748494166,
                          0.0757285227611846,
                          0.0641303992346064,
                          0.0538230728460491,
                          0.0447317023790150,
                          0.0367768217498934,
                          0.0298757528075581,
                          0.0239439421424684,
                          0.0188962078206995,
                          0.0146478855232069,
                          0.0111158669430303,
                          0.00821952639228226,
                          0.00588153433939356,
                          0.00402855900000000,
                          0,
                          0,
                          0,
                          0,
                          0,
                          0,
                          0,
                          0,
                          0,
                          0,
                          0,
                          0,
                          0,
                          0,
                          0,
                          0,
                          0,
                          0,
                          0,
                          0,
                          0,
                          0,
                          0,
                          0,
                          0,
                          0,
                          0,
                          0,
                          0,
                          0,
                          0,
                          0,
                          0,
                          0,
                          0,
                          0,
                          0,
                          0,
                          0,
                          0,
                          0,
                          0,
                          0,
                          0,
                          0,
                          0,
                          0,
                          0,
                          0,
                          0};
  t_real kb_values[]
      = {2.20285645956888e-08,  -5.65978970024670e-07, -1.14252882270768e-06, -1.65739161509457e-06,
         -2.05938910070430e-06, -2.30062311363058e-06, -2.34126053948774e-06, -2.15462115089311e-06,
         -1.73222442811505e-06, -1.08835503639540e-06, -2.63606763327717e-07, 6.73235843310269e-07,
         1.62571705015641e-06,  2.47346194701659e-06,  3.08007487561432e-06,  3.30616759649414e-06,
         3.02855243036611e-06,  2.16666613934683e-06,  7.17295598851346e-07,  -1.20134726343265e-06,
         -3.29520947210833e-06, -5.02873872825884e-06, -5.54720922895997e-06, -3.58642402031131e-06,
         2.63060855841303e-06,  1.55101829412959e-05,  3.82174662971212e-05,  7.48186915866401e-05,
         0.000130435187284355,  0.000211408082882889,  0.000325472133760779,  0.000481936699157212,
         0.000691871490627306,  0.000968294294094115,  0.00132635746585675,   0.00178352962162970,
         0.00235976858824408,   0.00307768138059059,   0.00396266671224037,   0.00504303535716691,
         0.00635010356180290,   0.00791825467018930,   0.00978496417804989,   0.0119907835808028,
         0.0145792786307987,    0.0175969179737239,    0.0210929085944489,    0.0251189750678849,
         0.0297290802776623,    0.0349790860294258,    0.0409263528386977,    0.0476292791057583,
         0.0551467808897721,    0.0635377145473196,    0.0728602455904844,    0.0831711682290191,
         0.0945251811705973,    0.106974126342665,     0.120566198247675,     0.135345132649515,
         0.151349384191101,     0.168611303340757,     0.187156323738390,     0.207002171542739,
         0.228158108751835,     0.250624222665355,     0.274390773668275,     0.299437613330604,
         0.325733684432752,     0.353236613937489,     0.381892409138879,     0.411635266231421,
         0.442387499366591,     0.474059596913169,     0.506550410126689,     0.539747477782625,
         0.573527488559878,     0.607756881101008,     0.642292579752258,     0.676982862028333,
         0.711668351886751,     0.746183130965846,     0.780355958072011,     0.814011585427741,
         0.846972158543965,     0.879058685087848,     0.910092556808574,     0.939897107483774,
         0.968299188980497,     0.995130746904553,     1.02023037695707,      1.04344484303436,
         1.06463053830531,      1.08365487097812,      1.10039755722211,      1.11475180473198,
         1.12662537169722,      1.13594148745054,      1.14263962279469,      1.14667609991988,
         1.14802453389575,      1.14667609991988,      1.14263962279469,      1.13594148745054,
         1.12662537169722,      1.11475180473198,      1.10039755722211,      1.08365487097812,
         1.06463053830531,      1.04344484303436,      1.02023037695707,      0.995130746904553,
         0.968299188980497,     0.939897107483774,     0.910092556808574,     0.879058685087848,
         0.846972158543965,     0.814011585427741,     0.780355958072011,     0.746183130965846,
         0.711668351886751,     0.676982862028333,     0.642292579752258,     0.607756881101008,
         0.573527488559878,     0.539747477782625,     0.506550410126689,     0.474059596913169,
         0.442387499366591,     0.411635266231421,     0.381892409138879,     0.353236613937489,
         0.325733684432752,     0.299437613330604,     0.274390773668275,     0.250624222665355,
         0.228158108751835,     0.207002171542739,     0.187156323738390,     0.168611303340757,
         0.151349384191101,     0.135345132649515,     0.120566198247675,     0.106974126342665,
         0.0945251811705973,    0.0831711682290191,    0.0728602455904844,    0.0635377145473196,
         0.0551467808897721,    0.0476292791057583,    0.0409263528386977,    0.0349790860294258,
         0.0297290802776623,    0.0251189750678849,    0.0210929085944489,    0.0175969179737239,
         0.0145792786307987,    0.0119907835808028,    0.00978496417804989,   0.00791825467018930,
         0.00635010356180290,   0.00504303535716691,   0.00396266671224037,   0.00307768138059059,
         0.00235976858824408,   0.00178352962162970,   0.00132635746585675,   0.000968294294094115,
         0.000691871490627306,  0.000481936699157212,  0.000325472133760779,  0.000211408082882889,
         0.000130435187284355,  7.48186915866401e-05,  3.82174662971212e-05,  1.55101829412959e-05,
         2.63060855841303e-06,  -3.58642402031131e-06, -5.54720922895997e-06, -5.02873872825884e-06,
         -3.29520947210833e-06, -1.20134726343265e-06, 7.17295598851346e-07,  2.16666613934683e-06,
         3.02855243036611e-06,  3.30616759649414e-06,  3.08007487561432e-06,  2.47346194701659e-06,
         1.62571705015641e-06,  6.73235843310269e-07,  -2.63606763327717e-07, -1.08835503639540e-06,
         -1.73222442811505e-06, -2.15462115089311e-06, -2.34126053948774e-06, -2.30062311363058e-06,
         -2.05938910070430e-06, -1.65739161509457e-06, -1.14252882270768e-06, -5.65978970024670e-07,
         2.20285645956888e-08};
  t_real gauss_values[]
      = {7.79493274433476e-06, 9.94228692044273e-06, 1.26502224466048e-05, 1.60563927676993e-05,
         2.03299239766315e-05, 2.56780166749567e-05, 3.23537890143938e-05, 4.06655613866274e-05,
         5.09878084634825e-05, 6.37740311948571e-05, 7.95718297198766e-05, 9.90404876110699e-05,
         0.000122971408011274, 0.000152311772467652, 0.000188191822900232, 0.000231956195286902,
         0.000285199759252389, 0.000349808439580846, 0.000428005512304208, 0.000522403877831003,
         0.000636064814761461, 0.000772563708591247, 0.000936063227281556, 0.00113139437837454,
         0.00136414582756920,  0.00164076178400285,  0.00196864866047992,  0.00235629059521048,
         0.00281337377309935,  0.00335091930736851,  0.00398142423479201,  0.00471900993907302,
         0.00557957704653091,  0.00658096553669976,  0.00774311847898104,  0.00908824744749947,
         0.0106409972833003,   0.0124286074708006,   0.0144810669801267,   0.0168312590062418,
         0.0195150916186365,   0.0225716099323464,   0.0260430850340541,   0.0299750745592136,
         0.0344164495317796,   0.0394193818623717,   0.0450392867692755,   0.0513347143555023,
         0.0583671846598817,   0.0662009607158526,   0.0749027545119931,   0.0845413612653258,
         0.0951872181015903,   0.106911884092501,    0.119787439631420,    0.133885804334611,
         0.149277974029365,    0.166033178921789,    0.184217966709446,    0.203895216195312,
         0.225123088841899,    0.247953927644749,    0.272433114664262,    0.298597900490936,
         0.326476220784425,    0.356085516771036,    0.387431578155354,    0.420507428246497,
         0.455292272165956,    0.491750529741991,    0.529830975058956,    0.569466004578028,
         0.610571055244923,    0.653044193025209,    0.696765890843863,    0.741599012948688,
         0.787389020275581,    0.833964408488515,    0.881137387032589,    0.928704803822208,
         0.976449316148087,    1.02414080409832,     1.07153801833231,     1.11839044951313,
         1.16444040219188,     1.20942525154877,     1.25307985723552,     1.29513910473391,
         1.33534054124762,     1.37342707027051,     1.40914966670890,     1.44227007284921,
         1.47256343461343,     1.49982083747333,     1.52385170212352,     1.54448600154692,
         1.56157626342703,     1.57499932493493,     1.58465781068938,     1.59048130908291,
         1.59242722709602,     1.59048130908291,     1.58465781068938,     1.57499932493493,
         1.56157626342703,     1.54448600154692,     1.52385170212352,     1.49982083747333,
         1.47256343461343,     1.44227007284921,     1.40914966670890,     1.37342707027051,
         1.33534054124762,     1.29513910473391,     1.25307985723552,     1.20942525154877,
         1.16444040219188,     1.11839044951313,     1.07153801833231,     1.02414080409832,
         0.976449316148087,    0.928704803822208,    0.881137387032589,    0.833964408488515,
         0.787389020275581,    0.741599012948688,    0.696765890843863,    0.653044193025209,
         0.610571055244923,    0.569466004578028,    0.529830975058956,    0.491750529741991,
         0.455292272165956,    0.420507428246497,    0.387431578155354,    0.356085516771036,
         0.326476220784425,    0.298597900490936,    0.272433114664262,    0.247953927644749,
         0.225123088841899,    0.203895216195312,    0.184217966709446,    0.166033178921789,
         0.149277974029365,    0.133885804334611,    0.119787439631420,    0.106911884092501,
         0.0951872181015903,   0.0845413612653258,   0.0749027545119931,   0.0662009607158526,
         0.0583671846598817,   0.0513347143555023,   0.0450392867692755,   0.0394193818623717,
         0.0344164495317796,   0.0299750745592136,   0.0260430850340541,   0.0225716099323464,
         0.0195150916186365,   0.0168312590062418,   0.0144810669801267,   0.0124286074708006,
         0.0106409972833003,   0.00908824744749947,  0.00774311847898104,  0.00658096553669976,
         0.00557957704653091,  0.00471900993907302,  0.00398142423479201,  0.00335091930736851,
         0.00281337377309935,  0.00235629059521048,  0.00196864866047992,  0.00164076178400285,
         0.00136414582756920,  0.00113139437837454,  0.000936063227281556, 0.000772563708591247,
         0.000636064814761461, 0.000522403877831003, 0.000428005512304208, 0.000349808439580846,
         0.000285199759252389, 0.000231956195286902, 0.000188191822900232, 0.000152311772467652,
         0.000122971408011274, 9.90404876110699e-05, 7.95718297198766e-05, 6.37740311948571e-05,
         5.09878084634825e-05, 4.06655613866274e-05, 3.23537890143938e-05, 2.56780166749567e-05,
         2.03299239766315e-05, 1.60563927676993e-05, 1.26502224466048e-05, 9.94228692044273e-06,
         7.79493274433476e-06};

  t_int number_of_samples = sizeof(x_values) / sizeof(*x_values);
  for(t_int i = 0; i < number_of_samples; ++i) {

    t_real pswf_difference = std::abs(kernels::ft_pswf(x_values[i], 6) - pswf_values[i]);
    CHECK(pswf_difference < 1e-13); // Test that pswf is the same as matlab calculations

    t_real kb_difference = std::abs(kernels::ft_kaiser_bessel(x_values[i], 6) - kb_values[i]);
    CHECK(kb_difference < 1e-13); // Test that kaiser_bessel is the same as matlab calculations

    t_real gauss_difference = std::abs(kernels::ft_gaussian(x_values[i], 6) - gauss_values[i]);
    CHECK(gauss_difference < 1e-13); // Test that gaussian is the same as matlab calculations
  }
}

TEST_CASE("Measurement Operator [Gridding]", "[Gridding]") {

  /*
    Tests each kernel by gridding visibility data and a point spread function, then comparing the
    gridded images to the same images generated from matlab.

    So far the maximum difference between the gridded images of at166B are 0.00015% (kb), 0.1071%
    (pswf), and 0.0362% (guass).

    Since there is small difference between the end results, it suggests that the C++ and Matlab
    codes perform the same process.
    But, it is not clear why the pswf and guass kernels are not as close to the matlab versions as
    the kb kernel.

  */

  utilities::vis_params uv_vis;
  t_real max;
  t_real max_diff;
  t_int over_sample;
  t_real cellsize;
  std::string kernel;

  // Gridding example
  over_sample = 2;
  t_int J = 6;
  uv_vis = utilities::read_visibility(
      vla_filename("at166B.3C129.c0.vis")); // visibility data being read in
  cellsize = 0.3;
  uv_vis = utilities::uv_symmetry(
      uv_vis); // Enforce condjugate symmetry by reflecting measurements in uv coordinates
  Vector<t_complex> point_source = uv_vis.vis * 0;
  point_source.setOnes(); // Creating model visibilities for point source
  Image<t_complex> psf;
  // Amount in which images are shifted relative to each other
  t_int j_shift = -0;
  t_int i_shift = -1;
  i_shift = 0;

  SECTION("Kaiser Bessel Gridding") {
    kernel = "kb";
    MeasurementOperator op(uv_vis, J, J, kernel, 1024, 1024, 5, over_sample, cellsize,
                           cellsize); // Generating gridding matrix

    psf = op.grid(point_source);
    max = psf.real().maxCoeff();
    psf = psf / max;
    pfitsio::write2d(psf.real(), output_filename("kb_psf.fits"));

    Image<t_real> kb_img = op.grid(uv_vis.vis).real();
    max = kb_img.maxCoeff();
    kb_img = kb_img / max;
    pfitsio::write2d(kb_img.real(), output_filename("grid_image_real_kb_6.fits"));

    Image<t_real> kb_test_image = pfitsio::read2d(gridding_filename("at166BtestJ6kb.fits")).real();

    max_diff = 0;
    for(t_int i = 2; i < kb_test_image.cols() - 2; ++i) {
      for(t_int j = 2; j < kb_test_image.rows() - 2; ++j) {
        if(std::abs(kb_img(j, i) - kb_test_image(j + j_shift, i + i_shift)) > max_diff) {
          max_diff = std::abs(kb_img(j, i) - kb_test_image(j + j_shift, i + i_shift));
        }
        if(std::abs(kb_img(j, i) - kb_test_image(j + j_shift, i + i_shift)) >= 1e-12) {
          INFO(i << " " << j << "\n");
        }
        CHECK(std::abs(kb_img(j, i) - kb_test_image(j + j_shift, i + i_shift)) < 1e-12);
      }
    }
    INFO("Percentage max difference in Kaiser Bessel gridding: " << max_diff * 100 << "%");
  }
  SECTION("Prolate Spheroidal Wave Functon Gridding") {
    kernel = "pswf";
    MeasurementOperator op(uv_vis, J, J, kernel, 1024, 1024, 5, over_sample, cellsize,
                           cellsize); // Generating gridding matrix
    Image<t_real> pswf_img = op.grid(uv_vis.vis).real();
    max = pswf_img.maxCoeff();
    pswf_img = pswf_img / max;
    pfitsio::write2d(pswf_img.real(), output_filename("grid_image_real_pswf_6.fits"));
    pfitsio::write2d(op.S.real(), output_filename("grid_scale_real_pswf_6.fits"));

    Image<t_real> pswf_test_image
        = pfitsio::read2d(gridding_filename("at166BtestJ6pswf.fits")).real();
    Image<t_real> pswf_difference = pswf_img - pswf_test_image;
    max_diff = pswf_difference.abs().maxCoeff();

    max_diff = 0;
    for(t_int i = 1; i < pswf_test_image.cols() - 1; ++i) {
      for(t_int j = 1; j < pswf_test_image.rows() - 1; ++j) {
        if(std::abs(pswf_img(j, i) - pswf_test_image(j + j_shift, i + i_shift)) > max_diff) {
          max_diff = std::abs(pswf_img(j, i) - pswf_test_image(j + j_shift, i + i_shift));
        }
        if(std::abs(pswf_img(j, i) - pswf_test_image(j + j_shift, i + i_shift)) >= 1e-12) {
          CAPTURE(i);
          CAPTURE(j);
        }
        CHECK(std::abs(pswf_img(j, i) - pswf_test_image(j + j_shift, i + i_shift)) < 1e-12);
      }
    }
    INFO("Percentage max difference in Prolate Spheroidal Wave Functon gridding: " << max_diff * 100
                                                                                   << "%");
  }
  SECTION("Gaussian Gridding") {
    kernel = "gauss";
    MeasurementOperator op(uv_vis, J, J, kernel, 1024, 1024, 5, over_sample, cellsize,
                           cellsize); // Generating gridding matrix
    Image<t_real> gauss_img = op.grid(uv_vis.vis).real();
    max = gauss_img.maxCoeff();
    gauss_img = gauss_img / max;

    Image<t_real> gauss_test_image
        = pfitsio::read2d(gridding_filename("at166BtestJ6gauss.fits")).real();
    Image<t_real> gauss_difference = gauss_img - gauss_test_image;
    max_diff = gauss_difference.abs().maxCoeff();

    max_diff = 0;
    for(t_int i = 1; i < gauss_test_image.cols() - 1; ++i) {
      for(t_int j = 1; j < gauss_test_image.rows() - 1; ++j) {
        if(std::abs(gauss_img(j, i) - gauss_test_image(j + j_shift, i + i_shift)) > max_diff) {
          max_diff = std::abs(gauss_img(j, i) - gauss_test_image(j + j_shift, i + i_shift));
        }
        if(std::abs(gauss_img(j, i) - gauss_test_image(j + j_shift, i + i_shift)) >= 1e-12) {
          CAPTURE(i);
          CAPTURE(j);
        }
        CHECK(std::abs(gauss_img(j, i) - gauss_test_image(j + j_shift, i + i_shift)) < 1e-12);
      }
    }
    INFO("Percentage max difference in Gaussian gridding: " << max_diff * 100 << "%");
  }
}

TEST_CASE("Measurement Operator [Degridding]", "[Degridding]") {
  /*
    Degrids a model point source, and then compares how close the degridded visibilities are to that
    of a point source.

    Also generates a degridded image of M31.

    We find the maximum difference in visibilities is 0.001% (kb), 0.842% (pswf), and 0.056%
    (gauss).

    Again, the kb Kernel gives closer to the expected result. PSWF gives a result that is worse than
    expected.
  */

  utilities::vis_params uv_vis;
  t_real max;
  t_real max_diff;
  t_int over_sample;
  t_real cellsize;
  std::string kernel;
  Image<t_complex> img;
  Image<t_complex> point;
  Vector<t_complex> psf_vis;
  Image<t_complex> psf;

  SECTION("Kaiser Bessel Degridding") {
    kernel = "kb";
    over_sample = 2;
    t_int J = 6;
    uv_vis = utilities::read_visibility(degridding_filename("M31_J6kb.vis"));
    img = pfitsio::read2d(image_filename("M31.fits"));
    uv_vis = utilities::uv_scale(uv_vis, img.cols() * over_sample, img.rows() * over_sample);
    uv_vis.v = -uv_vis.v;
    uv_vis.units = "pixels";
    MeasurementOperator op(uv_vis, J, J, kernel, img.cols(), img.rows(), 5, over_sample, cellsize,
                           cellsize); // Calculates gridding matrix

    point = Image<t_complex>::Zero(img.cols(), img.rows());
    point(floor(img.cols() / 2) - 1, floor(img.rows() / 2) - 1)
        = 1;                    // Creating model of point source in image
    psf_vis = op.degrid(point); // Creating visibilities from model
    max = psf_vis.cwiseAbs().maxCoeff();
    psf_vis = psf_vis / max; // normalized visibilities
    psf = op.grid(psf_vis);  // gridding model visibilities into image
    max = psf.real().maxCoeff();
    psf = psf / max;
    pfitsio::write2d(point.real(), output_filename("point.fits"));
    pfitsio::write2d(
        psf.real(),
        output_filename("kb_psf_M31_degridding.fits")); // saving image of degridded point source

    Vector<t_complex> kb_vis = op.degrid(img); // visibilities of degridded model
    Image<t_complex> kb_img = op.grid(kb_vis); // image of degridded model
    max = kb_img.real().maxCoeff();
    kb_img = kb_img / max; // normalisation of degridded model

    pfitsio::write2d(kb_img.real(), output_filename("kb_test.fits"));

    max = kb_vis.cwiseAbs().maxCoeff();
    kb_vis = kb_vis / max;
    max_diff = 0;

    for(int i = 0; i < psf_vis.size(); ++i) {
      if(std::abs(std::abs(psf_vis(i)) - 1) > max_diff) {
        max_diff = std::abs(std::abs(psf_vis(i)) - 1);
      }
      // checking visibilities of degridded point source
      CHECK(std::abs(std::abs(psf_vis(i)) - 1) < 0.0001);
    }
    INFO("Percentage max difference in Kaiser Bessel degridding: " << max_diff * 100 << "%");
    psf_vis.setOnes();
    psf = op.grid(psf_vis);
    max = psf.real().maxCoeff();
    psf = psf / max;
    pfitsio::write2d(
        psf.real(),
        output_filename("kb_psf_M31_gridding.fits")); // saving image of degridded point source
  }

  SECTION("Prolate Spheroidal Wave Functon Degridding") {
    kernel = "pswf";
    over_sample = 2;
    t_int J = 6;
    uv_vis = utilities::read_visibility(degridding_filename("M31_J6kb.vis"));
    img = pfitsio::read2d(image_filename("M31.fits"));
    uv_vis = utilities::uv_scale(uv_vis, img.cols() * over_sample, img.rows() * over_sample);
    uv_vis.v = -uv_vis.v;
    uv_vis.units = "pixels";
    MeasurementOperator op(uv_vis, J, J, kernel, img.cols(), img.rows(), 5, over_sample, cellsize,
                           cellsize); // Calculates gridding matrix

    point = Image<t_complex>::Zero(img.cols(), img.rows());
    point(floor(img.cols() / 2) - 1, floor(img.rows() / 2) - 1)
        = 1;                    // Creating model of point source in image
    psf_vis = op.degrid(point); // Creating visibilities from model
    max = psf_vis.cwiseAbs().maxCoeff();
    psf_vis = psf_vis / max; // normalized visibilities
    psf = op.grid(psf_vis);  // gridding model visibilities into image
    max = psf.real().maxCoeff();
    psf = psf / max;
    pfitsio::write2d(
        psf.real(),
        output_filename("pswf_psf_M31_degridding.fits")); // saving image of degridded point source

    Vector<t_complex> pswf_vis = op.degrid(img);   // visibilities of degridded model
    Image<t_complex> pswf_img = op.grid(pswf_vis); // image of degridded model
    max = pswf_img.real().maxCoeff();
    pswf_img = pswf_img / max; // normalisation of degridded model

    pfitsio::write2d(pswf_img.real(), output_filename("pswf_test.fits"));

    max = pswf_vis.cwiseAbs().maxCoeff();
    pswf_vis = pswf_vis / max;
    max_diff = 0;

    for(int i = 0; i < psf_vis.size(); ++i) {
      if(std::abs(std::abs(psf_vis(i)) - 1) > max_diff) {
        max_diff = std::abs(std::abs(psf_vis(i)) - 1);
      }
      // checking visibilities of degridded point source
      CHECK(std::abs(std::abs(psf_vis(i)) - 1) < 0.01);
    }
    INFO("Percentage max difference in Prolate Spheroidal Wave Functon degridding: "
         << max_diff * 100 << "%");
    psf_vis.setOnes();
    psf = op.grid(psf_vis);
    max = psf.real().maxCoeff();
    psf = psf / max;
    pfitsio::write2d(
        psf.real(),
        output_filename("pswf_psf_M31_gridding.fits")); // saving image of degridded point source
  }

  SECTION("Gaussian Degridding") {
    kernel = "gauss";
    over_sample = 2;
    t_int J = 6;
    uv_vis = utilities::read_visibility(degridding_filename("M31_J6kb.vis"));
    img = pfitsio::read2d(image_filename("M31.fits"));
    uv_vis = utilities::uv_scale(uv_vis, img.cols() * over_sample, img.rows() * over_sample);
    uv_vis.v = -uv_vis.v;
    uv_vis.units = "pixels";
    MeasurementOperator op(uv_vis, J, J, kernel, img.cols(), img.rows(), 5, over_sample, cellsize,
                           cellsize); // Calculates gridding matrix

    point = Image<t_complex>::Zero(img.cols(), img.rows());
    point(floor(img.cols() / 2) - 1, floor(img.rows() / 2) - 1)
        = 1;                    // Creating model of point source in image
    psf_vis = op.degrid(point); // Creating visibilities from model
    max = psf_vis.cwiseAbs().maxCoeff();
    psf_vis = psf_vis / max; // normalized visibilities
    psf = op.grid(psf_vis);  // gridding model visibilities into image
    max = psf.real().maxCoeff();
    psf = psf / max;
    pfitsio::write2d(
        psf.real(),
        output_filename("gauss_psf_M31_degridding.fits")); // saving image of degridded point source

    Vector<t_complex> gauss_vis = op.degrid(img);    // visibilities of degridded model
    Image<t_complex> gauss_img = op.grid(gauss_vis); // image of degridded model
    max = gauss_img.real().maxCoeff();
    gauss_img = gauss_img / max; // normalisation of degridded model

    pfitsio::write2d(gauss_img.real(), output_filename("gauss_test.fits"));

    max = gauss_vis.cwiseAbs().maxCoeff();
    gauss_vis = gauss_vis / max;
    max_diff = 0;

    for(int i = 0; i < psf_vis.size(); ++i) {
      if(std::abs(std::abs(psf_vis(i)) - 1) > max_diff) {
        max_diff = std::abs(std::abs(psf_vis(i)) - 1);
      }
      // checking visibilities of degridded point source
      CHECK(std::abs(std::abs(psf_vis(i)) - 1) < 0.01);
    }
    INFO("Percentage max difference in Gaussian degridding: " << max_diff * 100 << "%");
    psf_vis.setOnes();
    psf = op.grid(psf_vis);
    max = psf.real().maxCoeff();
    psf = psf / max;
    pfitsio::write2d(
        psf.real(),
        output_filename("gauss_psf_M31_gridding.fits")); // saving image of degridded point source
  }

}

 TEST_CASE("Flux") {
  //Test that checks flux scale is Jy/Pixel to Jy/lambda
  //const t_int factor = 1;
  //Vector<t_real> u = Vector<t_real>::Zero(256 * 256 * factor * factor);
  //Vector<t_real> v = Vector<t_real>::Zero(256 * 256 * factor * factor);
  //Vector<t_real> w = Vector<t_real>::Zero(256 * 256 * factor * factor);

  //Create data on a grid
  //t_int k = 0;
  //for (t_int i = 0; i < 256 * factor; i++) {
  //  for (t_int j = 0; j < 256 * factor; j++) {
  //    u(k) = (i / factor - 256.) * constant::pi / 256;
  //    v(k) = (j / factor - 256.) * constant::pi / 256;
  //    k++;
  //  }
 // }
 // utilities::vis_params grid_uv_data;
 // grid_uv_data.u = u;
 // grid_uv_data.v = v;
 // grid_uv_data.w = w;
 // grid_uv_data.weights = Vector<t_complex>::Constant(u.size(), 1.);
 // grid_uv_data.units = "radians";
 // grid_uv_data.vis = Vector<t_complex>::Constant(u.size(), 1.);
 // t_int const J = 4;
 // std::string const kernel = "kb";
 // std::cout << grid_uv_data.units << '\n';
  //MeasurementOperator op(grid_uv_data, J, J, kernel, 256, 256, 200, 2); // Generating gridding matrix
  //Image<t_complex> image = Image<t_complex>::Zero(256, 256);
  //image(0, 0) = 1.;
  //auto vis = op.degrid(image);
  //FFTOperator ftop;
  //auto const ft_grid = ftop.forward(image);

  //auto const ft_vis = Image<t_complex>::Map(ft_grid.data(), 1, 256 * 256);
  //auto const vis_grid = Image<t_complex>::Map(vis.data(), 256, 256);
  //auto const grid_image = ftop.inverse(vis_grid).array();
  //pfitsio::write2d(vis_grid.real(), output_filename("vis_real.fits") );
  //pfitsio::write2d(ft_grid.real(), output_filename("ft_vis_real.fits") );
  //pfitsio::write2d(vis_grid.imag(), output_filename("vis_imag.fits") );
  //pfitsio::write2d(ft_grid.imag(), output_filename("ft_vis_imag.fits") );
  //pfitsio::write2d(grid_image.real(), output_filename("vis_image.fits") );

  //CHECK(std::abs(image.abs().maxCoeff() - grid_image.abs().maxCoeff()) < 1e-2);
  // degrid point source and check scale. We find that the visiblities should scale with sqrt(M)/oversample_ratio
  // Image<t_complex> image = Image<t_complex>::Zero(256, 256);
 // t_real oversample = 2.;
  //image(0, 0) = 1.;
 // const t_int number_of_samples  = 1012;
 // auto uv_random = utilities::random_sample_density(number_of_samples, 0, constant::pi *0.5);
 // uv_random.u = uv_random.u;
 // uv_random.v = uv_random.v;
 // uv_random.units = "radians";
 // MeasurementOperator op_random(uv_random, J, J, kernel, image.rows(), image.cols(), 200, oversample, 1000); // Generating gridding matrix
 // Vector<t_real> const vis_random = op_random.degrid(image).array().abs();
 // auto temp_psf = op_random.grid(Vector<t_complex>::Constant(vis_random.size(), 1.));
 // auto mean = vis_random.array().abs().mean();
 // for (t_int i = 0; i < vis_random.size(); i++) {
  //  std::cout << vis_random(i)/mean  << '\n';
  //  CHECK(std::abs(std::abs(vis_random(i)/oversample - 1.)) < 1e-1);
 // }
 }
