/***************************************************************************
  hydrometertool.h
  -------------------
  A hydrometer correction utility for QBrew
  -------------------
  begin         February 21st, 2001
  author        Abe Kabakoff <abe_kabakoff@gmx.de>
  -------------------
  Copyright 2001, Abe Kabakoff
  Please see the header file for copyright and license information
 ***************************************************************************/

#include "resource.h"
#include "hydrometertool.h"

#ifdef HYDRO_DEBUG
#include <qapplication.h>
#endif // HYDRO_DEBUG
#include <qframe.h>
#include <qlabel.h>
#include <qlayout.h>
#include <qpushbutton.h>
#include <qsplitter.h>
#include <qstring.h>
#include <qwidget.h>

#include "fixedspinbox.h"

//////////////////////////////////////////////////////////////////////////////
// Constants

const double BOILING = 212.0;
const double FREEZING = 32.0;
const double TEMP_STEP = 0.2;
const double MINSG = 0.850;
const double MAXSG = 1.200;
const double SG_STEP = 0.001;

const double COEFF1 = 1.313454;
const double COEFF2 = 0.132674;
const double COEFF3 = 2.057793E-3;
const double COEFF4 = 2.627634E-6;

const QString SUFFIX = ' ' + AppResource::CHAR_LATIN_DEGREE + 'F';

//////////////////////////////////////////////////////////////////////////////
// HydrometerTool()
// ----------------
// Constructor

HydrometerTool::HydrometerTool(QWidget* parent) : QDialog(parent, "calculator")
{
    setCaption(AppResource::ID_TITLE + tr(" - Hydrometer Tool"));
    // Here we create everything we'll need
    box_ = new QGridLayout(this, 5, 3);
    roomtemp_ = new FixedSpinBox(FREEZING, BOILING, TEMP_STEP, 1, this, "roomtempspin");
    caltemp_ = new FixedSpinBox(FREEZING, BOILING, TEMP_STEP, 1, this, "caltempspin");
    sgread_ = new FixedSpinBox(MINSG, MAXSG, SG_STEP, 3, this, "sgreadspin");
    actsg_ = new QLabel(this, "actsg");
    QLabel *textlabel;
    split_ = new QFrame(this, "separator");
    finished_ = new QPushButton("Finished", this, "finished");

    // make the "finish" button close the dialog
    QObject::connect(finished_, SIGNAL(clicked()), this, SLOT(accept()));

    // set the default values for the dialog
    roomtemp_->setValue(60.0);
    caltemp_->setValue(60.0);
    sgread_->setValue(1.000);
    actsg_->setText("1.000");

    // put a degrees F suffix on the temperatures
    roomtemp_->setSuffix(SUFFIX);
    caltemp_->setSuffix(SUFFIX);

    // add text labels
    textlabel = new QLabel(this, "roomtemp");
    textlabel->setText("Sample Temperature");
    box_->addWidget(textlabel, 0, 0, Qt::AlignRight);
    textlabel = new QLabel(this, "caltemp");
    textlabel->setText("Calibrated Temperature");
    box_->addWidget(textlabel, 1, 0, Qt::AlignRight);
    textlabel = new QLabel(this, "hydroread");
    textlabel->setText("Hydrometer Reading");
    box_->addWidget(textlabel, 2, 0, Qt::AlignRight);
    textlabel = new QLabel(this, "corread");
    textlabel->setText("Corrected Reading");
    box_->addWidget(textlabel, 3, 0, Qt::AlignRight);

    // put the fields in their places
    box_->addWidget(roomtemp_, 0, 2);
    box_->addWidget(caltemp_, 1, 2);
    box_->addWidget(sgread_, 2, 2);
    box_->addWidget(actsg_, 3, 2);
    box_->addMultiCellWidget(finished_, 5, 5, 0, 3, Qt::AlignCenter);
    finished_->setDefault(true);

    // add margins and spacing
    box_->addColSpacing(1, 10);
    box_->setMargin(10);
    split_->setFrameStyle(QFrame::HLine | QFrame::Sunken);
    box_->addMultiCellWidget(split_, 4, 4, 0, 3);

    // make any changes to the values recalculate the actual SG
    QObject::connect(roomtemp_, SIGNAL(valueChanged(int)), this, SLOT(recalc()));
    QObject::connect(caltemp_, SIGNAL(valueChanged(int)), this, SLOT(recalc()));
    QObject::connect(sgread_, SIGNAL(valueChanged(int)), this, SLOT(recalc()));
}

//////////////////////////////////////////////////////////////////////////////
// recalc()
// --------
// the signal to calculate the actual SG

void HydrometerTool::recalc()
{
    // see http://www.hbd.org/brewery/library/HydromCorr0992.html

    double calterm;   // E is the correction factor for the calibration temp
    double cor;     // the corrected reading
    double temp = roomtemp_->value();
    double cal = caltemp_->value();
    double act = sgread_->value();

    // Correction(@59F) = 1.313454 - 0.132674*T + 2.057793e-3*T**2 - 2.627634e-6*T**3
    // the equation is set up for 59F (15C)
    calterm = COEFF1 - (COEFF2 * cal) + (COEFF3 * (cal*cal)) - (COEFF4 * (cal*cal*cal));
    cor = (COEFF1 - (COEFF2 * temp) + (COEFF3 * (temp*temp)) - (COEFF4 * (temp*temp*temp))) - calterm;
    cor /= 1000.0;
    cor = 1.0 - cor;
    cor = act / cor;

    // now that we have the value change the reading
    actsg_->setText(QString::number(cor, 'f', 3));
}

// we don't include main unless we're debugging just the hydrometer!

#ifdef HYDRO_DEBUG
int main(int argc, char **argv)
{
    QApplication a(argc, argv);
    HydrometerTool b;
    QVBox box(0, "box");
    QPushButton calc("Calc", &box, "calc");
    QPushButton quit("Quit", &box, "quit");
    a.setMainWidget(&box);

    QObject::connect((QObject *) &calc, SIGNAL(clicked()), &b, SLOT(show()));
    QObject::connect((QObject *) &quit, SIGNAL(clicked()), &a, SLOT(quit()));

    box.show();
    return a.exec();
}
#endif // HYDRO_DEBUG
