/***************************************************************************
  qbrewcalc.cpp
  -------------------
  Handles calculations for QBrew
  -------------------
  begin         September 26th, 1999
  author        David Johnson <david@usermode.org>
  -------------------
  Copyright 1999, 2001, David Johnson
  Please see the header file for copyright and license information
 ***************************************************************************/

#include <cmath>

#include <qdir.h>
#include <qdom.h>
#include <qtextstream.h>

#include "calcresource.h"
#include "resource.h"
#include "qbrewcalc.h"
#include "qbrewdoc.h"
#include "preferences.h"
#include "oldstore.h"

using namespace AppResource;
using namespace CalcResource;

//////////////////////////////////////////////////////////////////////////////
// Constants

const double STEEP_YIELD = 0.33;
const double COEFF1 = 1.65;
const double COEFF2 = 0.000125;
const double COEFF3 = 0.04;
const double COEFF4 = 4.15;

const double GPO = 28.3495;	// grams per ounce
const double LPG = 3.785;	// liters per gallon

//////////////////////////////////////////////////////////////////////////////
// Construction, Destruction                                                //
//////////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////////
// QBrewCalc()
// -----------
// Constructor

QBrewCalc::QBrewCalc(Preferences *preferences, QBrewDoc *doc)
    : preferences_(preferences), document_(doc), og_(0.0), ibu_(0.0), srm_(0.0),
    groupmap_(), utable_(), styles_(), grains_(), hops_(), misc_()
{
    utable_.clear();
    // build hashes
    groupmap_[groupStyles] = gidStyles;
    groupmap_[groupCalcGrains] = gidCalcGrains;
    groupmap_[groupCalcHops] = gidCalcHops;
    groupmap_[groupCalcMisc] = gidCalcMisc;
    groupmap_[groupUtilTable] = gidUtilTable;

    // load databases
    QString calcdir = preferences_->getString(ID_PREF_QBREWDIR, ID_PREF_QBREWDIR_DEFAULT) + ID_DATA_DIR;
    // try user home directory first, quietly...
    if (!loadData(QDir::homeDirPath() + "/." + ID_CALC_FILE, true)) {
        // then the default data file
        if (!loadData(calcdir + ID_CALC_FILE)) {
            // else load in some defaults
            styles_.replace("Generic Ale", Style());
            grains_.append(Grain());
            hops_.append(Hops());
            misc_.append(MiscIngredient());
            UEntry u = {0, 20};
            utable_.append(u);
        }
    }

    QObject::connect(document_,SIGNAL(recalc()), this, SLOT(slotRecalc()));
}

//////////////////////////////////////////////////////////////////////////////
// ~QBrewCalc()
// ------------
// Destructor

QBrewCalc::~QBrewCalc() {}

//////////////////////////////////////////////////////////////////////////////
// Database Access                                                          //
//////////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////////
// stylesList()
// ------------
// Return a string list of available styles

QStringList QBrewCalc::stylesList()
{
    QStringList list;
    QMap<QString, Style>::Iterator it;
    for (it=styles_.begin(); it!=styles_.end(); ++it) {
        list += it.key();
    }
    list.sort();
    return list;
}

//////////////////////////////////////////////////////////////////////////////
// style()
// -------
// Return a style given its name

Style* QBrewCalc::style(const QString &name)
{
    return &styles_[name];
}

//////////////////////////////////////////////////////////////////////////////
// grainsList()
// ------------
// Return string list of available grains

QStringList QBrewCalc::grainsList()
{
    QStringList list;
    GrainList::Iterator it;
    for (it=grains_.begin(); it!=grains_.end(); ++it) {
        list += (*it).name();
    }
    list.sort();
    return list;
}

//////////////////////////////////////////////////////////////////////////////
// grain()
// -------
// Return grain given its name

Grain* QBrewCalc::grain(const QString &name)
{
    GrainList::Iterator it;
    for (it=grains_.begin(); it!=grains_.end(); ++it) {
        if (name == (*it).name()) return &(*it);
    }
    return 0;
}

//////////////////////////////////////////////////////////////////////////////
// hopsList()
// ----------
// Return string list of available hops

QStringList QBrewCalc::hopsList()
{
    QStringList list;
    HopsList::Iterator it;
    for (it=hops_.begin(); it != hops_.end(); ++it) {
        list += (*it).name();
    }
    list.sort();
    return list;
}

//////////////////////////////////////////////////////////////////////////////
// formsList()
// ----------
// Return string list of available hop forms

QStringList QBrewCalc::formsList()
{
    QStringList list;
    list += "pellet";   // defaults
    list += "plug";
    list += "whole";
    HopsList::Iterator it;
    for (it=hops_.begin(); it!=hops_.end(); ++it) {
        if (list.contains((*it).form()) == 0) list += (*it).form();
    }
    list.sort();
    return list;
}

//////////////////////////////////////////////////////////////////////////////
// hop()
// -----
// Return hop given its name

Hops* QBrewCalc::hop(const QString &name)
{
    HopsList::Iterator it;
    for (it=hops_.begin(); it!=hops_.end(); ++it) {
        if (name == (*it).name()) return &(*it);
    }
    return 0;
}

//////////////////////////////////////////////////////////////////////////////
// miscList()
// ----------
// Return string list of available misc ingredients

QStringList QBrewCalc::miscList()
{
    QStringList list;
    MiscIngredientList::Iterator it;
    for (it=misc_.begin(); it!=misc_.end(); ++it) {
        list += (*it).name();
    }
    list.sort();
    return list;
}

//////////////////////////////////////////////////////////////////////////////
// misc()
// ------
// Return misc ingredient given its name

MiscIngredient* QBrewCalc::misc(const QString &name)
{
    MiscIngredientList::Iterator it;
    for (it=misc_.begin(); it!=misc_.end(); ++it) {
        if (name == (*it).name()) return &(*it);
    }
    return 0;
}

//////////////////////////////////////////////////////////////////////////////
// Slots                                                                    //
//////////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////////
// slot Recalc()
// -------------
// Received if we need to recalculate stuff

void QBrewCalc::slotRecalc()
{
    og_ = calcOG();
    ibu_ = calcIBU();
    srm_ = calcSRM();
    emit calcDone();
}

//////////////////////////////////////////////////////////////////////////////
// Calculations                                                             //
//////////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////////
// calcOG()
// --------
// Calculate the original gravity

double QBrewCalc::calcOG()
{
    GrainList *list = document_->grainList();
    double est = 0.0;
    GrainList::Iterator it;
    for (it=list->begin(); it!=list->end(); ++it) {
        double yield = (*it).yield();
        switch ((*it).use()) {
            case GRAIN_MASHED:
                // adjust for mash efficiency
                yield *= preferences_->getDouble(ID_PREF_EFFICIENCY, ID_PREF_EFFICIENCY_DEFAULT);
                break;
            case GRAIN_STEEPED:
                // steeped grains don't yield nearly as much as mashed grains
                yield *= STEEP_YIELD;
                break;
            case GRAIN_EXTRACT:
                break;  // no modifications necessary
            default:
                break;
        }
        est += yield;
    }
    est /= document_->size();
    return est + 1.0;
}

//////////////////////////////////////////////////////////////////////////////
// calcIBU()
// --------
// Calculate the bitterness

double QBrewCalc::calcIBU()
{
    // switch between two possible calculations
	if (preferences_->getBool(ID_PREF_TINSETH, ID_PREF_TINSETH_DEFAULT))
		return calcTinsethIBU();
	else
		return calcRagerIBU();
}

//////////////////////////////////////////////////////////////////////////////
// calcRagerIBU()
// --------
// Calculate the bitterness based on Rager's method (table method)

double QBrewCalc::calcRagerIBU()
{
    HopsList *list = document_->hopsList();
    double bitterness = 0.0;
    HopsList::Iterator it;
    for (it=list->begin(); it!=list->end(); ++it) {
        bitterness += (*it).HBU() * utilization((*it).time());
        // TODO: we should also correct for hop form
    }
    bitterness /= document_->size();
    // correct for boil gravity
    if (og_ > 1.050) bitterness /= 1.0 + ((og_ - 1.050) / 0.2);
    return bitterness;
}

//////////////////////////////////////////////////////////////////////////////
// calcTinsethIBU()
// --------
// Calculate the bitterness based on Tinseth's method (formula method)
// The formula used is:
// (1.65*0.000125^(gravity-1))*(1-EXP(-0.04*time))*alpha*mass*1000
// ---------------------------------------------------------------
// (volume*4.15)

// TODO: check and recheck this formula

double QBrewCalc::calcTinsethIBU()
{
    HopsList *list = document_->hopsList();
    double bitterness = 0.0;
    HopsList::Iterator it;
    for (it=list->begin(); it!=list->end(); ++it) {
        double ibu_ = (COEFF1 * pow(COEFF2, (og_ - 1.0))) *
        	(1.0 - exp(-COEFF3 * (*it).time())) *
        	((*it).alpha()) * (*it).quantity() * 1000.0;
    	ibu_ /= (document_->size()) * COEFF4;
    	bitterness += ibu_;
    }
    bitterness *= (GPO / LPG) / 100.0;
    return bitterness;
}

//////////////////////////////////////////////////////////////////////////////
// utilization
// -----------
// Look up the utilization for the given time

double QBrewCalc::utilization(const unsigned &time)
{
    QValueList<UEntry>::Iterator it;
    for (it=utable_.begin(); it!=utable_.end(); ++it) {
        if (time >= (*it).time) return (double((*it).utilization));
    }
    return 0.0;
}

//////////////////////////////////////////////////////////////////////////////
// calcSRM()
// ---------
// Calculate the color

double QBrewCalc::calcSRM()
{
    GrainList *list = document_->grainList();
    double est = 0.0;
    GrainList::Iterator it;
    for (it=list->begin(); it!=list->end(); ++it) {
        est += (*it).HCU();
    }
    est /= document_->size();
    if (est > 8.0) {
        est *= 0.2; est += 8.4;
    }
    return est;
}

// TODO: following formulas need to use constants

double QBrewCalc::FGEstimate() { return (((og_ - 1.0) * 0.25) + 1.0); }

double QBrewCalc::ABV() { return ((og_ - FGEstimate()) * 1.29); }

double QBrewCalc::ABW() { return (ABV() * 0.785); }

//////////////////////////////////////////////////////////////////////////////
// Serialization                                                            //
//////////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////////
// loadData()
// ----------
// Load the calc data file

bool QBrewCalc::loadData(const QString &filename, bool quiet)
{
    // TODO: need more error checking on tags and elements
    // open file
    QFile* datafile = new QFile(filename);
    if (!datafile->open(IO_ReadOnly)) {
        // error opening file
        if (!quiet) qWarning(tr("Error: Cannot open ") + filename);
        datafile->close();
        delete (datafile);
        return false;
    }

    // open dom document
    QDomDocument doc;
    doc.setContent(datafile);
    datafile->close();
    delete (datafile);

    // check the doc type and stuff
    if (doc.doctype().name() != tagDoc) {
        // wrong file type
        if (!quiet) qWarning(tr("Error: Wrong file type ") + filename);
        return false;
    }
    QDomElement root = doc.documentElement();

    // check file version
    if (root.attribute(attrVersion) < CALC_PREVIOUS) {
        // too old of a version
        if (!quiet) qWarning(tr("Error: Unsupported version ") + filename);
        return false;
    }

    // get all styles tags
    QDomNodeList nodes = root.elementsByTagName(tagStyles);
    QDomNodeList subnodes;
    QDomElement element;
    QDomElement subelement;
    for (unsigned n=0; n<nodes.count(); ++n) {
        if (nodes.item(n).isElement()) {
            element = nodes.item(n).toElement();
            // get all style tags
            subnodes = element.elementsByTagName(tagStyle);
            for (unsigned m=0; m<subnodes.count(); m++) {
                if (subnodes.item(m).isElement()) {
                    subelement = subnodes.item(m).toElement();
                    styles_.replace(subelement.text(),
                        Style(subelement.attribute(attrOGLow).toDouble(),
                            subelement.attribute(attrOGHigh).toDouble(),
                            subelement.attribute(attrIBULow).toDouble(),
                            subelement.attribute(attrIBUHigh).toDouble(),
                            subelement.attribute(attrSRMLow).toDouble(),
                            subelement.attribute(attrSRMHigh).toDouble()));
                }
            }
        }
    }

   // get all grains tags
    nodes = root.elementsByTagName(tagGrains);
    for (unsigned n=0; n<nodes.count(); ++n) {
        if (nodes.item(n).isElement()) {
            element = nodes.item(n).toElement();
            // get all grain tags
            subnodes = element.elementsByTagName(tagGrain);
            for (unsigned m=0; m<subnodes.count(); m++) {
                if (subnodes.item(m).isElement()) {
                    subelement = subnodes.item(m).toElement();
                    grains_.append(Grain(subelement.text(),
                        subelement.attribute(attrQuantity).toDouble(),
                        subelement.attribute(attrExtract).toDouble(),
                        subelement.attribute(attrColor).toDouble(),
                        subelement.attribute(attrUse)));
                }
            }
        }
    }

    // get all hops tags
    nodes = root.elementsByTagName(tagHops);
    for (unsigned n=0; n<nodes.count(); ++n) {
        if (nodes.item(n).isElement()) {
            element = nodes.item(n).toElement();
            // get all hop tags
            subnodes = element.elementsByTagName(tagHop);
            for (unsigned m=0; m<subnodes.count(); m++) {
                if (subnodes.item(m).isElement()) {
                    subelement = subnodes.item(m).toElement();
                    hops_.append(Hops(subelement.text(),
                        subelement.attribute(attrQuantity).toDouble(),
                        subelement.attribute(attrForm),
                        subelement.attribute(attrAlpha).toDouble(),
                        subelement.attribute(attrTime).toUInt()));
                }
            }
        }
    }

    // get all miscingredients tags
    nodes = root.elementsByTagName(tagMiscIngredients);
    for (unsigned n=0; n<nodes.count(); ++n) {
        if (nodes.item(n).isElement()) {
            element = nodes.item(n).toElement();
            // get all miscingredient tags
            subnodes = element.elementsByTagName(tagMiscIngredient);
            for (unsigned m=0; m<subnodes.count(); m++) {
                if (subnodes.item(m).isElement()) {
                    subelement = subnodes.item(m).toElement();
                    misc_.append(MiscIngredient(subelement.text(),
                        subelement.attribute(attrQuantity).toDouble(),
                        subelement.attribute(attrNotes)));
                }
            }
        }
    }

    // get all utilization tags
    nodes = root.elementsByTagName(tagUtilization);
    UEntry entry;
    for (unsigned n=0; n<nodes.count(); ++n) {
        if (nodes.item(n).isElement()) {
            element = nodes.item(n).toElement();
            // get all entry tags
            subnodes = element.elementsByTagName(tagEntry);
            for (unsigned m=0; m<subnodes.count(); m++) {
                if (subnodes.item(m).isElement()) {
                    subelement = subnodes.item(m).toElement();
                    entry.time = subelement.attribute(attrTime).toUInt();
                    entry.utilization = subelement.attribute(attrUtil).toUInt();
                    // keep the list sorted from highest time to lowest
                    QValueList<UEntry>::Iterator it;
                    if (utable_.isEmpty()) {
                        utable_.append(entry);
                    } else {
                        for (it=utable_.begin(); it != utable_.end(); ++it) {
                            if ((*it).time < entry.time)
                            break;
                        }
                        utable_.insert(it, entry);
                    }
                }
            }
        }
    }
    return true;
}

//////////////////////////////////////////////////////////////////////////////
// convert()
// ---------
// Convert filename to the new calc data file

bool QBrewCalc::convert(const QString& filename)
{
    // save the path
    QString path = "";
    int pos = filename.findRev('/');
    if (pos > 0) path = filename.left(pos+1);
    // open file
    OldStore *rfile = new OldStore(IO_ReadOnly, filename, CALC_OLDFORMAT, CALC_OLDVERSION);
    if (!rfile->good()) {
        qWarning(tr("Error: Cannot open ") + filename);
        delete rfile;
        return false;
    } else {
        // file opened without error
        if (rfile->getVersion() < CALC_OLDPREVIOUS) {
            qWarning(tr("Error: Cannot convert from this version of ") + filename);
            delete rfile;
            return false;
        }

        // clear existing data
        styles_.clear();
        grains_.clear();
        hops_.clear();
        misc_.clear();
        utable_.clear();

        // read in file line by line
        Style newstyle;
        Grain newgrain;
        Hops newhop;
        MiscIngredient newmisc;
        UEntry entry;
        QValueList<UEntry>::Iterator uit;
        unsigned ID;
        while (rfile->getLine()) {
            ID = groupmap_[rfile->getGroup()];
            if (rfile->getGroup() != rfile->getName()) {
                switch (ID) {
                    case gidStyles:
                        newstyle.obsoleteSerializeIn(rfile->getValue());
                        styles_.replace(rfile->getName(), newstyle);
                        break;
                    case gidCalcGrains:
                        newgrain.obsoleteSerializeIn(rfile->getName(), rfile->getValue());
                        grains_.append(newgrain);
                        break;
                    case gidCalcHops:
                        newhop.obsoleteSerializeIn(rfile->getName(), rfile->getValue());
                        hops_.append(newhop);
                        break;
                    case gidCalcMisc:
                        newmisc.obsoleteSerializeIn(rfile->getName(), rfile->getValue());
                        misc_.append(newmisc);
                        break;
                    case gidUtilTable:
                        entry.time = rfile->getName().toUInt();
                        entry.utilization = rfile->getValue().toUInt();
                        // keep the list sorted from highest time to lowest
                        if (utable_.isEmpty()) {
                            utable_.append(entry);
                        } else {
                            for (uit=utable_.begin(); uit != utable_.end(); ++uit) {
                                if ((*uit).time < entry.time)
                                    break;
                            }
                            utable_.insert(uit, entry);
                        }
                        break;
                    default:
                        qWarning("Warning: " + rfile->getGroup() + tr(" is not a valid group"));
                        break;
                }
            }
        }
    }
    delete rfile;
    saveData(path + ID_CALC_FILE);
    return true;
}

//////////////////////////////////////////////////////////////////////////////
// saveData()
// ------------
// Save info to calc data file

void QBrewCalc::saveData(const QString &filename)
{
    QDomDocument doc(tagDoc);

    // create the root element
    QDomElement root = doc.createElement(doc.doctype().name());
    root.setAttribute(attrVersion, VERSION);
    doc.appendChild(root);

    // styles elements
    QDomElement element = doc.createElement(tagStyles);
    QMap<QString, Style>::Iterator sit;
    QDomElement subelement;
    // iterate through styles_ list
    for (sit=styles_.begin(); sit!=styles_.end(); ++sit) {
        subelement = doc.createElement(tagStyle);
        subelement.appendChild(doc.createTextNode(sit.key()));
        subelement.setAttribute(attrOGLow, sit.data().OGLow());
        subelement.setAttribute(attrOGHigh, sit.data().OGHi());
        subelement.setAttribute(attrIBULow, sit.data().IBULow());
        subelement.setAttribute(attrIBUHigh, sit.data().IBUHi());
        subelement.setAttribute(attrSRMLow, sit.data().SRMLow());
        subelement.setAttribute(attrSRMHigh, sit.data().SRMHi());
        element.appendChild(subelement);
    }
    root.appendChild(element);

    // grains elements
    element = doc.createElement(tagGrains);
    GrainIterator git;
    // iterate through grains_ list
    for (git=grains_.begin(); git!=grains_.end(); ++git) {
        subelement = doc.createElement(tagGrain);
        subelement.appendChild(doc.createTextNode((*git).name()));
        subelement.setAttribute(attrQuantity, (*git).quantity());
        subelement.setAttribute(attrExtract, (*git).extract());
        subelement.setAttribute(attrColor, (*git).color());
        subelement.setAttribute(attrUse, (*git).useString());
        element.appendChild(subelement);
    }
    root.appendChild(element);

    // hops elements
    element = doc.createElement(tagHops);
    HopIterator hit;
    // iterate through hops_ list
    for (hit=hops_.begin(); hit!=hops_.end(); ++hit) {
        subelement = doc.createElement(tagHop);
        subelement.appendChild(doc.createTextNode((*hit).name()));
        subelement.setAttribute(attrQuantity, (*hit).quantity());
        subelement.setAttribute(attrForm, (*hit).form());
        subelement.setAttribute(attrAlpha, (*hit).alpha());
        subelement.setAttribute(attrTime, (*hit).time());
        element.appendChild(subelement);
    }
    root.appendChild(element);

    // miscingredients elements
    element = doc.createElement(tagMiscIngredients);
    MiscIngredientIterator mit;
    // iterate through misc_ list
    for (mit=misc_.begin(); mit!=misc_.end(); ++mit) {
        subelement = doc.createElement(tagMiscIngredient);
        subelement.appendChild(doc.createTextNode((*mit).name()));
        subelement.setAttribute(attrQuantity, (*mit).quantity());
        subelement.setAttribute(attrNotes, (*mit).notes());
        element.appendChild(subelement);
    }
    root.appendChild(element);

    // utilization elements
    element = doc.createElement(tagUtilization);
    QValueList<UEntry>::Iterator uit;
    // iterate through _uentry list
    for (uit=utable_.begin(); uit!=utable_.end(); ++uit) {
        subelement = doc.createElement(tagEntry);
        subelement.setAttribute(attrTime, (*uit).time);
        subelement.setAttribute(attrUtil, (*uit).utilization);
        element.appendChild(subelement);
    }
    root.appendChild(element);

    // open file
    QFile* datafile = new QFile(filename);
    if (!datafile->open(IO_WriteOnly)) {
        // error opening file
        qWarning(tr("Error: Cannot open file ") + filename);
        datafile->close();
    }

    // write it out
    QTextStream textstream(datafile);
    doc.save(textstream, 0);
    datafile->close();
    delete (datafile);
}
