/**********************************************************************************************
    Copyright (C) 2006, 2007 Oliver Eichler oliver.eichler@gmx.de

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111 USA

  Garmin and MapSource are registered trademarks or trademarks of Garmin Ltd.
  or one of its subsidiaries.

**********************************************************************************************/

#include "CGarminMap.h"
#include "CGarminImg.h"
#include "IProjection.h"
#include "GarminIcons.h"
#include "CCanvas.h"

#define MAX_IDX_ZOOM 34
#define MIN_IDX_ZOOM 0
#undef DEBUG_SHOW_SECTION_BORDERS

CGarminMap::scale_t CGarminMap::scales[] =
{
    {QString("7000 km"), 70000.0, 8}
    ,{QString("5000 km"), 50000.0, 8}
    ,{QString("3000 km"), 30000.0, 9}
    ,{QString("2000 km"), 20000.0, 9}
    ,{QString("1500 km"), 15000.0, 10}
    ,{QString("1000 km"), 10000.0, 10}
    ,{QString("700 km"), 7000.0, 11}
    ,{QString("500 km"), 5000.0, 11}
    ,{QString("300 km"), 3000.0, 13}
    ,{QString("200 km"), 2000.0, 13}
    ,{QString("150 km"), 1500.0, 13}
    ,{QString("100 km"), 1000.0, 14}
    ,{QString("70 km"), 700.0, 15}
    ,{QString("50 km"), 500.0, 16}
    ,{QString("30 km"), 300.0, 16}
    ,{QString("20 km"), 200.0, 17}
    ,{QString("15 km"), 150.0, 17}
    ,{QString("10 km"), 100.0, 18}
    ,{QString("7 km"), 70.0, 18}
    ,{QString("5 km"), 50.0, 19}
    ,{QString("3 km"), 30.0, 19}
    ,{QString("2 km"), 20.0, 20}
    ,{QString("1.5 km"), 15.0, 22}
    ,{QString("1 km"), 10.0, 24}
    ,{QString("700 m"), 7.0, 24}
    ,{QString("500 m"), 5.0, 24}
    ,{QString("300 m"), 3.0, 24}
    ,{QString("200 m"), 2.0, 24}
    ,{QString("150 m"), 1.5, 24}
    ,{QString("100 m"), 1.0, 24}
    ,{QString("70 m"), 0.7, 24}
    ,{QString("50 m"), 0.5, 24}
    ,{QString("30 m"), 0.3, 24}
    ,{QString("20 m"), 0.2, 24}
    ,{QString("15 m"), 0.1, 24}
    ,{QString("10 m"), 0.15, 24}

};

// #ifndef DYNAMIC_PROJ
// CGarminMap::scale_t CGarminMap::scales[] = {
//      {QString("7000 km"), 70000.0, 8}
//     ,{QString("5000 km"), 50000.0, 9}
//     ,{QString("3000 km"), 30000.0, 9}
//     ,{QString("2000 km"), 20000.0, 10}
//     ,{QString("1500 km"), 15000.0, 10}
//     ,{QString("1000 km"), 10000.0, 11}
//     ,{QString("700 km"), 7000.0, 11}
//     ,{QString("500 km"), 5000.0, 13}
//     ,{QString("300 km"), 3000.0, 13}
//     ,{QString("200 km"), 2000.0, 14}
//     ,{QString("150 km"), 1500.0, 15}
//     ,{QString("100 km"), 1000.0, 16}
//     ,{QString("70 km"), 700.0, 16}
//     ,{QString("50 km"), 500.0, 17}
//     ,{QString("30 km"), 300.0, 17}
//     ,{QString("20 km"), 200.0, 18}
//     ,{QString("15 km"), 150.0, 18}
//     ,{QString("10 km"), 100.0, 19}
//     ,{QString("7 km"), 70.0, 19}
//     ,{QString("5 km"), 50.0, 20}
//     ,{QString("3 km"), 30.0, 22}
//     ,{QString("2 km"), 20.0, 24}
//     ,{QString("1.5 km"), 15.0, 24}
//     ,{QString("1 km"), 10.0, 24}
//     ,{QString("700 m"), 7.0, 24}
//     ,{QString("500 m"), 5.0, 24}
//     ,{QString("300 m"), 3.0, 24}
//     ,{QString("200 m"), 2.0, 24}
//     ,{QString("150 m"), 1.5, 24}
//     ,{QString("100 m"), 1.0, 24}
//     ,{QString("70 m"), 0.7, 24}
//     ,{QString("50 m"), 0.5, 24}
//     ,{QString("30 m"), 0.3, 24}
//     ,{QString("20 m"), 0.2, 24}
//     ,{QString("15 m"), 0.1, 24}
//     ,{QString("10 m"), 0.15, 24}
// };
// #endif

struct defPolyline_t
{
    defPolyline_t(const QColor& p0, const QColor& p1, const quint32& t, const QFont& f, const Qt::PenStyle& s0, const Qt::PenStyle& s1, const float& scale = -1.0)
        : color0(p0)
        , color1(p1)
        , thickness(t)
        , scale(scale)
        , font(f)
        , style0(s0)
        , style1(s1)
        {}

    defPolyline_t()
        : color0(Qt::black)
        , color1(Qt::black)
        , thickness(2)
        , scale(-1)
        , font(QFont("Sans",10))
        , style0(Qt::SolidLine)
        , style1(Qt::SolidLine)
        {}

    QColor          color0;
    QColor          color1;
    quint32         thickness;
    float           scale;
    QFont           font;
    Qt::PenStyle    style0;
    Qt::PenStyle    style1;
};

struct defPolygon_t
{
    defPolygon_t(const QColor& p0, const QFont& f, const Qt::BrushStyle& s0,const QPen& pen = Qt::NoPen)
        : pen0(pen)
        , color0(p0)
        , font(f)
        , style0(s0)
        {}

    defPolygon_t()
        : pen0(Qt::magenta,2)
        , color0(Qt::magenta)
        , font(QFont("Sans",10))
        , style0(Qt::BDiagPattern)
        {}

    QPen            pen0;
    QColor          color0;
    QFont           font;
    Qt::BrushStyle    style0;

};

static QMap<quint32,defPolyline_t> dictPolyline;
static QMap<quint32,defPolygon_t> dictPolygon;

static void setSkinValue(defPolyline_t& def, const QString& token, const QString& value)
{
    if(token == "Color") {
        def.color0 = QColor(value);
    }
    else if(token == "Color1") {
        def.color1 = QColor(value);
    }
    else if(token == "Thickness") {
        def.thickness = value.toUInt();
    }
    else if(token == "Style") {
        if(value == "Solid") {
            def.style0 = Qt::SolidLine;
        }
        else if (value == "Dot") {
            def.style0 = Qt::DotLine;
        }
        else if (value == "Dash") {
            def.style0 = Qt::DashLine;
        }
        else if (value == "DashDot") {
            def.style0 = Qt::DashDotLine;
        }
    }
    else if(token == "Style1") {
        if(value == "Solid") {
            def.style1 = Qt::SolidLine;
        }
        else if (value == "Dot") {
            def.style1 = Qt::DotLine;
        }
        else if (value == "Dash") {
            def.style1 = Qt::DashLine;
        }
        else if (value == "DashDot") {
            def.style1 = Qt::DashDotLine;
        }
    }
}


static void setSkinValue(defPolygon_t& def, const QString& token, const QString& value)
{
    if(token == "BrushColor") {
        if(QColor(value) == QColor("#000000")) {
            def.color0 = Qt::transparent;
        }
        else {
            def.color0 = QColor(value);
        }
    }
    if(token == "Color") {
        def.pen0 = QPen(QColor(value));
    }
    else if(token == "BrushStyle") {
        if(value == "Solid") {
            def.style0 = Qt::SolidPattern;
        }
        else if(value == "Dense") {
            def.style0 = Qt::Dense5Pattern;
        }
        else if(value == "Diag") {
            def.style0 = Qt::DiagCrossPattern;
        }
    }
}


CGarminMap::CGarminMap(QObject * parent)
: IMap(parent)
, idxZoom(11)
, mapLevelMap(0)
, mapLevelOvl(0)
, bitBias(0)
{

    QSettings cfg;
    XY pt;
    pt.u            = cfg.value("map/left_margin", 5.5).toDouble() * DEG_TO_RAD;
    pt.v            = cfg.value("map/top_margin", 55.2).toDouble() * DEG_TO_RAD;
    idxZoom         = cfg.value("map/zoom",11).toUInt();
    showCityNames   = cfg.value("map/showCityNames",true).toBool();
    showPOIs        = cfg.value("map/showPOIs",true).toBool();
    showStreetNames = cfg.value("map/showStreetNames",true).toBool();

    pt = pj_fwd(pt,*gpProj);

    if(pt.u != HUGE_VAL) {
        x_off = pt.u;
        y_off = pt.v;
    }
    else {
        qDebug() << "error";
    }
    setScale( idxZoom,QPoint(0,0));

    connect(&gpResources->mapdb(),SIGNAL(sigMapChanged(QRectF&)),this,SLOT(slotMapChanged(QRectF&)));
    connect(&gpResources->mapdb(),SIGNAL(sigMapChanged()),this,SLOT(slotMapChanged()));

    loadSkin(QDir::home().filePath(".config/QLandkarte/skin.txt"));
}


CGarminMap::~CGarminMap()
{
    QSettings cfg;

    XY pt;
    pt.u = x_off;
    pt.v = y_off;
    pt = pj_inv(pt,*gpProj);
    if(pt.u != HUGE_VAL) {
        cfg.setValue("map/left_margin", pt.u * RAD_TO_DEG);
        cfg.setValue("map/top_margin", pt.v * RAD_TO_DEG);
    }
    cfg.setValue("map/zoom",idxZoom);
    cfg.setValue("map/showCityNames",showCityNames);
    cfg.setValue("map/showPOIs",showPOIs);
    cfg.setValue("map/showStreetNames",showStreetNames);
}


void CGarminMap::loadSkin(const QString& filename)
{
    QFile skinfile(filename);

    QFont defaultFont("Sans",10);
    /*"Major highway          "*/   dictPolyline[0x01] = defPolyline_t(Qt::black, Qt::blue, 5, defaultFont, Qt::SolidLine, Qt::SolidLine, 2.0);
    /*"Major highway connector"*/   dictPolyline[0x0B] = defPolyline_t(Qt::black, Qt::blue, 5, defaultFont, Qt::SolidLine, Qt::SolidLine, 2.0);
    /*"Highway ramp, low speed"*/   dictPolyline[0x08] = defPolyline_t(Qt::black, "#FF9900", 5, defaultFont, Qt::SolidLine, Qt::SolidLine, 2.0);
    /*"Highway ramp, high speed"*/  dictPolyline[0x09] = defPolyline_t(Qt::black, "#FF9900", 5, defaultFont, Qt::SolidLine, Qt::SolidLine, 2.0);
    /*"Principal highway"*/         dictPolyline[0x02] = defPolyline_t(Qt::black, "#FF9900", 5, defaultFont, Qt::SolidLine, Qt::SolidLine, 2.0);
    /*"Other highway"*/             dictPolyline[0x03] = defPolyline_t(Qt::black, Qt::yellow, 3, defaultFont, Qt::SolidLine, Qt::SolidLine, 2.0);
    /*"Arterial road",*/            dictPolyline[0x04] = defPolyline_t(Qt::black, Qt::black, 3, defaultFont, Qt::SolidLine, Qt::SolidLine, 2.0);
    /*"Roundabout"*/                dictPolyline[0x0C] = defPolyline_t(Qt::black, Qt::black, 3, defaultFont, Qt::SolidLine, Qt::SolidLine, 2.0);
    /*"Collector road"*/            dictPolyline[0x05] = defPolyline_t(Qt::black, Qt::black, 3, defaultFont, Qt::SolidLine, Qt::SolidLine, 2.0);
    /*"Residential street"*/        dictPolyline[0x06] = defPolyline_t(Qt::black, Qt::white, 4, defaultFont, Qt::SolidLine, Qt::SolidLine);
    /*"Unpaved road"*/              dictPolyline[0x0a] = defPolyline_t(Qt::darkGray, Qt::gray, 4, defaultFont, Qt::SolidLine, Qt::SolidLine);
    /*"Alley/Private road"*/        dictPolyline[0x07] = defPolyline_t(Qt::darkGray, Qt::gray, 4, defaultFont, Qt::SolidLine, Qt::SolidLine);
    /*"Railroad"*/                  dictPolyline[0x14] = defPolyline_t(Qt::black, Qt::white, 4, defaultFont, Qt::SolidLine, Qt::DashLine);
    /*"Shoreline*/                  dictPolyline[0x15] = defPolyline_t(Qt::yellow, Qt::transparent, 2, defaultFont, Qt::SolidLine, Qt::SolidLine);
    /*"Trail ???"*/                 dictPolyline[0x1e] = defPolyline_t(Qt::darkGray, Qt::white, 2, defaultFont, Qt::SolidLine, Qt::DotLine);
    /*"Trail ??"*/                  dictPolyline[0x0f] = defPolyline_t(Qt::darkGray, Qt::white, 2, defaultFont, Qt::SolidLine, Qt::DotLine);
    /*"Trail"*/                     dictPolyline[0x16] = defPolyline_t(Qt::darkGray, Qt::white, 2, defaultFont, Qt::SolidLine, Qt::DotLine);
    /*"River"*/                     dictPolyline[0x1f] = defPolyline_t(Qt::blue, Qt::transparent, 2, defaultFont, Qt::SolidLine, Qt::SolidLine);
    /*"Stream"*/                    dictPolyline[0x18] = defPolyline_t(Qt::blue, Qt::transparent, 2, defaultFont, Qt::SolidLine, Qt::SolidLine);
    /*"Intermittent stream"*/       dictPolyline[0x26] = defPolyline_t(Qt::blue, Qt::transparent, 2, defaultFont, Qt::SolidLine, Qt::SolidLine);
    /*"Time zone"*/                 dictPolyline[0x19] = defPolyline_t(Qt::red, Qt::transparent, 2, defaultFont, Qt::SolidLine, Qt::SolidLine);
    /*"Ferry"*/                     dictPolyline[0x1a] = defPolyline_t(Qt::black, Qt::transparent, 2, defaultFont, Qt::DashLine, Qt::SolidLine);
    /*"Ferry"*/                     dictPolyline[0x1b] = defPolyline_t(Qt::black, Qt::transparent, 2, defaultFont, Qt::DashLine, Qt::SolidLine);
    /*"State/province border"*/     dictPolyline[0x1c] = defPolyline_t(Qt::red, Qt::transparent, 2, defaultFont, Qt::DashDotLine, Qt::SolidLine);
    /*"County/parish border"*/      dictPolyline[0x1d] = defPolyline_t(Qt::red, Qt::transparent, 2, defaultFont, Qt::DashDotLine, Qt::SolidLine);
    /*"International border"*/      dictPolyline[0x1e] = defPolyline_t(Qt::red, Qt::transparent, 3, defaultFont, Qt::DashDotLine, Qt::SolidLine);
    /*"Minor depht contour"*/       dictPolyline[0x23] = defPolyline_t("#846030", Qt::transparent, 1, defaultFont, Qt::SolidLine, Qt::SolidLine);
    /*"Minor land contour"*/        dictPolyline[0x20] = defPolyline_t("#846030", Qt::transparent, 1, defaultFont, Qt::SolidLine, Qt::SolidLine);
    /*"Intermediate depth contour"*/dictPolyline[0x24] = defPolyline_t("#846030", Qt::transparent, 2, defaultFont, Qt::SolidLine, Qt::SolidLine);
    /*"Intermediate land contour"*/ dictPolyline[0x21] = defPolyline_t("#846030", Qt::transparent, 2, defaultFont, Qt::SolidLine, Qt::SolidLine);
    /*"Major depth contour",*/      dictPolyline[0x25] = defPolyline_t("#846030", Qt::transparent, 3, defaultFont, Qt::SolidLine, Qt::SolidLine);
    /*"Major land contour",*/       dictPolyline[0x22] = defPolyline_t("#846030", Qt::transparent, 3, defaultFont, Qt::SolidLine, Qt::SolidLine);
    /*"Airport runway",*/           dictPolyline[0x27] = defPolyline_t(Qt::darkGray, Qt::transparent, 3, defaultFont, Qt::SolidLine, Qt::SolidLine);
    /*"Pipeline",*/                 dictPolyline[0x28] = defPolyline_t(Qt::magenta, Qt::transparent, 2, defaultFont, Qt::SolidLine, Qt::SolidLine);
    /*"Powerline",*/                dictPolyline[0x29] = defPolyline_t(Qt::yellow, Qt::transparent, 2, defaultFont, Qt::SolidLine, Qt::SolidLine);

    /*                           */ dictPolygon[0x00] = defPolygon_t(Qt::transparent,defaultFont,Qt::SolidPattern);
    /*"Large urban area (>200K)",*/ dictPolygon[0x01] = defPolygon_t("#ff1c1c",defaultFont,Qt::SolidPattern);
    /*"Small urban area (<200K)",*/ dictPolygon[0x02] = defPolygon_t("#ff1c1c",defaultFont,Qt::SolidPattern);
    /*"Rural housing area",      */ dictPolygon[0x03] = defPolygon_t("#ff1c1c",defaultFont,Qt::SolidPattern);
    /*"Military base",           */ dictPolygon[0x04] = defPolygon_t("#666600",defaultFont,Qt::BDiagPattern);
    /*"Parking lot",             */ dictPolygon[0x05] = defPolygon_t("#66ccff",defaultFont,Qt::SolidPattern);
    /*"Parking garage",          */ dictPolygon[0x06] = defPolygon_t("#66ccff",defaultFont,Qt::SolidPattern);
    /*"Airport",                 */ dictPolygon[0x07] = defPolygon_t(Qt::gray,defaultFont,Qt::SolidPattern);
    /*"Shopping center",         */ dictPolygon[0x08] = defPolygon_t("#ffff00",defaultFont,Qt::SolidPattern);
    /*"Marina",                  */ dictPolygon[0x09] = defPolygon_t(Qt::darkRed,defaultFont,Qt::SolidPattern);
    /*"University/College",      */ dictPolygon[0x0a] = defPolygon_t(Qt::darkRed,defaultFont,Qt::SolidPattern);
    /*"Hospital",                */ dictPolygon[0x0b] = defPolygon_t(Qt::darkRed,defaultFont,Qt::SolidPattern);
    /*"Industrial complex",      */ dictPolygon[0x0c] = defPolygon_t(Qt::darkRed,defaultFont,Qt::SolidPattern);
    /*"Reservation",             */ dictPolygon[0x0d] = defPolygon_t(Qt::darkRed,defaultFont,Qt::SolidPattern);
    /*"Airport runway",          */ dictPolygon[0x0e] = defPolygon_t(Qt::darkGray,defaultFont,Qt::SolidPattern);
    /*"Man-made area",           */ dictPolygon[0x13] = defPolygon_t(Qt::darkRed,defaultFont,Qt::SolidPattern);
    /*"Sports complex",          */ dictPolygon[0x19] = defPolygon_t(Qt::darkRed,defaultFont,Qt::SolidPattern);
    /*"Golf course",             */ dictPolygon[0x18] = defPolygon_t(Qt::green,defaultFont,Qt::SolidPattern);
    /*"Cemetary",                */ dictPolygon[0x1a] = defPolygon_t(Qt::darkGreen,defaultFont,Qt::Dense5Pattern);
    /*"National park",           */ dictPolygon[0x14] = defPolygon_t("#67eb5d",defaultFont,Qt::SolidPattern);
    /*"National park",           */ dictPolygon[0x15] = defPolygon_t("#67eb5d",defaultFont,Qt::SolidPattern);
    /*"National park",           */ dictPolygon[0x16] = defPolygon_t("#67eb5d",defaultFont,Qt::SolidPattern);
    /*"City park",               */ dictPolygon[0x17] = defPolygon_t("#67eb5d",defaultFont,Qt::SolidPattern);
    /*"State park",              */ dictPolygon[0x1e] = defPolygon_t("#67eb5d",defaultFont,Qt::SolidPattern);
    /*"State park",              */ dictPolygon[0x1f] = defPolygon_t("#67eb5d",defaultFont,Qt::SolidPattern);
    /*"State park",              */ dictPolygon[0x20] = defPolygon_t("#67eb5d",defaultFont,Qt::SolidPattern);
    /*"Forest",                  */ dictPolygon[0x50] = defPolygon_t("#67eb5d",defaultFont,Qt::SolidPattern);
    /*"Ocean",                   */ dictPolygon[0x28] = defPolygon_t(Qt::blue,defaultFont,Qt::SolidPattern);
    /*"Blue (unknown)",          */ dictPolygon[0x29] = defPolygon_t(Qt::blue,defaultFont,Qt::SolidPattern);
    /*"Sea",                     */ dictPolygon[0x32] = defPolygon_t(Qt::blue,defaultFont,Qt::SolidPattern);
    /*"Blue (unknown)",          */ dictPolygon[0x3b] = defPolygon_t(Qt::blue,defaultFont,Qt::SolidPattern);
    /*"Large lake",              */ dictPolygon[0x3c] = defPolygon_t(Qt::blue,defaultFont,Qt::SolidPattern);
    /*"Large lake",              */ dictPolygon[0x3d] = defPolygon_t(Qt::blue,defaultFont,Qt::SolidPattern);
    /*"Medium lake",             */ dictPolygon[0x3e] = defPolygon_t(Qt::blue,defaultFont,Qt::SolidPattern);
    /*"Medium lake",             */ dictPolygon[0x3f] = defPolygon_t(Qt::blue,defaultFont,Qt::SolidPattern);
    /*"Small lake",              */ dictPolygon[0x40] = defPolygon_t(Qt::blue,defaultFont,Qt::SolidPattern);
    /*"Small lake",              */ dictPolygon[0x41] = defPolygon_t(Qt::blue,defaultFont,Qt::SolidPattern);
    /*"Major lake",              */ dictPolygon[0x42] = defPolygon_t(Qt::blue,defaultFont,Qt::SolidPattern);
    /*"Major lake",              */ dictPolygon[0x43] = defPolygon_t(Qt::blue,defaultFont,Qt::SolidPattern);
    /*"Large lake",              */ dictPolygon[0x44] = defPolygon_t(Qt::blue,defaultFont,Qt::SolidPattern);
    /*"Blue (unknown)",          */ dictPolygon[0x45] = defPolygon_t(Qt::blue,defaultFont,Qt::SolidPattern);
    /*"Major River",             */ dictPolygon[0x46] = defPolygon_t(Qt::blue,defaultFont,Qt::SolidPattern);
    /*"Large River",             */ dictPolygon[0x47] = defPolygon_t(Qt::blue,defaultFont,Qt::SolidPattern);
    /*"Medium River",            */ dictPolygon[0x48] = defPolygon_t(Qt::blue,defaultFont,Qt::SolidPattern);
    /*"Small River",             */ dictPolygon[0x49] = defPolygon_t(Qt::blue,defaultFont,Qt::SolidPattern);
    /*"Definition area"          */ dictPolygon[0x4a] = defPolygon_t(Qt::transparent,defaultFont,Qt::SolidPattern,QPen(Qt::gray));
    /*"Background",              */ dictPolygon[0x4b] = defPolygon_t(Qt::transparent,defaultFont,Qt::SolidPattern,QPen(Qt::gray));
    /*"Intermittent water",      */ dictPolygon[0x4c] = defPolygon_t(Qt::cyan,defaultFont,Qt::BDiagPattern);
    /*"Wetland/Swamp",           */ dictPolygon[0x51] = defPolygon_t(Qt::cyan,defaultFont,Qt::BDiagPattern);
    /*"Glacier",                 */ dictPolygon[0x4d] = defPolygon_t(Qt::cyan,defaultFont,Qt::SolidPattern);
    /*"Orchard/Plantation",      */ dictPolygon[0x4e] = defPolygon_t(Qt::green,defaultFont,Qt::Dense5Pattern);
    /*"Scrub",                   */ dictPolygon[0x4f] = defPolygon_t(Qt::green,defaultFont,Qt::Dense5Pattern);
    /*"Tundra",                  */ dictPolygon[0x52] = defPolygon_t(Qt::lightGray,defaultFont,Qt::Dense5Pattern);
    /*"Flat",                    */ dictPolygon[0x53] = defPolygon_t(Qt::lightGray,defaultFont,Qt::Dense5Pattern);
    /*"???",         */             dictPolygon[0x54] = defPolygon_t(Qt::NoPen,defaultFont,Qt::NoBrush);

    if(skinfile.open(QIODevice::ReadOnly)) {
        QRegExp re("(.*)\\.(.*):\\s+(.*)");

        while(!skinfile.atEnd()) {
            QString line = skinfile.readLine();
            line = line.trimmed();
            if(re.exactMatch(line)) {
                QStringList tokens = re.capturedTexts();

                // all polyline tokens
                if(tokens[1] == "Airport_runway") {
                    setSkinValue(dictPolyline[0x27],tokens[2],tokens[3]);
                }
                else if(tokens[1] == "Alley") {
                    setSkinValue(dictPolyline[0x07],tokens[2],tokens[3]);
                }
                else if(tokens[1] == "Arterial") {
                    setSkinValue(dictPolyline[0x04],tokens[2],tokens[3]);
                }
                else if(tokens[1] == "Collector") {
                    setSkinValue(dictPolyline[0x05],tokens[2],tokens[3]);
                }
                else if(tokens[1] == "County_border") {
                    setSkinValue(dictPolyline[0x1D],tokens[2],tokens[3]);
                }
                else if(tokens[1] == "Ferry1a") {
                    setSkinValue(dictPolyline[0x1a],tokens[2],tokens[3]);
                }
                else if(tokens[1] == "Ferry1b") {
                    setSkinValue(dictPolyline[0x1b],tokens[2],tokens[3]);
                }
                else if(tokens[1] == "HWY_connector") {
                    setSkinValue(dictPolyline[0x0B],tokens[2],tokens[3]);
                }
                else if(tokens[1] == "Intermediate_depth_contour") {
                    setSkinValue(dictPolyline[0x24],tokens[2],tokens[3]);
                }
                else if(tokens[1] == "Intermediate_land_contour") {
                    setSkinValue(dictPolyline[0x21],tokens[2],tokens[3]);
                }
                else if(tokens[1] == "Intermittent_stream") {
                    setSkinValue(dictPolyline[0x26],tokens[2],tokens[3]);
                }
                else if(tokens[1] == "International_border") {
                    setSkinValue(dictPolyline[0x1e],tokens[2],tokens[3]);
                }
                else if(tokens[1] == "Major_depth_contour") {
                    setSkinValue(dictPolyline[0x25],tokens[2],tokens[3]);
                }
                else if(tokens[1] == "Major_land_contour") {
                    setSkinValue(dictPolyline[0x22],tokens[2],tokens[3]);
                }
                else if(tokens[1] == "Major_highway") {
                    setSkinValue(dictPolyline[0x01],tokens[2],tokens[3]);
                }
                else if(tokens[1] == "Minor_depth_contour") {
                    setSkinValue(dictPolyline[0x23],tokens[2],tokens[3]);
                }
                else if(tokens[1] == "Minor_land_contour") {
                    setSkinValue(dictPolyline[0x20],tokens[2],tokens[3]);
                }
                else if(tokens[1] == "Marine_boundary") {
                    setSkinValue(dictPolyline[0x15],tokens[2],tokens[3]);
                }
                else if(tokens[1] == "Other_highway") {
                    setSkinValue(dictPolyline[0x03],tokens[2],tokens[3]);
                }
                else if(tokens[1] == "Pipeline") {
                    setSkinValue(dictPolyline[0x28],tokens[2],tokens[3]);
                }
                else if(tokens[1] == "Powerline") {
                    setSkinValue(dictPolyline[0x29],tokens[2],tokens[3]);
                }
                else if(tokens[1] == "Principal_highway") {
                    setSkinValue(dictPolyline[0x02],tokens[2],tokens[3]);
                }
                else if(tokens[1] == "Railroad") {
                    setSkinValue(dictPolyline[0x14],tokens[2],tokens[3]);
                }
                else if(tokens[1] == "Ramp_fast") {
                    setSkinValue(dictPolyline[0x09],tokens[2],tokens[3]);
                }
                else if(tokens[1] == "Ramp_slow") {
                    setSkinValue(dictPolyline[0x08],tokens[2],tokens[3]);
                }
                else if(tokens[1] == "Residential") {
                    setSkinValue(dictPolyline[0x06],tokens[2],tokens[3]);
                }
                else if(tokens[1] == "River") {
                    setSkinValue(dictPolyline[0x1f],tokens[2],tokens[3]);
                }
                else if(tokens[1] == "Roundabout") {
                    setSkinValue(dictPolyline[0x0C],tokens[2],tokens[3]);
                }
                else if(tokens[1] == "Shoreline") {
                    setSkinValue(dictPolyline[0x15],tokens[2],tokens[3]);
                }
                else if(tokens[1] == "State_border") {
                    setSkinValue(dictPolyline[0x1c],tokens[2],tokens[3]);
                }
                else if(tokens[1] == "Stream") {
                    setSkinValue(dictPolyline[0x18],tokens[2],tokens[3]);
                }
                else if(tokens[1] == "Timezone") {
                    setSkinValue(dictPolyline[0x19],tokens[2],tokens[3]);
                }
                else if(tokens[1] == "Trail") {
                    setSkinValue(dictPolyline[0x16],tokens[2],tokens[3]);
                }
                else if(tokens[1] == "Trail1e") {
                    setSkinValue(dictPolyline[0x1e],tokens[2],tokens[3]);
                }
                else if(tokens[1] == "Trail0f") {
                    setSkinValue(dictPolyline[0x0f],tokens[2],tokens[3]);
                }
                else if(tokens[1] == "Unpaved") {
                    setSkinValue(dictPolyline[0x0a],tokens[2],tokens[3]);
                }
                // all polygon tokens
                else if(tokens[1] == "Airport") {
                    setSkinValue(dictPolygon[0x07],tokens[2],tokens[3]);
                }
                else if(tokens[1] == "Airport_runwayP") {
                    setSkinValue(dictPolygon[0x0e],tokens[2],tokens[3]);
                }
                else if(tokens[1] == "Background") {
                    setSkinValue(dictPolygon[0x4b],tokens[2],tokens[3]);
                }
                else if(tokens[1] == "Blue_Unknown41") {
                    setSkinValue(dictPolygon[0x41],tokens[2],tokens[3]);
                }
                else if(tokens[1] == "Blue_Unknown59") {
                    setSkinValue(dictPolygon[0x59],tokens[2],tokens[3]);
                }
                else if(tokens[1] == "Blue_Unknown69") {
                    setSkinValue(dictPolygon[0x69],tokens[2],tokens[3]);
                }
                else if(tokens[1] == "Cemetery") {
                    setSkinValue(dictPolygon[0x1a],tokens[2],tokens[3]);
                }
                else if(tokens[1] == "City1") {
                    setSkinValue(dictPolygon[0x01],tokens[2],tokens[3]);
                }
                else if(tokens[1] == "City2") {
                    setSkinValue(dictPolygon[0x02],tokens[2],tokens[3]);
                }
                else if(tokens[1] == "City3") {
                    setSkinValue(dictPolygon[0x03],tokens[2],tokens[3]);
                }
                else if(tokens[1] == "City_park") {
                    setSkinValue(dictPolygon[0x17],tokens[2],tokens[3]);
                }
                else if(tokens[1] == "Definition_area") {
                    setSkinValue(dictPolygon[0x4a],tokens[2],tokens[3]);
                }
                else if(tokens[1] == "Flat") {
                    setSkinValue(dictPolygon[0x53],tokens[2],tokens[3]);
                }
                else if(tokens[1] == "Forest") {
                    setSkinValue(dictPolygon[0x50],tokens[2],tokens[3]);
                }
                else if(tokens[1] == "Glacier") {
                    setSkinValue(dictPolygon[0x4d],tokens[2],tokens[3]);
                }
                else if(tokens[1] == "Golf_course") {
                    setSkinValue(dictPolygon[0x18],tokens[2],tokens[3]);
                }
                else if(tokens[1] == "Hospital") {
                    setSkinValue(dictPolygon[0x0b],tokens[2],tokens[3]);
                }
                else if(tokens[1] == "Industrial_complex") {
                    setSkinValue(dictPolygon[0x0c],tokens[2],tokens[3]);
                }
                else if(tokens[1] == "Intermittent_water") {
                    setSkinValue(dictPolygon[0x4c],tokens[2],tokens[3]);
                }
                else if(tokens[1] == "Lake3c") {
                    setSkinValue(dictPolygon[0x3c],tokens[2],tokens[3]);
                }
                else if(tokens[1] == "Lake3d") {
                    setSkinValue(dictPolygon[0x3d],tokens[2],tokens[3]);
                }
                else if(tokens[1] == "Lake3e") {
                    setSkinValue(dictPolygon[0x3e],tokens[2],tokens[3]);
                }
                else if(tokens[1] == "Lake3f") {
                    setSkinValue(dictPolygon[0x3f],tokens[2],tokens[3]);
                }
                else if(tokens[1] == "Lake40") {
                    setSkinValue(dictPolygon[0x40],tokens[2],tokens[3]);
                }
                else if(tokens[1] == "Lake42") {
                    setSkinValue(dictPolygon[0x42],tokens[2],tokens[3]);
                }
                else if(tokens[1] == "Lake43") {
                    setSkinValue(dictPolygon[0x43],tokens[2],tokens[3]);
                }
                else if(tokens[1] == "Lake44") {
                    setSkinValue(dictPolygon[0x44],tokens[2],tokens[3]);
                }
                else if(tokens[1] == "Man_made_area") {
                    setSkinValue(dictPolygon[0x13],tokens[2],tokens[3]);
                }
                else if(tokens[1] == "Marina") {
                    setSkinValue(dictPolygon[0x09],tokens[2],tokens[3]);
                }
                else if(tokens[1] == "Marine_hazard") {
                    //                     setSkinValue(dictPolygon[0x0a],tokens[2],tokens[3]);
                }
                else if(tokens[1] == "Military_base") {
                    setSkinValue(dictPolygon[0x04],tokens[2],tokens[3]);
                }
                else if(tokens[1] == "National_park14") {
                    setSkinValue(dictPolygon[0x14],tokens[2],tokens[3]);
                }
                else if(tokens[1] == "National_park15") {
                    setSkinValue(dictPolygon[0x15],tokens[2],tokens[3]);
                }
                else if(tokens[1] == "National_park16") {
                    setSkinValue(dictPolygon[0x16],tokens[2],tokens[3]);
                }
                else if(tokens[1] == "Ocean") {
                    setSkinValue(dictPolygon[0x28],tokens[2],tokens[3]);
                }
                else if(tokens[1] == "Orchard_Plantation") {
                    setSkinValue(dictPolygon[0x4e],tokens[2],tokens[3]);
                }
                else if(tokens[1] == "Parking_garage") {
                    setSkinValue(dictPolygon[0x06],tokens[2],tokens[3]);
                }
                else if(tokens[1] == "Parking_lot") {
                    setSkinValue(dictPolygon[0x05],tokens[2],tokens[3]);
                }
                else if(tokens[1] == "Reservation") {
                    setSkinValue(dictPolygon[0x0d],tokens[2],tokens[3]);
                }
                else if(tokens[1] == "River46") {
                    setSkinValue(dictPolygon[0x46],tokens[2],tokens[3]);
                }
                else if(tokens[1] == "River47") {
                    setSkinValue(dictPolygon[0x47],tokens[2],tokens[3]);
                }
                else if(tokens[1] == "River48") {
                    setSkinValue(dictPolygon[0x48],tokens[2],tokens[3]);
                }
                else if(tokens[1] == "River49") {
                    setSkinValue(dictPolygon[0x49],tokens[2],tokens[3]);
                }
                else if(tokens[1] == "Scrub") {
                    setSkinValue(dictPolygon[0x4f],tokens[2],tokens[3]);
                }
                else if(tokens[1] == "Sea") {
                    setSkinValue(dictPolygon[0x32],tokens[2],tokens[3]);
                }
                else if(tokens[1] == "Shopping_center") {
                    setSkinValue(dictPolygon[0x08],tokens[2],tokens[3]);
                }
                else if(tokens[1] == "Small_lake") {
                    setSkinValue(dictPolygon[0x40],tokens[2],tokens[3]);
                }
                else if(tokens[1] == "Sports_complex") {
                    setSkinValue(dictPolygon[0x19],tokens[2],tokens[3]);
                }
                else if(tokens[1] == "State_park1e") {
                    setSkinValue(dictPolygon[0x1e],tokens[2],tokens[3]);
                }
                else if(tokens[1] == "State_park1f") {
                    setSkinValue(dictPolygon[0x1f],tokens[2],tokens[3]);
                }
                else if(tokens[1] == "State_park20") {
                    setSkinValue(dictPolygon[0x20],tokens[2],tokens[3]);
                }
                else if(tokens[1] == "Tundra") {
                    setSkinValue(dictPolygon[0x52],tokens[2],tokens[3]);
                }
                else if(tokens[1] == "University_College") {
                    setSkinValue(dictPolygon[0x0a],tokens[2],tokens[3]);
                }
                else if(tokens[1] == "Wetland_Swamp") {
                    setSkinValue(dictPolygon[0x51],tokens[2],tokens[3]);
                }
                else if(tokens[1] == "Wetland_Swamp54") {
                    setSkinValue(dictPolygon[0x54],tokens[2],tokens[3]);
                }
                // the left overs
                else {
                    qDebug() << tokens;
                }
            }
        }
    }
}


void CGarminMap::slotMapChanged()
{
    calcWorldRect();
    needRedraw = true;
}


void CGarminMap::slotMapChanged(QRectF& area)
{
    if(!area.intersects(worldrect)) {
        world();
    }
    else {
        calcWorldRect();
        needRedraw = true;
    }
}


void CGarminMap::world()
{
    bitBias = 0;
    gpResources->mapdb().releaseVisibleTiles();

    QRectF r = gpResources->mapdb().getWorldRect();

    for(idxZoom = MIN_IDX_ZOOM; idxZoom <= MAX_IDX_ZOOM; ++idxZoom) {
        if(rect.height() * scales[idxZoom].scale < r.height()) {
            --idxZoom;
            break;
        }
    }

    QPoint p1 = QPoint((int)rect.center().x(),(int)rect.center().y());
    setScale(idxZoom,p1);

    QPoint p2 = QPoint(u2px(r.center().x()),v2px(r.center().y()));
    move(p2,p1);

}


void CGarminMap::zoom(QRect& r)
{
    if(bitBias < 0) bitBias = 0;

    r = r.normalized();

    move(r.center(), rect.center().toPoint());

    if((rect.width() > r.width()) && (rect.height() > r.height())) {
        // have to zoom in
        for(; idxZoom <= MAX_IDX_ZOOM; ++idxZoom) {
            if((rect.width() * scales[idxZoom].scale < r.width() * scale)
            || (rect.height() * scales[idxZoom].scale < r.height() * scale)) {
                --idxZoom;
                break;
            }
        }
    }
    else {
        // have to zoom out
        for(; idxZoom >= 0; --idxZoom) {
            if((rect.width() * scales[idxZoom].scale > r.width() * scale)
            && (rect.height() * scales[idxZoom].scale > r.height() * scale)) {
                break;
            }
        }
    }
    setScale(idxZoom,rect.center().toPoint());
}


void CGarminMap::incDetails()
{
    ++bitBias;
    if(bitBias > 4) bitBias = 4;
    calcWorldRect();
    needRedraw = true;
}


void CGarminMap::decDetails()
{
    --bitBias;
    if(bitBias < -4) bitBias = -4;
    calcWorldRect();
    needRedraw = true;
}


void CGarminMap::toggleCityNames()
{
    showCityNames = !showCityNames;
    calcWorldRect();
    needRedraw = true;
}


void CGarminMap::togglePOIs()
{
    showPOIs = !showPOIs;
    calcWorldRect();
    needRedraw = true;
}


void CGarminMap::toggleStreetNames()
{
    showStreetNames = !showStreetNames;
    calcWorldRect();
    needRedraw = true;
}


void CGarminMap::zoom(bool zoomIn, const QPoint& p)
{
    if(bitBias > 0 && !zoomIn) bitBias = 0;
    if(bitBias < 0 && zoomIn) bitBias = 0;

    idxZoom += zoomIn ? +1 : -1;
    setScale(idxZoom,p);
}


void CGarminMap::setScale(quint32 idx,const QPoint& p)
{
    XY pt1,pt2;

    pt1.u = p.x() * scale;
    pt1.v = p.y() * scale;

    idxZoom = idx;
    if(idxZoom < MIN_IDX_ZOOM) idxZoom = MIN_IDX_ZOOM;
    if(idxZoom > MAX_IDX_ZOOM) idxZoom = MAX_IDX_ZOOM;
    scale       = scales[idxZoom].scale;
    scaleText   = scales[idxZoom].label;

    pt2.u = p.x() * scale;
    pt2.v = p.y() * scale;

    x_off += pt1.u - pt2.u;
    y_off -= pt1.v - pt2.v;

    calcWorldRect();

    needRedraw = true;

    qDebug() << "maplevel" << mapLevelMap << "(" << mapLevelOvl << ")" << "bits" << scales[idxZoom].bits;
}


void CGarminMap::drawBackground(QPainter * p)
{
    QPainter paintBottom;

    if(p == 0) {
        paintBottom.begin(&pixBottom);
        paintBottom.fillRect(rect,QColor("#ffffcc"));
        p = &paintBottom;
    }

    // draw active map
    const QList<CGarminImg*>& mapFiles = gpResources->mapdb().getVisibleTiles(CGarminDBMap::eMap);
    QList<CGarminImg*>::const_iterator file = mapFiles.begin();
    while(file != mapFiles.end()) {

        const QMap<QString,subfile_desc_t>& subfiles = (*file)->getSubFiles();
        QMap<QString,subfile_desc_t>::const_iterator map = subfiles.begin();
        while(map != subfiles.end()) {
            drawSubMap(*map,*p,mapLevelMap);
            ++map;
        }

        ++file;
    }

    // draw overlay map
    const QList<CGarminImg*>& ovlFiles = gpResources->mapdb().getVisibleTiles(CGarminDBMap::eOvl);
    file = ovlFiles.begin();
    while(file != ovlFiles.end()) {

        const QMap<QString,subfile_desc_t>& subfiles = (*file)->getSubFiles();
        QMap<QString,subfile_desc_t>::const_iterator map = subfiles.begin();
        while(map != subfiles.end()) {
            drawSubMap(*map,*p,mapLevelOvl);
            ++map;
        }

        ++file;
    }

    if(p == &paintBottom) {
        paintBottom.end();
    }
}


void CGarminMap::drawSubMap(const subfile_desc_t& map,QPainter& p, quint8 level)
{

    p.setFont(gpResources->getMapFont());

    drawPolygons(map,p,level);
    drawPolylines(map,p,level);
    drawPoints(map,p,level);

#ifdef DEBUG_SHOW_SECTION_BORDERS
    QVector<subdiv_desc_t>::const_iterator subdiv;
    p.setPen(QPen(Qt::magenta,2));
    p.setBrush(Qt::NoBrush);
    subdiv = map.subdivs.begin();
    while(subdiv != map.subdivs.end()) {
        if(subdiv->level != level) {
            ++subdiv;
            continue;
        }

        QRect rect;

        rect.setLeft(u2px(subdiv->area.left()));
        rect.setRight(u2px(subdiv->area.right()));
        rect.setTop(v2px(subdiv->area.top()));
        rect.setBottom(v2px(subdiv->area.bottom()));

        p.drawRect(rect);

        ++subdiv;
    }
#endif                       // DEBUG_SHOW_SECTION_BORDERS
}


void CGarminMap::drawTextOnPolyline(const QPolygonF& poly, const QString& text, QPainter& p)
{
    if(easyDraw) return;

    int idx = poly.count() >> 1;

    QPointF delta = poly[idx] - poly[idx - 1];
    if(delta.isNull()) return;

    double deg = (360 / (2*PI)) * atan(delta.y()/delta.x());

    p.save();
    p.translate(poly[idx-1]);
    p.rotate(deg);

    p.setPen(Qt::darkBlue);
    p.drawText((int)(delta.x()/2 + 0.5),-5,text);

    p.restore();

}


void CGarminMap::drawPolylines(const subfile_desc_t& map,QPainter& p, quint8 level)
{
    QPolygonF poly;
    poly.reserve(1000);
    QVector<CGarminPolygon>::const_iterator line;
    QVector<XY>::const_iterator pt;
    QVector<subdiv_desc_t>::const_iterator subdiv;

    // draw polylines
    subdiv = map.subdivs.begin();
    while(subdiv != map.subdivs.end()) {
        if(subdiv->level != level) {
            ++subdiv;
            continue;
        }

        if(!subdiv->area.intersects(worldrect)) {
            ++subdiv;
            continue;
        }

        line = subdiv->polylines.begin();
        while(line != subdiv->polylines.end()) {
            poly.clear();
            pt = line->points.begin();
            while(pt != line->points.end()) {
                poly.append(QPointF(u2px(pt->u),v2px(pt->v)));
                ++pt;
            }

            drawPolyline((*line).type,poly,p,(*line).label1());
            ++line;
        }

        ++subdiv;
    }
}


// void CGarminMap::drawPolyline(quint16 type, const QPolygonF& poly, QPainter& p, const QString& label)
// {
//     QPen pen;
//
//     defPolyline_t& def = dictPolyline[type];
//
//     pen.setCapStyle(Qt::FlatCap);
//     pen.setJoinStyle(Qt::RoundJoin);
//
//
//     quint32 thickness = scale > 7.0 ? def.thickness : def.thickness + (7.0/scale);
//
//     pen.setColor(def.color0);
//     pen.setWidth(thickness);
//     pen.setStyle(def.style0);
//     p.setPen(pen);
//     p.drawPolyline(poly);
//
//
//
//     if(thickness > 2){
//         pen.setColor(def.color1);
//         pen.setWidth(thickness - 2);
//         pen.setStyle(def.style1);
//         p.setPen(pen);
//         p.drawPolyline(poly);
//     }
//
//     if((def.scale > -1.0) && (scale < def.scale) && showStreetNames){
//         drawTextOnPolyline(poly, label,p);
//     }
// }

void CGarminMap::drawPolyline(quint16 type, const QPolygonF& poly, QPainter& p, const QString& label)
{
    QPen pen;

    defPolyline_t& def = dictPolyline[type];

    pen.setCapStyle(Qt::FlatCap);

    quint32 thickness;
                                 // Minor and Intermediate land contour
    if (type == 32 || type == 33) {
        pen.setJoinStyle(Qt::MiterJoin);
        thickness = 1;
    }
    else if ( type == 34 ) {     // Major land contour
        pen.setJoinStyle(Qt::MiterJoin);
        thickness = scale > 10.0 ? 1 : 2;
    }
                                 // Stream
    else if ( type == 0x18 || type == 0x1f) {
        pen.setJoinStyle(Qt::MiterJoin);
        thickness = scale > 10.0 ? 2 : 3;
    }
    else {
        pen.setJoinStyle(Qt::RoundJoin);
                                 // quint32 muss da hin, sonst gibt es eine Warnung
        thickness = scale > 7.0 ? def.thickness : quint32(def.thickness + (7.0/scale));
        if ( scale > 100.0 ) {   // to improve speed, especially for OSM tiles
                                 // Major highway
            if ( type == 1 ) thickness = 3;
            else thickness = 1;
        }
    }

    pen.setColor(def.color0);
    pen.setWidth(thickness);
    pen.setStyle(def.style0);
    p.setPen(pen);
    p.drawPolyline(poly);

    if(thickness > 2) {
        pen.setColor(def.color1);
        pen.setWidth(thickness - 2);
        pen.setStyle(def.style1);
        p.setPen(pen);
        p.drawPolyline(poly);
    }

    if((def.scale > -1.0) && (scale < def.scale) && showStreetNames) {
        drawTextOnPolyline(poly, label,p);
    }
}


static bool areaLessThan(const quint16 x1, const quint16 x2)
{
    return CGarminImg::polygon_typestr[x1].order < CGarminImg::polygon_typestr[x2].order;
}


void CGarminMap::drawPolygons(const subfile_desc_t& map,QPainter& p, quint8 level)
{
    if(easyDraw || map.isTransparent) return;

    /*
        All polygons have to be drawn in the order of their increasing type ID.
        'All' means all polygons of the viewport as a sum. Not just those of a
        subdivision. That is why polygons are structured into a multimap.
        Multimaps from all visible subdevisions are combined into one large,
        temporary multimap. Next the polygons stored in this multimap are rendered
        by the increasing order of their keys.

        And more: Guess we have reached the bottom line, now. It's not really the increasing order
        of the keys. It's some arbitrary order. Well, not really arbitrary, more 'based on physical
        rules' like: Urban areas are on top of national parks. City parks are on top of urban areas.
        Man made areas are on top of city parks. And so on... Thus, after inventing data formats,
        IDs and stuff Garmin ignored all these helpfull things to make it different. Looks like
        they fired the smart guy and kicked in the stupid one.

        For us this means: Add a sort count to CGarminImg::polygon_typestr. Sort keys by this sort
        counter. Great!

        TODO: This should be subject to optimizations
    */

    QVector<subdiv_desc_t>::const_iterator subdiv;
    QVector<XY>::const_iterator pt;
    QMultiMap<quint16,CGarminPolygon> allVisiblePolygons;

    QPolygonF poly;
    poly.reserve(1000);

    // collect all polygons from visible sections
    subdiv = map.subdivs.begin();
    while(subdiv != map.subdivs.end()) {
        if(subdiv->level != level) {
            ++subdiv;
            continue;
        }

        if(!subdiv->area.intersects(worldrect)) {
            ++subdiv;
            continue;
        }
        allVisiblePolygons += subdiv->polygons;
        ++subdiv;
    }
    // draw polygons by key
    QList<quint16> keys = allVisiblePolygons.uniqueKeys();
    qSort(keys.begin(),keys.end(),areaLessThan);

    quint16 key;
    foreach(key,keys) {

        const QList<CGarminPolygon>& lines = allVisiblePolygons.values(key);
        QList<CGarminPolygon>::const_iterator line = lines.begin();
        while(line != lines.end()) {
            poly.clear();
            pt = line->points.begin();

            while(pt != line->points.end()) {
                poly.append(QPointF(u2px(pt->u),v2px(pt->v)));
                ++pt;
            }
            //drawPolygon(poly,*line,p);
            drawPolygon((*line).type,poly,p,(*line).label1());
            ++line;
        }
    }
}


void CGarminMap::drawPolygon(const quint16 type, const QPolygonF& poly, QPainter& p, const QString& label)
{
    if(!dictPolygon.contains(type)) {
        qWarning() << "Unsupported polygon type:" << type << label;
    }

    defPolygon_t& def = dictPolygon[type];
    p.setPen(def.pen0);
    p.setBrush(QBrush(def.color0,def.style0));
    p.drawPolygon(poly);

}


/*
void CGarminMap::drawPolygon(const QPolygonF& poly, const CGarminPolygon& line, QPainter& p)
{
    switch(line.type){
        case 0x00:
            break;

        case 0x01: //"Large urban area (>200K)",
        case 0x02: //"Small urban area (<200K)",
        case 0x03: //"Rural housing area",
            p.setPen(Qt::NoPen);
            p.setBrush(QBrush("#ff1c1c"));
            p.drawPolygon(poly);
            break;

        case 0x04: //"Military base",
            p.setPen(Qt::NoPen);
            p.setBrush(QBrush("#666600"));
            p.drawPolygon(poly);
            p.setBrush(QBrush("#006633",Qt::BDiagPattern));
            p.drawPolygon(poly);
            break;

        case 0x05: //"Parking lot",
        case 0x06: //"Parking garage",
            p.setPen(Qt::NoPen);
            p.setBrush(QBrush("#66ccff"));
            p.drawPolygon(poly);
            break;

        case 0x07: //"Airport",
            p.setPen(Qt::NoPen);
            p.setBrush(Qt::gray);
            p.drawPolygon(poly);
            break;

        case 0x08: //"Shopping center",
            p.setPen(Qt::NoPen);
            p.setBrush(QBrush("#ffff00"));
            p.drawPolygon(poly);
            break;

        case 0x09: //"Marina",
        case 0x0a: //"University/College",
        case 0x0b: //"Hospital",
        case 0x0c: //"Industrial complex",
        case 0x0d: //"Reservation",
            p.setPen(Qt::NoPen);
            p.setBrush(Qt::darkRed);
            p.drawPolygon(poly);
            break;

        case 0x0e: //"Airport runway",
            p.setPen(Qt::NoPen);
            p.setBrush(Qt::darkGray);
            p.drawPolygon(poly);
            break;

        case 0x13: //"Man-made area",
        case 0x19: //"Sports complex",
            p.setPen(Qt::NoPen);
            p.setBrush(Qt::darkRed);
            p.drawPolygon(poly);
            break;

        case 0x18: //"Golf course",
            p.setPen(Qt::NoPen);
            p.setBrush(Qt::green);
            p.drawPolygon(poly);
            break;

        case 0x1a: //"Cemetary",
            p.setPen(Qt::NoPen);
            p.setBrush(Qt::darkGreen);
            p.drawPolygon(poly);
            p.setBrush(QBrush(Qt::lightGray,Qt::Dense5Pattern));
            p.drawPolygon(poly);
            break;

        case 0x14: //"National park",
        case 0x15: //"National park",
        case 0x16: //"National park",
        case 0x17: //"City park",
        case 0x1e: //"State park",
        case 0x1f: //"State park",
        case 0x20: //"State park",
        case 0x50: //"Forest",
            p.setPen(Qt::NoPen);
            p.setBrush(Qt::white);
            p.drawPolygon(poly);
            p.setBrush(QBrush("#67eb5d"));
            //p.setBrush(QBrush(Qt::green, Qt::DiagCrossPattern));
            p.drawPolygon(poly);
            break;

        case 0x28: //"Ocean",
        case 0x29: //"Blue (unknown)",
        case 0x32: //"Sea",
        case 0x3b: //"Blue (unknown)",
        case 0x3c: //"Large lake",
        case 0x3d: //"Large lake",
        case 0x3e: //"Medium lake",
        case 0x3f: //"Medium lake",
        case 0x40: //"Small lake",
        case 0x41: //"Small lake",
        case 0x42: //"Major lake",
        case 0x43: //"Major lake",
        case 0x44: //"Large lake",
        case 0x45: //"Blue (unknown)",
        case 0x46: //"Major River",
        case 0x47: //"Large River",
        case 0x48: //"Medium River",
        case 0x49: //"Small River",
            p.setPen(Qt::NoPen);
            p.setBrush(Qt::blue);
            p.drawPolygon(poly);
            break;

        case 0x4a: //"Definition area"
            p.setPen(Qt::gray);
            p.setBrush(Qt::NoBrush);
            p.drawPolygon(poly);
            break;

        case 0x4b: //"Background",
            p.setPen(Qt::gray);
            p.setBrush(Qt::NoBrush);
            p.drawPolygon(poly);
            break;

        case 0x4c: //"Intermittent water",
        case 0x51: //"Wetland/Swamp",
            p.setPen(Qt::NoPen);
            p.setBrush(QBrush(Qt::cyan,Qt::BDiagPattern));
            p.drawPolygon(poly);
            break;

        case 0x4d: //"Glacier",
            p.setPen(Qt::NoPen);
            p.setBrush(Qt::cyan);
            p.drawPolygon(poly);
            break;

        case 0x4e: //"Orchard/Plantation",
        case 0x4f: //"Scrub",
            p.setPen(Qt::NoPen);
            p.setBrush(QBrush(Qt::green,Qt::Dense5Pattern));
            p.drawPolygon(poly);
            break;

        case 0x52: //"Tundra",
        case 0x53: //"Flat",
            p.setPen(Qt::NoPen);
            p.setBrush(QBrush(Qt::lightGray,Qt::Dense5Pattern));
            p.drawPolygon(poly);
            break;

        default:
            qDebug() << "Missing polygon" << hex << line.type;
            p.setPen(Qt::magenta);
            p.setBrush(Qt::magenta);
            p.drawPolygon(poly);
    }
}
*/
void CGarminMap::drawPoints(const subfile_desc_t& map,QPainter& p, quint8 level)
{
    if(easyDraw) return;

    QVector<subdiv_desc_t>::const_iterator subdiv;
    QVector<CGarminPoint>::const_iterator point;

    // draw points and POIs

    const XY * ptOld = 0;

    subdiv = map.subdivs.begin();
    while(subdiv != map.subdivs.end()) {
        if(subdiv->level != level) {
            ++subdiv;
            continue;
        }

        if(!subdiv->area.intersects(worldrect)) {
            ++subdiv;
            continue;
        }

        ptOld = 0;
        point = subdiv->points.begin();
        while(point != subdiv->points.end()) {
            if(ptOld && ptOld->u == point->point.u && ptOld->v == point->point.v) {
                ++point;
                continue;
            }

            drawPoint(QPointF(u2px(point->point.u),v2px(point->point.v)),*point,p);
            ptOld = &point->point;
            ++point;
        }

        ptOld = 0;
        point = subdiv->pois.begin();
        while(point != subdiv->pois.end()) {
            if(ptOld && ptOld->u == point->point.u && ptOld->v == point->point.v) {
                ++point;
                continue;
            }
            drawPoint(QPointF(u2px(point->point.u),v2px(point->point.v)),*point,p);

            ptOld = &point->point;

            ++point;
        }

        ++subdiv;
    }
}


void CGarminMap::drawPoint(const QPointF& pt, const CGarminPoint& point, QPainter& p)
{
    if(point.type < 0x1600 && showCityNames) {
        switch (point.type) {
            case 0x0600:         //"City (pop. 200-500K)",
            case 0x0700:         //"City (pop. 100-200K)",
            case 0x0800:         //"City (pop. 50-100K)",
            case 0x0900:         //"City (pop. 20-50K)",
            case 0x0a00:         //"Town (pop. 10-20K)",
            case 0x0b00:         //"Town (pop. 4-10K)",
            case 0x0c00:         //"Town (pop. 2-4K)",
            case 0x0d00:         //"Town (pop. 1-2K)",
            case 0x0e00:         //"Town (pop. 0.5-1K)",
            case 0x0f00:         //"Town (pop. 200-500)",
            case 0x1000:         //"Town (pop. 100-200)",
            case 0x1100:         //"Town (pop. under 100)",
            case 0x1300:         //"Town",
                if(scale > 100.0) break;
            case 0x0100:         //"City (pop. over 8M)",
            case 0x0200:         //"City (pop. 4-8M)",
            case 0x0300:         //"City (pop. 2-4M)",
            case 0x0400:         //"City (pop. 1-2M)",
            case 0x0500:         //"City (pop. 0.5-1M)",
                p.setPen(Qt::black);
                p.setBrush(Qt::black);
                p.drawEllipse((int)pt.x() - 2 ,(int)pt.y() - 2,5,5);
                p.setPen(Qt::yellow);
                p.setBrush(Qt::yellow);
                p.drawEllipse((int)pt.x() - 1 ,(int)pt.y() - 1,3,3);

                QRect bounding = p.boundingRect(QRect(),Qt::AlignCenter,point.label1());
                bounding.moveTo((int)pt.x(),(int)pt.y());
                bounding = bounding.adjusted(3,3,7,3);

                p.setPen(Qt::darkBlue);
                p.setBrush(QColor(255,255,0));
                p.drawRect(bounding);
                p.drawText(bounding,Qt::AlignCenter,point.label1());
                break;

        }
        return;
    }

    if(point.type < 0x2000) {
        switch (point.type) {
            case 0x1600:         //Navaid",
            case 0x1601:         //Forg horn",
            case 0x1602:         //Radio beacon",
            case 0x1603:         //Racon",
            case 0x1604:         //Daybeacon (red triangle)",
            case 0x1605:         //Daybeacon (green square)",
            case 0x1606:         //Unlit navaid (white diamond)",
            case 0x1607:         //Unlit navaid (white)",
            case 0x1608:         //Unlit navaid (red)",
            case 0x1609:         //Unlit navaid (green)",
            case 0x160a:         //Unlit navaid (black)",
            case 0x160b:         //Unlit navaid (yellow/amber)",
            case 0x160c:         //Unlit navaid (orange)",
            case 0x160d:         //Unlit navaid (multi-colored)",
            case 0x160e:         //Navaid",
            case 0x160f:         //Navaid (white)",
            case 0x1610:         //Navaid (red)",
            case 0x1611:         //Navaid (green)",
            case 0x1612:         //Navaid (yellow/amber)",
            case 0x1613:         //Navaid (orange)",
            case 0x1614:         //Navaid (violet)",
            case 0x1615:         //Navaid (blue)",
            case 0x1616:         //Navaid (multi-colored)",

            case 0x1c00:         //Obstruction",
            case 0x1c01:         //Wreck",
            case 0x1c02:         //Submerged wreck, dangerous",
            case 0x1c03:         //Submerged wreck, non-dangerous",
            case 0x1c04:         //Wreck, cleared by wire-drag",
            case 0x1c05:         //Obstruction, visible at high water",
            case 0x1c06:         //Obstruction, awash",
            case 0x1c07:         //Obstruction, submerged",
            case 0x1c08:         //Obstruction, cleared by wire-drag",
            case 0x1c09:         //Rock, awash",
            case 0x1c0a:         //Rock, submerged at low water",
            case 0x1c0b:         //Sounding",
            case 0x1d00:         //Tide",
            case 0x1d01:         //Tide prediction",
            case 0x1d02:         //Tide prediction",
                p.setPen(Qt::black);
                p.setBrush(Qt::black);
                p.drawEllipse((int)pt.x() - 2 ,(int)pt.y() - 2,5,5);
                p.setPen(Qt::blue);
                p.setBrush(Qt::blue);
                p.drawEllipse((int)pt.x() - 1 ,(int)pt.y() - 1,3,3);

                if(scale < 5.0) {
                    CCanvas::drawText(point.label1(),p,(pt + QPoint(0,-2)).toPoint());
                }

                break;
        }
        return;
    }
    // 0x1e00,	"Region, medium (no point)",
    // 0x1f00,	"Region, medium (no point)",

    if(point.type < 0x4000 && showPOIs) {
        switch(point.type) {

            case 0x2000:         //"Exit",
            case 0x2100:         //"Exit, with facilities",
            case 0x210f:         //"Exit, service",
            case 0x2200:         //"Exit, restroom",
            case 0x2300:         //"Exit, convenience store",
            case 0x2400:         //"Exit, weigh station",
            case 0x2500:         //"Exit, toll booth",
            case 0x2600:         //"Exit, information",
            case 0x2700:         //"Exit",
            case 0x2800:         //"Region, small (no point)",
            case 0x2900:         //"Region",
            case 0x2a00:         //"Food & Drink",
            case 0x2a01:         //"Food & Drink, American",
            case 0x2a02:         //"Food & Drink, Asian",
            case 0x2a03:         //"Food & Drink, Barbeque",
            case 0x2a04:         //"Food & Drink, Chinese",
            case 0x2a05:         //"Food & Drink, Deli/Bakery",
            case 0x2a06:         //"Food & Drink, International",
            case 0x2a07:         //"Food & Drink, Fast Food",
            case 0x2a08:         //"Food & Drink, Italian",
            case 0x2a09:         //"Food & Drink, Mexican",
            case 0x2a0a:         //"Food & Drink, Pizza",
            case 0x2a0b:         //"Food & Drink, Seafood",
            case 0x2a0c:         //"Food & Drink, Steak/Grill",
            case 0x2a0d:         //"Food & Drink, Bagel/Doughnut",
            case 0x2a0e:         //"Food & Drink, Cafe/Diner",
            case 0x2a0f:         //"Food & Drink, French",
            case 0x2a10:         //"Food & Drink, German",
            case 0x2a11:         //"Food & Drink, British Isles",
            case 0x2b00:         //"Lodging",
            case 0x2b01:         //"Lodging, Hotel/Motel",
            case 0x2b02:         //"Lodging, Bed & Breakfast/Inn",
            case 0x2b03:         //"Lodging, Campground/RV Park",
            case 0x2b04:         //"Lodging, Resort",
            case 0x2c00:         //"Attraction",
            case 0x2c01:         //"Recreation, Amusement/Theme Park",
            case 0x2c02:         //"Attraction, Museum/Historical",
            case 0x2c03:         //"Community, Library",
            case 0x2c04:         //"Attraction, Landmark",
            case 0x2c05:         //"Community, School",
            case 0x2c06:         //"Attraction, Park/Garden",
            case 0x2c07:         //"Attraction, Zoo/Aquarium",
            case 0x2c08:         //"Recreation, Arena/Track",
            case 0x2c09:         //"Attraction, Hall/Auditorium",
            case 0x2c0a:         //"Attraction, Winery",
            case 0x2c0b:         //"Community, Place of Worship",
            case 0x2c0c:         //"Attraction, Hot Spring",
            case 0x2d00:         //"Entertainment",
            case 0x2d01:         //"Entertainment, Live Theater",
            case 0x2d02:         //"Entertainment, Bar/Nightclub",
            case 0x2d03:         //"Entertainment, Cinema",
            case 0x2d04:         //"Entertainment, Casino",
            case 0x2d05:         //"Entertainment, Golf Course",
            case 0x2d06:         //"Recreation, Skiing Center/Resort",
            case 0x2d07:         //"Entertainment, Bowling",
            case 0x2d08:         //"Entertainment, Ice Skating",
            case 0x2d09:         //"Entertainment, Swimming Pool",
            case 0x2d0a:         //"Entertainment, Sports/Fitness Center",
            case 0x2d0b:         //"Entertainment, Sport Airport",
            case 0x2e00:         //"Shopping",
            case 0x2e01:         //"Shopping, Department Store",
            case 0x2e02:         //"Shopping, Grocery Store",
            case 0x2e03:         //"Shopping, General Merchandise",
            case 0x2e04:         //"Shopping Center",
            case 0x2e05:         //"Shopping, Pharmacy",
            case 0x2e06:         //"Shopping, Convenience Store",
            case 0x2e07:         //"Shopping, Apparel",
            case 0x2e08:         //"Shopping, Home and Garden",
            case 0x2e09:         //"Shopping, Home Furnishings",
            case 0x2e0a:         //"Shopping, Specialty Retail",
            case 0x2e0b:         //"Shopping, Computer/Software",
            case 0x2e0c:         //"Shopping, Other",
            case 0x2f00:         //"Service",
            case 0x2f01:         //"Service, Auto Fuel",
            case 0x2f02:         //"Service, Auto Rental",
            case 0x2f03:         //"Service, Auto Repair",
            case 0x2f04:         //"Service, Air Transportation",
            case 0x2f05:         //"Service, Post Office",
            case 0x2f06:         //"Service, Bank/ATM",
            case 0x2f07:         //"Service, Dealer/Auto Parts",
            case 0x2f08:         //"Service, Ground Transportation",
            case 0x2f09:         //"Service, Marina/Boat Repair",
            case 0x2f0a:         //"Service, Wrecker",
            case 0x2f0c:         //"Service, Rest Area/Information",
            case 0x2f0d:         //"Service, Auto Club",
            case 0x2f0e:         //"Service, Car Wash",
            case 0x2f0f:         //"Service, Garmin Dealer",
            case 0x2f10:         //"Service, Personal",
            case 0x2f11:         //"Service, Business",
            case 0x2f12:         //"Service, Communication",
            case 0x2f13:         //"Service, Repair",
            case 0x2f14:         //"Service, Social",
            case 0x2f15:         //"Service, Public Utility",
            case 0x3000:         //"Emergency/Government",
            case 0x3010:         //"Community, Police Station",
            case 0x3020:         //"Hospital",
            case 0x3030:         //"Community, City Hall",
            case 0x3040:         //"Community, Court House",
            case 0x3050:         //"Community, Community Center",
            case 0x3060:         //"Community, Border Crossing",
                p.setPen(Qt::black);
                p.setBrush(Qt::black);
                p.drawEllipse((int)pt.x() - 2 ,(int)pt.y() - 2,5,5);
                p.setPen(Qt::white);
                p.setBrush(Qt::white);
                p.drawEllipse((int)pt.x() - 1 ,(int)pt.y() - 1,3,3);

                if(scale < 2.0) {
                    CCanvas::drawText(point.label1(),p, (pt + QPoint(0,-2)).toPoint());
                }

                break;

            case 0x2f0b:         //"Service, Parking",
                p.setPen(Qt::black);
                p.setBrush(Qt::black);
                p.drawEllipse((int)pt.x() - 2 ,(int)pt.y() - 2,5,5);
                p.setPen(Qt::white);
                p.setBrush(Qt::white);
                p.drawEllipse((int)pt.x() - 1 ,(int)pt.y() - 1,3,3);

                if(scale < 2.0) {
                    p.drawPixmap(pt + QPoint(-7,-7),QPixmap(":/icons/wpt/parking15x15.png"));
                    CCanvas::drawText(point.label1(), p, (pt + QPoint(0,-2)).toPoint());
                }

        }
        return;
    }

    if(point.type < 0x5900) {
        switch(point.type) {
            case 0x4000:         //"Golf",
            case 0x4100:         //"Fishing",
            case 0x4200:         //"Wreck",
            case 0x4300:         //"Marina",
            case 0x4400:         //"Gas Station",
            case 0x4500:         //"Food & Drink",
            case 0x4600:         //"Bar",
            case 0x4700:         //"Boat Ramp",
            case 0x4800:         //"Camping",
            case 0x4900:         //"Park",
            case 0x4a00:         //"Picnic Area",
            case 0x4b00:         //"First Aid",
            case 0x4c00:         //"Information",
            case 0x4d00:         //"Parking",
            case 0x4e00:         //"Restroom",
            case 0x4f00:         //"Shower",
            case 0x5000:         //"Drinking Water",
            case 0x5100:         //"Telephone",
            case 0x5200:         //"Scenic Area",
            case 0x5300:         //"Skiing",
            case 0x5400:         //"Swimming",
            case 0x5500:         //"Dam",
            case 0x5700:         //"Danger",
            case 0x5800:         //"Restrcited Area",
                p.setPen(Qt::black);
                p.setBrush(Qt::black);
                p.drawEllipse((int)pt.x() - 2 ,(int)pt.y() - 2,5,5);
                p.setPen(Qt::green);
                p.setBrush(Qt::green);
                p.drawEllipse((int)pt.x() - 1 ,(int)pt.y() - 1,3,3);

                if(scale < 5.0) {
                    CCanvas::drawText(point.label1(), p, (pt + QPoint(0,-2)).toPoint());
                }

                break;
        }
        return;
    }

    if(point.type < 0x5D00) {
        switch(point.type) {
            case 0x5900:         //"Airport",
            case 0x5901:         //"Airport, Large",
            case 0x5902:         //"Airport, Medium",
            case 0x5903:         //"Airport, Small",
            case 0x5904:         //"Heliport",
            case 0x5905:         //"Airport",
                p.setPen(Qt::black);
                p.setBrush(Qt::black);
                p.drawEllipse((int)pt.x() - 2 ,(int)pt.y() - 2,5,5);
                p.setPen(Qt::cyan);
                p.setBrush(Qt::cyan);
                p.drawEllipse((int)pt.x() - 1 ,(int)pt.y() - 1,3,3);
                if(scale < 5.0) {
                    CCanvas::drawText(point.label1(), p, (pt + QPoint(0,-2)).toPoint());
                }
                break;
        }
        return;
    }
    // 0x5d00,	"Daymark, green square",
    // 0x5e00,	"Daymark, red triangle",
    //
    switch(point.type) {
        case 0x6100:             //",
        case 0x6200:             //Depth",
        case 0x6300:             //Elevation",
        case 0x6400:             //Man-made Feature",
        case 0x6401:             //Bridge",
        case 0x6402:             //Building",
        case 0x6403:             //Cemetary",
        case 0x6404:             //Church",
        case 0x6405:             //Civil Building",
        case 0x6406:             //Crossing",
        case 0x6407:             //Dam",
        case 0x6408:             //Hospital",
        case 0x6409:             //Levee",
        case 0x640a:             //Locale",
        case 0x640b:             //Military",
        case 0x640c:             //Mine",
        case 0x640d:             //Oil Field",
        case 0x640e:             //Park",
        case 0x640f:             //Post Office",
        case 0x6410:             //School",
        case 0x6411:             //Tower",
        case 0x6412:             //Trail",
        case 0x6413:             //Tunnel",
        case 0x6414:             //Well",
        case 0x6415:             //Ghost Town",
        case 0x6416:             //Subdivision",
            //
        case 0x6500:             //Water Feature",
        case 0x6501:             //Arroyo",
        case 0x6502:             //Sand Bar",
        case 0x6503:             //Bay",
        case 0x6504:             //Bend",
        case 0x6505:             //Canal",
        case 0x6506:             //Channel",
        case 0x6507:             //Cove",
        case 0x6508:             //Falls",
        case 0x6509:             //Geyser",
        case 0x650a:             //Glacier",
        case 0x650b:             //Harbor",
        case 0x650c:             //Island",
        case 0x650d:             //Lake",
        case 0x650e:             //Rapids",
        case 0x650f:             //Resevoir",
        case 0x6510:             //Sea",
        case 0x6511:             //Spring",
        case 0x6512:             //Stream",
        case 0x6513:             //Swamp",
            //
        case 0x6600:             //Land Feature",
        case 0x6601:             //Arch",
        case 0x6602:             //Area",
        case 0x6603:             //Basin",
        case 0x6604:             //Beach",
        case 0x6605:             //Bench",
        case 0x6606:             //Cape",
        case 0x6607:             //Cliff",
        case 0x6608:             //Crater",
        case 0x6609:             //Flat",
        case 0x660a:             //Forest",
        case 0x660b:             //Gap",
        case 0x660c:             //Gut",
        case 0x660d:             //Isthmus",
        case 0x660e:             //Lava",
        case 0x660f:             //Pillar",
        case 0x6610:             //Plain",
        case 0x6611:             //Range",
        case 0x6612:             //Reserve",
        case 0x6613:             //Ridge",
        case 0x6614:             //Rock",
        case 0x6615:             //Slope",
        case 0x6617:             //Valley",
        case 0x6618:             //Woods",
            p.setPen(Qt::black);
            p.setBrush(Qt::black);
            p.drawEllipse((int)pt.x() - 2 ,(int)pt.y() - 2,5,5);
            p.setPen(QColor("#846030"));
            p.setBrush(QColor("#846030"));
            p.drawEllipse((int)pt.x() - 1 ,(int)pt.y() - 1,3,3);
            if(scale < 5.0) {
                CCanvas::drawText(point.label1(), p, (pt + QPoint(0,-2)).toPoint());
            }
            break;

        case 0x6616:             //Summit",
            p.setPen(Qt::black);
            p.setBrush(Qt::black);
            p.drawEllipse((int)pt.x() - 2 ,(int)pt.y() - 2,5,5);
            p.setPen(QColor("#846030"));
            p.setBrush(QColor("#846030"));
            p.drawEllipse((int)pt.x() - 1 ,(int)pt.y() - 1,3,3);
            if(scale < 5.0) {
                p.drawPixmap(pt + QPoint(-7,-7),QPixmap(":/icons/wpt/summit15x15.png"));
                CCanvas::drawText(point.label1(), p, (pt + QPoint(0,-2)).toPoint());
            }
            break;

    }

}
