// $Id: JobManager.cpp,v 1.99 2014/05/05 16:05:17 hupereir Exp $

/******************************************************************************
*
* Copyright (C) 2002 Hugo PEREIRA <mailto: hugo.pereira@free.fr>
*
* This is free software; you can redistribute it and/or modify it under the
* terms of the GNU General Public License as published by the Free Software
* Foundation; either version 2 of the License, or (at your option) any later
* version.
*
* This software is distributed in the hope that it will be useful, but WITHOUT
* Any WARRANTY; without even the implied warranty of MERCHANTABILITY or
* FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
* for more details.
*
* You should have received a copy of the GNU General Public License along with
* this program.  If not, see <http://www.gnu.org/licenses/>.
*
*******************************************************************************/

#include "JobManager.h"
#include "JobManager.moc"

#include "AnimatedTreeView.h"
#include "Command.h"
#include "Debug.h"
#include "File.h"
#include "Job.h"
#include "InformationDialog.h"
#include "Singleton.h"
#include "ToolTipWidget.h"
#include "UserSet.h"
#include "Util.h"
#include "XcbInterface.h"
#include "XmlOptions.h"

#include <QApplication>
#include <QHeaderView>
#include <QTextStream>

//_______________________________________________
JobManager::JobManager( QWidget *parent ):
    QWidget( parent ),
    showIcons_( false ),
    showFullNames_( false ),
    treeViewEnabled_( false ),
    user_( Util::user() )
{
    Debug::Throw("JobManager::JobManager.\n");

    // layout
    setLayout( new QVBoxLayout( this ) );
    layout()->setSpacing(0);
    layout()->setMargin(0);

    // tooltip widget
    toolTipWidget_ = new ToolTipWidget( this );

    // list
    list_ = new AnimatedTreeView( this );
    list_->setMouseTracking( true );
    list_->setModel( &model_ );
    list_->setSortingEnabled( true );
    list_->setRootIsDecorated( false );
    list_->setSelectionMode( TreeView::ContiguousSelection );
    list_->setOptionName( "JOBLIST" );

    // column visibility
    list_->lockColumnVisibility( JobModel::Empty );
    list_->lockColumnVisibility( JobModel::User );
    list_->setColumnHidden( JobModel::Empty, false );
    list_->setColumnHidden( JobModel::User, true );

    layout()->addWidget( list_ );

    connect( list_, SIGNAL(expanded(QModelIndex)), list_, SLOT(resizeColumns()) );
    connect( list_, SIGNAL(collapsed(QModelIndex)), list_, SLOT(resizeColumns()) );
    connect( list_, SIGNAL(hovered(QModelIndex)), SLOT(_showToolTip(QModelIndex)) );

    // configuration
    connect( Singleton::get().application(), SIGNAL(configurationChanged()), SLOT(_updateConfiguration()) );
    _updateConfiguration();

}

//_______________________________________________
JobManager::~JobManager( void )
{ Debug::Throw( "JobManager::~JobManager.\n"); }

//_______________________________________
bool JobManager::setUser( QString user )
{

    Debug::Throw() << "JobManager::setUser - " << user << endl;
    if( user.isEmpty() || user == user_ ) return false;
    user_ = user;
    model_.clear();
    list_->setColumnHidden( JobModel::User, !UserSet::isAllUsers( user ) );
    return true;

}

//____________________________________________________________
void JobManager::processJobList( Job::Set jobs )
{
    Debug::Throw( "JobManager::processJobList.\n" );

    QString localUser( user_.left(8) );

    // keep track of old jobs
    Job::List oldJobs;
    for( Job::List::iterator jobIter = jobs_.begin(); jobIter != jobs_.end(); ++jobIter )
    {

        // find matching job in set
        Job::Set::iterator newJobIter = jobs.find( Job( jobIter->id(), jobIter->parentId() ) );
        if( newJobIter == jobs.end() ) {

            if( jobIter->id() < 0 || jobIter->user() != localUser ) oldJobs.append( *jobIter );
            else jobIter->setId( -jobIter->id() );

        } else {

            jobIter->updateFrom( *newJobIter );
            jobs.erase( newJobIter );

        }

    }

    // remove jobs that have dissapear
    foreach( const Job& job, oldJobs )
    { jobs_.removeOne( job ); }

    // add new jobs
    foreach( const Job& job, jobs )
    {
        if( job.isValid() && ( user_.isEmpty() || UserSet::isAllUsers( user_ ) || localUser == job.user() ) )
        {
            jobs_.append( job );
            jobs_.back().readStartTime();
        }
    }

    // find pixmaps
    if( showIcons_ )
    {
        XcbInterface().findIcons( jobs_ );
        for( Job::List::iterator iter = jobs_.begin(); iter != jobs_.end(); ++iter )
        { iter->setIconInitialized( true ); }
    }

    // find full command names
    if( showFullNames_ )
    {
        for( Job::List::iterator iter = jobs_.begin(); iter != jobs_.end(); ++iter )
        { iter->readFromCommandLine(); }
    }

    // use parent id for tree view if enabled
    if( treeViewEnabled() )
    {
        for( Job::List::iterator iter = jobs_.begin(); iter != jobs_.end(); ++iter )
        {
            const bool parentFound( std::find_if( jobs_.begin(), jobs_.end(), Job::SameIdFTor( iter->parentId() ) ) != jobs_.end() );
            iter->setTreeParentId( parentFound ? iter->parentId():0 );
        }
    }

    // update model
    _updateModel();

    // update tooltip if visible
    if( toolTipWidget_->isVisible() )
    {
        const QModelIndex index( list_->indexAt( list_->viewport()->mapFromGlobal( QCursor::pos() ) ) );
        if( index.isValid() ) toolTipWidget_->setJob( model_.get( index ) );
        else toolTipWidget_->hide();
    }

    return;

}

//_______________________________________________
bool JobManager::toggleTreeView( bool state )
{

    Debug::Throw( "JobManager::toggleTreeView.\n" );
    if( treeViewEnabled_  == state ) return false;
    treeViewEnabled_ = state;
    list_->setRootIsDecorated( state );

    // explicitely reset job's parents when turning off
    if( !state )
    {
        for( Job::List::iterator iter = jobs_.begin(); iter != jobs_.end(); ++iter )
        { iter->setTreeParentId( 0 ); }
    }

    return true;

}

//________________________________________
Job::List JobManager::selectedJobs( void ) const
{ return model_.get( list_->selectionModel()->selectedRows() ); }

//______________________________________________________
void JobManager::_showToolTip( const QModelIndex& index )
{

    Debug::Throw() << "JobManager::_showToolTip." << endl;

    if( !index.isValid() ) toolTipWidget_->hide();
    else {

        // fileInfo
        const Job job( model_.get( index ) );

        // rect
        QRect rect( list_->visualRect( index ) );
        rect.translate( list_->viewport()->mapToGlobal( QPoint( 0, 0 ) ) );
        toolTipWidget_->setIndexRect( rect );

        // assign to tooltip widget
        toolTipWidget_->setJob( job );

        // show
        toolTipWidget_->showDelayed();

    }

}

//_______________________________________
void JobManager::_updateModel( void )
{

    // store selected jobs in model
    list_->setUpdatesEnabled( false );
    list_->storeScrollBarPosition();
    list_->storeSelectedIndexes();

    // assign to list  and restart timer
    model_.setToday( TimeStamp::now() );
    model_.set( jobs_ );

    // restore selected jobs
    list_->resizeColumns();
    list_->restoreSelectedIndexes();
    list_->restoreScrollBarPosition();
    list_->setUpdatesEnabled( true );
}

//_______________________________________
void JobManager::_updateConfiguration( void )
{

    Debug::Throw( "JobManager::_updateConfiguration.\n" );
    showIcons_ = XmlOptions::get().get<bool>( "SHOW_ICONS" );
    showFullNames_ = XmlOptions::get().get<bool>( "SHOW_FULL_NAMES" );

}
