/*
 Copyright (C) 2001, 2002 Sadruddin Rejeb

 This file is part of QuantLib, a free-software/open-source library
 for financial quantitative analysts and developers - http://quantlib.org/

 QuantLib is free software: you can redistribute it and/or modify it under the
 terms of the QuantLib license.  You should have received a copy of the
 license along with this program; if not, please email ferdinando@ametrano.net
 The license is also available online at http://quantlib.org/html/license.html

 This program is distributed in the hope that it will be useful, but WITHOUT
 ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 FOR A PARTICULAR PURPOSE.  See the license for more details.
*/
/*! \file model.cpp
    \brief Abstract interest rate model class

    \fullpath
    ql/InterestRateModelling/%model.cpp
*/

// $Id: model.cpp,v 1.18 2002/03/06 07:16:06 sadrejeb Exp $


#include "ql/InterestRateModelling/calibrationhelper.hpp"
#include "ql/Optimization/optimizer.hpp"

namespace QuantLib {

    namespace InterestRateModelling {

        using namespace Optimization;

        class Model::CalibrationFunction : public CostFunction {
          public:
            CalibrationFunction( 
              Model* model,
              CalibrationSet& instruments) 
            : model_(model, false),
              instruments_(instruments), prices_(instruments.size()) {
                for (Size i=0; i<prices_.size(); i++) {
                    instruments_[i]->setModel(model_);
                    prices_[i] = instruments_[i]->marketValue();
                }
            }
            virtual ~CalibrationFunction() {}
            
            virtual double value(
                const Array& params) {
                model_->setParams(params);
                std::cout << "parameters set to " << params << std::endl;

                double value = 0.0;
                for (Size i=0; i<prices_.size(); i++) {
                    double diff = instruments_[i]->calibrationError();
                    value += diff*diff;
                }

                std::cout << "Cost function: " << QL_SQRT(value) << std::endl;
                return QL_SQRT(value);
            }
            virtual double finiteDifferenceEpsilon() { return 1e-6; }
          private:
            Handle<Model> model_;
            CalibrationSet& instruments_;
            Array prices_;
        };

        void Model::calibrate(
            CalibrationSet& instruments,
            const Handle<OptimizationMethod>& method) {

            CalibrationFunction f(this, instruments);

            method->setInitialValue(params());
            method->endCriteria().setPositiveOptimization();
            OptimizationProblem prob(f, *constraint_, *method);
            prob.minimize();

            Array result(prob.minimumValue());
            setParams(result);

            std::cout << "Cost function: " << f.value(result) << std::endl;
            std::cout << "Model calibrated to these parameters:" << std::endl;
            for (Size i=0; i<result.size(); i++)
                std::cout << i << "   " << result[i]*100.0 << "%" << std::endl;
        }

    }

}
