;*=====================================================================*/
;*    serrano/prgm/project/scribe/scribeinfo/engine.scm                */
;*    -------------------------------------------------------------    */
;*    Author      :  Manuel Serrano                                    */
;*    Creation    :  Sun Oct 14 11:35:59 2001                          */
;*    Last change :  Fri Nov 23 10:48:46 2001 (serrano)                */
;*    Copyright   :  2001 Manuel Serrano                               */
;*    -------------------------------------------------------------    */
;*    The construction of the form that is a valid Scribe tree.        */
;*=====================================================================*/

;*---------------------------------------------------------------------*/
;*    The module                                                       */
;*---------------------------------------------------------------------*/
(module __scribeinfo_engine
   (import __scribeinfo_param)
   (export (bind-variable ::bstring ::obj)
	   (variable-bound?::bool ::bstring)
	   (syncodeindex ::symbol ::symbol)
	   (synindex ::symbol ::symbol)
	   (make-index ::symbol)
	   (index ::symbol ::obj)
	   (texi->scribe ::obj)))

;*---------------------------------------------------------------------*/
;*    Indexes                                                          */
;*---------------------------------------------------------------------*/
(define *indexes* (list (list 'cp)
			(list 'fn)
			(list 'vr)
			(list 'ky)
			(list 'pg)
			(list 'tp)))

(define *syncodeindex* '())
(define *synindex* '())

;*---------------------------------------------------------------------*/
;*    syncodeindex ...                                                 */
;*---------------------------------------------------------------------*/
(define (syncodeindex l1 l2)
   (set! *syncodeindex* (cons (cons l1 l2) *syncodeindex*)))

;*---------------------------------------------------------------------*/
;*    synindex ...                                                     */
;*---------------------------------------------------------------------*/
(define (synindex l1 l2)
   (set! *synindex* (cons (cons l1 l2) *synindex*)))

;*---------------------------------------------------------------------*/
;*    index ...                                                        */
;*---------------------------------------------------------------------*/
(define (index kind val)
   (let ((lbl (symbol->string (gensym))))
      (define (inner-index kind val)
	 (let ((i (assq kind *indexes*)))
	    (if (not (pair? i))
		(error "index" "Can't find index" i)
		(set-cdr! i (cons (cons val lbl) (cdr i))))
	    lbl))
      (let ((cell (assq kind *syncodeindex*)))
	 (if (pair? cell)
	     (inner-index (cdr cell) (list (list 'code (car val))))
	     (let ((cell (assq kind *synindex*)))
		(if (pair? cell)
		    (inner-index (cdr cell) val)))))
      (inner-index kind val)))

;*---------------------------------------------------------------------*/
;*    make-index ...                                                   */
;*---------------------------------------------------------------------*/
(define (make-index kind)
   ;; transform a index value into a string
   (define (val->string v)
      (let loop ((s (car v)))
	 (if (string? s)
	     s
	     (match-case s
		((?- ?v . ?rest)
		 (loop v))
		((?lst)
		 (loop lst))
		(else
		 "")))))
   ;; return the first letter of an index value
   (define (ref-first-letter ref)
      (let ((str (val->string ref)))
	 (if (=fx (string-length str) 0)
	     #\_
	     (string-ref str 0))))
   ;; process one index row
   (define (do-letter-row-index refs letter width)
      (let loop ((num 0)
		 (refs refs)
		 (new-letter letter)
		 (cells '()))
	 (cond
	    ((=fx num *index-columns*)
	     (values refs new-letter `(tr ,@(reverse! cells))))
	    ((or (null? refs) (not (char=? new-letter letter)))
	     (loop (+fx num 1)
		   refs
		   new-letter
		   (cons `(td :width ,width "") cells)))
	    (else
	     (let ((new-letter (ref-first-letter (car refs))))
		(if (not (char=? new-letter letter))
		    (loop num refs new-letter cells)
		    (loop (+fx num 1)
			  (cdr refs)
			  new-letter
			  (cons `(td (ref :id ,(cdar refs) ,@(caar refs)))
				cells))))))))
   ;; process one index entry (one index letter)
   (define (do-letter-index refs letter rows width)
      (let* ((str (string #\- #\- #\space letter #\space #\- #\-))
	     (th `(th :colspan ,*index-columns*
		      :align 'center
		      (bold (font :size "+1" ,str)))))
	 (let loop ((refs refs)
		    (new-letter letter)
		    (rows (append (list `(tr (th :colspan ,*index-columns* ""))
					`(tr :bg *scribe-tbackground* ,th)
					`(tr (th :colspan ,*index-columns* "")))
				  rows)))
	    (cond
	       ((null? refs)
		(values '() letter rows))
	       ((not (char=? new-letter letter))
		(values refs new-letter rows))
	       (else
		(multiple-value-bind (new-refs new-letter row)
		   (do-letter-row-index refs new-letter width)
		   (loop new-refs new-letter (cons row rows))))))))
   ;; process the entire index		       
   (define (do-index refs)
      (if (null? refs)
	  '()
	  (let* ((len (length refs))
		 (height (/fx len *index-columns*))
		 (width (/ 1 *index-columns*)))
	     (let loop ((refs refs)
			(letter (ref-first-letter (car refs)))
			(trows '()))
		(if (null? refs)
		    (list `(table :width 1.
				  :cellpadding 2
				  ,@(reverse! trows)))
		    (multiple-value-bind (new-refs new-letter new-trows)
		       (do-letter-index refs letter trows width)
		       (loop new-refs new-letter new-trows)))))))
   ;; sort the index entries
   (let ((i (assq kind *indexes*)))
      (if (not (pair? i))
	  (error "index" "Can't find index" i)
	  (try (do-index (sort (cdr i)
			       (lambda (v1 v2)
				  (string<? (val->string v1)
					    (val->string v2)))))
	       (lambda (a b c e)
		  (notify-error b c e))))))
   
;*---------------------------------------------------------------------*/
;*    *variables* ...                                                  */
;*---------------------------------------------------------------------*/
(define *variables* (make-hashtable))

;*---------------------------------------------------------------------*/
;*    bind-variable ...                                                */
;*---------------------------------------------------------------------*/
(define (bind-variable var val)
   (begin
      (hashtable-put! *variables* var val)
      ""))

;*---------------------------------------------------------------------*/
;*    variable-bound? ...                                              */
;*---------------------------------------------------------------------*/
(define (variable-bound? var)
   (hashtable-get *variables* var))

;*---------------------------------------------------------------------*/
;*    texi-eval ...                                                    */
;*---------------------------------------------------------------------*/
(define (texi-eval sexp)
   (define (inner sexp)
      (if (not (pair? sexp))
	  sexp
	  (match-case sexp
	     ((value ?var)
	      (let ((val (hashtable-get *variables* var)))
		 (if (not val)
		     (error "texi-eval" "Can't find variable definition" var)
		     (inner `(list ,@val)))))
	     (else
	      (if (null? (car sexp))
		  (inner (cdr sexp))
		  (cons (inner (car sexp)) (inner (cdr sexp))))))))
   (if (null? sexp)
       ""
       (let ((res (inner sexp)))
	  (if (and (pair? res) (string? (car res)) (null? (cdr res)))
	      (car res)
	      res))))

;*---------------------------------------------------------------------*/
;*    texi->scribe ...                                                 */
;*    -------------------------------------------------------------    */
;*    This function accepts a flow of sexps that it constructs         */
;*    a correclty formed Scribe expression representing the whole      */
;*    document.                                                        */
;*    -------------------------------------------------------------    */
;*    This function implements a kind of automata that keeps track     */
;*    of what it is currently doing. For instance, it has to           */
;*    remember what is the current chapter, section, ... in order      */
;*    to accumulate the texts and give them to the proper owner.       */
;*    That is, we have to turn expressions such as:                    */
;*       (chapter "chap")                                              */
;*       (section "sec")                                               */
;*       "Foo bar"                                                     */
;*    into:                                                            */
;*       (chapter "chap"                                               */
;*        (section "sec"                                               */
;*         "Foo bar"))	                                               */
;*---------------------------------------------------------------------*/
(define (texi->scribe sexp)
   ;; first we have to find the title
   (seek-title sexp))

;*---------------------------------------------------------------------*/
;*    seek-title ...                                                   */
;*    -------------------------------------------------------------    */
;*    This function evaluates but does not include in the generated    */
;*    Scribe file everything it finds before the title.                */
;*---------------------------------------------------------------------*/
(define (seek-title sexp)
   (if (pair? sexp)
       (match-case (car sexp)
	  ((title . ?sexps)
	   (let* ((title (title->scribe sexps))
		  (body (body->scribe (cdr sexp))))
	      (cond
		 ((not (list? title))
		  (error "scribeinfo" "Illegal title" title))
		 ((not (list? body))
		  (error "scribeinfo" "Illegal body" body))
		 (else
		  (append title body)))))
	  (else
	   (texi-eval (car sexp))
	   (seek-title (cdr sexp))))
       #f))

;*---------------------------------------------------------------------*/
;*    title->scribe ...                                                */
;*---------------------------------------------------------------------*/
(define (title->scribe sexp)
   (let liip ((sexp sexp)
	      (titles '())
	      (authors '())
	      (blurp '()))
      (if (pair? sexp)
	  (match-case (car sexp)
	     (()
	      (liip (cdr sexp)
		    titles
		    authors
		    blurp))
	     ((title . ?title)
	      (liip (cdr sexp)
		    (cons (texi-eval title) titles)
		    authors
		    blurp))
	     ((subtitle . ?stitle)
	      (liip (cdr sexp)
		    (append (reverse (map texi-eval stitle))
			    (cons '(linebreak) titles))
		    authors
		    blurp))
	     ((author . ?author)
	      (liip (cdr sexp)
		    titles
		    (cons (texi-eval author) authors)
		    blurp))
	     (else
	      (liip (cdr sexp)
		    titles
		    authors
		    (cons (texi-eval (car sexp)) blurp))))
	  `(document :title (list ,@(reverse! titles))
		     :author (list ,@(map (lambda (a)
					     (let ((a (if (string? a)
							  (list a)
							  a)))
						`(author :name (list ,@a))))
					  (reverse! authors)))
		     ,@(if (pair? blurp)
			   `((list ,@(reverse! blurp)))
			   '())))))

;*---------------------------------------------------------------------*/
;*    body->scribe ...                                                 */
;*    -------------------------------------------------------------    */
;*    Read all the remaining expressions and store them into one       */
;*    list that will be the body of the document.                      */
;*---------------------------------------------------------------------*/
(define (body->scribe sexp)
   (let laa ((sexp sexp)
	      (res '()))
      (if (null? sexp)
	  res
	  (multiple-value-bind (scribe rest)
	     (sexp->scribe sexp '())
	     (laa rest (append scribe res))))))

;*---------------------------------------------------------------------*/
;*    sexp->scribe ...                                                 */
;*---------------------------------------------------------------------*/
(define (sexp->scribe sexp stop)
   (define (nesting block)
      (memq block '(paragraph subsubsection subsection section chapter)))
   (let luup ((sexp sexp)
	      (res '()))
      (if (null? sexp)
	  (values (reverse! res) '())
	  (let ((exp (car sexp))
		(rest (cdr sexp)))
	     (cond
		((or (string? exp) (number? exp))
		 (luup rest (cons exp res)))
		((or (eq? exp #unspecified) (eq? exp '()))
		 (luup rest res))
		((pair? exp)
		 (match-case exp
		    ((list . ?exps)
		     (luup (append exps rest) res))
		    (((and ?block (or paragraph subsubsection subsection section chapter)) . ?sec)
		     (if (memq block stop)
			 (values (reverse! res) sexp)
			 (multiple-value-bind (body rest)
			    (sexp->scribe rest (append (nesting block) stop))
			    (luup rest (cons (append exp body) res)))))
		    (else
		     (luup rest (cons (texi-eval exp) res)))))
		(else
		 (error "sexp->scribe" "Illegal expression" exp)))))))



