/**
 * Copyright (C) 2009 Google Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

import org.apache.tools.ant.BuildException;
import org.apache.tools.ant.DirectoryScanner;
import org.apache.tools.ant.Task;
import org.apache.tools.ant.types.Commandline;
import org.apache.tools.ant.types.FileSet;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

/**
 * @author jessewilson@google.com (Jesse Wilson)
 */
public class MungeTask extends Task {

  private final byte[] buffer = new byte[1024];

  private final List<FileSet> fileSets = new ArrayList<FileSet>();
  private final List<Commandline.Argument> arguments = new ArrayList<Commandline.Argument>();
  private File todir;

  public void addFileset(FileSet fileSet) {
    fileSets.add(fileSet);
  }

  public void setTodir(File todir) {
    this.todir = todir;
  }

  public Commandline.Argument createArg() {
    Commandline.Argument args = new Commandline.Argument();
    arguments.add(args);
    return args;
  }

  public void execute() throws BuildException {
    List<String> argumentsList = new ArrayList<String>();
    for (Commandline.Argument argument : arguments) {
      argumentsList.addAll(Arrays.asList(argument.getParts()));
    }

    for (FileSet fileSet : fileSets) {
      DirectoryScanner directoryScanner = fileSet.getDirectoryScanner(getProject());
      for (String filename : directoryScanner.getIncludedFiles()) {
        String inFileName = directoryScanner.getBasedir() + "/" + filename;
        String outFileName = todir.getPath() + "/" + filename;

        File inFile = new File(inFileName);
        File outFile = new File(outFileName);

        try {
          if (!inFile.exists()) {
            throw new IllegalArgumentException("No such file " + inFile);
          }

          File parentFile = outFile.getParentFile();
          parentFile.mkdirs();
          if (!parentFile.isDirectory()) {
            throw new IllegalArgumentException("Couldn't make directory for " + outFile);
          }

          if (outFile.lastModified() > inFile.lastModified()) {
            continue; // skip this file
          }

          if (outFileName.endsWith(".java")) {
            System.out.println(" munging " + inFileName);
            List<String> argsThisCall = new ArrayList<String>(argumentsList);
            argsThisCall.add(inFileName);
            argsThisCall.add(outFileName);
            Munge.main(argsThisCall.toArray(new String[argsThisCall.size()]));

          } else {
            System.out.println(" copying " + inFileName);
            copy(inFile, outFile);
          }
        } catch (IOException e) {
          throw new BuildException("Failed to process " + inFile, e);
        }
      }
    }
  }

  private void copy(File source, File target) throws IOException {
    FileInputStream in = new FileInputStream(source);
    FileOutputStream out = new FileOutputStream(target);
    int bytesRead = 0;
    while ((bytesRead = in.read(buffer)) != -1) {
      out.write(buffer, 0, bytesRead);
    }
    out.close();
  }
}
