-------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
-------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--=============================================================================

with Debug;
with Unchecked_Conversion;

separate (Dictionary)
package body RawDict is
   --# hide RawDict;

   type Booleans is array (Dictionary.Abstractions) of Boolean;
   -- pragma pack (Booleans)

   type Symbols is array (Dictionary.Abstractions) of Dictionary.Symbol;
   -- pragma pack (Symbols)

   type Constraints is array (Dictionary.Abstractions) of ExaminerConstants.RefType;
   -- pragma pack (Constraints)

   type TypeInfo is record
      Name             : LexTokenManager.Lex_String;
      Discriminant     : TypeDiscriminant;
      IsPrivate        : TriState;
      IsLimited        : TriState;
      IsLimitedPrivate : Boolean;
      IsDerived        : Boolean;
      IsTagged         : Boolean;
      IsAbstract       : Boolean;
      IsOwnVarType     : Boolean; -- type can be used for abstract view of own variable
      IsAtomic         : Boolean;
      EqualityDefined  : Boolean;
      ContainsFloat    : Boolean;
      Constrained      : Boolean;
      Static           : Boolean;
      Wellformed       : Boolean;
      Parent           : Dictionary.Symbol;
      BaseType         : Dictionary.Symbol;
      Extends          : Dictionary.Symbol; -- root type for tagged type extension
      Accesses         : Dictionary.Symbol;
      Lower            : LexTokenManager.Lex_String;
      Upper            : LexTokenManager.Lex_String;
      Modulus          : LexTokenManager.Lex_String;
      ErrorBound       : LexTokenManager.Lex_String;
      --  linked list of indexes for an array type, or record
      --  components for a record type
      Head : Dictionary.Symbol;

      Tail                   : Dictionary.Symbol;
      ArrayComponent         : Dictionary.Symbol;
      Announcement           : Dictionary.Symbol;
      Declaration            : Dictionary.Symbol;
      PrivateTypeDeclaration : Dictionary.Symbol;
      TheVirtualElementList  : Symbol;
      AncilliaryFields       : Dictionary.Symbol; -- points to ProtectedInfo, TaskInfo or SubtypeInfo record
      SizeAttribute          : LexTokenManager.Lex_String;
   end record;
   -- pragma pack (TypeInfo)

   type TypeRef is access TypeInfo;

   type EnumerationLiteralInfo is record
      Name     : LexTokenManager.Lex_String;
      Position : LexTokenManager.Lex_String;
      TheType  : Dictionary.Symbol;
      Next     : Dictionary.Symbol;
   end record;
   -- pragma pack (EnumerationLiteralInfo)

   type EnumerationLiteralRef is access EnumerationLiteralInfo;

   type ArrayIndexInfo is record
      IndexType : Dictionary.Symbol;
      Next      : Dictionary.Symbol;
   end record;
   -- pragma pack (ArrayIndexInfo)

   type ArrayIndexRef is access ArrayIndexInfo;

   type RecordComponentInfo is record
      Name           : LexTokenManager.Lex_String;
      RecordType     : Dictionary.Symbol;
      ComponentType  : Dictionary.Symbol;
      InheritedField : Boolean; --tag
      Next           : Dictionary.Symbol;
   end record;
   -- pragma pack (RecordComponentInfo)

   type RecordComponentRef is access RecordComponentInfo;

   type SubcomponentInfo is record
      Object        : Dictionary.Symbol;
      Component     : Dictionary.Symbol;
      Next          : Dictionary.Symbol;
      Subcomponents : Dictionary.Symbol;
      MarkedValid   : Boolean;
   end record;
   -- pragma pack (SubcomponentInfo)

   type SubcomponentRef is access SubcomponentInfo;

   type RulePolicyInfo is record
      Scope : Dictionary.Scopes;
      Value : Dictionary.RulePolicies;
      Next  : Dictionary.Symbol;
   end record;

   type RulePolicyRef is access RulePolicyInfo;

   type ConstantInfo is record
      Name                        : LexTokenManager.Lex_String;
      Static                      : Boolean;
      TypeMark                    : Dictionary.Symbol;
      Value                       : LexTokenManager.Lex_String;
      ExpIsWellFormed             : Boolean;
      ExpNode                     : ExaminerConstants.RefType;
      Declaration                 : Dictionary.Symbol;
      DeferredConstantDeclaration : Dictionary.Symbol;
      AssociatedGenericParameter  : Dictionary.Symbol; -- points back to GenericParameterInfo if const is generic object
      FirstRulePolicy             : Dictionary.Symbol;
      LastRulePolicy              : Dictionary.Symbol;
   end record;
   -- pragma pack (ConstantInfo)

   type ConstantRef is access ConstantInfo;

   type VariableInfo is record
      Name             : LexTokenManager.Lex_String;
      TypeMark         : Dictionary.Symbol;
      AbstractTypeMark : Dictionary.Symbol;
      Initialized      : Boolean;
      HasAddressClause : Boolean;
      HasPragmaImport  : Boolean;
      IsAliased        : Boolean;
      MarkedValid      : Boolean;
      ExpNode          : ExaminerConstants.RefType;
      Declaration      : Dictionary.Symbol;
      GlobalReferences : Symbols;
      OwnVariable      : Dictionary.Symbol;
      OwnTask          : Dictionary.Symbol;
      VirtualElement   : Dictionary.Symbol;
      Constituent      : Dictionary.Symbol;
      Subcomponents    : Dictionary.Symbol;
   end record;
   -- pragma pack (VariableInfo)

   type VariableRef is access VariableInfo;

   type QuantifiedVariableInfo is record
      Name     : LexTokenManager.Lex_String;
      TypeMark : Dictionary.Symbol;
      Region   : Dictionary.Symbol;
   end record;
   -- pragma pack (QuantifiedVariableInfo)

   type QuantifiedVariableRef is access QuantifiedVariableInfo;

   type ImplicitReturnVariableInfo is record
      Name        : LexTokenManager.Lex_String;
      TheFunction : Dictionary.Symbol;
   end record;
   -- pragma pack (ImplicitReturnVariableInfo)

   type ImplicitReturnVariableRef is access ImplicitReturnVariableInfo;

   type OwnVariableInfo is record
      Variable                : Dictionary.Symbol;
      Owner                   : Dictionary.Symbol;
      Announced               : Boolean;
      Typed                   : Boolean;
      Initialized             : Boolean;
      Mode                    : Modes;
      IsProtected             : Boolean;
      IsInterruptStream       : Boolean;
      InterruptStreamMappings : Dictionary.Symbol;
      UnprotectedReference    : Dictionary.Symbol;
      SuspendsReference       : Dictionary.Symbol;
      ImplicitInStream        : Dictionary.Symbol; -- only set for un-moded, protected own variables
      Priority                : LexTokenManager.Lex_String; -- StorageRep of number
      Integrity               : LexTokenManager.Lex_String; -- StorageRep of number
      Suspendable             : Boolean;
      Interruptable           : Boolean;
      Constituents            : Dictionary.Symbol;
      Next                    : Dictionary.Symbol;
   end record;
   -- pragma pack (OwnVariableInfo)

   type OwnVariableRef is access OwnVariableInfo;

   type ImplicitInStreamInfo is record
      OwnVariable : Dictionary.Symbol; -- points back to the own variable with which this stream is associated
   end record;
   type ImplicitInStreamRef is access ImplicitInStreamInfo;

   type OwnTaskInfo is record
      Variable : Dictionary.Symbol;
      Owner    : Dictionary.Symbol;
      Next     : Symbol;
   end record;

   type OwnTaskRef is access OwnTaskInfo;

   type ConstituentInfo is record
      Subject  : Dictionary.Symbol;
      Variable : Dictionary.Symbol;
      Mode     : Modes;
      Next     : Dictionary.Symbol;
   end record;
   -- pragma pack (ConstituentInfo)

   type ConstituentRef is access ConstituentInfo;

   type PartInfo is record
      WithClauses          : Dictionary.Symbol;
      UseTypeClauses       : Dictionary.Symbol;
      RenamingDeclarations : Dictionary.Symbol;
      FirstDeclaration     : Dictionary.Symbol;
      LastDeclaration      : Dictionary.Symbol;
   end record;
   -- pragma pack (PartInfo)

   type ChildInfo is record
      FirstChild : Dictionary.Symbol;
      LastChild  : Dictionary.Symbol;
   end record;
   -- pragma pack (ChildInfo)

   type PackageInfo is record
      Name                 : LexTokenManager.Lex_String;
      Postcondition        : Constraints;
      PackageBody          : Dictionary.Symbol;
      PackageHasProperBody : Boolean;
      InheritClauses       : Dictionary.Symbol;
      OwnVariables         : Dictionary.Symbol;
      TaskList             : Dictionary.Symbol;
      FirstLoop            : Dictionary.Symbol;
      LastLoop             : Dictionary.Symbol;
      Specification        : Dictionary.Symbol;
      VisiblePart          : PartInfo;
      LocalPart            : PartInfo;
      PrivatePart          : PartInfo;
      ElaborateBodyFound   : Boolean;
      Parent               : Dictionary.Symbol;
      NextSibling          : Dictionary.Symbol;
      Extends              : Dictionary.Symbol; -- type extension/tagged types
      DeclaresTaggedType   : Boolean;
      PrivateChildren      : ChildInfo;
      PublicChildren       : ChildInfo;
      IsPrivate            : Boolean;
      --  1606
      --  points to generic package symbol if this package is an instantiation
      IsGeneric : Boolean;
      --  linked list of generic formals, only used if IsGeneric is true
      InstantiationOf       : Dictionary.Symbol;
      FirstGenericParameter : Dictionary.Symbol;
      LastGenericParameter  : Dictionary.Symbol;
      --  linked list of generic formal/actual pairings, only used if InstantiationOf is not null
      FirstGenericAssociation : Dictionary.Symbol;
      LastGenericAssociation  : Dictionary.Symbol;
   end record;
   -- pragma pack (PackageInfo)

   type PackageRef is access PackageInfo;

   type Pragmas is record
      Given : Boolean;                   -- true means has been declared
      Value : LexTokenManager.Lex_String; -- null string means no value, otherwise its a Maths StorageRep
   end record;

   type PragmaLists is array (Dictionary.RavenscarPragmas) of Pragmas;

   type ProtectedInfo is record
      OwnVariable       : Dictionary.Symbol; -- abstract "own var" with same name as type
      ElementsHidden    : Boolean;
      VisiblePart       : PartInfo;
      LocalPart         : PartInfo;
      PrivatePart       : PartInfo;
      TheEntry          : Symbol;
      FirstDiscriminant : Dictionary.Symbol;
      LastDiscriminant  : Dictionary.Symbol;
      Pragmas           : PragmaLists;
      ProtectedBody     : Symbol;
      HasProperBody     : Boolean;
   end record;

   type ProtectedRef is access ProtectedInfo;

   type TaskInfo is record
      SignatureIsWellformed    : Booleans;
      HasSecondAnnotation      : Boolean;
      HasDerivesAnnotation     : Boolean;
      LocalPart                : PartInfo;
      FirstDiscriminant        : Dictionary.Symbol;
      LastDiscriminant         : Dictionary.Symbol;
      Pragmas                  : PragmaLists;
      FirstGlobalVariable      : Symbols;
      LastGlobalVariable       : Symbols;
      TaskBody                 : Symbol;
      HasProperBody            : Boolean;
      BodyIsHidden             : Boolean;
      SuspendsList             : Symbol;
      FirstLoop                : Dictionary.Symbol;
      LastLoop                 : Dictionary.Symbol;
      UsesUnprotectedVariables : Boolean;
      UsesUncheckedConversion  : Boolean;
      AssignsFromExternal      : Boolean;
   end record;

   type TaskRef is access TaskInfo;

   type KnownDiscriminantInfo is record
      Name          : LexTokenManager.Lex_String;
      ProtectedType : Dictionary.Symbol;  -- could also be a task type
      TypeMark      : Dictionary.Symbol;
      SetsPriority  : Boolean;
      Next          : Dictionary.Symbol;
   end record;
   -- pragma pack (KnownDiscriminantInfo)

   type KnownDiscriminantRef is access KnownDiscriminantInfo;

   type SubtypeInfo is -- record used for extra info needed by subtypes of TASK and PROTECTED types
      record
         Priority        : LexTokenManager.Lex_String; -- StorageRep of number
         FirstConstraint : Dictionary.Symbol;         -- list of discriminant constraints
         LastConstraint  : Dictionary.Symbol;
      end record;

   type SubtypeRef is access SubtypeInfo;

   type DiscriminantConstraintInfo is record
      StaticValue    : LexTokenManager.Lex_String; -- only ONE of this and next field will have a value
      AccessedObject : Dictionary.Symbol;
      Next           : Dictionary.Symbol;
   end record;

   type DiscriminantConstraintRef is access DiscriminantConstraintInfo;

   type ContextClauseInfo is record
      Scope      : Dictionary.Scopes;
      ThePackage : Dictionary.Symbol;
      Explicit   : Boolean;  -- To allow for implicit withing of parents when child with'd
      Next       : Dictionary.Symbol;
   end record;
   -- pragma pack (ContextClauseInfo)

   type ContextClauseRef is access ContextClauseInfo;

   type UseTypeClauseInfo is record
      Scope   : Dictionary.Scopes;
      TheType : Dictionary.Symbol;
      Next    : Dictionary.Symbol;
   end record;
   -- pragma pack (UseTypeClauseInfo)

   type UseTypeClauseRef is access UseTypeClauseInfo;

   type SubprogramInfo is record
      Name                          : LexTokenManager.Lex_String;
      SignatureIsWellformed         : Booleans;
      ImplicitProofFunctions        : Symbols;
      ImplicitReturnVariables       : Symbols;
      Precondition                  : Constraints;
      Postcondition                 : Constraints;
      SubprogramBody                : Dictionary.Symbol;
      SubprogramHasProperBody       : Boolean;
      SubprogramHasSecondAnnotation : Boolean;
      SubprogramHasSecondConstraint : Boolean;
      -- true for procedures or entries with no explicit derives annotation
      HasDerivesAnnotation        : Boolean;
      HasDelayProperty            : Boolean;
      DelayPropertyIsAccountedFor : Boolean;
      -- true for generic subprogram
      IsGeneric : Boolean;
      -- points to generic subprogram symbol if this subprogram is an instantiation
      InstantiationOf : Dictionary.Symbol;
      -- linked list of generic formals, only used if IsGeneric is true
      FirstGenericParameter : Dictionary.Symbol;
      LastGenericParameter  : Dictionary.Symbol;
      -- linked list of generic formal/actual pairings, only used if InstantiationOf is not null
      FirstGenericAssociation  : Dictionary.Symbol;
      LastGenericAssociation   : Dictionary.Symbol;
      SuspendsList             : Dictionary.Symbol;
      IsEntry                  : Boolean;
      EntryBarrier             : Dictionary.Symbol;
      IsInterruptHandler       : Boolean;
      UsesUnprotectedVariables : Boolean;
      BodyIsHidden             : Boolean;
      UsesUncheckedConversion  : Boolean;
      AssignsFromExternal      : Boolean;
      TypeMark                 : Dictionary.Symbol;
      WithClauses              : Dictionary.Symbol;
      UseTypeClauses           : Dictionary.Symbol;
      FirstParameter           : Dictionary.Symbol;
      LastParameter            : Dictionary.Symbol;
      FirstGlobalVariable      : Symbols;
      LastGlobalVariable       : Symbols;
      RenamingDeclarations     : Dictionary.Symbol;
      FirstDeclaration         : Dictionary.Symbol;
      LastDeclaration          : Dictionary.Symbol;
      FirstLoop                : Dictionary.Symbol;
      LastLoop                 : Dictionary.Symbol;
      Specification            : Dictionary.Symbol;
   end record;
   -- pragma pack (SubprogramInfo)

   type SubprogramRef is access SubprogramInfo;

   type ImplicitProofFunctionRef is access Dictionary.Symbol;

   type SubprogramParameterInfo is record
      Name             : LexTokenManager.Lex_String;
      Mode             : Modes;
      Exported         : Booleans;
      Imported         : Booleans;
      Subprogram       : Dictionary.Symbol;
      TypeMark         : Dictionary.Symbol;
      Dependencies     : Symbols;
      GlobalReferences : Symbols;
      Subcomponents    : Dictionary.Symbol;
      IndexConstraints : Dictionary.Symbol; -- links to ParameterConstraintInfo record (see below)
      Next             : Dictionary.Symbol;
   end record;
   -- pragma pack (SubprogramParameterInfo)

   type SubprogramParameterRef is access SubprogramParameterInfo;

   -- If a subprogram parameter is unconstrained we need symbols to represent the anonymous constraints that
   -- are provided by the calling environment.  These are ParameterConstraint symbols and they are in a linked list
   -- because the unconstrained formal parameter may have more than one dimension.

   type ParameterConstraintInfo is record
      TheParameter : Dictionary.Symbol; -- links back to SubprogramParameterInfo
      Dimension    : Positive;
      Next         : Dictionary.Symbol; -- links to another ParameterConstraintInfo for next dimension
   end record;

   type ParameterConstraintRef is access ParameterConstraintInfo;

   -- PNA new record to describe generic parameters
   type GenericParameterInfo is record
      --Name      : LexTokenManager.Lex_String;
      OwningGeneric : Dictionary.Symbol;
      TypeOrObject  : Dictionary.Symbol;
      Kind          : Dictionary.GenericParameterKind;
      Next          : Dictionary.Symbol;
   end record;
   type GenericParameterRef is access GenericParameterInfo;

   type GenericAssociationInfo is record
      Formal       : Dictionary.Symbol;
      ActualSymbol : Dictionary.Symbol;
      Next         : Dictionary.Symbol;
   end record;
   type GenericAssociationRef is access GenericAssociationInfo;

   type GlobalVariableInfo is record
      Mode           : Modes;
      Exported       : Boolean;
      Imported       : Boolean;
      PrefixNeeded   : Boolean;
      Subprogram     : Dictionary.Symbol;
      Variable       : Dictionary.Symbol;
      Dependencies   : Symbols;
      NextSubprogram : Dictionary.Symbol;
      NextVariable   : Dictionary.Symbol;
   end record;
   -- pragma pack (GlobalVariableInfo)

   type GlobalVariableRef is access GlobalVariableInfo;

   type DependencyInfo is record
      Import : Dictionary.Symbol;
      Next   : Dictionary.Symbol;
   end record;
   -- pragma pack (DependencyInfo)

   type DependencyRef is access DependencyInfo;

   type VirtualElementInfo is record
      TheVariable : Dictionary.Symbol; -- VariableInfo
      TheOwner    : Dictionary.Symbol; -- VariableInfo
      SeenByOwner : Boolean;
      Next        : Dictionary.Symbol;
   end record;
   -- pragma pack (VirtualElementInfo)

   type VirtualElementInfoRef is access VirtualElementInfo;

   type SuspendsListItemInfo is record
      ThePOorSO      : Dictionary.Symbol;
      IsAccountedFor : Boolean;
      Next           : Dictionary.Symbol;
   end record;
   -- pragma pack (SuspendsListItemInfo)

   type SuspendsListItemInfoRef is access SuspendsListItemInfo;

   type InterruptStreamMapping is record
      TheHandler         : LexTokenManager.Lex_String;
      TheInterruptStream : LexTokenManager.Lex_String;
      Next               : Dictionary.Symbol;
   end record;
   -- pragma pack (InterruptStreamMapping)

   type InterruptStreamMappingRef is access InterruptStreamMapping;

   type OperatorInfo is record
      Name  : SP_Symbols.SP_Symbol;
      Left  : Dictionary.Symbol;
      Right : Dictionary.Symbol;
   end record;
   -- pragma pack (OperatorInfo)

   type OperatorRef is access OperatorInfo;

   type LoopInfo is record
      Name          : LexTokenManager.Lex_String;
      Region        : Dictionary.Symbol;
      LoopParameter : Dictionary.Symbol;  -- LoopParameterInfor
      OnEntryVars   : Dictionary.Symbol;  -- LoopEntryVariableInfo;

      -- the following field stores a forced type conversion of a Cells.Cell created
      -- by the VCG and representing the exit expression of a for loop.  Storing a
      -- reference to this expression in the Dictionary makes it easy to plant a loop
      -- invariant asserting that the ExitBound variable = the exit expression.
      ExitExpn : Natural;
      -- And similarly for the entry expression...
      EntryExpn : Natural;

      HasExits : Boolean;
      Next     : Dictionary.Symbol;
   end record;
   -- pragma pack (LoopInfo)

   type LoopRef is access LoopInfo;

   type LoopParameterInfo is record
      Name           : LexTokenManager.Lex_String;
      TypeMark       : Dictionary.Symbol;
      TheLoop        : Dictionary.Symbol;
      HasStaticRange : Boolean; -- true means bounds fixed, non-empty and for loop must be entered
      IsReverse      : Boolean; -- we assume for loops are forward unless this is set true
   end record;
   -- pragma pack (LoopParameterInfo)

   type LoopParameterRef is access LoopParameterInfo;

   -- Provides a linked list of variables used in the exit condition of a for loop.  Each
   -- variable is uniquely associated with the loop and the actual Ada variable used in the
   -- exit expression.  These special variables are used by teh VCG to store the value of variables
   -- on entry to the loop so that the loop bounds are properly frozen iaw Ada language definition.
   type LoopEntryVariableInfo is record
      OriginalVar : Dictionary.Symbol;
      TheLoop     : Dictionary.Symbol; -- we need this to construct the name of the var (<var>__entry__<loop>)
      Next        : Dictionary.Symbol;
   end record;

   type LoopEntryVariableRef is access LoopEntryVariableInfo;

   type DeclarationInfo is record
      Context : Contexts;
      Scope   : Dictionary.Scopes;
      Item    : Dictionary.Symbol;
      Next    : Dictionary.Symbol;
   end record;
   -- pragma pack (DeclarationInfo)

   type DeclarationRef is access DeclarationInfo;

   --------------------------------------------------------------------------------

   procedure DiscriminantDebug (CallerID        : in String;
                                Given, Expected : in Dictionary.SymbolDiscriminant) is
   begin
      SPARK_IO.Put_Line
        (SPARK_IO.Standard_Output,
         "*****************************************************************************",
         0);
      SPARK_IO.Put_String (SPARK_IO.Standard_Output, "* From: ", 0);
      SPARK_IO.Put_Line (SPARK_IO.Standard_Output, CallerID, 0);
      SPARK_IO.Put_String (SPARK_IO.Standard_Output, "* Symbol of type ", 0);
      SPARK_IO.Put_String (SPARK_IO.Standard_Output, Dictionary.SymbolDiscriminant'Image (Given), 0);
      SPARK_IO.Put_Line (SPARK_IO.Standard_Output, " found where ", 0);
      SPARK_IO.Put_String (SPARK_IO.Standard_Output, "* ", 0);
      SPARK_IO.Put_String (SPARK_IO.Standard_Output, Dictionary.SymbolDiscriminant'Image (Expected), 0);
      SPARK_IO.Put_Line (SPARK_IO.Standard_Output, " was expected.", 0);
   end DiscriminantDebug;

   --------------------------------------------------------------------------------

   function GetSymbolDiscriminant (Item : Dictionary.Symbol) return Dictionary.SymbolDiscriminant is
      Discriminant : Dictionary.SymbolDiscriminant;
   begin
      if Item = Dictionary.NullSymbol then
         Discriminant := NullSym;
      else
         Discriminant := Dict.Symbols.List (Item).Discriminant;
      end if;
      return Discriminant;
   end GetSymbolDiscriminant;

   --------------------------------------------------------------------------------

   function Get_Symbol_Compilation_Unit (Item : Dictionary.Symbol) return ContextManager.UnitDescriptors is
      Comp_Unit : ContextManager.UnitDescriptors;
   begin
      if Item = Dictionary.NullSymbol then
         Comp_Unit := ContextManager.NullUnit;
      else
         Comp_Unit := Dict.Symbols.List (Item).Comp_Unit;
      end if;
      return Comp_Unit;
   end Get_Symbol_Compilation_Unit;

   --------------------------------------------------------------------------------

   function Get_Symbol_Location (Item : Dictionary.Symbol) return LexTokenManager.Token_Position is
      Location : LexTokenManager.Token_Position;
   begin
      if Item = Dictionary.NullSymbol then
         Location := LexTokenManager.Null_Token_Position;
      else
         Location := Dict.Symbols.List (Item).Loc;
      end if;
      return Location;
   end Get_Symbol_Location;

   --------------------------------------------------------------------------------

   procedure Set_Symbol_Location (Item     : in Dictionary.Symbol;
                                  Location : in LexTokenManager.Token_Position) is
   begin
      if Item /= Dictionary.NullSymbol then
         Dict.Symbols.List (Item).Loc := Location;
      end if;
   end Set_Symbol_Location;

   --------------------------------------------------------------------------------

   function GetSymbolRef (Item : Dictionary.Symbol) return RefType is
   begin
      return Dict.Symbols.List (Item).Ref;
   end GetSymbolRef;

   --------------------------------------------------------------------------------

   function GetTypeRef (Item : Dictionary.Symbol) return TypeRef is
      function RefTypeToTypeRef is new Unchecked_Conversion (RefType, TypeRef);

      D : Dictionary.SymbolDiscriminant;
      R : TypeRef;
   begin
      D := GetSymbolDiscriminant (Item);
      if D = TypeSymbol then
         R := RefTypeToTypeRef (GetSymbolRef (Item));
      else
         DiscriminantDebug ("GetTypeRef", D, TypeSymbol);
         SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Invalid_Symbol_Table,
                                   Msg     => "");
      end if;
      return R;
   end GetTypeRef;

   --------------------------------------------------------------------------------

   function GetProtectedRef (Item : Dictionary.Symbol) return ProtectedRef is
      function RefTypeToProtectedRef is new Unchecked_Conversion (RefType, ProtectedRef);
   begin
      if GetSymbolDiscriminant (Item) /= ProtectedInfoSymbol then
         DiscriminantDebug ("GetProtectedRef", GetSymbolDiscriminant (Item), ProtectedInfoSymbol);
         SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Invalid_Symbol_Table,
                                   Msg     => "");
      end if;
      return RefTypeToProtectedRef (GetSymbolRef (Item));
   end GetProtectedRef;

   --------------------------------------------------------------------------------

   function GetTaskRef (Item : Dictionary.Symbol) return TaskRef is
      function RefTypeToTaskRef is new Unchecked_Conversion (RefType, TaskRef);
   begin
      if GetSymbolDiscriminant (Item) /= TaskInfoSymbol then
         DiscriminantDebug ("GetTaskRef", GetSymbolDiscriminant (Item), TaskInfoSymbol);
         SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Invalid_Symbol_Table,
                                   Msg     => "");
      end if;
      return RefTypeToTaskRef (GetSymbolRef (Item));
   end GetTaskRef;

   --------------------------------------------------------------------------------

   function GetKnownDiscriminantRef (Item : Dictionary.Symbol) return KnownDiscriminantRef is
      function RefTypeToKnownDiscriminantRef is new Unchecked_Conversion (RefType, KnownDiscriminantRef);
   begin
      if GetSymbolDiscriminant (Item) /= KnownDiscriminantSymbol then
         DiscriminantDebug ("GetKnownDiscriminantRef", GetSymbolDiscriminant (Item), KnownDiscriminantSymbol);
         SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Invalid_Symbol_Table,
                                   Msg     => "");
      end if;
      return RefTypeToKnownDiscriminantRef (GetSymbolRef (Item));
   end GetKnownDiscriminantRef;

   --------------------------------------------------------------------------------

   function GetSubtypeRef (Item : Dictionary.Symbol) return SubtypeRef is
      function RefTypeToSubtypeRef is new Unchecked_Conversion (RefType, SubtypeRef);
   begin
      if GetSymbolDiscriminant (Item) /= SubtypeSymbol then
         DiscriminantDebug ("GetSubtypeRef", GetSymbolDiscriminant (Item), SubtypeSymbol);
         SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Invalid_Symbol_Table,
                                   Msg     => "");
      end if;
      return RefTypeToSubtypeRef (GetSymbolRef (Item));
   end GetSubtypeRef;

   --------------------------------------------------------------------------------

   function GetDiscriminantConstraintRef (Item : Dictionary.Symbol) return DiscriminantConstraintRef is
      function RefTypeToDiscriminantConstraintRef is new Unchecked_Conversion (RefType, DiscriminantConstraintRef);
   begin
      if GetSymbolDiscriminant (Item) /= DiscriminantConstraintSymbol then
         DiscriminantDebug ("GetDiscriminantConstraintRef", GetSymbolDiscriminant (Item), DiscriminantConstraintSymbol);
         SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Invalid_Symbol_Table,
                                   Msg     => "");
      end if;
      return RefTypeToDiscriminantConstraintRef (GetSymbolRef (Item));
   end GetDiscriminantConstraintRef;

   --------------------------------------------------------------------------------

   function GetEnumerationLiteralRef (Item : Dictionary.Symbol) return EnumerationLiteralRef is
      function RefTypeToEnumerationLiteralRef is new Unchecked_Conversion (RefType, EnumerationLiteralRef);
   begin
      if GetSymbolDiscriminant (Item) /= EnumerationLiteralSymbol then
         DiscriminantDebug ("GetEnumerationLiteralRef", GetSymbolDiscriminant (Item), EnumerationLiteralSymbol);
         SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Invalid_Symbol_Table,
                                   Msg     => "");
      end if;
      return RefTypeToEnumerationLiteralRef (GetSymbolRef (Item));
   end GetEnumerationLiteralRef;

   --------------------------------------------------------------------------------

   function GetArrayIndexRef (Item : Dictionary.Symbol) return ArrayIndexRef is
      function RefTypeToArrayIndexRef is new Unchecked_Conversion (RefType, ArrayIndexRef);
   begin
      if GetSymbolDiscriminant (Item) /= ArrayIndexSymbol then
         DiscriminantDebug ("GetArrayIndexRef", GetSymbolDiscriminant (Item), ArrayIndexSymbol);
         SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Invalid_Symbol_Table,
                                   Msg     => "");
      end if;
      return RefTypeToArrayIndexRef (GetSymbolRef (Item));
   end GetArrayIndexRef;

   --------------------------------------------------------------------------------

   function GetRecordComponentRef (Item : Dictionary.Symbol) return RecordComponentRef is
      function RefTypeToRecordComponentRef is new Unchecked_Conversion (RefType, RecordComponentRef);
   begin
      if GetSymbolDiscriminant (Item) /= RecordComponentSymbol then
         DiscriminantDebug ("GetRecordComponentRef", GetSymbolDiscriminant (Item), RecordComponentSymbol);
         SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Invalid_Symbol_Table,
                                   Msg     => "");
      end if;
      return RefTypeToRecordComponentRef (GetSymbolRef (Item));
   end GetRecordComponentRef;

   --------------------------------------------------------------------------------

   function GetSubcomponentRef (Item : Dictionary.Symbol) return SubcomponentRef is
      function RefTypeToSubcomponentRef is new Unchecked_Conversion (RefType, SubcomponentRef);
   begin
      if GetSymbolDiscriminant (Item) /= SubcomponentSymbol then
         DiscriminantDebug ("GetSubcomponentRef", GetSymbolDiscriminant (Item), SubcomponentSymbol);
         SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Invalid_Symbol_Table,
                                   Msg     => "");
      end if;
      return RefTypeToSubcomponentRef (GetSymbolRef (Item));
   end GetSubcomponentRef;

   --------------------------------------------------------------------------------

   function GetRulePolicyRef (Item : Dictionary.Symbol) return RulePolicyRef is
      function RefTypeToRulePolicyRef is new Unchecked_Conversion (RefType, RulePolicyRef);
   begin
      if GetSymbolDiscriminant (Item) /= RulePolicySymbol then
         DiscriminantDebug ("GetPolicyRuleRef", GetSymbolDiscriminant (Item), RulePolicySymbol);
         SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Invalid_Symbol_Table,
                                   Msg     => "");
      end if;
      return RefTypeToRulePolicyRef (GetSymbolRef (Item));
   end GetRulePolicyRef;

   --------------------------------------------------------------------------------

   function GetConstantRef (Item : Dictionary.Symbol) return ConstantRef is
      function RefTypeToConstantRef is new Unchecked_Conversion (RefType, ConstantRef);
   begin
      if GetSymbolDiscriminant (Item) /= ConstantSymbol then
         DiscriminantDebug ("GetConstantRef", GetSymbolDiscriminant (Item), ConstantSymbol);
         SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Invalid_Symbol_Table,
                                   Msg     => "");
      end if;
      return RefTypeToConstantRef (GetSymbolRef (Item));
   end GetConstantRef;

   --------------------------------------------------------------------------------

   function GetVariableRef (Item : Dictionary.Symbol) return VariableRef is
      function RefTypeToVariableRef is new Unchecked_Conversion (RefType, VariableRef);
   begin
      if GetSymbolDiscriminant (Item) /= VariableSymbol then
         DiscriminantDebug ("RefTypeToVariableRef", GetSymbolDiscriminant (Item), VariableSymbol);
         SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Invalid_Symbol_Table,
                                   Msg     => "");
      end if;
      return RefTypeToVariableRef (GetSymbolRef (Item));
   end GetVariableRef;

   --------------------------------------------------------------------------------

   function GetQuantifiedVariableRef (Item : Dictionary.Symbol) return QuantifiedVariableRef is
      function RefTypeToQuantifiedVariableRef is new Unchecked_Conversion (RefType, QuantifiedVariableRef);
   begin
      if GetSymbolDiscriminant (Item) /= QuantifiedVariableSymbol then
         DiscriminantDebug ("GetQuantifiedVariableRef", GetSymbolDiscriminant (Item), QuantifiedVariableSymbol);
         SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Invalid_Symbol_Table,
                                   Msg     => "");
      end if;
      return RefTypeToQuantifiedVariableRef (GetSymbolRef (Item));
   end GetQuantifiedVariableRef;

   --------------------------------------------------------------------------------

   function GetImplicitReturnVariableRef (Item : Dictionary.Symbol) return ImplicitReturnVariableRef is
      function RefTypeToImplicitReturnVariableRef is new Unchecked_Conversion (RefType, ImplicitReturnVariableRef);
   begin
      if GetSymbolDiscriminant (Item) /= ImplicitReturnVariableSymbol then
         DiscriminantDebug ("GetImplicitReturnVariableRef", GetSymbolDiscriminant (Item), ImplicitReturnVariableSymbol);
         SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Invalid_Symbol_Table,
                                   Msg     => "");
      end if;
      return RefTypeToImplicitReturnVariableRef (GetSymbolRef (Item));
   end GetImplicitReturnVariableRef;

   --------------------------------------------------------------------------------

   function GetOwnVariableRef (Item : Dictionary.Symbol) return OwnVariableRef is
      function RefTypeToOwnVariableRef is new Unchecked_Conversion (RefType, OwnVariableRef);
   begin
      if GetSymbolDiscriminant (Item) /= OwnVariableSymbol then
         DiscriminantDebug ("GetOwnVariableRef", GetSymbolDiscriminant (Item), OwnVariableSymbol);
         SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Invalid_Symbol_Table,
                                   Msg     => "");
      end if;
      return RefTypeToOwnVariableRef (GetSymbolRef (Item));
   end GetOwnVariableRef;

   --------------------------------------------------------------------------------

   function GetImplicitInStreamRef (Item : Dictionary.Symbol) return ImplicitInStreamRef is
      function RefTypeToImplicitInStreamRef is new Unchecked_Conversion (RefType, ImplicitInStreamRef);
   begin
      if GetSymbolDiscriminant (Item) /= ProtectedImplicitInStreamSymbol then
         DiscriminantDebug ("GetImplicitInStreamRef", GetSymbolDiscriminant (Item), ProtectedImplicitInStreamSymbol);
         SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Invalid_Symbol_Table,
                                   Msg     => "");
      end if;
      return RefTypeToImplicitInStreamRef (GetSymbolRef (Item));
   end GetImplicitInStreamRef;

   --------------------------------------------------------------------------------

   function GetOwnTaskRef (Item : Dictionary.Symbol) return OwnTaskRef is
      function RefTypeToOwnTaskRef is new Unchecked_Conversion (RefType, OwnTaskRef);
   begin
      if GetSymbolDiscriminant (Item) /= OwnTaskSymbol then
         DiscriminantDebug ("GetOwnTaskRef", GetSymbolDiscriminant (Item), OwnTaskSymbol);
         SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Invalid_Symbol_Table,
                                   Msg     => "");
      end if;
      return RefTypeToOwnTaskRef (GetSymbolRef (Item));
   end GetOwnTaskRef;

   --------------------------------------------------------------------------------

   function GetConstituentRef (Item : Dictionary.Symbol) return ConstituentRef is
      function RefTypeToConstituentRef is new Unchecked_Conversion (RefType, ConstituentRef);
   begin
      if GetSymbolDiscriminant (Item) /= ConstituentSymbol then
         DiscriminantDebug ("GetConstituentRef", GetSymbolDiscriminant (Item), ConstituentSymbol);
         SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Invalid_Symbol_Table,
                                   Msg     => "");
      end if;
      return RefTypeToConstituentRef (GetSymbolRef (Item));
   end GetConstituentRef;

   --------------------------------------------------------------------------------

   function GetPackageRef (Item : Dictionary.Symbol) return PackageRef is
      function RefTypeToPackageRef is new Unchecked_Conversion (RefType, PackageRef);
   begin
      if GetSymbolDiscriminant (Item) /= PackageSymbol then
         DiscriminantDebug ("GetPackageRef", GetSymbolDiscriminant (Item), PackageSymbol);
         SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Invalid_Symbol_Table,
                                   Msg     => "");
      end if;
      return RefTypeToPackageRef (GetSymbolRef (Item));
   end GetPackageRef;

   --------------------------------------------------------------------------------

   function GetContextClauseRef (Item : Dictionary.Symbol) return ContextClauseRef is
      function RefTypeToContextClauseRef is new Unchecked_Conversion (RefType, ContextClauseRef);
   begin
      if GetSymbolDiscriminant (Item) /= ContextClauseSymbol then
         DiscriminantDebug ("GetContextClauseRef", GetSymbolDiscriminant (Item), ContextClauseSymbol);
         SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Invalid_Symbol_Table,
                                   Msg     => "");
      end if;
      return RefTypeToContextClauseRef (GetSymbolRef (Item));
   end GetContextClauseRef;

   --------------------------------------------------------------------------------

   function GetUseTypeClauseRef (Item : Dictionary.Symbol) return UseTypeClauseRef is
      function RefTypeToUseTypeClauseRef is new Unchecked_Conversion (RefType, UseTypeClauseRef);
   begin
      if GetSymbolDiscriminant (Item) /= UseTypeClauseSymbol then
         DiscriminantDebug ("GetUseTypeClauseRef", GetSymbolDiscriminant (Item), UseTypeClauseSymbol);
         SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Invalid_Symbol_Table,
                                   Msg     => "");
      end if;
      return RefTypeToUseTypeClauseRef (GetSymbolRef (Item));
   end GetUseTypeClauseRef;

   --------------------------------------------------------------------------------

   function GetSubprogramRef (Item : Dictionary.Symbol) return SubprogramRef is
      function RefTypeToSubprogramRef is new Unchecked_Conversion (RefType, SubprogramRef);
   begin
      if GetSymbolDiscriminant (Item) /= SubprogramSymbol then
         DiscriminantDebug ("GetSubprogramRef", GetSymbolDiscriminant (Item), SubprogramSymbol);
         SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Invalid_Symbol_Table,
                                   Msg     => "");
      end if;
      return RefTypeToSubprogramRef (GetSymbolRef (Item));
   end GetSubprogramRef;

   --------------------------------------------------------------------------------

   function GetImplicitProofFunctionRef (Item : Dictionary.Symbol) return ImplicitProofFunctionRef is
      function RefTypeToImplicitProofFunctionRef is new Unchecked_Conversion (RefType, ImplicitProofFunctionRef);
   begin
      if GetSymbolDiscriminant (Item) /= ImplicitProofFunctionSymbol then
         DiscriminantDebug ("GetImplicitProofFunctionRef", GetSymbolDiscriminant (Item), ImplicitProofFunctionSymbol);
         SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Invalid_Symbol_Table,
                                   Msg     => "");
      end if;
      return RefTypeToImplicitProofFunctionRef (GetSymbolRef (Item));
   end GetImplicitProofFunctionRef;

   --------------------------------------------------------------------------------

   function GetSubprogramParameterRef (Item : Dictionary.Symbol) return SubprogramParameterRef is
      function RefTypeToSubprogramParameterRef is new Unchecked_Conversion (RefType, SubprogramParameterRef);
   begin
      if GetSymbolDiscriminant (Item) /= SubprogramParameterSymbol then
         DiscriminantDebug ("GetSubprogramParameterRef", GetSymbolDiscriminant (Item), SubprogramParameterSymbol);
         SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Invalid_Symbol_Table,
                                   Msg     => "");
      end if;
      return RefTypeToSubprogramParameterRef (GetSymbolRef (Item));
   end GetSubprogramParameterRef;

   --------------------------------------------------------------------------------

   function GetParameterConstraintRef (Item : Dictionary.Symbol) return ParameterConstraintRef is
      function RefTypeToParameterConstraintRef is new Unchecked_Conversion (RefType, ParameterConstraintRef);
   begin
      if GetSymbolDiscriminant (Item) /= ParameterConstraintSymbol then
         DiscriminantDebug ("GetParameterConstraintRef", GetSymbolDiscriminant (Item), ParameterConstraintSymbol);
         SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Invalid_Symbol_Table,
                                   Msg     => "");
      end if;
      return RefTypeToParameterConstraintRef (GetSymbolRef (Item));
   end GetParameterConstraintRef;

   --------------------------------------------------------------------------------

   function GetGenericParameterRef (Item : Dictionary.Symbol) return GenericParameterRef is
      function RefTypeToGenericParameterRef is new Unchecked_Conversion (RefType, GenericParameterRef);
   begin
      if GetSymbolDiscriminant (Item) /= GenericParameterSymbol then
         DiscriminantDebug ("GetGenericParameterRef", GetSymbolDiscriminant (Item), GenericParameterSymbol);
         SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Invalid_Symbol_Table,
                                   Msg     => "");
      end if;
      return RefTypeToGenericParameterRef (GetSymbolRef (Item));
   end GetGenericParameterRef;

   --------------------------------------------------------------------------------

   function GetGenericAssociationRef (Item : Dictionary.Symbol) return GenericAssociationRef is
      function RefTypeToGenericAssociationRef is new Unchecked_Conversion (RefType, GenericAssociationRef);
   begin
      if GetSymbolDiscriminant (Item) /= GenericAssociationSymbol then
         DiscriminantDebug ("GetGenericAssociationRef", GetSymbolDiscriminant (Item), GenericAssociationSymbol);
         SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Invalid_Symbol_Table,
                                   Msg     => "");
      end if;
      return RefTypeToGenericAssociationRef (GetSymbolRef (Item));
   end GetGenericAssociationRef;

   --------------------------------------------------------------------------------

   function GetGlobalVariableRef (Item : Dictionary.Symbol) return GlobalVariableRef is
      function RefTypeToGlobalVariableRef is new Unchecked_Conversion (RefType, GlobalVariableRef);
   begin
      if GetSymbolDiscriminant (Item) /= GlobalVariableSymbol then
         DiscriminantDebug ("GetGlobalVariableRef", GetSymbolDiscriminant (Item), GlobalVariableSymbol);
         SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Invalid_Symbol_Table,
                                   Msg     => "");
      end if;
      return RefTypeToGlobalVariableRef (GetSymbolRef (Item));
   end GetGlobalVariableRef;

   --------------------------------------------------------------------------------

   function GetGlobalMode (Item : Dictionary.Symbol) return Dictionary.Modes is
   begin
      return GetGlobalVariableRef (Item).Mode;
   end GetGlobalMode;

   --------------------------------------------------------------------------------

   function GetDependencyRef (Item : Dictionary.Symbol) return DependencyRef is
      function RefTypeToDependencyRef is new Unchecked_Conversion (RefType, DependencyRef);
   begin
      if GetSymbolDiscriminant (Item) /= DependencySymbol then
         DiscriminantDebug ("GetDependencyRef", GetSymbolDiscriminant (Item), DependencySymbol);
         SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Invalid_Symbol_Table,
                                   Msg     => "");
      end if;
      return RefTypeToDependencyRef (GetSymbolRef (Item));
   end GetDependencyRef;

   --------------------------------------------------------------------------------

   function GetVirtualElementRef (Item : Dictionary.Symbol) return VirtualElementInfoRef is
      function RefTypeToVirtualElementRef is new Unchecked_Conversion (RefType, VirtualElementInfoRef);
   begin
      if GetSymbolDiscriminant (Item) /= VirtualElementSymbol then
         DiscriminantDebug ("GetVirtualElementRef", GetSymbolDiscriminant (Item), VirtualElementSymbol);
         SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Invalid_Symbol_Table,
                                   Msg     => "");
      end if;
      return RefTypeToVirtualElementRef (GetSymbolRef (Item));
   end GetVirtualElementRef;

   --------------------------------------------------------------------------------

   function GetSuspendsListItemRef (Item : Dictionary.Symbol) return SuspendsListItemInfoRef is
      function RefTypeToSuspendsListItemRef is new Unchecked_Conversion (RefType, SuspendsListItemInfoRef);
   begin
      if GetSymbolDiscriminant (Item) /= SuspendsListItemSymbol then
         DiscriminantDebug ("GetSuspendsListItemRef", GetSymbolDiscriminant (Item), SuspendsListItemSymbol);
         SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Invalid_Symbol_Table,
                                   Msg     => "");
      end if;
      return RefTypeToSuspendsListItemRef (GetSymbolRef (Item));
   end GetSuspendsListItemRef;

   --------------------------------------------------------------------------------

   function GetInterruptStreamMappingRef (Item : Dictionary.Symbol) return InterruptStreamMappingRef is
      function RefTypeToInterruptStreamMappingRef is new Unchecked_Conversion (RefType, InterruptStreamMappingRef);
   begin
      if GetSymbolDiscriminant (Item) /= InterruptStreamMappingSymbol then
         DiscriminantDebug ("GetInterruptStreamMappingRef", GetSymbolDiscriminant (Item), InterruptStreamMappingSymbol);
         SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Invalid_Symbol_Table,
                                   Msg     => "");
      end if;
      return RefTypeToInterruptStreamMappingRef (GetSymbolRef (Item));
   end GetInterruptStreamMappingRef;

   --------------------------------------------------------------------------------

   function GetOperatorRef (Item : Dictionary.Symbol) return OperatorRef is
      function RefTypeToOperatorRef is new Unchecked_Conversion (RefType, OperatorRef);
   begin
      if GetSymbolDiscriminant (Item) /= OperatorSymbol then
         DiscriminantDebug ("GetOperatorRef", GetSymbolDiscriminant (Item), OperatorSymbol);
         SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Invalid_Symbol_Table,
                                   Msg     => "");
      end if;
      return RefTypeToOperatorRef (GetSymbolRef (Item));
   end GetOperatorRef;

   --------------------------------------------------------------------------------

   function GetLoopRef (Item : Dictionary.Symbol) return LoopRef is
      function RefTypeToLoopRef is new Unchecked_Conversion (RefType, LoopRef);
   begin
      if GetSymbolDiscriminant (Item) /= LoopSymbol then
         DiscriminantDebug ("GetLoopRef", GetSymbolDiscriminant (Item), LoopSymbol);
         SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Invalid_Symbol_Table,
                                   Msg     => "");
      end if;
      return RefTypeToLoopRef (GetSymbolRef (Item));
   end GetLoopRef;

   --------------------------------------------------------------------------------

   function GetLoopParameterRef (Item : Dictionary.Symbol) return LoopParameterRef is
      function RefTypeToLoopParameterRef is new Unchecked_Conversion (RefType, LoopParameterRef);
   begin
      if GetSymbolDiscriminant (Item) /= LoopParameterSymbol then
         DiscriminantDebug ("GetLoopParameterRef", GetSymbolDiscriminant (Item), LoopParameterSymbol);
         SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Invalid_Symbol_Table,
                                   Msg     => "");
      end if;
      return RefTypeToLoopParameterRef (GetSymbolRef (Item));
   end GetLoopParameterRef;

   --------------------------------------------------------------------------------

   function GetLoopEntryVariableRef (Item : Dictionary.Symbol) return LoopEntryVariableRef is
      function RefTypeToLoopEntryVariableRef is new Unchecked_Conversion (RefType, LoopEntryVariableRef);
   begin
      if GetSymbolDiscriminant (Item) /= LoopEntryVariableSymbol then
         DiscriminantDebug ("GetLoopEntryVariableRef", GetSymbolDiscriminant (Item), LoopEntryVariableSymbol);
         SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Invalid_Symbol_Table,
                                   Msg     => "");
      end if;
      return RefTypeToLoopEntryVariableRef (GetSymbolRef (Item));
   end GetLoopEntryVariableRef;

   --------------------------------------------------------------------------------

   function GetDeclarationRef (Item : Dictionary.Symbol) return DeclarationRef is
      function RefTypeToDeclarationRef is new Unchecked_Conversion (RefType, DeclarationRef);
   begin
      if GetSymbolDiscriminant (Item) /= DeclarationSymbol then
         DiscriminantDebug ("GetDeclarationRef", GetSymbolDiscriminant (Item), DeclarationSymbol);
         SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Invalid_Symbol_Table,
                                   Msg     => "");
      end if;
      return RefTypeToDeclarationRef (GetSymbolRef (Item));
   end GetDeclarationRef;

   --------------------------------------------------------------------------------

   procedure AddSymbol
     (Discriminant : in     Dictionary.SymbolDiscriminant;
      Ref          : in     RefType;
      Comp_Unit    : in     ContextManager.UnitDescriptors;
      Loc          : in     LexTokenManager.Token_Position;
      Item         :    out Dictionary.Symbol)
   --# global in out Dict;
   --# derives Dict from *,
   --#                   Discriminant,
   --#                   Ref &
   --#         Item from Dict;
   is
   begin

      if Dict.Symbols.Current = Dictionary.Symbol'Last then
         SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Symbol_Table_Overflow_Static,
                                   Msg     => "");
      end if;

      Dict.Symbols.Current                     := Dict.Symbols.Current + 1;
      Dict.Symbols.List (Dict.Symbols.Current) :=
        Dictionary.SymbolInfo'(Discriminant => Discriminant,
                               Comp_Unit    => Comp_Unit,
                               Loc          => Loc,
                               Ref          => Ref);
      Item                                     := Dict.Symbols.Current;

   end AddSymbol;

   --------------------------------------------------------------------------------

   procedure CreateType
     (Name      : in     LexTokenManager.Lex_String;
      Comp_Unit : in     ContextManager.UnitDescriptors;
      Loc       : in     LexTokenManager.Token_Position;
      TypeMark  :    out Dictionary.Symbol) is
      NewType : TypeRef;
      function TypeRefToRefType is new Unchecked_Conversion (TypeRef, RefType);
   begin
      NewType :=
        new TypeInfo'
        (Name                   => Name,
         Discriminant           => UnknownType,
         IsPrivate              => Never,
         IsLimited              => Never,
         IsLimitedPrivate       => False,
         IsDerived              => False,
         IsTagged               => False,
         IsAbstract             => False,
         IsOwnVarType           => False,
         IsAtomic               => False,
         EqualityDefined        => True,
         ContainsFloat          => False,
         Constrained            => False,
         Static                 => True,
         Wellformed             => True,
         Parent                 => Dictionary.NullSymbol,
         BaseType               => Dictionary.NullSymbol,
         Extends                => Dictionary.NullSymbol,
         Accesses               => Dictionary.NullSymbol,
         Lower                  => LexTokenManager.Null_String,
         Upper                  => LexTokenManager.Null_String,
         Modulus                => LexTokenManager.Null_String,
         ErrorBound             => LexTokenManager.Null_String,
         Head                   => Dictionary.NullSymbol,
         Tail                   => Dictionary.NullSymbol,
         ArrayComponent         => Dictionary.NullSymbol,
         Announcement           => Dictionary.NullSymbol,
         Declaration            => Dictionary.NullSymbol,
         PrivateTypeDeclaration => Dictionary.NullSymbol,
         TheVirtualElementList  => Dictionary.NullSymbol,
         AncilliaryFields       => Dictionary.NullSymbol,
         SizeAttribute          => LexTokenManager.Null_String);
      AddSymbol
        (Discriminant => TypeSymbol,
         Ref          => TypeRefToRefType (NewType),
         Comp_Unit    => Comp_Unit,
         Loc          => Loc,
         Item         => TypeMark);
   exception
      when Storage_Error =>
         SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Symbol_Table_Overflow_Dynamic,
                                   Msg     => "");
   end CreateType;

   --------------------------------------------------------------------------------

   procedure CreateProtectedInfo
     (Comp_Unit : in     ContextManager.UnitDescriptors;
      Loc       : in     LexTokenManager.Token_Position;
      InfoSym   :    out Dictionary.Symbol) is
      NewProtected : ProtectedRef;
      function ProtectedRefToRefType is new Unchecked_Conversion (ProtectedRef, RefType);
   begin
      NewProtected :=
        new ProtectedInfo'
        (OwnVariable       => Dictionary.NullSymbol,
         ElementsHidden    => False,
         VisiblePart       => PartInfo'(WithClauses          => Dictionary.NullSymbol,
                                        UseTypeClauses       => Dictionary.NullSymbol,
                                        RenamingDeclarations => Dictionary.NullSymbol,
                                        FirstDeclaration     => Dictionary.NullSymbol,
                                        LastDeclaration      => Dictionary.NullSymbol),
         LocalPart         => PartInfo'(WithClauses          => Dictionary.NullSymbol,
                                        UseTypeClauses       => Dictionary.NullSymbol,
                                        RenamingDeclarations => Dictionary.NullSymbol,
                                        FirstDeclaration     => Dictionary.NullSymbol,
                                        LastDeclaration      => Dictionary.NullSymbol),
         PrivatePart       => PartInfo'(WithClauses          => Dictionary.NullSymbol,
                                        UseTypeClauses       => Dictionary.NullSymbol,
                                        RenamingDeclarations => Dictionary.NullSymbol,
                                        FirstDeclaration     => Dictionary.NullSymbol,
                                        LastDeclaration      => Dictionary.NullSymbol),
         TheEntry          => Dictionary.NullSymbol,
         FirstDiscriminant => Dictionary.NullSymbol,
         LastDiscriminant  => Dictionary.NullSymbol,
         Pragmas           => PragmaLists'(Dictionary.Priority          => Pragmas'(Given => False,
                                                                                    Value => LexTokenManager.Null_String),
                                           Dictionary.InterruptPriority => Pragmas'(Given => False,
                                                                                    Value => LexTokenManager.Null_String),
                                           Dictionary.AttachHandler     => Pragmas'(Given => False,
                                                                                    Value => LexTokenManager.Null_String)),
         ProtectedBody     => NullSymbol,
         HasProperBody     => False);
      AddSymbol
        (Discriminant => ProtectedInfoSymbol,
         Ref          => ProtectedRefToRefType (NewProtected),
         Comp_Unit    => Comp_Unit,
         Loc          => Loc,
         Item         => InfoSym);
   exception
      when Storage_Error =>
         SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Symbol_Table_Overflow_Dynamic,
                                   Msg     => "");
   end CreateProtectedInfo;

   --------------------------------------------------------------------------------

   procedure CreateProtectedImplicitInStream
     (TheOwnVar   : in     Dictionary.Symbol;
      Comp_Unit   : in     ContextManager.UnitDescriptors;
      Loc         : in     LexTokenManager.Token_Position;
      InStreamSym :    out Dictionary.Symbol) is
      NewImplicitInStream : ImplicitInStreamRef;
      function ImplicitInStreamRefToRefType is new Unchecked_Conversion (ImplicitInStreamRef, RefType);
   begin
      NewImplicitInStream := new ImplicitInStreamInfo'(OwnVariable => TheOwnVar);
      AddSymbol
        (Discriminant => ProtectedImplicitInStreamSymbol,
         Ref          => ImplicitInStreamRefToRefType (NewImplicitInStream),
         Comp_Unit    => Comp_Unit,
         Loc          => Loc,
         Item         => InStreamSym);
   exception
      when Storage_Error =>
         SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Symbol_Table_Overflow_Dynamic,
                                   Msg     => "");
   end CreateProtectedImplicitInStream;

   --------------------------------------------------------------------------------

   procedure CreateTaskInfo
     (Comp_Unit : in     ContextManager.UnitDescriptors;
      Loc       : in     LexTokenManager.Token_Position;
      InfoSym   :    out Dictionary.Symbol) is
      NewTask     : TaskRef;
      NullSymbols : constant Symbols := Symbols'(Dictionary.Abstractions => Dictionary.NullSymbol);
      function TaskRefToRefType is new Unchecked_Conversion (TaskRef, RefType);
   begin
      NewTask :=
        new TaskInfo'
        (SignatureIsWellformed    => Booleans'(Dictionary.Abstractions => True),
         HasSecondAnnotation      => False,
         HasDerivesAnnotation     => False,
         LocalPart                => PartInfo'(WithClauses          => Dictionary.NullSymbol,
                                               UseTypeClauses       => Dictionary.NullSymbol,
                                               RenamingDeclarations => Dictionary.NullSymbol,
                                               FirstDeclaration     => Dictionary.NullSymbol,
                                               LastDeclaration      => Dictionary.NullSymbol),
         FirstDiscriminant        => Dictionary.NullSymbol,
         LastDiscriminant         => Dictionary.NullSymbol,
         Pragmas                  => PragmaLists'(Dictionary.Priority          => Pragmas'(Given => False,
                                                                                           Value => LexTokenManager.Null_String),
                                                  Dictionary.InterruptPriority => Pragmas'(Given => False,
                                                                                           Value => LexTokenManager.Null_String),
                                                  Dictionary.AttachHandler     => Pragmas'(Given => False,
                                                                                           Value => LexTokenManager.Null_String)),
         FirstGlobalVariable      => NullSymbols,
         LastGlobalVariable       => NullSymbols,
         TaskBody                 => NullSymbol,
         HasProperBody            => False,
         BodyIsHidden             => False,
         SuspendsList             => NullSymbol,
         FirstLoop                => NullSymbol,
         LastLoop                 => NullSymbol,
         UsesUnprotectedVariables => False,
         UsesUncheckedConversion  => False,
         AssignsFromExternal      => False);
      AddSymbol
        (Discriminant => TaskInfoSymbol,
         Ref          => TaskRefToRefType (NewTask),
         Comp_Unit    => Comp_Unit,
         Loc          => Loc,
         Item         => InfoSym);
   exception
      when Storage_Error =>
         SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Symbol_Table_Overflow_Dynamic,
                                   Msg     => "");
   end CreateTaskInfo;

   --------------------------------------------------------------------------------

   procedure SetTypeDiscriminant (TypeMark     : in Dictionary.Symbol;
                                  Discriminant : in TypeDiscriminant) is
   begin
      GetTypeRef (TypeMark).Discriminant := Discriminant;
   end SetTypeDiscriminant;

   --------------------------------------------------------------------------------

   procedure SetTypePrivate (TypeMark  : in Dictionary.Symbol;
                             IsPrivate : in TriState) is
   begin
      GetTypeRef (TypeMark).IsPrivate := IsPrivate;
   end SetTypePrivate;

   --------------------------------------------------------------------------------

   procedure SetTypeLimited (TypeMark  : in Dictionary.Symbol;
                             IsLimited : in TriState) is
   begin
      GetTypeRef (TypeMark).IsLimited := IsLimited;
   end SetTypeLimited;

   --------------------------------------------------------------------------------

   procedure SetTypeIsTagged (TypeMark : in Dictionary.Symbol;
                              IsTagged : in Boolean) is
   begin
      GetTypeRef (TypeMark).IsTagged := IsTagged;
   end SetTypeIsTagged;

   --------------------------------------------------------------------------------

   procedure SetTypeIsAbstract (TypeMark   : in Dictionary.Symbol;
                                IsAbstract : in Boolean) is
   begin
      GetTypeRef (TypeMark).IsAbstract := IsAbstract;
   end SetTypeIsAbstract;

   --------------------------------------------------------------------------------

   procedure SetTypeIsOwnVarType (TypeMark     : in Dictionary.Symbol;
                                  IsOwnVarType : in Boolean) is
   begin
      GetTypeRef (TypeMark).IsOwnVarType := IsOwnVarType;
   end SetTypeIsOwnVarType;

   --------------------------------------------------------------------------------

   procedure SetTypeExtends (TypeMark : in Dictionary.Symbol;
                             RootType : in Dictionary.Symbol) is
   begin
      GetTypeRef (TypeMark).Extends := RootType;
   end SetTypeExtends;

   --------------------------------------------------------------------------------

   procedure SetTypeAccesses (TheType   : in Dictionary.Symbol;
                              TheAccess : in Dictionary.Symbol) is
   begin
      GetTypeRef (TheType).Accesses := TheAccess;
   end SetTypeAccesses;

   --------------------------------------------------------------------------------

   procedure SetTypeLimitedPrivate (TypeMark : in Dictionary.Symbol) is
   begin
      GetTypeRef (TypeMark).IsLimitedPrivate := True;
   end SetTypeLimitedPrivate;

   --------------------------------------------------------------------------------

   procedure SetTypeDerived (TypeMark : in Dictionary.Symbol;
                             Derived  : in Boolean) is
   begin
      GetTypeRef (TypeMark).IsDerived := Derived;
   end SetTypeDerived;

   --------------------------------------------------------------------------------

   procedure SetTypeEqualityDefined (TypeMark        : in Dictionary.Symbol;
                                     EqualityDefined : in Boolean) is
   begin
      GetTypeRef (TypeMark).EqualityDefined := EqualityDefined;
   end SetTypeEqualityDefined;

   --------------------------------------------------------------------------------
   procedure SetTypeContainsFloat (TypeMark      : in Dictionary.Symbol;
                                   ContainsFloat : in Boolean) is
   begin
      GetTypeRef (TypeMark).ContainsFloat := ContainsFloat;
   end SetTypeContainsFloat;

   --------------------------------------------------------------------------------

   procedure SetTypeConstrained (ArrayType   : in Dictionary.Symbol;
                                 Constrained : in Boolean) is
   begin
      GetTypeRef (ArrayType).Constrained := Constrained;
   end SetTypeConstrained;

   --------------------------------------------------------------------------------

   procedure SetTypeStatic (TypeMark : in Dictionary.Symbol;
                            Static   : in Boolean) is
   begin
      GetTypeRef (TypeMark).Static := Static;
   end SetTypeStatic;

   --------------------------------------------------------------------------------

   procedure SetTypeWellformed (TypeMark   : in Dictionary.Symbol;
                                Wellformed : in Boolean) is
   begin
      GetTypeRef (TypeMark).Wellformed := Wellformed;
   end SetTypeWellformed;

   --------------------------------------------------------------------------------

   procedure SetTypeLower (TypeMark : in Dictionary.Symbol;
                           Lower    : in LexTokenManager.Lex_String) is
   begin
      GetTypeRef (TypeMark).Lower := Lower;
   end SetTypeLower;

   --------------------------------------------------------------------------------

   procedure SetTypeUpper (TypeMark : in Dictionary.Symbol;
                           Upper    : in LexTokenManager.Lex_String) is
   begin
      GetTypeRef (TypeMark).Upper := Upper;
   end SetTypeUpper;

   --------------------------------------------------------------------------------

   procedure SetTypeModulus (TypeMark : in Dictionary.Symbol;
                             Modulus  : in LexTokenManager.Lex_String) is
   begin
      GetTypeRef (TypeMark).Modulus := Modulus;
   end SetTypeModulus;

   --------------------------------------------------------------------------------

   procedure SetTypeErrorBound (TypeMark   : in Dictionary.Symbol;
                                ErrorBound : in LexTokenManager.Lex_String) is
   begin
      GetTypeRef (TypeMark).ErrorBound := ErrorBound;
   end SetTypeErrorBound;

   --------------------------------------------------------------------------------

   procedure SetTypeParent (TheSubtype, Parent : in Dictionary.Symbol) is
   begin
      GetTypeRef (TheSubtype).Parent := Parent;
   end SetTypeParent;

   --------------------------------------------------------------------------------

   procedure SetTypeBaseType (TypeMark, BaseType : in Dictionary.Symbol) is
   begin
      GetTypeRef (TypeMark).BaseType := BaseType;
   end SetTypeBaseType;

   --------------------------------------------------------------------------------

   procedure SetTypeFirstEnumerationLiteral (EnumerationType, EnumerationLiteral : in Dictionary.Symbol) is
   begin
      GetTypeRef (EnumerationType).Head := EnumerationLiteral;
   end SetTypeFirstEnumerationLiteral;

   --------------------------------------------------------------------------------

   procedure SetTypeLastEnumerationLiteral (EnumerationType, EnumerationLiteral : in Dictionary.Symbol) is
   begin
      GetTypeRef (EnumerationType).Tail := EnumerationLiteral;
   end SetTypeLastEnumerationLiteral;

   --------------------------------------------------------------------------------

   procedure SetTypeFirstArrayIndex (ArrayType, ArrayIndex : in Dictionary.Symbol) is
   begin
      GetTypeRef (ArrayType).Head := ArrayIndex;
   end SetTypeFirstArrayIndex;

   --------------------------------------------------------------------------------

   procedure SetTypeLastArrayIndex (ArrayType, ArrayIndex : in Dictionary.Symbol) is
   begin
      GetTypeRef (ArrayType).Tail := ArrayIndex;
   end SetTypeLastArrayIndex;

   --------------------------------------------------------------------------------

   procedure SetTypeArrayComponent (ArrayType, ComponentType : in Dictionary.Symbol) is
   begin
      GetTypeRef (ArrayType).ArrayComponent := ComponentType;
   end SetTypeArrayComponent;

   --------------------------------------------------------------------------------

   procedure SetTypeFirstRecordComponent (RecordType, Component : in Dictionary.Symbol) is
   begin
      GetTypeRef (RecordType).Head := Component;
   end SetTypeFirstRecordComponent;

   --------------------------------------------------------------------------------

   procedure SetTypeLastRecordComponent (RecordType, Component : in Dictionary.Symbol) is
   begin
      GetTypeRef (RecordType).Tail := Component;
   end SetTypeLastRecordComponent;

   --------------------------------------------------------------------------------

   procedure SetTypeAnnouncement (TypeMark, Announcement : in Dictionary.Symbol) is
   begin
      GetTypeRef (TypeMark).Announcement := Announcement;
   end SetTypeAnnouncement;

   --------------------------------------------------------------------------------

   procedure SetTypeDeclaration (TypeMark, Declaration : in Dictionary.Symbol) is
   begin
      GetTypeRef (TypeMark).Declaration := Declaration;
   end SetTypeDeclaration;

   --------------------------------------------------------------------------------

   procedure SetTypePrivateTypeDeclaration (PrivateType, Declaration : in Dictionary.Symbol) is
   begin
      GetTypeRef (PrivateType).PrivateTypeDeclaration := Declaration;
   end SetTypePrivateTypeDeclaration;

   --------------------------------------------------------------------------------

   procedure SetTypeAncilliaryFields (TheType, Declaration : in Dictionary.Symbol) is
   begin
      GetTypeRef (TheType).AncilliaryFields := Declaration;
   end SetTypeAncilliaryFields;

   --------------------------------------------------------------------------------

   function GetTypeName (TypeMark : Dictionary.Symbol) return LexTokenManager.Lex_String is
   begin
      return GetTypeRef (TypeMark).Name;
   end GetTypeName;

   --------------------------------------------------------------------------------

   function GetTypeDiscriminant (TypeMark : Dictionary.Symbol) return TypeDiscriminant is
   begin
      return GetTypeRef (TypeMark).Discriminant;
   end GetTypeDiscriminant;

   --------------------------------------------------------------------------------

   function GetTypePrivate (TypeMark : Dictionary.Symbol) return TriState is
   begin
      return GetTypeRef (TypeMark).IsPrivate;
   end GetTypePrivate;

   --------------------------------------------------------------------------------

   function GetTypeLimited (TypeMark : Dictionary.Symbol) return TriState is
   begin
      return GetTypeRef (TypeMark).IsLimited;
   end GetTypeLimited;

   --------------------------------------------------------------------------------

   function GetTypeLimitedPrivate (TypeMark : Dictionary.Symbol) return Boolean is
   begin
      return GetTypeRef (TypeMark).IsLimitedPrivate;
   end GetTypeLimitedPrivate;

   --------------------------------------------------------------------------------

   function GetTypeDerived (TypeMark : Dictionary.Symbol) return Boolean is
   begin
      return GetTypeRef (TypeMark).IsDerived;
   end GetTypeDerived;

   --------------------------------------------------------------------------------

   function GetTypeIsTagged (TypeMark : Dictionary.Symbol) return Boolean is
   begin
      return GetTypeRef (TypeMark).IsTagged;
   end GetTypeIsTagged;

   --------------------------------------------------------------------------------

   function GetTypeIsAbstract (TypeMark : Dictionary.Symbol) return Boolean is
   begin
      return GetTypeRef (TypeMark).IsAbstract;
   end GetTypeIsAbstract;

   --------------------------------------------------------------------------------

   function GetTypeIsOwnVarType (TypeMark : Dictionary.Symbol) return Boolean is
   begin
      return GetTypeRef (TypeMark).IsOwnVarType;
   end GetTypeIsOwnVarType;

   --------------------------------------------------------------------------------

   function GetTypeExtends (TypeMark : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetTypeRef (TypeMark).Extends;
   end GetTypeExtends;

   --------------------------------------------------------------------------------

   function GetTypeAccesses (TypeMark : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetTypeRef (TypeMark).Accesses;
   end GetTypeAccesses;

   --------------------------------------------------------------------------------

   function GetTypeEqualityDefined (TypeMark : Dictionary.Symbol) return Boolean is
   begin
      return GetTypeRef (TypeMark).EqualityDefined;
   end GetTypeEqualityDefined;

   --------------------------------------------------------------------------------

   function GetTypeContainsFloat (TypeMark : Dictionary.Symbol) return Boolean is
   begin
      return GetTypeRef (TypeMark).ContainsFloat;
   end GetTypeContainsFloat;

   --------------------------------------------------------------------------------

   function GetTypeConstrained (ArrayType : Dictionary.Symbol) return Boolean is
   begin
      return GetTypeRef (ArrayType).Constrained;
   end GetTypeConstrained;

   --------------------------------------------------------------------------------

   function GetTypeStatic (TypeMark : Dictionary.Symbol) return Boolean is
   begin
      return GetTypeRef (TypeMark).Static;
   end GetTypeStatic;

   --------------------------------------------------------------------------------

   function GetTypeWellformed (TypeMark : Dictionary.Symbol) return Boolean is
   begin
      return GetTypeRef (TypeMark).Wellformed;
   end GetTypeWellformed;

   --------------------------------------------------------------------------------

   function GetTypeParent (TypeMark : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetTypeRef (TypeMark).Parent;
   end GetTypeParent;

   --------------------------------------------------------------------------------

   function GetTypeBaseType (TypeMark : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetTypeRef (TypeMark).BaseType;
   end GetTypeBaseType;

   --------------------------------------------------------------------------------

   function GetTypeLower (TypeMark : Dictionary.Symbol) return LexTokenManager.Lex_String is
   begin
      return GetTypeRef (TypeMark).Lower;
   end GetTypeLower;

   --------------------------------------------------------------------------------

   function GetTypeUpper (TypeMark : Dictionary.Symbol) return LexTokenManager.Lex_String is
   begin
      return GetTypeRef (TypeMark).Upper;
   end GetTypeUpper;

   --------------------------------------------------------------------------------

   function GetTypeModulus (TypeMark : Dictionary.Symbol) return LexTokenManager.Lex_String is
   begin
      return GetTypeRef (TypeMark).Modulus;
   end GetTypeModulus;

   --------------------------------------------------------------------------------

   function GetTypeErrorBound (TypeMark : Dictionary.Symbol) return LexTokenManager.Lex_String is
   begin
      return GetTypeRef (TypeMark).ErrorBound;
   end GetTypeErrorBound;

   --------------------------------------------------------------------------------

   function GetTypeFirstEnumerationLiteral (EnumerationType : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetTypeRef (EnumerationType).Head;
   end GetTypeFirstEnumerationLiteral;

   --------------------------------------------------------------------------------

   function GetTypeLastEnumerationLiteral (EnumerationType : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetTypeRef (EnumerationType).Tail;
   end GetTypeLastEnumerationLiteral;

   --------------------------------------------------------------------------------

   function GetTypeFirstArrayIndex (ArrayType : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetTypeRef (ArrayType).Head;
   end GetTypeFirstArrayIndex;

   --------------------------------------------------------------------------------

   function GetTypeLastArrayIndex (ArrayType : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetTypeRef (ArrayType).Tail;
   end GetTypeLastArrayIndex;

   --------------------------------------------------------------------------------

   function GetTypeArrayComponent (ArrayType : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetTypeRef (ArrayType).ArrayComponent;
   end GetTypeArrayComponent;

   --------------------------------------------------------------------------------

   function GetTypeFirstRecordComponent (RecordType : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetTypeRef (RecordType).Head;
   end GetTypeFirstRecordComponent;

   --------------------------------------------------------------------------------

   function GetTypeLastRecordComponent (RecordType : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetTypeRef (RecordType).Tail;
   end GetTypeLastRecordComponent;

   --------------------------------------------------------------------------------

   function GetTypeAnnouncement (TypeMark : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetTypeRef (TypeMark).Announcement;
   end GetTypeAnnouncement;

   --------------------------------------------------------------------------------

   function GetTypeDeclaration (TypeMark : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetTypeRef (TypeMark).Declaration;
   end GetTypeDeclaration;

   --------------------------------------------------------------------------------

   function GetTypePrivateTypeDeclaration (TypeMark : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetTypeRef (TypeMark).PrivateTypeDeclaration;
   end GetTypePrivateTypeDeclaration;

   --------------------------------------------------------------------------------

   function GetTypeAncilliaryFields (TypeMark : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetTypeRef (TypeMark).AncilliaryFields;
   end GetTypeAncilliaryFields;

   --------------------------------------------------------------------------------

   procedure CreateEnumerationLiteral
     (Name               : in     LexTokenManager.Lex_String;
      Position           : in     LexTokenManager.Lex_String;
      EnumerationType    : in     Dictionary.Symbol;
      Comp_Unit          : in     ContextManager.UnitDescriptors;
      Loc                : in     LexTokenManager.Token_Position;
      EnumerationLiteral :    out Dictionary.Symbol) is
      function EnumerationLiteralRefToRefType is new Unchecked_Conversion (EnumerationLiteralRef, RefType);
      NewLiteral : EnumerationLiteralRef;
   begin
      NewLiteral :=
        new EnumerationLiteralInfo'(Name     => Name,
                                    Position => Position,
                                    TheType  => EnumerationType,
                                    Next     => Dictionary.NullSymbol);
      AddSymbol
        (Discriminant => EnumerationLiteralSymbol,
         Ref          => EnumerationLiteralRefToRefType (NewLiteral),
         Comp_Unit    => Comp_Unit,
         Loc          => Loc,
         Item         => EnumerationLiteral);
   exception
      when Storage_Error =>
         SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Symbol_Table_Overflow_Dynamic,
                                   Msg     => "");
   end CreateEnumerationLiteral;

   --------------------------------------------------------------------------------

   procedure SetNextEnumerationLiteral (Current, Next : in Dictionary.Symbol) is
   begin
      GetEnumerationLiteralRef (Current).Next := Next;
   end SetNextEnumerationLiteral;

   --------------------------------------------------------------------------------

   function GetEnumerationLiteralName (Literal : Dictionary.Symbol) return LexTokenManager.Lex_String is
   begin
      return GetEnumerationLiteralRef (Literal).Name;
   end GetEnumerationLiteralName;

   --------------------------------------------------------------------------------

   function GetEnumerationLiteralPosition (Literal : Dictionary.Symbol) return LexTokenManager.Lex_String is
   begin
      return GetEnumerationLiteralRef (Literal).Position;
   end GetEnumerationLiteralPosition;

   --------------------------------------------------------------------------------

   function GetEnumerationLiteralType (Literal : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetEnumerationLiteralRef (Literal).TheType;
   end GetEnumerationLiteralType;

   --------------------------------------------------------------------------------

   function GetNextEnumerationLiteral (Literal : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetEnumerationLiteralRef (Literal).Next;
   end GetNextEnumerationLiteral;

   --------------------------------------------------------------------------------

   procedure CreateArrayIndex
     (IndexType  : in     Dictionary.Symbol;
      Comp_Unit  : in     ContextManager.UnitDescriptors;
      Loc        : in     LexTokenManager.Token_Position;
      ArrayIndex :    out Dictionary.Symbol) is
      function ArrayIndexRefToRefType is new Unchecked_Conversion (ArrayIndexRef, RefType);
      NewArrayIndex : ArrayIndexRef;
   begin
      NewArrayIndex := new ArrayIndexInfo'(IndexType => IndexType,
                                           Next      => Dictionary.NullSymbol);
      AddSymbol
        (Discriminant => ArrayIndexSymbol,
         Ref          => ArrayIndexRefToRefType (NewArrayIndex),
         Comp_Unit    => Comp_Unit,
         Loc          => Loc,
         Item         => ArrayIndex);
   exception
      when Storage_Error =>
         SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Symbol_Table_Overflow_Dynamic,
                                   Msg     => "");
   end CreateArrayIndex;

   procedure SetNextArrayIndex (Current, Next : in Dictionary.Symbol) is
   begin
      GetArrayIndexRef (Current).Next := Next;
   end SetNextArrayIndex;

   function GetArrayIndexType (ArrayIndex : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetArrayIndexRef (ArrayIndex).IndexType;
   end GetArrayIndexType;

   function GetNextArrayIndex (ArrayIndex : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetArrayIndexRef (ArrayIndex).Next;
   end GetNextArrayIndex;

   procedure CreateRecordComponent
     (Name           : in     LexTokenManager.Lex_String;
      RecordType     : in     Dictionary.Symbol;
      ComponentType  : in     Dictionary.Symbol;
      InheritedField : in     Boolean;
      Comp_Unit      : in     ContextManager.UnitDescriptors;
      Loc            : in     LexTokenManager.Token_Position;
      Component      :    out Dictionary.Symbol) is
      function RecordComponentRefToRefType is new Unchecked_Conversion (RecordComponentRef, RefType);
      NewComponent : RecordComponentRef;
   begin
      NewComponent :=
        new RecordComponentInfo'
        (Name           => Name,
         RecordType     => RecordType,
         ComponentType  => ComponentType,
         InheritedField => InheritedField,
         Next           => Dictionary.NullSymbol);
      AddSymbol
        (Discriminant => RecordComponentSymbol,
         Ref          => RecordComponentRefToRefType (NewComponent),
         Comp_Unit    => Comp_Unit,
         Loc          => Loc,
         Item         => Component);
   exception
      when Storage_Error =>
         SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Symbol_Table_Overflow_Dynamic,
                                   Msg     => "");
   end CreateRecordComponent;

   procedure SetNextRecordComponent (Current, Next : in Dictionary.Symbol) is
   begin
      GetRecordComponentRef (Current).Next := Next;
   end SetNextRecordComponent;

   function GetRecordComponentName (Component : Dictionary.Symbol) return LexTokenManager.Lex_String is
   begin
      return GetRecordComponentRef (Component).Name;
   end GetRecordComponentName;

   function GetRecordComponentRecordType (Component : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetRecordComponentRef (Component).RecordType;
   end GetRecordComponentRecordType;

   function GetRecordComponentType (Component : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetRecordComponentRef (Component).ComponentType;
   end GetRecordComponentType;

   function GetRecordComponentInheritedField (Component : Dictionary.Symbol) return Boolean is
   begin
      return GetRecordComponentRef (Component).InheritedField;
   end GetRecordComponentInheritedField;

   function GetNextRecordComponent (Component : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetRecordComponentRef (Component).Next;
   end GetNextRecordComponent;

   procedure CreateSubcomponent
     (Object       : in     Dictionary.Symbol;
      Component    : in     Dictionary.Symbol;
      Comp_Unit    : in     ContextManager.UnitDescriptors;
      Loc          : in     LexTokenManager.Token_Position;
      Subcomponent :    out Dictionary.Symbol) is
      function SubcomponentRefToRefType is new Unchecked_Conversion (SubcomponentRef, RefType);
      NewSubcomponent : SubcomponentRef;
   begin
      NewSubcomponent :=
        new SubcomponentInfo'
        (Object        => Object,
         Component     => Component,
         Next          => Dictionary.NullSymbol,
         Subcomponents => Dictionary.NullSymbol,
         MarkedValid   => True);
      AddSymbol
        (Discriminant => SubcomponentSymbol,
         Ref          => SubcomponentRefToRefType (NewSubcomponent),
         Comp_Unit    => Comp_Unit,
         Loc          => Loc,
         Item         => Subcomponent);
   exception
      when Storage_Error =>
         SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Symbol_Table_Overflow_Dynamic,
                                   Msg     => "");
   end CreateSubcomponent;

   procedure SetNextSubcomponent (Current, Next : in Dictionary.Symbol) is
   begin
      GetSubcomponentRef (Current).Next := Next;
   end SetNextSubcomponent;

   procedure SetSubcomponentSubcomponents (Parent, Sibling : in Dictionary.Symbol) is
   begin
      GetSubcomponentRef (Parent).Subcomponents := Sibling;
   end SetSubcomponentSubcomponents;

   function GetSubcomponentComponent (Subcomponent : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetSubcomponentRef (Subcomponent).Component;
   end GetSubcomponentComponent;

   function GetSubcomponentObject (Subcomponent : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetSubcomponentRef (Subcomponent).Object;
   end GetSubcomponentObject;

   function GetNextSubcomponent (Subcomponent : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetSubcomponentRef (Subcomponent).Next;
   end GetNextSubcomponent;

   function GetSubcomponentSubcomponents (Subcomponent : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetSubcomponentRef (Subcomponent).Subcomponents;
   end GetSubcomponentSubcomponents;

   procedure CreateRulePolicy
     (Scope         : in     Dictionary.Scopes;
      Value         : in     Dictionary.RulePolicies;
      Comp_Unit     : in     ContextManager.UnitDescriptors;
      Loc           : in     LexTokenManager.Token_Position;
      TheRulePolicy :    out Dictionary.Symbol) is
      function RulePolicyRefToRefType is new Unchecked_Conversion (RulePolicyRef, RefType);
      NewRulePolicy : RulePolicyRef;
   begin
      NewRulePolicy := new RulePolicyInfo'(Scope => Scope,
                                           Value => Value,
                                           Next  => Dictionary.NullSymbol);
      AddSymbol
        (Discriminant => RulePolicySymbol,
         Ref          => RulePolicyRefToRefType (NewRulePolicy),
         Comp_Unit    => Comp_Unit,
         Loc          => Loc,
         Item         => TheRulePolicy);
   exception
      when Storage_Error =>
         SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Symbol_Table_Overflow_Dynamic,
                                   Msg     => "");
   end CreateRulePolicy;

   procedure SetNextRulePolicy (TheRulePolicy : in Dictionary.Symbol;
                                Next          : in Dictionary.Symbol) is
   begin
      GetRulePolicyRef (TheRulePolicy).Next := Next;
   end SetNextRulePolicy;

   function GetNextRulePolicy (TheRulePolicy : in Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetRulePolicyRef (TheRulePolicy).Next;
   end GetNextRulePolicy;

   function GetRulePolicyScope (TheRulePolicy : in Dictionary.Symbol) return Dictionary.Scopes is
   begin
      return GetRulePolicyRef (TheRulePolicy).Scope;
   end GetRulePolicyScope;

   function GetRulePolicyValue (TheRulePolicy : in Dictionary.Symbol) return Dictionary.RulePolicies is
   begin
      return GetRulePolicyRef (TheRulePolicy).Value;
   end GetRulePolicyValue;

   procedure CreateConstant
     (Name        : in     LexTokenManager.Lex_String;
      TypeMark    : in     Dictionary.Symbol;
      Static      : in     Boolean;
      Comp_Unit   : in     ContextManager.UnitDescriptors;
      Loc         : in     LexTokenManager.Token_Position;
      TheConstant :    out Dictionary.Symbol) is
      function ConstantRefToRefType is new Unchecked_Conversion (ConstantRef, RefType);
      NewConstant : ConstantRef;
   begin
      NewConstant :=
        new ConstantInfo'
        (Name                        => Name,
         TypeMark                    => TypeMark,
         Value                       => LexTokenManager.Null_String,
         Static                      => Static,
         ExpIsWellFormed             => False,
         ExpNode                     => 0,
         Declaration                 => Dictionary.NullSymbol,
         DeferredConstantDeclaration => Dictionary.NullSymbol,
         AssociatedGenericParameter  => Dictionary.NullSymbol,
         FirstRulePolicy             => Dictionary.NullSymbol,
         LastRulePolicy              => Dictionary.NullSymbol);
      AddSymbol
        (Discriminant => ConstantSymbol,
         Ref          => ConstantRefToRefType (NewConstant),
         Comp_Unit    => Comp_Unit,
         Loc          => Loc,
         Item         => TheConstant);
   exception
      when Storage_Error =>
         SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Symbol_Table_Overflow_Dynamic,
                                   Msg     => "");
   end CreateConstant;

   procedure SetConstantValue (TheConstant : in Dictionary.Symbol;
                               Value       : in LexTokenManager.Lex_String) is
   begin
      GetConstantRef (TheConstant).Value := Value;
   end SetConstantValue;

   procedure SetConstantExpNode
     (TheConstant     : in Dictionary.Symbol;
      ExpIsWellFormed : in Boolean;
      ExpNode         : in ExaminerConstants.RefType) is
   begin
      GetConstantRef (TheConstant).ExpNode         := ExpNode;
      GetConstantRef (TheConstant).ExpIsWellFormed := ExpIsWellFormed;
   end SetConstantExpNode;

   procedure SetConstantStatic (TheConstant : in Dictionary.Symbol;
                                Static      : in Boolean) is
   begin
      GetConstantRef (TheConstant).Static := Static;
   end SetConstantStatic;

   procedure SetConstantDeclaration (TheConstant, Declaration : in Dictionary.Symbol) is
   begin
      GetConstantRef (TheConstant).Declaration := Declaration;
   end SetConstantDeclaration;

   procedure SetConstantDeferredConstantDeclaration (DeferredConstant, Declaration : in Dictionary.Symbol) is
   begin
      GetConstantRef (DeferredConstant).DeferredConstantDeclaration := Declaration;
   end SetConstantDeferredConstantDeclaration;

   procedure SetConstantAssociatedGenericParameter (TheConstant, TheGenericParameter : in Dictionary.Symbol) is
   begin
      GetConstantRef (TheConstant).AssociatedGenericParameter := TheGenericParameter;
   end SetConstantAssociatedGenericParameter;

   procedure SetConstantFirstRulePolicy (TheConstant : in Dictionary.Symbol;
                                         ThePolicy   : in Dictionary.Symbol) is
   begin
      GetConstantRef (TheConstant).FirstRulePolicy := ThePolicy;
   end SetConstantFirstRulePolicy;

   procedure SetConstantLastRulePolicy (TheConstant : in Dictionary.Symbol;
                                        ThePolicy   : in Dictionary.Symbol) is
   begin
      GetConstantRef (TheConstant).LastRulePolicy := ThePolicy;
   end SetConstantLastRulePolicy;

   function GetConstantName (TheConstant : Dictionary.Symbol) return LexTokenManager.Lex_String is
   begin
      return GetConstantRef (TheConstant).Name;
   end GetConstantName;

   function GetConstantType (TheConstant : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetConstantRef (TheConstant).TypeMark;
   end GetConstantType;

   function GetConstantValue (TheConstant : Dictionary.Symbol) return LexTokenManager.Lex_String is
   begin
      return GetConstantRef (TheConstant).Value;
   end GetConstantValue;

   function GetConstantFirstRulePolicy (TheConstant : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetConstantRef (TheConstant).FirstRulePolicy;
   end GetConstantFirstRulePolicy;

   function GetConstantLastRulePolicy (TheConstant : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetConstantRef (TheConstant).LastRulePolicy;
   end GetConstantLastRulePolicy;

   function GetConstantExpNode (TheConstant : Dictionary.Symbol) return ExaminerConstants.RefType is
   begin
      return GetConstantRef (TheConstant).ExpNode;
   end GetConstantExpNode;

   function ConstantExpIsWellFormed (TheConstant : Dictionary.Symbol) return Boolean is
   begin
      return GetConstantRef (TheConstant).ExpIsWellFormed;
   end ConstantExpIsWellFormed;

   function GetConstantStatic (TheConstant : Dictionary.Symbol) return Boolean is
   begin
      return GetConstantRef (TheConstant).Static;
   end GetConstantStatic;

   function GetConstantDeclaration (TheConstant : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetConstantRef (TheConstant).Declaration;
   end GetConstantDeclaration;

   function GetConstantDeferredConstantDeclaration (TheConstant : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetConstantRef (TheConstant).DeferredConstantDeclaration;
   end GetConstantDeferredConstantDeclaration;

   function GetConstantAssociatedGenericParameter (TheConstant : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetConstantRef (TheConstant).AssociatedGenericParameter;
   end GetConstantAssociatedGenericParameter;

   procedure CreateVariable
     (Name      : in     LexTokenManager.Lex_String;
      Comp_Unit : in     ContextManager.UnitDescriptors;
      Loc       : in     LexTokenManager.Token_Position;
      Variable  :    out Dictionary.Symbol) is
      function VariableRefToRefType is new Unchecked_Conversion (VariableRef, RefType);
      GlobalReferences : Symbols;
      NewVariable      : VariableRef;
   begin
      GlobalReferences := Symbols'(Dictionary.Abstractions => Dictionary.NullSymbol);
      NewVariable      :=
        new VariableInfo'
        (Name             => Name,
         TypeMark         => Dictionary.GetUnknownTypeMark,
         AbstractTypeMark => Dictionary.GetUnknownTypeMark,
         Initialized      => False,
         HasAddressClause => False,
         HasPragmaImport  => False,
         IsAliased        => False,
         MarkedValid      => True,
         ExpNode          => 0,
         Declaration      => Dictionary.NullSymbol,
         GlobalReferences => GlobalReferences,
         OwnVariable      => Dictionary.NullSymbol,
         OwnTask          => Dictionary.NullSymbol,
         VirtualElement   => Dictionary.NullSymbol,
         Constituent      => Dictionary.NullSymbol,
         Subcomponents    => Dictionary.NullSymbol);
      AddSymbol
        (Discriminant => VariableSymbol,
         Ref          => VariableRefToRefType (NewVariable),
         Comp_Unit    => Comp_Unit,
         Loc          => Loc,
         Item         => Variable);
   exception
      when Storage_Error =>
         SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Symbol_Table_Overflow_Dynamic,
                                   Msg     => "");
   end CreateVariable;

   procedure SetVariableType (Variable, TypeMark : in Dictionary.Symbol) is
   begin
      GetVariableRef (Variable).TypeMark := TypeMark;
   end SetVariableType;

   procedure SetVariableAbstractType (Variable, AbstractTypeMark : in Dictionary.Symbol) is
   begin
      GetVariableRef (Variable).AbstractTypeMark := AbstractTypeMark;
   end SetVariableAbstractType;

   procedure SetVariableInitialized (Variable : in Dictionary.Symbol) is
   begin
      GetVariableRef (Variable).Initialized := True;
   end SetVariableInitialized;

   procedure SetVariableHasAddressClause (Variable : in Dictionary.Symbol) is
   begin
      GetVariableRef (Variable).HasAddressClause := True;
   end SetVariableHasAddressClause;

   procedure SetVariableHasPragmaImport (Variable : in Dictionary.Symbol) is
   begin
      GetVariableRef (Variable).HasPragmaImport := True;
   end SetVariableHasPragmaImport;

   procedure SetTypeSizeAttribute (TypeMark : in Dictionary.Symbol;
                                   SizeVal  : in LexTokenManager.Lex_String) is
   begin
      GetTypeRef (TypeMark).SizeAttribute := SizeVal;
   end SetTypeSizeAttribute;

   procedure SetVariableIsAliased (Variable : in Dictionary.Symbol) is
   begin
      GetVariableRef (Variable).IsAliased := True;
   end SetVariableIsAliased;

   procedure SetVariableMarkedValid (Variable : in Dictionary.Symbol;
                                     Val      : in Boolean) is
   begin
      GetVariableRef (Variable).MarkedValid := Val;
   end SetVariableMarkedValid;

   procedure SetSubcomponentMarkedValid (Subcomponent : in Dictionary.Symbol;
                                         Val          : in Boolean) is
   begin
      GetSubcomponentRef (Subcomponent).MarkedValid := Val;
   end SetSubcomponentMarkedValid;

   procedure SetTypeAtomic (TypeMark : in Dictionary.Symbol) is
   begin
      GetTypeRef (TypeMark).IsAtomic := True;
   end SetTypeAtomic;

   procedure SetVariableDeclaration (Variable, Declaration : in Dictionary.Symbol) is
   begin
      GetVariableRef (Variable).Declaration := Declaration;
   end SetVariableDeclaration;

   procedure SetVariableExpNode (Variable : in Dictionary.Symbol;
                                 ExpNode  : in ExaminerConstants.RefType) is
   begin
      GetVariableRef (Variable).ExpNode := ExpNode;
   end SetVariableExpNode;

   procedure SetVariableOwnTask (Variable : in Dictionary.Symbol;
                                 OwnTask  : in Dictionary.Symbol) is
   begin
      GetVariableRef (Variable).OwnTask := OwnTask;
   end SetVariableOwnTask;

   procedure SetVariableVirtualElement (Variable       : in Dictionary.Symbol;
                                        VirtualElement : in Dictionary.Symbol) is
   begin
      GetVariableRef (Variable).VirtualElement := VirtualElement;
   end SetVariableVirtualElement;

   procedure SetVariableGlobalReferences
     (Abstraction : in Dictionary.Abstractions;
      Variable    : in Dictionary.Symbol;
      Reference   : in Dictionary.Symbol) is
   begin
      GetVariableRef (Variable).GlobalReferences (Abstraction) := Reference;
   end SetVariableGlobalReferences;

   procedure SetVariableOwnVariable (Variable, OwnVariable : in Dictionary.Symbol) is
   begin
      GetVariableRef (Variable).OwnVariable := OwnVariable;
   end SetVariableOwnVariable;

   procedure SetVariableConstituent (Variable, Constituent : in Dictionary.Symbol) is
   begin
      GetVariableRef (Variable).Constituent := Constituent;
   end SetVariableConstituent;

   procedure SetVariableSubComponents (Variable, Subcomponents : in Dictionary.Symbol) is
   begin
      GetVariableRef (Variable).Subcomponents := Subcomponents;
   end SetVariableSubComponents;

   function GetVariableName (Variable : Dictionary.Symbol) return LexTokenManager.Lex_String is
   begin
      return GetVariableRef (Variable).Name;
   end GetVariableName;

   function GetVariableType (Variable : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetVariableRef (Variable).TypeMark;
   end GetVariableType;

   function GetVariableAbstractType (Variable : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetVariableRef (Variable).AbstractTypeMark;
   end GetVariableAbstractType;

   function GetVariableInitialized (Variable : Dictionary.Symbol) return Boolean is
   begin
      return GetVariableRef (Variable).Initialized;
   end GetVariableInitialized;

   function GetVariableHasAddressClause (Variable : Dictionary.Symbol) return Boolean is
   begin
      return GetVariableRef (Variable).HasAddressClause;
   end GetVariableHasAddressClause;

   function GetVariableHasPragmaImport (Variable : Dictionary.Symbol) return Boolean is
   begin
      return GetVariableRef (Variable).HasPragmaImport;
   end GetVariableHasPragmaImport;

   function GetTypeSizeAttribute (TypeMark : Dictionary.Symbol) return LexTokenManager.Lex_String is
   begin
      return GetTypeRef (TypeMark).SizeAttribute;
   end GetTypeSizeAttribute;

   function GetVariableIsAliased (Variable : Dictionary.Symbol) return Boolean is
   begin
      return GetVariableRef (Variable).IsAliased;
   end GetVariableIsAliased;

   function GetVariableMarkedValid (Variable : Dictionary.Symbol) return Boolean is
   begin
      return GetVariableRef (Variable).MarkedValid;
   end GetVariableMarkedValid;

   function GetSubcomponentMarkedValid (Subcomponent : Dictionary.Symbol) return Boolean is
   begin
      return GetSubcomponentRef (Subcomponent).MarkedValid;
   end GetSubcomponentMarkedValid;

   function TypeIsAtomic (TypeMark : Dictionary.Symbol) return Boolean is
   begin
      return
        -- If this is a subtype then check the base type
        (IsSubtype (TypeMark) and then GetTypeRef (Dictionary.GetRootType (TypeMark)).IsAtomic)
        or else

        -- Otherwise just check the type itself
        GetTypeRef (TypeMark).IsAtomic;
   end TypeIsAtomic;

   function IsAtomic (TheSymbol : Dictionary.Symbol) return Boolean is
   begin
      return (IsVariable (TheSymbol) and then (TypeIsAtomic (GetType (TheSymbol))))
        or else (IsTypeMark (TheSymbol) and then TypeIsAtomic (TheSymbol));
   end IsAtomic;

   function IsPragmaAtomic (TheSymbol : Dictionary.Symbol) return Boolean is
   begin
      return IsAtomic (TheSymbol)
        and then not TypeIsProtected (Dictionary.GetType (TheSymbol))
        and then not IsPredefinedSuspensionObjectType (Dictionary.GetType (TheSymbol));
   end IsPragmaAtomic;

   function GetVariableDeclaration (Variable : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetVariableRef (Variable).Declaration;
   end GetVariableDeclaration;

   function GetVariableExpNode (Variable : Dictionary.Symbol) return ExaminerConstants.RefType is
   begin
      return GetVariableRef (Variable).ExpNode;
   end GetVariableExpNode;

   function GetVariableGlobalReferences
     (Abstraction : Dictionary.Abstractions;
      Variable    : Dictionary.Symbol)
     return        Dictionary.Symbol
   is
   begin
      return GetVariableRef (Variable).GlobalReferences (Abstraction);
   end GetVariableGlobalReferences;

   function GetVariableOwnVariable (Variable : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetVariableRef (Variable).OwnVariable;
   end GetVariableOwnVariable;

   function GetVariableOwnTask (Variable : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetVariableRef (Variable).OwnTask;
   end GetVariableOwnTask;

   function GetVariableConstituent (Variable : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetVariableRef (Variable).Constituent;
   end GetVariableConstituent;

   function GetVariableSubComponents (Variable : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetVariableRef (Variable).Subcomponents;
   end GetVariableSubComponents;

   procedure CreateQuantifiedVariable
     (Name      : in     LexTokenManager.Lex_String;
      TypeMark  : in     Dictionary.Symbol;
      Region    : in     Dictionary.Symbol;
      Comp_Unit : in     ContextManager.UnitDescriptors;
      Loc       : in     LexTokenManager.Token_Position;
      Variable  :    out Dictionary.Symbol) is
      QuantifiedVariable : QuantifiedVariableRef;
      function QuantifiedVariableRefToRefType is new Unchecked_Conversion (QuantifiedVariableRef, RefType);
   begin
      QuantifiedVariable := new QuantifiedVariableInfo'(Name     => Name,
                                                        TypeMark => TypeMark,
                                                        Region   => Region);
      AddSymbol
        (Discriminant => QuantifiedVariableSymbol,
         Ref          => QuantifiedVariableRefToRefType (QuantifiedVariable),
         Comp_Unit    => Comp_Unit,
         Loc          => Loc,
         Item         => Variable);
   exception
      when Storage_Error =>
         SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Symbol_Table_Overflow_Dynamic,
                                   Msg     => "");
   end CreateQuantifiedVariable;

   function GetQuantifiedVariableName (Variable : Dictionary.Symbol) return LexTokenManager.Lex_String is
   begin
      return GetQuantifiedVariableRef (Variable).Name;
   end GetQuantifiedVariableName;

   function GetQuantifiedVariableType (Variable : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetQuantifiedVariableRef (Variable).TypeMark;
   end GetQuantifiedVariableType;

   function GetQuantifiedVariableRegion (Variable : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetQuantifiedVariableRef (Variable).Region;
   end GetQuantifiedVariableRegion;

   procedure CreateImplicitReturnVariable
     (Name        : in     LexTokenManager.Lex_String;
      TheFunction : in     Dictionary.Symbol;
      Comp_Unit   : in     ContextManager.UnitDescriptors;
      Loc         : in     LexTokenManager.Token_Position;
      Variable    :    out Dictionary.Symbol) is
      ImplicitReturnVariable : ImplicitReturnVariableRef;
      function ImplicitReturnVariableRefToRefType is new Unchecked_Conversion (ImplicitReturnVariableRef, RefType);
   begin
      ImplicitReturnVariable := new ImplicitReturnVariableInfo'(Name        => Name,
                                                                TheFunction => TheFunction);
      AddSymbol
        (Discriminant => ImplicitReturnVariableSymbol,
         Ref          => ImplicitReturnVariableRefToRefType (ImplicitReturnVariable),
         Comp_Unit    => Comp_Unit,
         Loc          => Loc,
         Item         => Variable);
   exception
      when Storage_Error =>
         SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Symbol_Table_Overflow_Dynamic,
                                   Msg     => "");
   end CreateImplicitReturnVariable;

   function GetImplicitReturnVariableName (Variable : Dictionary.Symbol) return LexTokenManager.Lex_String is
   begin
      return GetImplicitReturnVariableRef (Variable).Name;
   end GetImplicitReturnVariableName;

   function GetImplicitReturnVariableFunction (Variable : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetImplicitReturnVariableRef (Variable).TheFunction;
   end GetImplicitReturnVariableFunction;

   procedure CreateOwnTask
     (Variable  : in     Dictionary.Symbol;
      Owner     : in     Dictionary.Symbol;
      Comp_Unit : in     ContextManager.UnitDescriptors;
      Loc       : in     LexTokenManager.Token_Position;
      OwnTask   :    out Dictionary.Symbol) is
      function OwnTaskRefToRefType is new Unchecked_Conversion (OwnTaskRef, RefType);
      NewOwnTask : OwnTaskRef;
   begin
      NewOwnTask := new OwnTaskInfo'(Variable => Variable,
                                     Owner    => Owner,
                                     Next     => Dictionary.NullSymbol);
      AddSymbol
        (Discriminant => OwnTaskSymbol,
         Ref          => OwnTaskRefToRefType (NewOwnTask),
         Comp_Unit    => Comp_Unit,
         Loc          => Loc,
         Item         => OwnTask);
   exception
      when Storage_Error =>
         SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Symbol_Table_Overflow_Dynamic,
                                   Msg     => "");
   end CreateOwnTask;

   procedure SetNextOwnTask (Current, Next : in Dictionary.Symbol) is
   begin
      GetOwnTaskRef (Current).Next := Next;
   end SetNextOwnTask;

   function GetNextOwnTask (Current : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetOwnTaskRef (Current).Next;
   end GetNextOwnTask;

   function GetOwnTaskVariable (OwnTask : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetOwnTaskRef (OwnTask).Variable;
   end GetOwnTaskVariable;

   function GetOwnTaskOwner (OwnTask : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetOwnTaskRef (OwnTask).Owner;
   end GetOwnTaskOwner;

   procedure CreateOwnVariable
     (Variable    : in     Dictionary.Symbol;
      Owner       : in     Dictionary.Symbol;
      Comp_Unit   : in     ContextManager.UnitDescriptors;
      Loc         : in     LexTokenManager.Token_Position;
      OwnVariable :    out Dictionary.Symbol) is
      function OwnVariableRefToRefType is new Unchecked_Conversion (OwnVariableRef, RefType);
      NewOwnVariable : OwnVariableRef;
   begin
      -- note use of InvalidMode in this aggregate.  The reasosn is that if we
      -- refine an own variable on to an embdedded package own variable then the
      -- own variable gets created by a call here when the refinement is processed.
      -- If there is an error when the embedded package own variable is eventually
      -- processed then the mode (set earlier to invalid) won't get overwitten so
      -- we are left with a nice detectable error case rather than an apaprently
      -- well ofrmed case of DefaultMode
      NewOwnVariable :=
        new OwnVariableInfo'
        (Variable                => Variable,
         Owner                   => Owner,
         Announced               => False,
         Typed                   => False,
         Initialized             => False,
         Mode                    => Dictionary.InvalidMode,
         IsProtected             => False,
         IsInterruptStream       => False,
         InterruptStreamMappings => Dictionary.NullSymbol,
         UnprotectedReference    => Dictionary.NullSymbol,
         SuspendsReference       => Dictionary.NullSymbol,
         ImplicitInStream        => Dictionary.NullSymbol,
         Priority                => LexTokenManager.Null_String,
         Integrity               => LexTokenManager.Null_String,
         Suspendable             => False,
         Interruptable           => False,
         Constituents            => Dictionary.NullSymbol,
         Next                    => Dictionary.NullSymbol);
      AddSymbol
        (Discriminant => OwnVariableSymbol,
         Ref          => OwnVariableRefToRefType (NewOwnVariable),
         Comp_Unit    => Comp_Unit,
         Loc          => Loc,
         Item         => OwnVariable);
   exception
      when Storage_Error =>
         SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Symbol_Table_Overflow_Dynamic,
                                   Msg     => "");
   end CreateOwnVariable;

   procedure SetOwnVariableAnnounced (OwnVariable : in Dictionary.Symbol) is
   begin
      GetOwnVariableRef (OwnVariable).Announced := True;
   end SetOwnVariableAnnounced;

   procedure SetOwnVariableTyped (OwnVariable : in Dictionary.Symbol) is
   begin
      GetOwnVariableRef (OwnVariable).Typed := True;
   end SetOwnVariableTyped;

   procedure SetOwnVariableInitialized (OwnVariable : in Dictionary.Symbol) is
   begin
      GetOwnVariableRef (OwnVariable).Initialized := True;
   end SetOwnVariableInitialized;

   procedure SetOwnVariableConstituents (Subject, Constituent : in Dictionary.Symbol) is
   begin
      GetOwnVariableRef (Subject).Constituents := Constituent;
   end SetOwnVariableConstituents;

   procedure SetOwnVariableMode (OwnVariable : in Dictionary.Symbol;
                                 Mode        : in Dictionary.Modes) is
   begin
      GetOwnVariableRef (OwnVariable).Mode := Mode;
   end SetOwnVariableMode;

   procedure SetOwnVariableProtected (OwnVariable : in Dictionary.Symbol;
                                      IsProtected : in Boolean) is
   begin
      GetOwnVariableRef (OwnVariable).IsProtected := IsProtected;
   end SetOwnVariableProtected;

   procedure SetNextOwnVariable (Current, Next : in Dictionary.Symbol) is
   begin
      GetOwnVariableRef (Current).Next := Next;
   end SetNextOwnVariable;

   function GetOwnVariableVariable (OwnVariable : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetOwnVariableRef (OwnVariable).Variable;
   end GetOwnVariableVariable;

   function GetOwnVariableOwner (OwnVariable : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetOwnVariableRef (OwnVariable).Owner;
   end GetOwnVariableOwner;

   function GetOwnVariableAnnounced (OwnVariable : Dictionary.Symbol) return Boolean is
   begin
      return GetOwnVariableRef (OwnVariable).Announced;
   end GetOwnVariableAnnounced;

   function GetOwnVariableTyped (OwnVariable : Dictionary.Symbol) return Boolean is
   begin
      return GetOwnVariableRef (OwnVariable).Typed;
   end GetOwnVariableTyped;

   function GetOwnVariableInitialized (OwnVariable : Dictionary.Symbol) return Boolean is
   begin
      return GetOwnVariableRef (OwnVariable).Initialized;
   end GetOwnVariableInitialized;

   function GetOwnVariableConstituents (Subject : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetOwnVariableRef (Subject).Constituents;
   end GetOwnVariableConstituents;

   function GetOwnVariableMode (OwnVariable : Dictionary.Symbol) return Dictionary.Modes is
   begin
      return GetOwnVariableRef (OwnVariable).Mode;
   end GetOwnVariableMode;

   function GetOwnVariableProtected (OwnVariable : Dictionary.Symbol) return Boolean is
   begin
      return GetOwnVariableRef (OwnVariable).IsProtected;
   end GetOwnVariableProtected;

   function GetOwnVariableImplicitInStream (OwnVariable : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetOwnVariableRef (OwnVariable).ImplicitInStream;
   end GetOwnVariableImplicitInStream;

   function GetImplicitInStreamOwnVariable (TheStream : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetImplicitInStreamRef (TheStream).OwnVariable;
   end GetImplicitInStreamOwnVariable;

   function GetVariableVirtualElement (Variable : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetVariableRef (Variable).VirtualElement;
   end GetVariableVirtualElement;

   function GetNextOwnVariable (OwnVariable : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetOwnVariableRef (OwnVariable).Next;
   end GetNextOwnVariable;

   procedure CreateConstituent
     (Subject     : in     Dictionary.Symbol;
      Variable    : in     Dictionary.Symbol;
      Comp_Unit   : in     ContextManager.UnitDescriptors;
      Loc         : in     LexTokenManager.Token_Position;
      Constituent :    out Dictionary.Symbol) is
      function ConstituentRefToRefType is new Unchecked_Conversion (ConstituentRef, RefType);
      NewConstituent : ConstituentRef;
   begin
      NewConstituent :=
        new ConstituentInfo'
        (Subject  => Subject,
         Variable => Variable,
         Mode     => Dictionary.DefaultMode,
         Next     => Dictionary.NullSymbol);
      AddSymbol
        (Discriminant => ConstituentSymbol,
         Ref          => ConstituentRefToRefType (NewConstituent),
         Comp_Unit    => Comp_Unit,
         Loc          => Loc,
         Item         => Constituent);
   exception
      when Storage_Error =>
         SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Symbol_Table_Overflow_Dynamic,
                                   Msg     => "");
   end CreateConstituent;

   procedure SetConstituentMode (Constituent : in Dictionary.Symbol;
                                 Mode        : in Dictionary.Modes) is
   begin
      GetConstituentRef (Constituent).Mode := Mode;
   end SetConstituentMode;

   procedure SetNextConstituent (Current, Next : in Dictionary.Symbol) is
   begin
      GetConstituentRef (Current).Next := Next;
   end SetNextConstituent;

   function GetConstituentSubject (Constituent : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetConstituentRef (Constituent).Subject;
   end GetConstituentSubject;

   function GetConstituentVariable (Constituent : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetConstituentRef (Constituent).Variable;
   end GetConstituentVariable;

   function GetConstituentMode (Constituent : Dictionary.Symbol) return Dictionary.Modes is
   begin
      return GetConstituentRef (Constituent).Mode;
   end GetConstituentMode;

   function GetNextConstituent (Constituent : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetConstituentRef (Constituent).Next;
   end GetNextConstituent;

   procedure CreatePackage
     (Name       : in     LexTokenManager.Lex_String;
      Comp_Unit  : in     ContextManager.UnitDescriptors;
      Loc        : in     LexTokenManager.Token_Position;
      ThePackage :    out Dictionary.Symbol) is
      Postcondition : Constraints;
      NewPackage    : PackageRef;
      function PackageRefToRefType is new Unchecked_Conversion (PackageRef, RefType);
   begin
      Postcondition := Constraints'(Dictionary.Abstractions => 0);
      NewPackage    :=
        new PackageInfo'
        (Name                    => Name,
         Postcondition           => Postcondition,
         PackageBody             => Dictionary.NullSymbol,
         PackageHasProperBody    => False,
         InheritClauses          => Dictionary.NullSymbol,
         OwnVariables            => Dictionary.NullSymbol,
         TaskList                => Dictionary.NullSymbol,
         FirstLoop               => Dictionary.NullSymbol,
         LastLoop                => Dictionary.NullSymbol,
         Specification           => Dictionary.NullSymbol,
         VisiblePart             => PartInfo'(WithClauses          => Dictionary.NullSymbol,
                                              UseTypeClauses       => Dictionary.NullSymbol,
                                              RenamingDeclarations => Dictionary.NullSymbol,
                                              FirstDeclaration     => Dictionary.NullSymbol,
                                              LastDeclaration      => Dictionary.NullSymbol),
         LocalPart               => PartInfo'(WithClauses          => Dictionary.NullSymbol,
                                              UseTypeClauses       => Dictionary.NullSymbol,
                                              RenamingDeclarations => Dictionary.NullSymbol,
                                              FirstDeclaration     => Dictionary.NullSymbol,
                                              LastDeclaration      => Dictionary.NullSymbol),
         PrivatePart             => PartInfo'(WithClauses          => Dictionary.NullSymbol,
                                              UseTypeClauses       => Dictionary.NullSymbol,
                                              RenamingDeclarations => Dictionary.NullSymbol,
                                              FirstDeclaration     => Dictionary.NullSymbol,
                                              LastDeclaration      => Dictionary.NullSymbol),
         ElaborateBodyFound      => False,
         Parent                  => Dictionary.NullSymbol,
         NextSibling             => Dictionary.NullSymbol,
         Extends                 => Dictionary.NullSymbol,
         DeclaresTaggedType      => False,
         PrivateChildren         => ChildInfo'(FirstChild => Dictionary.NullSymbol,
                                               LastChild  => Dictionary.NullSymbol),
         PublicChildren          => ChildInfo'(FirstChild => Dictionary.NullSymbol,
                                               LastChild  => Dictionary.NullSymbol),
         IsPrivate               => False,
         IsGeneric               => False,
         InstantiationOf         => Dictionary.NullSymbol,
         FirstGenericParameter   => Dictionary.NullSymbol,
         LastGenericParameter    => Dictionary.NullSymbol,
         FirstGenericAssociation => Dictionary.NullSymbol,
         LastGenericAssociation  => Dictionary.NullSymbol);
      AddSymbol
        (Discriminant => PackageSymbol,
         Ref          => PackageRefToRefType (NewPackage),
         Comp_Unit    => Comp_Unit,
         Loc          => Loc,
         Item         => ThePackage);
   exception
      when Storage_Error =>
         SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Symbol_Table_Overflow_Dynamic,
                                   Msg     => "");
   end CreatePackage;

   procedure SetPackagePostcondition
     (Abstraction   : in Dictionary.Abstractions;
      ThePackage    : in Dictionary.Symbol;
      Postcondition : in ExaminerConstants.RefType) is
   begin
      GetPackageRef (ThePackage).Postcondition (Abstraction) := Postcondition;
   end SetPackagePostcondition;

   procedure SetPackageBody (ThePackage, TheBody : in Dictionary.Symbol) is
   begin
      GetPackageRef (ThePackage).PackageBody := TheBody;
   end SetPackageBody;

   procedure SetPackageHasProperBody (ThePackage : in Dictionary.Symbol) is
   begin
      GetPackageRef (ThePackage).PackageHasProperBody := True;
   end SetPackageHasProperBody;

   procedure SetPackageElaborateBodyFound (ThePackage : in Dictionary.Symbol) is
   begin
      GetPackageRef (ThePackage).ElaborateBodyFound := True;
   end SetPackageElaborateBodyFound;

   procedure SetPackageInheritClauses (ThePackage, Clause : in Dictionary.Symbol) is
   begin
      GetPackageRef (ThePackage).InheritClauses := Clause;
   end SetPackageInheritClauses;

   procedure SetPackageVisibleWithClauses (ThePackage, Clause : in Dictionary.Symbol) is
   begin
      GetPackageRef (ThePackage).VisiblePart.WithClauses := Clause;
   end SetPackageVisibleWithClauses;

   procedure SetPackageLocalWithClauses (ThePackage, Clause : in Dictionary.Symbol) is
   begin
      GetPackageRef (ThePackage).LocalPart.WithClauses := Clause;
   end SetPackageLocalWithClauses;

   procedure SetPackageVisibleUseTypeClauses (ThePackage, Clause : in Dictionary.Symbol) is
   begin
      GetPackageRef (ThePackage).VisiblePart.UseTypeClauses := Clause;
   end SetPackageVisibleUseTypeClauses;

   procedure SetPackageLocalUseTypeClauses (ThePackage, Clause : in Dictionary.Symbol) is
   begin
      GetPackageRef (ThePackage).LocalPart.UseTypeClauses := Clause;
   end SetPackageLocalUseTypeClauses;

   procedure SetPackageOwnVariables (ThePackage, OwnVariable : in Dictionary.Symbol) is
   begin
      GetPackageRef (ThePackage).OwnVariables := OwnVariable;
   end SetPackageOwnVariables;

   procedure SetPackageTaskList (ThePackage, OwnTask : in Dictionary.Symbol) is
   begin
      GetPackageRef (ThePackage).TaskList := OwnTask;
   end SetPackageTaskList;

   procedure SetPackageFirstVisibleDeclaration (ThePackage, Declaration : in Dictionary.Symbol) is
   begin
      GetPackageRef (ThePackage).VisiblePart.FirstDeclaration := Declaration;
   end SetPackageFirstVisibleDeclaration;

   procedure SetPackageLastVisibleDeclaration (ThePackage, Declaration : in Dictionary.Symbol) is
   begin
      GetPackageRef (ThePackage).VisiblePart.LastDeclaration := Declaration;
   end SetPackageLastVisibleDeclaration;

   procedure SetPackageVisibleRenamingDeclarations (ThePackage, Declaration : in Dictionary.Symbol) is
   begin
      GetPackageRef (ThePackage).VisiblePart.RenamingDeclarations := Declaration;
   end SetPackageVisibleRenamingDeclarations;

   procedure SetPackageLocalRenamingDeclarations (ThePackage, Declaration : in Dictionary.Symbol) is
   begin
      GetPackageRef (ThePackage).LocalPart.RenamingDeclarations := Declaration;
   end SetPackageLocalRenamingDeclarations;

   procedure SetPackageFirstLocalDeclaration (ThePackage, Declaration : in Dictionary.Symbol) is
   begin
      GetPackageRef (ThePackage).LocalPart.FirstDeclaration := Declaration;
   end SetPackageFirstLocalDeclaration;

   procedure SetPackageLastLocalDeclaration (ThePackage, Declaration : in Dictionary.Symbol) is
   begin
      GetPackageRef (ThePackage).LocalPart.LastDeclaration := Declaration;
   end SetPackageLastLocalDeclaration;

   procedure SetPackageFirstPrivateDeclaration (ThePackage, Declaration : in Dictionary.Symbol) is
   begin
      GetPackageRef (ThePackage).PrivatePart.FirstDeclaration := Declaration;
   end SetPackageFirstPrivateDeclaration;

   procedure SetPackageLastPrivateDeclaration (ThePackage, Declaration : in Dictionary.Symbol) is
   begin
      GetPackageRef (ThePackage).PrivatePart.LastDeclaration := Declaration;
   end SetPackageLastPrivateDeclaration;

   procedure SetPackageParent (ThePackage, TheParent : in Dictionary.Symbol) is
   begin
      GetPackageRef (ThePackage).Parent := TheParent;
   end SetPackageParent;

   procedure SetPackageSibling (ThePackage, TheSibling : in Dictionary.Symbol) is
   begin
      GetPackageRef (ThePackage).NextSibling := TheSibling;
   end SetPackageSibling;

   procedure SetPackageExtends (ThePackage, ThePackageItExtends : in Dictionary.Symbol) is
   begin
      GetPackageRef (ThePackage).Extends := ThePackageItExtends;
   end SetPackageExtends;

   procedure SetPackageDeclaresTaggedType (ThePackage : in Dictionary.Symbol) is
   begin
      GetPackageRef (ThePackage).DeclaresTaggedType := True;
   end SetPackageDeclaresTaggedType;

   procedure SetPackageFirstChild (ThePackage, TheChild : in Dictionary.Symbol;
                                   NameSpace            : in Dictionary.PackageSort) is
   begin
      if NameSpace = Dictionary.Public then
         GetPackageRef (ThePackage).PublicChildren.FirstChild := TheChild;
      else
         GetPackageRef (ThePackage).PrivateChildren.FirstChild := TheChild;
      end if;
   end SetPackageFirstChild;

   procedure SetPackageLastChild (ThePackage, TheChild : in Dictionary.Symbol;
                                  NameSpace            : in Dictionary.PackageSort) is
   begin
      if NameSpace = Dictionary.Public then
         GetPackageRef (ThePackage).PublicChildren.LastChild := TheChild;
      else
         GetPackageRef (ThePackage).PrivateChildren.LastChild := TheChild;
      end if;
   end SetPackageLastChild;

   procedure SetPackageIsGeneric (ThePackage : in Dictionary.Symbol) is
   begin
      Debug.Print_Sym ("ThePackage ", ThePackage);
      Debug.PrintMsg ("IS GENERIC ", True);
      GetPackageRef (ThePackage).IsGeneric := True;
   end SetPackageIsGeneric;

   procedure SetPackageIsPrivate (ThePackage : in Dictionary.Symbol) is
   begin
      GetPackageRef (ThePackage).IsPrivate := True;
   end SetPackageIsPrivate;

   procedure SetPackageFirstLoop (ThePackage, TheLoop : in Dictionary.Symbol) is
   begin
      GetPackageRef (ThePackage).FirstLoop := TheLoop;
   end SetPackageFirstLoop;

   procedure SetPackageLastLoop (ThePackage, TheLoop : in Dictionary.Symbol) is
   begin
      GetPackageRef (ThePackage).LastLoop := TheLoop;
   end SetPackageLastLoop;

   procedure SetPackageFirstGenericParameter (ThePackage, TheParameter : in Dictionary.Symbol) is
   begin
      GetPackageRef (ThePackage).FirstGenericParameter := TheParameter;
   end SetPackageFirstGenericParameter;

   procedure SetPackageLastGenericParameter (ThePackage, TheParameter : in Dictionary.Symbol) is
   begin
      GetPackageRef (ThePackage).LastGenericParameter := TheParameter;
   end SetPackageLastGenericParameter;

   procedure SetPackageSpecification (ThePackage, Specification : in Dictionary.Symbol) is
   begin
      GetPackageRef (ThePackage).Specification := Specification;
   end SetPackageSpecification;

   procedure SetProtectedTypeOwnVariable (TheProtectedType, OwnVariable : in Dictionary.Symbol) is
   begin
      GetProtectedRef (GetTypeRef (TheProtectedType).AncilliaryFields).OwnVariable := OwnVariable;
   end SetProtectedTypeOwnVariable;

   procedure SetProtectedTypeElementsHidden (TheProtectedType : in Dictionary.Symbol) is
   begin
      GetProtectedRef (GetTypeRef (TheProtectedType).AncilliaryFields).ElementsHidden := True;
   end SetProtectedTypeElementsHidden;

   procedure SetProtectedTypeFirstVisibleDeclaration (TheProtectedType, Declaration : in Dictionary.Symbol) is
   begin
      GetProtectedRef (GetTypeRef (TheProtectedType).AncilliaryFields).VisiblePart.FirstDeclaration := Declaration;
   end SetProtectedTypeFirstVisibleDeclaration;

   procedure SetProtectedTypeLastVisibleDeclaration (TheProtectedType, Declaration : in Dictionary.Symbol) is
   begin
      GetProtectedRef (GetTypeRef (TheProtectedType).AncilliaryFields).VisiblePart.LastDeclaration := Declaration;
   end SetProtectedTypeLastVisibleDeclaration;

   procedure SetProtectedTypeFirstLocalDeclaration (TheProtectedType, Declaration : in Dictionary.Symbol) is
   begin
      GetProtectedRef (GetTypeRef (TheProtectedType).AncilliaryFields).LocalPart.FirstDeclaration := Declaration;
   end SetProtectedTypeFirstLocalDeclaration;

   procedure SetProtectedTypeLastLocalDeclaration (TheProtectedType, Declaration : in Dictionary.Symbol) is
   begin
      GetProtectedRef (GetTypeRef (TheProtectedType).AncilliaryFields).LocalPart.LastDeclaration := Declaration;
   end SetProtectedTypeLastLocalDeclaration;

   procedure SetProtectedTypeFirstPrivateDeclaration (TheProtectedType, Declaration : in Dictionary.Symbol) is
   begin
      GetProtectedRef (GetTypeRef (TheProtectedType).AncilliaryFields).PrivatePart.FirstDeclaration := Declaration;
   end SetProtectedTypeFirstPrivateDeclaration;

   procedure SetProtectedTypeLastPrivateDeclaration (TheProtectedType, Declaration : in Dictionary.Symbol) is
   begin
      GetProtectedRef (GetTypeRef (TheProtectedType).AncilliaryFields).PrivatePart.LastDeclaration := Declaration;
   end SetProtectedTypeLastPrivateDeclaration;

   procedure SetProtectedTypeTheEntry (TheProtectedType, TheEntry : in Dictionary.Symbol) is
   begin
      GetProtectedRef (GetTypeRef (TheProtectedType).AncilliaryFields).TheEntry := TheEntry;
   end SetProtectedTypeTheEntry;

   procedure SetProtectedTypeFirstDiscriminant (TheProtectedType, Discriminant : in Dictionary.Symbol) is
   begin
      GetProtectedRef (GetTypeRef (TheProtectedType).AncilliaryFields).FirstDiscriminant := Discriminant;
   end SetProtectedTypeFirstDiscriminant;

   procedure SetProtectedTypeLastDiscriminant (TheProtectedType, Discriminant : in Dictionary.Symbol) is
   begin
      GetProtectedRef (GetTypeRef (TheProtectedType).AncilliaryFields).LastDiscriminant := Discriminant;
   end SetProtectedTypeLastDiscriminant;

   procedure SetProtectedTypeHasPragma (TheProtectedType : in Dictionary.Symbol;
                                        ThePragma        : in Dictionary.RavenscarPragmas) is
   begin
      GetProtectedRef (GetTypeRef (TheProtectedType).AncilliaryFields).Pragmas (ThePragma).Given := True;
   end SetProtectedTypeHasPragma;

   procedure SetProtectedTypePragmaValue
     (TheProtectedType : in Dictionary.Symbol;
      ThePragma        : in Dictionary.RavenscarPragmasWithValue;
      TheValue         : in LexTokenManager.Lex_String) is
   begin
      GetProtectedRef (GetTypeRef (TheProtectedType).AncilliaryFields).Pragmas (ThePragma).Value := TheValue;
   end SetProtectedTypePragmaValue;

   procedure SetProtectedTypeBody (TheProtectedType, TheBody : in Dictionary.Symbol) is
   begin
      GetProtectedRef (GetTypeRef (TheProtectedType).AncilliaryFields).ProtectedBody := TheBody;
   end SetProtectedTypeBody;

   procedure SetProtectedTypeHasProperBody (TheProtectedType : in Dictionary.Symbol) is
   begin
      GetProtectedRef (GetTypeRef (TheProtectedType).AncilliaryFields).HasProperBody := True;
   end SetProtectedTypeHasProperBody;

   procedure SetProtectedTypeWithClauses (TheProtectedType, Clause : in Dictionary.Symbol) is
   begin
      GetProtectedRef (GetTypeRef (TheProtectedType).AncilliaryFields).LocalPart.WithClauses := Clause;
   end SetProtectedTypeWithClauses;

   procedure SetProtectedTypeUseTypeClauses (TheProtectedType, Clause : in Dictionary.Symbol) is
   begin
      GetProtectedRef (GetTypeRef (TheProtectedType).AncilliaryFields).LocalPart.UseTypeClauses := Clause;
   end SetProtectedTypeUseTypeClauses;

   procedure SetTaskTypeSignatureNotWellformed (Abstraction : in Dictionary.Abstractions;
                                                TaskType    : in Dictionary.Symbol) is
   begin
      GetTaskRef (GetTypeRef (TaskType).AncilliaryFields).SignatureIsWellformed (Abstraction) := False;
      --we may be marking a missing second anno as malformed so we must also mark
      --it as present, thus
      if Abstraction = Dictionary.IsRefined then
         GetTaskRef (GetTypeRef (TaskType).AncilliaryFields).HasSecondAnnotation := True;
      end if;
   end SetTaskTypeSignatureNotWellformed;

   procedure SetTaskTypeHasSecondAnnotation (TheTaskType : in Dictionary.Symbol) is
   begin
      GetTaskRef (GetTypeRef (TheTaskType).AncilliaryFields).HasSecondAnnotation := True;
   end SetTaskTypeHasSecondAnnotation;

   procedure SetTaskTypeFirstLocalDeclaration (TheTaskType, Declaration : in Dictionary.Symbol) is
   begin
      GetTaskRef (GetTypeRef (TheTaskType).AncilliaryFields).LocalPart.FirstDeclaration := Declaration;
   end SetTaskTypeFirstLocalDeclaration;

   procedure SetTaskTypeLastLocalDeclaration (TheTaskType, Declaration : in Dictionary.Symbol) is
   begin
      GetTaskRef (GetTypeRef (TheTaskType).AncilliaryFields).LocalPart.LastDeclaration := Declaration;
   end SetTaskTypeLastLocalDeclaration;

   procedure SetTaskTypeWithClauses (TheTaskType, Clause : in Dictionary.Symbol) is
   begin
      GetTaskRef (GetTypeRef (TheTaskType).AncilliaryFields).LocalPart.WithClauses := Clause;
   end SetTaskTypeWithClauses;

   procedure SetTaskTypeUseTypeClauses (TheTaskType, Clause : in Dictionary.Symbol) is
   begin
      GetTaskRef (GetTypeRef (TheTaskType).AncilliaryFields).LocalPart.UseTypeClauses := Clause;
   end SetTaskTypeUseTypeClauses;

   procedure SetTaskTypeFirstDiscriminant (TheTaskType, Discriminant : in Dictionary.Symbol) is
   begin
      GetTaskRef (GetTypeRef (TheTaskType).AncilliaryFields).FirstDiscriminant := Discriminant;
   end SetTaskTypeFirstDiscriminant;

   procedure SetTaskTypeLastDiscriminant (TheTaskType, Discriminant : in Dictionary.Symbol) is
   begin
      GetTaskRef (GetTypeRef (TheTaskType).AncilliaryFields).LastDiscriminant := Discriminant;
   end SetTaskTypeLastDiscriminant;

   procedure SetTaskTypeFirstGlobalVariable
     (Abstraction            : in Dictionary.Abstractions;
      TheTaskType, TheGlobal : in Dictionary.Symbol) is
   begin
      GetTaskRef (GetTypeRef (TheTaskType).AncilliaryFields).FirstGlobalVariable (Abstraction) := TheGlobal;
   end SetTaskTypeFirstGlobalVariable;

   procedure SetTaskTypeLastGlobalVariable
     (Abstraction            : in Dictionary.Abstractions;
      TheTaskType, TheGlobal : in Dictionary.Symbol) is
   begin
      GetTaskRef (GetTypeRef (TheTaskType).AncilliaryFields).LastGlobalVariable (Abstraction) := TheGlobal;
   end SetTaskTypeLastGlobalVariable;

   procedure SetTaskTypeHasPragma (TheTaskType : in Dictionary.Symbol;
                                   ThePragma   : in Dictionary.RavenscarPragmas) is
   begin
      GetTaskRef (GetTypeRef (TheTaskType).AncilliaryFields).Pragmas (ThePragma).Given := True;
   end SetTaskTypeHasPragma;

   procedure SetTaskTypeFirstLoop (TheTaskType, TheLoop : in Dictionary.Symbol) is
   begin
      GetTaskRef (GetTypeRef (TheTaskType).AncilliaryFields).FirstLoop := TheLoop;
   end SetTaskTypeFirstLoop;

   procedure SetTaskTypeLastLoop (TheTaskType, TheLoop : in Dictionary.Symbol) is
   begin
      GetTaskRef (GetTypeRef (TheTaskType).AncilliaryFields).LastLoop := TheLoop;
   end SetTaskTypeLastLoop;

   procedure SetVirtualElementList (TheProtectedType : in Dictionary.Symbol;
                                    TheList          : in Dictionary.Symbol) is
   begin
      GetTypeRef (TheProtectedType).TheVirtualElementList := TheList;
   end SetVirtualElementList;

   procedure SetTaskTypeSuspendsList (TheTaskType     : in Dictionary.Symbol;
                                      TheSuspendsList : in Dictionary.Symbol) is
   begin
      GetTaskRef (GetTypeRef (TheTaskType).AncilliaryFields).SuspendsList := TheSuspendsList;
   end SetTaskTypeSuspendsList;

   procedure SetProcedureSuspendsList (TheProcedure    : in Dictionary.Symbol;
                                       TheSuspendsList : in Dictionary.Symbol) is
   begin
      GetSubprogramRef (TheProcedure).SuspendsList := TheSuspendsList;
   end SetProcedureSuspendsList;

   procedure SetTaskTypePragmaValue
     (TheTaskType : in Dictionary.Symbol;
      ThePragma   : in Dictionary.RavenscarPragmasWithValue;
      TheValue    : in LexTokenManager.Lex_String) is
   begin
      GetTaskRef (GetTypeRef (TheTaskType).AncilliaryFields).Pragmas (ThePragma).Value := TheValue;
   end SetTaskTypePragmaValue;

   procedure SetTaskTypeBody (TheTaskType, TheBody : in Dictionary.Symbol) is
   begin
      GetTaskRef (GetTypeRef (TheTaskType).AncilliaryFields).TaskBody := TheBody;
   end SetTaskTypeBody;

   procedure SetTaskTypeHasProperBody (TheTaskType : in Dictionary.Symbol;
                                       IsHidden    : in Boolean) is
   begin
      GetTaskRef (GetTypeRef (TheTaskType).AncilliaryFields).HasProperBody := True;
      GetTaskRef (GetTypeRef (TheTaskType).AncilliaryFields).BodyIsHidden  := IsHidden;
   end SetTaskTypeHasProperBody;

   function GetPackageName (ThePackage : Dictionary.Symbol) return LexTokenManager.Lex_String is
   begin
      return GetPackageRef (ThePackage).Name;
   end GetPackageName;

   function GetPackagePostcondition
     (Abstraction : Dictionary.Abstractions;
      ThePackage  : Dictionary.Symbol)
     return        ExaminerConstants.RefType
   is
   begin
      return GetPackageRef (ThePackage).Postcondition (Abstraction);
   end GetPackagePostcondition;

   function GetPackageBody (ThePackage : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetPackageRef (ThePackage).PackageBody;
   end GetPackageBody;

   function GetPackageHasProperBody (ThePackage : Dictionary.Symbol) return Boolean is
   begin
      return GetPackageRef (ThePackage).PackageHasProperBody;
   end GetPackageHasProperBody;

   function GetPackageElaborateBodyFound (ThePackage : Dictionary.Symbol) return Boolean is
   begin
      return GetPackageRef (ThePackage).ElaborateBodyFound;
   end GetPackageElaborateBodyFound;

   function GetPackageInheritClauses (ThePackage : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetPackageRef (ThePackage).InheritClauses;
   end GetPackageInheritClauses;

   function GetPackageVisibleWithClauses (ThePackage : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetPackageRef (ThePackage).VisiblePart.WithClauses;
   end GetPackageVisibleWithClauses;

   function GetPackageLocalWithClauses (ThePackage : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetPackageRef (ThePackage).LocalPart.WithClauses;
   end GetPackageLocalWithClauses;

   function GetPackageVisibleUseTypeClauses (ThePackage : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetPackageRef (ThePackage).VisiblePart.UseTypeClauses;
   end GetPackageVisibleUseTypeClauses;

   function GetPackageLocalUseTypeClauses (ThePackage : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetPackageRef (ThePackage).LocalPart.UseTypeClauses;
   end GetPackageLocalUseTypeClauses;

   function GetPackageOwnVariables (ThePackage : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetPackageRef (ThePackage).OwnVariables;
   end GetPackageOwnVariables;

   function Get_Package_Is_Generic (The_Package : Dictionary.Symbol) return Boolean is
   begin
      return GetPackageRef (The_Package).IsGeneric;
   end Get_Package_Is_Generic;

   function GetPackageTaskList (ThePackage : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetPackageRef (ThePackage).TaskList;
   end GetPackageTaskList;

   function GetPackageFirstVisibleDeclaration (ThePackage : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetPackageRef (ThePackage).VisiblePart.FirstDeclaration;
   end GetPackageFirstVisibleDeclaration;

   function GetPackageLastVisibleDeclaration (ThePackage : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetPackageRef (ThePackage).VisiblePart.LastDeclaration;
   end GetPackageLastVisibleDeclaration;

   function GetPackageVisibleRenamingDeclarations (ThePackage : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetPackageRef (ThePackage).VisiblePart.RenamingDeclarations;
   end GetPackageVisibleRenamingDeclarations;

   function GetPackageLocalRenamingDeclarations (ThePackage : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetPackageRef (ThePackage).LocalPart.RenamingDeclarations;
   end GetPackageLocalRenamingDeclarations;

   function GetPackageFirstLocalDeclaration (ThePackage : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetPackageRef (ThePackage).LocalPart.FirstDeclaration;
   end GetPackageFirstLocalDeclaration;

   function GetPackageLastLocalDeclaration (ThePackage : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetPackageRef (ThePackage).LocalPart.LastDeclaration;
   end GetPackageLastLocalDeclaration;

   function GetPackageFirstPrivateDeclaration (ThePackage : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetPackageRef (ThePackage).PrivatePart.FirstDeclaration;
   end GetPackageFirstPrivateDeclaration;

   function GetPackageLastPrivateDeclaration (ThePackage : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetPackageRef (ThePackage).PrivatePart.LastDeclaration;
   end GetPackageLastPrivateDeclaration;

   function GetPackageParent (ThePackage : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetPackageRef (ThePackage).Parent;
   end GetPackageParent;

   function GetPackageSibling (ThePackage : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetPackageRef (ThePackage).NextSibling;
   end GetPackageSibling;

   function GetPackageExtends (ThePackage : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetPackageRef (ThePackage).Extends;
   end GetPackageExtends;

   function GetPackageDeclaresTaggedType (ThePackage : Dictionary.Symbol) return Boolean is
   begin
      return GetPackageRef (ThePackage).DeclaresTaggedType;
   end GetPackageDeclaresTaggedType;

   function GetPackageFirstChild (ThePackage : Dictionary.Symbol;
                                  NameSpace  : Dictionary.PackageSort) return Dictionary.Symbol is
      Result : Dictionary.Symbol;
   begin
      if NameSpace = Public then
         Result := GetPackageRef (ThePackage).PublicChildren.FirstChild;
      else
         Result := GetPackageRef (ThePackage).PrivateChildren.FirstChild;
      end if;
      return Result;
   end GetPackageFirstChild;

   function GetPackageLastChild (ThePackage : Dictionary.Symbol;
                                 NameSpace  : Dictionary.PackageSort) return Dictionary.Symbol is
      Result : Dictionary.Symbol;
   begin
      if NameSpace = Public then
         Result := GetPackageRef (ThePackage).PublicChildren.LastChild;
      else
         Result := GetPackageRef (ThePackage).PrivateChildren.LastChild;
      end if;
      return Result;
   end GetPackageLastChild;

   function GetPackageIsPrivate (ThePackage : in Dictionary.Symbol) return Boolean is
   begin
      return GetPackageRef (ThePackage).IsPrivate;
   end GetPackageIsPrivate;

   function GetPackageFirstLoop (ThePackage : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetPackageRef (ThePackage).FirstLoop;
   end GetPackageFirstLoop;

   function GetPackageLastLoop (ThePackage : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetPackageRef (ThePackage).LastLoop;
   end GetPackageLastLoop;

   function GetPackageFirstGenericParameter (ThePackage : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetPackageRef (ThePackage).FirstGenericParameter;
   end GetPackageFirstGenericParameter;

   function GetPackageLastGenericParameter (ThePackage : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetPackageRef (ThePackage).LastGenericParameter;
   end GetPackageLastGenericParameter;

   function GetPackageSpecification (ThePackage : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetPackageRef (ThePackage).Specification;
   end GetPackageSpecification;

   function GetProtectedTypeOwnVariable (TheProtectedType : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetProtectedRef (GetTypeRef (TheProtectedType).AncilliaryFields).OwnVariable;
   end GetProtectedTypeOwnVariable;

   function GetProtectedTypeElementsHidden (TheProtectedType : Dictionary.Symbol) return Boolean is
   begin
      return GetProtectedRef (GetTypeRef (TheProtectedType).AncilliaryFields).ElementsHidden;
   end GetProtectedTypeElementsHidden;

   function GetProtectedTypeHasEntry (TheProtectedType : Dictionary.Symbol) return Boolean is
   begin
      return GetProtectedRef (GetTypeRef (TheProtectedType).AncilliaryFields).TheEntry /= Dictionary.NullSymbol;
   end GetProtectedTypeHasEntry;

   function GetProtectedTypeFirstVisibleDeclaration (TheProtectedType : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetProtectedRef (GetTypeRef (TheProtectedType).AncilliaryFields).VisiblePart.FirstDeclaration;
   end GetProtectedTypeFirstVisibleDeclaration;

   function GetProtectedTypeLastVisibleDeclaration (TheProtectedType : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetProtectedRef (GetTypeRef (TheProtectedType).AncilliaryFields).VisiblePart.LastDeclaration;
   end GetProtectedTypeLastVisibleDeclaration;

   function GetProtectedTypeFirstLocalDeclaration (TheProtectedType : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetProtectedRef (GetTypeRef (TheProtectedType).AncilliaryFields).LocalPart.FirstDeclaration;
   end GetProtectedTypeFirstLocalDeclaration;

   function GetProtectedTypeLastLocalDeclaration (TheProtectedType : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetProtectedRef (GetTypeRef (TheProtectedType).AncilliaryFields).LocalPart.LastDeclaration;
   end GetProtectedTypeLastLocalDeclaration;

   function GetProtectedTypeFirstPrivateDeclaration (TheProtectedType : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetProtectedRef (GetTypeRef (TheProtectedType).AncilliaryFields).PrivatePart.FirstDeclaration;
   end GetProtectedTypeFirstPrivateDeclaration;

   function GetProtectedTypeLastPrivateDeclaration (TheProtectedType : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetProtectedRef (GetTypeRef (TheProtectedType).AncilliaryFields).PrivatePart.LastDeclaration;
   end GetProtectedTypeLastPrivateDeclaration;

   function GetProtectedTypeTheEntry (TheProtectedType : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetProtectedRef (GetTypeRef (TheProtectedType).AncilliaryFields).TheEntry;
   end GetProtectedTypeTheEntry;

   function GetProtectedTypeFirstDiscriminant (TheProtectedType : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetProtectedRef (GetTypeRef (TheProtectedType).AncilliaryFields).FirstDiscriminant;
   end GetProtectedTypeFirstDiscriminant;

   function GetProtectedTypeLastDiscriminant (TheProtectedType : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetProtectedRef (GetTypeRef (TheProtectedType).AncilliaryFields).LastDiscriminant;
   end GetProtectedTypeLastDiscriminant;

   function GetProtectedTypeHasPragma
     (TheProtectedType : Dictionary.Symbol;
      ThePragma        : Dictionary.RavenscarPragmas)
     return             Boolean
   is
   begin
      return GetProtectedRef (GetTypeRef (TheProtectedType).AncilliaryFields).Pragmas (ThePragma).Given;
   end GetProtectedTypeHasPragma;

   function GetProtectedTypePragmaValue
     (TheProtectedType : Dictionary.Symbol;
      ThePragma        : Dictionary.RavenscarPragmasWithValue)
     return             LexTokenManager.Lex_String
   is
   begin
      return GetProtectedRef (GetTypeRef (TheProtectedType).AncilliaryFields).Pragmas (ThePragma).Value;
   end GetProtectedTypePragmaValue;

   function GetProtectedTypeBody (TheProtectedType : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetProtectedRef (GetTypeRef (TheProtectedType).AncilliaryFields).ProtectedBody;
   end GetProtectedTypeBody;

   function GetProtectedTypeHasProperBody (TheProtectedType : Dictionary.Symbol) return Boolean is
   begin
      return GetProtectedRef (GetTypeRef (TheProtectedType).AncilliaryFields).HasProperBody;
   end GetProtectedTypeHasProperBody;

   function GetProtectedTypeWithClauses (TheProtectedType : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetProtectedRef (GetTypeRef (TheProtectedType).AncilliaryFields).LocalPart.WithClauses;
   end GetProtectedTypeWithClauses;

   function GetProtectedTypeUseTypeClauses (TheProtectedType : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetProtectedRef (GetTypeRef (TheProtectedType).AncilliaryFields).LocalPart.UseTypeClauses;
   end GetProtectedTypeUseTypeClauses;

   function GetTaskTypeSignatureIsWellformed
     (Abstraction : Dictionary.Abstractions;
      TaskType    : Dictionary.Symbol)
     return        Boolean
   is
   begin
      return GetTaskRef (GetTypeRef (TaskType).AncilliaryFields).SignatureIsWellformed (Abstraction);
   end GetTaskTypeSignatureIsWellformed;

   function GetTaskTypeHasSecondAnnotation (TaskType : Dictionary.Symbol) return Boolean is
   begin
      return GetTaskRef (GetTypeRef (TaskType).AncilliaryFields).HasSecondAnnotation;
   end GetTaskTypeHasSecondAnnotation;

   function GetTaskTypeFirstLocalDeclaration (TheTaskType : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetTaskRef (GetTypeRef (TheTaskType).AncilliaryFields).LocalPart.FirstDeclaration;
   end GetTaskTypeFirstLocalDeclaration;

   function GetTaskTypeLastLocalDeclaration (TheTaskType : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetTaskRef (GetTypeRef (TheTaskType).AncilliaryFields).LocalPart.LastDeclaration;
   end GetTaskTypeLastLocalDeclaration;

   function GetTaskTypeWithClauses (TheTaskType : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetTaskRef (GetTypeRef (TheTaskType).AncilliaryFields).LocalPart.WithClauses;
   end GetTaskTypeWithClauses;

   function GetTaskTypeUseTypeClauses (TheTaskType : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetTaskRef (GetTypeRef (TheTaskType).AncilliaryFields).LocalPart.UseTypeClauses;
   end GetTaskTypeUseTypeClauses;

   function GetTaskTypeFirstDiscriminant (TheTaskType : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetTaskRef (GetTypeRef (TheTaskType).AncilliaryFields).FirstDiscriminant;
   end GetTaskTypeFirstDiscriminant;

   function GetTaskTypeLastDiscriminant (TheTaskType : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetTaskRef (GetTypeRef (TheTaskType).AncilliaryFields).LastDiscriminant;
   end GetTaskTypeLastDiscriminant;

   function GetTaskTypeFirstGlobalVariable
     (Abstraction : Dictionary.Abstractions;
      TheTaskType : Dictionary.Symbol)
     return        Dictionary.Symbol
   is
   begin
      return GetTaskRef (GetTypeRef (TheTaskType).AncilliaryFields).FirstGlobalVariable (Abstraction);
   end GetTaskTypeFirstGlobalVariable;

   function GetTaskTypeLastGlobalVariable
     (Abstraction : Dictionary.Abstractions;
      TheTaskType : Dictionary.Symbol)
     return        Dictionary.Symbol
   is
   begin
      return GetTaskRef (GetTypeRef (TheTaskType).AncilliaryFields).LastGlobalVariable (Abstraction);
   end GetTaskTypeLastGlobalVariable;

   function GetTaskTypeHasPragma (TheTaskType : Dictionary.Symbol;
                                  ThePragma   : Dictionary.RavenscarPragmas) return Boolean is
   begin
      return GetTaskRef (GetTypeRef (TheTaskType).AncilliaryFields).Pragmas (ThePragma).Given;
   end GetTaskTypeHasPragma;

   function GetTaskTypePragmaValue
     (TheTaskType : Dictionary.Symbol;
      ThePragma   : Dictionary.RavenscarPragmasWithValue)
     return        LexTokenManager.Lex_String
   is
   begin
      return GetTaskRef (GetTypeRef (TheTaskType).AncilliaryFields).Pragmas (ThePragma).Value;
   end GetTaskTypePragmaValue;

   function GetTaskTypeFirstLoop (TheTaskType : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetTaskRef (GetTypeRef (TheTaskType).AncilliaryFields).FirstLoop;
   end GetTaskTypeFirstLoop;

   function GetTaskTypeLastLoop (TheTaskType : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetTaskRef (GetTypeRef (TheTaskType).AncilliaryFields).LastLoop;
   end GetTaskTypeLastLoop;

   function GetVirtualElementList (TheProtectedType : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetTypeRef (TheProtectedType).TheVirtualElementList;
   end GetVirtualElementList;

   function GetTaskTypeSuspendsList (TheTaskType : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetTaskRef (GetTypeRef (TheTaskType).AncilliaryFields).SuspendsList;
   end GetTaskTypeSuspendsList;

   function GetProcedureSuspendsList (TheProcedure : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetSubprogramRef (TheProcedure).SuspendsList;
   end GetProcedureSuspendsList;

   function GetTaskTypeBody (TheTaskType : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetTaskRef (GetTypeRef (TheTaskType).AncilliaryFields).TaskBody;
   end GetTaskTypeBody;

   function GetTaskTypeHasProperBody (TheTaskType : Dictionary.Symbol) return Boolean is
   begin
      return GetTaskRef (GetTypeRef (TheTaskType).AncilliaryFields).HasProperBody;
   end GetTaskTypeHasProperBody;

   procedure CreateContextClause
     (Scope         : in     Dictionary.Scopes;
      ThePackage    : in     Dictionary.Symbol;
      Explicit      : in     Boolean;
      Comp_Unit     : in     ContextManager.UnitDescriptors;
      Loc           : in     LexTokenManager.Token_Position;
      ContextClause :    out Dictionary.Symbol) is
      NewContextClause : ContextClauseRef;
      function ContextClauseRefToRefType is new Unchecked_Conversion (ContextClauseRef, RefType);
   begin
      NewContextClause :=
        new ContextClauseInfo'(Scope      => Scope,
                               ThePackage => ThePackage,
                               Explicit   => Explicit,
                               Next       => Dictionary.NullSymbol);
      AddSymbol
        (Discriminant => ContextClauseSymbol,
         Ref          => ContextClauseRefToRefType (NewContextClause),
         Comp_Unit    => Comp_Unit,
         Loc          => Loc,
         Item         => ContextClause);
   exception
      when Storage_Error =>
         SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Symbol_Table_Overflow_Dynamic,
                                   Msg     => "");
   end CreateContextClause;

   procedure CreateUseTypeClause
     (Scope         : in     Dictionary.Scopes;
      TheType       : in     Dictionary.Symbol;
      Comp_Unit     : in     ContextManager.UnitDescriptors;
      Loc           : in     LexTokenManager.Token_Position;
      UseTypeClause :    out Dictionary.Symbol) is
      NewUseTypeClause : UseTypeClauseRef;
      function UseTypeClauseRefToRefType is new Unchecked_Conversion (UseTypeClauseRef, RefType);
   begin
      NewUseTypeClause := new UseTypeClauseInfo'(Scope   => Scope,
                                                 TheType => TheType,
                                                 Next    => Dictionary.NullSymbol);
      AddSymbol
        (Discriminant => UseTypeClauseSymbol,
         Ref          => UseTypeClauseRefToRefType (NewUseTypeClause),
         Comp_Unit    => Comp_Unit,
         Loc          => Loc,
         Item         => UseTypeClause);
   exception
      when Storage_Error =>
         SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Symbol_Table_Overflow_Dynamic,
                                   Msg     => "");
   end CreateUseTypeClause;

   procedure SetNextContextClause (Current, Next : in Dictionary.Symbol) is
   begin
      GetContextClauseRef (Current).Next := Next;
   end SetNextContextClause;

   procedure SetContextClauseExplicit (Current : in Dictionary.Symbol) is
   begin
      GetContextClauseRef (Current).Explicit := True;
   end SetContextClauseExplicit;

   procedure SetNextUseTypeClause (Current, Next : in Dictionary.Symbol) is
   begin
      GetUseTypeClauseRef (Current).Next := Next;
   end SetNextUseTypeClause;

   function GetContextClauseScope (ContextClause : Dictionary.Symbol) return Dictionary.Scopes is
   begin
      return GetContextClauseRef (ContextClause).Scope;
   end GetContextClauseScope;

   function GetContextClauseExplicit (ContextClause : Dictionary.Symbol) return Boolean is
   begin
      return GetContextClauseRef (ContextClause).Explicit;
   end GetContextClauseExplicit;

   function GetContextClausePackage (ContextClause : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetContextClauseRef (ContextClause).ThePackage;
   end GetContextClausePackage;

   function GetUseTypeClauseType (UseTypeClause : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetUseTypeClauseRef (UseTypeClause).TheType;
   end GetUseTypeClauseType;

   function GetNextContextClause (ContextClause : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetContextClauseRef (ContextClause).Next;
   end GetNextContextClause;

   function GetNextUseTypeClause (UseTypeClause : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetUseTypeClauseRef (UseTypeClause).Next;
   end GetNextUseTypeClause;

   -- subprograms --------------------------------------------------------------

   procedure CreateSubprogram
     (Name       : in     LexTokenManager.Lex_String;
      Comp_Unit  : in     ContextManager.UnitDescriptors;
      Loc        : in     LexTokenManager.Token_Position;
      Subprogram :    out Dictionary.Symbol) is
      Constraint    : Constraints;
      NullSymbols   : Symbols;
      NewSubprogram : SubprogramRef;
      function SubprogramRefToRefType is new Unchecked_Conversion (SubprogramRef, RefType);
   begin
      Constraint    := Constraints'(Dictionary.Abstractions => 0);
      NullSymbols   := Symbols'(Dictionary.Abstractions => Dictionary.NullSymbol);
      NewSubprogram :=
        new SubprogramInfo'
        (Name                          => Name,
         SignatureIsWellformed         => Booleans'(Dictionary.Abstractions => True),
         ImplicitProofFunctions        => NullSymbols,
         ImplicitReturnVariables       => NullSymbols,
         Precondition                  => Constraint,
         Postcondition                 => Constraint,
         SubprogramBody                => Dictionary.NullSymbol,
         SubprogramHasProperBody       => False,
         SubprogramHasSecondAnnotation => False,
         SubprogramHasSecondConstraint => False,
         HasDerivesAnnotation          => False,
         HasDelayProperty              => False,
         DelayPropertyIsAccountedFor   => False,
         IsGeneric                     => False,
         InstantiationOf               => Dictionary.NullSymbol,
         FirstGenericParameter         => Dictionary.NullSymbol,
         LastGenericParameter          => Dictionary.NullSymbol,
         FirstGenericAssociation       => Dictionary.NullSymbol,
         LastGenericAssociation        => Dictionary.NullSymbol,
         SuspendsList                  => Dictionary.NullSymbol,
         IsEntry                       => False,
         EntryBarrier                  => Dictionary.NullSymbol,
         IsInterruptHandler            => False,
         UsesUnprotectedVariables      => False,
         BodyIsHidden                  => False,
         UsesUncheckedConversion       => False,
         AssignsFromExternal           => False,
         TypeMark                      => Dictionary.NullSymbol,
         WithClauses                   => Dictionary.NullSymbol,
         UseTypeClauses                => Dictionary.NullSymbol,
         FirstParameter                => Dictionary.NullSymbol,
         LastParameter                 => Dictionary.NullSymbol,
         FirstGlobalVariable           => NullSymbols,
         LastGlobalVariable            => NullSymbols,
         RenamingDeclarations          => Dictionary.NullSymbol,
         FirstDeclaration              => Dictionary.NullSymbol,
         LastDeclaration               => Dictionary.NullSymbol,
         FirstLoop                     => Dictionary.NullSymbol,
         LastLoop                      => Dictionary.NullSymbol,
         Specification                 => Dictionary.NullSymbol);
      AddSymbol
        (Discriminant => SubprogramSymbol,
         Ref          => SubprogramRefToRefType (NewSubprogram),
         Comp_Unit    => Comp_Unit,
         Loc          => Loc,
         Item         => Subprogram);
   exception
      when Storage_Error =>
         SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Symbol_Table_Overflow_Dynamic,
                                   Msg     => "");
   end CreateSubprogram;

   procedure SetSubprogramImplicitProofFunction
     (Abstraction   : in Dictionary.Abstractions;
      AdaFunction   : in Dictionary.Symbol;
      ProofFunction : in Dictionary.Symbol) is
   begin
      GetSubprogramRef (AdaFunction).ImplicitProofFunctions (Abstraction) := ProofFunction;
   end SetSubprogramImplicitProofFunction;

   procedure SetSubprogramImplicitReturnVariable
     (Abstraction : in Dictionary.Abstractions;
      TheFunction : in Dictionary.Symbol;
      Variable    : in Dictionary.Symbol) is
   begin
      GetSubprogramRef (TheFunction).ImplicitReturnVariables (Abstraction) := Variable;
   end SetSubprogramImplicitReturnVariable;

   procedure SetSubprogramSignatureNotWellformed (Abstraction : in Dictionary.Abstractions;
                                                  Subprogram  : in Dictionary.Symbol) is
   begin
      GetSubprogramRef (Subprogram).SignatureIsWellformed (Abstraction) := False;
      --we may be marking a missing second anno as malformed so we must also mark
      --it as present, thus
      if Abstraction = Dictionary.IsRefined then
         GetSubprogramRef (Subprogram).SubprogramHasSecondAnnotation := True;
      end if;
   end SetSubprogramSignatureNotWellformed;

   procedure SetSubprogramIsEntry (Subprogram : in Dictionary.Symbol) is
   begin
      GetSubprogramRef (Subprogram).IsEntry := True;
   end SetSubprogramIsEntry;

   procedure SetSubprogramEntryBarrier (Subprogram, TheBarrier : in Dictionary.Symbol) is
   begin
      GetSubprogramRef (Subprogram).EntryBarrier := TheBarrier;
   end SetSubprogramEntryBarrier;

   procedure SetIsInterruptHandler (Subprogram : in Dictionary.Symbol) is
   begin
      GetSubprogramRef (Subprogram).IsInterruptHandler := True;
   end SetIsInterruptHandler;

   procedure SetSubprogramUsesUncheckedConversion (Subprogram : in Dictionary.Symbol) is
   begin
      GetSubprogramRef (Subprogram).UsesUncheckedConversion := True;
   end SetSubprogramUsesUncheckedConversion;

   procedure SetSubprogramAssignsFromExternal (Subprogram : in Dictionary.Symbol) is
   begin
      GetSubprogramRef (Subprogram).AssignsFromExternal := True;
   end SetSubprogramAssignsFromExternal;

   procedure SetSubprogramPrecondition
     (Abstraction  : in Dictionary.Abstractions;
      Subprogram   : in Dictionary.Symbol;
      Precondition : in ExaminerConstants.RefType) is
   begin
      GetSubprogramRef (Subprogram).Precondition (Abstraction) := Precondition;
   end SetSubprogramPrecondition;

   procedure SetSubprogramPostcondition
     (Abstraction   : in Dictionary.Abstractions;
      Subprogram    : in Dictionary.Symbol;
      Postcondition : in ExaminerConstants.RefType) is
   begin
      GetSubprogramRef (Subprogram).Postcondition (Abstraction) := Postcondition;
   end SetSubprogramPostcondition;

   procedure SetSubprogramBody (Subprogram, TheBody : in Dictionary.Symbol) is
   begin
      GetSubprogramRef (Subprogram).SubprogramBody := TheBody;
   end SetSubprogramBody;

   procedure SetSubprogramInstantiationOf (Subprogram, TheGeneric : in Dictionary.Symbol)
   --# global in out Dictionary.Dict;
   --# derives Dictionary.Dict from *,
   --#                              Subprogram,
   --#                              TheGeneric;
   is
   begin
      GetSubprogramRef (Subprogram).InstantiationOf := TheGeneric;
   end SetSubprogramInstantiationOf;

   procedure SetSubprogramHasProperBody (Subprogram : in Dictionary.Symbol;
                                         IsHidden   : in Boolean) is
   begin
      GetSubprogramRef (Subprogram).SubprogramHasProperBody := True;
      GetSubprogramRef (Subprogram).BodyIsHidden            := IsHidden;
   end SetSubprogramHasProperBody;

   procedure SetSubprogramIsGeneric (Subprogram : in Dictionary.Symbol)
   --# global in out Dictionary.Dict;
   --# derives Dictionary.Dict from *,
   --#                              Subprogram;
   is
   begin
      GetSubprogramRef (Subprogram).IsGeneric := True;
   end SetSubprogramIsGeneric;

   procedure SetSubprogramHasSecondAnnotation (Subprogram : in Dictionary.Symbol) is
   begin
      GetSubprogramRef (Subprogram).SubprogramHasSecondAnnotation := True;
   end SetSubprogramHasSecondAnnotation;

   procedure SetSubprogramHasSecondConstraint (Subprogram : in Dictionary.Symbol) is
   begin
      GetSubprogramRef (Subprogram).SubprogramHasSecondConstraint := True;
   end SetSubprogramHasSecondConstraint;

   procedure SetHasDerivesAnnotation (Task_Or_Proc : in Dictionary.Symbol) is
   begin

      case GetSymbolDiscriminant (Task_Or_Proc) is
         when SubprogramSymbol | ImplicitProofFunctionSymbol =>
            GetSubprogramRef (Task_Or_Proc).HasDerivesAnnotation := True;
         when others =>
            -- The only other possibility is a task. We could explicitly check this and raise
            -- a fatal error here, but the same check will be done by GetTaskRef.
            GetTaskRef (GetTypeRef (Task_Or_Proc).AncilliaryFields).HasDerivesAnnotation := True;
      end case;

   end SetHasDerivesAnnotation;

   procedure SetSubprogramReturnType (TheFunction, TypeMark : in Dictionary.Symbol) is
   begin
      GetSubprogramRef (TheFunction).TypeMark := TypeMark;
   end SetSubprogramReturnType;

   procedure SetSubprogramWithClauses (Subprogram, Clause : in Dictionary.Symbol) is
   begin
      GetSubprogramRef (Subprogram).WithClauses := Clause;
   end SetSubprogramWithClauses;

   procedure SetSubprogramUseTypeClauses (Subprogram, Clause : in Dictionary.Symbol) is
   begin
      GetSubprogramRef (Subprogram).UseTypeClauses := Clause;
   end SetSubprogramUseTypeClauses;

   procedure SetSubprogramFirstParameter (Subprogram, Parameter : in Dictionary.Symbol) is
   begin
      GetSubprogramRef (Subprogram).FirstParameter := Parameter;
   end SetSubprogramFirstParameter;

   procedure SetSubprogramLastParameter (Subprogram, Parameter : in Dictionary.Symbol) is
   begin
      GetSubprogramRef (Subprogram).LastParameter := Parameter;
   end SetSubprogramLastParameter;

   procedure SetSubprogramFirstGenericParameter (Subprogram, Parameter : in Dictionary.Symbol) is
   begin
      GetSubprogramRef (Subprogram).FirstGenericParameter := Parameter;
   end SetSubprogramFirstGenericParameter;

   procedure SetSubprogramLastGenericParameter (Subprogram, Parameter : in Dictionary.Symbol) is
   begin
      GetSubprogramRef (Subprogram).LastGenericParameter := Parameter;
   end SetSubprogramLastGenericParameter;

   procedure SetSubprogramFirstGenericAssociation (Subprogram, TheAssociation : in Dictionary.Symbol) is
   begin
      GetSubprogramRef (Subprogram).FirstGenericAssociation := TheAssociation;
   end SetSubprogramFirstGenericAssociation;

   procedure SetSubprogramLastGenericAssociation (Subprogram, TheAssociation : in Dictionary.Symbol) is
   begin
      GetSubprogramRef (Subprogram).LastGenericAssociation := TheAssociation;
   end SetSubprogramLastGenericAssociation;

   procedure SetSubprogramFirstGlobalVariable
     (Abstraction : in Dictionary.Abstractions;
      Subprogram  : in Dictionary.Symbol;
      Variable    : in Dictionary.Symbol) is
   begin
      GetSubprogramRef (Subprogram).FirstGlobalVariable (Abstraction) := Variable;
   end SetSubprogramFirstGlobalVariable;

   procedure SetSubprogramLastGlobalVariable
     (Abstraction : in Dictionary.Abstractions;
      Subprogram  : in Dictionary.Symbol;
      Variable    : in Dictionary.Symbol) is
   begin
      GetSubprogramRef (Subprogram).LastGlobalVariable (Abstraction) := Variable;
   end SetSubprogramLastGlobalVariable;

   procedure SetSubprogramRenamingDeclarations (Subprogram, Declaration : in Dictionary.Symbol) is
   begin
      GetSubprogramRef (Subprogram).RenamingDeclarations := Declaration;
   end SetSubprogramRenamingDeclarations;

   procedure SetSubprogramFirstDeclaration (Subprogram, Declaration : in Dictionary.Symbol) is
   begin
      GetSubprogramRef (Subprogram).FirstDeclaration := Declaration;
   end SetSubprogramFirstDeclaration;

   procedure SetSubprogramLastDeclaration (Subprogram, Declaration : in Dictionary.Symbol) is
   begin
      GetSubprogramRef (Subprogram).LastDeclaration := Declaration;
   end SetSubprogramLastDeclaration;

   procedure SetSubprogramFirstLoop (Subprogram, TheLoop : in Dictionary.Symbol) is
   begin
      GetSubprogramRef (Subprogram).FirstLoop := TheLoop;
   end SetSubprogramFirstLoop;

   procedure SetSubprogramLastLoop (Subprogram, TheLoop : in Dictionary.Symbol) is
   begin
      GetSubprogramRef (Subprogram).LastLoop := TheLoop;
   end SetSubprogramLastLoop;

   procedure SetSubprogramSpecification (Subprogram, Specification : in Dictionary.Symbol) is
   begin
      GetSubprogramRef (Subprogram).Specification := Specification;
   end SetSubprogramSpecification;

   function GetSubprogramName (Subprogram : Dictionary.Symbol) return LexTokenManager.Lex_String is
   begin
      return GetSubprogramRef (Subprogram).Name;
   end GetSubprogramName;

   function GetSubprogramImplicitProofFunction
     (Abstraction : Dictionary.Abstractions;
      AdaFunction : Dictionary.Symbol)
     return        Dictionary.Symbol
   is
   begin
      return GetSubprogramRef (AdaFunction).ImplicitProofFunctions (Abstraction);
   end GetSubprogramImplicitProofFunction;

   function GetSubprogramImplicitReturnVariable
     (Abstraction : Dictionary.Abstractions;
      TheFunction : Dictionary.Symbol)
     return        Dictionary.Symbol
   is
   begin
      return GetSubprogramRef (TheFunction).ImplicitReturnVariables (Abstraction);
   end GetSubprogramImplicitReturnVariable;

   function GetSubprogramSignatureIsWellformed
     (Abstraction : Dictionary.Abstractions;
      Subprogram  : Dictionary.Symbol)
     return        Boolean
   is
   begin
      return GetSubprogramRef (Subprogram).SignatureIsWellformed (Abstraction);
   end GetSubprogramSignatureIsWellformed;

   function GetSubprogramIsEntry (Subprogram : Dictionary.Symbol) return Boolean is
   begin
      return GetSubprogramRef (Subprogram).IsEntry;
   end GetSubprogramIsEntry;

   function GetSubprogramEntryBarrier (Subprogram : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetSubprogramRef (Subprogram).EntryBarrier;
   end GetSubprogramEntryBarrier;

   function IsInterruptHandler (Subprogram : Dictionary.Symbol) return Boolean is
   begin
      return GetSubprogramRef (Subprogram).IsInterruptHandler;
   end IsInterruptHandler;

   function GetSubprogramUsesUncheckedConversion (Subprogram : Dictionary.Symbol) return Boolean is
   begin
      return GetSubprogramRef (Subprogram).UsesUncheckedConversion;
   end GetSubprogramUsesUncheckedConversion;

   function GetSubprogramAssignsFromExternal (Subprogram : Dictionary.Symbol) return Boolean is
   begin
      return GetSubprogramRef (Subprogram).AssignsFromExternal;
   end GetSubprogramAssignsFromExternal;

   function GetSubprogramPrecondition
     (Abstraction : Dictionary.Abstractions;
      Subprogram  : Dictionary.Symbol)
     return        ExaminerConstants.RefType
   is
   begin
      return GetSubprogramRef (Subprogram).Precondition (Abstraction);
   end GetSubprogramPrecondition;

   function GetSubprogramPostcondition
     (Abstraction : Dictionary.Abstractions;
      Subprogram  : Dictionary.Symbol)
     return        ExaminerConstants.RefType
   is
   begin
      return GetSubprogramRef (Subprogram).Postcondition (Abstraction);
   end GetSubprogramPostcondition;

   function GetSubprogramBody (Subprogram : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetSubprogramRef (Subprogram).SubprogramBody;
   end GetSubprogramBody;

   function GetSubprogramInstantiationOf (Subprogram : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetSubprogramRef (Subprogram).InstantiationOf;
   end GetSubprogramInstantiationOf;

   function GetSubprogramHasProperBody (Subprogram : Dictionary.Symbol) return Boolean is
   begin
      return GetSubprogramRef (Subprogram).SubprogramHasProperBody;
   end GetSubprogramHasProperBody;

   function Get_Subprogram_Is_Generic (Subprogram : Dictionary.Symbol) return Boolean is
   begin
      return GetSubprogramRef (Subprogram).IsGeneric;
   end Get_Subprogram_Is_Generic;

   function GetSubprogramHasSecondAnnotation (Subprogram : Dictionary.Symbol) return Boolean is
   begin
      return GetSubprogramRef (Subprogram).SubprogramHasSecondAnnotation;
   end GetSubprogramHasSecondAnnotation;

   function GetSubprogramHasSecondConstraint (Subprogram : Dictionary.Symbol) return Boolean is
   begin
      return GetSubprogramRef (Subprogram).SubprogramHasSecondConstraint;
   end GetSubprogramHasSecondConstraint;

   function GetHasDerivesAnnotation (Task_Or_Proc : Dictionary.Symbol) return Boolean is
      Result : Boolean;
   begin

      case GetSymbolDiscriminant (Task_Or_Proc) is
         when SubprogramSymbol | ImplicitProofFunctionSymbol =>
            Result := GetSubprogramRef (Task_Or_Proc).HasDerivesAnnotation;
         when others =>
            -- The only other possibility is a task. We could explicitly check this and raise
            -- a fatal error here, but the same check will be done by GetTaskRef.
            Result := GetTaskRef (GetTypeRef (Task_Or_Proc).AncilliaryFields).HasDerivesAnnotation;
      end case;

      return Result;

   end GetHasDerivesAnnotation;

   function GetSubprogramReturnType (Subprogram : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetSubprogramRef (Subprogram).TypeMark;
   end GetSubprogramReturnType;

   function GetSubprogramWithClauses (Subprogram : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetSubprogramRef (Subprogram).WithClauses;
   end GetSubprogramWithClauses;

   function GetSubprogramUseTypeClauses (Subprogram : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetSubprogramRef (Subprogram).UseTypeClauses;
   end GetSubprogramUseTypeClauses;

   function GetSubprogramFirstParameter (Subprogram : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetSubprogramRef (Subprogram).FirstParameter;
   end GetSubprogramFirstParameter;

   function GetSubprogramLastParameter (Subprogram : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetSubprogramRef (Subprogram).LastParameter;
   end GetSubprogramLastParameter;

   function GetSubprogramFirstGenericParameter (Subprogram : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetSubprogramRef (Subprogram).FirstGenericParameter;
   end GetSubprogramFirstGenericParameter;

   function GetSubprogramLastGenericParameter (Subprogram : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetSubprogramRef (Subprogram).LastGenericParameter;
   end GetSubprogramLastGenericParameter;

   function GetSubprogramFirstGenericAssociation (Subprogram : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetSubprogramRef (Subprogram).FirstGenericAssociation;
   end GetSubprogramFirstGenericAssociation;

   function GetSubprogramLastGenericAssociation (Subprogram : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetSubprogramRef (Subprogram).LastGenericAssociation;
   end GetSubprogramLastGenericAssociation;

   function GetSubprogramFirstGlobalVariable
     (Abstraction : Dictionary.Abstractions;
      Subprogram  : Dictionary.Symbol)
     return        Dictionary.Symbol
   is
   begin
      return GetSubprogramRef (Subprogram).FirstGlobalVariable (Abstraction);
   end GetSubprogramFirstGlobalVariable;

   function GetSubprogramLastGlobalVariable
     (Abstraction : Dictionary.Abstractions;
      Subprogram  : Dictionary.Symbol)
     return        Dictionary.Symbol
   is
   begin
      return GetSubprogramRef (Subprogram).LastGlobalVariable (Abstraction);
   end GetSubprogramLastGlobalVariable;

   function GetSubprogramRenamingDeclarations (Subprogram : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetSubprogramRef (Subprogram).RenamingDeclarations;
   end GetSubprogramRenamingDeclarations;

   function GetSubprogramFirstDeclaration (Subprogram : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetSubprogramRef (Subprogram).FirstDeclaration;
   end GetSubprogramFirstDeclaration;

   function GetSubprogramLastDeclaration (Subprogram : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetSubprogramRef (Subprogram).LastDeclaration;
   end GetSubprogramLastDeclaration;

   function GetSubprogramFirstLoop (Subprogram : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetSubprogramRef (Subprogram).FirstLoop;
   end GetSubprogramFirstLoop;

   function GetSubprogramLastLoop (Subprogram : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetSubprogramRef (Subprogram).LastLoop;
   end GetSubprogramLastLoop;

   function GetSubprogramSpecification (Subprogram : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetSubprogramRef (Subprogram).Specification;
   end GetSubprogramSpecification;

   procedure CreateImplicitProofFunction
     (AdaFunction   : in     Dictionary.Symbol;
      Comp_Unit     : in     ContextManager.UnitDescriptors;
      Loc           : in     LexTokenManager.Token_Position;
      ProofFunction :    out Dictionary.Symbol) is
      NewProofFunction : ImplicitProofFunctionRef;
      function ImplicitProofFunctionRefToRefType is new Unchecked_Conversion (ImplicitProofFunctionRef, RefType);
   begin
      NewProofFunction := new Dictionary.Symbol'(AdaFunction);
      AddSymbol
        (Discriminant => ImplicitProofFunctionSymbol,
         Ref          => ImplicitProofFunctionRefToRefType (NewProofFunction),
         Comp_Unit    => Comp_Unit,
         Loc          => Loc,
         Item         => ProofFunction);
   exception
      when Storage_Error =>
         SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Symbol_Table_Overflow_Dynamic,
                                   Msg     => "");
   end CreateImplicitProofFunction;

   function GetImplicitProofFunctionAdaFunction (ProofFunction : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetImplicitProofFunctionRef (ProofFunction).all;
   end GetImplicitProofFunctionAdaFunction;

   -- subprogram parameters -----------------------------------------------------------------

   procedure CreateSubprogramParameter
     (Name       : in     LexTokenManager.Lex_String;
      Subprogram : in     Dictionary.Symbol;
      TypeMark   : in     Dictionary.Symbol;
      Mode       : in     Modes;
      Comp_Unit  : in     ContextManager.UnitDescriptors;
      Loc        : in     LexTokenManager.Token_Position;
      Parameter  :    out Dictionary.Symbol) is
      ImportExport     : Booleans;
      Dependencies     : Symbols;
      GlobalReferences : Symbols;
      NewParameter     : SubprogramParameterRef;
      function SubprogramParameterRefToRefType is new Unchecked_Conversion (SubprogramParameterRef, RefType);
   begin
      ImportExport := Booleans'(Dictionary.Abstractions => False);

      Dependencies     := Symbols'(Dictionary.Abstractions => Dictionary.NullSymbol);
      GlobalReferences := Symbols'(Dictionary.Abstractions => Dictionary.NullSymbol);
      NewParameter     :=
        new SubprogramParameterInfo'
        (Name             => Name,
         Subprogram       => Subprogram,
         TypeMark         => TypeMark,
         Mode             => Mode,
         Exported         => ImportExport,
         Imported         => ImportExport,
         Dependencies     => Dependencies,
         GlobalReferences => GlobalReferences,
         Subcomponents    => Dictionary.NullSymbol,
         IndexConstraints => Dictionary.NullSymbol,
         Next             => Dictionary.NullSymbol);
      AddSymbol
        (Discriminant => SubprogramParameterSymbol,
         Ref          => SubprogramParameterRefToRefType (NewParameter),
         Comp_Unit    => Comp_Unit,
         Loc          => Loc,
         Item         => Parameter);
   exception
      when Storage_Error =>
         SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Symbol_Table_Overflow_Dynamic,
                                   Msg     => "");
   end CreateSubprogramParameter;

   procedure SetSubprogramParameterExported (Abstraction : in Dictionary.Abstractions;
                                             Parameter   : in Dictionary.Symbol) is
   begin
      GetSubprogramParameterRef (Parameter).Exported (Abstraction) := True;
   end SetSubprogramParameterExported;

   procedure SetSubprogramParameterImported (Abstraction : in Dictionary.Abstractions;
                                             Parameter   : in Dictionary.Symbol) is
   begin
      GetSubprogramParameterRef (Parameter).Imported (Abstraction) := True;
   end SetSubprogramParameterImported;

   procedure SetSubprogramParameterDependencies
     (Abstraction : in Dictionary.Abstractions;
      Parameter   : in Dictionary.Symbol;
      Dependency  : in Dictionary.Symbol) is
   begin
      GetSubprogramParameterRef (Parameter).Dependencies (Abstraction) := Dependency;
   end SetSubprogramParameterDependencies;

   procedure SetSubprogramParameterGlobalReferences
     (Abstraction : in Dictionary.Abstractions;
      Parameter   : in Dictionary.Symbol;
      Reference   : in Dictionary.Symbol) is
   begin
      GetSubprogramParameterRef (Parameter).GlobalReferences (Abstraction) := Reference;
   end SetSubprogramParameterGlobalReferences;

   procedure SetSubprogramParameterSubComponents (Parameter, Subcomponents : in Dictionary.Symbol) is
   begin
      GetSubprogramParameterRef (Parameter).Subcomponents := Subcomponents;
   end SetSubprogramParameterSubComponents;

   procedure SetSubprogramParameterIndexConstraints (Parameter, IndexConstraints : in Dictionary.Symbol) is
   begin
      GetSubprogramParameterRef (Parameter).IndexConstraints := IndexConstraints;
   end SetSubprogramParameterIndexConstraints;

   procedure SetNextSubprogramParameter (Current, Next : in Dictionary.Symbol) is
   begin
      GetSubprogramParameterRef (Current).Next := Next;
   end SetNextSubprogramParameter;

   function GetSubprogramParameterName (Parameter : Dictionary.Symbol) return LexTokenManager.Lex_String is
   begin
      return GetSubprogramParameterRef (Parameter).Name;
   end GetSubprogramParameterName;

   function GetSubprogramParameterSubprogram (Parameter : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetSubprogramParameterRef (Parameter).Subprogram;
   end GetSubprogramParameterSubprogram;

   function GetSubprogramParameterType (Parameter : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetSubprogramParameterRef (Parameter).TypeMark;
   end GetSubprogramParameterType;

   function GetSubprogramParameterMode (Parameter : Dictionary.Symbol) return Modes is
   begin
      return GetSubprogramParameterRef (Parameter).Mode;
   end GetSubprogramParameterMode;

   function GetSubprogramParameterExported (Abstraction : Dictionary.Abstractions;
                                            Parameter   : Dictionary.Symbol) return Boolean is
   begin
      return GetSubprogramParameterRef (Parameter).Exported (Abstraction);
   end GetSubprogramParameterExported;

   function GetSubprogramParameterImported (Abstraction : Dictionary.Abstractions;
                                            Parameter   : Dictionary.Symbol) return Boolean is
   begin
      return GetSubprogramParameterRef (Parameter).Imported (Abstraction);
   end GetSubprogramParameterImported;

   function GetSubprogramParameterDependencies
     (Abstraction : Dictionary.Abstractions;
      Parameter   : Dictionary.Symbol)
     return        Dictionary.Symbol
   is
   begin
      return GetSubprogramParameterRef (Parameter).Dependencies (Abstraction);
   end GetSubprogramParameterDependencies;

   function GetSubprogramParameterGlobalReferences
     (Abstraction : Dictionary.Abstractions;
      Parameter   : Dictionary.Symbol)
     return        Dictionary.Symbol
   is
   begin
      return GetSubprogramParameterRef (Parameter).GlobalReferences (Abstraction);
   end GetSubprogramParameterGlobalReferences;

   function GetSubprogramParameterSubComponents (Parameter : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetSubprogramParameterRef (Parameter).Subcomponents;
   end GetSubprogramParameterSubComponents;

   function GetSubprogramParameterIndexConstraints (Parameter : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetSubprogramParameterRef (Parameter).IndexConstraints;
   end GetSubprogramParameterIndexConstraints;

   function GetNextSubprogramParameter (Parameter : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetSubprogramParameterRef (Parameter).Next;
   end GetNextSubprogramParameter;

   -- constraints associated with unconstrained formal parameters

   procedure CreateParameterConstraint
     (TheParameter     : in     Dictionary.Symbol;
      Dimension        : in     Positive;
      Comp_Unit        : in     ContextManager.UnitDescriptors;
      Loc              : in     LexTokenManager.Token_Position;
      ConstraintSymbol :    out Dictionary.Symbol) is
      NewParameterConstraint : ParameterConstraintRef;
      function ParameterConstraintRefToRefType is new Unchecked_Conversion (ParameterConstraintRef, RefType);
   begin
      NewParameterConstraint :=
        new ParameterConstraintInfo'(TheParameter => TheParameter,
                                     Dimension    => Dimension,
                                     Next         => Dictionary.NullSymbol);
      AddSymbol
        (Discriminant => ParameterConstraintSymbol,
         Ref          => ParameterConstraintRefToRefType (NewParameterConstraint),
         Comp_Unit    => Comp_Unit,
         Loc          => Loc,
         Item         => ConstraintSymbol);
   exception
      when Storage_Error =>
         SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Symbol_Table_Overflow_Dynamic,
                                   Msg     => "");
   end CreateParameterConstraint;

   function GetParameterConstraintParameter (TheConstraint : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetParameterConstraintRef (TheConstraint).TheParameter;
   end GetParameterConstraintParameter;

   function GetParameterConstraintDimension (TheConstraint : Dictionary.Symbol) return Positive is
   begin
      return GetParameterConstraintRef (TheConstraint).Dimension;
   end GetParameterConstraintDimension;

   function GetParameterConstraintNext (TheConstraint : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetParameterConstraintRef (TheConstraint).Next;
   end GetParameterConstraintNext;

   procedure SetParameterConstraintNext (Current, Next : in Dictionary.Symbol) is
   begin
      GetParameterConstraintRef (Current).Next := Next;
   end SetParameterConstraintNext;

   -- generic parameters -----------------------------------------------------------------

   procedure CreateGenericParameter
     (OwningGeneric : in     Dictionary.Symbol;
      TypeOrObject  : in     Dictionary.Symbol;
      Kind          : in     Dictionary.GenericParameterKind;
      Comp_Unit     : in     ContextManager.UnitDescriptors;
      Loc           : in     LexTokenManager.Token_Position;
      Parameter     :    out Dictionary.Symbol) is
      NewParameter : GenericParameterRef;
      function GenericParameterRefToRefType is new Unchecked_Conversion (GenericParameterRef, RefType);
   begin
      NewParameter :=
        new GenericParameterInfo'
        (OwningGeneric => OwningGeneric,
         TypeOrObject  => TypeOrObject,
         Kind          => Kind,
         Next          => Dictionary.NullSymbol);

      AddSymbol
        (Discriminant => GenericParameterSymbol,
         Ref          => GenericParameterRefToRefType (NewParameter),
         Comp_Unit    => Comp_Unit,
         Loc          => Loc,
         Item         => Parameter);
   exception
      when Storage_Error =>
         SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Symbol_Table_Overflow_Dynamic,
                                   Msg     => "");

   end CreateGenericParameter;

   procedure SetNextGenericParameter (Current, Next : in Dictionary.Symbol) is
   begin
      GetGenericParameterRef (Current).Next := Next;
   end SetNextGenericParameter;

   function GetGenericParameterTypeOrObject (Parameter : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetGenericParameterRef (Parameter).TypeOrObject;
   end GetGenericParameterTypeOrObject;

   function GetGenericParameterKind (Parameter : Dictionary.Symbol) return Dictionary.GenericParameterKind is
   begin
      return GetGenericParameterRef (Parameter).Kind;
   end GetGenericParameterKind;

   function GetGenericParameterName (Parameter : Dictionary.Symbol) return LexTokenManager.Lex_String is
      Result : LexTokenManager.Lex_String;
   begin
      -- A generic parameter might be a type or an object.  The Kind field tells us which.  The name
      -- lives in either the type or the object pointed at.
      case GetGenericParameterKind (Parameter) is
         when GenericTypeParameter =>
            Result := GetTypeName (GetGenericParameterTypeOrObject (Parameter));
         when GenericObjectParameter =>
            Result := GetConstantName (GetGenericParameterTypeOrObject (Parameter));
      end case;
      return Result;
   end GetGenericParameterName;

   function GetGenericParameterNext (Parameter : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetGenericParameterRef (Parameter).Next;
   end GetGenericParameterNext;

   function GetGenericParameterOwningGeneric (Parameter : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetGenericParameterRef (Parameter).OwningGeneric;
   end GetGenericParameterOwningGeneric;

   -- generic associations -----------------------------------------------------------------

   procedure CreateGenericAssociation
     (Formal       : in     Dictionary.Symbol;
      ActualSymbol : in     Dictionary.Symbol;
      Comp_Unit    : in     ContextManager.UnitDescriptors;
      Loc          : in     LexTokenManager.Token_Position;
      Association  :    out Dictionary.Symbol) is
      NewAssociation : GenericAssociationRef;
      function GenericAssociationRefToRefType is new Unchecked_Conversion (GenericAssociationRef, RefType);
   begin
      NewAssociation :=
        new GenericAssociationInfo'(Formal       => Formal,
                                    ActualSymbol => ActualSymbol,
                                    Next         => Dictionary.NullSymbol);

      AddSymbol
        (Discriminant => GenericAssociationSymbol,
         Ref          => GenericAssociationRefToRefType (NewAssociation),
         Comp_Unit    => Comp_Unit,
         Loc          => Loc,
         Item         => Association);
   exception
      when Storage_Error =>
         SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Symbol_Table_Overflow_Dynamic,
                                   Msg     => "");
   end CreateGenericAssociation;

   procedure SetNextGenericAssociation (Current, Next : in Dictionary.Symbol) is
   begin
      GetGenericAssociationRef (Current).Next := Next;
   end SetNextGenericAssociation;

   function GetGenericAssociationFormal (Association : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetGenericAssociationRef (Association).Formal;
   end GetGenericAssociationFormal;

   function GetGenericAssociationActualSymbol (Association : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetGenericAssociationRef (Association).ActualSymbol;
   end GetGenericAssociationActualSymbol;

   function GetGenericAssociationNext (Association : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetGenericAssociationRef (Association).Next;
   end GetGenericAssociationNext;

   -- discriminants -----------------------------------------------------------------------

   procedure CreateKnownDiscriminant
     (Name              : in     LexTokenManager.Lex_String;
      ProtectedType     : in     Dictionary.Symbol;
      TypeMark          : in     Dictionary.Symbol;
      Comp_Unit         : in     ContextManager.UnitDescriptors;
      Loc               : in     LexTokenManager.Token_Position;
      KnownDiscriminant :    out Dictionary.Symbol) is
      NewKnownDiscriminant : KnownDiscriminantRef;
      function KnownDiscriminantRefToRefType is new Unchecked_Conversion (KnownDiscriminantRef, RefType);
   begin
      NewKnownDiscriminant :=
        new KnownDiscriminantInfo'
        (Name          => Name,
         ProtectedType => ProtectedType,
         TypeMark      => TypeMark,
         SetsPriority  => False,
         Next          => Dictionary.NullSymbol);
      AddSymbol
        (Discriminant => KnownDiscriminantSymbol,
         Ref          => KnownDiscriminantRefToRefType (NewKnownDiscriminant),
         Comp_Unit    => Comp_Unit,
         Loc          => Loc,
         Item         => KnownDiscriminant);
   exception
      when Storage_Error =>
         SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Symbol_Table_Overflow_Dynamic,
                                   Msg     => "");
   end CreateKnownDiscriminant;

   procedure SetNextDiscriminant (Current, Next : in Dictionary.Symbol) is
   begin
      GetKnownDiscriminantRef (Current).Next := Next;
   end SetNextDiscriminant;

   procedure SetDiscriminantSetsPriority (Discriminant : in Dictionary.Symbol) is
   begin
      GetKnownDiscriminantRef (Discriminant).SetsPriority := True;
   end SetDiscriminantSetsPriority;

   function GetNextDiscriminant (Discriminant : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetKnownDiscriminantRef (Discriminant).Next;
   end GetNextDiscriminant;

   function GetDiscriminantName (Discriminant : Dictionary.Symbol) return LexTokenManager.Lex_String is
   begin
      return GetKnownDiscriminantRef (Discriminant).Name;
   end GetDiscriminantName;

   function GetDiscriminantTypeMark (Discriminant : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetKnownDiscriminantRef (Discriminant).TypeMark;
   end GetDiscriminantTypeMark;

   function GetDiscriminantProtectedType (Discriminant : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetKnownDiscriminantRef (Discriminant).ProtectedType;
   end GetDiscriminantProtectedType;

   function GetDiscriminantSetsPriority (Discriminant : Dictionary.Symbol) return Boolean is
   begin
      return GetKnownDiscriminantRef (Discriminant).SetsPriority;
   end GetDiscriminantSetsPriority;

   -- note the following creates a record for extra info about task and protected subtypes, not subtypes in general
   procedure CreateSubtype
     (Comp_Unit  : in     ContextManager.UnitDescriptors;
      Loc        : in     LexTokenManager.Token_Position;
      TheSubtype :    out Dictionary.Symbol) is
      NewSubtype : SubtypeRef;
      function SubtypeRefToRefType is new Unchecked_Conversion (SubtypeRef, RefType);
   begin
      NewSubtype :=
        new SubtypeInfo'
        (Priority        => LexTokenManager.Null_String,
         FirstConstraint => Dictionary.NullSymbol,
         LastConstraint  => Dictionary.NullSymbol);
      AddSymbol
        (Discriminant => SubtypeSymbol,
         Ref          => SubtypeRefToRefType (NewSubtype),
         Comp_Unit    => Comp_Unit,
         Loc          => Loc,
         Item         => TheSubtype);
   exception
      when Storage_Error =>
         SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Symbol_Table_Overflow_Dynamic,
                                   Msg     => "");
   end CreateSubtype;

   procedure SetSubtypeInfoPriority (TheSubtype : in Dictionary.Symbol;
                                     Priority   : in LexTokenManager.Lex_String) is
   begin
      GetSubtypeRef (TheSubtype).Priority := Priority;
   end SetSubtypeInfoPriority;

   procedure SetSubtypeInfoFirstConstraint (TheSubtype    : in Dictionary.Symbol;
                                            TheConstraint : in Dictionary.Symbol) is
   begin
      GetSubtypeRef (TheSubtype).FirstConstraint := TheConstraint;
   end SetSubtypeInfoFirstConstraint;

   procedure SetSubtypeInfoLastConstraint (TheSubtype    : in Dictionary.Symbol;
                                           TheConstraint : in Dictionary.Symbol) is
   begin
      GetSubtypeRef (TheSubtype).LastConstraint := TheConstraint;
   end SetSubtypeInfoLastConstraint;

   function GetSubtypeInfoPriority (TheSubtype : Dictionary.Symbol) return LexTokenManager.Lex_String is
   begin
      return GetSubtypeRef (TheSubtype).Priority;
   end GetSubtypeInfoPriority;

   function GetSubtypeInfoFirstConstraint (TheSubtype : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetSubtypeRef (TheSubtype).FirstConstraint;
   end GetSubtypeInfoFirstConstraint;

   function GetSubtypeInfoLastConstraint (TheSubtype : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetSubtypeRef (TheSubtype).LastConstraint;
   end GetSubtypeInfoLastConstraint;

   procedure CreateDiscriminantConstraint
     (Comp_Unit                 : in     ContextManager.UnitDescriptors;
      Loc                       : in     LexTokenManager.Token_Position;
      TheDiscriminantConstraint :    out Dictionary.Symbol) is
      NewDiscriminantConstraint : DiscriminantConstraintRef;
      function DiscriminantConstraintRefToRefType is new Unchecked_Conversion (DiscriminantConstraintRef, RefType);
   begin
      NewDiscriminantConstraint :=
        new DiscriminantConstraintInfo'
        (StaticValue    => LexTokenManager.Null_String,
         AccessedObject => Dictionary.NullSymbol,
         Next           => Dictionary.NullSymbol);
      AddSymbol
        (Discriminant => DiscriminantConstraintSymbol,
         Ref          => DiscriminantConstraintRefToRefType (NewDiscriminantConstraint),
         Comp_Unit    => Comp_Unit,
         Loc          => Loc,
         Item         => TheDiscriminantConstraint);
   exception
      when Storage_Error =>
         SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Symbol_Table_Overflow_Dynamic,
                                   Msg     => "");
   end CreateDiscriminantConstraint;

   procedure SetDiscriminantConstraintStaticValue
     (TheConstraint : in Dictionary.Symbol;
      TheValue      : in LexTokenManager.Lex_String) is
   begin
      GetDiscriminantConstraintRef (TheConstraint).StaticValue := TheValue;
   end SetDiscriminantConstraintStaticValue;

   procedure SetDiscriminantConstraintAccessedObject (TheConstraint : in Dictionary.Symbol;
                                                      TheObject     : in Dictionary.Symbol) is
   begin
      GetDiscriminantConstraintRef (TheConstraint).AccessedObject := TheObject;
   end SetDiscriminantConstraintAccessedObject;

   procedure SetNextDiscriminantConstraint (TheConstraint : in Dictionary.Symbol;
                                            Next          : in Dictionary.Symbol) is
   begin
      GetDiscriminantConstraintRef (TheConstraint).Next := Next;
   end SetNextDiscriminantConstraint;

   function GetDiscriminantConstraintStaticValue (TheConstraint : Dictionary.Symbol) return LexTokenManager.Lex_String is
   begin
      return GetDiscriminantConstraintRef (TheConstraint).StaticValue;
   end GetDiscriminantConstraintStaticValue;

   function GetDiscriminantConstraintAccessedObject (TheConstraint : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetDiscriminantConstraintRef (TheConstraint).AccessedObject;
   end GetDiscriminantConstraintAccessedObject;

   function GetNextDiscriminantConstraint (TheConstraint : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetDiscriminantConstraintRef (TheConstraint).Next;
   end GetNextDiscriminantConstraint;

   procedure CreateGlobalVariable
     (Subprogram     : in     Dictionary.Symbol;
      Variable       : in     Dictionary.Symbol;
      Mode           : in     Dictionary.Modes;
      PrefixNeeded   : in     Boolean;
      NextSubprogram : in     Dictionary.Symbol;
      Comp_Unit      : in     ContextManager.UnitDescriptors;
      Loc            : in     LexTokenManager.Token_Position;
      GlobalVariable :    out Dictionary.Symbol) is
      Dependencies      : Symbols;
      NewGlobalVariable : GlobalVariableRef;
      function GlobalVariableRefToRefType is new Unchecked_Conversion (GlobalVariableRef, RefType);
   begin
      Dependencies      := Symbols'(Dictionary.Abstractions => Dictionary.NullSymbol);
      NewGlobalVariable :=
        new GlobalVariableInfo'
        (Subprogram     => Subprogram,
         Variable       => Variable,
         Mode           => Mode,
         PrefixNeeded   => PrefixNeeded,
         Exported       => False,
         Imported       => False,
         Dependencies   => Dependencies,
         NextSubprogram => NextSubprogram,
         NextVariable   => Dictionary.NullSymbol);
      AddSymbol
        (Discriminant => GlobalVariableSymbol,
         Ref          => GlobalVariableRefToRefType (NewGlobalVariable),
         Comp_Unit    => Comp_Unit,
         Loc          => Loc,
         Item         => GlobalVariable);
   exception
      when Storage_Error =>
         SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Symbol_Table_Overflow_Dynamic,
                                   Msg     => "");
   end CreateGlobalVariable;

   procedure SetGlobalVariableExported (GlobalVariable : in Dictionary.Symbol) is
   begin
      GetGlobalVariableRef (GlobalVariable).Exported := True;
   end SetGlobalVariableExported;

   procedure SetGlobalVariableImported (GlobalVariable : in Dictionary.Symbol) is
   begin
      GetGlobalVariableRef (GlobalVariable).Imported := True;
   end SetGlobalVariableImported;

   procedure SetGlobalVariableDependencies
     (Abstraction    : in Dictionary.Abstractions;
      GlobalVariable : in Dictionary.Symbol;
      Dependency     : in Dictionary.Symbol) is
   begin
      GetGlobalVariableRef (GlobalVariable).Dependencies (Abstraction) := Dependency;
   end SetGlobalVariableDependencies;

   procedure SetGlobalVariableNextVariable (Previous, Next : in Dictionary.Symbol) is
   begin
      GetGlobalVariableRef (Previous).NextVariable := Next;
   end SetGlobalVariableNextVariable;

   function GetGlobalVariableSubprogram (GlobalVariable : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetGlobalVariableRef (GlobalVariable).Subprogram;
   end GetGlobalVariableSubprogram;

   function GetGlobalVariableVariable (GlobalVariable : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetGlobalVariableRef (GlobalVariable).Variable;
   end GetGlobalVariableVariable;

   function GetGlobalVariablePrefixNeeded (GlobalVariable : Dictionary.Symbol) return Boolean is
   begin
      return GetGlobalVariableRef (GlobalVariable).PrefixNeeded;
   end GetGlobalVariablePrefixNeeded;

   function GetGlobalVariableExported (GlobalVariable : Dictionary.Symbol) return Boolean is
   begin
      return GetGlobalVariableRef (GlobalVariable).Exported;
   end GetGlobalVariableExported;

   function GetGlobalVariableImported (GlobalVariable : Dictionary.Symbol) return Boolean is
   begin
      return GetGlobalVariableRef (GlobalVariable).Imported;
   end GetGlobalVariableImported;

   function GetGlobalVariableDependencies
     (Abstraction    : Dictionary.Abstractions;
      GlobalVariable : Dictionary.Symbol)
     return           Dictionary.Symbol
   is
   begin
      return GetGlobalVariableRef (GlobalVariable).Dependencies (Abstraction);
   end GetGlobalVariableDependencies;

   function GetGlobalVariableNextSubprogram (GlobalVariable : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetGlobalVariableRef (GlobalVariable).NextSubprogram;
   end GetGlobalVariableNextSubprogram;

   function GetNextGlobalVariable (GlobalVariable : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetGlobalVariableRef (GlobalVariable).NextVariable;
   end GetNextGlobalVariable;

   procedure CreateDependency
     (TheImport  : in     Dictionary.Symbol;
      Comp_Unit  : in     ContextManager.UnitDescriptors;
      Loc        : in     LexTokenManager.Token_Position;
      Dependency :    out Dictionary.Symbol) is
      NewDependency : DependencyRef;
      function DependencyRefToRefType is new Unchecked_Conversion (DependencyRef, RefType);
   begin
      NewDependency := new DependencyInfo'(Import => TheImport,
                                           Next   => Dictionary.NullSymbol);
      AddSymbol
        (Discriminant => DependencySymbol,
         Ref          => DependencyRefToRefType (NewDependency),
         Comp_Unit    => Comp_Unit,
         Loc          => Loc,
         Item         => Dependency);
   exception
      when Storage_Error =>
         SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Symbol_Table_Overflow_Dynamic,
                                   Msg     => "");
   end CreateDependency;

   procedure SetNextDependency (Current, Next : in Dictionary.Symbol) is
   begin
      GetDependencyRef (Current).Next := Next;
   end SetNextDependency;

   function GetDependencyImport (Dependency : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetDependencyRef (Dependency).Import;
   end GetDependencyImport;

   function GetNextDependency (Dependency : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetDependencyRef (Dependency).Next;
   end GetNextDependency;

   procedure CreateInterruptStreamMapping
     (TheHandler         : in     LexTokenManager.Lex_String;
      TheInterruptStream : in     LexTokenManager.Lex_String;
      Comp_Unit          : in     ContextManager.UnitDescriptors;
      Loc                : in     LexTokenManager.Token_Position;
      TheMapping         :    out Dictionary.Symbol) is
      NewMapping : InterruptStreamMappingRef;
      function InterruptStreamMappingRefToRefType is new Unchecked_Conversion (InterruptStreamMappingRef, RefType);
   begin
      NewMapping :=
        new InterruptStreamMapping'
        (TheHandler         => TheHandler,
         TheInterruptStream => TheInterruptStream,
         Next               => Dictionary.NullSymbol);
      AddSymbol
        (Discriminant => InterruptStreamMappingSymbol,
         Ref          => InterruptStreamMappingRefToRefType (NewMapping),
         Comp_Unit    => Comp_Unit,
         Loc          => Loc,
         Item         => TheMapping);
   exception
      when Storage_Error =>
         SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Symbol_Table_Overflow_Dynamic,
                                   Msg     => "");
   end CreateInterruptStreamMapping;

   procedure SetNextInterruptStreamMapping (Current, Next : in Dictionary.Symbol) is
   begin
      GetInterruptStreamMappingRef (Current).Next := Next;
   end SetNextInterruptStreamMapping;

   function GetNextInterruptStreamMapping (TheMapping : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetInterruptStreamMappingRef (TheMapping).Next;
   end GetNextInterruptStreamMapping;

   procedure SetOwnVariableInterruptStreamMappings
     (TheOwnVariable             : in Dictionary.Symbol;
      TheInterruptStreamMappings : in Dictionary.Symbol) is
   begin
      GetOwnVariableRef (TheOwnVariable).InterruptStreamMappings := TheInterruptStreamMappings;
   end SetOwnVariableInterruptStreamMappings;

   procedure SetOwnVariableImplicitInStream (TheOwnVariable      : in Dictionary.Symbol;
                                             TheImplicitInStream : in Dictionary.Symbol) is
   begin
      GetOwnVariableRef (TheOwnVariable).ImplicitInStream := TheImplicitInStream;
   end SetOwnVariableImplicitInStream;

   function GetOwnVariableInterruptStreamMappings (TheOwnVariable : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetOwnVariableRef (TheOwnVariable).InterruptStreamMappings;
   end GetOwnVariableInterruptStreamMappings;

   procedure SetOwnVariableIsInterruptStream (TheOwnVariable    : in Dictionary.Symbol;
                                              IsInterruptStream : in Boolean) is
   begin
      GetOwnVariableRef (TheOwnVariable).IsInterruptStream := IsInterruptStream;
   end SetOwnVariableIsInterruptStream;

   function GetOwnVariableIsInterruptStream (TheOwnVariable : Dictionary.Symbol) return Boolean is
   begin
      return GetOwnVariableRef (TheOwnVariable).IsInterruptStream;
   end GetOwnVariableIsInterruptStream;

   function GetInterruptStreamMappingHandler (TheMapping : in Symbol) return LexTokenManager.Lex_String is
   begin
      return GetInterruptStreamMappingRef (TheMapping).TheHandler;
   end GetInterruptStreamMappingHandler;

   function GetInterruptStreamMappingStream (TheMapping : in Symbol) return LexTokenManager.Lex_String is
   begin
      return GetInterruptStreamMappingRef (TheMapping).TheInterruptStream;
   end GetInterruptStreamMappingStream;

   procedure CreateSuspendsListItem
     (ThePOorSO        : in     Dictionary.Symbol;
      Comp_Unit        : in     ContextManager.UnitDescriptors;
      Loc              : in     LexTokenManager.Token_Position;
      SuspendsListItem :    out Dictionary.Symbol) is
      NewSuspendsListItem : SuspendsListItemInfoRef;
      function SuspendsListItemRefToRefType is new Unchecked_Conversion (SuspendsListItemInfoRef, RefType);
   begin
      NewSuspendsListItem :=
        new SuspendsListItemInfo'(ThePOorSO      => ThePOorSO,
                                  IsAccountedFor => False,
                                  Next           => Dictionary.NullSymbol);
      AddSymbol
        (Discriminant => SuspendsListItemSymbol,
         Ref          => SuspendsListItemRefToRefType (NewSuspendsListItem),
         Comp_Unit    => Comp_Unit,
         Loc          => Loc,
         Item         => SuspendsListItem);
   exception
      when Storage_Error =>
         SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Symbol_Table_Overflow_Dynamic,
                                   Msg     => "");
   end CreateSuspendsListItem;

   procedure SetNextSuspendsListItem (Current, Next : in Dictionary.Symbol) is
   begin
      GetSuspendsListItemRef (Current).Next := Next;
   end SetNextSuspendsListItem;

   function GetSuspendsListItem (SuspendsListItem : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetSuspendsListItemRef (SuspendsListItem).ThePOorSO;
   end GetSuspendsListItem;

   procedure SetSuspendsListItemIsAccountedFor (SuspendsListItem : in Dictionary.Symbol) is
   begin
      GetSuspendsListItemRef (SuspendsListItem).IsAccountedFor := True;
   end SetSuspendsListItemIsAccountedFor;

   function GetSuspendsListItemIsAccountedFor (SuspendsListItem : Dictionary.Symbol) return Boolean is
   begin
      return GetSuspendsListItemRef (SuspendsListItem).IsAccountedFor;
   end GetSuspendsListItemIsAccountedFor;

   function GetNextSuspendsListItem (SuspendsListItem : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetSuspendsListItemRef (SuspendsListItem).Next;
   end GetNextSuspendsListItem;

   procedure CreateVirtualElement
     (TheVariable       : in     Dictionary.Symbol;
      TheOwner          : in     Dictionary.Symbol;
      Comp_Unit         : in     ContextManager.UnitDescriptors;
      Loc               : in     LexTokenManager.Token_Position;
      TheVirtualElement :    out Dictionary.Symbol) is
      NewVirtualElement : VirtualElementInfoRef;
      function VirtualElementRefToRefType is new Unchecked_Conversion (VirtualElementInfoRef, RefType);
   begin
      NewVirtualElement :=
        new VirtualElementInfo'
        (TheVariable => TheVariable,
         TheOwner    => TheOwner,
         SeenByOwner => False,
         Next        => Dictionary.NullSymbol);
      AddSymbol
        (Discriminant => VirtualElementSymbol,
         Ref          => VirtualElementRefToRefType (NewVirtualElement),
         Comp_Unit    => Comp_Unit,
         Loc          => Loc,
         Item         => TheVirtualElement);
   exception
      when Storage_Error =>
         SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Symbol_Table_Overflow_Dynamic,
                                   Msg     => "");
   end CreateVirtualElement;

   function GetVirtualElementVariable (VirtualElement : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetVirtualElementRef (VirtualElement).TheVariable;
   end GetVirtualElementVariable;

   function GetVirtualElementOwner (VirtualElement : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetVirtualElementRef (VirtualElement).TheOwner;
   end GetVirtualElementOwner;

   procedure SetVirtualElementSeenByOwner (VirtualElement : in Dictionary.Symbol) is
   begin
      GetVirtualElementRef (VirtualElement).SeenByOwner := True;
   end SetVirtualElementSeenByOwner;

   function GetVirtualElementSeenByOwner (VirtualElement : Dictionary.Symbol) return Boolean is
   begin
      return GetVirtualElementRef (VirtualElement).SeenByOwner;
   end GetVirtualElementSeenByOwner;

   procedure SetNextVirtualElement (Current, Next : in Dictionary.Symbol) is
   begin
      GetVirtualElementRef (Current).Next := Next;
   end SetNextVirtualElement;

   function GetNextVirtualElement (VirtualElement : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetVirtualElementRef (VirtualElement).Next;
   end GetNextVirtualElement;

   procedure CreateUnaryOperator
     (Name      : in     SP_Symbols.SP_Symbol;
      Operand   : in     Dictionary.Symbol;
      Comp_Unit : in     ContextManager.UnitDescriptors;
      Loc       : in     LexTokenManager.Token_Position;
      Operator  :    out Dictionary.Symbol) is
      NewOperator : OperatorRef;
      function OperatorRefToRefType is new Unchecked_Conversion (OperatorRef, RefType);
   begin
      NewOperator := new OperatorInfo'(Name  => Name,
                                       Left  => Dictionary.NullSymbol,
                                       Right => Operand);
      AddSymbol
        (Discriminant => OperatorSymbol,
         Ref          => OperatorRefToRefType (NewOperator),
         Comp_Unit    => Comp_Unit,
         Loc          => Loc,
         Item         => Operator);
   exception
      when Storage_Error =>
         SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Symbol_Table_Overflow_Dynamic,
                                   Msg     => "");
   end CreateUnaryOperator;

   procedure CreateBinaryOperator
     (Name        : in     SP_Symbols.SP_Symbol;
      Left, Right : in     Dictionary.Symbol;
      Comp_Unit   : in     ContextManager.UnitDescriptors;
      Loc         : in     LexTokenManager.Token_Position;
      Operator    :    out Dictionary.Symbol) is
      NewOperator : OperatorRef;
      function OperatorRefToRefType is new Unchecked_Conversion (OperatorRef, RefType);
   begin
      NewOperator := new OperatorInfo'(Name  => Name,
                                       Left  => Left,
                                       Right => Right);
      AddSymbol
        (Discriminant => OperatorSymbol,
         Ref          => OperatorRefToRefType (NewOperator),
         Comp_Unit    => Comp_Unit,
         Loc          => Loc,
         Item         => Operator);
   exception
      when Storage_Error =>
         SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Symbol_Table_Overflow_Dynamic,
                                   Msg     => "");
   end CreateBinaryOperator;

   function GetOperatorName (Operator : Dictionary.Symbol) return SP_Symbols.SP_Symbol is
   begin
      return GetOperatorRef (Operator).Name;
   end GetOperatorName;

   function GetOperatorLeftOperand (Operator : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetOperatorRef (Operator).Left;
   end GetOperatorLeftOperand;

   function GetOperatorRightOperand (Operator : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetOperatorRef (Operator).Right;
   end GetOperatorRightOperand;

   procedure CreateLoop
     (Region    : in     Dictionary.Symbol;
      Comp_Unit : in     ContextManager.UnitDescriptors;
      Loc       : in     LexTokenManager.Token_Position;
      TheLoop   :    out Dictionary.Symbol) is
      NewLoop : LoopRef;
      function LoopRefToRefType is new Unchecked_Conversion (LoopRef, RefType);
   begin
      NewLoop :=
        new LoopInfo'
        (Name          => LexTokenManager.Null_String,
         Region        => Region,
         LoopParameter => Dictionary.NullSymbol,
         OnEntryVars   => Dictionary.NullSymbol,
         ExitExpn      => 0,
         EntryExpn     => 0,
         HasExits      => False,
         Next          => Dictionary.NullSymbol);
      AddSymbol
        (Discriminant => LoopSymbol,
         Ref          => LoopRefToRefType (NewLoop),
         Comp_Unit    => Comp_Unit,
         Loc          => Loc,
         Item         => TheLoop);
   exception
      when Storage_Error =>
         SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Symbol_Table_Overflow_Dynamic,
                                   Msg     => "");
   end CreateLoop;

   procedure SetLoopName (Name    : in LexTokenManager.Lex_String;
                          TheLoop : in Dictionary.Symbol) is
   begin
      GetLoopRef (TheLoop).Name := Name;
   end SetLoopName;

   procedure SetLoopParameter (ForLoop, LoopParameter : in Dictionary.Symbol) is
   begin
      GetLoopRef (ForLoop).LoopParameter := LoopParameter;
   end SetLoopParameter;

   procedure SetLoopOnEntryVars (ForLoop, OnEntryVars : in Dictionary.Symbol) is
   begin
      GetLoopRef (ForLoop).OnEntryVars := OnEntryVars;
   end SetLoopOnEntryVars;

   procedure SetLoopExitExpn (ForLoop : in Dictionary.Symbol;
                              Expn    : in Natural) is
   begin
      GetLoopRef (ForLoop).ExitExpn := Expn;
   end SetLoopExitExpn;

   procedure SetLoopEntryExpn (ForLoop : in Dictionary.Symbol;
                               Expn    : in Natural) is
   begin
      GetLoopRef (ForLoop).EntryExpn := Expn;
   end SetLoopEntryExpn;

   procedure SetLoopHasExits (TheLoop : in Dictionary.Symbol) is
   begin
      GetLoopRef (TheLoop).HasExits := True;
   end SetLoopHasExits;

   procedure SetNextLoop (Current, Next : in Dictionary.Symbol) is
   begin
      GetLoopRef (Current).Next := Next;
   end SetNextLoop;

   function GetLoopName (TheLoop : Dictionary.Symbol) return LexTokenManager.Lex_String is
   begin
      return GetLoopRef (TheLoop).Name;
   end GetLoopName;

   function GetLoopRegion (TheLoop : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetLoopRef (TheLoop).Region;
   end GetLoopRegion;

   function GetLoopParameter (TheLoop : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetLoopRef (TheLoop).LoopParameter;
   end GetLoopParameter;

   function GetLoopOnEntryVars (TheLoop : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetLoopRef (TheLoop).OnEntryVars;
   end GetLoopOnEntryVars;

   function GetLoopExitExpn (TheLoop : Dictionary.Symbol) return Natural is
   begin
      return GetLoopRef (TheLoop).ExitExpn;
   end GetLoopExitExpn;

   function GetLoopEntryExpn (TheLoop : Dictionary.Symbol) return Natural is
   begin
      return GetLoopRef (TheLoop).EntryExpn;
   end GetLoopEntryExpn;

   function GetLoopHasExits (TheLoop : Dictionary.Symbol) return Boolean is
   begin
      return GetLoopRef (TheLoop).HasExits;
   end GetLoopHasExits;

   function GetNextLoop (TheLoop : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetLoopRef (TheLoop).Next;
   end GetNextLoop;

   procedure CreateLoopParameter
     (Name           : in     LexTokenManager.Lex_String;
      TypeMark       : in     Dictionary.Symbol;
      TheLoop        : in     Dictionary.Symbol;
      HasStaticRange : in     Boolean;
      IsReverse      : in     Boolean;
      Comp_Unit      : in     ContextManager.UnitDescriptors;
      Loc            : in     LexTokenManager.Token_Position;
      LoopParameter  :    out Dictionary.Symbol) is
      NewParameter : LoopParameterRef;
      function LoopParameterRefToRefType is new Unchecked_Conversion (LoopParameterRef, RefType);
   begin
      NewParameter :=
        new LoopParameterInfo'
        (Name           => Name,
         TypeMark       => TypeMark,
         TheLoop        => TheLoop,
         HasStaticRange => HasStaticRange,
         IsReverse      => IsReverse);
      AddSymbol
        (Discriminant => LoopParameterSymbol,
         Ref          => LoopParameterRefToRefType (NewParameter),
         Comp_Unit    => Comp_Unit,
         Loc          => Loc,
         Item         => LoopParameter);
   exception
      when Storage_Error =>
         SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Symbol_Table_Overflow_Dynamic,
                                   Msg     => "");
   end CreateLoopParameter;

   procedure CreateLoopEntryVariable
     (OriginalVar       : in     Dictionary.Symbol;
      TheLoop           : in     Dictionary.Symbol;
      Comp_Unit         : in     ContextManager.UnitDescriptors;
      Loc               : in     LexTokenManager.Token_Position;
      LoopEntryVariable :    out Dictionary.Symbol) is
      NewBound : LoopEntryVariableRef;
      function LoopEntryVariableRefToRefType is new Unchecked_Conversion (LoopEntryVariableRef, RefType);
   begin
      NewBound := new LoopEntryVariableInfo'(OriginalVar => OriginalVar,
                                             TheLoop     => TheLoop,
                                             Next        => Dictionary.NullSymbol);
      AddSymbol
        (Discriminant => LoopEntryVariableSymbol,
         Ref          => LoopEntryVariableRefToRefType (NewBound),
         Comp_Unit    => Comp_Unit,
         Loc          => Loc,
         Item         => LoopEntryVariable);
   exception
      when Storage_Error =>
         SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Symbol_Table_Overflow_Dynamic,
                                   Msg     => "");
   end CreateLoopEntryVariable;

   function GetLoopParameterName (LoopParameter : Dictionary.Symbol) return LexTokenManager.Lex_String is
   begin
      return GetLoopParameterRef (LoopParameter).Name;
   end GetLoopParameterName;

   function GetLoopParameterType (LoopParameter : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetLoopParameterRef (LoopParameter).TypeMark;
   end GetLoopParameterType;

   function GetLoopParameterLoop (LoopParameter : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetLoopParameterRef (LoopParameter).TheLoop;
   end GetLoopParameterLoop;

   function GetLoopParameterHasStaticRange (LoopParameter : Dictionary.Symbol) return Boolean is
   begin
      return GetLoopParameterRef (LoopParameter).HasStaticRange;
   end GetLoopParameterHasStaticRange;

   function GetLoopParameterIsReverse (LoopParameter : Dictionary.Symbol) return Boolean is
   begin
      return GetLoopParameterRef (LoopParameter).IsReverse;
   end GetLoopParameterIsReverse;

   procedure SetLoopEntryVariableNext (LoopEntryVariable : in Dictionary.Symbol;
                                       Next              : in Dictionary.Symbol) is
   begin
      GetLoopEntryVariableRef (LoopEntryVariable).Next := Next;
   end SetLoopEntryVariableNext;

   function GetLoopEntryVariableOriginalVar (LoopEntryVariable : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetLoopEntryVariableRef (LoopEntryVariable).OriginalVar;
   end GetLoopEntryVariableOriginalVar;

   function GetLoopEntryVariableTheLoop (LoopEntryVariable : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetLoopEntryVariableRef (LoopEntryVariable).TheLoop;
   end GetLoopEntryVariableTheLoop;

   function GetLoopEntryVariableNext (LoopEntryVariable : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetLoopEntryVariableRef (LoopEntryVariable).Next;
   end GetLoopEntryVariableNext;

   procedure CreateDeclaration
     (Item        : in     Dictionary.Symbol;
      Context     : in     Contexts;
      Comp_Unit   : in     ContextManager.UnitDescriptors;
      Loc         : in     LexTokenManager.Token_Position;
      Declaration :    out Dictionary.Symbol) is
      NewDeclaration : DeclarationRef;
      function DeclarationRefToRefType is new Unchecked_Conversion (DeclarationRef, RefType);
   begin
      NewDeclaration :=
        new DeclarationInfo'
        (Context => Context,
         Scope   => Dictionary.Scopes'(TypeOfScope => Visible,
                                       Region      => NullSymbol),
         Item    => Item,
         Next    => Dictionary.NullSymbol);
      AddSymbol
        (Discriminant => DeclarationSymbol,
         Ref          => DeclarationRefToRefType (NewDeclaration),
         Comp_Unit    => Comp_Unit,
         Loc          => Loc,
         Item         => Declaration);
   exception
      when Storage_Error =>
         SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Symbol_Table_Overflow_Dynamic,
                                   Msg     => "");
   end CreateDeclaration;

   procedure SetDeclarationScope (Declaration : in Dictionary.Symbol;
                                  Scope       : in Scopes) is
   begin
      GetDeclarationRef (Declaration).Scope := Scope;
   end SetDeclarationScope;

   procedure SetDeclarationContext (Declaration : in Dictionary.Symbol;
                                    Context     : in Contexts) is
   begin
      GetDeclarationRef (Declaration).Context := Context;
   end SetDeclarationContext;

   procedure SetNextDeclaration (Current, Next : in Dictionary.Symbol) is
   begin
      GetDeclarationRef (Current).Next := Next;
   end SetNextDeclaration;

   function GetDeclarationContext (Declaration : Dictionary.Symbol) return Contexts is
   begin
      return GetDeclarationRef (Declaration).Context;
   end GetDeclarationContext;

   function GetDeclarationScope (Declaration : Dictionary.Symbol) return Dictionary.Scopes is
   begin
      return GetDeclarationRef (Declaration).Scope;
   end GetDeclarationScope;

   function GetDeclarationItem (Declaration : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetDeclarationRef (Declaration).Item;
   end GetDeclarationItem;

   function GetNextDeclaration (Declaration : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetDeclarationRef (Declaration).Next;
   end GetNextDeclaration;

   procedure SetHasDelayProperty (TheProcedure : in Dictionary.Symbol) is
   begin
      GetSubprogramRef (TheProcedure).HasDelayProperty := True;
   end SetHasDelayProperty;

   procedure SetTaskUsesUnprotectedVariables (TheTaskType : in Dictionary.Symbol) is
   begin
      GetTaskRef (GetTypeRef (TheTaskType).AncilliaryFields).UsesUnprotectedVariables := True;
   end SetTaskUsesUnprotectedVariables;

   procedure SetTaskUsesUncheckedConversion (TheTaskType : in Dictionary.Symbol) is
   begin
      GetTaskRef (GetTypeRef (TheTaskType).AncilliaryFields).UsesUncheckedConversion := True;
   end SetTaskUsesUncheckedConversion;

   procedure SetTaskAssignsFromExternal (TheTaskType : in Dictionary.Symbol) is
   begin
      GetTaskRef (GetTypeRef (TheTaskType).AncilliaryFields).AssignsFromExternal := True;
   end SetTaskAssignsFromExternal;

   procedure SetSubprogUsesUnprotectedVariables (SubprogSym : in Dictionary.Symbol) is
   begin
      GetSubprogramRef (SubprogSym).UsesUnprotectedVariables := True;
   end SetSubprogUsesUnprotectedVariables;

   procedure SetUnprotectedReference (OwnVariable : in Dictionary.Symbol;
                                      ByThread    : in Dictionary.Symbol) is
   begin
      GetOwnVariableRef (OwnVariable).UnprotectedReference := ByThread;
   end SetUnprotectedReference;

   procedure SetSuspendsReference (OwnVariable : in Dictionary.Symbol;
                                   ByThread    : in Dictionary.Symbol) is
   begin
      GetOwnVariableRef (OwnVariable).SuspendsReference := ByThread;
   end SetSuspendsReference;

   function HasDelayProperty (TheProcedure : Dictionary.Symbol) return Boolean is
   begin
      return GetSubprogramRef (TheProcedure).HasDelayProperty;
   end HasDelayProperty;

   function DelayPropertyIsAccountedFor (TheProcedure : Symbol) return Boolean is
   begin
      return GetSubprogramRef (TheProcedure).DelayPropertyIsAccountedFor;
   end DelayPropertyIsAccountedFor;

   procedure MarkAccountsForDelay (TheProcedure : in Symbol) is
   begin
      GetSubprogramRef (TheProcedure).DelayPropertyIsAccountedFor := True;
   end MarkAccountsForDelay;

   procedure SetIsSuspendable (OwnVariable : in Dictionary.Symbol) is
   begin
      GetOwnVariableRef (OwnVariable).Suspendable := True;
   end SetIsSuspendable;

   function GetIsSuspendable (OwnVariable : Dictionary.Symbol) return Boolean is
   begin
      return GetOwnVariableRef (OwnVariable).Suspendable;
   end GetIsSuspendable;

   procedure SetHasInterruptProperty (OwnVariable : in Dictionary.Symbol) is
   begin
      GetOwnVariableRef (OwnVariable).Interruptable := True;
   end SetHasInterruptProperty;

   function GetHasInterruptProperty (OwnVariable : Dictionary.Symbol) return Boolean is
   begin
      return GetOwnVariableRef (OwnVariable).Interruptable;
   end GetHasInterruptProperty;

   procedure SetPriorityProperty (OwnVariable : in Dictionary.Symbol;
                                  TheValue    : in LexTokenManager.Lex_String) is
   begin
      GetOwnVariableRef (OwnVariable).Priority := TheValue;
   end SetPriorityProperty;

   procedure SetIntegrityProperty (OwnVariable : in Dictionary.Symbol;
                                   TheValue    : in LexTokenManager.Lex_String) is
   begin
      GetOwnVariableRef (OwnVariable).Integrity := TheValue;
   end SetIntegrityProperty;

   function GetPriorityProperty (OwnVariable : Dictionary.Symbol) return LexTokenManager.Lex_String is
   begin
      return GetOwnVariableRef (OwnVariable).Priority;
   end GetPriorityProperty;

   function GetIntegrityProperty (OwnVariable : Dictionary.Symbol) return LexTokenManager.Lex_String is
   begin
      return GetOwnVariableRef (OwnVariable).Integrity;
   end GetIntegrityProperty;

   function HasValidPriorityProperty (OwnVariable : Dictionary.Symbol) return Boolean is
   begin
      return LexTokenManager.Lex_String_Case_Insensitive_Compare
        (Lex_Str1 => GetPriorityProperty (OwnVariable),
         Lex_Str2 => LexTokenManager.Null_String) /=
        LexTokenManager.Str_Eq;
   end HasValidPriorityProperty;

   function TaskUsesUnprotectedVariables (TheTaskType : Dictionary.Symbol) return Boolean is
   begin
      return GetTaskRef (GetTypeRef (TheTaskType).AncilliaryFields).UsesUnprotectedVariables;
   end TaskUsesUnprotectedVariables;

   function GetTaskUsesUncheckedConversion (TheTaskType : Dictionary.Symbol) return Boolean is
   begin
      return GetTaskRef (GetTypeRef (TheTaskType).AncilliaryFields).UsesUncheckedConversion;
   end GetTaskUsesUncheckedConversion;

   function GetTaskAssignsFromExternal (TheTaskType : Dictionary.Symbol) return Boolean is
   begin
      return GetTaskRef (GetTypeRef (TheTaskType).AncilliaryFields).AssignsFromExternal;
   end GetTaskAssignsFromExternal;

   function SubprogUsesUnprotectedVariables (SubprogSym : Dictionary.Symbol) return Boolean is
   begin
      return GetSubprogramRef (SubprogSym).UsesUnprotectedVariables;
   end SubprogUsesUnprotectedVariables;

   function GetUnprotectedReference (OwnVariable : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetOwnVariableRef (OwnVariable).UnprotectedReference;
   end GetUnprotectedReference;

   function GetSuspendsReference (OwnVariable : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetOwnVariableRef (OwnVariable).SuspendsReference;
   end GetSuspendsReference;

   function ProcedureBodyIsHidden (Subprogram : in Dictionary.Symbol) return Boolean is
   begin
      return GetSubprogramRef (Subprogram).BodyIsHidden;
   end ProcedureBodyIsHidden;

   function TaskBodyIsHidden (TheTask : in Dictionary.Symbol) return Boolean is
   begin
      return GetTaskRef (GetTypeRef (TheTask).AncilliaryFields).BodyIsHidden;
   end TaskBodyIsHidden;

end RawDict; -- hidden because of non-SPARK code
