-------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
-------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--=============================================================================

separate (Sem.CompUnit.WalkStatements)
procedure Wf_Delay_Until
  (Node           : in     STree.SyntaxNode;
   Scope          : in     Dictionary.Scopes;
   Table          : in out RefList.HashTable;
   Component_Data : in out ComponentManager.ComponentData) is
   Enc_Unit_Sym : Dictionary.Symbol;
   Result_Type  : Exp_Record;
   Ref_Var      : SeqAlgebra.Seq;
   Exp_Node     : STree.SyntaxNode;
begin
   SeqAlgebra.CreateSeq (TheHeap, Ref_Var);
   Exp_Node := Next_Sibling (Current_Node => Child_Node (Current_Node => Node));
   -- ASSUME Exp_Node = expression
   SystemErrors.RT_Assert
     (C       => Syntax_Node_Type (Node => Exp_Node) = SP_Symbols.expression,
      Sys_Err => SystemErrors.Invalid_Syntax_Tree,
      Msg     => "Expect Exp_Node = expression in Wf_Delay_Until");
   -- Process the expression of the delay_statement. Must be of type Ada.Real_Time.
   Walk_Expression_P.Walk_Expression
     (Exp_Node                => Exp_Node,
      Scope                   => Scope,
      Type_Context            => Dictionary.GetPredefinedTimeType,
      Context_Requires_Static => False,
      Ref_Var                 => Ref_Var,
      Result                  => Result_Type,
      Component_Data          => Component_Data,
      The_Heap                => TheHeap);

   -- Verify that Result_Type is compatible with Ada.Real_Time
   if not Dictionary.CompatibleTypes
     (Scope => Scope,
      Left  => Result_Type.Type_Symbol,
      Right => Dictionary.GetPredefinedTimeType) then
      ErrorHandler.Semantic_Error
        (Err_Num   => 866,
         Reference => ErrorHandler.No_Reference,
         Position  => Node_Position (Node => Exp_Node),
         Id_Str    => LexTokenManager.Null_String);
   end if;

   -- Verify that if the enclosing unit is a procedure then
   -- it has a delay annotation. Other enclosing units are invalid.
   -- Also mark the enclosing unit as having seen a delay statement.
   Enc_Unit_Sym := Dictionary.GetEnclosingCompilationUnit (Scope);

   if Dictionary.IsProcedure (Enc_Unit_Sym) and then not Dictionary.IsOrIsInProtectedScope (Scope) then
      if Dictionary.HasDelayProperty (Enc_Unit_Sym) then
         Dictionary.MarkAccountsForDelay (Enc_Unit_Sym);
      else
         ErrorHandler.Semantic_Error
           (Err_Num   => 867,
            Reference => ErrorHandler.No_Reference,
            Position  => Node_Position (Node => Node),
            Id_Str    => Dictionary.GetSimpleName (Enc_Unit_Sym));
      end if;
   elsif not Dictionary.IsTaskType (Enc_Unit_Sym) then
      -- Cannot call delay from the enclosing construct.
      ErrorHandler.Semantic_Error
        (Err_Num   => 907,
         Reference => ErrorHandler.No_Reference,
         Position  => Node_Position (Node => Node),
         Id_Str    => LexTokenManager.Null_String);
   end if;

   -- NullVariable is an out stream and so need to be considered an import
   Add_Stream_Effects
     (Table    => Table,
      The_Heap => TheHeap,
      Node     => Node,
      Export   => Dictionary.GetNullVariable,
      Imports  => Ref_Var);
   -- add reference variable list to RefList hash table for the statement node
   RefList.AddRelation (Table, TheHeap, Node, Dictionary.GetNullVariable, -- ref vars dumped in data sink
                        Ref_Var);
end Wf_Delay_Until;
