/*   EXTRAITS DE LA LICENCE
	Copyright CEA, contributeurs : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)
  
	Adresse ml :
	BILLARD, non joignable par ml ;
	CALISTE, damien P caliste AT cea P fr.

	Ce logiciel est un programme informatique servant  visualiser des
	structures atomiques dans un rendu pseudo-3D. 

	Ce logiciel est rgi par la licence CeCILL soumise au droit franais et
	respectant les principes de diffusion des logiciels libres. Vous pouvez
	utiliser, modifier et/ou redistribuer ce programme sous les conditions
	de la licence CeCILL telle que diffuse par le CEA, le CNRS et l'INRIA 
	sur le site "http://www.cecill.info".

	Le fait que vous puissiez accder  cet en-tte signifie que vous avez 
	pris connaissance de la licence CeCILL, et que vous en avez accept les
	termes (cf. le fichier Documentation/licence.fr.txt fourni avec ce logiciel).
*/

/*   LICENCE SUM UP
	Copyright CEA, contributors : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)

	E-mail address:
	BILLARD, not reachable any more ;
	CALISTE, damien P caliste AT cea P fr.

	This software is a computer program whose purpose is to visualize atomic
	configurations in 3D.

	This software is governed by the CeCILL  license under French law and
	abiding by the rules of distribution of free software.  You can  use, 
	modify and/ or redistribute the software under the terms of the CeCILL
	license as circulated by CEA, CNRS and INRIA at the following URL
	"http://www.cecill.info". 

	The fact that you are presently reading this means that you have had
	knowledge of the CeCILL license and that you accept its terms. You can
	find a copy of this licence shipped with this software at Documentation/licence.en.txt.
*/
#include "panelElements.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <support.h>
#include <visu_object.h>
#include <visu_tools.h>
#include <visu_data.h>
#include <visu_elements.h>
#include <extraGtkFunctions/gtk_colorComboBoxWidget.h>
#include <extraGtkFunctions/gtk_toolPanelWidget.h>
#include <coreTools/toolColor.h>
#include <openGLFunctions/light.h>

#include "gtkAtomic.h"
#include "gtkSpin.h"

/* Functions used to initialise the
 * specific area in the panel of elements.
 */
static VisuToolsInitFunc listInitRendenringGtkPanelFunc[] = {
  panelElementAtomicInit_gtkPanel,
  panelElementSpinInit_gtkPanel,
  (VisuToolsInitFunc)0};

GtkListStore *listStoreVisuElements;
enum
  {
    COLUMN_NAME,  /* The label shown */
    COLUMN_POINTER_TO_DATA,  /* Pointer to the VisuElement. */
    N_COLUMNS
  };
/* sensitive widgets */
GtkWidget *vBoxElements;
GtkWidget *renderingMethodElements;
GtkWidget *elementsCombo;
static GtkWidget *checkRendered;
static GtkWidget *checkMasked;
GtkWidget *widgetVisuElementColor;
gint sortVisuElements(gpointer a, gpointer b);
char *getLabelOfElement(VisuElement *element);

/* Structure to store the hooks that are required for
   the specific zone for elements. */
struct GtkRenderingInterface_struct
{
/*   setSensitiveFunc sensitiveInterface; */
  panelElements_changeFunc newElementInterface;
  createGtkWidgetFunc createInterface;
};
typedef struct GtkRenderingInterface_struct GtkRenderingInterface;
GHashTable *listOfRenderingInterfaces;
panelElements_changeFunc updateRenderingMethodResources;

/* Local methods. */
void createInteriorElements(GtkWidget *toolPanel);
static void sendElementToSpecificArea();
void createCallBacksElements();
void setInitialValuesElements();
char *getDefaultLabelOfElement(VisuElement *element);
/* It adds the interface widget in the subPanel. The variable
   renderingMethodElements pints then to this widget. If
   renderingMethodElements has already a target, this target is
   freed via gtk_widget_destroy. */
void changeRenderingMethodInterface(GtkWidget* interface);
/* It allows the client to set the method use to update
   user defined values. */
void setElementChangedFunc(panelElements_changeFunc func);
/* It reads the value of the interface methods. */
void getRenderingInterface(RenderingMethod *method,
			   panelElements_changeFunc *change,
			   createGtkWidgetFunc *create);

/* Callbacks */
guint idRefreshComboElements;
static void comboVisuElementsChanged(GtkComboBox *combobox, gpointer data);
static void resourcesChanged(GObject *object, VisuData *dataObj, gpointer data);
void colorVisuElementChanged(ColorComboBox *colorWd, Color *color, gpointer userData);
static void checkRenderedChanged(GtkToggleButton *button, gpointer data);
static void checkMaskedChanged(GtkToggleButton *button, gpointer data);
void refreshComboElements(GObject *obj, VisuData *dataObj, gpointer userData);
static void onRenderingMethodChanged(GObject *obj, RenderingMethod *method,
				     gpointer userData);
static void onColorChanged(ColorComboBox *colorComboBox, guint colorId, gpointer data);
static void onMaterialChanged(ColorComboBox *colorComboBox, Material mat, gpointer data);

ToolPanel* panelElementsInit()
{
  gchar *cl = _("Set elements caracteristics");
  gchar *tl = _("Elements");
  int i;

  panelElements = toolPanelNew_withIconFromPath("Panel_elements", cl, tl,
						"stock-elements_20.png");
  createInteriorElements(panelElements);
  toolPanelSet_dockable(TOOL_PANEL(panelElements), TRUE);

  if (!panelElements)
    return (ToolPanel*)0;

  /* Create the callbacks of all the sensitive widgets. */
  createCallBacksElements();

  /* Create the list of the available rendering method
     interfaces. */
  listOfRenderingInterfaces = g_hash_table_new_full(g_direct_hash,
						    g_direct_equal,
						    NULL, NULL);
  g_return_val_if_fail(listOfRenderingInterfaces, (ToolPanel*)0);

  for (i = 0; listInitRendenringGtkPanelFunc[i]; i++)
    listInitRendenringGtkPanelFunc[i]();

  return TOOL_PANEL(panelElements);
}

void createInteriorElements(GtkWidget *toolPanel)
{
  GtkWidget *label, *expand, *image;
  GtkWidget *hbox;
  GtkWidget *align;
  GtkWidget *scrollView;
  GtkCellRenderer *renderer;
  GtkTooltips *tooltips;

  tooltips = gtk_tooltips_new ();

  /* We create the structure that store the VisuElements */
  listStoreVisuElements = gtk_list_store_new (N_COLUMNS,
					      G_TYPE_STRING,
					      G_TYPE_POINTER);
  
  scrollView = gtk_scrolled_window_new((GtkAdjustment*)0,
				       (GtkAdjustment*)0);
  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrollView),
				  GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
  gtk_scrolled_window_set_shadow_type (GTK_SCROLLED_WINDOW (scrollView), GTK_SHADOW_NONE);
  vBoxElements = gtk_vbox_new(FALSE, 0);
  gtk_widget_set_sensitive(vBoxElements, FALSE);
  gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(scrollView), vBoxElements);


  hbox = gtk_hbox_new(FALSE, 0);
  gtk_box_pack_start(GTK_BOX(vBoxElements), hbox, FALSE, FALSE, 5);
  label = gtk_label_new(_("<b>Set caracteristics of: </b>"));
  gtk_widget_set_name(label, "label_head");
  gtk_misc_set_alignment(GTK_MISC(label), 0., 0.5);
  gtk_label_set_use_markup(GTK_LABEL(label), TRUE);
  gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 2);
  /* We create the tree widget that show the methods. */
  elementsCombo = gtk_combo_box_new_with_model(GTK_TREE_MODEL(listStoreVisuElements));
  renderer = gtk_cell_renderer_text_new();
  gtk_cell_layout_pack_start(GTK_CELL_LAYOUT(elementsCombo), renderer, FALSE);
  gtk_cell_layout_add_attribute(GTK_CELL_LAYOUT(elementsCombo), renderer, "text", 0);
  gtk_box_pack_start(GTK_BOX(hbox), elementsCombo, TRUE, TRUE, 2);

  label = gtk_label_new("");
  gtk_label_set_markup(GTK_LABEL(label), _("<b>Standard resources</b>"));
  gtk_misc_set_alignment(GTK_MISC(label), 0., 0.5);
  gtk_widget_set_name(label, "label_head_2");
  gtk_box_pack_start(GTK_BOX(vBoxElements), label, FALSE, FALSE, 5);

  hbox = gtk_hbox_new(FALSE, 0);
  gtk_box_pack_start(GTK_BOX(vBoxElements), hbox, FALSE, FALSE, 0);
/*   label = gtk_label_new(_("Color: ")); */
/*   gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 2); */
  widgetVisuElementColor = colorComboBox_newWithRanges(TRUE);
  gtk_box_pack_start(GTK_BOX(hbox), widgetVisuElementColor, FALSE, FALSE, 2);
  align = gtk_alignment_new(1., 0.5, 0., 0.);
  gtk_box_pack_start(GTK_BOX(hbox), align, TRUE, TRUE, 2);
  checkRendered = gtk_check_button_new_with_label(_("rendered"));
  gtk_container_add(GTK_CONTAINER(align), checkRendered);
  align = gtk_alignment_new(0.5, 0.5, 0., 0.);
  gtk_box_pack_start(GTK_BOX(hbox), align, FALSE, FALSE, 2);
  checkMasked = gtk_check_button_new();
  gtk_container_add(GTK_CONTAINER(align), checkMasked);
  image = create_pixmap((GtkWidget*)0, "stock-masking.png");
  gtk_container_add(GTK_CONTAINER(checkMasked), image);
  gtk_tooltips_set_tip(tooltips, checkMasked,
		       _("Make nodes sensitive to the masking effect of planes."),
		       NULL);

  expand = colorComboBoxGet_rangesWidgets(COLOR_COMBOX(widgetVisuElementColor));
  gtk_box_pack_start(GTK_BOX(vBoxElements), expand, FALSE, FALSE, 0);

  label = gtk_label_new(_("<b>Rendering specific resources</b>"));
  gtk_widget_set_name(label, "label_head_2");
  gtk_misc_set_alignment(GTK_MISC(label), 0., 0.5);
  gtk_label_set_use_markup(GTK_LABEL(label), TRUE);
  gtk_box_pack_start(GTK_BOX(vBoxElements), label, FALSE, FALSE, 5);

  renderingMethodElements = (GtkWidget*)0;

  gtk_widget_show_all(scrollView);

  gtk_container_add(GTK_CONTAINER(toolPanel), scrollView);
}

void createCallBacksElements()
{
  idRefreshComboElements = g_signal_connect(G_OBJECT(elementsCombo), "changed",
					    G_CALLBACK(comboVisuElementsChanged), (gpointer)0);
  g_signal_connect(G_OBJECT(widgetVisuElementColor), "color-selected",
		   G_CALLBACK(colorVisuElementChanged), (gpointer)0);
  g_signal_connect(G_OBJECT(widgetVisuElementColor), "material-value-changed",
		   G_CALLBACK(onMaterialChanged), (gpointer)0);
  g_signal_connect(G_OBJECT(widgetVisuElementColor), "color-value-changed",
		   G_CALLBACK(onColorChanged), (gpointer)0);
  g_signal_connect(G_OBJECT(checkRendered), "toggled",
		   G_CALLBACK(checkRenderedChanged), (gpointer)0);
  g_signal_connect(G_OBJECT(checkMasked), "toggled",
		   G_CALLBACK(checkMaskedChanged), (gpointer)0);
  /* Connect the signal emitted by visu when a new file is loaded. */
  g_signal_connect(G_OBJECT(visu), "dataReadyForRendering",
		   G_CALLBACK(refreshComboElements), (gpointer)0);
  /* Connect the signal emitted by visu when
     a new rendering method is selected. */
  g_signal_connect(G_OBJECT(visu), "renderingChanged",
		   G_CALLBACK(onRenderingMethodChanged), (gpointer)0);
  /* Connect the signal emitted by visu when
     a new rendering method is selected. */
  g_signal_connect(G_OBJECT(visu), "resourcesLoaded",
		   G_CALLBACK(resourcesChanged), (gpointer)0);
}

/*****************/
/* Miscellaneous */
/*****************/
gint sortVisuElements(gpointer a, gpointer b)
{
  return strcmp(((VisuElement*)a)->name, ((VisuElement*)b)->name);
}

/*************/
/* Callbacks */
/*************/
void refreshComboElements(GObject *obj _U_, VisuData *dataObj, gpointer userData _U_)
{
  unsigned int i;
  GList *tmpLst, *currentListOfElements;
  char *label;
  GtkTreeIter iter, selectedIter;
  Color *color;
  int posColor;
  float *rgba;
  VisuElement *prevSelectedElement;
  gboolean selected;

  DBG_fprintf(stderr, "Panel Element : catch 'dataReadyForRendering' signal,"
	      " trying to rebuild element list.\n");
  if (dataObj)
    {
      if (visuDataGet_changeElementFlag(dataObj))
	{
	  DBG_fprintf(stderr,"Panel Element : not removing old elements"
		      " from the liststore of labels.\n");
	  return;
	}
    }

  DBG_fprintf(stderr,"Panel Element : removing old elements"
	      " from the liststore of labels.\n");

  /* Get the selected iter. */
  selected = gtk_combo_box_get_active_iter(GTK_COMBO_BOX(elementsCombo), &iter);
  if (selected)
    gtk_tree_model_get(GTK_TREE_MODEL(listStoreVisuElements), &iter,
		       COLUMN_POINTER_TO_DATA, &prevSelectedElement, -1);
  else
    prevSelectedElement = (VisuElement*)0;

  /* Clear old list, and clean all pointers. */
  g_signal_handler_block(G_OBJECT(elementsCombo),
			 idRefreshComboElements);
  gtk_list_store_clear(listStoreVisuElements);
  currentListOfElements = (GList*)0;
  g_signal_handler_unblock(G_OBJECT(elementsCombo),
			   idRefreshComboElements);

  if (dataObj)
    {
      DBG_fprintf(stderr,"Panel Element : creating %d labels for the"
		  " combobox of VisuElements.\n", dataObj->ntype);
      for(i = 0; i < dataObj->ntype; i++)
	currentListOfElements = g_list_prepend(currentListOfElements,
					       (gpointer)dataObj->fromIntToVisuElement[i]);

      gtk_widget_set_sensitive(vBoxElements, TRUE);
    }
  else
    {
      DBG_fprintf(stderr, "Panel Element : there is no element available.\n");
      gtk_widget_set_sensitive(vBoxElements, FALSE);
    }
  currentListOfElements = g_list_sort(currentListOfElements,
				      (GCompareFunc)sortVisuElements);

  /* Add a first item that apply changes on all elements. */
  gtk_list_store_append(listStoreVisuElements, &iter);
  gtk_list_store_set(listStoreVisuElements, &iter,
		     COLUMN_NAME, _("All elements"),
		     COLUMN_POINTER_TO_DATA, (gpointer)0,
		     -1);
  tmpLst = currentListOfElements;
  while(tmpLst)
    {
      label = getLabelOfElement((VisuElement*)tmpLst->data);
      rgba = ((VisuElement*)tmpLst->data)->rgb;
      color = colorGet_byValues(&posColor, rgba[0], rgba[1], rgba[2], rgba[3]);
      gtk_list_store_append(listStoreVisuElements, &iter);
      gtk_list_store_set(listStoreVisuElements, &iter,
			 COLUMN_NAME, label,
			 COLUMN_POINTER_TO_DATA, tmpLst->data,
			 -1);
      g_free(label);
      if (tmpLst == currentListOfElements) /* If first. */
	selectedIter = iter;
      if (tmpLst->data == (gpointer)prevSelectedElement)
	selectedIter = iter;
      currentListOfElements = g_list_prepend(currentListOfElements, tmpLst->data);
      tmpLst = g_list_next(tmpLst);
    }
  if (currentListOfElements)
    gtk_combo_box_set_active_iter(GTK_COMBO_BOX(elementsCombo), &selectedIter);

  DBG_fprintf(stderr,"Panel Element : Refresh OK.\n");
}
static void onRenderingMethodChanged(GObject *obj _U_, RenderingMethod *method,
				     gpointer userData _U_)
{
/*   setSensitiveFunc sensitive; */
  panelElements_changeFunc change;
  createGtkWidgetFunc create;
  int sans;

  sans = 1;
  if (method)
    {
      getRenderingInterface(method, &change, &create);
      if (change && create)
	{
	  sans = 0;
	  DBG_fprintf(stderr,"Panel Element : set the gtk interface for"
		      " the method '%s'.\n", method->name);
	  setElementChangedFunc(change);
	  changeRenderingMethodInterface(create());
	}
    }
  if (sans)
    {
      DBG_fprintf(stderr,"Panel Element : set the gtk interface to null.\n");
      setElementChangedFunc((panelElements_changeFunc)0);
      changeRenderingMethodInterface((GtkWidget*)0);
    }
}
static void comboVisuElementsChanged(GtkComboBox *combobox, gpointer data _U_)
{
  GtkTreeModel *tree;
  gboolean selected;
  GtkTreeIter iter;
  VisuElement *ele;

  ele = (VisuElement*)0;
  selected = gtk_combo_box_get_active_iter(combobox, &iter);
  if (selected)
    {
      tree = gtk_combo_box_get_model(combobox);
      gtk_tree_model_get(tree, &iter,
			 COLUMN_POINTER_TO_DATA, &ele, -1);

    }
  else
    {
      g_warning("Can't find the selected VisuElement in the"
		" GtkComboBox of the Panel Element.");
      return;
    }

  if (ele)
    {
      DBG_fprintf(stderr, "Panel Element : set the selected VisuElement to '%s'.\n", ele->name);
      colorComboBoxSet_color(COLOR_COMBOX(widgetVisuElementColor), ele->rgb, FALSE);
      colorComboBoxSet_material(COLOR_COMBOX(widgetVisuElementColor), ele->material, FALSE);
      gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(checkRendered),
				   visuElementGet_rendered(ele));
      gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(checkMasked),
				   visuElementGet_sensitiveToPlanes(ele));
    }

  /* Force refresh on the specific area. */
  sendElementToSpecificArea();
}
void colorVisuElementChanged(ColorComboBox *colorWd _U_, Color *color,
			     gpointer userData _U_)
{
  int res;
  int selected, refresh;
  GtkTreeIter iter;
  VisuData *dataObj;
  VisuElement *currentSelectedElement;

  DBG_fprintf(stderr, "Panel Elements : Catch 'color-selected' signal (%p).\n", (gpointer) color);

  /* Get the selected iter. */
  selected = gtk_combo_box_get_active_iter(GTK_COMBO_BOX(elementsCombo), &iter);
  g_return_if_fail(selected);
    
  /* If currentSelectedElement is NULL we apply the changes
     on all elements. */
  dataObj = toolPanelGet_visuData(TOOL_PANEL(panelElements));

  /* Get a pointer on the current visuElement */
  gtk_tree_model_get(GTK_TREE_MODEL(listStoreVisuElements), &iter,
		     COLUMN_POINTER_TO_DATA, &currentSelectedElement, -1);

  refresh = FALSE;
  if (!currentSelectedElement)
    {
      /* Change the color for all elements and
	 unset the stored color for all the GtkListStore */
      selected = gtk_tree_model_get_iter_first(GTK_TREE_MODEL(listStoreVisuElements), &iter);
      while (selected)
	{
	  gtk_tree_model_get(GTK_TREE_MODEL(listStoreVisuElements), &iter,
			     COLUMN_POINTER_TO_DATA, &currentSelectedElement, -1);
	  if (currentSelectedElement)
	    {
	      res = visuElementSet_allRGBValues(currentSelectedElement, color->rgba);
	      if (res > 0)
		visuData_createNodes(dataObj, currentSelectedElement);
	      refresh = refresh || (res != 0);
	    }
	  selected = gtk_tree_model_iter_next(GTK_TREE_MODEL(listStoreVisuElements), &iter);
	}
    }
  else
    {
      /* Change the color for the selected element only. */
      res = visuElementSet_allRGBValues(currentSelectedElement, color->rgba);
      if (res > 0)
	visuData_createNodes(dataObj, currentSelectedElement);
      refresh = (res != 0);
    }
  if (refresh)
    {
      DBG_fprintf(stderr, "Panel Element : color change, refresh.\n");
      g_signal_emit (visu, VISU_GET_CLASS (visu)->OpenGLAskForReDraw_signal_id,
		     0 , NULL);
    }
}
static void onColorChanged(ColorComboBox *colorComboBox, guint colorId,
			   gpointer data _U_)
{
  int res;
  gboolean selected, refresh;
  GtkTreeIter iter;
  VisuData *dataObj;
  VisuElement *currentSelectedElement;
  float *rgbVal;

  /* Get the selected iter. */
  selected = gtk_combo_box_get_active_iter(GTK_COMBO_BOX(elementsCombo), &iter);
  g_return_if_fail(selected);
    
  /* Get the values. */
  rgbVal = colorComboBoxGet_color(colorComboBox);

  /* If currentSelectedElement is NULL we apply the changes
     on all elements. */
  dataObj = toolPanelGet_visuData(TOOL_PANEL(panelElements));

  /* Get a pointer on the current visuElement */
  gtk_tree_model_get(GTK_TREE_MODEL(listStoreVisuElements), &iter,
		     COLUMN_POINTER_TO_DATA, &currentSelectedElement, -1);

  refresh = FALSE;
  if (!currentSelectedElement)
    {
      /* Change the color for all elements and
	 unset the stored color for all the GtkListStore */
      selected = gtk_tree_model_get_iter_first(GTK_TREE_MODEL(listStoreVisuElements), &iter);
      while (selected)
	{
	  gtk_tree_model_get(GTK_TREE_MODEL(listStoreVisuElements), &iter,
			     COLUMN_POINTER_TO_DATA, &currentSelectedElement, -1);
	  if (currentSelectedElement)
	    {
	      res = visuElementSet_allRGBValues(currentSelectedElement, rgbVal);
	      if (res > 0)
		visuData_createNodes(dataObj, currentSelectedElement);
	      refresh = refresh || (res != 0);
	    }
	  selected = gtk_tree_model_iter_next(GTK_TREE_MODEL(listStoreVisuElements), &iter);
	}
    }
  else
    {
      /* Change the color for the selected element only. */
      res = visuElementSet_RGBValue(currentSelectedElement, colorId,
				    rgbVal[colorId]);
      if (res > 0)
	visuData_createNodes(dataObj, currentSelectedElement);
      refresh = (res != 0);
    }
  if (refresh)
    {
      DBG_fprintf(stderr, "Panel Element : RGB change, refresh.\n");
      g_signal_emit (visu, VISU_GET_CLASS (visu)->OpenGLAskForReDraw_signal_id,
		     0 , NULL);
    }
  g_free(rgbVal);
}
static void onMaterialChanged(ColorComboBox *colorComboBox, Material mat,
			      gpointer data _U_)
{
  int res;
  int selected, refresh;
  GtkTreeIter iter;
  VisuData *dataObj;
  VisuElement *currentSelectedElement;
  float *matVal;

  /* Get the selected iter. */
  selected = gtk_combo_box_get_active_iter(GTK_COMBO_BOX(elementsCombo), &iter);
  g_return_if_fail(selected);
    
  /* Get the values. */
  matVal = colorComboBoxGet_material(colorComboBox);

  /* If currentSelectedElement is NULL we apply the changes
     on all elements. */
  dataObj = toolPanelGet_visuData(TOOL_PANEL(panelElements));

  /* Get a pointer on the current visuElement */
  gtk_tree_model_get(GTK_TREE_MODEL(listStoreVisuElements), &iter,
		     COLUMN_POINTER_TO_DATA, &currentSelectedElement, -1);

  refresh = FALSE;
  if (!currentSelectedElement)
    {
      /* Change the material for all elements. */
      selected = gtk_tree_model_get_iter_first(GTK_TREE_MODEL(listStoreVisuElements), &iter);
      while (selected)
	{
	  gtk_tree_model_get(GTK_TREE_MODEL(listStoreVisuElements), &iter,
			     COLUMN_POINTER_TO_DATA, &currentSelectedElement, -1);
	  if (currentSelectedElement)
	    {
	      res = visuElementSet_allMaterialValues(currentSelectedElement, matVal);
	      if (res > 0)
		visuData_createNodes(dataObj, currentSelectedElement);
	      refresh = refresh || (res != 0);
	    }
	  selected = gtk_tree_model_iter_next(GTK_TREE_MODEL(listStoreVisuElements), &iter);
	}
    }
  else
    {
      /* Change the color for the selected element only. */
      res = visuElementSet_materialValue(currentSelectedElement, mat, matVal[mat]);
      if (res > 0)
	visuData_createNodes(dataObj, currentSelectedElement);
      refresh = (res != 0);
    }
  if (refresh)
    {
      DBG_fprintf(stderr, "Panel Element : material change, refresh.\n");
      g_signal_emit (visu, VISU_GET_CLASS (visu)->OpenGLAskForReDraw_signal_id,
		     0 , NULL);
    }
  g_free(matVal);
}

static void resourcesChanged(GObject *object _U_, VisuData *dataObj _U_,
			     gpointer data _U_)
{
  gboolean validIter;
  GtkTreeIter iter;
  VisuElement *ele;

  /* The resources loading could have changed Color of the element,
     so we check all the list. */
  DBG_fprintf(stderr, "Panel Elements : catch the 'resourcesChanged'"
	      " signal, checking color pointers.\n");

  /* Get the selected iter. */
  validIter = gtk_combo_box_get_active_iter(GTK_COMBO_BOX(elementsCombo), &iter);
  if (!validIter)
    return;
  /* Get a pointer on the current visuElement */
  gtk_tree_model_get(GTK_TREE_MODEL(listStoreVisuElements), &iter,
		     COLUMN_POINTER_TO_DATA, &ele, -1);
  if (!ele)
    return;

  colorComboBoxSet_color(COLOR_COMBOX(widgetVisuElementColor), ele->rgb, FALSE);
  colorComboBoxSet_material(COLOR_COMBOX(widgetVisuElementColor), ele->material, FALSE);

  /* Force refresh on the specific area. */
  sendElementToSpecificArea();
}
static void checkMaskedChanged(GtkToggleButton *button, gpointer data _U_)
{
  gboolean validIter, setValue, refresh;
  GtkTreeIter iter;
  VisuData *dataObj;
  VisuElement *ele;

  dataObj = toolPanelGet_visuData(TOOL_PANEL(panelElements));

  /* Get the selected iter. */
  validIter = gtk_combo_box_get_active_iter(GTK_COMBO_BOX(elementsCombo), &iter);
  g_return_if_fail(validIter);

  /* Get a pointer on the current visuElement */
  gtk_tree_model_get(GTK_TREE_MODEL(listStoreVisuElements), &iter,
		     COLUMN_POINTER_TO_DATA, &ele, -1);
  setValue = gtk_toggle_button_get_active(button);

  refresh = FALSE;
  if (ele)
    /* We change one specific element. */
    refresh = visuElementSet_sensitiveToPlanes(ele, setValue);
  else
    {
      /* We change for all elements. */
      validIter = gtk_tree_model_get_iter_first
	(GTK_TREE_MODEL(listStoreVisuElements), &iter);
      while (validIter)
	{
	  gtk_tree_model_get(GTK_TREE_MODEL(listStoreVisuElements), &iter,
			     COLUMN_POINTER_TO_DATA, &ele, -1);
	  if (ele)
	    refresh = visuElementSet_sensitiveToPlanes(ele, setValue) || refresh;
	  validIter = gtk_tree_model_iter_next
	    (GTK_TREE_MODEL(listStoreVisuElements), &iter);
	}
    }
  if (refresh)
    {
      visuDataEmit_askForShowHideNodes(dataObj, &refresh);

      if (refresh)
	{
	  visuDataEmit_nodeRenderedChange(dataObj);
	  visuData_createAllNodes(dataObj);
	  g_idle_add(visuObjectRedraw, (gpointer)0);
	}
    }
}
static void checkRenderedChanged(GtkToggleButton *button, gpointer data _U_)
{
  int res;
  gboolean validIter, setValue, refresh;
  GtkTreeIter iter;
  VisuData *dataObj;
  VisuElement *ele;

  dataObj = toolPanelGet_visuData(TOOL_PANEL(panelElements));

  /* Get the selected iter. */
  validIter = gtk_combo_box_get_active_iter(GTK_COMBO_BOX(elementsCombo), &iter);
  g_return_if_fail(validIter);

  /* Get a pointer on the current visuElement */
  gtk_tree_model_get(GTK_TREE_MODEL(listStoreVisuElements), &iter,
		     COLUMN_POINTER_TO_DATA, &ele, -1);

  setValue = gtk_toggle_button_get_active(button);
  refresh = FALSE;
  if (ele)
    {
      res = visuElementSet_rendered(ele, setValue);
      if (res)
	{
	  refresh = TRUE;
	  visuDataEmit_elementRenderedChange(dataObj, ele);
	  visuData_createNodes(dataObj, ele);
	}
    }
  else
    {
      validIter = gtk_tree_model_get_iter_first(GTK_TREE_MODEL(listStoreVisuElements), &iter);
      while (validIter)
	{
	  gtk_tree_model_get(GTK_TREE_MODEL(listStoreVisuElements), &iter,
			     COLUMN_POINTER_TO_DATA, &ele, -1);
	  if (ele)
	    {
	      res = visuElementSet_rendered(ele, setValue);
	      if (res)
		{
		  refresh = TRUE;
		  visuDataEmit_elementRenderedChange(dataObj, ele);
		  visuData_createNodes(dataObj, ele);
		}
	    }
	  validIter = gtk_tree_model_iter_next(GTK_TREE_MODEL(listStoreVisuElements), &iter);
	}
    }
  if (refresh)
    g_idle_add(visuObjectRedraw, (gpointer)0);
}


/********************/
/* Private methods. */
/********************/
char *getLabelOfElement(VisuElement *element)
{
  GString *str;
  char *res;

  str = g_string_new("");
  g_string_append_printf(str, _("Element '%s'"), element->name);
  res = str->str;
  g_string_free(str, FALSE);
  return res;
}
static void sendElementToSpecificArea()
{
  gboolean validIter;
  GtkTreeIter iter;
  VisuElement *ele;
  GList *tmpLst;

  if (!updateRenderingMethodResources)
    return;

  /* Get the selected iter. */
  validIter = gtk_combo_box_get_active_iter(GTK_COMBO_BOX(elementsCombo), &iter);
  if (!validIter)
    return;

  /* Get a pointer on the current visuElement */
  gtk_tree_model_get(GTK_TREE_MODEL(listStoreVisuElements), &iter,
		     COLUMN_POINTER_TO_DATA, &ele, -1);

  tmpLst = (GList*)0;
  if (ele)
    tmpLst = g_list_prepend(tmpLst, (gpointer)ele);
  else
    {
      validIter = gtk_tree_model_get_iter_first(GTK_TREE_MODEL(listStoreVisuElements), &iter);
      while (validIter)
	{
	  gtk_tree_model_get(GTK_TREE_MODEL(listStoreVisuElements), &iter,
			     COLUMN_POINTER_TO_DATA, &ele, -1);
	  if (ele)
	    tmpLst = g_list_prepend(tmpLst, (gpointer)ele);
	  validIter = gtk_tree_model_iter_next(GTK_TREE_MODEL(listStoreVisuElements), &iter);
	}
    }
  updateRenderingMethodResources(tmpLst);
  g_list_free(tmpLst);
}


/*******************/
/* Public Methods. */
/*******************/
/* It allows the client to set the method use to update
   user defined values. */
void setElementChangedFunc(panelElements_changeFunc func)
{
  updateRenderingMethodResources = func;
}
/* It adds the interface widget in the subPanel. The variable
   renderingMethodElements pints then to this widget. If
   renderingMethodElements has already a target, this target is
   freed via gtk_widget_destroy. If interface is a null pointer
   the previous renderingMethodElements is removed, but
   nothing replaces it. */
void changeRenderingMethodInterface(GtkWidget* interface)
{
  DBG_fprintf(stderr, "Panel Element : caught 'resourcesLoaded' signal.\n");
  if (renderingMethodElements)
    {
      /* note that gtk_destroy automatically remove
	 renderingMethodElements for its container, so
	 a call to gtk_container_remove is not necessary. */
      DBG_fprintf(stderr, "Panel Element : removing old rendering specific widget.\n");
      gtk_widget_destroy(renderingMethodElements);
    }

  renderingMethodElements = interface;

  if (interface)
    {
      gtk_box_pack_start(GTK_BOX(vBoxElements), interface,
			 FALSE, FALSE, 5);
      gtk_box_reorder_child(GTK_BOX(vBoxElements), interface, 6);
      gtk_widget_show(interface);
    }

  /* Force refresh on the specific area. */
  sendElementToSpecificArea();
}


/* RenderingMethod gtk interface. */

/* This method allows to initiate the method
   to deal with the interface. */
void panelElements_setInterfaceMethods(RenderingMethod *method,
				      panelElements_changeFunc change,
				      createGtkWidgetFunc create)
{
  GtkRenderingInterface *interface;

  if (!method)
    return;

  /* Search for an already existing GtkRenderingInterface
     for this name. */
  interface = g_hash_table_lookup(listOfRenderingInterfaces,
				  (gpointer)method);
  if (interface)
    {
      interface->newElementInterface = change;
      interface->createInterface = create;
    }
  else
    {
      interface = g_malloc(sizeof(GtkRenderingInterface));
      interface->newElementInterface = change;
      interface->createInterface = create;
      g_hash_table_insert(listOfRenderingInterfaces,
			  (gpointer)method,
			  (gpointer)interface);
    }
}
/* It reads the value of the interface methods. */
void getRenderingInterface(RenderingMethod *method,
			   panelElements_changeFunc *change,
			   createGtkWidgetFunc *create)
{
  GtkRenderingInterface *interface;

  *change = (panelElements_changeFunc)0;
  *create = (createGtkWidgetFunc)0;
  if (!method)
    return;

  interface = g_hash_table_lookup(listOfRenderingInterfaces,
				  (gpointer)method);
  if (interface)
    {
      *change = interface->newElementInterface;
      *create = interface->createInterface;
    }
}
