/*   EXTRAITS DE LA LICENCE
	Copyright CEA, contributeurs : Damien
	CALISTE, laboratoire L_Sim, (2001-2005)
  
	Adresse ml :
	CALISTE, damien P caliste AT cea P fr.

	Ce logiciel est un programme informatique servant  visualiser des
	structures atomiques dans un rendu pseudo-3D. 

	Ce logiciel est rgi par la licence CeCILL soumise au droit franais et
	respectant les principes de diffusion des logiciels libres. Vous pouvez
	utiliser, modifier et/ou redistribuer ce programme sous les conditions
	de la licence CeCILL telle que diffuse par le CEA, le CNRS et l'INRIA 
	sur le site "http://www.cecill.info".

	Le fait que vous puissiez accder  cet en-tte signifie que vous avez 
	pris connaissance de la licence CeCILL, et que vous en avez accept les
	termes (cf. le fichier Documentation/licence.fr.txt fourni avec ce logiciel).
*/

/*   LICENCE SUM UP
	Copyright CEA, contributors : Damien
	CALISTE, laboratoire L_Sim, (2001-2005)

	E-mail address:
	CALISTE, damien P caliste AT cea P fr.

	This software is a computer program whose purpose is to visualize atomic
	configurations in 3D.

	This software is governed by the CeCILL  license under French law and
	abiding by the rules of distribution of free software.  You can  use, 
	modify and/ or redistribute the software under the terms of the CeCILL
	license as circulated by CEA, CNRS and INRIA at the following URL
	"http://www.cecill.info". 

	The fact that you are presently reading this means that you have had
	knowledge of the CeCILL license and that you accept its terms. You can
	find a copy of this licence shipped with this software at Documentation/licence.en.txt.
*/
#include "dumpToSVG.h"

#ifdef HAVE_CAIRO

#include <math.h>
#include <string.h>
#include <glib.h>
#include <cairo.h>
#include <cairo-svg.h>
#include <GL/gl.h>

#if CAIRO_VERSION_MINOR > 1

#include <visu_tools.h>
#include <visu_rendering.h>
#include <visu_pairs.h>
#include <opengl.h>
#include <openGLFunctions/light.h>
#include <openGLFunctions/view.h>
#include <extensions/box.h>
#include <extensions/fogAndBGColor.h>
#include <extensions/axes.h>
#include <extensions/legend.h>

#define DEBUG_USE_FOG 0

/**
 * SECTION:dumpToSVG
 * @short_description: add an export capability into SVG files.
 * @include: extensions/box.h, extensions/axes.h, visu_pairs.h and visu_data.h
 *
 * This provides a write routine to export V_Sim views into SVG
 * files. Currently, this is an experimental feature. Not all V_Sim
 * elements are rendered, only the nodes, the box, the pairs and the
 * axes. All the characteristics are not used (no line stipple for
 * instance). In spin mode, nodes are only atomic.
 *
 * Since: 3.4
 */

struct _pairs
{
  VisuPairData *data;
  VisuNode *node1, *node2;
  VisuElement *ele1, *ele2;
  float alpha;
};

static gboolean writeViewInSvgFormat(FileFormat *format, const char* filename,
				     int width, int height, VisuData *dataObj,
				     guchar* imageData, GError **error,
				     voidDataFunc functionWait, gpointer data);

DumpType* dumpToSVG_init()
{
  DumpType *svg;
  char *typeSVG[] = {"*.svg", (char*)0};
#define descrSVG _("Scalar Vector Graphic (SVG) file")
  FileFormat* fmt;

  svg = g_malloc(sizeof(DumpType));
  fmt = fileFormatNew(descrSVG, typeSVG);
  if (!fmt)
    {
      g_error("Can't initialize the SVG dump module, aborting.\n");
    }

  svg->bitmap    = FALSE;
  svg->hasAlpha  = FALSE;
  svg->fileType  = fmt;
  svg->writeFunc = writeViewInSvgFormat;

  fileFormatAdd_propertyBoolean(fmt, "use_flat_rendering",
				_("Use flat colours for scheme rendering"), FALSE);
  
  return svg;
}

static void sort_by_z(float *coordinates, float *buffer,
		      int n, int z, int begin, int end)
{
  int i;
  int middle;

  if( begin >= end ) return;
  
  memcpy(buffer, coordinates + begin * n, sizeof(float) * n);
  memcpy(coordinates + begin * n, coordinates + (end + begin) / 2 * n, sizeof(float) * n);
  memcpy(coordinates + (end + begin) / 2 * n, buffer, sizeof(float) * n);

  middle = begin;
  for(i = begin + 1; i <= end; i++)
    {
      if ( coordinates[i * n + z] > coordinates[begin * n + z] )
	{
	  middle += 1;
	  memcpy(buffer, coordinates + i * n, sizeof(float) * n);
	  memcpy(coordinates + i * n, coordinates + middle * n, sizeof(float) * n);
	  memcpy(coordinates + middle * n, buffer, sizeof(float) * n);
	}
    }
  memcpy(buffer, coordinates + begin * n, sizeof(float) * n);
  memcpy(coordinates + begin * n, coordinates + middle * n, sizeof(float) * n);
  memcpy(coordinates + middle * n, buffer, sizeof(float) * n);
  sort_by_z(coordinates, buffer, n, z, begin, middle - 1);
  sort_by_z(coordinates, buffer, n, z, middle + 1, end);
}

gboolean writeDataToCairoSurface(cairo_surface_t *cairo_surf, FileFormat *format,
				 VisuData *dataObj, GError **error,
				 voidDataFunc functionWait _U_,
				 gpointer user_data _U_)
{
  cairo_t *cr;
  cairo_status_t status;
  cairo_pattern_t **pat;
#define FONT_SIZE 16.
  cairo_matrix_t scale = {1., 0., 0., -1., 0., 0.};
  cairo_matrix_t scaleFont = {FONT_SIZE, 0., 0., -FONT_SIZE, 0., 0.};
  VisuDataIter iter;
  int i, nNodes, nEle;
  float xyz[3], radius, norm, *rgb, rgbaFog[4], rgbaBg[4];
  GLfloat *coordinates, *box, *pairs, *axes, *legend;
  GLint nValuesBox, nValues, nValuesAxes, nValuesLegend;
  OpenGLView *view;
  float modelView[16];
  int viewport[4], *width;
  VisuRendering *method;
  GList *lights;
  Light *light;
#define NVERT 2
#define NVALS 8
#define NPASS 1
#define NBUFF (NVERT * NVALS + NPASS * 2)
  float tmpFloat[NBUFF];
#define PAIRS_NVERT 2
#define PAIRS_NVALS 4
#define PAIRS_NPASS 5
#define PAIRS_NBUFF (PAIRS_NVERT * PAIRS_NVALS + PAIRS_NPASS * 2)
#define PAIRS_NCMPT (4 + 1 + PAIRS_NPASS)
  VisuDataIter iter1, iter2;
  float d2, d2min, d2max, d2min_buffered, d2max_buffered, l;
  VisuPairData *data;
  float xyz1[3], xyz2[3], u[3], alpha, hsl[3], rgba[3], lum;
  GList *pairsLst, *tmpLst, *prop;
  struct _pairs *pairData;
  int nPairs, iPairs;
  float tmpPairs[6];
  Color *color;
  gboolean flat;
  gchar *name;
  VisuElement *ele;
  gchar *lbl;
  VisuNodeArray *array;
  float lgWidth, lgHeight, max;
  
  cr = cairo_create(cairo_surf);
  status = cairo_status(cr);
  if (status != CAIRO_STATUS_SUCCESS)
    {
      *error = g_error_new(VISU_ERROR_DUMP, DUMP_ERROR_FILE,
			   cairo_status_to_string(status));
      cairo_destroy(cr);
      return FALSE;
    }
  cairo_set_line_cap(cr, CAIRO_LINE_CAP_ROUND);

  flat = FALSE;
  prop = fileFormatGet_propertiesList(format);
  while (prop)
    {
      name = fileFormatGet_propertyName((FileFormatProperty*)(prop->data));
      if (strcmp(name, "use_flat_rendering") == 0)
	flat = fileFormatGet_propertyBoolean((FileFormatProperty*)(prop->data));
      prop = g_list_next(prop);
    }

  visuDataIter_new(dataObj, &iter);
  view = visuDataGet_openGLView(dataObj);
  method = visuRenderingClassGet_current();
  lights = lightEnvironnementGet_lightList(openGLGet_currentLights());
  g_return_val_if_fail(lights, FALSE);
  light = (Light*)lights->data;

  /* We get the fog color. */
  if (fogGet_isOn() && DEBUG_USE_FOG)
    {
      if (fogGet_useSpecificColor())
	fogGet_value(rgbaFog);
      else
	bgGet_value(rgbaFog);
    }
  else
    {
      rgbaFog[0] = 0.;
      rgbaFog[1] = 0.;
      rgbaFog[2] = 0.;
      rgbaFog[3] = 0.;
    }

  /* We create a feedback mode to get node coordinates. */
  coordinates = g_malloc(sizeof(GLfloat) * (iter.nAllStoredNodes * NBUFF));
  glFeedbackBuffer((iter.nAllStoredNodes * NBUFF), GL_3D_COLOR, coordinates);
  glRenderMode(GL_FEEDBACK);
  glGetFloatv(GL_MODELVIEW_MATRIX, modelView);
  glGetIntegerv(GL_VIEWPORT, viewport);
  scale.y0 = (double)viewport[3];
  cairo_set_matrix(cr, &scale);

  glPushMatrix();
  glTranslated(-view->box->dxxs2, -view->box->dyys2, -view->box->dzzs2);

  /* First thing is to order nodes along z axes. */
  for (visuDataIter_startVisible(dataObj, &iter); iter.node;
       visuDataIter_nextVisible(dataObj, &iter))
    {
      glBegin(GL_POINTS);
      visuDataGet_nodePosition(dataObj, iter.node, xyz);
      /* We compute the node position in the eyes coordinates. */
      glVertex3fv(xyz);
      /* We compute the node apparent radius using the real radius in
	 X direction in ModelView. */
      radius = visuRenderingGet_sizeOfElement(method, iter.element);
      glVertex3f(xyz[0] + modelView[0] * radius,
		 xyz[1] + modelView[4] * radius,
		 xyz[2] + modelView[8] * radius);
      glEnd();

      /* We store the number of the VisuElement. */
      glPassThrough(iter.iElement);
    }

  /* We sort the coordinates along z. */
  nValues = glRenderMode(GL_RENDER);
  /* We compact coordinates to keep complete vertex list. */
  i = 0;
  nNodes = 0;
  while (i < nValues)
    {
      if (coordinates[i] == GL_POINT_TOKEN &&
	  coordinates[i + NVALS] == GL_POINT_TOKEN)
	{
/* 	  fprintf(stderr, "Found a complete node %d, at %gx%g.\n", */
/* 		  i, coordinates[i + 1], coordinates[i + 2]); */
/* 	  fprintf(stderr, " | move it from %d to %d.\n", i, nNodes); */
	  /* A complete set, copying it to nNodes location. */
	  if (nNodes != i)
	    memcpy(coordinates + nNodes, coordinates + i, sizeof(GLfloat) * NBUFF);
	  i += NBUFF;
	  nNodes += NBUFF;
	}
      else
	{
/* 	  fprintf(stderr, "Found a uncomplete node at %d.\n", i); */
	  /* Incomplete set, go on till the GL_PASS_THROUGH_TOKEN. */
	  while (coordinates[i] != GL_PASS_THROUGH_TOKEN)
	    i += 1;
	  /* Remove the GL_POINT_TOKEN. */
	  i += 2;
/* 	  fprintf(stderr, " | jump to %d.\n", i); */
	}
    }
  sort_by_z(coordinates, tmpFloat, NBUFF, 3, 0, nNodes / NBUFF - 1);

  /* Draw the pairs. */
  nPairs = -1;
  pairs = (GLfloat*)0;
  if (visuPairGet_status())
    {
      pairsLst = (GList*)0;

      visuDataIter_new(dataObj, &iter1);
      visuDataIter_new(dataObj, &iter2);
      for(visuDataIter_start(dataObj, &iter1); iter1.element;
	  visuDataIter_nextElement(dataObj, &iter1))
	{
	  if (!visuElementGet_rendered(iter1.element))
	    continue;

	  for(visuDataIter_start(dataObj, &iter2);
	      iter2.element && iter2.iElement <= iter1.iElement ;
	      visuDataIter_nextElement(dataObj, &iter2))
	    {
	      if (!visuElementGet_rendered(iter2.element))
		continue;
	  
	      for (tmpLst = visuPairGet_links(iter1.element, iter2.element);
		   tmpLst; tmpLst = g_list_next(tmpLst))
		{
		  data = (VisuPairData*)tmpLst->data;
		  if (!data->drawn)
		    continue;
		  d2min = data->minMax[PAIRS_MIN] * data->minMax[PAIRS_MIN];
		  d2max = data->minMax[PAIRS_MAX] * data->minMax[PAIRS_MAX];
		  if(d2min >= d2max || d2max <= 0.)
		    continue;

		  l = data->minMax[PAIRS_MAX] - data->minMax[PAIRS_MIN];
		  d2min_buffered = (data->minMax[PAIRS_MIN] - 0.15 * l);
		  d2min_buffered *= d2min_buffered;
		  d2max_buffered = (data->minMax[PAIRS_MAX] + 0.15 * l);
		  d2max_buffered *= d2max_buffered;

		  for(visuDataIter_restartNode(dataObj, &iter1); iter1.node;
		      visuDataIter_nextNode(dataObj, &iter1))
		    {
		      if (!iter1.node->rendered)
			continue;

		      for(visuDataIter_restartNode(dataObj, &iter2); iter2.node;
			  visuDataIter_nextNode(dataObj, &iter2))
			{
			  if (!iter2.node->rendered)
			    continue;
			  /* Don't draw the inter element pairs two times. */
			  if (iter1.element == iter2.element &&
			      iter2.node >= iter1.node)
			    break;

			  visuDataGet_nodePosition(dataObj, iter1.node, xyz1);
			  visuDataGet_nodePosition(dataObj, iter2.node, xyz2);
			  d2 = (xyz1[0] - xyz2[0]) * (xyz1[0] - xyz2[0]) + 
			    (xyz1[1] - xyz2[1]) * (xyz1[1] - xyz2[1]) + 
			    (xyz1[2] - xyz2[2]) * (xyz1[2] - xyz2[2]);
			  if(d2 <= 0. || d2 < d2min_buffered || d2 > d2max_buffered)
			    continue;

			  if (d2 < d2min)
			    alpha = (d2 - d2min_buffered) /
			      (d2min - d2min_buffered);
			  else if (d2 > d2max)
			    alpha = (d2max_buffered - d2) /
			      (d2max_buffered - d2max);
			  else
			    alpha = 1.f;

#if GLIB_MINOR_VERSION > 9
			  pairData = g_slice_alloc(sizeof(struct _pairs));
#else
			  pairData = g_malloc(sizeof(struct _pairs));
#endif
			  pairData->data  = data;
			  pairData->node1 = iter1.node;
			  pairData->node2 = iter2.node;
			  pairData->ele1  = iter1.element;
			  pairData->ele2  = iter2.element;
			  pairData->alpha = alpha;
			  pairsLst = g_list_prepend(pairsLst, pairData);
			}
		    }
		}
	    }
	}
      nPairs = g_list_length(pairsLst);
      DBG_fprintf(stderr, "Dump SVG: found %d pairs to draw.\n", nPairs);

      if (nPairs > 0)
	{
	  pairs = g_malloc(sizeof(GLfloat) * nPairs * PAIRS_NBUFF);
	  glFeedbackBuffer(nPairs * PAIRS_NBUFF, GL_3D, pairs);
	  glRenderMode(GL_FEEDBACK);
	  glPushMatrix();

	  /* Render the list of pairs and free them each time. */
	  tmpLst = pairsLst;
	  while (tmpLst)
	    {
	      pairData = (struct _pairs*)tmpLst->data;

	      visuDataGet_nodePosition(dataObj, pairData->node1, xyz1);
	      visuDataGet_nodePosition(dataObj, pairData->node2, xyz2);
	      u[0] = xyz2[0] - xyz1[0];
	      u[1] = xyz2[1] - xyz1[1];
	      u[2] = xyz2[2] - xyz1[2];
	      norm = sqrt(u[0] * u[0] + u[1] * u[1] + u[2] * u[2]);
	      u[0] /= norm;
	      u[1] /= norm;
	      u[2] /= norm;
	      radius = visuRenderingGet_sizeOfElement(method, pairData->ele1);
	      xyz1[0] += radius * u[0];
	      xyz1[1] += radius * u[1];
	      xyz1[2] += radius * u[2];
	      radius = visuRenderingGet_sizeOfElement(method, pairData->ele2);
	      xyz2[0] -= radius * u[0];
	      xyz2[1] -= radius * u[1];
	      xyz2[2] -= radius * u[2];
	      glBegin(GL_POINTS);
	      glVertex3fv(xyz1);
	      glVertex3fv(xyz2);
	      glEnd();
	      /* We save colour channel as passthrough. */
	      color = visuPairGet_color(pairData->data);
	      glPassThrough(color->rgba[0]);
	      glPassThrough(color->rgba[1]);
	      glPassThrough(color->rgba[2]);
	      glPassThrough(pairData->alpha);
	      /* We save the width in a passthrough. */
	      width = (int*)visuPairGet_linkProperty(pairData->data, "width");
	      if (width)
		glPassThrough((float)*width);
	      else
		glPassThrough((float)boxGet_lineWidth());
	  
	      /* Free the data. */
#if GLIB_MINOR_VERSION > 9
	      g_slice_free1(sizeof(struct _pairs), tmpLst->data);
#else
	      g_free(tmpLst->data);
#endif
	      tmpLst = g_list_next(tmpLst);
	    }
	  /* Free the list itself. */
	  if (pairsLst)
	    g_list_free(pairsLst);
	  /* Analyse the OpenGL results. */
	  nValues = glRenderMode(GL_RENDER);
	  DBG_fprintf(stderr, " | OpenGL returns %d.\n", nValues);
	  i = 0;
	  nPairs = 0;
	  while (i < nValues)
	    {
	  
	      if (pairs[i] == GL_POINT_TOKEN && 
		  pairs[i + PAIRS_NVALS] == GL_POINT_TOKEN &&
		  pairs[i + 2 * PAIRS_NVALS] == GL_PASS_THROUGH_TOKEN)
		{
/* 	      DBG_fprintf(stderr, " | pair %d at (%f;%f;%f) - (%f;%f;%f) %g\n", */
/* 			  nPairs, */
/* 			  pairs[i + 1], pairs[i + 2], pairs[i + 3], */
/* 			  pairs[i + 5], pairs[i + 6], pairs[i + 7], pairs[i + 9]); */
		  /* Copy all these values into the beginning of the pairs
		     array. */
		  norm = (pairs[i + 3] + pairs[i + 7]) / 2.f;
		  pairs[nPairs * PAIRS_NCMPT + 0] = pairs[i + 1]; /* x1 */
		  pairs[nPairs * PAIRS_NCMPT + 1] = pairs[i + 2]; /* y1 */
		  pairs[nPairs * PAIRS_NCMPT + 2] = pairs[i + 5]; /* x2 */
		  pairs[nPairs * PAIRS_NCMPT + 3] = pairs[i + 6]; /* y2 */
		  pairs[nPairs * PAIRS_NCMPT + 4] = norm;         /* altitude */
		  pairs[nPairs * PAIRS_NCMPT + 5] = pairs[i + PAIRS_NVALS * PAIRS_NVERT + 7]; /* alpha */
		  pairs[nPairs * PAIRS_NCMPT + 6] = pairs[i + PAIRS_NVALS * PAIRS_NVERT + 1]; /* red */
		  pairs[nPairs * PAIRS_NCMPT + 7] = pairs[i + PAIRS_NVALS * PAIRS_NVERT + 3]; /* green */
		  pairs[nPairs * PAIRS_NCMPT + 8] = pairs[i + PAIRS_NVALS * PAIRS_NVERT + 5]; /* blue */
		  pairs[nPairs * PAIRS_NCMPT + 9] = pairs[i + PAIRS_NVALS * PAIRS_NVERT + 9]; /* width */
		  i += PAIRS_NBUFF;
		  nPairs += 1;
		}
	      else if (pairs[i] == GL_POINT_TOKEN &&
		       pairs[i + PAIRS_NVALS] == GL_PASS_THROUGH_TOKEN)
		{
		  DBG_fprintf(stderr, "| uncomplete pair for i=%d\n", i);
		  i += PAIRS_NVALS + 2 * PAIRS_NPASS;
		}
	      else if (pairs[i] == GL_PASS_THROUGH_TOKEN)
		{
		  DBG_fprintf(stderr, "| no pair for i=%d\n", i);
		  i += 2 * PAIRS_NPASS;
		}
	    }
	  glPopMatrix();
	  sort_by_z(pairs, tmpPairs, PAIRS_NCMPT, 4, 0, nPairs - 1);
	  DBG_fprintf(stderr, " | will draw %d pairs.\n", nPairs);
	}
    }
  
  /* We draw the background colour. */
  bgGet_value(rgbaBg);
  cairo_set_source_rgba(cr, rgbaBg[0], rgbaBg[1], rgbaBg[2], rgbaBg[3]);
  cairo_paint(cr);

  /* Draw the back box. */
  nValuesBox = -1;
  box = (GLfloat*)0;
  if (boxGet_isOn())
    {
      box = g_malloc(sizeof(GLfloat) * 7000);
      glFeedbackBuffer(7000, GL_3D, box);
      glRenderMode(GL_FEEDBACK);
      OpenGLExtensionCall_list(EXT_BOX_ID, FALSE);
      nValuesBox = glRenderMode(GL_RENDER);
      cairo_set_line_width(cr, (double)boxGet_lineWidth());
      rgb = boxGet_RGBValues();
      cairo_set_source_rgb(cr, rgb[0], rgb[1], rgb[2]);
      /* We draw the lines that have a boundary hidden by elements. */
      for (i = 0; i < nValuesBox; i += 7)
	if (box[i + 3] >= coordinates[nNodes - NBUFF + 3] &&
	    box[i + 6] >= coordinates[nNodes - NBUFF + 3])
	  {
	    cairo_move_to(cr, box[i + 1], box[i + 2]);
	    cairo_line_to(cr, box[i + 4], box[i + 5]);
	    cairo_stroke(cr);
	  }
    }

  /* We create the pattern for the different elements. */
  pat = g_malloc(sizeof(cairo_pattern_t*) * dataObj->ntype);
  for (i = 0; i < (int)dataObj->ntype; i++)
    {
      if (flat)
	{
	  pat[i] =
	    cairo_pattern_create_rgba(dataObj->fromIntToVisuElement[i]->rgb[0],
				      dataObj->fromIntToVisuElement[i]->rgb[1],
				      dataObj->fromIntToVisuElement[i]->rgb[2],
				      dataObj->fromIntToVisuElement[i]->rgb[3]);
	}
      else
	{
	  pat[i] = cairo_pattern_create_radial(.4f, .4f, .1f, 0.f, 0.f, 1.f);
	  /* We get the Element colour in HSL. */
	  color_RGBtoHSL(hsl, dataObj->fromIntToVisuElement[i]->rgb);
	  lum = hsl[2];
	  hsl[2] = CLAMP(lum + 0.2f, 0.f, 1.f);
	  color_HSLtoRGB(rgba, hsl);
	  cairo_pattern_add_color_stop_rgba(pat[i], 0, rgba[0], rgba[1], rgba[2], 1.f);
	  hsl[2] = CLAMP(lum + 0.05f, 0.f, 1.f);
	  color_HSLtoRGB(rgba, hsl);
	  cairo_pattern_add_color_stop_rgba(pat[i], 0.3, rgba[0], rgba[1], rgba[2], 1.f);
	  hsl[2] = CLAMP(lum - 0.05f, 0.f, 1.f);
	  color_HSLtoRGB(rgba, hsl);
	  cairo_pattern_add_color_stop_rgba(pat[i], 0.7, rgba[0], rgba[1], rgba[2], 1.f);
	  hsl[2] = CLAMP(lum - 0.2f, 0.f, 1.f);
	  color_HSLtoRGB(rgba, hsl);
	  cairo_pattern_add_color_stop_rgba(pat[i], 1, rgba[0], rgba[1], rgba[2], 1.f);
	}
    }

  cairo_set_line_width(cr, 1.);
  iPairs = 0;
  alpha = 0.f;
  for (i = 0; i < nNodes; i+= NBUFF)
    {
      if (nPairs > 0 && iPairs < PAIRS_NCMPT * nPairs)
	{
	  /* We draw the pairs in between. */
	  while (pairs[iPairs + 4] > coordinates[i + 3] &&
		 iPairs < PAIRS_NCMPT * nPairs)
	    {
	      DBG_fprintf(stderr, " | pair %d at (%f;%f) - (%f;%f) %g,%g\n",
			  iPairs / PAIRS_NCMPT,
			  pairs[iPairs + 0], pairs[iPairs + 1],
			  pairs[iPairs + 2], pairs[iPairs + 3],
			  pairs[iPairs + 4], pairs[iPairs + 5]);
	      cairo_set_line_width(cr, pairs[iPairs + 9]);
	      cairo_set_source_rgba(cr, pairs[iPairs + 6], pairs[iPairs + 7],
				    pairs[iPairs + 8], pairs[iPairs + 5]);
	      cairo_move_to(cr, pairs[iPairs + 0], pairs[iPairs + 1]);
	      cairo_line_to(cr, pairs[iPairs + 2], pairs[iPairs + 3]);
	      cairo_stroke(cr);
	      iPairs += PAIRS_NCMPT;
	    }
	  cairo_set_line_width(cr, 1.);
	}

      /* Compute the alpha for the fog. */
      if (fogGet_isOn() && DEBUG_USE_FOG)
	alpha = 1.f - CLAMP((fogGet_end() - coordinates[i + 3]) /
			    (fogGet_end() - fogGet_start()), 0.f, 1.f);
      
      radius = (coordinates[i + NVALS + 1] - coordinates[i + 1]) *
	(coordinates[i + NVALS + 1] - coordinates[i + 1]) +
	(coordinates[i + NVALS + 2] - coordinates[i + 2]) *
	(coordinates[i + NVALS + 2] - coordinates[i + 2]);
      /*       fprintf(stderr, "%gx%g %g %g %g\n", coordinates[i + 1], coordinates[i + 2], */
      /* 	      sqrt(radius), coordinates[i + 3], coordinates[i + NBUFF - 1]); */
      radius = sqrt(radius);

      cairo_save(cr);
      cairo_translate(cr, coordinates[i + 1], coordinates[i + 2]);
      cairo_arc(cr, 0.f, 0.f, radius, 0., 2 * G_PI);
      cairo_save(cr);
      cairo_scale(cr, radius, radius);
      if (alpha < 1.f)
	{
	  cairo_set_source(cr, pat[(int)coordinates[i + NVERT * NVALS + 1]]);
	  cairo_fill_preserve(cr);
	}
      if (alpha > 0.f)
	{
	  fprintf(stderr, "%g %g %g %g.\n", rgbaFog[0], rgbaFog[1], rgbaFog[2], alpha);
	  cairo_set_source_rgba(cr, rgbaFog[0], rgbaFog[1], rgbaFog[2], alpha);
	  cairo_fill_preserve(cr);
	}
      cairo_restore(cr);
      cairo_set_source_rgb(cr, rgbaFog[0] * alpha, rgbaFog[1] * alpha,
			   rgbaFog[2] * alpha);
      cairo_stroke(cr);
      cairo_restore(cr);
    }
  /* We draw the remaining pairs. */
  if (nPairs > 0 && iPairs < PAIRS_NCMPT * nPairs)
    {
      /* We draw the pairs in between. */
      while (iPairs < PAIRS_NCMPT * nPairs)
	{
	  DBG_fprintf(stderr, " | pair %d at (%f;%f) - (%f;%f) %g,%g\n",
		      iPairs / PAIRS_NCMPT,
		      pairs[iPairs + 0], pairs[iPairs + 1],
		      pairs[iPairs + 2], pairs[iPairs + 3],
		      pairs[iPairs + 4], pairs[iPairs + 5]);
	  cairo_set_line_width(cr, pairs[iPairs + 9]);
	  cairo_set_source_rgba(cr, pairs[iPairs + 6], pairs[iPairs + 7],
				pairs[iPairs + 8], pairs[iPairs + 5]);
	  cairo_move_to(cr, pairs[iPairs + 0], pairs[iPairs + 1]);
	  cairo_line_to(cr, pairs[iPairs + 2], pairs[iPairs + 3]);
	  cairo_stroke(cr);
	  iPairs += PAIRS_NCMPT;
	}
      cairo_set_line_width(cr, 1.);
    }
  if (nPairs > 0)
    g_free(pairs);

  /* Draw the front box. */
  if (boxGet_isOn())
    {
      cairo_set_line_width(cr, (double)boxGet_lineWidth());
      rgb = boxGet_RGBValues();
      cairo_set_source_rgb(cr, rgb[0], rgb[1], rgb[2]);
      /* We draw the lines that have a boundary hidden by elements. */
      for (i = 0; i < nValuesBox; i += 7)
	if (box[i + 3] < coordinates[nNodes - NBUFF + 3] ||
	    box[i + 6] < coordinates[nNodes - NBUFF + 3])
	  {
	    cairo_move_to(cr, box[i + 1], box[i + 2]);
	    cairo_line_to(cr, box[i + 4], box[i + 5]);
	    cairo_stroke(cr);
	  }
    }
  g_free(box);
  g_free(coordinates);

  /* We draw the axes. */
  /* We create the feedback for the axes. */
  nValuesAxes = -1;
  axes = (GLfloat*)0;
  if (axesGet_areOn())
    {
      axes = g_malloc(sizeof(GLfloat) * 7000);
      glFeedbackBuffer(7000, GL_3D, axes);
      glRenderMode(GL_FEEDBACK);
      OpenGLExtensionCall_list(EXT_AXES_ID, TRUE);
      nValuesAxes = glRenderMode(GL_RENDER);

      cairo_set_line_width(cr, (double)axesGet_lineWidth());
      rgb = axesGet_RGBvalues();
      cairo_set_source_rgb(cr, rgb[0], rgb[1], rgb[2]);
      cairo_select_font_face(cr, "Serif", CAIRO_FONT_SLANT_NORMAL,
			     CAIRO_FONT_WEIGHT_NORMAL);
      cairo_set_font_size(cr, 12.0);
      /* We draw the 3 lines of the axes. */
      if (nValuesAxes >= 21 && axes[0] == GL_LINE_RESET_TOKEN &&
	  axes[7] == GL_LINE_RESET_TOKEN && axes[14] == GL_LINE_RESET_TOKEN)
	{
	  DBG_fprintf(stderr, "Dump SVG: output axes.\n");
	  cairo_move_to(cr, axes[0 + 1], axes[0 + 2]);
	  cairo_line_to(cr, axes[0 + 4], axes[0 + 5]);
	  cairo_stroke(cr);
	  cairo_move_to(cr, axes[7 + 1], axes[7 + 2]);
	  cairo_line_to(cr, axes[7 + 4], axes[7 + 5]);
	  cairo_stroke(cr);
	  cairo_move_to(cr, axes[14 + 1], axes[14 + 2]);
	  cairo_line_to(cr, axes[14 + 4], axes[14 + 5]);
	  cairo_stroke(cr);
	}
      cairo_set_source_rgb(cr, 1.f - rgb[0], 1.f - rgb[1], 1.f - rgb[2]);
      cairo_set_font_matrix(cr, &scaleFont);
      if (nValuesAxes >= 33 && axes[21] == GL_BITMAP_TOKEN &&
	  axes[25] == GL_BITMAP_TOKEN && axes[29] == GL_BITMAP_TOKEN)
	{
	  DBG_fprintf(stderr, "Dump SVG: output axes names.\n");
	  cairo_move_to(cr, axes[21 + 1], axes[21 + 2]);
	  cairo_show_text(cr, "x");
	  cairo_move_to(cr, axes[25 + 1], axes[25 + 2]);
	  cairo_show_text(cr, "y");
	  cairo_move_to(cr, axes[29 + 1], axes[29 + 2]);
	  cairo_show_text(cr, "z");
	}
      g_free(axes);
    }

  /* We draw a legend, if required. */
  nValuesLegend = -1;
  legend = (GLfloat*)0;
  if (extLegendGet_isOn())
    {
      legend = g_malloc(sizeof(GLfloat) * 70000);
      glFeedbackBuffer(70000, GL_3D, legend);
      glRenderMode(GL_FEEDBACK);
      OpenGLExtensionCall_list(EXT_LEGEND_ID, TRUE);
      nValuesLegend = glRenderMode(GL_RENDER);

      cairo_select_font_face(cr, "Serif", CAIRO_FONT_SLANT_NORMAL,
			     CAIRO_FONT_WEIGHT_NORMAL);
      cairo_set_font_size(cr, 12.0);
      /* The two first polygons are the white frame. */
      if (nValuesLegend >= 22)
	{
	  cairo_set_source_rgba(cr, 1.f, 1.f, 1.f, 0.4f);
	  lgWidth  = legend[5] - legend[2];
	  lgHeight = legend[9] - legend[3];
	  cairo_rectangle(cr, legend[2], legend[3], lgWidth, lgHeight);
	  cairo_fill(cr);
	  /* Run to the names and the elements... */
	  nEle = 0;
	  array = visuDataGet_nodeArray(dataObj);
	  max = visuDataGet_allElementExtens(dataObj);
	  for (i = 0; i < nValuesLegend; i++)
	    if (legend[i] == GL_BITMAP_TOKEN)
	      {
		ele = dataObj->fromIntToVisuElement[nEle];

		/* Write the name. */
		cairo_move_to(cr, legend[i + 1], legend[i + 2]);
		lbl = g_strdup_printf("%s (%d)", ele->name,
				      array->numberOfStoredNodes[nEle]);
		cairo_set_source_rgb(cr, 0.f, 0.f, 0.f);
		cairo_set_font_matrix(cr, &scaleFont);
		cairo_show_text(cr, lbl);
		g_free(lbl);

		/* Draw the element. */
		radius = 0.4f * lgHeight *
		  visuRenderingGet_sizeOfElement(method, ele) / max;
		cairo_new_path(cr);
		cairo_save(cr);
		cairo_translate(cr, legend[i + 1] - 0.6f * lgHeight,
				legend[3] + 0.5f * lgHeight);
		cairo_arc(cr, 0.f, 0.f, radius, 0., 2 * G_PI);
		cairo_save(cr);
		cairo_scale(cr, radius, radius);
		cairo_set_source(cr, pat[nEle]);
		cairo_fill_preserve(cr);
		cairo_restore(cr);
		cairo_set_source_rgb(cr, 0.f, 0.f, 0.f);
		cairo_stroke(cr);
		cairo_restore(cr);

		nEle += 1;
		do
		  i += 4;
		while (legend[i] == GL_BITMAP_TOKEN && i < nValuesLegend);
	      }
	}
      g_free(legend);
    }

  for (i = 0; i < (int)dataObj->ntype; i++)
    cairo_pattern_destroy(pat[i]);
  g_free(pat);

  glPopMatrix();
  
  cairo_show_page(cr);
  cairo_destroy(cr);

  return TRUE;
}

static gboolean writeViewInSvgFormat(FileFormat *format, const char* filename,
				     int width, int height, VisuData *dataObj,
				     guchar* imageData _U_, GError **error,
				     voidDataFunc functionWait, gpointer data)
{
  cairo_surface_t *svg_surface;
  cairo_status_t status;
  OpenGLView *view;
  guint old_width, old_height;

  g_return_val_if_fail(error && !*error, FALSE);

  DBG_fprintf(stderr, "Dump SVG: begin export to SVG.\n");

  svg_surface = cairo_svg_surface_create(filename, (double)width, (double)height);
  status = cairo_surface_status(svg_surface);
  if (status != CAIRO_STATUS_SUCCESS)
    {
      *error = g_error_new(VISU_ERROR_DUMP, DUMP_ERROR_FILE,
			   cairo_status_to_string(status));
      cairo_surface_destroy(svg_surface);
      return FALSE;
    }

  view = visuDataGet_openGLView(dataObj);

  /* Change the view port to match width and height. */
  old_width = view->window->width;
  old_height = view->window->height;
  OpenGLViewSet_windowSize(view, (guint)width, (guint)height);
  writeDataToCairoSurface(svg_surface, format, dataObj, error, functionWait, data);
  cairo_surface_destroy(svg_surface);

  /* Set back the viewport. */
  OpenGLViewSet_windowSize(view, old_width, old_height);

  return TRUE;
}

#endif
#endif
