/*   EXTRAITS DE LA LICENCE
	Copyright CEA, contributeurs : Luc BILLARD, Damien
	CALISTE, Olivier D'Astier, laboratoire L_Sim, (2001-2005)
  
	Adresse ml :
	BILLARD, non joignable par ml ;
	CALISTE, damien P caliste AT cea P fr.

	Ce logiciel est un programme informatique servant  visualiser des
	structures atomiques dans un rendu pseudo-3D. 

	Ce logiciel est rgi par la licence CeCILL soumise au droit franais et
	respectant les principes de diffusion des logiciels libres. Vous pouvez
	utiliser, modifier et/ou redistribuer ce programme sous les conditions
	de la licence CeCILL telle que diffuse par le CEA, le CNRS et l'INRIA 
	sur le site "http://www.cecill.info".

	Le fait que vous puissiez accder  cet en-tte signifie que vous avez 
	pris connaissance de la licence CeCILL, et que vous en avez accept les
	termes (cf. le fichier Documentation/licence.fr.txt fourni avec ce logiciel).
*/

/*   LICENCE SUM UP
	Copyright CEA, contributors : Luc BILLARD and Damien
	CALISTE and Olivier D'Astier, laboratoire L_Sim, (2001-2005)

	E-mail addresses :
	BILLARD, not reachable any more ;
	CALISTE, damien P caliste AT cea P fr.
	D'ASTIER, dastier AT iie P cnam P fr.

	This software is a computer program whose purpose is to visualize atomic
	configurations in 3D.

	This software is governed by the CeCILL  license under French law and
	abiding by the rules of distribution of free software.  You can  use, 
	modify and/ or redistribute the software under the terms of the CeCILL
	license as circulated by CEA, CNRS and INRIA at the following URL
	"http://www.cecill.info". 

	The fact that you are presently reading this means that you have had
	knowledge of the CeCILL license and that you accept its terms. You can
	find a copy of this licence shipped with this software at Documentation/licence.en.txt.
*/
#include "surfaces.h"

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <math.h>

#include <GL/gl.h>
#include <GL/glu.h>

#include <opengl.h>
#include <visu_object.h>
#include <visu_configFile.h>
#include <visu_tools.h>
#include <coreTools/toolMatrix.h>
#include <coreTools/toolConfigFile.h>

/**
 * SECTION:surfaces
 * @short_description: Supports loading of .surf files and drawing of
 * surfaces through OpenGL.
 *
 * <para>Originally written by Luc Billard for his Visualize program. This
 * module allows loading of .surf files to draw scalar fields on top of
 * the current display scene. .surf files are text files which specs are
 * the following :
 * <itemizedlist>
 * <listitem><para>
 * 1st line is arbitrary
 * </para></listitem>
 * <listitem><para>
 * 2nd line must contain 3 real (float) values: dxx dyx dyy
 * </para></listitem>
 * <listitem><para>
 * 3rd line must contain 3 real (float) values: dzx dzy dzz
 * </para></listitem>
 * <listitem><para>
 * 4th line must contain 3 positive integers which
 * represents respectively the number of surfaces, the total number of
 * polys, and the total number of points
 * </para></listitem>
 * <listitem><para>
 * Then, for each of these surfaces :      
 *     
 * <itemizedlist> 
 * <listitem><para>
 * next line must contain the name of the surface : it is a
 * string which should match the pattern surface_*
 * </para></listitem>
 * <listitem><para>
 * next line must contain 2 positive integer values: the number of polys
 * (num_polys) and
 * the number of points (num_points) used by the surface
 * </para></listitem>
 * <listitem><para>
 * each of the following num_polys lines must match the pattern 
 * [n i_1 i_2 i_3 ... i_n] where n is the number of vertices in the poly (n >= 3)
 * and [i_1 i_2 i_3 ... i_n] are the numbering of these vertices (vertices numbered from 1 to num_points)
 * </para></listitem>
 * <listitem><para>
 * each of the following num_points lines must contain 6 real values for the 
 * successive (1 to num_points) points : [x y z nx ny nz], where x y z are the coordinates of the point
 * and nx ny nz are the coordinates of the unit normal at the point
 * </para></listitem>
 * </itemizedlist>
 * 
 * </para></listitem>
 * </itemizedlist>
 * 
 * It is the responsibility of the user to guarantee that
 * dxx, dyx, dyy, dzx, dzy, dzz match the one currently loaded in V_Sim's
 * current context. Though if you use <link
 * linkend="v-sim-panelSurfaces">panelSurfaces</link> you can ask
 * to resize the surfaces so that they fit in the current loaded box.
 * </para>
 */

#define ISOSURFACES_FLAG_POTENTIAL "# potentialValue"

#define DESC_RESOURCE_INTRA "Choose if the interior is drawn in color inverse ;"\
  " a boolean (0 or 1)"
#define FLAG_RESOURCE_INTRA "isosurfaces_drawIntra"

static GQuark quark;

typedef struct SurfacesProperties_struct
{
  /* The element (G_TYPE_INT...). */
  guint type;
  /* A pointer to the surf it belongs to. */
  Surfaces *surf;
  /* The data. */
  gpointer data;
} SurfacesProperties;

struct SurfacesOrder_struct
{
  /* The size of the allocated arrays. */
  int allocatedSize;

  /* Any_pointer[i][0:1] gives the id for surfaces and id for poly i
     in the z sorted from back to front. */
  int **any_pointer;
  /* Store the z value.
     The array is recomputed each time, but stored here to
     avoid constant malloc. */
  double *any_z;
  /* Give for all poly the id for surfaces object and the id for poly
     in this object. any_pointer elements point to that array. */
  int *polygon_number;
};

/* Local variables. */
static gboolean drawIntra = FALSE;

/* Local methods. */
static gboolean isosurfaces_read_intra(gchar **lines, int nbLines, int position,
					    VisuData *dataObj, GError **error);
static void isosurfaces_export_resources(GString *data, VisuData *dataObj);

static void freeSurfacesProperties(gpointer data);
static void propertiesRemoveSurf(gpointer key, gpointer value, gpointer data);
static void propertiesReallocateSurf(gpointer key, gpointer value, gpointer data);


void isosurfacesInit()
{
  VisuConfigFileEntry *entry;

  quark = g_quark_from_static_string("visu_isosurfaces");

  entry = visuConfigFileAdd_entry(VISU_CONFIGFILE_RESOURCE,
				  FLAG_RESOURCE_INTRA,
				  DESC_RESOURCE_INTRA,
				  1, isosurfaces_read_intra);
  visuConfigFileSet_version(entry, 3.4f);
  visuConfigFileAdd_exportFunction(VISU_CONFIGFILE_RESOURCE,
				   isosurfaces_export_resources);
  
  isosurfacesResourcesInit();
}
GQuark isosurfacesGet_quark()
{
  return quark;
}

void isosurfacesCheck_consistency(Surfaces* surf)
{
  DBG_fprintf(stderr, "Isosurfaces : Check consistency.\n");

  DBG_fprintf(stderr, " | Base points\n");
  isosurfacesPointsCheck(&surf->basePoints);
  DBG_fprintf(stderr, " | Volatile planes\n");
  isosurfacesPointsCheck(&surf->volatilePlanes);
}
gboolean isosurfacesRemove(Surfaces *surf, int idSurf)
{
  int i, pos;

  g_return_val_if_fail(surf, FALSE);

  pos = isosurfacesGet_surfacePosition(surf, idSurf);
  g_return_val_if_fail(pos >= 0 && pos < surf->nsurf, FALSE);

  if (surf->nsurf == 1)
    return TRUE;

  DBG_fprintf(stderr, "Isosurfaces: remove surface %d from %p.\n", pos, (gpointer)surf);

  DBG_fprintf(stderr, "Isosurfaces: remove base points.\n");
  isosurfacesPointsRemove(&surf->basePoints, pos);
  DBG_fprintf(stderr, "Isosurfaces: remove volatile planes.\n");
  isosurfacesPointsRemove(&surf->volatilePlanes, pos);
  surf->nsurf -= 1;
  
  /* If the resource has no name, then we free it since it is not shared. */
  DBG_fprintf(stderr, " | Additional removing.\n");
  if (!surf->resources[pos]->surfnom)
    isosurfacesFree_resource(surf->resources[pos]);
  for (i = pos; i < surf->nsurf; i++)
    {
      surf->ids[i] = surf->ids[i + 1];
      /* Move resource pointers. */
      surf->resources[i] = surf->resources[i + 1];
    }
  surf->ids = g_realloc(surf->ids, sizeof(int) * surf->nsurf);
  surf->resources = g_realloc(surf->resources,
			      sizeof(SurfaceResource*) * surf->nsurf);
  g_hash_table_foreach(surf->properties, propertiesRemoveSurf, GINT_TO_POINTER(pos));
  
#if DEBUG == 1
  isosurfacesCheck_consistency(surf);
#endif

  return FALSE;
}

Surfaces* isosurfacesNew(int bufferSize)
{
  Surfaces *surf;

  surf = g_malloc(sizeof(Surfaces));
  DBG_fprintf(stderr, "Surfaces: new surfaces %p.\n", (gpointer)surf);

  surf->nsurf      = 0;
  isosurfacesPointsInit(&surf->basePoints,     bufferSize);
  isosurfacesPointsInit(&surf->volatilePlanes, bufferSize);

  surf->ids        = (int*)0;
  surf->resources  = (SurfaceResource**)0;
  surf->properties = g_hash_table_new_full(g_str_hash, g_str_equal,
					   NULL, freeSurfacesProperties);

  return surf;
}
void isosurfacesFree(Surfaces *surf)
{
  int i;

  g_return_if_fail(surf);

  DBG_fprintf(stderr, "Surfaces: free %p.\n", (gpointer)surf);
  isosurfacesPointsFree(&surf->basePoints);
  isosurfacesPointsFree(&surf->volatilePlanes);
  if (surf->ids)
    g_free(surf->ids);
  if (surf->resources)
    {
      /* We free all resource that has no name (i.e. that are not stored in
	 the hashtable. */
      for (i = 0; i < surf->nsurf; i++)
	if (surf->resources[i] && !surf->resources[i]->surfnom)
	  isosurfacesFree_resource(surf->resources[i]);
      g_free(surf->resources);
    }
  if (surf->properties)
    g_hash_table_destroy(surf->properties);

  g_free(surf);
}
 
void isosurfacesAllocate(Surfaces *surf, int nsurf, int npolys, int npoints)
{
  int i;
  
  DBG_fprintf(stderr, "Surfaces: allocate %p to %d %d %d.\n",
	      (gpointer)surf, nsurf, npolys, npoints);
  surf->nsurf = nsurf;
  isosurfacesPointsAllocate(&surf->basePoints, nsurf, npolys, npoints);
  isosurfacesPointsAllocate(&surf->volatilePlanes, nsurf, 0, 0);

  DBG_fprintf(stderr, "Surfaces: allocate complement.\n");
  /* Allocating pointers. */
  surf->ids       = g_malloc(nsurf * sizeof(int));
  surf->resources = g_malloc(nsurf * sizeof(SurfaceResource*));

  /* Set initial values. */
  for(i = 0; i < surf->nsurf; i++)
    surf->resources[i] = (SurfaceResource*)0;
  DBG_fprintf(stderr, " | OK.\n");
}
void isosurfacesAddSurfaces(Surfaces *surf, int nsurf, int npolys, int npoints)
{
  int i, nsurf_old;

  g_return_if_fail(surf);
  
  nsurf_old = surf->nsurf;
  DBG_fprintf(stderr, "Isosurfaces : add surface(s) %d %d %d.\n",
	      nsurf, npolys, npoints);
  surf->nsurf                        += nsurf;
  surf->basePoints.nsurf             += nsurf;
  surf->volatilePlanes.nsurf         += nsurf;
  surf->basePoints.num_polys         += npolys;
  surf->basePoints.num_points        += npoints;
  surf->basePoints.num_polys_surf     = g_realloc(surf->basePoints.num_polys_surf,
						  surf->nsurf * sizeof(int));
  surf->volatilePlanes.num_polys_surf = g_realloc(surf->volatilePlanes.num_polys_surf,
						  surf->nsurf * sizeof(int));
  for (i = 0; i < nsurf; i++)
    {
      surf->basePoints.num_polys_surf[nsurf_old + i]     = 0;
      surf->volatilePlanes.num_polys_surf[nsurf_old + i] = 0;
    }

  surf->basePoints.poly_surf_index    = g_realloc(surf->basePoints.poly_surf_index,
						  surf->basePoints.num_polys *
						  sizeof(int));
  surf->basePoints.poly_num_vertices  = g_realloc(surf->basePoints.poly_num_vertices,
						  surf->basePoints.num_polys *
						  sizeof(int));
  surf->basePoints.poly_vertices      = g_realloc(surf->basePoints.poly_vertices,
						  surf->basePoints.num_polys *
						  sizeof(int*));

  surf->basePoints.poly_points_data   = g_realloc(surf->basePoints.poly_points_data,
						  surf->basePoints.num_points *
						  sizeof(float *));
  surf->basePoints.poly_points_data[0]= g_realloc(surf->basePoints.poly_points_data[0],
						  (SurfacesPoints_userOffset +
						   surf->basePoints.bufferSize) *
						  surf->basePoints.num_points *
						  sizeof(float));
  for(i = 0; i < surf->basePoints.num_points; i++)
    surf->basePoints.poly_points_data[i]   = surf->basePoints.poly_points_data[0] +
      i * (SurfacesPoints_userOffset + surf->basePoints.bufferSize);

  surf->ids                           = g_realloc(surf->ids,
						  surf->nsurf * sizeof(int));
  surf->resources                     = g_realloc(surf->resources,
						  surf->nsurf * sizeof(SurfaceResource*));

  /* Dealing with properties reallocation. */
  g_hash_table_foreach(surf->properties, propertiesReallocateSurf,
		       GINT_TO_POINTER(surf->nsurf));
}

void isosurfacesSet_box(Surfaces *surf, double box[6])
{
  g_return_if_fail(surf);

  DBG_fprintf(stderr, "Surfaces: copy box.\n");
  memcpy(surf->local_box, box, sizeof(double) * 6);
}
double* isosurfacesGet_box(Surfaces *surf)
{
  g_return_val_if_fail(surf, (double*)0);

  DBG_fprintf(stderr, "Surfaces: get box.\n");
  return surf->local_box;
}
static void freeSurfacesProperties(gpointer data)
{
  SurfacesProperties *prop;

  prop = (SurfacesProperties*)data;
  g_free(prop->data);
  g_free(prop);
}
static void propertiesRemoveSurf(gpointer key _U_, gpointer value, gpointer data)
{
  int idSurf, i;
  SurfacesProperties *prop;

  idSurf = GPOINTER_TO_INT(data);
  prop = (SurfacesProperties*)value;

  switch (prop->type)
    {
    case G_TYPE_FLOAT:
      for (i = idSurf; i < prop->surf->nsurf; i++)
	((float*)prop->data)[i] = ((float*)prop->data)[i + 1];
      prop->data = g_realloc(prop->data, prop->surf->nsurf * sizeof(float));
      break;
    default:
      g_warning("Unimplemented format.");
    }
}
static void propertiesReallocateSurf(gpointer key _U_, gpointer value, gpointer data)
{
  int nsurf;
  SurfacesProperties *prop;

  nsurf = GPOINTER_TO_INT(data);
  prop = (SurfacesProperties*)value;

  switch (prop->type)
    {
    case G_TYPE_FLOAT:
      prop->data = g_realloc(prop->data, nsurf * sizeof(float));
      break;
    default:
      g_warning("Unimplemented format.");
    }
}
/* Dealing with properties as floats. */
float* isosurfacesAdd_floatProperty(Surfaces *surf, const gchar* name)
{
  float *data;
  SurfacesProperties *prop;

  g_return_val_if_fail(surf && surf->nsurf > 0, (float*)0);
  g_return_val_if_fail(name && name[0],         (float*)0);

  prop = g_malloc(sizeof(SurfacesProperties));
  prop->type = G_TYPE_FLOAT;
  prop->surf = surf;
  data = g_malloc(sizeof(float) * surf->nsurf);
  prop->data = (gpointer)data;
  g_hash_table_insert(surf->properties, (gpointer)name, (gpointer)prop);
  return data;
}
gboolean isosurfacesAdd_floatPropertyValue(Surfaces *surf, int idSurf,
					   const gchar* name, float value)
{
  float* data;
  int id;
  SurfacesProperties *prop;

  g_return_val_if_fail(surf, FALSE);
  
  id = isosurfacesGet_surfacePosition(surf, idSurf);
  g_return_val_if_fail(id >= 0 && id < surf->nsurf, FALSE);

  prop = (SurfacesProperties*)g_hash_table_lookup(surf->properties, name);
  if (!prop)
    return FALSE;
  g_return_val_if_fail(prop->surf != surf, FALSE);

  data = (float*)prop->data;
  data[id] = value;
  return TRUE;
}
gboolean isosurfacesGet_floatPropertyValue(Surfaces *surf, int idSurf,
					   const gchar *name, float *value)
{
  float* data;
  int id;
  SurfacesProperties *prop;

  g_return_val_if_fail(surf && value, FALSE);
  
  id = isosurfacesGet_surfacePosition(surf, idSurf);
  g_return_val_if_fail(id >= 0 && id < surf->nsurf, FALSE);

  prop = (SurfacesProperties*)g_hash_table_lookup(surf->properties, name);
  if (!prop)
    return FALSE;
  g_return_val_if_fail(prop->surf != surf, FALSE);

  data = (float*)prop->data;
  *value = data[id];
  return TRUE;
}
float* isosurfacesGet_floatProperty(Surfaces *surf, const gchar *name)
{
  SurfacesProperties *prop;

  g_return_val_if_fail(surf, (float*)0);

  prop = (SurfacesProperties*)g_hash_table_lookup(surf->properties, name);
  if (prop)
    return (float*)prop->data;
  else
    return (float*)0;
}

gboolean isosurfacesLoad_file(const char *file, Surfaces **surf, GError **error)
{
  GIOChannel* surf_file;
  int line_number = 0;
  double dxx=0, dyx=0, dyy=0;
  double dzx=0, dzy=0, dzz=0;
  int file_nsurfs=0, file_npolys=0, file_npoints=0;
  int npolys_loaded=0, npoints_loaded=0;
  int i,j,k, res;
  int sum_polys=0, sum_points=0;
  GIOStatus io_status;
  GString *current_line;
  float *densityData, densityValue, *fval;

  g_return_val_if_fail(surf, FALSE);
  g_return_val_if_fail(error, FALSE);

  DBG_fprintf(stderr, "Isosurfaces : trying to load %s file.\n", file);

  *surf = (Surfaces*)0;

  current_line = g_string_new("");

  *error = (GError*)0;
  surf_file = g_io_channel_new_file(file, "r", error);
  if(!surf_file)
    return FALSE;

  /*   DBG_fprintf(stderr, "File opened\n"); */
  *error = (GError*)0;
  io_status = g_io_channel_read_line_string(surf_file, current_line, NULL, error);
  if(io_status != G_IO_STATUS_NORMAL)
    {
      g_string_free(current_line, TRUE);
      g_io_channel_unref(surf_file);
      return FALSE;
    }
  line_number++;  
  /*   DBG_fprintf(stderr, "Line %d read successfully\n", line_number); */

  *surf = isosurfacesNew(0);

  *error = (GError*)0;
  io_status = g_io_channel_read_line_string(surf_file, current_line, NULL, error);
  if(io_status != G_IO_STATUS_NORMAL)
    {
      g_string_free(current_line, TRUE);
      g_io_channel_unref(surf_file);
      isosurfacesFree(*surf);
      *surf = (Surfaces*)0;
      return FALSE;
    }
  line_number++;
  /*   DBG_fprintf(stderr, "Line %d read successfully\n", line_number); */
  if(sscanf(current_line->str, "%lf %lf %lf", &dxx, &dyx, &dyy) != 3)
    {
      *error = g_error_new(VISU_ERROR_ISOSURFACES,
			   VISU_ERROR_FORMAT,
			   _("Line %d doesn't match the [float, float, float]"
			     " pattern."), line_number);
      g_string_free(current_line, TRUE);
      g_io_channel_unref(surf_file);
      isosurfacesFree(*surf);
      *surf = (Surfaces*)0;
      return FALSE;
    }

  *error = (GError*)0;
  io_status = g_io_channel_read_line_string(surf_file, current_line, NULL, error);
  if(io_status != G_IO_STATUS_NORMAL)
    {
      g_string_free(current_line, TRUE);
      g_io_channel_unref(surf_file);
      isosurfacesFree(*surf);
      *surf = (Surfaces*)0;
      return FALSE;
    }
  line_number++;
  /*   DBG_fprintf(stderr, "Line %d read successfully\n", line_number); */
  if(sscanf(current_line->str, "%lf %lf %lf", &dzx, &dzy, &dzz) != 3)
    {
      *error = g_error_new(VISU_ERROR_ISOSURFACES,
			   VISU_ERROR_FORMAT,
			   _("Line %d doesn't match the [float, float, float]"
			     " pattern."), line_number);
      g_string_free(current_line, TRUE);
      g_io_channel_unref(surf_file);
      isosurfacesFree(*surf);
      *surf = (Surfaces*)0;
      return FALSE;
    }
      
  *error = (GError*)0;
  io_status = g_io_channel_read_line_string(surf_file, current_line, NULL, error);
  if(io_status != G_IO_STATUS_NORMAL)
    {
      g_string_free(current_line, TRUE);
      g_io_channel_unref(surf_file);
      isosurfacesFree(*surf);
      *surf = (Surfaces*)0;
      return FALSE;
    }
  line_number++;
  /*   DBG_fprintf(stderr, "Line %d read successfully\n", line_number); */
  res = sscanf(current_line->str, "%d %d %d", &file_nsurfs, &file_npolys, &file_npoints);
  if(res != 3 || file_nsurfs <= 0 || file_npolys <= 0 || file_npoints <= 0)
    {
      *error = g_error_new(VISU_ERROR_ISOSURFACES,
			   VISU_ERROR_FORMAT,
			   _("Line %d doesn't match the [int > 0, int > 0, int > 0]"
			     " pattern."), line_number);
      g_string_free(current_line, TRUE);
      g_io_channel_unref(surf_file);
      isosurfacesFree(*surf);
      *surf = (Surfaces*)0;
      return FALSE;
    }
  /* From now on, the file is supposed to be a valid surface file. */
  (*surf)->nsurf = file_nsurfs;
  (*surf)->local_box[0] = dxx;
  (*surf)->local_box[1] = dyx;
  (*surf)->local_box[2] = dyy;
  (*surf)->local_box[3] = dzx;
  (*surf)->local_box[4] = dzy;
  (*surf)->local_box[5] = dzz;

  /* Allocate storage arrays. */
  isosurfacesAllocate(*surf, file_nsurfs, file_npolys, file_npoints);

  /* Create a table to store the density values. */
  densityData = isosurfacesAdd_floatProperty(*surf, ISOSURFACES_PROPERTY_POTENTIAL);

  /* For each surf */
  for(i = 0; i < (*surf)->nsurf; i++) 
    {
      int surf_npolys=0, surf_npoints=0;
      
      densityData[i] = 0.;
      (*surf)->ids[i] = i;
      /* Allow some commentaries here begining with a '#' character. */
      do
	{
	  /* Get the first line that should contains current surf' name */
	  *error = (GError*)0;
	  io_status = g_io_channel_read_line_string(surf_file, current_line, NULL, error);
	  if(io_status != G_IO_STATUS_NORMAL)
	    {
	      g_string_free(current_line, TRUE);
	      g_io_channel_unref(surf_file);
	      isosurfacesFree(*surf);
	      *surf = (Surfaces*)0;
	      return TRUE;
	    }
	  line_number++;

	  /* If the line begins with a '#', then we try to find a density value. */
	  if (current_line->str[0] == '#')
	    {
	      res = sscanf(current_line->str, ISOSURFACES_FLAG_POTENTIAL" %f", &densityValue);
	      if (res == 1)
		densityData[i] = densityValue;
	    }
	}
      while (current_line->str[0] == '#');
      g_strdelimit(current_line->str, "\n", ' ');
      g_strstrip(current_line->str);
      (*surf)->resources[i] = isosurfacesGet_resourceFromName(current_line->str, (gboolean*)0);
      DBG_fprintf(stderr, "Line %d ('%s' -> %p) read successfully %f\n",
		  line_number, current_line->str, (gpointer)(*surf)->resources[i],
		  densityData[i]);

      *error = (GError*)0;
      io_status = g_io_channel_read_line_string(surf_file, current_line, NULL, error);
      if(io_status != G_IO_STATUS_NORMAL)
	{
	  g_string_free(current_line, TRUE);
	  g_io_channel_unref(surf_file);
	  isosurfacesFree(*surf);
	  *surf = (Surfaces*)0;
	  return TRUE;
	}
      line_number++;
      /*        DBG_fprintf(stderr, "Line %d read successfully\n", line_number); */
      res = sscanf(current_line->str, "%d %d", &surf_npolys, &surf_npoints);
      if(res != 2 || surf_npolys <= 0 || surf_npoints <= 0)
	{
	  *error = g_error_new(VISU_ERROR_ISOSURFACES,
			       VISU_ERROR_FORMAT,
			       _("Line %d doesn't match the [int > 0, int > 0]"
				 " pattern."), line_number);
	  g_string_free(current_line, TRUE);
	  g_io_channel_unref(surf_file);
	  isosurfacesFree(*surf);
	  *surf = (Surfaces*)0;
	  return TRUE;
	}

      sum_polys += surf_npolys;
      if(sum_polys > file_npolys)
	{
	  *error = g_error_new(VISU_ERROR_ISOSURFACES,
			       VISU_ERROR_CHECKSUM,
			       _("Error on line %d. Declared number of polygons"
				 " reached."), line_number);
	  g_string_free(current_line, TRUE);
	  g_io_channel_unref(surf_file);
	  isosurfacesFree(*surf);
	  *surf = (Surfaces*)0;
	  return TRUE;
	}

      sum_points += surf_npoints;
      if(sum_points > file_npoints)
	{
	  *error = g_error_new(VISU_ERROR_ISOSURFACES,
			       VISU_ERROR_CHECKSUM,
			       _("Error on line %d. Declared number of points"
				 " reached."), line_number);
	  g_string_free(current_line, TRUE);
	  g_io_channel_unref(surf_file);
	  isosurfacesFree(*surf);
	  *surf = (Surfaces*)0;
	  return TRUE;
	}

      for(j = 0; j < surf_npolys; j++)
	{
	  int nvertex=0, nvertex_i=0;

	  gchar **split_line;
	   
	  *error = (GError*)0;
	  io_status = g_io_channel_read_line_string(surf_file, current_line, NULL, error);
	  if(io_status != G_IO_STATUS_NORMAL)
	    {
	      g_string_free(current_line, TRUE);
	      g_io_channel_unref(surf_file);
	      isosurfacesFree(*surf);
	      *surf = (Surfaces*)0;
	      return TRUE;
	    }
	  line_number++;

	  split_line = g_strsplit_set(current_line->str, " ", -1);

	  for(k = 0; split_line[k] != NULL; k++)
	    {
	      if(g_ascii_strcasecmp(split_line[k], "") == 0) 
		continue;	       
	      if(nvertex == 0)
		{
		  if(sscanf(split_line[k], "%d", &nvertex) != 1 || nvertex < 3)
		    {
		      *error = g_error_new(VISU_ERROR_ISOSURFACES,
					   VISU_ERROR_FORMAT,
					   _("Line %dmust begin by an int."),
					   line_number);
		      g_string_free(current_line, TRUE);
		      g_io_channel_unref(surf_file);
		      isosurfacesFree(*surf);
		      return TRUE;
		    }
		  (*surf)->basePoints.poly_num_vertices[npolys_loaded] = nvertex;
		  (*surf)->basePoints.poly_vertices[npolys_loaded] = g_malloc(nvertex * sizeof(double));
		  (*surf)->basePoints.poly_surf_index[npolys_loaded] = i + 1;
		  (*surf)->basePoints.num_polys_surf[i] += 1;
		  npolys_loaded++;
		  continue;
		}
	      res = sscanf(split_line[k], "%d", &(*surf)->basePoints.poly_vertices[npolys_loaded-1][nvertex_i]);
	      if(res != 1)
		{
		  *error = g_error_new(VISU_ERROR_ISOSURFACES,
				       VISU_ERROR_FORMAT,
				       _("Line %d doesn't match the [int > 3, ...]"
					 " required pattern."), line_number);
		  g_string_free(current_line, TRUE);
		  g_io_channel_unref(surf_file);
		  isosurfacesFree(*surf);
		  return TRUE;
		}
	      (*surf)->basePoints.poly_vertices[npolys_loaded-1][nvertex_i] += -1 + npoints_loaded;
	      nvertex_i++;
	      if(nvertex_i >= nvertex)
		break;
	    }

	  g_strfreev(split_line);
	}
 
      for(j = 0; j < surf_npoints; j++) 
	{
	  *error = (GError*)0;
	  io_status = g_io_channel_read_line_string(surf_file, current_line, NULL, error);
	  if(io_status != G_IO_STATUS_NORMAL)
	    {
	      g_string_free(current_line, TRUE);
	      g_io_channel_unref(surf_file);
	      isosurfacesFree(*surf);
	      *surf = (Surfaces*)0;
	      return TRUE;
	    }
	  line_number++;

	  fval = (*surf)->basePoints.poly_points_data[npoints_loaded];
	  if(sscanf(current_line->str, "%f %f %f %f %f %f", 
		    fval, fval + 1, fval + 2,
		    fval + SurfacesPoints_normalOffset,
		    fval + SurfacesPoints_normalOffset + 1,
		    fval + SurfacesPoints_normalOffset + 2) != 6) 
	    {
	      *error = g_error_new(VISU_ERROR_ISOSURFACES,
				   VISU_ERROR_FORMAT,
				   _("Line %d doesn't match the [float x 6]"
				     " required pattern."), line_number);
	      g_string_free(current_line, TRUE);
	      g_io_channel_unref(surf_file);
	      isosurfacesFree(*surf);
	      *surf = (Surfaces*)0;
	      return TRUE;
	    }
	  npoints_loaded++;
	}
    }

  g_string_free(current_line, TRUE);
  g_io_channel_unref(surf_file);

  return TRUE;
}  

void isosurfacesSet_showAll(Surfaces *surf, gboolean show)
{
  int i;

  g_return_if_fail(surf);

  for (i = 0; i < surf->nsurf; i++)
    surf->resources[i]->rendered = show;
}

gboolean isosurfacesHide(Surfaces *surf, Plane **planes)
{
  int i, j, k, iPoly, isurf, iVertice, n, iv;
  gboolean *verticeStatus;
  gboolean visibility, redraw, boundary, valid;
  int *boundaryPolys, nBounadryPolys;
  int npolys, npoints;
  SurfacesPoints *ptFr, *ptTo;

  g_return_val_if_fail(surf, FALSE);

  DBG_fprintf(stderr, "Isosurfaces: compute masked polygons"
	      " for surface %p.\n", (gpointer)surf);
  /* Free previous volatile points. */
  isosurfacesPointsFree(&surf->volatilePlanes);

  verticeStatus = g_malloc(sizeof(gboolean) * surf->basePoints.num_points);
  redraw = FALSE;
  /* Compute a visibility flag for each vertice. */
  for (i = 0; i < surf->basePoints.num_points; i++)
    verticeStatus[i] =
      planesGet_visibility(planes, surf->basePoints.poly_points_data[i]);
  /* We store the id of boundary polygons. */
  boundaryPolys = g_malloc(sizeof(int) * surf->basePoints.num_polys);
  nBounadryPolys = 0;
  /* Hide polygons. */
  for (i = 0; i < surf->basePoints.num_polys; i++)
    {
      visibility = TRUE;
      boundary = FALSE;
      if (surf->resources[ABS(surf->basePoints.poly_surf_index[i]) - 1]->sensitiveToMaskingPlanes)
	{
	  for (j = 0; j < surf->basePoints.poly_num_vertices[i]; j++)
	    {
	      visibility = visibility && verticeStatus[surf->basePoints.poly_vertices[i][j]];
	      boundary = boundary || verticeStatus[surf->basePoints.poly_vertices[i][j]];
	    }
	  boundary = !visibility && boundary;
	}
      if (!visibility && surf->basePoints.poly_surf_index[i] > 0)
	{
	  /* Hide this polygon. */
	  surf->basePoints.num_polys_surf[surf->basePoints.poly_surf_index[i] - 1] -= 1;
	  surf->basePoints.poly_surf_index[i] = -surf->basePoints.poly_surf_index[i];
	  redraw = TRUE;
	}
      else if (visibility && surf->basePoints.poly_surf_index[i] < 0)
	{
	  /* Show this polygon. */
	  surf->basePoints.poly_surf_index[i] = -surf->basePoints.poly_surf_index[i];
	  surf->basePoints.num_polys_surf[surf->basePoints.poly_surf_index[i] - 1] += 1;
	  redraw = TRUE;
	}
      if (boundary)
	boundaryPolys[nBounadryPolys++] = i;
    }
  if (DEBUG)
    for (i = 0; i < surf->nsurf; i++)
      fprintf(stderr, " | surface %2d -> %7d polygons\n", i, surf->basePoints.num_polys_surf[i]);
  /* We count the number of boundaries per surface and allocate
     accordingly the volatile. */
  npolys  = nBounadryPolys;
  npoints = 0;
  for (i = 0; i < nBounadryPolys; i++)
    {
      /* The number of points to add is the number of visible points plus 2. */
      npoints += 2;
      for (j = 0; j < surf->basePoints.poly_num_vertices[boundaryPolys[i]]; j++)
	if (verticeStatus[surf->basePoints.poly_vertices[boundaryPolys[i]][j]])
	  npoints += 1;
    }
  DBG_fprintf(stderr, "Isosurfaces: volatile polygons.\n");
  DBG_fprintf(stderr, " | polys %d, points %d\n", npolys, npoints);
  isosurfacesPointsAllocate(&surf->volatilePlanes, surf->nsurf, npolys, npoints);
  surf->volatilePlanes.num_polys  = 0;
  surf->volatilePlanes.num_points = 0;
  /* We copy the polygons and the vertices in the volatile part. */
  for (i = 0; i < nBounadryPolys; i++)
    {
      ptFr = &surf->basePoints;
      isurf = -ptFr->poly_surf_index[boundaryPolys[i]] - 1;
      ptTo = &surf->volatilePlanes;
      iPoly = ptTo->num_polys;
      iVertice = ptTo->num_points;

      ptTo->num_polys_surf[isurf] += 1;
      ptTo->poly_surf_index[iPoly] = isurf + 1;
      n = 2;
      for (j = 0; j < ptFr->poly_num_vertices[boundaryPolys[i]]; j++)
	if (verticeStatus[ptFr->poly_vertices[boundaryPolys[i]][j]])
	  n += 1;
      ptTo->poly_num_vertices[iPoly] = n;
      ptTo->poly_vertices[iPoly] = g_malloc(sizeof(int) * n);
      n = ptFr->poly_num_vertices[boundaryPolys[i]];
      /* Compute new vertex. */
      iv = 0;
      for (k = 0; k < n; k++)
	{
	  /* If vertex is visible, we put it. */
	  if (verticeStatus[ptFr->poly_vertices[boundaryPolys[i]][k]])
	    {
	      memcpy(ptTo->poly_points_data[iVertice],
		     ptFr->poly_points_data[ptFr->poly_vertices[boundaryPolys[i]][k]],
		     sizeof(float) * (SurfacesPoints_userOffset + ptTo->bufferSize));
	      g_return_val_if_fail(iv < ptTo->poly_num_vertices[iPoly], redraw);
	      ptTo->poly_vertices[iPoly][iv] = iVertice;
	      iVertice += 1;
	      iv += 1;
	    }
	  /* If next vertex brings a visibility change, we compute intersection. */
	  if ((!verticeStatus[ptFr->poly_vertices[boundaryPolys[i]][k]] &&
	       verticeStatus[ptFr->poly_vertices[boundaryPolys[i]][(k + 1)%n]]))
	    {
	      /* We take the previous and compute the intersection. */
	      valid = planesGet_intersection
		(planes,
		 ptFr->poly_points_data[ptFr->poly_vertices[boundaryPolys[i]][(k + 1)%n]],
		 ptFr->poly_points_data[ptFr->poly_vertices[boundaryPolys[i]][k]],
		 ptTo->poly_points_data[iVertice]);
	      g_return_val_if_fail(valid, redraw);
	      /* Other values than coordinates are copied. */
	      memcpy(ptTo->poly_points_data[iVertice] + 3,
		     ptFr->poly_points_data[ptFr->poly_vertices[boundaryPolys[i]][(k + 1)%n]] + 3,
		     sizeof(float) * (SurfacesPoints_userOffset + ptTo->bufferSize - 3));
	      g_return_val_if_fail(iv < ptTo->poly_num_vertices[iPoly], redraw);
	      ptTo->poly_vertices[iPoly][iv] = iVertice;
	      iVertice += 1;
	      iv += 1;
	    }
	  if ((verticeStatus[ptFr->poly_vertices[boundaryPolys[i]][k]] &&
	       !verticeStatus[ptFr->poly_vertices[boundaryPolys[i]][(k + 1)%n]]))
	    {
	      /* We take the previous and compute the intersection. */
	      valid = planesGet_intersection
		(planes, ptFr->poly_points_data[ptFr->poly_vertices[boundaryPolys[i]][k]],
		 ptFr->poly_points_data[ptFr->poly_vertices[boundaryPolys[i]][(k + 1)%n]],
		 ptTo->poly_points_data[iVertice]);
	      g_return_val_if_fail(valid, redraw);
	      memcpy(ptTo->poly_points_data[iVertice] + 3,
		     ptFr->poly_points_data[ptFr->poly_vertices[boundaryPolys[i]][k]] + 3,
		     sizeof(float) * (SurfacesPoints_userOffset + ptTo->bufferSize - 3));
	      g_return_val_if_fail(iv < ptTo->poly_num_vertices[iPoly], redraw);
	      ptTo->poly_vertices[iPoly][iv] = iVertice;
	      iVertice += 1;
	      iv += 1;
	    }
	}
      ptTo->num_polys += 1;
      ptTo->num_points = iVertice;
    }
  g_free(verticeStatus);
  g_free(boundaryPolys);

#if DEBUG == 1
  isosurfacesCheck_consistency(surf);
#endif
  return redraw;
}

static void sort_by_z(int *pointer[], double *zs, int begin, int end) {
   int i;
   int middle;
   int *temp;

   if( begin >= end ) return;
   temp = pointer[begin];
   pointer[begin] = pointer[(end+begin)/2];
   pointer[(end+begin)/2] = temp;
   middle = begin;
   for(i = begin +1; i <= end; i++) {
      if ( zs[*pointer[i]] < zs[*pointer[begin]] ) {
         temp = pointer[i];
         pointer[i] = pointer[++middle];
         pointer[middle] = temp;
      }
   }
   temp = pointer[begin];
   pointer[begin] = pointer[middle];
   pointer[middle] = temp;
   sort_by_z(pointer, zs, begin, middle-1);
   sort_by_z(pointer, zs, middle+1, end);
}


static double z_eye(float mat[16], float points[3])
{
  return
    (mat[ 2]*points[0]+
     mat[ 6]*points[1]+
     mat[10]*points[2]+
     mat[14]*1.)/
    (mat[ 3]*points[0]+
     mat[ 7]*points[1]+
     mat[11]*points[2]+
     mat[15]*1.);
}

SurfacesOrder* isosurfacesOrder_new()
{
  SurfacesOrder *order;

  order = g_malloc(sizeof(SurfacesOrder));
  order->allocatedSize = 0;
  order->any_pointer = (int**)0;
  order->any_z = (double*)0;
  order->polygon_number = (int*)0;

  return order;
}
void isosurfacesOrder_free(SurfacesOrder *order)
{
  g_return_if_fail(order);

  if (order->any_pointer)
    g_free(order->any_pointer);
  if (order->any_z)
    g_free(order->any_z);
  if (order->polygon_number)
    g_free(order->polygon_number);

  g_free(order);
}


void isosurfacesOrder_polygons(SurfacesOrder *order, Surfaces *surf[])
{
  int i,j, k, idSurf, nb;
  float mat[16];
  SurfacesPoints *points;

  g_return_if_fail(surf && order);

  DBG_fprintf(stderr, "Isosurfaces : re-ordering polygons in back to front order.\n");
  
  glGetFloatv(GL_MODELVIEW_MATRIX, mat);
  
  /* We compute the number of visible polygons. */
  nb = 0;
  for( idSurf = 0; surf[idSurf]; idSurf++)
    for ( i = 0; i < surf[idSurf]->nsurf; i++)
      {
	if (surf[idSurf]->resources[i]->rendered)
	  {
	    nb += surf[idSurf]->basePoints.num_polys_surf[i];
	    nb += surf[idSurf]->volatilePlanes.num_polys_surf[i];
	  }
      }
  DBG_fprintf(stderr, "Isosurfaces : found %d visible polygons.\n", nb);
  /* If the given order object need to be reallocated, we do it. */
  if (nb != order->allocatedSize)
    {
      DBG_fprintf(stderr, "Isosurfaces : need to reallocate the order object(%d %d).\n",
		  order->allocatedSize, nb);
      order->any_z = g_realloc(order->any_z, sizeof(double) * nb);
      order->any_pointer = g_realloc(order->any_pointer, sizeof(int*) * nb);
      order->polygon_number = g_realloc(order->polygon_number, sizeof(int) * nb * 4);
      order->allocatedSize = nb;
    }

  /* For all polygons, we compute the z position of the isobarycentre. */
  DBG_fprintf(stderr, "Isosurfaces : compute z values.\n");
  nb = 0;
  for( idSurf = 0; surf[idSurf]; idSurf++)
    {
      for (k = 0; k < 2; k++)
	{
	  points = (k == 0)?&surf[idSurf]->basePoints:&surf[idSurf]->volatilePlanes;
	  for( i = 0; i < points->num_polys; i++)
	    {
	      if (points->poly_surf_index[i] > 0 &&
		  surf[idSurf]->resources[points->poly_surf_index[i] - 1]->rendered)
		{
		  /* Ok, we handle polygon nb. */
		  order->polygon_number[4 * nb] = nb;
		  /* We store the surfaces id of this polygon. */
		  order->polygon_number[4 * nb + 1] = idSurf;
		  /* We store the id of this polygon. */
		  order->polygon_number[4 * nb + 2] = i;
		  /* We store k to say that this polygon is from base or volatile. */
		  order->polygon_number[4 * nb + 3] = k;
		  order->any_pointer[nb] = &(order->polygon_number[4 * nb]);
		  order->any_z[nb] = 0.0;
		  for(j = 0; j < points->poly_num_vertices[i]; j++)
		    order->any_z[nb] +=
		      z_eye(mat, points->poly_points_data[points->poly_vertices[i][j]]);
		  order->any_z[nb] /= points->poly_num_vertices[i];
		  nb += 1;
		}
	    }
	}
    }
  if (nb != order->allocatedSize)
    {
      g_error("Incorrect checksum in ordering (%d | %d).", nb, order->allocatedSize);
    }
   
  DBG_fprintf(stderr, "Isosurfaces : sorting...");
  sort_by_z(order->any_pointer, order->any_z, 0, nb - 1);
  DBG_fprintf(stderr, "OK\n");
}


void isosurfacesDraw_surfaces(int OpenGLId, Surfaces *surf[], SurfacesOrder *order)
{
  int i,j, ip, itp, idSurf;
  int nb;
  SurfacesPoints *points;
  int itp_old = -1, idSurf_old = -1;
  SurfaceResource *res;
  float rgba[4];
   
  g_return_if_fail(surf && order);
   
  DBG_fprintf(stderr, "Isosurfaces : rebuilding surfaces list\n");

  glNewList(OpenGLId, GL_COMPILE);  
  if (drawIntra)
    glEnable(GL_CULL_FACE);
  else
    glDisable(GL_CULL_FACE);
   
  glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
  glEnable(GL_BLEND);   
    
  /* We compute the number of visible polygons. */
  nb = 0;
  for( idSurf = 0; surf[idSurf]; idSurf++)
    for ( i = 0; i < surf[idSurf]->nsurf; i++)
      {
	if (surf[idSurf]->resources[i]->rendered)
	  {
	    nb += surf[idSurf]->basePoints.num_polys_surf[i];
	    nb += surf[idSurf]->volatilePlanes.num_polys_surf[i];
	  }
	DBG_fprintf(stderr, " |  - %d surface %d -> %d.\n",
		    idSurf, i, surf[idSurf]->basePoints.num_polys_surf[i]);
      }
  DBG_fprintf(stderr, " | found %d visible polygons.\n", nb);

  /* If order is out of date, we update. */
  if (nb != order->allocatedSize)
    isosurfacesOrder_polygons(order, surf);

  DBG_fprintf(stderr, " | draw polygons.\n");
  res = (SurfaceResource*)0;
  for(i = 0; i < nb; i++) {
    idSurf = *(order->any_pointer[i] + 1);
    /* ip is index of polygon in surf[idSurf]. */
    ip = *(order->any_pointer[i] + 2);
    points = (*(order->any_pointer[i] + 3) == 0)?
      &surf[idSurf]->basePoints:&surf[idSurf]->volatilePlanes;
    /* itp is index of surface of polygon ip. */
    itp = points->poly_surf_index[ip] - 1;
    /*      if(surf_draw_flag[itp] == 0) continue; */
    if(itp != itp_old || idSurf != idSurf_old)
      {
	res = surf[idSurf]->resources[itp];
/* 	if(res->rendered == FALSE) */
/* 	  continue; */
	if (drawIntra)
	  {
	    rgba[0] = 1. - res->color->rgba[0];
	    rgba[1] = 1. - res->color->rgba[1];
	    rgba[2] = 1. - res->color->rgba[2];
	    rgba[3] = res->color->rgba[3];
	  }
	else
	  openGLSet_color(res->material, res->color->rgba);

	/*          glCallList(lists_location + itp); */
	itp_old = itp;
	idSurf_old = idSurf;
      }
    /* This is where to find the points and the normals. */
    glBegin(GL_POLYGON);
    if (drawIntra)
      openGLSet_color(res->material, res->color->rgba);
    for(j = 0; j < points->poly_num_vertices[ip]; j++ ) {
      glNormal3fv(points->poly_points_data[points->poly_vertices[ip][j]] +
		  SurfacesPoints_normalOffset);
      glVertex3fv(points->poly_points_data[points->poly_vertices[ip][j]]);
    }
    glEnd();
    
    if (drawIntra)
      {
	glBegin(GL_POLYGON);
	openGLSet_color(res->material, rgba);
	for(j = points->poly_num_vertices[ip] - 1; j >= 0; j-- )
	  {
	    glNormal3f(-points->poly_points_data[points->poly_vertices[ip][j]][SurfacesPoints_normalOffset + 0],
		       -points->poly_points_data[points->poly_vertices[ip][j]][SurfacesPoints_normalOffset + 1],
		       -points->poly_points_data[points->poly_vertices[ip][j]][SurfacesPoints_normalOffset + 2]);
	    glVertex3fv(points->poly_points_data[points->poly_vertices[ip][j]]);
	  }
	glEnd();
      }
  }
   
  glEndList();
}

static void sort_block_by_z(int *order, float *z, int begin, int end) {
   int i;
   int middle;
   int temp;

   if( begin >= end ) return;

   /* We make sure end + begin / 2 has found its place. */
   temp = order[begin];
   order[begin] = order[(end + begin) / 2];
   order[(end + begin) / 2] = temp;

   middle = begin;
   for(i = begin + 1; i <= end; i++) {
      if ( z[order[i]] < z[order[begin]] ) {
         temp = order[i];
	 middle += 1;
         order[i] = order[middle];
         order[middle] = temp;
      }
   }
   temp = order[begin];
   order[begin] = order[middle];
   order[middle] = temp;
   sort_block_by_z(order, z, begin, middle-1);
   sort_block_by_z(order, z, middle+1, end);
}

void isosurfacesDuplicate(int totalList, int simpleBlockList,
			  VisuData *dataObj, gboolean reorder)
{
  float ext[3], *xyzTrans, boxTrans[3], *z;
  int i, j, k, n, *order;
  float mat[16];

  DBG_fprintf(stderr, "Isosurfaces: duplicate the primitive block.\n");
  /* Duplicate the surface according to the box extension. */
  visuDataGet_extension(dataObj, ext);
  
  if (reorder)
    glGetFloatv(GL_MODELVIEW_MATRIX, mat);

  n = (1 + 2 * (int)ext[0]) * (1 + 2 * (int)ext[1]) * (1 + 2 * (int)ext[2]);
  xyzTrans = g_malloc(sizeof(int) * 3 * n);
  z = (float*)0;
  if (reorder)
    z = g_malloc(sizeof(float) * n);
  order = g_malloc(sizeof(int) * n);
  n = 0;
  for (i = -(int)ext[0]; i < 1 + (int)ext[0]; i++)
    for (j = -(int)ext[1]; j < 1 + (int)ext[1]; j++)
      for (k = -(int)ext[2]; k < 1 + (int)ext[2]; k++)
	{
	  boxTrans[0] = (float)i;
	  boxTrans[1] = (float)j;
	  boxTrans[2] = (float)k;
	  visuDataConvert_boxCoordinatestoXYZ(dataObj, xyzTrans + 3 * n, boxTrans);
	  if (reorder)
	    z[n] = (mat[ 2] * xyzTrans[3 * n + 0] +
		    mat[ 6] * xyzTrans[3 * n + 1] +
		    mat[10] * xyzTrans[3 * n + 2] +
		    mat[14] * 1.) /
	      (mat[ 3] * xyzTrans[3 * n + 0] +
	       mat[ 7] * xyzTrans[3 * n + 1] +
	       mat[11] * xyzTrans[3 * n + 2] +
	       mat[15] * 1.);
	  order[n] = n;
	  n += 1;
	}

  if (reorder)
    /* we sort xyzTrans following z values. */
    sort_block_by_z(order, z, 0, n - 1);

  glNewList(totalList, GL_COMPILE);  
  for (i = 0; i < n; i++)
    {
/*       DBG_fprintf(stderr, "Isosurfaces: translate surfaces to box %d.\n", */
/* 		  order[i]); */
      glPushMatrix();
      glTranslated(xyzTrans[3 * order[i] + 0],
		   xyzTrans[3 * order[i] + 1],
		   xyzTrans[3 * order[i] + 2]);
      glCallList(simpleBlockList);
      glPopMatrix();
    }
  glEndList();

  g_free(order);
  g_free(xyzTrans);
  if (reorder)
    g_free(z);
}

int isosurfacesGet_nbSurfaces(Surfaces *surf)
{
  if (!surf)
    return -1;

  return surf->nsurf;
}

const gchar* isosurfacesGet_surfaceName(Surfaces *surf, int surf_index)
{
  int id;

  g_return_val_if_fail(surf, (gchar*)0);

  id = isosurfacesGet_surfacePosition(surf, surf_index);
  g_return_val_if_fail(id >= 0 && id < surf->nsurf, (gchar*)0);

/*   fprintf(stderr, "%d %d %p '%s'\n", surf_index, id, (gpointer)surf->resources[id], surf->resources[id]->surfnom); */
  return surf->resources[id]->surfnom;
}
gboolean isosurfacesGet_surfaceRendered(Surfaces *surf, int surf_index)
{
  int id;

  g_return_val_if_fail(surf, FALSE);

  id = isosurfacesGet_surfacePosition(surf, surf_index);
  g_return_val_if_fail(id >= 0 && id < surf->nsurf, FALSE);

  return surf->resources[id]->rendered;
}
SurfaceResource* isosurfacesGet_surfaceResource(Surfaces *surf, int surf_index)
{
  int id;

  g_return_val_if_fail(surf, (SurfaceResource*)0);

  id = isosurfacesGet_surfacePosition(surf, surf_index);
  g_return_val_if_fail(id >= 0 && id < surf->nsurf, (SurfaceResource*)0);

/*   fprintf(stderr, "%d %p\n", id, (gpointer)surf->resources); */
  return surf->resources[id];
}
void isosurfacesSet_surfaceResource(Surfaces *surf, int surf_index, SurfaceResource *res)
{
  int id;

  g_return_if_fail(surf && res);

  id = isosurfacesGet_surfacePosition(surf, surf_index);
  g_return_if_fail(id >= 0 && id < surf->nsurf);

  if (!surf->resources[id]->surfnom)
    isosurfacesFree_resource(surf->resources[id]);
  DBG_fprintf(stderr, "Surfaces: set resource %p (%s) to surface %d (%d).\n",
	      (gpointer)res, res->surfnom, surf_index, id);
  surf->resources[id] = res;
}
int isosurfacesGet_surfaceId(Surfaces *surf, int i)
{
  g_return_val_if_fail(i >= 0 && i < surf->nsurf, -1);
  
  return surf->ids[i];
}
int* isosurfacesGet_surfaceSortedById(Surfaces *surf)
{
  int *ids, i, j, tmp;

  g_return_val_if_fail(surf, (int*)0);

  ids = g_malloc(sizeof(int) * surf->nsurf);
  for (i = 0; i < surf->nsurf; i++)
    ids[i] = surf->ids[i];

  /* Idiotic algorithm to sort all surfaces. */
  for (i = 0; i < surf->nsurf; i++)
    for (j = 0; j < surf->nsurf; j++)
      if (ids[j] > ids[i])
        {
          tmp = ids[i];
          ids[i] = ids[j];
          ids[j] = tmp;
        }
  return ids;
}
int isosurfacesGet_newId(Surfaces *surf)
{
  int i, id;
  
  if (!surf)
    return 0;
  
  id = -1;
  for (i = 0; i < surf->nsurf; i++)
    id = MAX(surf->ids[i], id);
  DBG_fprintf(stderr, "Surfaces : get new id for surfaces %d.\n", id + 1);
  return id + 1;
}

int isosurfacesGet_surfacePosition(Surfaces *surf, int id)
{
  int i;

  g_return_val_if_fail(surf, -1);

  for (i = 0; i < surf->nsurf; i++)
    if (surf->ids[i] == id)
      return i;

  g_warning("Unfound surface with id %d.", id);
  return -1;
}

gboolean isosurfacesGet_drawIntra()
{
  return drawIntra;
}
gboolean isosurfacesSet_drawIntra(gboolean status)
{
  if (drawIntra == status)
    return FALSE;

  drawIntra = status;
  return TRUE;
}

void isoSurfacesSet_fitToBox(VisuData *data, Surfaces *surf)
{
  enum
    {
      _DXX = 0,
      _DYX,
      _DYY,
      _DZX,
      _DZY,
      _DZZ
    };
  int i,j;
  float inverted_local_box_matrix[3][3];
  float transform_box_matrix[3][3];
  float boxCoordinatestoXYZ[3][3];
  float old_poly_points[6];

  g_return_if_fail(data && surf);

  DBG_fprintf(stderr, "Isosurfaces: change the current box to fit to %p.\n",
	      (gpointer) data);

  inverted_local_box_matrix[0][0] = 1 / surf->local_box[_DXX];
  inverted_local_box_matrix[0][1] = - surf->local_box[_DYX] /
    surf->local_box[_DXX] / surf->local_box[_DYY];
  inverted_local_box_matrix[0][2] 
    = - ((surf->local_box[_DZX] / surf->local_box[_DXX] -
	  surf->local_box[_DYX] * surf->local_box[_DZY] / 
	  surf->local_box[_DXX] / surf->local_box[_DYY] ) /
	 surf->local_box[_DZZ] );
  inverted_local_box_matrix[1][0] = 0.;
  inverted_local_box_matrix[1][1] = 1 / surf->local_box[_DYY];
  inverted_local_box_matrix[1][2] = - surf->local_box[_DZY] /
    surf->local_box[_DYY] / surf->local_box[_DZZ];
  inverted_local_box_matrix[2][0] = 0.;
  inverted_local_box_matrix[2][1] = 0.;
  inverted_local_box_matrix[2][2] = 1 / surf->local_box[_DZZ];

  visuDataGet_boxMatrix(data, boxCoordinatestoXYZ);
  for(i = 0; i < 3; i++)
    for(j = 0; j < 3; j++)
      transform_box_matrix[i][j] =
	(boxCoordinatestoXYZ[i][0] * inverted_local_box_matrix[0][j] +
	 boxCoordinatestoXYZ[i][1] * inverted_local_box_matrix[1][j] +
	 boxCoordinatestoXYZ[i][2] * inverted_local_box_matrix[2][j]);
  DBG_fprintf(stderr, " | (%6.3f %6.3f %6.3f)\n", transform_box_matrix[0][0],
	      transform_box_matrix[0][1], transform_box_matrix[0][2]);
  DBG_fprintf(stderr, " | (%6.3f %6.3f %6.3f)\n", transform_box_matrix[1][0],
	      transform_box_matrix[1][1], transform_box_matrix[1][2]);
  DBG_fprintf(stderr, " | (%6.3f %6.3f %6.3f)\n", transform_box_matrix[2][0],
	      transform_box_matrix[2][1], transform_box_matrix[2][2]);

  DBG_fprintf(stderr, " | apply change to %d points.\n",
	      surf->basePoints.num_points);
  for (i = 0; i < surf->basePoints.num_points; i++)
    {
      old_poly_points[0] = surf->basePoints.poly_points_data[i][0];
      old_poly_points[1] = surf->basePoints.poly_points_data[i][1];
      old_poly_points[2] = surf->basePoints.poly_points_data[i][2];
      old_poly_points[3] = surf->basePoints.poly_points_data[i][SurfacesPoints_normalOffset + 0];
      old_poly_points[4] = surf->basePoints.poly_points_data[i][SurfacesPoints_normalOffset + 1];
      old_poly_points[5] = surf->basePoints.poly_points_data[i][SurfacesPoints_normalOffset + 2];
      matrix_productVector(surf->basePoints.poly_points_data[i],
			   transform_box_matrix, old_poly_points);
      matrix_productVector(surf->basePoints.poly_points_data[i] + SurfacesPoints_normalOffset,
			   transform_box_matrix, old_poly_points + 3);
    }
  surf->local_box[_DXX] = (double)boxCoordinatestoXYZ[0][0];
  surf->local_box[_DYX] = (double)boxCoordinatestoXYZ[0][1];
  surf->local_box[_DYY] = (double)boxCoordinatestoXYZ[1][1];
  surf->local_box[_DZX] = (double)boxCoordinatestoXYZ[0][2];
  surf->local_box[_DZY] = (double)boxCoordinatestoXYZ[1][2];
  surf->local_box[_DZZ] = (double)boxCoordinatestoXYZ[2][2];
}  

/***************/
/* Properties. */
/***************/
static gboolean isosurfaces_read_intra(gchar **lines, int nbLines, int position,
				       VisuData *dataObj _U_, GError **error)
{
  gboolean intra;
  
  g_return_val_if_fail(nbLines == 1, FALSE);

  if (!configFileRead_boolean(lines[0], position, &intra, 1, error))
    return FALSE;
  drawIntra = intra;
  return TRUE;
}
static void isosurfaces_export_resources(GString *data, VisuData *dataObj _U_)
{
  g_string_append_printf(data, "# %s\n", DESC_RESOURCE_INTRA);
  g_string_append_printf(data, "%s:\n", FLAG_RESOURCE_INTRA);
  g_string_append_printf(data, "    %d\n\n", drawIntra);
}
