/*   EXTRAITS DE LA LICENCE
	Copyright CEA, contributeurs : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)
  
	Adresse ml :
	BILLARD, non joignable par ml ;
	CALISTE, damien P caliste AT cea P fr.

	Ce logiciel est un programme informatique servant  visualiser des
	structures atomiques dans un rendu pseudo-3D. 

	Ce logiciel est rgi par la licence CeCILL soumise au droit franais et
	respectant les principes de diffusion des logiciels libres. Vous pouvez
	utiliser, modifier et/ou redistribuer ce programme sous les conditions
	de la licence CeCILL telle que diffuse par le CEA, le CNRS et l'INRIA 
	sur le site "http://www.cecill.info".

	Le fait que vous puissiez accder  cet en-tte signifie que vous avez 
	pris connaissance de la licence CeCILL, et que vous en avez accept les
	termes (cf. le fichier Documentation/licence.fr.txt fourni avec ce logiciel).
*/

/*   LICENCE SUM UP
	Copyright CEA, contributors : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)

	E-mail address:
	BILLARD, not reachable any more ;
	CALISTE, damien P caliste AT cea P fr.

	This software is a computer program whose purpose is to visualize atomic
	configurations in 3D.

	This software is governed by the CeCILL  license under French law and
	abiding by the rules of distribution of free software.  You can  use, 
	modify and/ or redistribute the software under the terms of the CeCILL
	license as circulated by CEA, CNRS and INRIA at the following URL
	"http://www.cecill.info". 

	The fact that you are presently reading this means that you have had
	knowledge of the CeCILL license and that you accept its terms. You can
	find a copy of this licence shipped with this software at Documentation/licence.en.txt.
*/
#ifndef VISU_DRAWING_H
#define VISU_DRAWING_H

#include <glib.h>
#include <stdio.h>

#include "visu_data.h"
#include "openGLFunctions/view.h"
#include "coreTools/toolFileFormat.h"

G_BEGIN_DECLS

/* The way visu renders its data is done by modules. They are called
   rendering methods and they describes how data are drawn on
   the screen. Many can be defined but only one is used at a
   time to render the data.
   One or more file type are associated with a rendering method. And
   a rendering method must specify the way to load the data it needs.
   A rendering method can introduced new resources and it must
   implement methods and create variable to deal with them :
    - one GHashTable that store as keys the name of all
    the resources introduced by the method and as value
    a description that is used as a commentary when the
    resources are exported to a file.
    - one method to read from a line the values of a specified
    resource.
    - one method to write to a file the values of a specified
    resource. */

#define VISU_ERROR_RENDERING visuRenderingClassGet_quark()
GQuark visuRenderingClassGet_quark();
/**
 * RenderingErrorFlag:
 * @RENDERING_ERROR_METHOD: Error from the rendering method.
 * @RENDERING_ERROR_FILE: Error when opening.
 * @RENDERING_ERROR_FORMAT: Wrongness in format.
 *
 * Thiese are flags used when reading a file with a rendering method.
 */
typedef enum
  {
    RENDERING_ERROR_METHOD,   /* Error from the rendering method. */
    RENDERING_ERROR_FILE,     /* Error when opening. */
    RENDERING_ERROR_FORMAT    /* Wrongness in format. */
  } RenderingErrorFlag;

/**
 * VISU_RENDERING_TYPE:
 *
 * return the type of #VisuRendering.
 */
#define VISU_RENDERING_TYPE	     (visu_rendering_get_type ())
/**
 * VISU_RENDERING:
 * @obj: a #GObject to cast.
 *
 * Cast the given @obj into #VisuRendering type.
 */
#define VISU_RENDERING(obj)	     (G_TYPE_CHECK_INSTANCE_CAST(obj, VISU_RENDERING_TYPE, VisuRendering))
/**
 * VISU_RENDERING_CLASS:
 * @klass: a #GClassObject to cast.
 *
 * Cast the given @klass into #VisuRenderingClass.
 */
#define VISU_RENDERING_CLASS(klass)    (G_TYPE_CHECK_CLASS_CAST(klass, VISU_RENDERING_TYPE, VisuRenderingClass))
/**
 * IS_VISU_RENDERING_TYPE:
 * @obj: a #GObject to test.
 *
 * Test if the given @ogj is of the type of #VisuRendering object.
 */
#define IS_VISU_RENDERING_TYPE(obj)    (G_TYPE_CHECK_INSTANCE_TYPE(obj, VISU_RENDERING_TYPE))
/**
 * IS_VISU_RENDERING_CLASS:
 * @klass: a #GClassObject to test.
 *
 * Test if the given @klass is of the type of #VisuRenderingClass class.
 */
#define IS_VISU_RENDERING_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE(klass, VISU_RENDERING_TYPE))
/**
 * VISU_RENDERING_GET_CLASS:
 * @obj: a #GObject to get the class of.
 *
 * It returns the class of the given @obj.
 */
#define VISU_RENDERING_GET_CLASS(obj)  (G_TYPE_INSTANCE_GET_CLASS(obj, VISU_RENDERING_TYPE, VisuRenderingClass))

/**
 * VisuRendering:
 *
 * A short way to identify #_VisuRendering structure.
 */
typedef struct _VisuRendering VisuRendering;
/**
 * VisuRenderingClass:
 *
 * A short way to identify #_VisuRenderingClass structure.
 */
typedef struct _VisuRenderingClass VisuRenderingClass;

/**
 * visu_rendering_get_type:
 *
 * This method returns the type of #VisuRendering, use VISU_RENDERING_TYPE instead.
 *
 * Returns: the type of #VisuRendering.
 */
GType visu_rendering_get_type(void);










/***************/
/* Public area */
/***************/



/**
 * visuRenderingLoadFunc:
 * @data: an already allocated #VisuData ;
 * @format: a pointer on a format (can be NULL if format is to be guessed) ;
 * @nSet: an integer ;
 * @error: a pointer to store a possible error,
 *         location must be initialized to (GError*)0.
 *
 * Interface of functions that allow to load data. The filenames are
 * retrieve from the given #VisuData, using the visuDataGet_file()
 * method. If no @format argument is given, then the format should be
 * guessed, otherwise, the given format is the one chosen by the
 * the user. WARNING : this format may not be the right format since
 * user selected. The @nSet argument is used to load a specific set of
 * nodes if the input format supports it. If @nSet is 0, then the
 * default set of nodes is loaded.
 *
 * Returns: TRUE if everything goes right.
 */
typedef gboolean (*visuRenderingLoadFunc) (VisuData *data, FileFormat *format,
					   int nSet, GError **error);


/**
 * renderingFormatLoadFunc:
 * @data: a #VisuData object ;
 * @filename: the access to the file to load ;
 * @format: a #FileFormat object (can be NULL) ;
 * @nSet: an integer ;
 * @error: a pointer to store possible errors.
 * 
 * This is an interface for a generic load method. This method read the file
 * positionned on @filename and populate or change the arrays in
 * @data. When enter this method, the @data argument is already
 * allocated but its arrays may be empty and unallocated (depending on
 * context). If the load method fails (because the format is wrong or
 * anything else), the @data argument should not be modified. If some
 * errors occur, the pointer @error will be instanciated. A
 * #VisuRendering can have several #renderingFormatLoadFunc methods
 * for each format they support. The @nSet argument is used to load a
 * specific set of nodes if the input format supports it. If @nSet is
 * 0, then the default set of nodes is loaded.
 *
 * Returns: FALSE if @data is unchanged (wrong format), TRUE otherwise (even if
 *          some minor errors have happened).
 */
typedef gboolean (*renderingFormatLoadFunc)(VisuData *data, const gchar* filename,
					    FileFormat *format,
					    int nSet, GError **error);
/**
 * RenderingFormatLoad_struct:
 * @name: a descriptive name ;
 * @fmt: file formats associated to this load method ;
 * @load: the load method ;
 * @priority: an int.
 *
 * The priority argument is used when a file is tested to be opened.
 * The smaller, the earlier the load method is tested.
 */
struct RenderingFormatLoad_struct
{
  gchar* name;

  FileFormat *fmt;
  renderingFormatLoadFunc load;

  int priority;
};
/**
 * RenderingFormatLoad:
 *
 * Short way to address #RenderingFormatLoad_struct objects.
 */
typedef struct RenderingFormatLoad_struct RenderingFormatLoad;

/**
 * createOpenGLElementFunc:
 * @visuData: a #VisuData object ;
 * @ele: a #VisuElement ;
 *
 * Such functions are called whenever a newElement is registered.
 *
 * Returns: an id representing an OpenGL list in which the element
 *          has been created.
 */
typedef int (*visuRenderingCreate_elementFunc) (VisuData *visuData, VisuElement* ele);
/**
 * createOpenGLNodeFunc:
 * @visuData: a #VisuData object ;
 * @node: a #VisuElement ;
 * @ele: a #VisuElement.
 *
 * Such functions are called when the OpenGl list of all the nodes is created.
 * The @ele parameter is the #VisuElement of the given @node and the @visuData one
 * points to the #VisuData object that contains this node.
 */
typedef void (*visuRenderingCreate_nodeFunc) (VisuData *visuData, VisuNode* node,
					      VisuElement* ele);
/**
 * getExtensOfNodeFunc:
 * @ele: a @VisuElement.
 *
 * This function is required to inform the OpenGL drawer
 * and to adapt the maximum size of the drawing area.
 *
 * Returns: the geometrical size of the element.
 */
typedef float (*visuRenderingGet_nodeExtendFunc) (VisuElement* ele);

/**
 * visuRendering_new:
 * @name: a string to label the method ;
 * @printName: a string to label the method that can be safely translated ;
 * @description: a string to describe the method ;
 * @nbFileType: an integer ;
 * @loadMethod: a @loadDataFromFileFunc which the load function of the method.
 *
 * Method used to create rendering methods. The @nbFileType argument is used to declare
 * the number of files that the method needs to read to render a set of data.
 * Use visuRenderingSet_fileType() method to set some #FileFormat for each kind.
 *
 * Returns: the newly created method of a null pointer if something goes
 * wrong.
 */
VisuRendering* visuRendering_new(const char* name, const char* printName,
				 const char* description, guint nbFileType,
				 visuRenderingLoadFunc           loadMethod,
				 visuRenderingCreate_elementFunc createElement,
				 visuRenderingCreate_nodeFunc    createNode,
				 visuRenderingGet_nodeExtendFunc getExtend);
/**
 * visuRenderingSet_fileType:
 * @method: a method ;
 * @fileTypeList: a list of #FileFormat ;
 * @fileType: an integer used as a key, must >= 0 and < method->nbFilesType ;
 * @name: a string to shortly describe the kind of file type (not NULL).
 *
 * Store a list of #FileFormat for the kind of file @fileType. The @name
 * argument is copied. but warning, the @fileTypeList GList* is not copied.
 */
void visuRenderingSet_fileType(VisuRendering *method, GList *fileTypeList,
			       guint fileType, const gchar *name);
/**
 * visuRenderingAdd_fileFormat:
 * @method: a method ;
 * @fmt: a #FileFormat ;
 * @fileType: an integer used as a key, must >= 0 and < method->nbFilesType.
 *
 * Add a file format descriptor to the list of already known file formats
 * of the key @fileType.
 */
void visuRenderingAdd_fileFormat(VisuRendering *method, FileFormat *fmt,
				 guint fileType);

/**
 * visuRenderingGet_nbFileType:
 * @method: a method.
 *
 * This method is used to get the number of kind of files needed to
 * render a set of data.
 *
 * Returns: how many kind of files are handled by the given VisuRendering.
 */
guint visuRenderingGet_nbFileType(VisuRendering *method);
/**
 * visuRenderingGet_fileType:
 * @method: a method ;
 * @fileType: an integer used as a key, must >= 0 and < method->nbFilesType.
 *
 * This method is used to get the file formats associated to a kind of input file
 * handled by the rendering method.
 *
 * Returns: a GList* with the #FileFormat. This GList should been considered read-only.
 */
GList* visuRenderingGet_fileType(VisuRendering *method, guint fileType);

/**
 * visuRenderingGet_fileTypeName:
 * @method: a method ;
 * @fileType: an integer used as a key, must >= 0 and < method->nbFilesType.
 *
 * This method is used to get the name associated to a kind of input file
 * handled by the rendering method.
 *
 * Returns: a string own by V_Sim. This string should been considered read-only.
 */
const gchar* visuRenderingGet_fileTypeName(VisuRendering *method, guint fileType);
const gchar* visuRenderingGet_name(VisuRendering *method, gboolean UTF8);
const gchar* visuRenderingGet_description(VisuRendering *method);

/**
 * visuRenderingGet_AllMethods:
 *
 * This method gives a GList with pointers to each rendering method.
 * Warning : the returned GList is not a copy, it must not be modified,
 * just read.
 *
 * Returns: A GList containing all the registered rendering methods.
 */
GList* visuRenderingClassGet_Allobjects();

/**
 * visuRenderingSet_icon:
 * @method: a method ;
 * @path: a path to an image file.
 *
 * This method is used to set the path to an icon for the specified method.
 * The path is copied, and the given name can be freed freely after a call to
 * this method.
 */
void visuRenderingSet_icon(VisuRendering *method, const gchar *path);
const gchar* visuRenderingGet_iconPath(VisuRendering *method);

/**
 * visuRenderingGet_sizeOfElement:
 * @method: a method ;
 * @ele: a #VisuElement to get the size of.
 *
 * This method is used to retrieve the radius of the sphere that contains
 * the @ele.
 *
 * Returns: the radius of the given element.
 */
float visuRenderingGet_sizeOfElement(VisuRendering *method, VisuElement *ele);
void visuRenderingEmit_elementSize(VisuRendering *method, float value);

/**
 * setVisuRenderingInUse:
 * @method: a VisuRendering.
 *
 * Choose the method used to render the data.
 */
void visuRenderingClassSet_current(VisuRendering* method);

/**
 * setVisuRenderingByName:
 * @name: the name of a rendering method.
 *
 * Choose the method used to render the data.
 *
 * Returns: 1 if the method exists and is set, 0 otherwise.
 */
gboolean visuRenderingClassSet_currentByName(const gchar* name);

/**
 * getVisuRenderingInUse:
 *
 * Get the current method used to render the data.
 *
 * Returns: the name of the currently used rendering method.
 */
VisuRendering* visuRenderingClassGet_current();
visuRenderingCreate_elementFunc visuRenderingClassGet_currentElementFunc();
visuRenderingCreate_nodeFunc visuRenderingClassGet_currentNodeFunc();
visuRenderingLoadFunc visuRenderingClassGet_currentLoadFunc();
VisuRendering* visuRenderingClassGet_methodByName(const gchar *name);
gboolean visuRenderingClassIs_currentByName(const gchar *name);

gint visuRenderingFormatCompare_priority(gconstpointer a, gconstpointer b);

G_END_DECLS


#endif
