/*   EXTRAITS DE LA LICENCE
	Copyright CEA, contributeurs : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)
  
	Adresse ml :
	BILLARD, non joignable par ml ;
	CALISTE, damien P caliste AT cea P fr.

	Ce logiciel est un programme informatique servant  visualiser des
	structures atomiques dans un rendu pseudo-3D. 

	Ce logiciel est rgi par la licence CeCILL soumise au droit franais et
	respectant les principes de diffusion des logiciels libres. Vous pouvez
	utiliser, modifier et/ou redistribuer ce programme sous les conditions
	de la licence CeCILL telle que diffuse par le CEA, le CNRS et l'INRIA 
	sur le site "http://www.cecill.info".

	Le fait que vous puissiez accder  cet en-tte signifie que vous avez 
	pris connaissance de la licence CeCILL, et que vous en avez accept les
	termes (cf. le fichier Documentation/licence.fr.txt fourni avec ce logiciel).
*/

/*   LICENCE SUM UP
	Copyright CEA, contributors : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)

	E-mail address:
	BILLARD, not reachable any more ;
	CALISTE, damien P caliste AT cea P fr.

	This software is a computer program whose purpose is to visualize atomic
	configurations in 3D.

	This software is governed by the CeCILL  license under French law and
	abiding by the rules of distribution of free software.  You can  use, 
	modify and/ or redistribute the software under the terms of the CeCILL
	license as circulated by CEA, CNRS and INRIA at the following URL
	"http://www.cecill.info". 

	The fact that you are presently reading this means that you have had
	knowledge of the CeCILL license and that you accept its terms. You can
	find a copy of this licence shipped with this software at Documentation/licence.en.txt.
*/
#include "visu_tools.h"
#include "visu_object.h"

#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h> /* For the access markers R_OK, W_OK ... */
#include <math.h>

/**
 * SECTION:visu_tools
 * @short_description: Basic tools and variables that are used all
 * around the program.
 *
 * <para>This module gives tools to handle strings (trim, append,
 * allocate...), GList and GHashTable (compare, delete functions), the
 * file system (test for existing file, for read/write permissions),
 * mathematical functions (floating point modulo, matrix product,
 * coordinates handling)...</para>
 */

void freeProperty(gpointer prop)
{
  g_return_if_fail(prop);

  DBG_fprintf(stderr, "Visu tools: free the data property '%s' (%p).\n",
	      ((Property*)prop)->name, ((Property*)prop)->data);
  if (((Property*)prop)->freeFunc)
    ((Property*)prop)->freeFunc(((Property*)prop)->data);
  g_free(prop);
}


gchar* getValidPath(GList **pathList, char *fileName, int accessMode)
{
  gchar *validPath;
  int fileOk;

  validPath = (char*)0;
  /* look for a directory to save or read a file. */
  fileOk = 0;
  while (*pathList && !fileOk)
    {
      validPath = g_build_filename((gchar*)(*pathList)->data, fileName, NULL);
      DBG_fprintf(stderr, "Visu Tools : test access (%d) for '%s' ... ", accessMode, validPath);
      fileOk = !access((char*)validPath, accessMode); /* return 0 if success */
      if (!fileOk)
	{
	  /* if access mode is write access and the file does not already exist :
	     we test if the directory has written permitions. */
	  if ( accessMode == W_OK && !g_file_test(validPath, G_FILE_TEST_EXISTS) )
	    fileOk = !access((char*)(*pathList)->data, accessMode); /* return 0 if success */
	  if (!fileOk)
	    {
	      DBG_fprintf(stderr, " failed.\n");
	      g_free(validPath);
	      *pathList = g_list_next(*pathList);
	    }
	}
      
    }
  if (fileOk)
    {
      DBG_fprintf(stderr, " OK.\n");
      return validPath;
    }
  else
      return (gchar*)0;
}


float fModulo(float a, int b) {
  float fb = (float)b;
  while(a < fb) a += fb;
  while(a >= fb) a -= fb;
  return a;
}

gchar* normalize_path(const gchar* path)
{
#if SYSTEM_X11 == 1
#define FILE_SYSTEM_SEP "/"
#endif
#if SYSTEM_WIN32 == 1
#define FILE_SYSTEM_SEP "\\"
#endif
  gchar **tokens;
  int i;
  GString *normPath;
  GList *lst, *tmplst;
  gchar *allPath, *dir;

  if (!path)
    return (gchar*)0;

  if (!g_path_is_absolute(path))
    {
      dir = g_get_current_dir();
      allPath = g_build_filename(dir, path, NULL);
      g_free(dir);
    }
  else
    allPath = g_strdup(path);

  tokens = g_strsplit(allPath, FILE_SYSTEM_SEP, -1);

  normPath = g_string_new("");

  lst = (GList*)0;
  for (i = 0; tokens[i]; i++)
    {
      /* If tokens[i] == . or is empty (because of //), we ignore. */
	if (!strcmp(tokens[i], "."))
	  continue;
      if (!tokens[i][0])
	continue;
      /* If token[i] == .. then we pop one element from lst. */
      if (!strcmp(tokens[i], ".."))
	{
	  lst = g_list_delete_link(lst, lst);
	  continue;
	}
      /* Token[i] is a valid chain, then we prepend it to the list. */
      lst = g_list_prepend(lst, tokens[i]);
    }
    /* Write the lst to the string. */
    tmplst = lst;
  while(tmplst)
    {
      g_string_prepend(normPath, (gchar*)tmplst->data);
      g_string_prepend(normPath, FILE_SYSTEM_SEP);
      tmplst = g_list_next(tmplst);
    }
  g_list_free(lst);
#if SYSTEM_WIN32 == 1
  g_string_erase(normPath, 0,1);
#endif
  g_strfreev(tokens);
  g_free(allPath);
  if (!normPath->str[0])
    g_string_append(normPath, FILE_SYSTEM_SEP);
  DBG_fprintf(stderr, "Visu Tools : normalizing path, from '%s' to '%s'.\n",
	      path, normPath->str);
  return g_string_free(normPath, FALSE);
}

#if GLIB_MINOR_VERSION < 5
gboolean g_file_set_contents(const gchar *fileName, const gchar *str,
			     gsize len _U_, GError **error _U_)
{
  FILE *f;

  f = fopen(fileName, "w");
  if (!f)
    return FALSE;
  if (fwrite(str, strlen(str), 1, f) != strlen(str))
    return FALSE;
  fclose(f);

  return TRUE;
}
#endif

static gchar* tagLookup(const gchar *tag, const gchar *buffer)
{
  char *ptTag, *ptStart, *ptEnd;

  ptTag = strstr(buffer, tag);
  if (!ptTag)
    return (gchar*)0;

  /* We check that tag was not in a commentary section. */
  ptStart = g_strrstr_len(buffer, (gssize)(ptTag - buffer), "<!--");
  if (!ptStart)
    return ptTag;
  ptEnd = g_strstr_len(ptStart, (gssize)(ptTag - ptStart), "-->");
  if (ptEnd)
    return ptTag;
  return tagLookup(tag, ptTag + strlen(tag));
}

gboolean visuToolsSubstitute_XML(GString *output, const gchar *filename,
				 const gchar *tag, GError **error)
{
  gchar *contents, *ptStart, *ptStop;
  gchar *tgStart, *tgEnd;
  gboolean valid;

    /* If file does exist, we read it and replace only the tag part. */
  contents = (gchar*)0;
  ptStart = (gchar*)0;
  if (g_file_test(filename, G_FILE_TEST_EXISTS))
    {
      valid = g_file_get_contents(filename, &contents, (gsize*)0, error);
      if (!valid)
	return FALSE;

      tgStart = g_strdup_printf("<%s", tag);

      ptStart = tagLookup(tgStart, contents);
      if (ptStart)
	g_string_prepend_len(output, contents, (gssize)(ptStart - contents));
      else
	{
	  ptStop = tagLookup("</v_sim>", contents);
	  if (ptStop)
	    g_string_prepend_len(output, contents, (gssize)(ptStop - contents));
	  else
	    {
	      ptStop = tagLookup("<v_sim>", contents);
	      if (ptStop)
		g_string_prepend(output, contents);
	      else
		{
		  g_string_prepend(output, contents);
		  g_string_prepend(output, "<?xml version=\"1.0\""
				   " encoding=\"utf-8\"?>\n<v_sim>");
		}
	    }
	  g_string_prepend(output, "\n");
	}

      g_free(tgStart);
    }
  else
    g_string_prepend(output, "<?xml version=\"1.0\" encoding=\"utf-8\"?>\n<v_sim>\n");

  /* If file does exist, we add the remaining parts. */
  if (contents && ptStart)
    {
      tgEnd = g_strdup_printf("</%s>", tag);
      
      ptStop = tagLookup(tgEnd, ptStart);
      if (ptStop)
	g_string_append(output, ptStop + strlen(tgEnd));
      else
	g_string_append(output, "\n</v_sim>");

      g_free(tgEnd);
    }
  else
    g_string_append(output, "</v_sim>\n");

  if (contents)
    g_free(contents);

  return TRUE;
}

static const gchar* unitNames[nb_units + 1] = {"undefined", "bohr", "angstroem", "nanometer", NULL};
static const gchar* unitNamesAll[nb_units + 1][4] =
  {{"undefined", NULL, NULL, NULL},
   {"bohr", "Bohr", "bohrd0", "Bohrd0"},
   {"angstroem", "Angstroem", "angstroemd0", "Angstroemd0"},
   {"nanometer", "Nanometer", "nanometerd0", "Nanometerd0"},
   {NULL, NULL, NULL, NULL}};
const gchar** visuToolsGet_unitNames()
{
  return unitNames;
}
static float unitValues[nb_units] = {-1.f, 5.291772108e-11f, 1e-10f, 1e-9f};
float visuToolsGet_unitValueInMeter(VisuUnits unit)
{
  g_return_val_if_fail(unit != unit_undefined && unit != nb_units, -1.f);

  DBG_fprintf(stderr, "Visu Tools: get unit (%d) length.\n", unit);
  return unitValues[unit];
}
VisuUnits visuToolsGet_unitFromName(const gchar *name)
{
  int i, j;

  for (i = 0; i < nb_units; i++)
    for (j = 0; j < 4 && unitNamesAll[i][j]; j++)
      if (!strcmp(name, unitNamesAll[i][j]))
	return i;
  
  return unit_undefined;
}
