/*  $Id: conjugation.cpp,v 1.4 2008/01/24 03:28:21 sarrazip Exp $
    conjugation.cpp - Generic conjugation interface

    verbiste - French conjugation system
    Copyright (C) 2003-2006 Pierre Sarrazin <http://sarrazip.com/>

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
    02111-1307, USA.
*/

#include "conjugation.h"

#include <libintl.h>
#define _(x) gettext(x)
#define N_(x) (x)

#include <iostream>
#include <string.h>

using namespace std;
using namespace verbiste;


FrenchVerbDictionary *
initVerbDict() throw(logic_error)
{
    /*  Create the French verb dictionary, which can conjugate and
	deconjugate French verbs.
    */
    const char *libdatadir = getenv("LIBDATADIR");
    if (libdatadir == NULL)
	libdatadir = LIBDATADIR;
    string conjFN  = libdatadir + string("/") + "conjugation-fr.xml";
    string verbsFN = libdatadir + string("/") + "verbs-fr.xml";
    return new FrenchVerbDictionary(conjFN, verbsFN);  // may throw
}


void
getConjugation(const FrenchVerbDictionary &fvd,
			const string &infinitive,
			VVVS &dest,
			bool includePronouns)
{
    const char *tname = fvd.getVerbTemplate(infinitive);
    if (tname == NULL)
	return;
    const TemplateSpec *templ = fvd.getTemplate(tname);
    if (templ == NULL)
	return;

    try
    {
	static const struct { Mode m; Tense t; } table[] =
	{
	    { INFINITIVE_MODE, PRESENT_TENSE },
	    { INDICATIVE_MODE, PRESENT_TENSE },
	    { INDICATIVE_MODE, IMPERFECT_TENSE },
	    { INDICATIVE_MODE, FUTURE_TENSE },
	    { INDICATIVE_MODE, PAST_TENSE },
	    { CONDITIONAL_MODE, PRESENT_TENSE },
	    { SUBJUNCTIVE_MODE, PRESENT_TENSE },
	    { SUBJUNCTIVE_MODE, IMPERFECT_TENSE },
	    { IMPERATIVE_MODE, PRESENT_TENSE },
	    { PARTICIPLE_MODE, PRESENT_TENSE },
	    { PARTICIPLE_MODE, PAST_TENSE },
	    { INVALID_MODE, INVALID_TENSE }  // marks the end
	};


	string radical = FrenchVerbDictionary::getRadical(infinitive, tname);

	for (int j = 0; table[j].m != INVALID_MODE; j++)
	{
	    dest.push_back(VVS());
	    VVS &tenseDest = dest.back();

	    VVS conjug;
	    fvd.generateTense(radical, *templ, table[j].m, table[j].t, conjug,
				includePronouns,
				fvd.isVerbStartingWithAspirateH(infinitive));

	    for (VVS::const_iterator p = conjug.begin(); p != conjug.end(); p++)
	    {
		tenseDest.push_back(VS());
		for (VS::const_iterator i = p->begin(); i != p->end(); i++)
		{
		    string s = *i;
		    tenseDest.back().push_back(s);
		}
	    }
	}
    }
    catch (logic_error &e)
    {
	dest.clear();
	return;
    }
}


static const char *tn[16] =
{
    N_("inf. pres."),
    "",
    "",
    "",
    N_("ind. pres."),
    N_("ind. imperf."),
    N_("ind. fut."),
    N_("ind. past"),
    N_("cond. pres."),
    N_("subj. pres."),
    N_("subj. imperf."),
    "",
    N_("imp. pres."),
    N_("part. pres."),
    N_("part. past"),
    ""
};


string getTenseNameForTableCell(int row, int col)
{
    if (row < 0 || row > 3 || col < 0 || col > 3)
    {
	assert(false);
	return "";
    }

    return _(tn[row * 4 + col]);
}


static
string
removePronoun(const string &s)
{
    string::size_type start = 0;
    if (s.find("que ") == 0)
	start = 4;
    else if (s.find("qu'") == 0)
	start = 3;

    static const char *pronouns[] =
    {
	"je ", "j'", "tu ", "il ", "nous ", "vous ", "ils ", NULL
    };

    for (size_t i = 0; pronouns[i] != NULL; i++)
    {
	size_t len = strlen(pronouns[i]);
	if (s.find(pronouns[i], start) == start)
	{
	    start += len;
	    break;
	}
    }

    return string(s, start, string::npos);
}


void
createTableCellText(const VVS &tense,
			const string &lowerCaseLatin1UserText,
			const string &openMark,
			const string &closeMark,
			VS &inflectionList)
{
    for (VVS::const_iterator it = tense.begin(); it != tense.end(); it++)
    {
	const VS &person = *it;

	string inf;
	for (VS::const_iterator i = person.begin(); i != person.end(); i++)
	{
	    if (i != person.begin())
		inf += ", ";

	    if (removePronoun(*i) == lowerCaseLatin1UserText)
		inf += openMark + *i + closeMark;
	    else
		inf += *i;
	}

	inflectionList.push_back(inf);
    }
}


string
createTableCellText(const VVS &tense,
			const string &lowerCaseLatin1UserText,
			const string &openMark,
			const string &closeMark)
{
    string persons;
    for (VVS::const_iterator it = tense.begin(); it != tense.end(); it++)
    {
	const VS &person = *it;

	if (it != tense.begin())
	    persons += "\n";

	for (VS::const_iterator i = person.begin(); i != person.end(); i++)
	{
	    if (i != person.begin())
		persons += ", ";

	    if (removePronoun(*i) == lowerCaseLatin1UserText)
		persons += openMark + *i + closeMark;
	    else
		persons += *i;
	}
    }
    return persons;
}
