/*******************************************************************************
*                                                                              *
*                                   Viewmol                                    *
*                                                                              *
*                               I N T E R N . C                                *
*                                                                              *
*                 Copyright (c) Joerg-R. Hill, December 2000                   *
*                                                                              *
********************************************************************************
*
* $Id: intern.c,v 1.5 2000/12/10 15:09:37 jrh Exp $
* $Log: intern.c,v $
* Revision 1.5  2000/12/10 15:09:37  jrh
* Release 2.3
*
* Revision 1.4  1999/05/24 01:26:04  jrh
* Release 2.2.1
*
* Revision 1.3  1999/02/07 21:51:39  jrh
* Release 2.2
*
* Revision 1.2  1998/01/26 00:48:12  jrh
* Release 2.1
*
* Revision 1.1  1996/12/10  18:41:35  jrh
* Initial revision
*
*/
#include<math.h>
#include<stdio.h>
#include "viewmol.h"

double bondAverage(struct MOLECULE *, int);
double bondLength(struct MOLECULE *, int, int);
double bondAngle(struct MOLECULE *, int, int, int);
double torsionAngle(struct MOLECULE *, int, int, int, int);
double dist(double, double, double, double, double, double);
double angle(double, double, double, double, double, double, double,
             double, double);
double torsion(double, double, double, double, double, double, double,
               double, double, double, double, double);

extern struct MOLECULE *molecules;
extern double bndfac;
extern int debug;

int calcInternal(struct MOLECULE *mol, int n)
{
  register int i, j;

  switch (mol->internals[n].type)
  {
    case BONDAVERAGE: mol->internals[n].value=bndfac*bondAverage(mol,
                                              mol->internals[n].atoms[0]);
                      i=mol->internals[n].atoms[0];
                      mol->internals[n].x=mol->atoms[i].x+0.05;
                      mol->internals[n].y=mol->atoms[i].y+0.05;
                      mol->internals[n].z=mol->atoms[i].z+0.05;
                      break;
    case BONDLENGTH:  mol->internals[n].value=bndfac*bondLength(mol,
                        mol->internals[n].atoms[0], mol->internals[n].atoms[1]);
                      i=mol->internals[n].atoms[0];
                      j=mol->internals[n].atoms[1];
                      mol->internals[n].x=(mol->atoms[i].x+mol->atoms[j].x)*0.5+0.05;
                      mol->internals[n].y=(mol->atoms[i].y+mol->atoms[j].y)*0.5+0.05;
                      mol->internals[n].z=(mol->atoms[i].z+mol->atoms[j].z)*0.5+0.05;
                      break;
    case ANGLE:       mol->internals[n].value=bondAngle(mol, mol->internals[n].atoms[0],
                                                             mol->internals[n].atoms[1],
                                                             mol->internals[n].atoms[2]);
                      i=mol->internals[n].atoms[1];
                      mol->internals[n].x=mol->atoms[i].x+0.05;
                      mol->internals[n].y=mol->atoms[i].y+0.05;
                      mol->internals[n].z=mol->atoms[i].z+0.05;
                      break;
    case TORSION:     mol->internals[n].value=torsionAngle(mol, mol->internals[n].atoms[0],
                                                                mol->internals[n].atoms[1],
                                                                mol->internals[n].atoms[2],
                                                                mol->internals[n].atoms[3]);
                      if (mol->internals[n].value > 360.) return(FALSE);
                      i=mol->internals[n].atoms[1];
                      j=mol->internals[n].atoms[2];
                      mol->internals[n].x=(mol->atoms[i].x+mol->atoms[j].x)*0.5+0.05;
                      mol->internals[n].y=(mol->atoms[i].y+mol->atoms[j].y)*0.5+0.05;
                      mol->internals[n].z=(mol->atoms[i].z+mol->atoms[j].z)*0.5+0.05;
                      break;
  }
  if (debug) printf("Internal coordinate %d: %d-%d-%d-%d, %f\n", n,
                    mol->internals[n].atoms[0], mol->internals[n].atoms[1],
                    mol->internals[n].atoms[2], mol->internals[n].atoms[3],
                    mol->internals[n].value);
  return(TRUE);
}

double bondAverage(struct MOLECULE *mol, int i)
{
  register double value=0.0;
  register int j, k=0;

  for (j=0; j<mol->nb; j++)
  {
    if (mol->bonds[j].first == i ||
        mol->bonds[j].second == i)
    {
      value+=bondLength(mol, mol->bonds[j].first, mol->bonds[j].second);
      k++;
    }
  }
  if (k > 0)
    value/=(double)k;
  else
    value=0.0;
  return(value);
}

double bondLength(struct MOLECULE *mol, int i, int j)
{
  return(dist(mol->atoms[i].x, mol->atoms[i].y, mol->atoms[i].z,
              mol->atoms[j].x, mol->atoms[j].y, mol->atoms[j].z));
}

double bondAngle(struct MOLECULE *mol, int i, int j, int k)
{
  return(angle(mol->atoms[i].x, mol->atoms[i].y, mol->atoms[i].z,
               mol->atoms[j].x, mol->atoms[j].y, mol->atoms[j].z,
               mol->atoms[k].x, mol->atoms[k].y, mol->atoms[k].z));
}

double torsionAngle(struct MOLECULE *mol, int i, int j, int k, int l)
{
  return(torsion(mol->atoms[i].x, mol->atoms[i].y, mol->atoms[i].z,
                 mol->atoms[j].x, mol->atoms[j].y, mol->atoms[j].z,
                 mol->atoms[k].x, mol->atoms[k].y, mol->atoms[k].z,
                 mol->atoms[l].x, mol->atoms[l].y, mol->atoms[l].z));
}

double dist(double x1, double y1, double z1, double x2, double y2, double z2)
{
  return(sqrt((x1-x2)*(x1-x2)+(y1-y2)*(y1-y2)+(z1-z2)*(z1-z2)));
}

double angle(double x1, double y1, double z1, double x2, double y2, double z2,
             double x3, double y3, double z3)
{
  double todeg=45.0/atan(1.0);
  double xa, ya, za, xb, yb, zb, r;

  xa=x1-x2;
  ya=y1-y2;
  za=z1-z2;
  xb=x3-x2;
  yb=y3-y2;
  zb=z3-z2;
  r=dist(x1, y1, z1, x2, y2, z2)*dist(x3, y3, z3, x2, y2, z2);
  r=(xa*xb+ya*yb+za*zb)/r;
  r=r > 1.0 ? 1.0 : r < -1.0 ? -1.0 : r;
  return(todeg*acos(r));
}

double torsion(double x1, double y1, double z1, double x2, double y2, double z2,
               double x3, double y3, double z3, double x4, double y4, double z4)
{
  double todeg=45.0/atan(1.0);
  double xa, ya, za, xb, yb, zb, xc, yc, zc, xd, yd, zd, xe, ye, ze, xf, yf, zf;
  double sgn, r;

  xa=x1-x2;
  ya=y1-y2;
  za=z1-z2;
  xb=x3-x2;
  yb=y3-y2;
  zb=z3-z2;
  xc=x4-x3;
  yc=y4-y3;
  zc=z4-z3;
  xd=ya*zb-yb*za;
  yd=xb*za-xa*zb;
  zd=xa*yb-xb*ya;
  xe=yc*zb-yb*zc;
  ye=xb*zc-xc*zb;
  ze=xc*yb-xb*yc;
  xf=yd*ze-ye*zd;
  yf=xe*zd-xd*ze;
  zf=xd*ye-xe*yd;
  sgn=xf*xb+yf*yb+zf*zb;
  r=sqrt((xd*xd+yd*yd+zd*zd)*(xe*xe+ye*ye+ze*ze));

/* The torsion angle is undefined (linear arrangement of atoms),
   return 400.0 as torsion angle which is outside the expected
   range */

  if (fabs(r) < 1.0e-6) return((double)400.);
  r=(xd*xe+yd*ye+zd*ze)/r;
  if (fabs(r) > 1.0) r/=fabs(r);
  if (sgn < 0.0)
    return(-todeg*acos(r));
  else
    return(todeg*acos(r));
}
