/* @(#) im_lindetect: Optimised convolution for line detection
 * @(#) Uses the entered mask and 3 rotated versions of it (each by 45 degrees)
 * @(#)
 * @(#) Usage 
 * @(#) int im_lindetect( in, out, m )
 * @(#) IMAGE *in, *out;
 * @(#) INTMASK *m;
 * @(#)
 * @(#) Returns 0 on sucess and -1 on error
 * @(#)
 * @(#) Returns an int pointer to valid offsets for rotating a square mask 
 * @(#) of odd size by 45 degrees.
 * @(#)
 * @(#) Usage 
 * @(#) int *im_offsets45( size )
 * @(#) int size;
 * @(#)
 * @(#) Returns an int pointer to valid offsets on sucess and -1 on error
 * @(#)
 * 
 * Author: N. Dessipris (Copyright, N. Dessipris 1991)
 * Written on: 08/05/1991
 * Modified on: 
 * 
 * old, horrible code kept for reference ... replaced by a small thing in 
 * im_compass.c
 */

/*

    This file is part of VIPS.
    
    VIPS is free software; you can redistribute it and/or modify
    it under the terms of the GNU Lesser General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 */

/*

    These files are distributed with VIPS - http://www.vips.ecs.soton.ac.uk

 */

#define MASKS	 4	/* rotate initial mask three times by 45 degrees each */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif /*HAVE_CONFIG_H*/
#include <vips/intl.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <vips/vips.h>

#ifdef WITH_DMALLOC
#include <dmalloc.h>
#endif /*WITH_DMALLOC*/

static void 
freebuffers( lutcnt, lut_orig, lut,
	lut_mask,
	cplut, lut_current, 
	pnts, cpnts,
	line, rot_offs )
int lutcnt;
int **lut_orig, **lut;
int **lut_mask[];
int **cplut, **lut_current;
PEL **pnts, **cpnts;
int *line;
int *rot_offs;
{
	int i;

	for (i=0; i<lutcnt; i++)
		free( (char*)lut_orig[i] );
	free( (char*)lut_orig ); free( (char*)lut );

	for (i=0; i<MASKS; i++)
		free( (char*)lut_mask[i] );
		
	free( (char*)cplut ); free( (char*)lut_current );
	free( (char*)pnts ); free( (char*)cpnts );
	free( (char*)line ); free( (char*)rot_offs );
}

/* 
*/
#define PIM_RINT 1
int im_lindetect_old(in, out, m)
IMAGE *in, *out;
INTMASK *m;
{
	int *pm;	/* pointer to mask */
	int ms;
	int *rot_offs; /* offsets for rotating the input mask 45 degrees */
	int x, y, i, ma;	/* counters*/
	int xstart, xend, ystart, yend;
	PEL **pnt, **pnts, **cpnt, **cpnts;
	int **lut_orig, **lut;	/* set by the function im_create_int_luts */
	int lutcnt; /* needed for freeing pointers malloc by im_create_... */
	int **lut_mask[MASKS];	/* points at rotated mask luts */
	int **lut_current, **cplut; /* used for processing */
	PEL *input;
	int *line, *cpline;	/* combined output */
	int os;			/* size of a line */
	int sum, temp, rounding;

/* check input arguments
 */
	if ( m->xsize != m->ysize )
		{
		im_errormsg("im_lindetect: Unable to accept non square mask");
		return(-1);
		}
/* Prepare output descriptor
 */
	if(im_iocheck(in, out) == -1)
		{ im_errormsg("im_lindetect: im_iocheck failed"); return(-1); }
	if((in->Coding!=IM_CODING_NONE)||(in->Bbits!=8)||(in->BandFmt!=IM_BANDFMT_UCHAR))
		{
		im_errormsg("im_lindetect: input not unsigned char uncoded");
		return(-1);
		}
	if ( im_cp_desc(out, in) == -1)	/* copy image descriptors */
		{ im_errormsg("im_lindetect: im_cp_desc failed"); return(-1); }
	out->Bbits = IM_BBITS_INT; out->BandFmt = IM_BANDFMT_INT;
 
	if ( im_setupout(out) == -1)
		{ im_errormsg("im_lindetect: im_setupout failed"); return(-1); }

/* find the coefficients necessary to rotate input mask by 45 degrees
 */
	if ( (rot_offs = im_offsets45( m->xsize )) == NULL )
		{im_errormsg("im_lindetect: im_offsets90 failed");return(-1);}

/* calculate luts
 */
	ms = m->xsize * m->ysize;
	lut_orig = (int**)calloc((unsigned)ms, sizeof(int**) );
	lut = (int**)calloc((unsigned)ms, sizeof(int**) );
	cplut = (int**)calloc((unsigned)ms, sizeof(int**) );
	lut_current = (int**)calloc((unsigned)ms, sizeof(int**) );
	if ( (lut_orig == NULL)||(lut == NULL)||
	     (lut_current == NULL)||(cplut == NULL))
		{ im_errormsg("im_lindetect: calloc failed(1)"); return(-1); }

	for (ma=0; ma<MASKS; ma++)
		{
		lut_mask[ma] = (int**)calloc((unsigned)ms, sizeof(int**) );
		if ( lut_mask[ma] == NULL )
			{
			im_errormsg("im_lindetect: calloc failed(2)");
			return(-1);
			}
		}
	ms = m->xsize * m->ysize;
	pm = m->coeff;
	if ( im__create_int_luts( pm, ms, lut_orig, lut, &lutcnt ) == -1 )
		{im_errormsg("im_lindetect: lut creation failed(1)");return(-1);}

/* Make lut_mask[...] point input and to 3 rotated masks
 */
	for ( i=0; i<ms; i++ )
		cplut[i] = lut[i];

	for (ma=0; ma<MASKS; ma++)
		{
		for ( i=0; i<ms; i++ )
			lut_current[i] = cplut[i];
		for ( i=0; i<ms; i++ )
			*(lut_mask[ma] + i) = lut_current[i];

		for ( i=0; i<ms; i++ ) /* rotate cplut */
			cplut[i] = lut_current[ rot_offs[i] ];
		}
	
/* allocate line buffers */
	os = out->Xsize * out->Bands;
	line = (int *)calloc( (unsigned) os, sizeof(int) );
	if ( line == NULL )
		{ im_errormsg("im_lindetect: calloc failed(2)"); return(-1); }

/* Do the processing */
	xstart = m->xsize/2 * out->Bands;
        xend = ( out->Xsize - (m->xsize - m->xsize/2) ) * out->Bands;
        ystart = m->ysize/2;
        yend = out->Ysize - (m->ysize - m->ysize/2);
        rounding = m->scale/2;
	
/* set input pointers */
	pnts = (PEL**)calloc((unsigned)ms, sizeof(PEL**) );
	cpnts = (PEL**)calloc((unsigned)ms, sizeof(PEL**) );
	if ( (pnts == NULL) || (cpnts == NULL) )
		{ im_errormsg("im_lindetect: calloc failed(3)"); return(-1); }
	pnt = pnts; 
	cpnt = cpnts;

	input = (PEL*)in->data;
	i = 0;
	for ( y=0; y<m->ysize; y++ )
		for ( x=0; x<m->xsize; x++ )
			pnt[i++] = (input+(x + y*in->Xsize)* in->Bands);

/* ZERO top part */
	memset((char*)line,0, os * sizeof(int));
	for (y=0; y<ystart; y++)
		if (im_writeline( y, out, (PEL*)line ) == -1)
			{
			freebuffers( lutcnt, lut_orig, lut,
				lut_mask,
				cplut, lut_current, 
				pnts, cpnts,
				line, rot_offs );
			im_errormsg("im_lindetect: im_writeline failed (1)");
			return(-1);
			}

#ifdef PIM_RINT
	fprintf(stderr, "input and rotated mask\n");
	pm = m->coeff;
	i=0;
	for (y=0; y<m->ysize; y++)
		{
		for (x=0; x<m->xsize; x++)
			{
			fprintf(stderr, "%4d\t", *(pm+i));
			i++;
			}
		fprintf(stderr, "\n");
		}
	fprintf(stderr, "\n");
	i=0;
	for (y=0; y<m->ysize; y++)
		{
		for (x=0; x<m->xsize; x++)
			fprintf(stderr, "%4d\t", *(pm+rot_offs[i++]));
		fprintf(stderr, "\n");
		}
#endif

	for ( y = ystart; y < yend; y++ )
		{
		cpline = line;
		for ( x = 0; x < xstart ; x++ )
			*cpline++ = (int)0;
		/* advance input pointers one input line */
		for ( i=0; i<ms; i++ )
			{
			cpnt[i] = pnt[i];
			pnt[i] += os;
			}

		for ( x = xstart; x < xend ; x++ )
			{
			sum = 0;
			for ( ma=0; ma<MASKS; ma++ )
				{
				temp = 0;
				lut_current = lut_mask[ma];
				for ( i=0; i<ms; i++ )
					temp += *(lut_current[i] + (*cpnt[i]) );
				temp = ( (temp+rounding)/m->scale ) + m->offset;
				if ( abs (temp) > sum )
					sum = abs( temp );
				}
			*cpline++ = sum;
			/* Advance pointers and get the next pel */
			for ( i=0; i<ms; i++ )
				cpnt[i]++;
			}

		for ( x = xend; x < os; x++ )
			*cpline++ = (int)0;

		if (im_writeline( y, out, (PEL*)line ) == -1)
			{
			freebuffers( lutcnt, lut_orig, lut,
				lut_mask,
				cplut, lut_current, 
				pnts, cpnts,
				line, rot_offs );
			im_errormsg("im_lindetect: im_writeline failed (2)");
			return(-1);
			}
		
		}	/* end of the y loop */

/* ZERO bottom part */
	memset((char*)line,0, os * sizeof(int));
	for (y=yend; y<out->Ysize; y++)
		if (im_writeline( y, out, (PEL*)line ) == -1)
			{
			freebuffers( lutcnt, lut_orig, lut,
				lut_mask,
				cplut, lut_current, 
				pnts, cpnts,
				line, rot_offs );
			im_errormsg("im_lindetect: im_writeline failed (3)");
			return(-1);
			}

	freebuffers( lutcnt, lut_orig, lut,
		lut_mask,
		cplut, lut_current, 
		pnts, cpnts,
		line, rot_offs );
	return(0);
}
