#include "global.h"

#include "net.h"
#include "netdefs.h"	/* SD_R_RTP */
#include "netImpl.h"
#include "mcast.h"
#include "pkt.h"	/* sendBYE */
#include "netobj.h"	/* my_host_id */
#include "stat.h"	/* ptime_net */
#include "channel.h"
#include "payload.h"	/* resetPayload */
#include "rtpsess.h"	/* RtpClearSessionsList */

#include "defaults.h"	/* DEF */
#include "world.h"	/* setChannelName */
#include "gui.h"	/* GuiAddInputTable */


/* fds */
int *tab_fd, *tab_manager_fd;
int cnt_fd, cnt_manager_fd;

/* channels */
Channel *pchannel_curr = NULL;		/* current channel */
Channel *pchannel_manager = NULL;	/* manager channel */

static Channel *channels_list = NULL;	/* channel head */
static u_int8 nb_chan;

/* unicast tunnel: Multicast Unicast Proxy (MUP) */
boolean mup;
char * mupname;
u_int32 mupaddr;
u_int16 mupctrlport;
u_int16 mupdataport;


struct _Channel * getCurrentChannel(void)
{
  return pchannel_curr;
}

struct _Channel * getManagerChannel(void)
{
  return pchannel_manager;
}

/*
 * Decode the string format "group[/port[/ttl]]". 
 * return group, port, ttl 
 * If chan_str == "" or NULL, we get the default values
 */
static void
decodeChannel(const char *chan_str, u_int32 *group, u_int16 *port, u_int8 *ttl)
{
  char chanstr[CHAN_LEN];
  char *ttlstr, *portstr, *groupstr;

  memset(chanstr, 0, sizeof(chanstr));
  strncpy(chanstr, chan_str, CHAN_LEN - 1);
  groupstr = chanstr;
  portstr = strchr(chanstr, '/');
  if (portstr != NULL)
    *portstr = '\0';
  *group = inet_addr(groupstr);
  if (portstr != NULL) {
    portstr++;
    ttlstr = strchr(portstr, '/');
    if (ttlstr != NULL)
      *ttlstr = '\0';
    *port = atoi(portstr);
    if (ttlstr != NULL) {
      *ttlstr = '\0';
      ttlstr++;
      *ttl = atoi(ttlstr);
    }
  }
}

/* check if multicast proxy */
static
void mcastProxy(void)
{
#ifndef VRENGD
  char *p;
  struct hostent *ph;
  struct in_addr *pa;
  static boolean done = FALSE;

  if (done)
    return;
  if (mcastproxystr)
    p = mcastproxystr;
  else
    p = getenv(MCAST_PROXY);	/* syntax: mcast_proxy=host:port */
  if (p && *p) {
    mupname = (char *) calloc(1, strlen(p));
    strcpy(mupname, p);
    p = strrchr(mupname, ':');
    *p = '\0';
    mupctrlport = atoi(++p);
    mupdataport = mupctrlport + 1;
    if ((ph = my_gethostbyname(mupname)) == NULL)
      return;
    if ((pa = (struct in_addr*) (ph->h_addr)) == NULL) {
      my_free_netdb(ph);
      return;
    }
    mupaddr = pa->s_addr;   
    my_free_netdb(ph);
    trace(DBG_NET, "mproxy=%s:%d", mupname, mupctrlport);
    mup = TRUE;
  }
  else
    mup = FALSE;
  done = TRUE;
#endif /* !VRENGD */
}

/*
 * Network Initialization
 */
void clearChannelsList(void)
{
  static boolean first = TRUE;

  if (first) {
    startTime(&ptime_net);
    mcastProxy();
    first = FALSE;
  }
  channels_list = NULL;
  RtpClearSessionsList();
  clearNetObjectsList();
}

/*
 * Alloc a Channel
 */
Channel * allocChannel(void)
{
  Channel *pchan;

  if ((pchan = (Channel *) calloc(1, sizeof(Channel))) == NULL)
    return ((Channel *) NULL);
  if (channels_list == (Channel *) NULL) {
    channels_list = pchan;
    pchan->next = NULL;
  }
  else {
    pchan->next = channels_list;
    channels_list = pchan;
  }
  trace(DBG_NET, "allocChannel: pchan=%x", pchan);
  return pchan;
}

/*
 * Free a Channel
 */
void freeChannel(Channel *pchannel)
{
  Channel *pchan;

  trace(DBG_NET, "freeChannel: pchannel=%x", pchannel);
  if (pchannel == pchannel_manager)
    return;
  if (pchannel == NULL) {
    trace(DBG_FORCE, "freeChannel: channel already released");
    return;
  }
  if (channels_list == NULL) {
    trace(DBG_FORCE, "freeChannel channels_list NULL");
    return;
  }
  for (pchan = channels_list; pchan != NULL; pchan = pchan->next) {
    if (pchan == pchannel) {
      channels_list = pchan->next;
      pchannel_curr = pchan->next;
      pchan->next = NULL;
      if (pchan == channels_list)
        channels_list = NULL;
      free(pchan);
      return;
    }
  }
  channels_list = NULL;
}

/*
 * Channel naming
 * init my_host_id, my_port_id, my_obj_id
 */
static
void namingChannel(Channel *pchannel)
{
  struct sockaddr_in ad;
  socklen_t adlen = sizeof(ad);
  Payload pl;			/* dummy payload */
  char hostname[MAXHOSTNAMELEN];
  struct hostent *ph;
  struct in_addr *pa;

  gethostname(hostname, sizeof(hostname));
  if ((ph = my_gethostbyname(hostname)) == NULL)
    return;
  if ((pa = (struct in_addr*) (ph->h_addr)) == NULL) {
    my_free_netdb(ph);
    return;
  }
  my_host_id = pa->s_addr;   
  my_free_netdb(ph);
 
  /* first packet to learn my_port_id */
  resetPayload(&pl);
  sendPayload(pchannel->sa[SA_RTP], &pl); /* needed for naming (port number) */
  if (getsockname(pchannel->sd[SD_W_RTP], (struct sockaddr *) &ad, &adlen) < 0)
    fatal("getsockname: %s", strerror(errno));

#if NEEDLOOPBACK
  my_port_id = (u_int16) (my_ssrc_id & 0x00007FFF);
#else
   my_port_id = ntohs(ad.sin_port);
#endif

  sendPayload(pchannel->sa[SA_RTCP], &pl); /* needed for proxy (source port) */
  trace(DBG_NET, "my_port_id = %d", my_port_id);
  my_obj_id = 0;
}

/*
 * Open a Channel
 * channel string is given by decodeChannel
 * channel structure is given bye Channel
 * return number of fd to survey
 *
 * Usage:
 * int *fds,i;
 * int count = openChannel(pchannel, chan_str, &fds);
 * for (i=0; i < count; i++) {
 *   addThisFdToWatchList(fds[i]);
 * }
 * create sockets mcast_recv_rtp and mcast_send_rtp
 *        sockets mcast_recv_rtcp and mcast_send_rtcp
 */
int openChannel(Channel *pchannel, const char *chan_str, int **pfds)
{
  u_int8 cntfds = 0;	/* number of fd */
  u_int32 group;	/* IP Mcast addr in network format */
  u_int16 port;		/* port */
  u_int8 ttl;
  
  assert(chan_str); assert(pchannel);
  decodeChannel(chan_str, &group, &port, &ttl);
  pchannel->group = group;
  pchannel->ttl = ttl;

  port &= ~1;	/* RTP compliant: even port */

  /* RTP channel */
  memset((char *) &(pchannel->sin_mcast_rtp), 0, sizeof(struct sockaddr_in));
  pchannel->sin_mcast_rtp.sin_family = AF_INET; 
  pchannel->sin_mcast_rtp.sin_port = htons(port);
  pchannel->sin_mcast_rtp.sin_addr.s_addr = group; 
  pchannel->sa[SA_RTP] = &(pchannel->sin_mcast_rtp);
  if ((pchannel->sd[SD_R_RTP] = createMcastRecvSocket(pchannel,
                                                      pchannel->sa[SA_RTP])) <0)
    fatal("can't open receive socket RTP");

  pchannel->sd[SD_W_RTP] = createSendSocket(ttl);
  cntfds += 2;
  trace(DBG_RTP, "pchannel->sa[SA_RTP]=%x", pchannel->sa[SA_RTP]);
  trace(DBG_RTP, "pchannel->sd[SD_R_RTP]=%d, pchannel->sd[SD_W_RTP]=%d",
                  pchannel->sd[SD_R_RTP], pchannel->sd[SD_W_RTP]);
  trace(DBG_RTP, "SA_RTP: port=%d", pchannel->sa[SA_RTP]->sin_port);

  /* RTCP channel */
  memset((char *) &(pchannel->sin_mcast_rtcp), 0, sizeof(struct sockaddr_in));
  pchannel->sin_mcast_rtcp.sin_family = AF_INET;
  pchannel->sin_mcast_rtcp.sin_port = htons(port + 1);	/* not ++port !!! */
  pchannel->sin_mcast_rtcp.sin_addr.s_addr = group;
  pchannel->sa[SA_RTCP] = &(pchannel->sin_mcast_rtcp);
  if ((pchannel->sd[SD_R_RTCP] = createMcastRecvSocket(pchannel,
                                                       pchannel->sa[SA_RTCP]))<0)
    fatal("can't open receive socket RTCP");

  pchannel->sd[SD_W_RTCP] = createSendSocket(ttl);
  cntfds += 2;
  trace(DBG_RTP, "pchannel->sa[SA_RTCP]=%x", pchannel->sa[SA_RTCP]);
  trace(DBG_RTP, "pchannel->sd[SD_R_RTCP]=%d, pchannel->sd[SD_W_RTCP]=%d",
                  pchannel->sd[SD_R_RTCP], pchannel->sd[SD_W_RTCP]);
  trace(DBG_RTP, "SA_RTCP: port=%d", pchannel->sa[SA_RTCP]->sin_port);

  /* UDP channel */
  pchannel->sa[SA_UDP] = NULL;

  /* RTP session initialization */
  pchannel->ssrc = RtpCreateSession(pchannel, group, port, ttl);
  if (pchannel->ssrc == 0)
    fatal("openChannel: can't create session");

  if (pchannel == pchannel_manager)
    my_mgr_ssrc_id = pchannel->ssrc;
  else {
    my_ssrc_id = pchannel->ssrc;
    pchannel->session->type = 1;	/* world */
  }

  namingChannel(pchannel);

  *pfds = pchannel->sd;
  return cntfds;
}

/*
 * Closing all sockets
 */
void closeChannel(Channel *pchannel)
{
  if (pchannel == NULL) {
    trace(DBG_FORCE, "closeChannel: pchannel NULL");
    return;	// abnormal
  }
  /* respect this order! */
  if (pchannel != pchannel_manager) {
    trace(DBG_RTP, "closeChannel: send BYE on channel=%x", pchannel);
    sendBYE(pchannel);
    trace(DBG_RTP, "closeChannel: statRTP on session=%x", pchannel->session);
    statRTP(pchannel->session);
    RtpCloseSession(pchannel->session, pchannel->ssrc);
    closeMcastSocket(pchannel);
    freeChannel(pchannel);
  }
  else {
    strcpy(worlds->name, "manager");	// HUGLY !
    setChannelName(MANAGER_CHANNEL);	/* -> channelname */
    statRTP(pchannel->session);
    RtpCloseSession(pchannel->session, pchannel->ssrc);
    closeMcastSocket(pchannel);
    free(pchannel);
  }
}

/* quitChannel */
void quitChannel(Channel *pchannel)
{
  int i;

  if (pchannel == NULL) {
    trace(DBG_FORCE, "quitChannel: pchannel NULL");
    return;	// abnormal
  }
  closeChannel(pchannel);
#ifndef VRENGD
  GuiRemoveInputTable(cnt_fd, WORLD_MODE);
#endif /* !VRENGD */
  for (i=0; i < cnt_fd; i++)
    close(tab_fd[i]);
  nb_chan--;
  trace(DBG_IPMC, "quitChannel: nb_chan=%d", nb_chan);
}

/* joinChannel: join the channel and return the new channel string */
/* set the global variables: pchannel_curr, cnt_fd, tab_fd */
char * joinChannel(const char *chan_str)
{
  Channel *pchannel;
  static char chanstr[CHAN_LEN];

  if ((pchannel = allocChannel()) == (Channel *) NULL)
    fatal("joinChannel: can't alloc channel");

  pchannel_curr = pchannel;
  memset(chanstr, 0, sizeof(chanstr));
  strcpy(chanstr, newvrmc(chan_str));
  trace(DBG_NET, "joinChannel: chan_str=%s -> chanstr=%s", chan_str, chanstr);
  notice("channel: %s", chanstr);
  cnt_fd = openChannel(pchannel, chanstr, &tab_fd);
#ifndef VRENGD
  GuiAddInputTable(cnt_fd, tab_fd, WORLD_MODE);
#endif /* !VRENGD */
  nb_chan++;
  trace(DBG_IPMC, "joinChannel: pchannel=%x nb_chan=%d cnt_fd=%d",
        pchannel, nb_chan, cnt_fd);
  return chanstr;
}

/* joinManagerChannel: only for vreng client */
/* set the global variables: pchannel_manager, cnt_manager_fd, tab_manager_fd */
char * joinManagerChannel(const char *chan_str)
{
  Channel *pchannel;
  static char chanstr[CHAN_LEN];

  if ((pchannel = allocChannel()) == (Channel *) NULL)
    fatal("joinManagerChannel: can't alloc channel");

  pchannel_manager = pchannel;
  memset(chanstr, 0, sizeof(chanstr));
  strcpy(chanstr, newvrmc(chan_str));
  trace(DBG_NET, "joinManagerChannel: manager.chan: %s", chanstr);
  cnt_manager_fd = openChannel(pchannel_manager, chanstr, &tab_manager_fd);
#ifndef VRENGD
  GuiAddInputTable(cnt_manager_fd, tab_manager_fd, MANAGER_MODE);
#endif /* !VRENGD */
  trace(DBG_INIT, "joinManagerChannel: pchannel=%x cnt_manager_fd=%d",
        pchannel, cnt_manager_fd);
  return chanstr;
}

/* joinDaemonChannel: only for vrengd */
/* pchannel_manager, cnt_manager_fd, tab_manager_fd: are global */
char * joinDaemonChannel(const char *chan_str)
{
  Channel *pchannel;
  static char chanstr[CHAN_LEN];

  if ((pchannel = allocChannel()) == (Channel *) NULL) {
    fprintf(stderr, "joinDaemonChannel: can't alloc channel\n");
    exit(2);
  }
  pchannel_manager = pchannel;
  memset(chanstr, 0, sizeof(chanstr));
  strncpy(chanstr, newvrmc(chan_str), sizeof(chanstr));
  fprintf(stderr, "joinManagerChannel: manager.chan: %s\n", chanstr);
  cnt_manager_fd = openChannel(pchannel_manager, chanstr, &tab_manager_fd);
  fprintf(stderr, "joinDaemonChannel: pchannel=%x cnt_manager_fd=%d\n",
                  (int) pchannel, cnt_manager_fd);
  return chanstr;
}


/*
 * get a socket fd by sockaddr_in*
 */
int getfdbysa(struct sockaddr_in *sa, int i_fd)
{
  int ret = -2;
  Channel *pchan;

  if (sa == NULL) {
    trace(DBG_FORCE, "getfdbysa: sa NULL");
    return -1;
  }
  if (channels_list == NULL) {
    trace(DBG_FORCE, "getfdbysa: channels_list NULL");
    return -1;
  }
  for (pchan = channels_list; pchan != NULL; pchan = pchan->next) {
    if (pchan->sa[SA_RTP] == sa || pchan->sa[SA_RTCP] == sa) {
      if (pchan->sd[i_fd] < 0) {
        ret = -1;
        break;
      }
      return(pchan->sd[i_fd]);
    }
  }
  ret = channels_list->sd[i_fd]; //octobre
  trace(DBG_FORCE, "getfdbysa: sa=%x not in table, force fd=%d", sa, ret);
  return ret;
}

int getFdSendRTP(struct sockaddr_in *sa)
{
  return getfdbysa(sa, SD_W_RTP);
}

int getFdSendRTCP(struct sockaddr_in *sa)
{
  return getfdbysa(sa, SD_W_RTCP);
}

/*
 * get a socket address by sockaddr_in*
 */
struct sockaddr_in * getsabysa(struct sockaddr_in *sa, int i_sa)
{
  Channel *pchan;

  if (sa == NULL) {
    trace(DBG_FORCE, "getsabysa: sa NULL");
    return ((struct sockaddr_in *) NULL);
  }
  if (channels_list == NULL) {
    trace(DBG_FORCE, "getsabysa: channels_list NULL");
    return ((struct sockaddr_in *) NULL);
  }
  for (pchan = channels_list; pchan != NULL; pchan = pchan->next) {
    if (pchan->sa[SA_RTP] == sa || pchan->sa[SA_RTCP] == sa) {
      return pchan->sa[i_sa];
    }
    trace(DBG_FORCE, "getsabysa error: pchan->sa[0]=%x, pchan->sa[1]=%x, sa=%x, i_sa=%d pchan->sa[i_sa]=%x", pchan->sa[SA_RTP], pchan->sa[SA_RTCP], sa, i_sa, pchan->sa[i_sa]);
  }
  return ((struct sockaddr_in *) NULL);
}

struct sockaddr_in * getSaRTCP(struct sockaddr_in *sa)
{
  return getsabysa(sa, SA_RTCP);
}

/*
 * get a channel by sockaddr_in*
 */
Channel * getChannelbysa(struct sockaddr_in *sa)
{
  Channel *pchan;

  if (sa == NULL) {
    trace(DBG_FORCE, "getChannelbysa: sa NULL");
    return ((Channel *) NULL);
  }
  if (channels_list == NULL) {
    trace(DBG_FORCE, "getChanelbysa: channels_list NULL");
    return ((Channel *) NULL);
  }
  for (pchan = channels_list; pchan != NULL; pchan = pchan->next) {
    if (pchan->sa[SA_RTP] == sa)
      return pchan;
    if (pchan->sa[SA_RTCP] == sa)
      return pchan;
    if (pchan->sa[SA_UDP] == sa)
      return pchan;
  }
  pchan = getCurrentChannel();	// hack
  trace(DBG_FORCE, "getChannelbysa: sa=%x not in table, force curr channel", sa);
  return pchan;
}


/*
 * manage group/port/ttl strings
 */

char * getvrgroup(const char *chan_str)
{
  char *p;
  char *chanstr;
  static char grpstr[GROUP_LEN];

  trace(DBG_IPMC, "getvrgroup: chan_str=%s", chan_str);
  memset(grpstr, 0, sizeof(grpstr));
  chanstr = strdup(chan_str);
  if ((p = strchr(chanstr, '/')) == NULL) {
    trace(DBG_FORCE, "getvrgroup: chan_str=%s", chan_str);
    return NULL;
  }
  *p = '\0';
  strcpy(grpstr, chanstr);
  free(chanstr);
  return grpstr;
}

u_int16 getvrport(const char *chan_str)
{
  u_int16 port;
  const char *p;

  if ((p = strchr(chan_str, '/')) == NULL)
    port = (u_int16) atoi(DEF_VRE_MPORT);
  else
    port = (u_int16) atoi(++p);
  trace(DBG_IPMC, "getvrport: port=%u", port);
  return port;
}

u_int8 getCurrentTtl(void)
{
  return universe.ttlorigin;
}

u_int8 getvrttl(const char *chan_str)
{
  u_int8 ttl;
  const char *p;

  if ((p = strrchr(chan_str, '/')) == NULL)
    ttl = (u_int8) atoi(DEF_VRE_TTL);
  else
    ttl = (u_int8) atoi(++p);
  trace(DBG_IPMC, "getvrttl: ttl=%u", ttl);
  return ttl;
}

char * newvrmc(const char *chan_str)
{
  u_int16 port;
  char *grpstr = NULL;
  static char chanstr[CHAN_LEN];

  trace(DBG_IPMC, "newvrmc: chan_str=%s", chan_str);
  memset(chanstr, 0, sizeof(chanstr));
  grpstr = strdup(getvrgroup(chan_str));
  port = (u_int16) getvrport(chan_str);
  strcpy(chanstr, createmc(grpstr, port, getCurrentTtl()));
  free(grpstr);
  trace(DBG_IPMC, "newvrmc: chanstr=%s", chanstr);
  return chanstr;
}

char * createmc(char *groupstr, u_int16 port, u_int8 ttl)
{
  static char chanstr[CHAN_LEN];

  sprintf(chanstr, "%s/%u/%d", groupstr, port, ttl);
  trace(DBG_IPMC, "createmc: chanstr=%s", chanstr);
  return chanstr;
}

int getworldid(const char *chan_str)
{
  int wid;
  char lowip;
  const char *p;
  u_int16 port;
  char grpstr[GROUP_LEN];

  memset(grpstr, 0, sizeof(grpstr));
  strncpy(grpstr, getvrgroup(chan_str), sizeof(grpstr));
  if ((p = strrchr(grpstr, '.')) == NULL)
    lowip = 255;
  else
    lowip = (char) atoi(++p);
  port = getvrport(chan_str);
  wid = lowip * port * getCurrentTtl();
  return wid;
}

