#include "global.h"

#include "net.h"
#include "netdefs.h"	/* SD_R_RTP */
#include "netImpl.h"
#include "mcast.h"
#include "channel.h"	/* Channel */
#include "defaults.h"	/* IPMC_ENABLE */


/*
 * Do REUSEADDR on the socket
 * return sock if OK, else -1
 */
static
int setReuseAddr(int sock)
{
  int one = 1;
  
  if (setsockopt(sock, SOL_SOCKET, SO_REUSEADDR, 
		 (char *) &one, sizeof(one)) < 0) {
    trace(DBG_FORCE, "SO_REUSEADDR: %s (%d)", strerror(errno), errno);
    return -1;
  }
  return sock;
}

/*
 * Control blocking(1)/non blocking(0)
 * return sock if OK, else -1
 */
static
int setBlocking(int sock, int block)
{
#if !defined(WIN32) || defined(CYGWIN32)
  int flags;

  if ((flags = fcntl(sock, F_GETFL)) < 0) {
    trace(DBG_FORCE, "F_GETFL: %s (%d)", strerror(errno), errno);
    return -1;
  }
  if (block)
    flags &= ~O_NONBLOCK;
  else
    flags |= O_NONBLOCK;
  if (fcntl(sock, F_SETFL, flags) < 0) {
    trace(DBG_FORCE, "F_SETFL: %s (%d)", strerror(errno), errno);
    return -1;
  }
#endif /* !WIN32 */
  return sock;
}

/*
 * Set the ttl
 * return sock if OK, else -1
 */
static
int setScope(int sock, u_int8 ttl)
{
  if (setsockopt(sock, IPPROTO_IP, IP_MULTICAST_TTL, &ttl, sizeof(ttl)) < 0) {
    trace(DBG_FORCE, "IP_MULTICAST_TTL: %s (%d)", strerror(errno), errno);    
    return -1;
  }
  return sock;
}

/*
 * Set loopback: active (1) either inactive (0)
 * return sock if OK, else -1
 */
int setLoopback(int sock, u_int8 loop)
{
  if (setsockopt(sock, IPPROTO_IP, IP_MULTICAST_LOOP, 
		 &loop, sizeof(loop)) < 0) {
#if IPMC_ENABLE
    trace(DBG_FORCE, "IP_MULTICAST_LOOP: %s (%d)", strerror(errno), errno);
#endif
    return -1;
  }
  return sock;
}

/*
 * Join group (ADD_MEMBERSHIP)
 * "group" in network format
 * return sock if OK, else -1
 */
static
int joinGroup(int sock, Channel *pchannel)
{
  memset((char *) &(pchannel->mreq), 0, sizeof(struct ip_mreq));
  pchannel->mreq.imr_multiaddr.s_addr = pchannel->group;
  pchannel->mreq.imr_interface.s_addr = htonl(INADDR_ANY);
  if (setsockopt(sock, IPPROTO_IP, IP_ADD_MEMBERSHIP, 
		 (void *) &(pchannel->mreq), sizeof(struct ip_mreq)) < 0) {
#if IPMC_ENABLE
    trace(DBG_FORCE, "IP_ADD_MEMBERSHIP: %s (%d)", strerror(errno), errno);
#endif
    return -1;
  }
  return sock;
}

/*
 * Leave group (DROP_MEMBERSHIP)
 * return sock if OK, else -1
 */
static
int leaveGroup(int sock, Channel *pchannel)
{
  if (setsockopt(sock, IPPROTO_IP, IP_DROP_MEMBERSHIP, 
		 (void *) &(pchannel->mreq), sizeof(struct ip_mreq)) < 0) {
#if IPMC_ENABLE
    trace(DBG_FORCE, "IP_DROP_MEMBERSHIP: %s (%d) on sock=%d channel=%x", strerror(errno), errno, sock, pchannel);
#endif
    return -1;
  }
  return sock;
}

static struct sockaddr_in s_proxy_ctrl;
static struct sockaddr_in s_proxy_data;

static
int16 setLocalPort(int sock)
{
  struct sockaddr_in s_local;
  socklen_t lg = sizeof(struct sockaddr_in);

  s_local.sin_family = AF_INET;
  s_local.sin_port = 0;
  s_local.sin_addr.s_addr = htonl(INADDR_ANY);
  if (getsockname(sock, (struct sockaddr *) &(s_local), (socklen_t *) &lg) < 0)
    return -1;
  return s_local.sin_port;
}

static
int createProxyCtrlSocket(void)
{
  static int sock_proxy_ctrl = -1;

  if (sock_proxy_ctrl != -1)
    return sock_proxy_ctrl;	/* already done */
  /* do once */
  if ((sock_proxy_ctrl = socket(PF_INET, SOCK_DGRAM, 0)) < 0) {
    perror("socket proxy_ctrl");
    return -1;
  }
  memcpy(&s_proxy_ctrl.sin_addr, &mupaddr, sizeof(u_int32));
  s_proxy_ctrl.sin_family = AF_INET;
  s_proxy_ctrl.sin_port = htons(mupctrlport);
  return sock_proxy_ctrl;
}

static
int createProxyDataSocket(struct sockaddr_in *sa)
{
  int sock_proxy_data = -1;

  if ((sock_proxy_data = socket(PF_INET, SOCK_DGRAM, 0)) < 0) {
    perror("socket proxy_data");
    return -1;
  }
  memcpy(&s_proxy_data.sin_addr, &mupaddr, sizeof(u_int32));
  s_proxy_data.sin_family = AF_INET;
  s_proxy_data.sin_port = htons(mupdataport);

  /* overload original socket by s_proxy_data */
  *sa = s_proxy_data;
  return sock_proxy_data;
}

static
int sendCmdToProxy(char *cmd, int sock, Channel *pchannel, struct sockaddr_in *sa)
{
  int sock_proxy_ctrl;
  int localport;
  struct in_addr groupaddr;
  char pkt_cmd[512];

  if ((sock_proxy_ctrl = createProxyCtrlSocket()) < 0)
    return -1;
  if ((localport = ntohs(setLocalPort(sock))) < 0)
    return -1;
  memcpy((char *) &groupaddr, (char *) &(pchannel->group), sizeof(u_int32));
  trace(DBG_FORCE, "sendCmdToProxy: %s %s %d %d (sa->sin_port=%d)",
                    cmd, inet_ntoa(groupaddr), localport, pchannel->ttl,
                    sa->sin_port);
  sprintf(pkt_cmd, "%s %s %d %d",
                    cmd, inet_ntoa(groupaddr), localport, pchannel->ttl);
  if (sendto(sock_proxy_ctrl, pkt_cmd, strlen(pkt_cmd), 0,
             (struct sockaddr *) &s_proxy_ctrl, sizeof(struct sockaddr_in)) < 0) {
    perror("sendto proxy_ctrl");
    return -1;
  }
  return sock_proxy_ctrl;
}

static
void leaveGroupByProxy(int ix_sa, int sock, Channel *pchannel)
{
  if (mup)
    sendCmdToProxy("drop", sock, pchannel, pchannel->sa[ix_sa]);
}

/*
 * Create a Multicast listen socket on the channel defined by group/port.
 * Return sock else -1 if problem
 */
int createMcastRecvSocket(Channel *pchannel, struct sockaddr_in *sa)
{
  int sock = -1;

  if ((sock = socket(PF_INET, SOCK_DGRAM, 0)) < 0) {
    trace(DBG_FORCE, "receive socket: %s", strerror(errno));
    return -1;
  }
  if (setReuseAddr(sock) < 0)
    perror("reuse failed");
  if (bind(sock, (struct sockaddr *) sa, sizeof(struct sockaddr_in)) < 0)
    trace(DBG_FORCE, "receive bind: %s", strerror(errno));

 /* test if unicast-multicast proxy */
  if (mup) {
    if (sendCmdToProxy("join", sock, pchannel, sa) < 0)
      return -1;
    sock = createProxyDataSocket(sa);
  }
  else if (joinGroup(sock, pchannel) < 0)
#if IPMC_ENABLE
    perror("join failed");
#else
    ;
#endif

  return sock;
}

/*
 * Create an Unicast socket
 * return fd, -1 if problem
 */
int createUcastSocket(u_int32 uni_addr, u_int16 port)
{
  int sock = -1;
  struct sockaddr_in ad;

  if ((sock = socket(PF_INET, SOCK_DGRAM, 0)) < 0) {
    trace(DBG_FORCE, "receive unicast socket: %s", strerror(errno));
    return -1;
  }
  memset((char *) &ad, 0, sizeof(struct sockaddr_in));
  ad.sin_family = AF_INET; 
  ad.sin_port = htons(port);
  ad.sin_addr.s_addr = htonl(uni_addr);

  setReuseAddr(sock);
  if (bind(sock, (struct sockaddr *) &ad, sizeof(struct sockaddr_in)) < 0) {
    trace(DBG_FORCE, "receive unicast bind: %s", strerror(errno));
  }
  return sock;
}

/*
 * Set a Multicast socket
 */
void setSendSocket(int sock, u_int8 ttl)
{
  if (!mup) {
    setScope(sock, ttl);
#if NEEDLOOPBACK
    setLoopback(sock, 1);		/* loopback */
#else
    setLoopback(sock, 0);		/* no loopback */
#endif
  }
  setBlocking(sock, 0);
}

/*
 * Create an UDP socket
 * Prevue pour emettre (uni et mcast) et recevoir (unicast donc)
 * Do setScope (ttl) and setLoopback (off)
 * return sock if OK, else -1
 */
int createSendSocket(u_int8 ttl)
{
  int sock = -1;

  if ((sock = socket(PF_INET, SOCK_DGRAM, 0)) < 0) {
    trace(DBG_FORCE, "send socket: %s", strerror(errno));    
    return -1;
  }
  setSendSocket(sock, ttl);
  return sock;
}

void closeMcastSocket(Channel *pchannel)
{
  if (mup)
    leaveGroupByProxy(SA_RTP, pchannel->sd[SD_R_RTP], pchannel);
  else
    leaveGroup(pchannel->sd[SD_R_RTP], pchannel);
  close(pchannel->sd[SD_R_RTP]);
  pchannel->sd[SD_R_RTP] = -1;
  close(pchannel->sd[SD_W_RTP]);
  pchannel->sd[SD_W_RTP] = -1;

  if (mup)
    leaveGroupByProxy(SA_RTCP, pchannel->sd[SD_R_RTCP], pchannel);
  else
    leaveGroup(pchannel->sd[SD_R_RTCP], pchannel);
  close(pchannel->sd[SD_R_RTCP]);
  pchannel->sd[SD_R_RTCP] = -1;
  close(pchannel->sd[SD_W_RTCP]);
  pchannel->sd[SD_W_RTCP] = -1;
}

void closeUcastSocket(Channel *pchannel)
{
  close(pchannel->sd[SD_R_UDP]);
  pchannel->sd[SD_R_UDP] = -1;
}
