#ifndef VRENGD

#include "global.h"
#include "net.h"
#include "wobject.h"
#include "wmgt.h"
#include "parse.h"
#include "col.h"	/* projectMovementOnObject */
#include "zv.h"		/* parseGeometry */
#include "door.h"

#include "payload.h"	/* Payload */
#include "helpers.h"	/* audio */


const WClass Door::wclass(DOOR_TYPE, "Door", Door::creator);

static u_int16 oid = 1;


/* angular variation modulo 2PI */
static
float deltaAngle(float a1, float a2)
{
  float d = a1 - a2;

  d -= M_2PI * ((int)(d / M_2PI));
  return d;
}

/* door initialization from a file */
void Door::creator(char *l)
{
  new Door(l);
}

Door::Door(char *l)
{
  l = parseName(l, this);
  l = parseFileLine(l);
  /* hinge */
  center.v[0] = (float) atof(l); l = strtok(NULL, SEP);
  center.v[1] = (float) atof(l); l = strtok(NULL, SEP);
  center.v[2] = (float) atof(l); l = strtok(NULL, SEP);
  pos.az = (float) atof(l); l = strtok(NULL, SEP);
  aopen = (float) atof(l); l = strtok(NULL, SEP);
  aclose = (float) atof(l); l = strtok(NULL, SEP);
  if (aclose == pos.az)
    status = DOOR_CLOSED;
  else
    status = DOOR_OPENED;
  aspeed = (float) atof(l);  l = strtok(NULL, SEP);
  size.v[0] = (float) atof(l); l = strtok(NULL, SEP);
  soh = parseGeometry(l);

  pos.x = center.v[0] + size.v[0] * (double) Cos((float)pos.az);
  pos.y = center.v[1] + size.v[0] * (double) Sin((float)pos.az);
  pos.z = center.v[2];

  initializeObject(this, DOOR_TYPE, VR_MOBILE);
  maxlasting[DOOR_TYPE] = DOOR_LASTING;

  initializeNetObject(this, oid++, DOOR_PROPS, VR_PERMANENT);

  /* here call persistency manager to know the door state */
  /* TODO */
  nature.persistency = TRUE;
  //BUG declarePersistInfo(&(noh));
  /* wait answer, may be not */
  
  move.ttl = 0.0;
} 

/* update times array */
void Door::updateTime(time_t sec, time_t usec, float *lasting)
{
  if (move.aspeed.v[0] == aspeed)
    move.ttl = MINI(ABSF(deltaAngle(pos.az,aopen)/move.aspeed.v[0]), move.ttl);
  else 
    move.ttl = MINI(ABSF(deltaAngle(pos.az,aclose)/move.aspeed.v[0]), move.ttl);

  *lasting = (float) (sec - move.sec) + (float) (usec - move.usec) / 1e6;
  if (*lasting < move.ttl) {
    move.ttl -= *lasting;
    move.sec = sec;
    move.usec = usec;
  }
  else {
    *lasting = move.ttl;
    move.ttl = 0;
    move.sec = move.usec = 0;
  }
}

/* equations system handling imposed motions */
void Door::changePosition(float lasting)
{
  pos.az += lasting * move.aspeed.v[0];
  pos.x = center.v[0] + size.v[0] * (double) Cos((float)pos.az);
  pos.y = center.v[1] + size.v[0] * (double) Sin((float)pos.az);
  pos.z = center.v[2];
}

/* condition to do position modifications */
boolean Door::change()
{
  return (move.ttl > 0.0005) ? TRUE : FALSE;
}

/* update a door towards the network */
boolean Door::updateToNetwork(const Pos &oldpos)
{
  boolean change = FALSE;
  
  if ((pos.x != oldpos.x) || (pos.y != oldpos.y)) {
    declareObjDelta(&(noh), DOOR_PROPXY);
    change = TRUE;
  }
  if (pos.az != oldpos.az) {
    declareObjDelta(&(noh), DOOR_PROPAZ);
    change = TRUE;
  }
  return change;
}

/* open */
static
void doorOpen(Door *po, void *data, time_t sec, time_t usec)
{
  if (po->status == DOOR_OPENED || po->status == DOOR_LOCKED)
    return;
  po->move.lspeed.v[0] = 0;
  po->move.lspeed.v[1] = 0;
  po->move.lspeed.v[2] = 0;
  po->move.aspeed.v[0] = po->aspeed;
  po->move.aspeed.v[1] = 0;
  po->move.aspeed.v[2] = 0;
  po->move.sec = sec;
  po->move.usec = usec;
  po->move.ttl = ABSF(deltaAngle(po->pos.az, po->aopen) / po->move.aspeed.v[0]);
  playSound(DOOROPENSND);
  po->status = DOOR_OPENED;

  /* here call persistency manager to inform change state */
  /* TODO */
  /* declarePersistSet(&(po->noh)); */
}

/* close */
static
void doorClose(Door *po, void *data, time_t sec, time_t usec)
{
  if (po->status == DOOR_CLOSED)
    return;
  po->move.lspeed.v[0] = 0;
  po->move.lspeed.v[1] = 0;
  po->move.lspeed.v[2] = 0;
  po->move.aspeed.v[0] = -po->aspeed;
  po->move.aspeed.v[1] = 0;
  po->move.aspeed.v[2] = 0;
  po->move.sec = sec;
  po->move.usec = usec;
  po->move.ttl = ABSF(deltaAngle(po->pos.az, po->aclose) / po->move.aspeed.v[0]);
  playSound(DOORCLOSESND);
  po->status = DOOR_CLOSED;

  /* here call persistency manager to inform change state */
  /* TODO */
}

/* unlock */
static
void doorUnlock(Door *po, void *data, time_t sec, time_t usec)
{
  switch (po->status) {
  case DOOR_LOCKED:
    po->status = DOOR_UNLOCKED;

    /* here call persistency manager to inform change state */
    /* TODO */
  default:
    break;
  }
}

/* lock */
static
void doorLock(Door *po, void *data, time_t sec, time_t usec)
{
  switch (po->status) {
  case DOOR_UNLOCKED:
  case DOOR_CLOSED:
    po->status = DOOR_LOCKED;

    /* here call persistency manager to inform change state */
    /* TODO */
  default:
    break;
  }
}

/* intersect with an user: stop */
void Door::whenIntersect(WObject *pcur, WObject *pold)
{
  if (status == DOOR_OPENED && projectMovementOnObject(pcur->pos, pold->pos, this->pos))
    updateReplica(pcur, pold->pos);
  else
    copyPositionAndBB(pold, pcur);
}

void Door::quit()
{
  oid = 1;
}

void doorInitFuncList(void)
{
  setFuncList[DOOR_PROPXY][DOOR_TYPE].pf = WO_PAYLOAD set_xy;
  setFuncList[DOOR_PROPZ][DOOR_TYPE].pf = WO_PAYLOAD set_z;
  setFuncList[DOOR_PROPAZ][DOOR_TYPE].pf = WO_PAYLOAD set_az;
  setFuncList[DOOR_PROPHNAME][DOOR_TYPE].pf = WO_PAYLOAD set_hname;

  getFuncList[DOOR_PROPXY][DOOR_TYPE].pf = WO_PAYLOAD get_xy;
  getFuncList[DOOR_PROPZ][DOOR_TYPE].pf = WO_PAYLOAD get_z;
  getFuncList[DOOR_PROPAZ][DOOR_TYPE].pf = WO_PAYLOAD get_az;
  getFuncList[DOOR_PROPHNAME][DOOR_TYPE].pf = WO_PAYLOAD get_hname;

  setMethodFunc(DOOR_TYPE, 0, WO_ACTION doorOpen, "Open");
  setMethodFunc(DOOR_TYPE, 1, WO_ACTION doorClose, "Close");
  setMethodFunc(DOOR_TYPE, 2, WO_ACTION doorUnlock, "Unlock");
  setMethodFunc(DOOR_TYPE, 3, WO_ACTION doorLock, "Lock");
}

#endif /* !VRENGD */
