/*
 * Interface entre vreng et zbuffer/zrender
 * (c) 1996,1997 Fabrice Bellard
 */

#include <GL/gl.h>

#include "global.h"
#include "zv.h"
#include "texture.h"	/* getTextureFromCache */
#include "md2.h"
#include "solid.h"
#include "draw.h"	/* SLICES, STACKS */
#include "net.h"
#include "wobject.h"	/* WObject */

#ifdef WANT_GLUT
#include "glutgui.h"	/* SetWindow3D */
#endif /* WANT_GLUT */

#ifndef VRENGD

/* zv_context global */
ZVContext zv_context;

static int solid_id = 0;
 
static const char *sid_str[] = {
  "box",
  "sphere",
  "cone",
  "torus",
  "rect",
  "disk",
  "line",
  "statue",
  "size",
  "radius",
  "radius2",
  "height",
  "texture",
  "tex_xn",
  "tex_yp",
  "tex_xp",
  "tex_yn",
  "tex_zp",
  "tex_zn",
  "emission",
  "diffuse",
  "ambient",
  "specular",
  "shininess",
  "geometry",
  "scale",
  "frames",
  "boxblend",
  "spot_direction",
  "spot_cutoff",
  "constant_attenuation",
  "linear_attenuation",
  "quadratic_attenuation",
  "style",
  "bbox",
  "dsphere",
  NULL,
};


/*
 * computes bounding box and draws vertex
 */
void zvVertex3fv(float *v)
{
  int i;
  ZVContext *c = &zv_context;

  if (c->first_bbox) {
    for (i=0; i < 3; i++) {
      c->bbox_min.v[i] = v[i];
      c->bbox_max.v[i] = v[i];
    }
    c->first_bbox = 0;
  }
  else {
    for (i=0; i < 3; i++) {
      if (v[i] < c->bbox_min.v[i]) c->bbox_min.v[i] = v[i]; /* min */
      if (v[i] > c->bbox_max.v[i]) c->bbox_max.v[i] = v[i]; /* max */
    }
  }
  glVertex3f(v[0], v[1], v[2]);
}

void zvVertex3f(float x, float y, float z)
{
  float v[3];

  v[0] = x; v[1] = y; v[2] = z;
  zvVertex3fv(v);
}


/*
 * Parser for solids
 */
static const char *solid_str, *curr_ptr;
static int ch;

static
void solidParserInit(const char *geometry)
{
  curr_ptr = solid_str = geometry;
  ch = curr_ptr[0];
}

static
int isGeometrySep(char c)
{
  return (c == ' ') || (c == ',') || (c == '\0') || (c == ENDFRAME);
}

static
void nextChar(void)
{
  if (ch)
    ch = *++curr_ptr;
}

static
void skipSpace(void)
{
  while (isspace(ch))
    nextChar();
}

static
void skipChar(int c)
{
  if (ch != c)
    trace(DBG_FORCE, "'%c' expected at position %d in '%s'",
		     c, curr_ptr-solid_str, solid_str);
  nextChar();
}

static
float getGeomFloat(void) 
{
  int sign = 1;
  char *p;
  char buf[20];

  if (ch == '-') {
    sign = -1;
    nextChar();
  }
  for (p = buf; isdigit(ch); nextChar())
    *p++ = ch;
  if (ch == '.') {
    *p++ = ch;
    nextChar();
    for ( ; isdigit(ch); nextChar())
      *p++ = ch;
  }
  *p++ = '\0';
  return sign * atof(buf);
}

static
u_int16 getGeomInt(void) 
{
  char *p;
  char buf[20];

  for (p = buf; isdigit(ch); nextChar())
    *p++ = ch;
  *p++ = '\0';
  return atoi(buf);
}

static
int findChar(char c)
{
  while (isGeometrySep(ch) && (ch != c))
    nextChar();
  return (ch == c);
}

static
void getGeomStr(char *buf)
{
  char *p;

  for (p = buf; !isGeometrySep(ch); nextChar())
    *p++ = ch;
  *p++ = '\0';
}

static
int getGeomId(void)
{
  char *p;
  const char **q;
  char buf[BUFSIZ];

  for (p = buf; isalnum(ch) || ch == '_'; nextChar())
    *p++ = ch;
  *p++ = '\0';
  for (q = sid_str; *q != NULL && (strcmp(*q, buf) != 0); )
    q++;
  if (*q == NULL) {
    trace(DBG_FORCE, "getGeomId: \"%s\" at pos='%d'", buf, curr_ptr-solid_str);
    return -1;
  }
  return q - sid_str;
}

/*
 * Routines to create a solid
 *
 * TODO: delete textures
 */

void SolidClose(void)
{
  solid_id = 1;	/* preserve localuser (sid=1) */
}

static
void addSolidToList(ZVContext *c, Solid *s)
{
  s->prev = c->last_solid;
  if (c->first_solid == NULL)
    c->first_solid = s;
  else 
    c->last_solid->next = s;
  c->last_solid = s;
}

void deleteSolidFromList(Solid *s)
{
  ZVContext *c = &zv_context;

  if (s) {
    if (s->next)
      s->next->prev = s->prev;
    else
      c->last_solid = s->prev;
    if (s->prev)
      s->prev->next = s->next;
    else
      c->first_solid = s->next;
    if (s->displaylist) {
      //PD for (int i=0; i < s->nbframes; i++)
      //PD   glDeleteLists(s->displaylist[i], 1);
      free(s->displaylist);
      s->displaylist = NULL;
    }
    if (s->object) {
      if (s->object->soh == s)
        s->object->soh = NULL; //ELC
      else if (s->object->soh == NULL)
	trace(DBG_FORCE, "deleteSolidFromList: null link s->object=%p s=%p name=%s", s->object, s, s->object->name.class_name);
      else
	trace(DBG_FORCE, "deleteSolidFromList: bad link s->object=%p s=%p name=%s", s->object, s, s->object->name.class_name);
    }
    free(s);
  }
}

/* computes and returns bbmax, bbmin in parseGeometry */
static
void solidSetBB(V3 *bbmax, V3 *bbmin, u_int16 frame)
{
  ZVContext *c = &zv_context;

  if (frame == 0) {
    *bbmax = c->bbox_max;
    *bbmin = c->bbox_min;
  }
  else {	/* frames */
    int j;

    for (j = 0; j < 3; j++) {
      bbmin->v[j] = (c->bbox_min.v[j] < bbmin->v[j] ?
                     c->bbox_min.v[j] : bbmin->v[j]);
      bbmax->v[j] = (c->bbox_max.v[j] > bbmax->v[j] ?
                     c->bbox_max.v[j] : bbmax->v[j]);
    }    
  }
}

static
int statueParser(V3 *bbmax, V3 *bbmin, int32 *disp_list, u_int16 frame)
{
  ZVContext *c = &zv_context;
  int nbf = 0, texture = -1, dlist = -1, geom_dlist = -1;
  float scale = 1.0;
  Md2 *md2;
  int32 frames[MAXFRAMES];
  char url_geom[URL_LEN];
  
  c->first_bbox = 1;
  
  while (1) {
    int id;
    
    skipSpace();
    if ((ch == '\0') || (ch == ENDFRAME))
      break;
    if (ch==',')
      nextChar();

    id = getGeomId(); skipChar('=');
    switch (id) {
    case ID_TEXTURE: {
      char url[URL_LEN];
      
      getGeomStr(url);
      texture = getTextureFromCache(url);
    } break;
    case ID_GEOMETRY:
      getGeomStr(url_geom);
      dlist = 0;
      break;
    case ID_SCALE:
      scale = getGeomFloat();
      break;
    case ID_FRAMES: {
      u_int16 cur_frame, end_frame, frame = 0;
      
      nbf = 1;
      skipChar(BEGINFRAME);
      while (ch != ENDFRAME) {
	cur_frame = getGeomInt();
	if (ch == SEPFRAME) {
	  skipChar(SEPFRAME);
	  end_frame = getGeomInt();
	}
        else {
	  end_frame = cur_frame;
	  if (ch != ENDFRAME)
	    skipChar(',');
	}
	while (cur_frame <= end_frame)
	  frames[frame++] = cur_frame++;
      }
      frames[frame] = -1;
      skipChar(ENDFRAME);
      break;
    }
    default:
      trace(DBG_FORCE, "solid parser: bad id=%d", id);
      return -1;
    }
  }
  if ((texture == -1) || (dlist == -1))
    return -1;
  if ((md2 = loadModelMd2(url_geom)) == NULL) 
    return -1;
  if (nbf == 0) {
    frames[0] = 0;
    frames[1] = -1;
  }
  
  for (nbf = 0; frames[nbf] != -1 && nbf < MAXFRAMES; nbf++) {
    if ((geom_dlist = getDlistModelMd2(md2, frames[nbf], frames[nbf], 0., scale)) <0) {
      freeModelMd2(md2);
      return -1;
    }
    dlist = glGenLists(1);
    glNewList(dlist, GL_COMPILE);
    glEnable(GL_TEXTURE_2D);
    glBindTexture(GL_TEXTURE_2D, texture);
    glCallList(geom_dlist);
    glDisable(GL_TEXTURE_2D);
    glEndList();
    disp_list[frame + nbf] = dlist;
    solidSetBB(bbmax, bbmin, frame);
  }
  freeModelMd2(md2);
  return nbf;
}

static
int solidParser(int shape, V3 *bbmax, V3 *bbmin, int32 *disp_list, u_int16 frame)
{
  ZVContext *c = &zv_context;
  int sid, i;
  int32 list;
  u_int8 style = STYLE_FILL;
  V3 size, v;
  float radius, radius2, height = 0.;
  float scale = 1.0;
  int32 texture;
  int32 box_textures[6];
  char url[URL_LEN];
  static float mat_emission[] = {0, 0, 0, 0};
  static float mat_diffuse[] = {1, 1, 1, 1};
  static float mat_ambient[] = {1, 1, 1, 1};
  static float mat_specular[] = {1, 1, 1, 1};
  static float mat_shininess[] = {20};
  static float light_spot_direction[] = {1, 1, 1, 1};
  static float light_spot_cutoff[] = {180};
  static float light_constant_attenuation[] = {1};
  static float light_linear_attenuation[] = {0};
  static float light_quadratic_attenuation[] = {0};
  static float scales[] = {1, 1, 1};

  c->first_bbox = 1;
  size = V3_New(1, 1, 1);
  radius = radius2 = 1;
  for (texture = -1, i=0; i < 6; box_textures[i++] = -1) ;
  for (i=0; i<4; i++) {
    mat_emission[i] = 0.0;
    mat_diffuse[i] = 1.0;
    mat_ambient[i] = 1.0;
    mat_specular[i] = 1.0;
    light_spot_direction[i] = 1.0;
  }
  for (i=0; i<3; i++) {
    scales[i] = 1.0;
  }
  mat_shininess[0] = 20.0;
  light_spot_cutoff[0] = 180.0;
  light_constant_attenuation[0] = 1.0;
  light_linear_attenuation[0] = 0.0;
  light_quadratic_attenuation[0] = 0.0;

  while (1) {
    skipSpace();
    if ((ch == '\0') || (ch == ENDFRAME))
      break;	/* end of geometry */
    if (ch==',') nextChar();
    sid = getGeomId(); skipChar('=');

    switch (sid) {
    case ID_RADIUS:
      radius = getGeomFloat();
      break;
    case ID_RADIUS2:
      radius2 = getGeomFloat();
      break;
    case ID_HEIGHT:
      height = getGeomFloat();
      break;
    case ID_SIZE:
    case ID_EMISSION:
    case ID_DIFFUSE:
    case ID_AMBIENT:
    case ID_SPECULAR:
    case ID_SPOT_DIRECTION:
#if 0
    case ID_SCALES:
#endif
      v.V_X = getGeomFloat(); skipChar(',');
      v.V_Y = getGeomFloat(); skipChar(',');
      v.V_Z = getGeomFloat();
      switch (sid) {
      case ID_SIZE:
	size = v;
	break;
      case ID_EMISSION:
	for (i=0; i<3; i++)
	  mat_emission[i] = mat_ambient[i] = v.v[i];
	break;
      case ID_DIFFUSE:
	for (i=0; i<3; i++)
	  mat_diffuse[i] = mat_ambient[i] = v.v[i];
	break;
      case ID_AMBIENT:
	for (i=0; i<3; i++)
	  mat_ambient[i] = v.v[i];
	break;
      case ID_SPECULAR:
	for (i=0; i<3; i++)
	  mat_specular[i] = v.v[i];
	break;
      case ID_SPOT_DIRECTION:
	for (i=0; i<3; i++)
	  light_spot_direction[i] = v.v[i];
	break;
#if 0
      case ID_SCALES:
	for (i=0; i<3; i++)
	  scales[i] = v.v[i];
	break;
#endif
      }
      break;
    case ID_SCALE:
      scale = getGeomFloat();
      break;
    case ID_STYLE:
      style = getGeomInt();
      break;
    case ID_SHININESS:
      mat_shininess[0] = getGeomFloat();
      break;
    case ID_SPOT_CUTOFF:
      light_spot_cutoff[0] = getGeomFloat();
      break;
    case ID_CONSTANT_ATTENUATION:
      light_constant_attenuation[0] = getGeomFloat();
      break;
    case ID_LINEAR_ATTENUATION:
      light_linear_attenuation[0] = getGeomFloat();
      break;
    case ID_QUADRATIC_ATTENUATION:
      light_quadratic_attenuation[0] = getGeomFloat();
      break;
    case ID_TEXTURE:
      getGeomStr(url);
      texture = getTextureFromCache(url);
      break;
    case ID_TEX_XP:
    case ID_TEX_XN:
    case ID_TEX_YP:
    case ID_TEX_YN:
    case ID_TEX_ZP:
    case ID_TEX_ZN:
      getGeomStr(url);
      box_textures[sid - ID_TEX_XP] = getTextureFromCache(url);
      break ;
    default:
      trace(DBG_FORCE, "solid parser: bad sid=%d", sid);
      return -1;
    }
  } /* end while */

#if WANT_GLUT
  SetWindow3D();	/* we must be on the 3D window */
#endif /* WANT_GLUT */

  /* displaylists generation */
  list = glGenLists(1);
  glNewList(list, GL_COMPILE);
  glCullFace(GL_BACK);
  glMaterialfv(GL_FRONT, GL_SHININESS, mat_shininess);
  glMaterialfv(GL_FRONT, GL_SPECULAR, mat_specular);
  glMaterialfv(GL_FRONT, GL_AMBIENT, mat_ambient);
  glMaterialfv(GL_FRONT, GL_DIFFUSE, mat_diffuse);
  glMaterialfv(GL_FRONT, GL_EMISSION, mat_emission);

  switch (shape) {
  case ID_BBOX:
    zv_context.bbox_max=size;
    zv_context.bbox_min.v[0]=-size.v[0];
    zv_context.bbox_min.v[1]=-size.v[1];
    zv_context.bbox_min.v[2]=-size.v[2];
    break;
  case ID_BOX:
  case ID_BOX_BLEND:
    glShadeModel(GL_FLAT);
    glMaterialfv(GL_FRONT, GL_SPECULAR, mat_specular);
    if (style == STYLE_LINES)
      drawBox(-size.V_X, size.V_X, -size.V_Y, size.V_Y, -size.V_Z, size.V_Z,
	      GL_LINE_LOOP, box_textures, style);
    else if (style == STYLE_POINTS)
      drawBox(-size.V_X, size.V_X, -size.V_Y, size.V_Y, -size.V_Z, size.V_Z,
	      GL_POINTS, box_textures, style);
    else
      drawBox(-size.V_X, size.V_X, -size.V_Y, size.V_Y, -size.V_Z, size.V_Z,
	      GL_QUADS, box_textures, style);
    break;
  case ID_SPHERE:
    glShadeModel(GL_SMOOTH);
    if (texture > 0) {
      glEnable(GL_TEXTURE_2D);
      glBindTexture(GL_TEXTURE_2D, texture);
      drawSphere(radius, SPHERE_SLICES, SPHERE_STACKS, STYLE_FILL);
      glDisable(GL_TEXTURE_2D);
    }
    else
      drawSphere(radius, SPHERE_SLICES, SPHERE_STACKS, style);
    break;
  case ID_DSPHERE:
    glShadeModel(GL_SMOOTH);
    drawSphere(radius, SPHERE_SLICES, SPHERE_STACKS, style);
    drawTorus(0.05, TORUS_CYLINDER, radius2, TORUS_CIRCLES, style);
    break;
  case ID_CONE:
    glDisable(GL_CULL_FACE);
    glShadeModel(GL_SMOOTH);
    glMaterialfv(GL_FRONT, GL_SPECULAR, mat_specular);
    if (texture > 0) {
      glEnable(GL_TEXTURE_2D);
      glBindTexture(GL_TEXTURE_2D, texture);
      drawCylinder(radius, radius2, height, CYLINDER_SLICES, CYLINDER_STACKS, STYLE_FILL);
      glDisable(GL_TEXTURE_2D);
    }
    else
      drawCylinder(radius, radius2, height, CYLINDER_SLICES, CYLINDER_STACKS, style);
    break;
  case ID_TORUS:
    glShadeModel(GL_SMOOTH);
    drawTorus(radius2, TORUS_CYLINDER, radius, TORUS_CIRCLES, style);
    break;
  case ID_RECT:
    glDisable(GL_CULL_FACE);
    if (texture > 0) {
      glEnable(GL_TEXTURE_2D);
      glBindTexture(GL_TEXTURE_2D, texture);
    }
    drawRect(size.V_X, size.V_Y, style);
    if (texture > 0)
      glDisable(GL_TEXTURE_2D);
    break;
  case ID_DISK:
    glDisable(GL_CULL_FACE);
    glShadeModel(GL_SMOOTH);
    glMaterialfv(GL_FRONT, GL_SPECULAR, mat_specular);
    if (texture > 0) {
      glEnable(GL_TEXTURE_2D);
      glBindTexture(GL_TEXTURE_2D, texture);
      drawDisk(radius, radius2, DISK_SLICES, DISK_LOOPS, STYLE_FILL);
      glDisable(GL_TEXTURE_2D);
    }
    else
      drawDisk(radius, radius2, DISK_SLICES, DISK_LOOPS, style);
    break;
  case ID_LINE:
    drawLine(size.V_X, size.V_Y);
    break;
  }
  glEndList();

  /* new solid in s->displaylist */
  disp_list[frame] = list;

  switch (shape) {
  case ID_RECT:
  case ID_LINE:
    /* solid without Bounding Box */
    break;
  default:
    /* solid with Bounding Box */
    solidSetBB(bbmax, bbmin, frame);
  }
  return 1;	/* only one frame */
}

static
int getNumberOfFrames(void)
{
  int nbframes = 1;	/* by default 1 frame */

  if (findChar(BEGINFRAME)) {
    /* this object has more that one frame */
    skipChar(BEGINFRAME);
    nbframes = getGeomInt();
    skipChar(ENDFRAME);
  }
  return nbframes;
}

/* Interface with WMGT */
Solid *parseGeometry(const char *geom)
{
  ZVContext *c = &zv_context;
  Solid *s;
  V3 bbmax, bbmin;

  if (strlen(geom) == 0) {
    trace(DBG_FORCE, "parseGeometry: no geometry");
    return NULL;
  }
  solidParserInit(geom);

  /* allocate the new solid */
  if ((s = (Solid *) calloc(1, sizeof(Solid))) == NULL) {
    trace(DBG_FORCE, "parseGeometry: can't calloc s");
    return NULL;
  }
  s->id = ++solid_id;
  IdM4(&s->posmat);
  s->visible = TRUE;
  s->nbframes = getNumberOfFrames();

  if ((s->displaylist = (int32 *) calloc(1, s->nbframes * sizeof(int32))) == NULL) {
    trace(DBG_FORCE, "parseGeometry: can't alloc displaylist");
    return NULL;
  }
  s->bbflag = TRUE;
  //dax-obs s->optsize = -1;
  addSolidToList(c, s);

  /* for each frame */
  for (u_int16 frame = 0; frame < s->nbframes; ) {
    int f = 0;

    if (s->nbframes > 1)
      skipChar(BEGINFRAME);

    switch (s->type = getGeomId()) {
    case ID_BBOX:
    case ID_BOX:
    case ID_SPHERE:
    case ID_DSPHERE:
    case ID_CONE:
    case ID_TORUS:
    case ID_RECT:
    case ID_DISK:
    case ID_LINE:
    case ID_BOX_BLEND:
      f = solidParser(s->type, &bbmax, &bbmin, s->displaylist, frame);
      break;
    case ID_STATUE:
      f = statueParser(&bbmax, &bbmin, s->displaylist, frame);
      break;
    default:
      trace(DBG_FORCE, "parseGeometry: s->type=%d geometry=%s", s->type, geom);
      return NULL;
    }
    if (f == -1) {
      trace(DBG_FORCE, "parseGeometry: f=-1 geometry=%s", geom);
      deleteSolidFromList(s);
      return NULL;
    }
    if (s->nbframes > 1)
      skipChar(ENDFRAME);
    frame += f;
  }

  /* BBox in solid: bbcenter and bbsize */
  for (int i = 0; i < 3; i++) {
    s->bbcenter.v[i] = (bbmax.v[i] + bbmin.v[i]) / 2;
    s->bbsize.v[i] = (bbmax.v[i] - bbmin.v[i]) / 2;
  }
  return s;
}

void getSolidBB(Solid *s, V3 *center, V3 *size)
{
  if (!s) {
    warning("GetSolidBB called with a NULL solid");
    return;
  }
  if (!s->bbflag)
    return;	/* this solid has no BB */

  V3 p[2], v, v1, qmin, qmax;

  for (int i=0; i < 3; i++) {
    p[0].v[i] = s->bbcenter.v[i] - s->bbsize.v[i]; /* min */
    p[1].v[i] = s->bbcenter.v[i] + s->bbsize.v[i]; /* max */
  }
  for (int l=0; l < 8; l++) {
    v.V_X = p[l % 2].V_X;
    v.V_Y = p[(l/2) % 2].V_Y;
    v.V_Z = p[(l/4) % 2].V_Z;
    MulM4V3(&v1, &s->posmat, &v);	/* v1 = posmat * v */
    if (l == 0)
      qmin = qmax = v1;
    else {
      for (int i=0; i < 3; i++) {
	if (v1.v[i] < qmin.v[i]) qmin.v[i] = v1.v[i];
	if (v1.v[i] > qmax.v[i]) qmax.v[i] = v1.v[i];
      }
    }
  }

  for (int i=0; i < 3; i++) {
    center->v[i] = (qmax.v[i] + qmin.v[i])/2;
    size->v[i] = (qmax.v[i] - qmin.v[i])/2;
  }
#ifdef DEBUG_BB
  trace(DBG_ZV, "BB: center=%g %g %g size=%g %g %g",
	center->X, center->Y, center->Z, size->X, size->Y, size->Z);
#endif
}

void setSolidPosition(Solid *s, M4 *mpos)
{
  if (s) s->posmat = *mpos;
}

void getSolidPosition(Solid *s, M4 *mpos)
{
  if (s) *mpos = s->posmat;
}

void setSolidVisible(Solid *s, boolean flag)
{
  if (s) s->visible = flag;
}

void setSolidBBFlag(Solid *s, boolean flag)
{
  if (s) s->bbflag = flag;
}

void setSolidFrame(Solid *s, u_int16 frame)
{
  if (s) s->curframe = frame % s->nbframes;
}

#endif /* !VRENGD */
