#ifndef STATE_MANAGER_H
#define STATE_MANAGER_H

// Copyright (c) The University of Cincinnati.  
// All rights reserved.

// UC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF 
// THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
// TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
// PARTICULAR PURPOSE, OR NON-INFRINGEMENT.  UC SHALL NOT BE LIABLE
// FOR ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING,
// RESULT OF USING, MODIFYING OR DISTRIBUTING THIS SOFTWARE OR ITS
// DERIVATIVES.

// By using or copying this Software, Licensee agrees to abide by the
// intellectual property laws, and all other applicable laws of the
// U.S., and the terms of this license.

// Authors: Malolan Chetlur             mal@ececs.uc.edu
//          Jorgen Dahl                 dahlj@ececs.uc.edu
//          Dale E. Martin              dmartin@cliftonlabs.com
//          Radharamanan Radhakrishnan  ramanan@ececs.uc.edu
//          Dhananjai Madhava Rao       dmadhava@ececs.uc.edu
//          Philip A. Wilsey            phil.wilsey@uc.edu

#include "warped.h"
#include "Configurable.h"

class State;
class SimulationObject;

/** The StateManager abstract base class.

    This is the abstract base class for all state manager implementations in
    the kernel. Different state manager implementations MUST be derived from 
    this abstract base class.  

*/
class StateManager : virtual public Configurable {
public:
   /**@name Public Class Methods of StateManager. */
   //@{

   /// Virtual Destructor
   virtual ~StateManager(){};

   /// return the state saving period
   virtual unsigned int getStatePeriod() = 0;

   /** Save the state of the object at the specified time.
       
       This is a pure virtual function that has to be overridden.

       @param VTime Time at which the state is saved.
       @param SimulationObject The object whose state is saved.
   */
  virtual void saveState( const VTime &, SimulationObject* ) = 0;
   
   /** Restore the state of the object at the specified time.
       
       This is a pure virtual function that has to be overridden.

       @param VTime State-restoring time.
       @param SimulationObject The object whose state should be restored.
       @return time of restored state
   */
   virtual const VTime &restoreState( const VTime&, SimulationObject* ) = 0;

   /** Garbage collect old state info for this object.

       This is a pure virtual function that has to be overridden.

       @param VTime Garbage collect upto this time.
       @param SimulationObject Object whose states are gbg collected.
       @return lowest time-stamped state in state queue
   */
   virtual const VTime &garbageCollect( const VTime&, SimulationObject* ) = 0;

   /** Print state queue.

       This is a pure virtual function that has to be overridden.

       @param currentTime Print statequeue upto current time.
       @param SimulationObject Object whose statequeue is printed.
   */
   virtual void printStateQueue(const VTime &currentTime, SimulationObject*,
                                ostream &) = 0;
  
   //@} // End of Public Class Methods of StateManager.

protected:

   /**@name Protected Class Methods of StateManager. */
  //@{
   
   /// Default Constructor.
   StateManager() {};

   //@} // End of Protected Class Methods of StateManager.
};

#endif
