#ifndef TIME_WARP_EVENT_SET_H
#define TIME_WARP_EVENT_SET_H

// Copyright (c) The University of Cincinnati.  
// All rights reserved.

// UC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF 
// THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
// TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
// PARTICULAR PURPOSE, OR NON-INFRINGEMENT.  UC SHALL NOT BE LIABLE
// FOR ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING,
// RESULT OF USING, MODIFYING OR DISTRIBUTING THIS SOFTWARE OR ITS
// DERIVATIVES.

// By using or copying this Software, Licensee agrees to abide by the
// intellectual property laws, and all other applicable laws of the
// U.S., and the terms of this license.

// Authors: Malolan Chetlur             mal@ececs.uc.edu
//          Jorgen Dahl                 dahlj@ececs.uc.edu
//          Dale E. Martin              dmartin@cliftonlabs.com
//          Radharamanan Radhakrishnan  ramanan@ececs.uc.edu
//          Dhananjai Madhava Rao       dmadhava@ececs.uc.edu
//          Philip A. Wilsey            phil.wilsey@uc.edu

#include "warped.h"
#include "Configurable.h"
#include "EventId.h"
class Event;
class SimulationObject;
class TimeWarpEventSetFactory;

/** The TimeWarpEventSet abstract base class.

This is the abstract base class for all Time Warp specific event set
implementations in the kernel. Different Time Warp event set
implementations must be derived from this abstract class.

The TimeWarpEventSet contains input events to the simulation kernel.  It
"owns" them and is responsible for their reclaimation at garbage
collection time.
*/
class TimeWarpEventSet : virtual public Configurable {

  /**@type friend class declarations */
  //@{
   
  /// Factory class for TimeWarpEventSet
friend class TimeWarpEventSetFactory;

  //@} // End of friend class declarations

public:
  /**@name Public Class Methods of TimeWarpEventSet. */
  //@{
   
  /// Virtual Destructor
  virtual ~TimeWarpEventSet();
   
  /** Insert an event into the event set.

  This is a pure virtual function that has to be overridden.

  @param Event The event to be inserted.

  @return Returns "true" if the event is in the past of the simulation
  object for which the event was inserted, and false if it's in the
  object's future.
  */

  virtual bool insert( const Event *event ) = 0;

  /** Cancel out positive message corresponding to the anti message.

  @param reclaimer The SimulationObject that will be used to reclaim the
  event.
  @param cancelId The EventId of the event to cancel.
  */
  virtual void handleAntiMessage( SimulationObject *reclaimer,
				  const EventId cancelId ) = 0;

  /** Remove an event from the event set.

  This is a pure virtual function that has to be overridden.

  @param Event The event to be removed.
  @param findMode Option for how to find the event.
  */
  virtual void remove( const Event *, findMode ) = 0;

  /** Remove and return the next event in the event set.

  This is a pure virtual function that has to be overridden.

  @return The removed event.
  */
  virtual const Event *getEvent(SimulationObject *) = 0;

  /** Remove and return the next event for a simulation object.

  This method differs from the getEvent(SimulationObject *)
  in that it returns an event for a specific simulation
  object and that event must have a receive time less than the
  time given by the input parameter. If there is no event
  in the input event set with a time less than the given time
  for the specified simulation object, NULL is returned.

  This is a pure virtual function and has to overridden.

  @param SimulationObject Reference to the simulation object.
  @param VTime Time constraint for the returned event.
  @return The event, or NULL if none found.
  */
  virtual const Event *getEvent( SimulationObject *, const VTime& ) = 0;

  /** Return a pointer to the next event in the event set.

  This is a pure virtual function and has to be overridden.

  @return A pointer to the next event in the event set.
  */
  virtual const Event *peekEvent(SimulationObject *) = 0;

  /** Return a pointer the next event for a simulation object.

  This method differs from the peekEvent(SimulationObject *)
  in that it returns an event for a specific simulation
  object and that event must have a receive time less than the
  time given by the input parameter. If there is no event
  in the input event set with a time less than the given time
  for the specified simulation object, NULL is returned.

  This is a pure virtual function and has to overridden.

  @param SimulationObject Reference to the simulation object.
  @param VTime Time constraint for the returned event.
  @return The peeked event, or NULL if none found.
  */
  virtual const Event *peekEvent(SimulationObject *, const VTime&) = 0;

  /** Garbage collect the event set upto a given time.

  This is a pure virtual function that has to be overridden.

  @param VTime Time upto which to garbage collect.
  */
  virtual void garbageCollect( SimulationObject *, const VTime & ) = 0;

  /**
     Rollback to the time passed in.  Could involve unprocessing events,
     i.e. taking events we've considered processed and making them go
     back into the unprocessed state.
  */
  virtual void rollback( SimulationObject *object, const VTime &rollbackTime ) = 0;

  virtual void debugDump( const string &, ostream & ){}

  //@} // End of Public Class Methods of TimeWarpEventSet.

protected:
  /**@name Protected Class Methods of TimeWarpEventSet. */
  //@{

  /// Default Constructor
  TimeWarpEventSet() {};

  //@} // End of Protected Class Methods of TimeWarpEventSet.
};

#endif
