/**************************************************************************
* This file is part of the WebIssues program
* Copyright (C) 2006 Michał Męciński
* Copyright (C) 2007-2008 WebIssues Team
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
**************************************************************************/

#ifndef ISSUECONDITION_H
#define ISSUECONDITION_H

#include <QString>

#include "abstracttablemodel.h"

namespace WebIssues
{

/**
* Structure representing a condition for a single column.
*
* It is used to filter table rows according to a list of conditions.
*/
class ColumnCondition
{
public:
    /**
    * Type of the condition.
    */
    enum Condition
    {
        String_Equals,
        String_NotEquals,
        String_Contains,
        String_NotContains,
        String_StartsWith,
        String_NotStartsWith,
        String_EndsWith,
        String_NotEndsWith,

        Numeric_Equals = 100,
        Numeric_NotEquals,
        Numeric_GreaterThan,
        Numeric_GreaterOrEqual,
        Numeric_LessThan,
        Numeric_LessOrEqual,

        DateTime_Equals = 200,
        DateTime_NotEquals,
        DateTime_Before,
        DateTime_After,

        Age_OlderThan = 300,
        Age_NewerThan,

        Check_IsEmpty = 400,
        Check_IsNotEmpty,
        Check_IsMe,
        Check_IsNotMe
    };

public:
    /**
    * Default constructor.
    */
    ColumnCondition();

    /**
    * Destructor.
    */
    ~ColumnCondition();

public:
    /**
    * Set the column to filter.
    */
    void setColumn( Column column ) { m_column = column; }

    /**
    * Return the filtered column.
    */
    Column column() const { return m_column; }

    /**
    * Set the type of the condition.
    */
    void setCondition( Condition condition ) { m_condition = condition; }

    /**
    * Return the type of the condition.
    */
    Condition condition() const { return m_condition; }

    /**
    * Set the operand of the condition.
    */
    void setOperand( const QString& operand ) { m_operand = operand; }

    /**
    * Return the operand of the condition.
    */
    const QString& operand() const { return m_operand; }

public:
    /**
    * Match the value with the condition.
    *
    * @param value The value to test.
    *
    * @return @c true if the value matches the condition, @c false otherwise.
    */
    bool match( const QString& value ) const;

public:
    /**
    * Return the user-friendly name of the condition.
    */
    static QString name( Condition condition );

private:
    bool matchString( const QString& value ) const;
    bool matchNumeric( const QString& value ) const;
    bool matchDateTime( const QString& value ) const;
    bool matchAge( const QString& value ) const;

    bool checkIsMe( const QString& value ) const;

private:
    static QString tr( const char* text );

private:
    Column m_column;
    Condition m_condition;
    QString m_operand;
};

}

#endif
