/**************************************************************************
* This file is part of the WebIssues program
* Copyright (C) 2006 Michał Męciński
* Copyright (C) 2007-2009 WebIssues Team
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
**************************************************************************/

#include "preferencesdialog.h"

#include <QLayout>
#include <QLabel>
#include <QTabWidget>
#include <QGroupBox>
#include <QLineEdit>
#include <QRadioButton>
#include <QComboBox>
#include <QTimeEdit>
#include <QPushButton>
#include <QMessageBox>

#include "attributes/frequencyinfo.h"
#include "attributes/preferenceshelper.h"
#include "attributes/datetimehelper.h"
#include "data/datamanager.h"
#include "data/preferencesbatch.h"
#include "data/connectioninfo.h"
#include "connectionmanager.h"
#include "iconloader.h"

PreferencesDialog::PreferencesDialog( int userId, QWidget* parent ) : CommandDialog( parent ),
    m_userId( userId ),
    m_preferences( NULL )
{
    const UserRow* user = dataManager->users()->find( userId );
    QString name = user ? user->name() : QString();

    setWindowTitle( tr( "User Preferences" ) );
    if ( userId == connectionManager->connectionInfo()->userId() )
        setPrompt( tr( "Configure your user preferences:" ) );
    else
        setPrompt( tr( "Configure preferences for user <b>%1</b>:" ).arg( name ) );
    setPromptPixmap( IconLoader::pixmap( "preferences", 22 ) );

    showInfo( tr( "Edit preferences" ) );

    QVBoxLayout* layout = new QVBoxLayout();

    QTabWidget* tabWidget = new QTabWidget( this );
    layout->addWidget( tabWidget );

    QWidget* generalPage = new QWidget( tabWidget );
    tabWidget->addTab( generalPage, IconLoader::icon( "configure" ), tr( "General" ) );

    QVBoxLayout* generalLayout = new QVBoxLayout( generalPage );

    QGroupBox* emailGroup = new QGroupBox( tr( "&Email Address" ), generalPage );
    generalLayout->addWidget( emailGroup );

    QVBoxLayout* emailLayout = new QVBoxLayout( emailGroup );

    m_emailEdit = new QLineEdit( emailGroup );
    emailLayout->addWidget( m_emailEdit );

    QLabel* emailLabel = new QLabel( tr( "This address will only be visible to the system administrator." ), emailGroup );
    emailLayout->addWidget( emailLabel );

    QGroupBox* timeZoneGroup = new QGroupBox( tr( "Time Zone" ), generalPage );
    generalLayout->addWidget( timeZoneGroup );

    QVBoxLayout* timeZoneLayout = new QVBoxLayout( timeZoneGroup );

    m_defaultZoneRadio = new QRadioButton( tr( "&Use the same time zone as the server" ), timeZoneGroup );
    timeZoneLayout->addWidget( m_defaultZoneRadio );

    QHBoxLayout* selectedZoneLayout = new QHBoxLayout();
    timeZoneLayout->addLayout( selectedZoneLayout );

    m_selectedZoneRadio = new QRadioButton( tr( "Use the &selected time zone:" ), timeZoneGroup );
    selectedZoneLayout->addWidget( m_selectedZoneRadio );

    m_zoneCombo = new QComboBox( timeZoneGroup );
    selectedZoneLayout->addWidget( m_zoneCombo );

    QDateTime current = QDateTime::currentDateTime();
    QDateTime currentUtc = current.toUTC();

    int currentZone = currentUtc.date().daysTo( current.date() ) * 86400 + currentUtc.time().secsTo( current.time() );

    static const double zones[] = {
        -11, -10, -9.5, -9, -8, -7, -6, -5, -4, -3.5, -3, -2, -1, 0, 1, 2, 3, 3.5, 4,
        5, 5.5, 5.75, 6, 6.5, 7, 8, 9, 9.5, 10, 10.5, 11, 11.5, 12, 12.75, 13, 14
    };

    QLocale currentLocale;

    for ( int i = 0; i < (int)( sizeof( zones ) / sizeof( double ) ); i++ ) {
        int zone = (int)( zones[ i ] * 3600.0 );
        QDateTime zoneTime = currentUtc.addSecs( zone );
        // NOTE: "dddd" returns the day of week for system locale, not for current one
        QString time = currentLocale.dayName( zoneTime.date().dayOfWeek() )
            + zoneTime.toString( ", yyyy-MM-dd, hh:mm " )
            + PreferencesHelper::formatTimeZone( zone );
        m_zoneCombo->addItem( time, QVariant( zone ) );
        if ( zone == currentZone )
            m_zoneCombo->setCurrentIndex( i );
    }

    selectedZoneLayout->addStretch( 1 );
    generalLayout->addStretch( 1 );

    m_defaultZoneRadio->setChecked( true );
    m_zoneCombo->setEnabled( false );

    connect( m_selectedZoneRadio, SIGNAL( toggled( bool ) ), m_zoneCombo, SLOT( setEnabled( bool ) ) );

    if ( connectionManager->connectionInfo()->checkFeature( "notify" ) ) {
        QWidget* notifyPage = new QWidget( tabWidget );
        tabWidget->addTab( notifyPage, IconLoader::icon( "mail-send" ), tr( "Notifications" ) );

        QVBoxLayout* notifyLayout = new QVBoxLayout( notifyPage );

        QGroupBox* frequencyGroup = new QGroupBox( tr( "Frequency" ), notifyPage );
        notifyLayout->addWidget( frequencyGroup );

        QVBoxLayout* frequencyLayout = new QVBoxLayout( frequencyGroup );

        m_noNotifyRadio = new QRadioButton( tr( "Disable &notifications" ), frequencyGroup );
        frequencyLayout->addWidget( m_noNotifyRadio );
        m_immediateNotifyRadio = new QRadioButton( tr( "Send notifications i&mmediately" ), frequencyGroup );
        frequencyLayout->addWidget( m_immediateNotifyRadio );
        m_dailyNotifyRadio = new QRadioButton( tr( "Send daily &summary at given time" ), frequencyGroup );
        frequencyLayout->addWidget( m_dailyNotifyRadio );
        m_weeklyNotifyRadio = new QRadioButton( tr( "Send &weekly summary at given day and time" ), frequencyGroup );
        frequencyLayout->addWidget( m_weeklyNotifyRadio );

        QGroupBox* dayTimeGroup = new QGroupBox( tr( "Day and Time" ), notifyPage );
        notifyLayout->addWidget( dayTimeGroup );

        QHBoxLayout* dayTimeLayout = new QHBoxLayout( dayTimeGroup );

        QLabel* dayLabel = new QLabel( tr( "&Day:" ), dayTimeGroup );
        dayTimeLayout->addWidget( dayLabel );

        m_dayCombo = new QComboBox( dayTimeGroup );
        dayTimeLayout->addWidget( m_dayCombo );

        Qt::DayOfWeek firstDayOfWeek = DateTimeHelper::firstDayOfWeek();
        for ( int i = 0; i < 7; i++ ) {
            int day = firstDayOfWeek + i;
            if ( day > Qt::Sunday )
                day -= 7;
            m_dayCombo->addItem( currentLocale.dayName( day ) );
            if ( day == Qt::Monday )
                m_dayCombo->setCurrentIndex( i );
        }

        dayLabel->setBuddy( m_dayCombo );

        dayTimeLayout->addSpacing( 20 );

        QLabel* timeLabel = new QLabel( tr( "&Time:" ), dayTimeGroup );
        dayTimeLayout->addWidget( timeLabel );

        m_timeEdit = new QTimeEdit( dayTimeGroup );
        dayTimeLayout->addWidget( m_timeEdit );

        m_timeEdit->setDisplayFormat( "hh:mm" );
        m_timeEdit->setTime( QTime( 8, 0 ) );

        timeLabel->setBuddy( m_timeEdit );

        dayTimeLayout->addStretch( 1 );
        notifyLayout->addStretch( 1 );

        m_noNotifyRadio->setChecked( true );
        m_dayCombo->setEnabled( false );
        m_timeEdit->setEnabled( false );

        connect( m_noNotifyRadio, SIGNAL( clicked() ), this, SLOT( frequencyChanged() ) );
        connect( m_immediateNotifyRadio, SIGNAL( clicked() ), this, SLOT( frequencyChanged() ) );
        connect( m_dailyNotifyRadio, SIGNAL( clicked() ), this, SLOT( frequencyChanged() ) );
        connect( m_weeklyNotifyRadio, SIGNAL( clicked() ), this, SLOT( frequencyChanged() ) );
    } else {
        m_noNotifyRadio = NULL;
        m_immediateNotifyRadio = NULL;
        m_dailyNotifyRadio = NULL;
        m_weeklyNotifyRadio = NULL;
        m_dayCombo = NULL;
        m_timeEdit = NULL;
    }

    setContentLayout( layout, true );

    if ( userId == connectionManager->connectionInfo()->userId() ) {
        m_preferences = dataManager->preferences();
        loadPreferences();
    } else {
        LoadPreferencesBatch* batch = new LoadPreferencesBatch( userId );
        executeBatch( batch, tr( "Loading preferences..." ) );
    }
}

PreferencesDialog::~PreferencesDialog()
{
    if ( m_userId != connectionManager->connectionInfo()->userId() )
        delete m_preferences;
}

void PreferencesDialog::accept()
{
    PreferencesData data;

    QString email = m_emailEdit->text();
    if ( !email.isEmpty() && !PreferencesHelper::isValidEmail( email ) ) {
        showWarning( tr( "Invalid email address" ) );
        return;
    }
    data.setValue( "email", email );

    QString timeZone;
    if ( m_selectedZoneRadio->isChecked() )
        timeZone = PreferencesHelper::formatTimeZone( m_zoneCombo->itemData( m_zoneCombo->currentIndex() ).toInt() );
    data.setValue( "timezone", timeZone );

    if ( m_noNotifyRadio != NULL ) {
        QString frequency;
        if ( m_immediateNotifyRadio->isChecked() ) {
            frequency = "IMMEDIATELY";
        } else if ( m_dailyNotifyRadio->isChecked() ) {
            frequency = QString( "DAILY time=\"%1\"" ).arg( m_timeEdit->text() );
        } else if ( m_weeklyNotifyRadio->isChecked() ) {
            frequency = QString( "WEEKLY day=%1 time=\"%2\"" ).arg( m_dayCombo->currentIndex() ).arg( m_timeEdit->text() );
        }
        if ( !frequency.isEmpty() && email.isEmpty() ) {
            if ( QMessageBox::warning( this, tr( "Warning" ),
                tr( "Notifications will not be sent until you enter a valid email address. Do you want to continue?" ),
                QMessageBox::Ok | QMessageBox::Cancel ) != QMessageBox::Ok )
                    return;
        }
        data.setValue( "notify-frequency", frequency );
    }

    PreferencesData changes = data.compareWith( *m_preferences );

    if ( changes.isEmpty() ) {
        QDialog::accept();
        return;
    }

    SavePreferencesBatch* batch = new SavePreferencesBatch( m_userId, changes );

    executeBatch( batch );
}

bool PreferencesDialog::batchSuccessful( AbstractBatch* batch )
{
    if ( m_preferences == NULL ) {
        LoadPreferencesBatch* loadBatch = (LoadPreferencesBatch*)batch;
        m_preferences = loadBatch->preferencesData();
        loadPreferences();
        showInfo( tr( "Edit preferences" ) );
        return false;
    }

    return true;
}

bool PreferencesDialog::batchFailed( AbstractBatch* /*batch*/ )
{
    if ( m_preferences == NULL )
        return false;
    return true;
}

void PreferencesDialog::loadPreferences()
{
    m_emailEdit->setText( m_preferences->value( "email" ) );

    QString timeZone = m_preferences->value( "timezone" );
    if ( !timeZone.isEmpty() ) {
        m_selectedZoneRadio->setChecked( true );
        int zone = PreferencesHelper::parseTimeZone( timeZone );
        int index = m_zoneCombo->findData( QVariant( zone ) );
        if ( index >= 0 )
            m_zoneCombo->setCurrentIndex( index );
    }

    if ( m_noNotifyRadio != NULL ) {
        QString frequency = m_preferences->value( "notify-frequency" );
        if ( !frequency.isEmpty() ) {
            FrequencyInfo info = FrequencyInfo::fromString( frequency );
            if ( info.isValid() ) {
                switch ( info.type() ) {
                    case FrequencyInfo::Immediately:
                        m_immediateNotifyRadio->setChecked( true );
                        break;
                    case FrequencyInfo::Daily:
                        m_dailyNotifyRadio->setChecked( true );
                        m_timeEdit->setTime( info.time() );
                        break;
                    case FrequencyInfo::Weekly:
                        m_weeklyNotifyRadio->setChecked( true );
                        m_dayCombo->setCurrentIndex( info.day() );
                        m_timeEdit->setTime( info.time() );
                        break;
                    default:
                        break;
                }
                frequencyChanged();
            }
        }
    }
}

void PreferencesDialog::frequencyChanged()
{
    m_dayCombo->setEnabled( m_weeklyNotifyRadio->isChecked() );
    m_timeEdit->setEnabled( m_dailyNotifyRadio->isChecked() || m_weeklyNotifyRadio->isChecked() );
}
