/**************************************************************************
* This file is part of the WebIssues program
* Copyright (C) 2006 Michał Męciński
* Copyright (C) 2007-2009 WebIssues Team
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
**************************************************************************/

#include "editlistbox.h"

#include <QListView>
#include <QStringListModel>
#include <QLineEdit>
#include <QPushButton>
#include <QLayout>

EditListBox::EditListBox( QWidget* parent ) : QWidget( parent )
{
    QHBoxLayout* mainLayout = new QHBoxLayout( this );
    mainLayout->setMargin( 0 );

    QVBoxLayout* listLayout = new QVBoxLayout();
    mainLayout->addLayout( listLayout );

    m_lineEdit = new QLineEdit( this );
    m_lineEdit->setEnabled( false );
    listLayout->addWidget( m_lineEdit );

    m_listView = new QListView( this );
    listLayout->addWidget( m_listView );

    m_model = new QStringListModel( this );
    m_listView->setModel( m_model );

    QVBoxLayout* buttonLayout = new QVBoxLayout();
    mainLayout->addLayout( buttonLayout );

    buttonLayout->addStretch( 1 );

    m_addButton = new QPushButton( tr( "&Add" ), this );
    buttonLayout->addWidget( m_addButton );

    m_removeButton = new QPushButton( tr( "&Remove" ), this );
    buttonLayout->addWidget( m_removeButton );

    buttonLayout->addSpacing( 10 );

    m_upButton = new QPushButton( tr( "Move &Up" ), this );
    buttonLayout->addWidget( m_upButton );

    m_downButton = new QPushButton( tr( "Move &Down" ), this );
    buttonLayout->addWidget( m_downButton );

    buttonLayout->addStretch( 1 );

    connect( m_lineEdit, SIGNAL( textChanged( const QString& ) ), this, SLOT( textChanged( const QString& ) ) );
    connect( m_listView->selectionModel(), SIGNAL( currentChanged( const QModelIndex&, const QModelIndex& ) ),
        this, SLOT( currentChanged( const QModelIndex& ) ) );

    connect( m_addButton, SIGNAL( clicked() ), this, SLOT( addItem() ) );
    connect( m_removeButton, SIGNAL( clicked() ), this, SLOT( removeItem() ) );
    connect( m_upButton, SIGNAL( clicked() ), this, SLOT( moveItemUp() ) );
    connect( m_downButton, SIGNAL( clicked() ), this, SLOT( moveItemDown() ) );

    currentChanged( QModelIndex() );
}

EditListBox::~EditListBox()
{
}

void EditListBox::setStringList( const QStringList& list )
{
    m_model->setStringList( list );
}

QStringList EditListBox::stringList() const
{
    return m_model->stringList();
}

void EditListBox::setMaxLength( int length )
{
    m_lineEdit->setMaxLength( length );
}

void EditListBox::textChanged( const QString& text )
{
    QModelIndex index = m_listView->selectionModel()->currentIndex();
    if ( index.isValid() )
        m_model->setData( index, text );
}

void EditListBox::currentChanged( const QModelIndex& index )
{
    if ( index.isValid() ) {
        m_lineEdit->setEnabled( true );
        m_lineEdit->setText( m_model->data( index, Qt::DisplayRole ).toString() );
    } else {
        m_lineEdit->setEnabled( false );
        m_lineEdit->clear();
    }

    m_removeButton->setEnabled( index.isValid() );
    m_upButton->setEnabled( index.isValid() && index.row() > 0 );
    m_downButton->setEnabled( index.isValid() && index.row() < m_model->rowCount() - 1 );
}

void EditListBox::addItem()
{
    QStringList list = m_model->stringList();

    QModelIndex index = m_listView->selectionModel()->currentIndex();
    int row;
    if ( index.isValid() )
        row = index.row() + 1;
    else
        row = list.count();

    list.insert( row, tr( "New Item" ) );

    m_model->setStringList( list );

    m_listView->selectionModel()->setCurrentIndex( m_model->index( row ), QItemSelectionModel::ClearAndSelect );

    m_lineEdit->setFocus();
}

void EditListBox::removeItem()
{
    QStringList list = m_model->stringList();

    QModelIndex index = m_listView->selectionModel()->currentIndex();
    if ( !index.isValid() )
        return;

    int row = index.row();

    list.removeAt( row );

    m_model->setStringList( list );

    if ( row >= list.count() )
        row = list.count() - 1;

    if ( row >= 0 )
        m_listView->selectionModel()->setCurrentIndex( m_model->index( row ), QItemSelectionModel::ClearAndSelect );
    else
        currentChanged( QModelIndex() );
}

void EditListBox::moveItemUp()
{
    QModelIndex index = m_listView->selectionModel()->currentIndex();
    if ( !index.isValid() )
        return;

    int row = index.row();
    if ( row < 0 )
        return;

    QModelIndex prevIndex = m_model->index( row - 1 );

    QVariant temp = m_model->data( index, Qt::DisplayRole );
    m_model->setData( index, m_model->data( prevIndex, Qt::DisplayRole ) );
    m_model->setData( prevIndex, temp );

    m_listView->selectionModel()->setCurrentIndex( prevIndex, QItemSelectionModel::ClearAndSelect );
}

void EditListBox::moveItemDown()
{
    QModelIndex index = m_listView->selectionModel()->currentIndex();
    if ( !index.isValid() )
        return;

    int row = index.row();
    if ( row >= m_model->rowCount() )
        return;

    QModelIndex nextIndex = m_model->index( row + 1 );

    QVariant temp = m_model->data( index, Qt::DisplayRole );
    m_model->setData( index, m_model->data( nextIndex, Qt::DisplayRole ) );
    m_model->setData( nextIndex, temp );

    m_listView->selectionModel()->setCurrentIndex( nextIndex, QItemSelectionModel::ClearAndSelect );
}

QSize EditListBox::sizeHint() const
{
    return QSize( 300, 150 );
}
