#!/usr/bin/python
# Copyright 2016 The ANGLE Project Authors. All rights reserved.
# Use of this source code is governed by a BSD-style license that can be
# found in the LICENSE file.
#
# gen_vk_format_table.py:
#  Code generation for vk format map

from datetime import date
import json
import math
import pprint
import os
import re
import sys

sys.path.append('..')
import angle_format

template_table_autogen_cpp = """// GENERATED FILE - DO NOT EDIT.
// Generated by {script_name} using data from {input_file_name}
//
// Copyright {copyright_year} The ANGLE Project Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
//
// {out_file_name}:
//   Queries for full Vulkan format information based on GL format.

#include "libANGLE/renderer/vulkan/formatutilsvk.h"

#include "image_util/copyimage.h"
#include "image_util/generatemip.h"
#include "image_util/loadimage.h"

using namespace angle;

namespace rx
{{

namespace vk
{{

void Format::initialize(VkPhysicalDevice physicalDevice, const angle::Format &angleFormat)
{{
    switch (angleFormat.id)
    {{
{format_case_data}
        default:
            UNREACHABLE();
            break;
    }}
}}

}}  // namespace vk

}}  // namespace rx
"""

empty_format_entry_template = """{space}case angle::Format::ID::{format_id}:
{space}    // This format is not implemented in Vulkan.
{space}    break;
"""

format_entry_template = """{space}case angle::Format::ID::{format_id}:
{space}{{
{space}    internalFormat = {internal_format};
{space}    textureFormatID = angle::Format::ID::{texture_format_id};
{space}    vkTextureFormat = {vk_texture_format};
{space}    bufferFormatID = angle::Format::ID::{buffer_format_id};
{space}    vkBufferFormat = {vk_buffer_format};
{space}    dataInitializerFunction = {initializer};
{space}    break;
{space}}}
"""

def gen_format_case(angle, internal_format, vk_map):

    if angle not in vk_map or angle == 'NONE':
        return empty_format_entry_template.format(
            space = '        ',
            format_id = angle)

    texture_format_id = angle
    buffer_format_id = angle

    vk_format_name = vk_map[angle]
    vk_buffer_format = vk_format_name
    vk_texture_format = vk_format_name

    if isinstance(vk_format_name, dict):
        info = vk_format_name
        vk_format_name = info["native"]

        if "buffer" in info:
            buffer_format_id = info["buffer"]
            vk_buffer_format = vk_map[buffer_format_id]
            assert(not isinstance(vk_buffer_format, dict))
        else:
            vk_buffer_format = vk_format_name

        if "texture" in info:
            texture_format_id = info["texture"]
            vk_texture_format = vk_map[texture_format_id]
            assert(not isinstance(vk_texture_format, dict))
        else:
            vk_texture_format = vk_format_name

    initializer = angle_format.get_internal_format_initializer(
        internal_format, texture_format_id)

    return format_entry_template.format(
        space = '        ',
        internal_format = internal_format,
        format_id = angle,
        texture_format_id = texture_format_id,
        buffer_format_id = buffer_format_id,
        vk_buffer_format = vk_buffer_format,
        vk_texture_format = vk_texture_format,
        initializer = initializer)

input_file_name = 'vk_format_map.json'
out_file_name = 'vk_format_table'

angle_to_gl = angle_format.load_inverse_table(os.path.join('..', 'angle_format_map.json'))
vk_map = angle_format.load_json(input_file_name)
vk_cases = [gen_format_case(angle, gl, vk_map) for angle, gl in sorted(angle_to_gl.iteritems())]

output_cpp = template_table_autogen_cpp.format(
    copyright_year = date.today().year,
    format_case_data = "\n".join(vk_cases),
    script_name = __file__,
    out_file_name = out_file_name,
    input_file_name = input_file_name)

with open(out_file_name + '_autogen.cpp', 'wt') as out_file:
    out_file.write(output_cpp)
    out_file.close()
