#
# WebminSLBackup.pm -- Webmin module for the Skolelinux Backup service
# By Morten Werner Olsen <werner@skolelinux.no>
#
# Copyright (c) 2003, Skolelinux. 
# See COPYING in the source distribution for license details.
#
# $Id: WebminSLBackup.pm,v 1.8 2004/03/26 18:25:02 werner-guest Exp $
#

# This is a subclass of CGI::Application
package WebminSLBackup;
use base 'CGI::Application';
use Expect;
use Net::Ping;
use POSIX qw(strftime);
use SLBackup;

$Expect::Exp_Internal = 0;
$Expect::Log_Stdout   = 0;

use strict;

my $conffile = "/etc/slbackup/slbackup.conf";
my $logfile = "/var/log/slbackup/webmin-slbackup.log";

# unfortunately webmin's web-lib.pl uses a lot of global functions so because 
# we use strict we have to predeclare them here.
use vars qw(%config %gconfig $module_name $module_config_directory $tb $cb 
  $scriptname $remote_user $base_remote_user $current_theme $root_directory 
  $module_root_directory %module_info %text);


# This function is the first one called when the script is run.  Ideally this 
# and cgiapp_postrun would be in a class which contained all the infrastructure
# that webmin modules using CGI::Application share in common.
#  The inheritance tree would look like this:
#
#                                                 ,---> SLBackup.pm
#  CGI::Application --> CGI::Application::Webmin -+---> Foo.pm
#                       (or some similiar name)   `---> Bar.pm
#                                                       (etc.)
sub cgiapp_init
{
  my ($self) = @_;

  # Import the webmin helper functions.
  do '/usr/share/webmin/web-lib.pl';

  # initalize webmin configuration.
  &init_config();

  # Convert all the webmin global variables into class parameters to make
  # it more OOP.
  map { $self->param($_, $config{$_}) } keys %config;
  map { $self->param($_, $config{$_}) } keys %gconfig;
  map { $self->param($_, $config{$_}) } keys %module_info;
  $self->param('module_name', $module_name);
  $self->param('module_config_directory', $module_config_directory);
  $self->param('tb', $tb);
  $self->param('cb', $cb);
  $self->param('scriptname', $scriptname);
  $self->param('remote_user', $remote_user);
  $self->param('base_remote_user', $base_remote_user);
  $self->param('current_theme', $current_theme);
  $self->param('root_directory', $root_directory);
  $self->param('module_root_directory', $module_root_directory);
}


#
# This function is called after each run mode (See CGI::Application docs.)
# Here we wrap the output of each template with the webmin header and footer
#
sub cgiapp_postrun
{
  # $output is a reference to the results of a run mode.
  my ($self, $output) = @_;

  # We want to send out the input unbuffered.
  local $| = 1;

  # We don't want CGI::Application to send the http headers.  The function
  # below will do that.
  $self->header_type('none');

  # print the webmin header.  
  header($self->param('module_name'), '');

  print $$output;

  # print the webmin footer
  #footer();
  &footer("/", $text{'index_return'});

  # If we don't erase the contents of $output, the body of the page will be
  # displayed twice
  $$output = '';
}

#
# Here we set up the SLBackup class.  Think of it as the constructor.
#
sub setup
{
  my ($self) = @_;

  # The initial run mode or if no run mode is specified.
  $self->start_mode('general');

  #  Our run modes and the functions they map to.
  $self->run_modes(
    'general' => 'general',
    'backup' => 'backup',
    'backup_configure_client' => 'backup_configure_client',
    'backup_add_client' => 'backup_add_client',
    'backup_delete_client' => 'backup_delete_client',
    'backup_configure_server' => 'backup_configure_server',
    'restore' => 'restore',
    'restore_choose' => 'restore_choose',
    'restore_choose_snapshot' => 'restore_choose_snapshot',
    'restore_execute' => 'restore_execute',
    'maint' => 'maint',
    'maint_delete_older' => 'maint_delete_older',
    'maint_delete_confirm' => 'maint_delete_confirm',
    'sshkeys' => 'sshkeys',
    'sshkeys_add' => 'sshkeys_add',
    'sshkeys_create' => 'sshkeys_create',
    'sshkeys_delete' => 'sshkeys_delete',
    # AUTOLOAD is called if something other than the above run modes is asked
    # for.  In this case we just fall back to general.
    AUTOLOAD => 'general',
  );

}

#
#  This run mode ...
#
sub general
{
  my ($self) = @_;
  my $last_session;
  my $session_finished = 0;
  my $sshkeys_ok = 0;

  # fetch configuration data
  my $config = slbackup_readconfig ($conffile);

  # find out details about the last session
  if (open (LOGFILE, "/var/log/slbackup/slbackup.log")) {
      my @finished_sessions = grep (/Finished slbackup/, <LOGFILE>);
      close (LOGFILE);

      for my $session (@finished_sessions) {
	  ($last_session) = split (/ - /, $session);
      }
      $session_finished = 1;
  }

  ## does the ssh keys work properly?
  my $sshkeys_server = 0;
  my $sshkeys_client = 0;

  # localhost test
  my $sshkeys_localhost = ssh_test_local();
  
  # server test
  my ($server_address, $server_destdir, $server_type, $server_user) =
      list_server_config();
  $sshkeys_server = ssh_test_client ($server_address, $server_type,
				    $server_user);
  if ($sshkeys_localhost) {
      my $execstr = "ssh -o PasswordAuthentication=no $server_user" . "@" .
	  "$server_address 'echo 1'";
      if (($server_type eq "local") or (`$execstr` eq "1")) {
	  $sshkeys_server = 1;
      }
  }

  # clients test
  my @clients = list_clients();
  
  if ($sshkeys_localhost and $sshkeys_server) {
      for my $key (keys %{$config->{client}}) {
	  my $client_address = $config->{client}->{$key}->{address};
	  my $client_type = $config->{client}->{$key}->{type};
	  my $client_user = $config->{client}->{$key}->{user};
	  
	  if (ssh_test_client ($client_address, $client_type, 
			       $client_user) gt 0) {
	      $sshkeys_client = 1;
	  } else {
	      $sshkeys_client = 0;
	      last;
	  }
      }
  }

  if ($sshkeys_client) {
      $sshkeys_ok = 1;
  }

  # find the clients in configuration
  @clients = list_clients_html();

  # See the HTML::Template man page to understand all this.
  my $template = $self->load_tmpl('general', die_on_bad_params => 0);

  # fill template with text-strings
  fill_in_template($template);

  # fill template with "local" variables
  $template->param(last_session => $last_session,
		   session_finished => $session_finished,
		   clients => \@clients, sshkeys_ok => $sshkeys_ok,);

  return $template->output;
}

#
# This run mode ...
#
sub backup
{
  my ($self) = @_;
  my $backup_enable = 1;
  my $backuptime_changed;
  my $client;
  my $client_deleted = 0;
  my $newtime = "";
  my $output = "";

  # Fetching query from webform
  my $q = $self->query();

  # fetch details from cron job
  open (CRONFILE, "/etc/cron.d/slbackup");
  my @cronline = grep (/slbackup-cron/, <CRONFILE>);
  close (CRONFILE);

  if (grep (/^\#/, $cronline[0])) {
      $cronline[0] =~ s/\#//g;
      $backup_enable = 0;
  }
  my ($min, $hour) = split(/ /, $cronline[0], 3);  
  

  # Check if a new time was submitted
  if ($q->param("backuptime-submit")) {
      my $enable = $q->param("backup_enable");
      $newtime = $q->param("newtime");
      # make new cron-line
      ($hour, $min) = split(/:/, $newtime);
      my $newcron = "# cron job for Skolelinux Backup ";
      $newcron .= "(every night at $hour:$min)\n";
      if ($enable =~ /[Nn]o/) {
	  $newcron .= "#";
	  $backup_enable = 0;
      } else {
	  $backup_enable = 1;
      }
      $newcron .= "$min $hour * * * root ";
      $newcron .= "if [ -x /usr/share/slbackup/slbackup-cron " .
	  "-a -r /etc/slbackup/slbackup.conf ]; then " .
	  "/usr/share/slbackup/slbackup-cron ; fi\n";

      # save this in cron job
      open(CRONFILE, ">/etc/cron.d/slbackup");
      print(CRONFILE "$newcron");
      close(CRONFILE);
      
      # set bool newtime_changed
      $backuptime_changed = 1;
  } else {
      if (length($min) le 1) {
	  $min = "0$min";
      }
      if (length($hour) le 1) {
	  $hour = "0$hour";
      }
      $newtime = "$hour:$min";

      # set bool newtime_changed
      $backuptime_changed = 0;
  }

  # Check if a client has been deleted
  if ($q->param("action") eq "delete_client") {
      # which client is to be deleted
      $client = $q->param("client");

      # fetch configuration
      my $config = slbackup_readconfig ($conffile);
      
      # delete client from config
      delete ($config->{client}->{$client});
      
      # write configuration
      slbackup_writeconfig($conffile, $config);      

      # set bool client_deleted
      $client_deleted = 1;
  } 

  # find the clients and servername in configuration
  my $config = slbackup_readconfig ($conffile);
  my $backupserver = $config->{server_address};
  my @clients = list_clients_html();

  # See the HTML::Template man page to understand all this.
  my $template = $self->load_tmpl('backup', die_on_bad_params => 0);

  # fill template with text-strings
  fill_in_template($template);

  $template->param(newtime => $newtime, 
		   backuptime_changed => $backuptime_changed,
		   backupserver => $backupserver,
		   backup_enable => $backup_enable,
		   client => $client,
		   clients => \@clients,
		   client_deleted => $client_deleted,
		   configure => $text{"configure"},);
  return $output . $template->output;
}

#
#  This run mode ...
#
sub backup_add_client
{
  my ($self) = @_;

  # Fetching query from webform
  my $q = $self->query();

  # find which client to add
  my $client = $q->param("clientname");

  # See the HTML::Template man page to understand all this.
  my $template = $self->load_tmpl('backup_add_client', \
				  die_on_bad_params => 0);

  # fill template with text-strings
  fill_in_template($template);

  $template->param(client => $client,);
  return $template->output;
}

#
#  This run mode ...
#
sub backup_configure_client
{
  my ($self) = @_;
  my $client;
  my $client_added = 0;
  my $client_added_success = 0;
  my $client_added_error = "";
  my $client_confd = 0;
  my $client_changed = 0;
  my $client_hostname;
  my $client_unreach = 0;
  my $client_unreach_err = "";
  my $client_type_extbool;
  my $client_user;
  my $client_keep;
  my $delete_dir;
  my $delete_dir_bool = 0;
  my $delete_dir_deleted = 0;
  my $output = "";

  # Fetching query from webform
  my $q = $self->query();

  # fetch old configuration
  my $config = slbackup_readconfig($conffile);

  # is the user coming from backup_configure_client?
  if ($client = $q->param("client_confd")) {
      # fetch old configuration
      my @oldfiles = list_files ($client);
      my $oldfiles_len = scalar (@oldfiles);
      my @newfiles;
      my $files_changed = 0;
      my $type_changed = 0;
      my $hostname_changed = 0;
      my $user_changed = 0;
      my $keep_changed = 0;

      ## update configuration
      # check if client type is updated
      if ($q->param("type") ne $config->{client}->{$client}->{type}) {
	  $config->{client}->{$client}->{type} = $q->param("type");
	  $type_changed = 1;
      }

      # check if client hostname is updated
      if ($q->param("hostname") ne $config->{client}->{$client}->{address}) {
	  $config->{client}->{$client}->{address} = $q->param("hostname");
	  $hostname_changed = 1;
      }

      # check if client username is updated
      if ($q->param("username") ne $config->{client}->{$client}->{user}) {
	  $config->{client}->{$client}->{user} = $q->param("username");
	  $user_changed = 1;
      }

      # check if days to keep is updated
      if ($q->param("keep") ne $config->{client}->{$client}->{keep}) {
	  if ($q->param("keep") =~ /\d?/) {
	      $config->{client}->{$client}->{keep} = $q->param("keep");
	      $keep_changed = 1;
	  }
      }

      # check for changed file
      for my $file (@oldfiles) {
	  my $newfile = $q->param($file);
	  push (@newfiles, $newfile);
	  if ($file ne $newfile) {
	      $files_changed = 1;
	  }
      }

      # check if new file(s) are added
      my $newfile0 = $q->param("newfile0");
      my $newfile1 = $q->param("newfile1");
      logger("newfile0: .$newfile0., newfile1: .$newfile1.");
      if ($newfile0 ne "") {
	  push (@newfiles, $newfile0);
	  $files_changed = 1;
      }
      if ($newfile1 ne "") {
	  push (@newfiles, $newfile1);
	  $files_changed = 1;
      }

      # save the new configuration if change has
      if ($type_changed or $hostname_changed or $user_changed or
	  $files_changed or $keep_changed) {
	  delete ($config->{client}->{$client}->{location});
	  if (scalar (@newfiles) eq 1) {
	      $config->{client}->{$client}->{location} = $newfiles[0];
	  } else {
	      @{$config->{client}->{$client}->{location}} = @newfiles;
	  }
	  slbackup_writeconfig ($conffile, $config);

	  # set bool client_changed
	  $client_changed = 1;
      }

      # set bool client_confd
      $client_confd = 1;
  } else {
      # find which client to configure
      $client = $q->param("client");
  }

  # check if some directories is to be deleted and delete it
  if ($delete_dir = $q->param("delete_dir")) {
      # delete file from configuration:
      # build a new list of files to backup
      my @newfiles;
      
      # check how many locations in config file
      if (ref ($config->{client}->{$client}->{location}) eq "") {
	  # check that location actually is something
	  if ($config->{client}->{$client}->{location} ne "") {
	      if ($config->{client}->{$client}->{location} ne $delete_dir) {
		  push (@newfiles, $config->{client}->{$client}->{location});
	      }	else {
		  $delete_dir_deleted = 1;
	      }   
	  }
      } elsif (ref ($config->{client}->{$client}->{location}) eq "ARRAY") {
	  for my $file (@{$config->{client}->{$client}->{location}}) {
	      if ($file ne $delete_dir) {
		  push( @newfiles, $file );
	      } else {
		  $delete_dir_deleted = 1;
	      }
	  }
      }

      # rewrite configuration
      # check if @newfiles is empty
      if (scalar (@newfiles) eq 0) {
	  delete ($config->{client}->{$client}->{location});
      } else {
	  @{$config->{client}->{$client}->{location}} = @newfiles;
      }
      slbackup_writeconfig ($conffile, $config);
      
      # set bool delete_dir_bool
      $delete_dir_bool = 1;
  }
  
  # is the client added?
  if ($q->param("addclient")) {
      # set bool client_added
      $client_added = 1;

      # fetch data from web form
      my $client = $q->param("client");
      my $client_type = $q->param("type");
      my $client_hostname = $q->param("hostname");
      my $client_username = $q->param("username");
      my $client_keep = $q->param("keep");
      my @files;
      for (my $i = 0; $i lt 5; $i++) {
	  my $file = $q->param("newfile" . $i);
	  if ($file ne "") {
	      push (@files, $file);
	  }
      }

      # check that user has given us enough information
      if ($client eq "") {
	  # client name has to ...
	  $client_added_error = $text{"added_error_name"};
      } elsif (($client_type ne "local") and ($client_type ne "extern")) {
	  # something really strange has happened
	  # (not post from original form or something)
	  $client_added_error = $text{"added_error_type"};	  
      } elsif (($client_type eq "extern") and ($client_hostname eq "")) {
	  # client hostname cannot be nothing
	  $client_added_error = $text{"added_error_type_name"};
      # check that client does not already exist
      } elsif (! $config->{client}->{$client}) {
	  # client does not exist -> adding
	  # building hash
	  my %newclient;
	  $newclient{"type"} = $client_type;
	  if ($client_hostname ne "") {
	      $newclient{"address"} = $client_hostname;
	  }
	  if ($client_username eq "") {
	      $newclient{"user"} = "root";
	  } else {
	      $newclient{"user"} = $client_username;
	  }
	  if ($client_keep =~ /\d?/) {
	      $newclient{"keep"} = $client_keep;
	  } else {
	      #FIXME test if this really is a number, not just set it to 185
	      $newclient{"keep"} = 185;
	  }
	      
	  if (scalar (@files) eq 1) {
	      $newclient{"location"} = $files[0];
	  } elsif (scalar (@files) gt 1) {
	      $newclient{"location"} = \@files;
	  }
	  
	  # store configuration
	  $config->{client}->{$client} = \%newclient;
	  slbackup_writeconfig($conffile, $config);
	  
	  # set client_added_success-bool
	  $client_added_success = 1;
      } else {
	  # client existed (?)
	  $client_added_error = $text{"added_error_client_existed"};
      }
  } else {
      # set bool client_added
      $client_added = 0;
  }

  # fetch data about client from configuration file
  # check the hostname-validity if client is extern
  if ($config->{client}->{$client}->{address}) {
      $client_hostname = $config->{client}->{$client}->{address};
  } else {
      $client_hostname = "";
  }
  if ($config->{client}->{$client}->{type} eq "extern") {
      $client_type_extbool = 1;

      # check that the hostname/ip-address is valid
      $client_unreach_err = hostname_validate($client_hostname);
      
      if ($client_unreach_err) {
	  $client_unreach = 1;
      }
  } else {
      $client_type_extbool = 0;
  }

  $client_user = $config->{client}->{$client}->{user};
  $client_keep = $config->{client}->{$client}->{keep};
  my @files = list_files_html ($client);

  # See the HTML::Template man page to understand all this.
  my $template = $self->load_tmpl('backup_configure_client', \
				  die_on_bad_params => 0);

  # fill template with text-strings
  fill_in_template($template);

  $template->param(client => $client, client_added => $client_added,
		   client_confd => $client_confd,
		   client_changed => $client_changed,
		   client_added => $client_added,
		   client_added_error => $client_added_error,
		   client_added_success => $client_added_success,
		   client_hostname => $client_hostname,
		   client_unreach => $client_unreach,
		   client_unreach_err => $client_unreach_err,
		   client_type_extbool => $client_type_extbool,
		   client_user => $client_user,
		   client_keep => $client_keep,
		   delete_dir => $delete_dir,
		   delete_dir_bool => $delete_dir_bool,
		   delete_dir_deleted => $delete_dir_deleted,
		   files => \@files,);

  return $output . $template->output;
}


#
#  This run mode ...
#
sub backup_delete_client
{
  my ($self) = @_;
  my $client;

  # Fetching query from webform
  my $q = $self->query();

  # find which client to delete public key on
  $client = $q->param("client");

  # See the HTML::Template man page to understand all this.
  my $template = $self->load_tmpl('backup_delete_client', \
				  die_on_bad_params => 0);

  # fill template with text-strings
  fill_in_template($template);

  $template->param(client => $client,);
  return $template->output;
}

#
#  This run mode ...
#
sub backup_configure_server
{
  my ($self) = @_;
  my $backupdir;
  my $serveraddr;
  my $servertype;
  my $serveruser;
  my $serverconf_changed = 0;
  my $localhost = 1;
  my $output = "";

  # Fetching query from webform
  my $q = $self->query();

  # fetch config from file
  my $config = slbackup_readconfig ($conffile);

  # Check if some information has changed
  if ($q->param("serverconf")) {
      # get data from form
      $backupdir = $q->param("backupdir");
      $config->{server_destdir} = $backupdir;

      $servertype = $q->param("servertype");
      $config->{server_type} = $servertype;

      $serveruser = $q->param("serveruser");
      if ($serveruser eq "") {
	  $serveruser = "root";
      }
      $config->{server_user} = $serveruser;

      $serveraddr = $q->param("serveraddr");
      $config->{server_address} = $serveraddr;

      # write to configuration file
      slbackup_writeconfig ($conffile, $config);

      $serverconf_changed = 1;
  } else { #fetch from config-file
      $backupdir = $config->{server_destdir};
      $servertype = $config->{server_type};
      $serveruser = $config->{server_user};
      $serveraddr = $config->{server_address};
  }
  
  # set value of servertype (info to html-template)
  if ($servertype eq "local") {
      $localhost = 1;
  } else {
      $localhost = 0;
  }

  # See the HTML::Template man page to understand all this.
  my $template = $self->load_tmpl('backup_configure_server', \
				  die_on_bad_params => 0);

  # fill template with text-strings
  fill_in_template($template);

  $template->param(backupdir => $backupdir,
		   localhost => $localhost,
		   serveraddr => $serveraddr,
		   serveruser => $serveruser,
		   serverconf_changed => $serverconf_changed,);
  return $output . $template->output;
}

#
#  This run mode ...
#
sub restore
{
  my ($self) = @_;

  # fetch names on all computers that is possible to restore from
  my @clients = list_clients_html();

  # See the HTML::Template man page to understand all this.
  my $template = $self->load_tmpl('restore', die_on_bad_params => 0);

  # fill template with text-strings
  fill_in_template($template);

  $template->param(clients => \@clients, error => 0);
  return $template->output;
}


#
#  This run mode ...
#
sub restore_choose
{
  my ($self) = @_;

  # Fetching query from webform
  my $q = $self->query();

  # find which client to restore to
  my $client = $q->param("client");

  # See the HTML::Template man page to understand all this.
  my $template = $self->load_tmpl('restore_choose', \
				  die_on_bad_params => 0);
  
  # fill template with text-strings
  fill_in_template($template);

  $template->param(client => $client, error => 0, error_msg => "",);
  return $template->output;
}

#
#  This run mode ...
#
sub restore_choose_snapshot
{
  my ($self) = @_;
  my $restoredir;
  my @restoredirs;

  # Fetching query from webform
  my $q = $self->query();

  # find which client to restore to and what type of restore (full or not)
  my $client = $q->param("client");
  my $type = $q->param("type");

  if ($type eq "full") {
      my @files = list_files($client);
      for (my $i = 0; $i < scalar (@files); $i++) {
	  push (@restoredirs, { id => "dir$i", location => $files[$i]});
      }
  } else {
      # not full restore -> directory has been passed with a web-form
      $restoredir = $q->param("dir0");
      @restoredirs = ( { id => 'dir0', location => $q->param("dir0") } );
      $type = "not_full";
  }

  # find the available snapshots from the client
  my @snapshots = list_snapshots ($client, $type, $restoredir);

  # if there are no snapshots and type not equals full, give warning
  if (scalar (@snapshots) eq 0 and $q->param("type") ne "full") {
      my $error_msg = $text{"error_no_snapshots_avail"};

      # See the HTML::Template man page to understand all this.
      my $template = $self->load_tmpl('restore_choose', \
				      die_on_bad_params => 0);

      # fill template with text-strings
      fill_in_template($template);

      $template->param(client => $client, error => 1,
		       error_msg => $error_msg);
      return $template->output;
  } elsif (scalar (@snapshots) eq 0 and $q->param("type") eq "full") {
      # if there are no snapshots and type equals full
      my $error_msg = $text{"error_no_snapshots_avail_client"};

      # fetch names on all computers that is possible to restore from
      my @clients = list_clients_html();

      # See the HTML::Template man page to understand all this.
      my $template = $self->load_tmpl('restore', die_on_bad_params => 0);

      # fill template with text-strings
      fill_in_template($template);

      $template->param(clients => \@clients, error => 1,
                       error_msg => $error_msg,);
      return $template->output;
  } else {
      # See the HTML::Template man page to understand all this.
      my $template = $self->load_tmpl('restore_choose_snapshot', \
				      die_on_bad_params => 0);

      # fill template with text-strings
      fill_in_template($template);

      $template->param(backuptype => $type, client => $client,
		       restoredirs => \@restoredirs,
		       snapshots => \@snapshots,);
      return $template->output;
  }
}

#
#  This run mode ...
#
sub restore_execute
{
  my ($self) = @_;

  # Fetching query from webform
  my $q = $self->query();

  # fetch config from configuration file
  my $config = slbackup_readconfig ($conffile);

  # find which client to restore to
  my $client = $q->param("client");

  # find what kind of backup (full / not full)
  my $backuptype = $q->param("backuptype");

  # find which snapshot to restore from
  my $temp = $q->param("snapshot");
  my ($timestamp, $filetype) = split (/;/, $q->param("snapshot"));
  my $snapshot = ssepoch_to_iso ($timestamp);

  # find which directory to restore to
  my $targetdir = $q->param("targetdir");

  # find which directories/files to restore
  my $i = 0;
  my $tempentry;
  my @restoredirs;
  while ($tempentry = $q->param("dir" . $i)) {
      push @restoredirs, { id => "dir" . $i, location => $tempentry };
      $i++;
  }

  # build the execute string
  my $execstr = "rdiff-backup --print-statistics --restore-as-of $timestamp ";

  my $overwrite = $q->param("overwrite");
  if ($overwrite eq "on") {
      $execstr .= "--force ";
  }

  if ($config->{server_type} eq "extern") {
      my $user = $config->{server_user};
      if ($user eq "") {
	  $user = "root";
      }
      my $address = $config->{server_address};
      $execstr .= "$user" . "@" . "$address" . "::"
  }
  my $destdir = $config->{server_destdir};
  $execstr .=  "$destdir" . "/" . "$client";
  my $restoredir = $restoredirs[0]->{location};

  # restore file to the computer webmin runs on (until later)...
  #if ($config->{client}->{$client}->{type} eq "extern") {
  #    my $user = $config->{client}->{$client}->{user};
  #    if ($user eq "") {
  #      $user = "root";
  #    }
  #    my $address = $config->{client}->{$client}->{address};
  #    $execstr .= "$user" . "@" . "$address" . "::";
  #}
  if ($backuptype eq "full") {
      $execstr .= "/ $targetdir";
  } else {
      $execstr .= "$restoredir $targetdir";
  }

  # execute restore
  my $output;
  my $target;
  my $retval;
  if ($targetdir eq "/") {
      # we wont let any user do this...
      $output = $text{"restore_not_allow"};
      $retval = 0;
  } else {
      $output = `$execstr 2>&1`;
      $retval = $?;
      if ($retval ne 0) {
	  $retval = 0;
      } else {
	  $retval = 1;
      }
  }

  # See the HTML::Template man page to understand all this.
  my $template = $self->load_tmpl('restore_execute', \
				  die_on_bad_params => 0);

  # fill template with text-strings
  fill_in_template($template);

  $template->param(client => $client, restoredirs => \@restoredirs,
		   restorelog => $output, snapshot => $snapshot,
		   success => $retval, targetdir => $targetdir,);
  return $template->output;
}

#
#  This run mode ...
#
sub maint
{
  my ($self) = @_;
  my $client;
  my $delete_confirm;
  my $execstr;
  my $output;
  my $retval;

  # Fetching query from webform
  my $q = $self->query();

  # check if this is the return from deleted snapshots
  if ($q->param("client")) {
      # set the bool delete_confirm
      $delete_confirm = 1;

      # fetch data about the client and snapshot to delete older than
      $client = $q->param("client");
      my $snapshot = $q->param("snapshot");

      # fetch data about backup server
      my $config = slbackup_readconfig ($conffile);
      my ($server_address, $server_destdir, $server_type, $server_user) = 
	  list_server_config();

      # delete data
      $execstr = "rdiff-backup --force --remove-older-than $snapshot ";
      if ($server_type eq "extern") {
	  $execstr .= "$server_user" . "@" . "$server_address" . "::";
      }
      if (grep (/\/$/, $server_destdir)) {
	  $execstr .= "$server_destdir";
      } else {
	  $execstr .= "$server_destdir" . "/";
      }
      $execstr .= "$client";

      # execute restore
      $output = `$execstr 2>&1`;
      $retval = $?;
      if ($retval ne 0) {
          $retval = 0;
      } else {
          $retval = 1;
      }
      
      $delete_confirm = 1;
  } else {
      # set the bool delete_confirm to 0 and the string client to ""
      $delete_confirm = 0;
      $client = "";
  }

  # find all available computers
  my @clients = list_clients();

  # See the HTML::Template man page to understand all this.
  my $template = $self->load_tmpl('maint', die_on_bad_params => 0);

  # fill template with text-strings
  fill_in_template($template);

  $template->param(client => $client, clients => \@clients,
		   delete_confirm => $delete_confirm,
		   delete_log => $output, success => $retval,);
  return $template->output;
}

#
#  This run mode ...
#
sub maint_delete_older
{
  my ($self) = @_;
  my $deleting_none = 0;

  # Fetching query from webform
  my $q = $self->query();

  # find which client to delete snapshots for
  my $client = $q->param("client");

  # find all available snapshots
  my @snapshots_all = list_snapshots($client, "full", "");
  my @snapshots;
  for (my $i = (scalar (@snapshots_all) - 2); $i ge 0; $i--) {
      push (@snapshots, $snapshots_all[$i]);
  }

  # check number of snapshots available
  # (only one, none is going to be deleted)
  if (scalar (@snapshots) le 0) {
      $deleting_none = 1;
  }

  # See the HTML::Template man page to understand all this.
  my $template = $self->load_tmpl('maint_delete_older', \
				  die_on_bad_params => 0);

  # fill template with text-strings
  fill_in_template($template);

  $template->param(client => $client, deleting_none => $deleting_none,
		   snapshots => \@snapshots,);
  return $template->output;
}

#
#  This run mode ...
#
sub maint_delete_confirm
{
  my ($self) = @_;

  # Fetching query from webform
  my $q = $self->query();

  # find which client to delete snapshots for
  my $client = $q->param("client");

  # find the latest snapshot to keep
  my ($snapshot, $stype) = split (/;/, $q->param("snapshot"));

  # find which snapshots will be deleted
  my @snapshots = list_snapshots($client, "full", "");
  my @snapshots_deleted;
  for my $snap (@snapshots) {
      #FIXME - something wrong here...
      ($snap, my $temp) = split (/;/, $snap->{id});
      if ($snap lt $snapshot) {
	  my $time = ssepoch_to_iso ($snap);
	  push (@snapshots_deleted, {date => $time});
      }
  }

  # See the HTML::Template man page to understand all this.
  my $template = $self->load_tmpl('maint_delete_confirm', \
				  die_on_bad_params => 0);

  # fill template with text-strings
  fill_in_template($template);

  $template->param(client => $client, snapshot => $snapshot,
		   snapshots_deleted => \@snapshots_deleted,);
  return $template->output;
}

#
#  This run mode ...
#
sub sshkeys
{
  my ($self) = @_;
  my $client;
  my $client_publickey_added = 0;
  my $client_publickey_deleted = 0;
  my $client_keys_recreated = 0;
  my $error_msg;
  my $retval;

  # Fetching query from webform
  my $q = $self->query();

  ## find information about the localhost
  my $config = slbackup_readconfig ($conffile);

  # check if some action was done and returned to this page
  # we would not want to delete sshkeys until slbackup is using it's own
  # SSH private/public-keys
  #if ($q->param("action") eq "delete") {
  #    # delete clients public-key
  #    $client = $q->param("client");
  #    $client_publickey_deleted = 1;
  #    my $type = $q->param("type");
  #    my $address;
  #    my $username;
  #    if ($type eq "server") {
#	  $address = $config->{server_address};
#	  $username = $config->{server_user};
#      } else {
#	  $address = $config->{client}->{$client}->{address};
#	  $username = $config->{client}->{$client}->{user};
#      }
#      $error_msg = ssh_session ("delsshpubkey", $address, $username, "");

  if ($q->param("action") eq "add") {
      # add public key to the client
      $client = $q->param("client");
      $client_publickey_added = 1;
      my $password = $q->param("password");
      my $address;
      my $username;
      if ($q->param("type") eq "backupserver") {
	  $address = $config->{server_address};
	  $username = $config->{server_user};
      } else {
	  $address = $config->{client}->{$client}->{address};
	  $username = $config->{client}->{$client}->{user};
      }
      $error_msg = ssh_session ("addsshpubkey", $address, $username,
				$password);
  } elsif ($q->param("action") eq "create") {
      # create private/public key pair on localhost
      my $cmd = 'ssh-keygen -t dsa -N "" -f ~/.ssh/id_dsa';
      my $output = `$cmd 2>&1`;
      $retval = $?;
      if ($retval ne 0) {
	  $retval = 0;
      } else {
	  $retval = 1;
      }

      if (! $retval) {
	  $error_msg = "<br>\n" . $output;
      }
  }

  # hostname
  my $localhost = `hostname`;
  for my $tmpclient (keys %{$config->{client}}) {
      if ($config->{client}->{$tmpclient}->{type} eq "local") {
	  $localhost = "$tmpclient";
      }
  }

  # status of the host
  my $homedir = $ENV{"HOME"} || $ENV{"LOGDIR"} || (getpwuid($<))[7];
  my @sshprivstat = stat ("$homedir/.ssh/id_dsa");
  my @sshpubstat = stat ("$homedir/.ssh/id_dsa.pub");
  my $localhost_status;
  my $localhost_choises;
  # only checking that the size is greater then zero
  if (($sshprivstat[7] gt 0) and ($sshpubstat[7] gt 0)) {
      $localhost_status = $text{"sshkeys_pair_avail"};
      # we don't want to be able to recreate ssh key pair after all
      # it can break other things depending on the ssh key pair located
      # in ~root/.ssh/id_dsa[.pub]
      $localhost_choises = "&nbsp;";
  } else {
      $localhost_status = $text{"sshkeys_pair_unavail"};
      $localhost_choises = "&nbsp;";
      $localhost_choises = 
	  "(<a href=\"?rm=sshkeys_create\">" . 
	  $text{create} . "</a>)";
  }

  ## find information about the backup server
  my ($server_address, $server_destdir, $server_type, $server_user) =
      list_server_config();

  my $server_status;  
  my $server_choises = "&nbsp";
  # check if server _is_ localhost
  if ($server_type eq "local") {
      $server_status = $text{"sshkeys_not_needed"};
  } elsif ($localhost_status eq $text{"sshkeys_pair_unavail"}) {
      $server_status = $text{"sshkeys_pub_unavail"} . " $localhost";
      $server_choises = "&nbsp;";
  } else { 
      # if not, test ssh connection with rdiff-backup -V
      my $execstr = "ssh $server_user" . "@" .
	  "$server_address rdiff-backup -V";
      if (`$execstr` eq `rdiff-backup -V`) {
	  $server_status = $text{"sshkeys_pub_work"};
	  $server_choises = "&nbsp;";
# we do not want to delete keys until slbackup uses it's own SSH keys
#	  $server_choises = 
#	      "(<a href=\"?rm=sshkeys_delete&type=server\">" .
#	      $text{"delete"} . "</a>)";
      } else {
	  $server_status = $text{"sshkeys_pub_notwork"};
	  $server_choises = "&nbsp;";
	  $server_choises = 
	      "(<a href=\"?rm=sshkeys_add&type=server\">" . 
	      $text{"add"} . "</a>)";
      }
  }
  
  ## find the computers available in the configuration
  # hostname, status, choises
  my @clients = list_clients();
  my @clients_var;

  for my $bclient (keys %{$config->{client}}) {
      my $client_status;
      my $client_choises;
      if ($config->{client}->{$bclient}->{type} eq "extern") {
	  # if keypair is not available on localhost, none of the clients
	  # public keys could work
	  if ($localhost_status eq $text{"sshkeys_pair_unavail"}) {
	      # set same status and choises on all clients
	      $client_status = $text{"sshkeys_pub_unavail"} . " $localhost";
	      $client_choises = "&nbsp;";
	  } else {
	      my $client_address = $config->{client}->{$bclient}->{address};
	      my $client_user = $config->{client}->{$bclient}->{user};
	      my $client_type = $config->{client}->{$bclient}->{type};

	      if (ssh_test_client ($client_address, $client_type, 
				   $client_user) eq 0) {
		  $client_status = $text{"sshkeys_pub_notwork"};
		  $client_choises = 
		      "(<a href=\"?rm=sshkeys_add&client=$bclient\">" . 
		          $text{"add"} . "</a>)";
# we do not want to delete keys until slbackup uses it's own keys
#		  $client_choises = 
#		      "(<a href=\"?rm=sshkeys_delete&client=$bclient\">" .
#			  $text{"delete"} . "</a>)";
	      } else {
		  $client_status = $text{"sshkeys_pub_work"};
	      }
	  }
	  push (@clients_var, { client => $bclient,
				client_status => $client_status,
				client_choises => $client_choises });
      }
  }

  # See the HTML::Template man page to understand all this.
  my $template = $self->load_tmpl('sshkeys', die_on_bad_params => 0);

  # fill template with text-strings
  fill_in_template($template);

  # check if something returned an error
  my $error = 0;
  if ($error_msg ne "") {
      $error = 1;
  }

  $template->param(localhost => $localhost, 
		   localhost_status => $localhost_status,
		   localhost_choises => $localhost_choises,
		   backupserver => $server_address,
		   backupserver_status => $server_status,
		   backupserver_choises => $server_choises,
		   clients => \@clients_var,
		   client => $client,
		   client_publickey_added => $client_publickey_added,
		   client_publickey_deleted => $client_publickey_deleted,
		   client_keys_recreated => $client_keys_recreated,
		   error => $error, error_msg => $error_msg,);
  return $template->output;
}


#
#  This run mode ...
#
sub sshkeys_add
{
  my ($self) = @_;

  # Fetching query from webform
  my $q = $self->query();


  # check if the computer to add a pubkey to is the server
  # and set address and user-variables
  my $type = $q->param("type");
  my $client;
  my $username;
  my $address;
  my $config = slbackup_readconfig();
  if ($type eq "server") {
      $client = $config->{server_address};
      $username = $config->{server_user};
      $address = $client;
      $type = "backupserver";
  } else {
      $client = $q->param("client");
      $username = $config->{client}->{$client}->{user};
      $address = $config->{client}->{$client}->{address};
      $type = "client";
  }

  # See the HTML::Template man page to understand all this.
  my $template = $self->load_tmpl('sshkeys_add', \
				  die_on_bad_params => 0);

  # fill template with text-strings
  fill_in_template($template);

  $template->param(client => $client, username => $username,
		   address => $address, type => $type,);
  return $template->output;
}


#
#  This run mode ...
#
sub sshkeys_create
{
  my ($self) = @_;

  # See the HTML::Template man page to understand all this.
  my $template = $self->load_tmpl('sshkeys_create', \
				  die_on_bad_params => 0);

  # fill template with text-strings
  fill_in_template($template);

  return $template->output;
}


#
#  This run mode ...
#
sub sshkeys_delete
{
  my ($self) = @_;

  # Fetching query from webform
  my $q = $self->query();

  # fetch configuration
  my $config = slbackup_readconfig ($conffile);

  # find which client to delete public key on
  my $client = $q->param("client");
  my $type = $q->param("type");

  # find server address if type eq server
  if ($type eq "server") {
      $client = $config->{server_address};
  }

  # See the HTML::Template man page to understand all this.
  my $template = $self->load_tmpl('sshkeys_delete', \
				  die_on_bad_params => 0);

  # fill template with text-strings
  fill_in_template($template);

  $template->param(client => $client, type => $type,);
  return $template->output;
}

#
#  This run mode displays the information for one alternative group.
#
#sub list
#{
#  my ($self) = @_;
#  my $alternative = $self->query->param('name');
#
#  my $file = $self->parse_alternative(\$alternative);
#  my $template = $self->load_tmpl('list', die_on_bad_params => 0);
#  $template->param(name => $file->{name}, links => $file->{links},
#    scriptname => $self->param('scriptname'), tb => $self->param('tb'),
#    cb => $self->param('cb'),);
#
#  return $template->output;
#}


# function to fill in a template
# (Thanks to Andreas Schuldei - webmin-ldap-skolelinux)
sub fill_in_template {
    my ($template) = @_;
    
    my @parameter_names = $template->param();
    
    for my $name (@parameter_names) {
        
        if( $text{$name} ){
            $template->param ($name => $text{$name} );
        } 
    }
    $template->param( JavaScriptStatus => generate_status() ) 
        if($template->query(name => "JavaScriptStatus")); 
    $template->param( cb               => $cb               )
        if($template->query(name => "cb" ));
    $template->param( tb               => $tb               )
        if($template->query(name => "tb" ));
    $template->param( CgiName          => "entermanyusers.cgi")
        if($template->query(name => "CgiName" ));
}


# function to fill in a template
# (Thanks to Andreas Schuldei - webmin-ldap-skolelinux)
sub generate_status {
    my $status =  "\"";
    $status .= $ENV{'ANONYMOUS_USER'} ? "Anonymous user" :$remote_user;
    $status .= $ENV{'SSL_USER'} ? " (SSL certified)" : 
        $ENV{'LOCAL_USER'} ? " (Local user)" : "";
    $status .= " logged into " .$text{'programname'};
    $status .= " " . get_webmin_version(); 
    $status .= " on " . get_system_hostname();
    $status .= "(" . $ENV{'HTTP_USER_AGENT'}; 
    $status .= ")\"";
    
    return $status;
}


# function that lists the clients in the configuration
# and applies some language data used in html-templates
# (do not use data from this functions to other things than html-templates)
sub list_clients_html {
    # fetch configuration
    my $config = slbackup_readconfig($conffile);
    
    my @clients;
    for my $client (keys %{$config->{client}}) {
	push (@clients, {client => $client,
			 configure => $text{"configure"},
			 delete_str => $text{"delete_str"},
			 choose_filesdir => $text{"choose_filesdir"},
			 restore_full => $text{"restore_full"}});
    }
    return @clients;
}


# function that lists the clients in the configuration
sub list_clients {
    # fetch configuration
    my $config = slbackup_readconfig($conffile);
    
    my @clients;
    for my $client (keys %{$config->{client}}) {
	push (@clients, {client => $client});
    }
    return @clients;
}


# function that lists all files/directories to back up on a clients
sub list_files {
    my ($client) = @_;;
    # fetch configuration
    my $config = slbackup_readconfig($conffile);

    # check if location is scalar or array, and put value(s) in @files
    my @files;
    if (ref ($config->{client}->{$client}->{location}) eq "") {
	# check that location actually is something
	if ($config->{client}->{$client}->{location} ne "") {
	    push (@files, $config->{client}->{$client}->{location});
	}
    } elsif (ref ($config->{client}->{$client}->{location}) eq "ARRAY") {
	foreach my $file (@{$config->{client}->{$client}->{location}}) {
	    push (@files, $file);
	}
    }
    return @files;
}


# function that lists all files/directories to back up on a clients
# and applies some language data used in html-templates
# (do not use data from this functions to other things than html-templates)
sub list_files_html {
    my ($client) = @_;
    # fetch configuration
    my $config = slbackup_readconfig($conffile);

    # check if location is scalar or array, and put value(s) in @files
    my @files;
    if (ref ($config->{client}->{$client}->{location}) eq "") {
	# check that location actually is something
	if ($config->{client}->{$client}->{location} ne "") {
	    push (@files, {file => $config->{client}->{$client}->{location},
			   remove => $text{"remove"},
			   client => $client});
	}
    } elsif (ref ($config->{client}->{$client}->{location}) eq "ARRAY") {
	foreach my $file (@{$config->{client}->{$client}->{location}}) {
	    push (@files, {file => $file,
			   remove => $text{"remove"},
			   client => $client});
	}
    }
    return @files;
}


#
sub list_snapshots {
    my ($client, $backup_type, $restoredir) = @_;

    # fetch client configuration
    my $config = slbackup_readconfig ($conffile);
    my $client_user = $config->{client}->{$client}->{user};
    my $server_address = $config->{server_address};
    my $server_destdir = $config->{server_destdir};
    my $server_type = $config->{server_type};
    my $server_user = $config->{server_user};

    # build execute string
    my $execstr = "rdiff-backup --parsable-output --list-increments ";

    if ($server_type eq "extern") {
	$execstr .= "$server_user" . "@" . "$server_address" . "::";
    }

    if ($backup_type eq "full" ) {
	$execstr .= "$server_destdir" . "/" . "$client";
    } else {
	$execstr .= "$server_destdir" . "/" . "$client" . "$restoredir";
    }

    my $output .= `$execstr 2>&1 | grep -v missing`;
    my $retval = $?;
    if ($retval ne 0) {
	$retval = 0;
    } else {
	$retval = 1;
    }

    my @snapshots;
    
    if ($retval eq 1 and ! ($output =~ /error/i)) {
	my @lines = split (/\n/, $output);
	my $increments = scalar (@lines);

	for (my $i = $increments - 1; $i >= 0; $i--) {
	    $lines[$i] =~ /^(\d*)\s(\w*)$/;
	    my ($time_se, $type) = ($1, $2);
	    my $id = "$time_se;$type";
	    my $time = ssepoch_to_iso($time_se);
	    push (@snapshots, { id => $id, time => $time });
	}

    } else { # error handling
	#FIXME, please :)
    }

    return @snapshots;
}


# return configuration data about server
sub list_server_config {
    # fetch config from configuration files
    my $config = slbackup_readconfig ($conffile);

    # get server configuration
    my $address = $config->{server_address};
    my $destdir = $config->{server_destdir};
    my $type = $config->{server_type};
    my $user = $config->{server_user};

    return ($address, $destdir, $type, $user);
}


# reformat "seconds since epoch to "DDMMYYYY HH:MM"-format
sub ssepoch_to_iso {
    my ($timestamp) = @_;

    return `date -d "1970-01-01 $timestamp sec UTC" --iso-8601=minutes`;
}


#
sub ssh_session {
    my ($action, $address, $username, $password) = @_;
    my $cmd;
    my $homedir = $ENV{"HOME"} || $ENV{"LOGDIR"} || (getpwuid($<))[7];

    # check action
    if ($action eq "addsshpubkey" ) {
	# fetch ssh public key from ~/.ssh/id_dsa.pub
	open (DSAPUB, "$homedir/.ssh/id_dsa.pub") ||
	    return "Could not open public key on localhost";
	my $sshpubkey = readline (\*DSAPUB);
	close (DSAPUB);
	
	# provide command to the SSH session
	$cmd = "install -d -m 700 ~$username/.ssh; " .
	       "echo -n '$sshpubkey' >> ~$username/.ssh/authorized_keys";
    } elsif ($action eq "delsshpubkey") {
	# fetch ssh public key from ~/.ssh/id_dsa.pub
	open (DSAPUB, "$homedir/.ssh/id_dsa.pub") ||
	    return "Could not open public key on localhost";
	my $sshpubkey = readline (\*DSAPUB);
	close (DSAPUB);

	# delete ssh public key from ~/.ssh/authorized_keys
	$cmd = "grep -v '$sshpubkey' ~$username/.ssh/authorized_keys > " .
	    "~$username/.ssh/authorized_keys";
    }

    # Check that host is up'n'running
    #FIXME

    # connect to host and execute command
    (my $connect = Expect->spawn("ssh -o StrictHostKeyChecking=no " .
				 "$username\@$address")) ||
	return "Problems with SSH connection";
    
    if ($action ne "delsshpubkey") {
	if ($connect->expect(10, "assword: ")) {
	    print $connect "$password\r\n";
	} else {
	    return "Host did not ask for password";
	}
    }
    
    # recognise prompt...
    #FIXME - need a better solution here... (regexp and which fetches
    # more possible prompts)
    if ($connect->expect(10, '# ') || $connect->expect(10, '$ ')) {
	print $connect "$cmd\r";
    } else {
	return "No prompt given by host";
    }
    
    if ($connect->expect(10, '# ') || $connect->expect(10, '$ ')) {
	print $connect "exit\r";
    } else {
	return "Prompt is not returned after executing command";
    }
    
    if (! ($connect->expect(10, "closed"))) {
	return "Connection was not properly closed";
    }
    
    $connect->hard_close();
    
    return "";
}


sub hostname_validate {
    my ($hostip, $trash) = @_;
    my $hostip_valid = 0;

    my $hosttest = '^([a-z]([-a-z\d]*[a-z\d])?[\.]?)+?$';

    my $iptest = '^(([01]?[0-9][0-9]?|2[0-4][0-9]|25[0-5])\.){3}([01]?[0-9][0-9]?|2[0-4][0-9]|25[0-5])$';

    # test if the hostname or ip-address is in a valid format
    if (! ($hostip =~ /$hosttest/i or $hostip =~ /$iptest/)) {
	return "<b>" . $text{"error_str"} . ":</b> " . 
	    $text{"hostname_not_valid"};
    }

    # test if the hostname or ip-address is reachable
    my $p = Net::Ping->new("icmp", 5);
    if ($p->ping($hostip)) {
	logger("hostname: $hostip, ping successfull\n");
	$p->close();
	return undef;
    } else {
	$p->close();
	logger("hostname: .$hostip., ping NOT successfull\n");
	return "<b>" . $text{'warn_str'} . ":</b> " . 
	    $text{'host_not_reachable'};
    }
}


# testing ssh local environment:
#  - $home/.ssh/id_dsa(.pub)
#  - is the files above needed (checking configurations)
#  
# return:
#  0: missing id_dsa and/or id_dsa.pub
#  1: missing id_dsa and/or id_dsa.pub but not needed
#  2: everything ok
sub ssh_test_local {
    my $homedir = $ENV{"HOME"} || $ENV{"LOGDIR"} || (getpwuid($<))[7];
    my @sshprivstat = stat ("$homedir/.ssh/id_dsa");
    my @sshpubstat = stat ("$homedir/.ssh/id_dsa.pub");

    # checking that the size is greater then zero and
    # that there is any need for them
    my $sshfiles_present = 0;
    my $sshfiles_needed = 0;
    my ($server_address, $server_destdir, $server_type, $server_user) =
	list_server_config();
    my $config = slbackup_readconfig ($conffile);
    
    if (($sshprivstat[7] gt 0) and ($sshpubstat[7] gt 0)) {
	$sshfiles_present = 1;
    }

    if ($server_type eq "extern") {
	if ($sshfiles_present) {
	    return 2;
	} else {
	    return 0;
	}
    } else {
	for my $key (keys %{$config->{client}}) {
	    my $client_type = $config->{client}->{$key}->{type};
	    if ($client_type eq "extern") {
		if ($sshfiles_present) {
		    return 2;
		} else {
		    return 0;
		}
	    }
	}
	if ($sshfiles_present) {
	    return 2;
	} else {
	    return 1;
	}
    }
    return 0;
}


# testing that ssh-connection to a client works properly:
#
# return:
#  0: does not work (could not log in)
#  1: rdiff-backup is on the wrong version
#  2: everything ok
sub ssh_test_client {
    my ($client_address, $client_type, $client_user) = @_;
    if ($client_type eq "local") {
	return 2;
    }

    my $execstr = "ssh -l $client_user -o PasswordAuthentication=no " . 
	"$client_address";

    if (`$execstr echo 1` eq `echo 1`) {
	if (`$execstr rdiff-backup -V` eq `rdiff-backup -V`) {
	    return 2;
	} else {
	    return 1;}
    }
    return 0;
}

sub logger {
    open (LOG, ">>$logfile");
    my ($comment) = @_;
    my $now = strftime "%b %d %H:%M:%S", localtime;
    printflush LOG ("$now - $comment\n");
    close (LOG);
}

1;

__END__
