/*
  libwftk - Worldforge Toolkit - a widget library
  Copyright (C) 2002 Malcolm Walker <malcolm@worldforge.org>
  Based on code copyright  (C) 1999-2002  Karsten Laux 

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.
  
  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.
  
  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the
  Free Software Foundation, Inc., 59 Temple Place - Suite 330,
  Boston, MA  02111-1307, SA.
*/

#include <cassert>

// for Fatal constructor
#include <iostream>
#include <sigc++/object_slot.h>
#include <sigc++/bind.h>

#include <wftk/application.h>

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <wftk/debug.h>

#include <wftk/poll.h>
#include <wftk/timer.h>
#include <wftk/sdlhandler.h>

// for destructor
#include <wftk/font.h>
#include <wftk/mixer.h>
#include <wftk/surface.h>

#include "mapinit.h"

/* this is a gimp header-imagefile
   width and height are defined as well as HEADER_PIXEL()
   and header_data
*/
#include "logo_data.h"

namespace wftk {

///static members

Application* Application::instance_ = NULL;

Fatal::Fatal(const std::string& reason) : reason_(reason)
{
  std::cerr << "libwftk: FATAL ERROR: " << reason << std::endl;
}

Application::Application(int& argc, char **&argv,
	const DebugMap& other_flags, Uint32 update) :
  next_source_(FIRST_SOURCE),
  event_pump_(update),
  exitcode_(0),
  running_(true)
{
  // need to check for --wftk-debug-startup before
  // declaring our debugging variable
#ifdef DEBUG
  for(int i = 1; i < argc; ++i) {
    if(strcmp(argv[i], "--") == 0)
      break;
    if(strcmp(argv[i], "--wftk-debug-startup") == 0) {
      Debug::addChannels(Debug::STARTUP);
      break;
    }
  }
#endif

  Debug out(Debug::STARTUP);

  /* Initialize SDL step by step*/
  out << "trying to initialize SDLvideo ... ";
  if ( SDL_Init(SDL_INIT_VIDEO | SDL_INIT_NOPARACHUTE) < 0 )
    throw SDLFatal("SDL_Init");
  out << " OK." << Debug::endl;

  // Don't set instance_ until after SDL is initialized, since
  // other parts of the lib use Application::instance() to check
  // if SDL has been initialized.

  assert(!instance_);
  instance_ = this;

  parseArgs(argc, argv, other_flags);

  out << "Enabling SDL Unicode support." << Debug::endl;
  SDL_EnableUNICODE(1); 

  SDL_EnableKeyRepeat(SDL_DEFAULT_REPEAT_DELAY, SDL_DEFAULT_REPEAT_INTERVAL);

  loadResources();
}

void
Application::parseArgs(int& argc, char**& argv, const DebugMap& other_flags)
{
  Debug out(Debug::STARTUP);

  out << "Before parsing, " << argc - 1 << " args:" << Debug::endl;
  for(int i = 1; i < argc; ++i)
    Debug::out << argv[i] << Debug::endl;

  // parse command line args first, so we can turn on debugging

  const PairInit<DebugMap::value_type> debug_flags[] = {
	{"--wftk-debug-user",		Debug::APP_MASK},
	{"--wftk-debug-lib",		Debug::LIB_MASK},
	{"--wftk-debug-all",		0xffffffff},
	{"--wftk-debug-invalidate",	Debug::INVALIDATE},
	{"--wftk-debug-generic",	Debug::GENERIC},
	{"--wftk-debug-drawing",	Debug::DRAWING},
	{"--wftk-debug-mainloop",	Debug::MAINLOOP},
	{"--wftk-debug-startup",	Debug::STARTUP},
	{"--wftk-debug-sound",		Debug::SOUND},
	{"--wftk-debug-events",		Debug::EVENTS},
	{"--wftk-debug-widget-create",	Debug::WIDGET_CREATE},
	{"--wftk-debug-text-widgets",	Debug::TEXT_WIDGETS},
	{"--wftk-debug-fonts",		Debug::FONTS},
	{"--wftk-debug-packing",	Debug::PACKING},
	{"--wftk-debug-opengl",		Debug::OPENGL},
	{"--wftk-debug-draw-timing",	Debug::DRAW_TIMING},
  };
  const unsigned num_debug_flags = sizeof(debug_flags)/sizeof(debug_flags[0]);
  DebugMap debug_map(debug_flags->itr(), debug_flags->itr(num_debug_flags));

  debug_map.insert(other_flags.begin(), other_flags.end());

  char **end_arg = argv + argc, **next_arg = argv + 1, **this_arg = next_arg;

  while(this_arg < end_arg) {

    // we modify argc, argv to remove the --wftk-foo args,
    // this_arg points to the current argument, next_arg
    // points to where in argv it will be copied to if
    // it's not a --wftk-foo argument

    if(strcmp(*this_arg, "--") == 0) { // no more options
      if(this_arg != next_arg) // we've removed some args, need to finish copying
        while(this_arg < end_arg)
          *next_arg++ = *this_arg++;
      break;
    }

    const char flag_prefix[] = "--wftk-debug-flag-";
    const unsigned flag_prefix_len = sizeof(flag_prefix) - 1;
    const char wftk_prefix[] = "--wftk";
    const unsigned wftk_prefix_len = sizeof(wftk_prefix) - 1;
    DebugMap::const_iterator I;

    if(strcmp(*this_arg, "--wftk-no-sound") == 0) {
      if(Mixer::isInit()) // shouldn't happen, but...
        delete Mixer::instance();
      new Mixer(false);
    }
    else if(strncmp(*this_arg, flag_prefix, flag_prefix_len) == 0) {
      // allow flags to be specified by number, particularly
      // useful for the 0-15 'user' flags
      int flag = atoi(*this_arg + flag_prefix_len);
      if(flag >= 0 && (unsigned) flag < sizeof(Debug::Mask) * 8) {
        Debug::addChannels(1 << flag);
        Debug::out << "Adding debugging on channel " << flag << Debug::endl;
      }
    }
    else if((I = debug_map.find(*this_arg)) != debug_map.end()) {
      Debug::addChannels(I->second);
      Debug::out << "Adding debugging for " << I->first << Debug::endl;
    }
    else if(strncmp(*this_arg, wftk_prefix, wftk_prefix_len) != 0) {
      if(next_arg != this_arg)
        *next_arg = *this_arg;
      ++next_arg;
    }

    ++this_arg;
  }

  // if we've removed arguments, we need to decrement argc
  argc -= this_arg - next_arg;

  out << "After parsing, " << argc - 1 << " args:" << Debug::endl;
  for(int i = 1; i < argc; ++i)
    out << argv[i] << Debug::endl;
}

void Application::loadResources()
{
  // code stolen from old Logo class

  /** The following code requires that the image be saved from GIMP as type C-source.  
   *
   * I used Gimp 1.2.1 and saved a 32x32 RGB image, with only the 'use Macros instead
   * of Struct' option set.
   * 
   * The resulting array of data is referred to by GIMP_IMAGE_PIXEL_DATA.
   */

  Debug out(Debug::STARTUP);

  out <<"creating logo.." << Debug::endl;
  Surface *surf = new Surface();
  surf->readFromHeader(GIMP_IMAGE_PIXEL_DATA, (unsigned int)GIMP_IMAGE_WIDTH,
	(unsigned int)GIMP_IMAGE_HEIGHT);
  Surface::Resource* res = new Surface::Resource(surf);
  Surface::registry.insert("wftk_logo", res);
  res->free(); // drop constructor ref count, insert() creates its own
  out <<"success."<< Debug::endl;
}

Application::~Application()
{
  Debug out(Debug::STARTUP);

  out << "In application destructor" << Debug::endl;

  destroyed.emit();

  out << "Doing video shutdown" << Debug::endl;

  Color::registry.unregisterAll();
  Surface::registry.unregisterAll();
  Font::registry.unregisterAll();

  // shut down video (to kill the window) before shutting down
  // audio, since audio shutdown takes _so_ long
  SDL_QuitSubSystem(SDL_INIT_VIDEO);

  out << "Doing audio shutdown" << Debug::endl;

  if(Mixer::isInit())
    delete Mixer::instance();
  
  out << "Quitting SDL" << Debug::endl;

  // clear instance_ before we quit SDL

  assert(instance_ == this);
  instance_ = 0;

  SDL_Quit();
}

void
Application::handleEvent(bool can_block)
{
  if(queue_.empty()) {
    switch(next_source_) {
      case POLLING:
        Poll::poll(can_block ? Timer::limitWait(event_pump_) : 0);
        break;
      case TIMERS:
        Timer::processAllTimers();
        break;
      case GUI:
        SDLHandler::queueEvents();
        break;
      case UPDATE:
        update();
        break;
      case DRAW:
        draw();
        break;
      case NUM_SOURCES:
        assert(false);
        break;
      // no default, so it'll warn us if we add a source to the enum
    }

    // Apparently, enums don't have operator++()
    next_source_ = (Source) (next_source_ + 1);
    if(next_source_ == NUM_SOURCES)
      next_source_ = FIRST_SOURCE;
  }
  else {
    // pop before calling, so reentrant calls (through keepAlive(),
    // waitFor(), etc.), won't see the same event
    Event* event = queue_.front();
    queue_.pop();
    (*event)();
    delete event;
  }
}

int
Application::exec()
{

  // application main loop
  //
  // * timer coordination of the three timers: 50ms, 100ms and 500ms
  // * limit cpu usage by limiting the revolutions to 1000ms/Application::idleTime_ per second
  // * handle event everytime
  // * handle mouse updates everytime, including screen update
  // * update and draw the screen at 20fps
  // * implement key repeat

  while(running_)
    handleEvent(true);

  return exitcode_;
}

bool
Application::keepAlive()
{
  if(!running_)
    return false;

  // bail out when we get back to the top of the mainloop,
  // but run through once if we're already there

  do {
    handleEvent(false);
  } while(running_ && (next_source_ != FIRST_SOURCE || !queue_.empty()));

  return running_;
}

void
Application::waitFor(bool& var, bool wait_val)
{
  while(running_ && var != wait_val)
    handleEvent(true);
}

void
Application::waitFor(SigC::Slot0<bool> func, bool wait_val)
{
  while(running_ && func() != wait_val)
    handleEvent(true);
}

void 
Application::quit(int exitcode)
{
  Debug::channel(Debug::MAINLOOP)
	<< "Application: was requested to quit." << Debug::endl;
  running_ = false;
  exitcode_ = exitcode;
}

SigC::Slot0<void>
Application::quitSlot(int exitcode)
{
  return SigC::bind(SigC::slot(*this, &Application::quit), exitcode);
}

void 
Application::abort()
{
  Debug::channel(Debug::MAINLOOP) << "Application: aborting." << Debug::endl;
  running_ = false;
  exitcode_ = -1;
}

} // namespace wftk

