/*
  libwftk - Worldforge Toolkit - a widget library
  Copyright (C) 2002 Malcolm Walker <malcolm@worldforge.org>
  Based on code copyright  (C) 1999-2002  Karsten Laux 

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.
  
  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.
  
  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the
  Free Software Foundation, Inc., 59 Temple Place - Suite 330,
  Boston, MA  02111-1307, SA.
*/

#include "mixer.h"

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "debug.h"
#include "music.h"
#include "sample.h"
#include "application.h"
#include "timer.h"

#ifdef HAVE_MIXER
#include <SDL/SDL_mixer.h>
#endif

// We complain to stderr if mixer can't be initted
#include <iostream>

#include <SDL/SDL.h>
#include <sigc++/object_slot.h>

namespace wftk {

Mixer* Mixer::instance_ = 0;

Mixer::Mixer(bool use_sound) :
  audio_open_(false),
  music_(0),
  volume_(Sound::VOLUME_MAX)
{
  assert(!use_sound || Application::instance()); // Need SDL_Init()

  assert(!instance_);
  instance_ = this;

  if(!use_sound)
    return;

  Debug out(Debug::SOUND);

#ifdef HAVE_MIXER
 
  out << "trying to initialize SDLaudio ... ";
  if ( SDL_InitSubSystem(SDL_INIT_AUDIO) < 0 )
    {
      out << " Couldn't initialize: " << SDL_GetError() << Debug::endl;
      std::cerr << "Audio init failed; will proceed without sound support." << std::endl;
      return;
    }
  out << " OK." << Debug::endl;

  int chunksize;

  audio_format_ = MIX_DEFAULT_FORMAT;
  audio_channels_ = MIX_DEFAULT_CHANNELS;

//undef this (HIFI) to reduce CPU by a 5%
#if 1
  audio_rate_ = 44100;
  chunksize = 2048;
#else
  audio_rate_ = MIX_DEFAULT_FREQUENCY;
  chunksize = 1024;
#endif

  /* Open the audio device */
  if (Mix_OpenAudio(audio_rate_, audio_format_, 
		    audio_channels_, chunksize) < 0) 
    {
      out << "Couldn't open audio: " << SDL_GetError() << Debug::endl;
      SDL_QuitSubSystem(SDL_INIT_AUDIO);
    } 
  else 
    {
      Mix_QuerySpec(&audio_rate_, &audio_format_, &audio_channels_);
      
      out << "Opened audio at "<< audio_rate_ << " Hz  ";
      out << (audio_format_&0xFF) << "bit  ";
      if(audio_channels_ > 1)
        out << " stereo"; 
      else
        out << " mono" ;
      out << Debug::endl;  
      
      audio_open_ = true;
    }
#endif // HAVE_MIXER

  if(!audio_open_)
    std::cerr << "Audio init failed; will proceed without sound." << std::endl;
}

Mixer::~Mixer()
{
  assert(instance_ == this);
  instance_ = 0;

  //this is important here, because unloading music when the mixer is already dead  
  //likely leads to segfaults

  Sample::registry.unregisterAll();
  Music::registry.unregisterAll();
  
#ifdef HAVE_MIXER
  if ( audio_open_ ) 
    {
      Debug out(Debug::SOUND);

      Time start;
      if(out)
        start = Time::now();

      out <<"Mixer shutting down audio...";
      Mix_CloseAudio();
      audio_open_ = false;
      out <<"[OK]"<< Debug::endl;

      SDL_QuitSubSystem(SDL_INIT_AUDIO);

      out << "Took " << (out ? Time::now() - start : 0) <<
	" milliseconds to shut down sound" << Debug::endl;
    }
#endif
}

void Mixer::setMusic(const std::string& data)
{
  if(!audio_open_)
    return;

  const Music* music = Music::registry.find(data);
  if(music)
    setMusic(*music);
}

void Mixer::setMusic(const Music& music)
{
  if(!audio_open_)
    return;

  bool playing = music_ && music_->playing();

  if(playing)
    music_->stop();

  music_ = &music;

  if(playing)
    music_->play(0);
}

void Mixer::startMusic()
{
  if(!audio_open_)
    return;

  if(music_)
    music_->play(0);
}

void Mixer::stopMusic()
{
  if(!audio_open_)
    return;

  if(music_ != 0)
    music_->stop();
}

void Mixer::playSample(const std::string& data)
{
  const Sample *res =Sample::registry.find(data);
  if(res)
    playSample(*res);
}

void Mixer::playSample(const Sample& sample)
{
  if(audio_open_)
    sample.play(); 
}

int 
Mixer::setVolume(int data)
{
  volume_ = data;

  if(!audio_open_)
    return 0;

#ifdef HAVE_MIXER
  //set volume for all channels
  return  Mix_Volume(-1, data);
#else
  // no mixer!
  return 0;
#endif
}

int 
Mixer::setMusicVolume(int data)
{
  if(!audio_open_)
    return 0;

#ifdef HAVE_MIXER
  //set volume for all channels
  return  Mix_VolumeMusic(data);
#else
  return 0;
#endif
}

void
Mixer::pauseMusic()
{
  if(!audio_open_)
    return ;
#ifdef HAVE_MIXER
  Mix_PauseMusic();
#endif
}

void
Mixer::resumeMusic()
{
  if(!audio_open_)
    return ;
#ifdef HAVE_MIXER
  Mix_ResumeMusic();
#endif
}
void
Mixer::rewindMusic()
{
  if(!audio_open_)
    return ;
#ifdef HAVE_MIXER
  Mix_RewindMusic();
#endif
}

void
Mixer::mixer_callback(void*, Uint8* data, int len)
{
  if(instance_)
    instance_->data.emit(data, len);
}

void 
Mixer::enableHook(bool enable)
{
  if(!audio_open_)
    return ;
#ifdef HAVE_MIXER
  Mix_SetPostMix(enable ? &Mixer::mixer_callback : 0, 0);
#endif
}

} // namespace wftk
