/*
  libwftk - Worldforge Toolkit - a widget library
  Copyright (C) 2002 Malcolm Walker <malcolm@worldforge.org>
  Based on code copyright  (C) 1999-2002  Karsten Laux

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.
  
  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.
  
  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the
  Free Software Foundation, Inc., 59 Temple Place - Suite 330,
  Boston, MA  02111-1307, SA.
*/

#include "mouse.h"

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "rootwindow.h"
#include "application.h"
#include "widget.h"
#include "pointer.h"
#include "dialog.h"

#include "debug.h"
#include <assert.h>

#include "mouse_data.h"

#include <SDL/SDL_mouse.h>
#include <sigc++/object_slot.h>

//we may only control one mouse
wftk::Mouse* wftk::Mouse::instance_ = NULL;

wftk::Mouse::Mouse() :
    visible_(true)
{
  assert(Application::instance()); // need SDL_Init()

  assert(!instance_);
  instance_ = this;

  Application::instance()->destroyed.connect(SigC::slot(*this,&Mouse::destroy));

  SDL_GetMouseState(&pos_.x, &pos_.y);

  // make sure "default" is registered
  Pointer::ResInval()("");

  ptr_ = Pointer::registry.get("default");
  assert(ptr_);
  ptr_->bind();
  ptr_->res()->show();

  update();
}


wftk::Mouse::~Mouse()
{
  ptr_->free();

  Pointer::registry.unregisterAll();

  assert(instance_ == this);
  instance_ = NULL;
}

void
wftk::Mouse::setPointer(const Surface& surf, const Point& hotspot)
{
  Surface::Resource *surf_res = new Surface::Resource(new Surface(surf));
  Pointer::Resource *res = new Pointer::Resource(new SoftPointer(surf_res, hotspot));
  surf_res->free();
  setPointer(res);
  res->free();
}

void
wftk::Mouse::setPointer(Pointer::Resource* res)
{
  if(!res) {
    res = Pointer::registry.get("default");
    // "default" is initialized in the Mouse constructor
    assert(res);
  }

  if(res == ptr_)
    return;

  if(visible_) {
    ptr_->res()->hide();
    res->res()->show();
  }

  ptr_->free();
  ptr_ = res;
  ptr_ ->bind();

  update();
}

bool
wftk::Mouse::handleEvent(const SDL_Event* event)
{
  Debug out(Debug::EVENTS);

  switch(event->type) {
    case SDL_MOUSEMOTION:
      {
        Point pos(event->motion.x, event->motion.y);
        Point rel(event->motion.xrel, event->motion.yrel);
        Point old_pos = pos_;
        Button mask = (Button) event->motion.state;
        pos_ = pos;
        update();
        if(mouseMove.emit(pos, rel, mask))
          return true;
        if(RootWindow::instance()) {
          ScreenArea* oldsa = RootWindow::instance()->getContainer(old_pos);
          ScreenArea* newsa = RootWindow::instance()->getContainer(pos);
          ScreenArea* sa = newsa;
          // handle mouse gain/loss
          while(oldsa && (!newsa || !oldsa->contains(*newsa))) {
            oldsa->lostMouse();
            oldsa = oldsa->parent();
          }
          // oldsa is now the common parent, or 0
          while(newsa != oldsa) {
            newsa->gainedMouse();
            newsa = newsa->parent();
          }
          // mouse motion event
          Widget* w = dynamic_cast<Widget*>(sa);
          if(w && !w->isEnabled())
            return false;
	  if(sa)
	    pos = sa->localCoord(wftk::Rect(pos.x, pos.y, 1, 1)).origin();
          while(sa) {
            if(sa->mouseEvent(pos, rel, mask))
              return true;
	    pos += sa->getRect().origin();
            sa = sa->parent();
          }
        }
        return false;
      }
    case SDL_MOUSEBUTTONDOWN:
    case SDL_MOUSEBUTTONUP:
      {
        Point pos(event->button.x, event->button.y);
        out << "Got a click at " << pos << Debug::endl;
        bool pressed = (event->button.state == SDL_PRESSED);
        Button button = (Button) SDL_BUTTON(event->button.button);
        if(mouseClick.emit(button, pressed, pos))
          return true;
        if(RootWindow::instance()) {
          ScreenArea* top = Dialog::getTop();
          if(!top)
            top = RootWindow::instance();
	  else
	    out << "Got a dialog" << Debug::endl;
          ScreenArea* sa = top->getContainer(pos - top->screenRect().origin());
          out << "Click lies inside " << (sa ? sa->name() : "nothing") << ' ' << sa << Debug::endl;
          Widget* w = dynamic_cast<Widget*>(sa);
          if(w && !w->isEnabled()) {
            out << "Got a disabled widget" << Debug::endl;
            return false;
          }
	  if(sa)
	    pos = sa->localCoord(wftk::Rect(pos.x, pos.y, 1, 1)).origin();
          while(sa) {
            out << "Emitting button event for " << sa << Debug::endl;
            if(sa->buttonEvent(button, pressed, pos))
              return true;
	    pos += sa->getRect().origin();
            sa = sa->parent();
          }
        }
        return false;
      }
    case SDL_ACTIVEEVENT:
      assert(event->active.state & SDL_APPMOUSEFOCUS);
      update();
      if(!event->active.gain) {
        if(RootWindow::instance()) {
          // This is the reason Mouse needs to keep track of pos_,
          // all other cases could be handled by SDL_GetMouseState()
          ScreenArea* sa = RootWindow::instance()->getContainer(pos_);
          while(sa) {
            sa->lostMouse();
            sa = sa->parent();
          }
        }
        hidePointer();
        return lostMouse.emit();
      }
      else {
        SDL_GetMouseState(&pos_.x, &pos_.y);
        if(RootWindow::instance()) {
          ScreenArea* sa = RootWindow::instance()->getContainer(pos_);
          while(sa) {
            sa->gainedMouse();
            sa = sa->parent();
          }
        }
        showPointer();
        return gotMouse.emit();
      }
    default:
      assert(false);
      return false;
  }
}

void
wftk::Mouse::update()
{
  // Deal with multiple root windows here

  if(RootWindow::instance())
    RootWindow::instance()->updateMouse();
}
