/*
  libwftk - Worldforge Toolkit - a widget library
  Copyright (C) 2002 Malcolm Walker <malcolm@worldforge.org>
  Based on code copyright  (C) 1999-2002  Karsten Laux

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.
  
  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.
  
  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the
  Free Software Foundation, Inc., 59 Temple Place - Suite 330,
  Boston, MA  02111-1307, SA.
*/

#ifndef _RECT_H
#define _RECT_H

#include <SDL/SDL.h>

#include <wftk/point.h>

namespace wftk {

/// Rectangle class.  Has corners, fill, etc.
class Rect : public SDL_Rect
{
 public:
  
  /// Create an empty rectangle at (0, 0)
  Rect() : valid_(true) {x = y = 0; w = h = 0;}
  /// Create a rectangle
  Rect(int x_, int y_, int w_, int h_);
  /// Copy a rectangle from an SDL_Rect
  explicit Rect( const SDL_Rect& r) : SDL_Rect(r), valid_(true) {}
  ///
//  Rect( const Point& upperLeft, const Point& lowerRight);
  ///
  ~Rect() {}

  /// An invalid rectangle
  static Rect invalid;
  /// An empty (w == h == 0) rectangle
  static Rect empty;
  /// Returns true if the rectangle is empty
  bool isEmpty() const {return w == 0 && h == 0;}
  /// Returns true if the rectangle is valid (dimensions non-negative)
  bool isValid() const {return valid_;}
  /// Set the validity of the rectangle.
  void setValid(bool val = true) {valid_ = val;}
  /// Returns true if the rectangle contains p
  bool contains( const Point &p ) const;
  /// Returns true if the rectangle contains r
  bool contains( const Rect &r )  const;

  ///move the rectangle by (dx, dy)
  void translate( int dx, int dy );
  ///move the rectangle by p
  void translate(const Point& p) {translate(p.x, p.y);}
  ///move the rectangle, specifying new upperleft corner
  void warp(const Point &newOrigin);
  ///this keeps the upper left corner and moves the lower right one
  void resize(int w_, int h_);
  /// Return the smallest rectangle which contains both *this and r
  Rect unite( const Rect &r )    const;
  /// Return the largest rectangle which is contained by both *this and r
  /**
   * Returns an invalid rectangle if *this and r don't intersect
   **/
  Rect intersect( const Rect &r) const;  

  /// Deprecated, use origin()
  Point upperLeft() const { return origin(); }

  /// Return the upper left corner of the rectangle
  Point origin() const {return Point(x, y);}
  /// Return the width of the rectangle
  Uint16 width() const {return w;}
  /// Return the height of the rectangle
  Uint16 height() const {return h;}

  /// Compare two rectangles for equality
  friend bool operator==(const Rect &p, const Rect &q)
    {return p.x == q.x && p.y == q.y && p.w == q.w && p.h == q.h;}
	     
  /// Compare two rectangles
  friend bool operator!=(const Rect &p, const Rect &q) {return !(p == q);}
  
  /// binary predicate, for std::map only
  friend bool operator<(const Rect &p, const Rect &q)
    {return p.x < q.x || (p.x == q.x &&
           (p.y < q.y || (p.y == q.y &&
           (p.w < q.w || (p.w == q.w && p.h < q.h)))));} 
 
  /// Print a rectangle to a stream, formatted as (x,y,w,h)
  friend std::ostream& operator<<(std::ostream& s, const Rect& p);       
 
 private:
  ///
  bool valid_;
};

} // namespace wftk

#endif // _RECT_H
