/*
  libwftk - Worldforge Toolkit - a widget library
  Copyright (C) 2002 Malcolm Walker <malcolm@worldforge.org>
  Based on code copyright  (C) 1999-2002  Karsten Laux

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.
  
  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.
  
  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the
  Free Software Foundation, Inc., 59 Temple Place - Suite 330,
  Boston, MA  02111-1307, SA.
*/

#ifndef _ROOTWINDOW_H
#define _ROOTWINDOW_H

#include <set>

#include <sigc++/object.h>
#if SIGC_MAJOR_VERSION == 1 && SIGC_MINOR_VERSION == 0
#include <sigc++/signal_system.h>
#else
#include <sigc++/signal.h>
#endif

#include <wftk/singlecontainer.h>

namespace wftk {

class ScreenSurface;
class SoftPointer;

/// singleton class, handles access to the entire drawing surface of the application
class RootWindow : public SingleContainer
{
 public:
  /// Constructor
  RootWindow(
			int resX,
			int resY,
			int bpp = 0,
			bool fullscreen = false,
			bool resizeable = true,
			const Surface& icon = Surface(),
			Uint32 extra_flags = 0,
			unsigned padding = 10);
  ///
  virtual ~RootWindow();
  /// return a pointer to the singleton RootWindow
  static RootWindow* instance() {return instance_;}

  /// application minimized / iconified
  SigC::Signal0<void> iconified;
  /// application restored (uniconified)
  SigC::Signal0<void> restored;
  /// application window has been resized
  SigC::Signal2<void,int,int> resized;

  
  /** Check if this video mode is possible.
      If SDL will emulate the requested pixeldepth, true_bpp (if != NULL)
      will contain the bitdepth of the actually used video mode.
      This is static so you can call it before the root window
      is created. Returns the value for bpp that will actually
      be used, or 0 on failure.
  */
  static int modeAvailable(int resX, int resY, int bpp, bool fullscreen);
  ///immediate display content update
  void sync();
  /// Set title, as displayed by window manager
  void setTitle(const std::string& title, const std::string& icon_name = "");
  /// This is static for when we have multiple root windows
  static bool handleEvent(const SDL_Event*);
  /// Return a surface representing the entire root window
  ScreenSurface* screen() {return screen_;}
  /// return fullscreen state
  bool fullscreen() const;
  /// true if application is minimized / iconified
  bool isIconified() { return iconified_; }

  /// resize the screen
  void resize(Uint16 w, Uint16 h);
  /// resize the screen to fit the widgets
  void resize();

  /// to be called by Mouse
  void updateMouse() {mouseBuffer_.changed();}
	/// for openGL only
	void convertSurface();
	void addSurface(Surface *);

 protected:
  ///
  virtual void drawAfter(Surface& target, const Point& offset, const Region& r);
  /// In RootWindow, instead of telling its (nonexistent) parent we need
  /// to update the packing information, we actually do the repacking.
  /// We don't resize the window, letting the user control that.
  virtual void packingUpdateParent() {if(!isHidden()) handleResize(width(), height());}
 private:

  void destroy() {delete this;} // signal handler

  ///
  std::set<Surface*> surfaces_;
  ///
  ScreenSurface* screen_;
  ///
  bool iconified_;
  ///
  bool updateMouse_; // WFTK_DISABLE_DEPRECATED
  /// rectangle to resize the root window to
  Rect resize_;
  ///
  int resize_wait_;

  /// background buffer for software mouse pointer
  class MouseBuffer {
   public:
    MouseBuffer() : ptr_(0), changed_(true) {}

    /// update when there's no other redraw
    void update(ScreenSurface& screen);
    /// update when there's other redraw, returns added screen region drawn to
    Region update(ScreenSurface& screen, const Region& dirty);

    /// call when mouse position or icon has changed
    void changed() {changed_ = true;}

   private:
    void checkBufferSize(const Surface&, const Pixelformat&);

    Surface buffer_;
    const SoftPointer* ptr_;
    Rect screen_loc_;
    bool changed_;

  } mouseBuffer_;

  ///
  static RootWindow* instance_;
};

} // namespace wftk

#endif // _ROOTWINDOW_H
