/*
  libwftk - Worldforge Toolkit - a widget library
  Copyright (C) 2002 Malcolm Walker <malcolm@worldforge.org>
  Based on code copyright  (C) 1999-2002  Karsten Laux

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.
  
  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.
  
  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the
  Free Software Foundation, Inc., 59 Temple Place - Suite 330,
  Boston, MA  02111-1307, SA.
*/
// written by Karsten Laux, June 1999  

#include "screensurface.h"
#include <iostream>
#include <cassert>

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "debug.h"
#include "region.h"
#include "application.h"
#include "sdlhandler.h" // for SDLFatal

#ifdef USE_SDL_OPENGL
  // shut up a redefinition warning
  #ifdef __WIN32__
  #undef NOMINMAX
  #endif
#include <SDL/SDL_opengl.h>
#endif

namespace wftk {

ScreenSurface::ScreenSurface(int w, int h, int bitPerPixel,
	bool fullscreen_, bool resizeable, Uint32 extra_flags)
{
  Debug out(Debug::GENERIC);

  Uint32 flags = SDL_ANYFORMAT | SDL_HWSURFACE | SDL_HWPALETTE | extra_flags ;

  if(fullscreen_)
    flags |= SDL_FULLSCREEN;
  else if(resizeable) // don't allow resizable fullscreen surfaces
    flags |= SDL_RESIZABLE;

#ifdef USE_SDL_OPENGL
	if (flags & SDL_OPENGL) {
       int rgb_size[4]; 

		switch (bitPerPixel) {
			case 8:
				rgb_size[0] = 2;
				rgb_size[1] = 3;
				rgb_size[2] = 3;
				rgb_size[3] = 4;
				break;
			case 15:
			case 16:
				rgb_size[0] = 5;
				rgb_size[1] = 5;
				rgb_size[2] = 5;
				rgb_size[3] = 4;
				break;
			default:
				rgb_size[0] = 8;
				rgb_size[1] = 8;
				rgb_size[2] = 8;
				break;
		}
		SDL_GL_SetAttribute( SDL_GL_RED_SIZE, rgb_size[0] );
		SDL_GL_SetAttribute( SDL_GL_GREEN_SIZE, rgb_size[1] );
		SDL_GL_SetAttribute( SDL_GL_BLUE_SIZE, rgb_size[2] );
//		SDL_GL_SetAttribute( SDL_GL_ALPHA_SIZE, rgb_size[3] );
//		SDL_GL_SetAttribute( SDL_GL_DEPTH_SIZE, bitPerPixel );
		SDL_GL_SetAttribute( SDL_GL_DOUBLEBUFFER, 1 );
		
	}
#endif

  /* switch to video mode */
  out << "switching to video mode..."<<Debug::endl<<Debug::flush;
  sdlSurface_ = SDL_SetVideoMode(w,h,bitPerPixel, flags);
  if (sdlSurface_ == 0)
    throw SDLFatal("SDL_SetVideoMode");
  
  //set static Pixelformat::member to spread information
  Pixelformat::displayFormat = Pixelformat(sdlSurface_);

  assert(Pixelformat::displayFormat.valid());

  /* Print out information about our surface */
  out << "\tScreen is in format "<< Pixelformat::displayFormat << Debug::endl;
  
  if (sdlSurface_->flags & SDL_HWSURFACE) 
    out << "  Screen is in video memory" << Debug::endl;
  else 
    out << "  Screen is in system memory" << Debug::endl;

  if (doublebuffered()) 
    out << "  Screen has double-buffering enabled" << Debug::endl;

  if (fullscreen()) 
    out << "  Running in Fullscreen Mode" << Debug::endl;
  else
    out << "  Running in Windowed Mode" << Debug::endl;

#ifdef USE_SDL_OPENGL
  if (flags & SDL_OPENGL) {
    glViewport( 0, 0, w, h );
    glMatrixMode( GL_PROJECTION );
    glLoadIdentity( );
	
    // clear the screen
    glClearColor( 0.0, 0.0, 0.0, 1.0 );
    glClear( GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

    glOrtho( 0, w, 0, h, -1.0, 1.0 );
 
    glMatrixMode( GL_MODELVIEW );
    glLoadIdentity( );
						 
    glEnable(GL_DEPTH_TEST);

    // Set the default color (transparent black!)
    glColor4i(0,0,0,0);
    // get the video surface the other way
    sdlSurface_ = SDL_GetVideoSurface();
  }
#endif
  flags_ = flags;
}

void 
ScreenSurface::update()
{
  SDL_UpdateRects(sdlSurface_,0,0);
}
  
void
ScreenSurface::update(const Rect& dest)
{
  SDL_UpdateRect(sdlSurface_, dest.x, dest.y, dest.w, dest.h); 
}

void
ScreenSurface::update(const Region& r)
{
  if(r.empty()) // could cause a problem with zero size array
    return;

#ifdef USE_SDL_OPENGL
  if (sdlSurface_->flags & SDL_OPENGL) {
 //   Debug::channel(Debug::OPENGL) << "ScreenSurface UPDATE: SDL_GL_SwapBuffers()\n";
//    SDL_GL_SwapBuffers();
    return;
  }
#endif

  SDL_Rect rects[r.nRects()];
  for(unsigned i = 0; i < r.nRects(); ++i)
    rects[i] = r.getRect(i);
  SDL_UpdateRects(sdlSurface_, r.nRects(), rects);
}

void
ScreenSurface::flip()
{
	if(sdlSurface_->flags & SDL_OPENGL) {
		Debug::channel(Debug::OPENGL) << "ScreenSurf FLIP\n";
		SDL_GL_SwapBuffers();
	}
	else {
    SDL_Flip(sdlSurface_);
	}
}

bool
ScreenSurface::setGamma(float r, float g, float b)
{
  if(!SDL_WasInit(SDL_INIT_VIDEO))
    return false;

  return SDL_SetGamma(r,g,b) == 0;
}

bool
ScreenSurface::setGammaRamp(Uint16 *red, Uint16 *green, Uint16 *blue)
{
  if(!SDL_WasInit(SDL_INIT_VIDEO))
    return false;

  return SDL_SetGammaRamp(red, green, blue) == 0;
}

void
ScreenSurface::resize(unsigned w, unsigned h)
{
  if(!resizeable() || (w == width() && h == height()))
    return;

  sdlSurface_ = SDL_SetVideoMode(w, h,
	sdlSurface_->format->BitsPerPixel, sdlSurface_->flags);

  Resized(w,h);
}

} // namespace

