/* button.cc
 * This file belongs to Worker, a filemanager for UNIX/X11.
 * Copyright (C) 2001 Ralf Hoffmann.
 * You can contact me at: ralf.hoffmann@epost.de
 *   or http://www.boomerangsworld.de/worker
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
/* $Id: button.cc,v 1.13 2002/01/29 00:52:24 ralf Exp $ */

#include "button.h"
#include "awindow.h"
#include "guielement.h"

const char *Button::type="Button";

Button::~Button()
{
  if(text[0]!=NULL) _freesafe(text[0]);
  if(text[1]!=NULL) _freesafe(text[1]);
  if(parent!=NULL) {
    parent->removeSubWin(win);
  }
}

Button::Button(AGUIX *taguix):GUIElement(taguix)
{
  this->dual=false;
  this->data=0;
  this->state=0;
  this->instate=0;
  this->text[0]=(char*)_allocsafe(1);
  this->text[1]=(char*)_allocsafe(1);
  this->text[0][0]=0;
  this->text[1][0]=0;
  this->fg[0]=0;
  this->fg[1]=0;
  this->bg[0]=0;
  this->bg[1]=0;
  lastbg=-1;
  showdual=true;
  font=NULL;
  active[0]=true;
  active[1]=true;
}

Button::Button(AGUIX *taguix,int tx,int ty,int width,
               const char *ttext,int tfg,int tbg,int tdata):GUIElement(taguix)
{
  this->x=tx;
  this->y=ty;
  this->w=width;
  this->h=taguix->getCharHeight()+4;
  this->dual=false;
  this->data=tdata;
  this->state=0;
  this->instate=0;
  this->text[0]=dupstring(ttext);
  this->text[1]=(char*)_allocsafe(1);
  this->text[1][0]=0;
  this->fg[0]=tfg;
  this->fg[1]=0;
  this->bg[0]=tbg;
  this->bg[1]=0;
  lastbg=-1;
  showdual=true;
  font=NULL;
  active[0]=true;
  active[1]=true;
}

Button::Button(AGUIX *taguix,int tx,int ty,
               const char *ttext,int tfg,int tbg,int tdata):GUIElement(taguix)
{
  this->x=tx;
  this->y=ty;
  this->w = ( strlen( ttext ) + 2 ) * taguix->getCharWidth();
  this->h=taguix->getCharHeight()+4;
  this->dual=false;
  this->data=tdata;
  this->state=0;
  this->instate=0;
  this->text[0]=dupstring(ttext);
  this->text[1]=(char*)_allocsafe(1);
  this->text[1][0]=0;
  this->fg[0]=tfg;
  this->fg[1]=0;
  this->bg[0]=tbg;
  this->bg[1]=0;
  lastbg=-1;
  showdual=true;
  font=NULL;
  active[0]=true;
  active[1]=true;
}

Button::Button(AGUIX *taguix,int tx,int ty,int width,int height,
               const char *ttext,int tfg,int tbg,int tdata):GUIElement(taguix)
{
  this->x=tx;
  this->y=ty;
  this->w=width;
  this->h=height;
  this->dual=false;
  this->data=tdata;
  this->state=0;
  this->instate=0;
  this->text[0]=dupstring(ttext);
  this->text[1]=(char*)_allocsafe(1);
  this->text[1][0]=0;
  this->fg[0]=tfg;
  this->fg[1]=0;
  this->bg[0]=tbg;
  this->bg[1]=0;
  lastbg=-1;
  showdual=true;
  font=NULL;
  active[0]=true;
  active[1]=true;
}

Button::Button(AGUIX *taguix,int tx,int ty,int width,
               const char *text_normal,const char *text_high,
               int fg_normal,int fg_high,
               int bg_normal,int bg_high,int tdata):GUIElement(taguix)
{
  this->x=tx;
  this->y=ty;
  this->w=width;
  this->h=taguix->getCharHeight()+4;
  this->dual=true;
  this->data=tdata;
  this->state=0;
  this->instate=0;
  this->text[0]=dupstring(text_normal);
  this->text[1]=dupstring(text_high);
  this->fg[0]=fg_normal;
  this->fg[1]=fg_high;
  this->bg[0]=bg_normal;
  this->bg[1]=bg_high;
  lastbg=-1;
  showdual=true;
  font=NULL;
  active[0]=true;
  active[1]=true;
}

const char *Button::getText(int index2)
{
  if((index2<0)||(index2>1)) return NULL;
  return dupstring(text[index2]);
}

void Button::setText(int index2,const char *new_text)
{
  if((index2<0)||(index2>1)) return;
  if(text[index2]!=NULL) _freesafe(text[index2]);
  text[index2]=dupstring(new_text);
  redraw();
}

bool Button::getDualState()
{
  return dual;
}

void Button::setDualState(bool new_dual)
{
  dual=new_dual;
  redraw();
}

void Button::setFG(int index2,int tfg)
{
  if((index2<0)||(index2>1)) return;
  if((tfg<(aguix->getMaxCols()))&&(tfg>=0)) {
    this->fg[index2]=tfg;
  }
  redraw();
}

int Button::getFG(int index2)
{
  if((index2<0)||(index2>1)) return 0;
  return fg[index2];
}

void Button::setBG(int index2,int tbg)
{
  if((index2<0)||(index2>1)) return;
  if((tbg<(aguix->getMaxCols()))&&(tbg>=0)) {
    this->bg[index2]=tbg;
  }
  redraw();
}

int Button::getBG(int index2)
{
  if((index2<0)||(index2>1)) return 0;
  return bg[index2];
}

int Button::getData()
{
  return data;
}

void Button::setData(int tdata)
{
  this->data=tdata;
}

void Button::redraw()
{
  int newbg;
  GC usegc;
  bool showe=false;
  int x1=0,y1=0,x2=0,y2=0,x3=0,y3=0;
  int dw,dw2;
  int dy;
  unsigned int maxdisplay;
  char *tstr;
  int ch,cw;

  if(parent==NULL) return;
  if(win==0) return;

  /* Clear the window */

  if((dual==false)||(state<2)) {
    newbg=bg[0];
  } else {
    newbg=bg[1];
  }
  aguix->SetWindowBG(win,newbg);
  lastbg=newbg;
  aguix->ClearWin(win);
  if(font==NULL) usegc=0; else usegc=font->getGC();

  /* calculate dimension of the turned-down corner */

  if ( ( dual == true ) && ( showdual == true ) && ( state < 2 ) ) {
    showe = true;

    dw=w/4;
    dw2=(int)((double)(h/3)*1.3);
    if( dw2 < dw ) dw = dw2;

    x1 = w-dw;
    y1 = 0;
    x2 = w-dw;
    y2 = dw-1;
    x3 = w-1;
    y3 = dw-1;
  }

  /* draw the text */

  if((dual==false)||(state<2)) {
    aguix->setFG(usegc,fg[0]);
  } else {
    aguix->setFG(usegc,fg[1]);
  }
  if(font==NULL) {
    ch=aguix->getCharHeight();
    cw=aguix->getCharWidth();
  } else {
    ch=font->getCharHeight();
    cw=font->getCharWidth();
  }

  dy=h/2-ch/2;
  maxdisplay=(w-2)/cw;
  if((dual==false)||(state<2)) {
    tstr=dupstring(text[0]);
  } else {
    tstr=dupstring(text[1]);
  }
  if(strlen(tstr)>maxdisplay) tstr[maxdisplay]=0;
  dw=strlen(tstr)*cw;
  int dx=w/2-dw/2;
  if(font==NULL) {
    aguix->DrawText(win,tstr,dx,dy);
  } else {
    aguix->DrawText(win,font,tstr,dx,dy);
  }
  _freesafe(tstr);

  /* now draw the button */

  if( showe == true ) {
    aguix->setFG( usegc, 2 );
    aguix->DrawTriangleFilled( win, usegc, x1, y1, x2, y2, x3, y3 );
    aguix->setFG( usegc, 0 );
    aguix->DrawTriangleFilled( win, usegc, x1, y1, w-1, 0, x3, y3 );
  }

  if( state == 0 )
    aguix->setFG( usegc, 2 );
  else
    aguix->setFG( usegc, 1 );

  aguix->DrawLine( win, usegc, 0, h-1, 0, 0 );

  if( showe == true ) {
    aguix->DrawLine( win, usegc, 0, 0, x1, y1 );
    aguix->setFG( usegc, 1 );
    aguix->DrawLine( win, usegc, x1, y1, x2, y2 );
  } else
    aguix->DrawLine( win, usegc, 0, 0, w-1, 0 );

  if( state == 0 )
    aguix->setFG( usegc, 1 );
  else
    aguix->setFG( usegc, 2 );

  aguix->DrawLine( win, usegc, 0, h-1, w-1, h-1 );

  if( showe == true ) {
    aguix->DrawLine( win, usegc, x3, y3, w-1, h-1 );
    aguix->setFG( usegc, 1 );
    aguix->DrawLine( win, usegc, x2, y2, x3, y3 );
    aguix->DrawLine( win, usegc, x1, y1, x3, y3 );
  } else
    aguix->DrawLine(win,usegc,w-1,h-1,w-1,1);

  aguix->Flush();
}

void Button::flush()
{
}

void Button::setState(int tstate)
{
  if((dual==false)&&(tstate==2)) {
    this->state=1;
  } else {
    this->state=tstate;
  }
  redraw();
}

int Button::getState()
{
  return state;
}

bool Button::isInside(int px,int py)
{
  if((px>0)&&(px<=w)) {
    if((py>0)&&(py<=h)) return true;
  }
  return false;
}

bool Button::handleMessage(XEvent *E,Message *msg)
{
  bool returnvalue;
  returnvalue=false;
  if(msg->window!=win) return false;
  if((msg->type==ButtonPress)||(msg->type==ButtonRelease)) {
    if((msg->button==Button3)||(msg->button==Button1)) {
      int mx,my;
      mx=msg->mousex;
      my=msg->mousey;
      if(msg->type==ButtonPress) {
        if(msg->window==win) {
	  if( ( ( msg->button == Button1 ) && ( active[0] == true ) ) ||
	      ( ( msg->button == Button3 ) && ( active[1] == true ) ) ) {
	    AGMessage *agmsg=(AGMessage*)_allocsafe(sizeof(AGMessage));
	    agmsg->type=AG_BUTTONPRESSED;
	    agmsg->button.button=this;
	    if((msg->button==Button3)&&(dual==true)) {
	      setState(2);
	      instate=2;
	      agmsg->button.state=2;
	    } else {
	      setState(1);
	      instate=1;
	      agmsg->button.state=1;
	    }
	    aguix->putAGMsg(agmsg);
	    returnvalue=true;
	  }
        }
      } else {
        if((state!=0)&&(instate!=0)) {
          AGMessage *agmsg=(AGMessage*)_allocsafe(sizeof(AGMessage));
          agmsg->type=AG_BUTTONRELEASED;
          agmsg->button.button=this;
          agmsg->button.state=instate;
          aguix->putAGMsg(agmsg);
          agmsg=(AGMessage*)_allocsafe(sizeof(AGMessage));
          agmsg->type=AG_BUTTONCLICKED;
          agmsg->button.button=this;
          agmsg->button.state=instate;
          aguix->putAGMsg(agmsg);
        }
        if(instate!=0) {
          setState(0);
          instate=0;
          returnvalue=true;
        }
      }
    }
  } else if(msg->type==EnterNotify) {
    // alles hier und alles mit instate wird benutzt, damit Button sich anpat, wenn
    // Mauszeiger im Button oder auerhalb des Buttons ist
    if(instate!=0) {
      if(state!=instate) {
        setState(instate);
      }
    }
  } else if(msg->type==LeaveNotify) {
    // alles hier und alles mit instate wird benutzt, damit Button sich anpat, wenn
    // Mauszeiger im Button oder auerhalb des Buttons ist
    if(instate!=0) {
      setState(0);
    }
  } else if(msg->type==Expose) {
    redraw();
  }
  if(returnvalue==true) {
    // jetzt noch die Message mit den Werten fllen
    msg->gadget=this;
    msg->gadgettype=BUTTON_GADGET;
  }
//  return returnvalue;
  return false; // see cyclebutton.cc
}

bool Button::getShowDual()
{
  return showdual;
}

void Button::setShowDual(bool new_val)
{
  if(showdual!=new_val) {
    showdual=new_val;
    redraw();
  } else showdual=new_val;
}

int Button::setFont(char *fontname)
{
  font=aguix->getFont(fontname);
  if(font==NULL) return -1;
  return 0;
}

const char *Button::getType()
{
  return type;
}

bool Button::isType(const char *qtype)
{
  if(strcmp(type,qtype)==0) return true;
  return false;
}

void Button::deactivate()
{
  deactivate(0);
  deactivate(1);
}

void Button::activate()
{
  activate(0);
  activate(1);
}

void Button::deactivate(int mode)
{
  if( ( mode == 0 ) || ( mode == 1 ) ) {
    active[mode]=false;
  }
}

void Button::activate(int mode)
{
  if( ( mode == 0 ) || ( mode == 1 ) ) {
    active[mode]=true;
  }
}
