/* listview.cc
 * This file belongs to Worker, a filemanager for UNIX/X11.
 * Copyright (C) 2001 Ralf Hoffmann.
 * You can contact me at: ralf.hoffmann@epost.de
 *   or http://www.boomerangsworld.de/worker
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
/* $Id: listview.cc,v 1.25 2002/03/23 01:59:02 ralf Exp $ */

#include "listview.h"
#include "awindow.h"
#include "guielement.h"

const char *ListView::type="ListView";

ListView::~ListView()
{
  if(parent!=NULL) {
    for(int i=0;i<maxdisplayv;i++) {
      parent->removeSubWin(subwins[i]);
    }
    parent->removeSubWin(hbarwin);
    parent->removeSubWin(vbarwin);    
    parent->removeSubWin(win);
  }
  if(subwins!=NULL) delete subwins;
  if(vislvcs!=NULL) delete vislvcs;
  int id=lvcs->initEnum();
  ListViewContent *tlvc=(ListViewContent*)lvcs->getFirstElement(id);
  while(tlvc!=NULL) {
    delete tlvc;
    tlvc=(ListViewContent*)lvcs->getNextElement(id);
  }
  lvcs->closeEnum(id);
  delete lvcs;
}

ListView::ListView(AGUIX *taguix):GUIElement(taguix)
{
  data=0;
  w=100;
  h=100;
  xoffset=0;
  yoffset=0;
  bgset=false;
  hbar=0;
  vbar=0;
  vbar_width=10;
  hbar_height=10;
  hbar_dir=0;
  vbar_dir=0;
  vbar_pressed=false;
  hbar_pressed=false;
  maxlen=0;
  subwins=NULL;
  winbgs=NULL;
  vislvcs=NULL;
  lvcs=new ArrayList();
  selecthandler=NULL;
  hbarwin=0;
  vbarwin=0;
  activelvc=NULL;
  font=NULL;
  quickAdd=false;
  gettimeofday(&lastwheel,NULL);
  mbg=0;
}

ListView::ListView(AGUIX *taguix,int tx,int ty,int width,int height,int tdata):GUIElement(taguix)
{
  this->data=tdata;
  this->x=tx;
  this->y=ty;
  this->w=width;
  this->h=height;
  xoffset=0;
  yoffset=0;
  bgset=false;
  hbar=1;
  vbar=1;
  vbar_width=10;
  hbar_height=10;
  hbar_dir=0;
  vbar_dir=0;
  vbar_pressed=false;
  hbar_pressed=false;
  maxlen=0;
  subwins=NULL;
  winbgs=NULL;
  vislvcs=NULL;
  lvcs=new ArrayList();
  hbarwin=0;
  vbarwin=0;
  selecthandler=NULL;
  activelvc=NULL;
  font=NULL;
  quickAdd=false;
  gettimeofday(&lastwheel,NULL);
  mbg=0;
}

void ListView::resize(int tw,int th)
{
  if((tw<1)||(th<1)) return;
  this->w=tw;
  this->h=th;
  if(parent!=NULL) {
    parent->resizeSubWin(win,tw,th);
    int oldmaxdv;
    oldmaxdv=maxdisplayv;
    recalcMAXValues();
    rebuildSubWins(oldmaxdv);
    reconfSubWins();
    setupBars();
    mredraw();
  }
}

int ListView::getData()
{
  return data;
}

void ListView::setData(int tdata)
{
  this->data=tdata;
}

void ListView::redraw()
{
  mredraw();
  hbarredraw();
  vbarredraw();
  redrawalllvc();
}

void ListView::flush()
{
}

void ListView::handleExpose(Window msgwin)
{
  ListViewContent *tlvc;
  if(msgwin==win) mredraw();
  else if(msgwin==hbarwin) hbarredraw();
  else if(msgwin==vbarwin) vbarredraw();
  else {
    if(quickAdd==false) {
      int elements=getElements();
      for(int i=0;i<maxdisplayv;i++) {
        if((i+yoffset)>=elements) break;
        if(subwins[i]==msgwin) {
          tlvc=vislvcs[i];
  	  if(tlvc!=NULL)
            winbgs[i]=tlvc->redraw(aguix,msgwin,font,winbgs[i],maxdisplayh);
          break;
	}
      }
    }
  }
}

bool ListView::handleMessage(XEvent *E,Message *msg)
{
//TODO
  bool returnvalue;
  returnvalue=false;
  struct timeval t2;
  int dt,s,us,scrollspeed;
  if(msg->type==Expose) {
    handleExpose(msg->window);
  } else if(msg->type==ButtonPress) {
    // jetzt men wir was machen
    bool isc=false;
    if(msg->window==hbarwin) {
      handleHBar(msg);
      isc=true;
    } else if(msg->window==vbarwin) {
      handleVBar(msg);
      isc=true;
    } else if(msg->window==win) {
      isc=true;
    } else {
      bool fulls;
      if(msg->button==Button3) fulls=true;
      else fulls=false;
      int elements=getElements();
      bool handle=true,found=false;
      for(int i=0;i<maxdisplayv;i++) {
        if(((i+yoffset)>=elements)&&(fulls==false)) handle=false;
        if(subwins[i]==msg->window) {
	  found=true;
	  break;
	}
      }
      if((handle==true)&&(found==true)) {
        // handleSelect, handleThird
        if(msg->button==Button3) handleThird(msg);
	else if((msg->button==Button4)||(msg->button==Button5)) {
	  gettimeofday(&t2,NULL);
	  s=abs(t2.tv_sec-lastwheel.tv_sec);
	  us=t2.tv_usec-lastwheel.tv_usec;
	  if(us<0) {
	    us+=1000000;
	    s--;
	  }
	  dt=us+s*1000000;
	  dt/=100000;
	  if(dt<2) scrollspeed=5;
	  else if(dt<4) scrollspeed=2;
	  else scrollspeed=1;
	  if(msg->button==Button4) scrollV1(-scrollspeed);
	  else if(msg->button==Button5) scrollV1(scrollspeed);
	  lastwheel=t2;
        } else if( msg->button == Button2 )
	  handleMiddle(msg);
	else
	  handleSelect(msg);
      }
      if(found==true) isc=true;
    }
    if(isc==true) {
      AGMessage *agmsg=(AGMessage*)_allocsafe(sizeof(AGMessage));
      agmsg->listview.lv=this;
      agmsg->listview.lvc=NULL;
      agmsg->listview.time=msg->time;
      agmsg->type=AG_LISTVIEW_PRESSED;
      aguix->putAGMsg(agmsg);
    }
  } else if((msg->type==KeyPress)||(msg->type==KeyRelease)) {
    if(hasFocus()==true) {
      // we have the focus so let's react to some keys
      // lets call an extra handler so it can be overwritten
      returnvalue=handleKeys(msg);
    }
  }
  return returnvalue;
}

void ListView::setParent(AWindow *tparent)
{
  this->parent=tparent;
  win=tparent->getSubWindow(0,x,y,w,h);
//  int tbg=tparent->getBG();
  int tbg=mbg;
  recalcMAXValues();
  subwins=new Window[maxdisplayv];
  winbgs=new int[maxdisplayv];
  vislvcs=new ListViewContent*[maxdisplayv];
  int tx,ty,tw,th;
  tw=w-2;
  if(vbar!=0) tw-=vbar_width;
  if(font==NULL) th=aguix->getCharHeight();
  else th=font->getCharHeight();
  if(hbar==1) ty=1+hbar_height; else ty=1;
  if(vbar==1) tx=1+vbar_width; else tx=1;
  for(int i=0;i<maxdisplayv;i++) {
    subwins[i]=tparent->getSubWindow(win,tx,ty+th*i,tw,th);
    aguix->SetWindowBG(subwins[i],tbg);
    aguix->ClearWin(subwins[i]);
    winbgs[i]=tbg;
    vislvcs[i]=NULL;
  }
  if(hbar!=0) hbarwin=tparent->getSubWindow(win,0,0,5,5);
  if(vbar!=0) vbarwin=tparent->getSubWindow(win,0,0,5,5);  
  setupBars();
}

void ListView::hbarredraw()
{
  if(hbar==0) return;
  int tx,ty,tw,th;
  tw=w-2;
  if(vbar!=0) tw-=vbar_width;
  th=hbar_height;
  ty=0;
  tx=0;

  aguix->SetWindowBG(hbarwin,parent->getBG());
  aguix->ClearWin(hbarwin);
  if(hbar_dir==1) {
    aguix->setFG(1);
  } else {
    aguix->setFG(2);
  }
  aguix->DrawLine(hbarwin,tx,ty+th-1,tx,ty);
  aguix->DrawLine(hbarwin,tx,ty,tx+th-1,ty);
  if(hbar_dir==1) {
    aguix->setFG(2);
  } else {
    aguix->setFG(1);
  }
  aguix->DrawLine(hbarwin,tx+1,ty+th-1,tx+th-1,ty+th-1);
  aguix->DrawLine(hbarwin,tx+th-1,ty+th-1,tx+th-1,ty+1);
  // Left-Arrow
  aguix->setFG(1);
  aguix->DrawLine(hbarwin,tx+th*4/5,ty+th/5,tx+th/5,ty+th/2);
  aguix->DrawLine(hbarwin,tx+th/5,ty+th/2,tx+th*4/5,ty+th*4/5);

  aguix->DrawLine(hbarwin,tx+th,ty+th-1,tx+tw-th-1,ty+th-1);
  aguix->setFG(2);
  aguix->DrawLine(hbarwin,tx+th,ty,tx+tw-th-1,ty);

  if(hbar_dir==2) {
    aguix->setFG(1);
  } else {
    aguix->setFG(2);
  }
  aguix->DrawLine(hbarwin,tx+tw-th,ty+th-1,tx+tw-th,ty);
  aguix->DrawLine(hbarwin,tx+tw-th,ty,tx+tw-1,ty);
  if(hbar_dir==2) {
    aguix->setFG(2);
  } else {
    aguix->setFG(1);
  }
  aguix->DrawLine(hbarwin,tx+tw-th+1,ty+th-1,tx+tw-1,ty+th-1);
  aguix->DrawLine(hbarwin,tx+tw-1,ty+th-1,tx+tw-1,ty+1);
  // Right-Arrow
  aguix->setFG(1);
  aguix->DrawLine(hbarwin,tx+tw-th*4/5,ty+th/5,tx+tw-th/5,ty+th/2);
  aguix->DrawLine(hbarwin,tx+tw-th/5,ty+th/2,tx+tw-th*4/5,ty+th*4/5);
  // jetzt noch die eigentliche Leiste
  tw-=2*th;  // Breite wenn alles anzeigbar ist
  th-=2;
  ty++;
  if(maxlen>maxdisplayh) {
    int dw=tw*maxdisplayh;
    dw/=maxlen;
    int a=maxlen-maxdisplayh;
    double b=tw-dw;  // das was brig bleibt
    b/=a;            // verteilt sich auch die nichtsichtbaren Zeichen
    tx=tx+hbar_height+(int)(xoffset*b);
    tw=dw;
  } else {
    tx=tx+hbar_height;
  }
  if(hbar_pressed==false) {
    aguix->setFG(2);
  } else {
    aguix->setFG(1);
  }
  aguix->DrawLine(hbarwin,tx,ty+th-1,tx,ty);
  aguix->DrawLine(hbarwin,tx,ty,tx+tw-1,ty);
  if(hbar_pressed==false) {
    aguix->setFG(1);
  } else {
    aguix->setFG(2);
  }
  aguix->DrawLine(hbarwin,tx+1,ty+th-1,tx+tw-1,ty+th-1);
  aguix->DrawLine(hbarwin,tx+tw-1,ty+th-1,tx+tw-1,ty+1);  
}

void ListView::vbarredraw()
{
  if(vbar==0) return;
  int tx,ty,tw,th;
  th=h-2;
  if(hbar!=0) th-=hbar_height;
  tw=vbar_width;
  tx=0;
  ty=0;

  aguix->SetWindowBG(vbarwin,parent->getBG());
  aguix->ClearWin(vbarwin);
  if(vbar_dir==1) {
    aguix->setFG(1);
  } else {
    aguix->setFG(2);
  }
  aguix->DrawLine(vbarwin,tx,ty+tw-1,tx,ty);
  aguix->DrawLine(vbarwin,tx,ty,tx+tw-1,ty);
  if(vbar_dir==1) {
    aguix->setFG(2);
  } else {
    aguix->setFG(1);
  }
  aguix->DrawLine(vbarwin,tx+1,ty+tw-1,tx+tw-1,ty+tw-1);
  aguix->DrawLine(vbarwin,tx+tw-1,ty+tw-1,tx+tw-1,ty+1);
  // Up-Arrow
  aguix->setFG(1);
  aguix->DrawLine(vbarwin,tx+tw/5,ty+tw*4/5,tx+tw/2,ty+tw/5);
  aguix->DrawLine(vbarwin,tx+tw/2,ty+tw/5,tx+tw*4/5,ty+tw*4/5);

  aguix->DrawLine(vbarwin,tx+tw-1,ty+tw,tx+tw-1,ty+th-tw-1);
  aguix->setFG(2);
  aguix->DrawLine(vbarwin,tx,ty+tw,tx,ty+th-tw-1);

  if(vbar_dir==2) {
    aguix->setFG(1);
  } else {
    aguix->setFG(2);
  }
  aguix->DrawLine(vbarwin,tx,ty+th-1,tx,ty+th-tw);
  aguix->DrawLine(vbarwin,tx,ty+th-tw,tx+tw-1,ty+th-tw);
  if(vbar_dir==2) {
    aguix->setFG(2);
  } else {
    aguix->setFG(1);
  }
  aguix->DrawLine(vbarwin,tx+1,ty+th-1,tx+tw-1,ty+th-1);
  aguix->DrawLine(vbarwin,tx+tw-1,ty+th-1,tx+tw-1,ty+th-tw);
  // Down-Arrow
  aguix->setFG(1);
  aguix->DrawLine(vbarwin,tx+tw/5,ty+th-tw*4/5,tx+tw/2,ty+th-tw/5);
  aguix->DrawLine(vbarwin,tx+tw/2,ty+th-tw/5,tx+tw*4/5,ty+th-tw*4/5);
  // jetzt noch die eigentliche Leiste
  th-=2*tw;  // Hhe wenn alles anzeigbar ist
  tw-=2;
  tx++;
  int elements=getElements();
  if(elements>maxdisplayv) {
    int dh=th*maxdisplayv;
    dh/=elements;
    int a=elements-maxdisplayv;
    double b=th-dh;  // das was brig bleibt
    b/=a;            // verteilt sich auch die nichtsichtbaren Zeichen
    ty=ty+vbar_width+(int)(yoffset*b);
    th=dh;
  } else {
    ty=ty+vbar_width;
  }
  if(vbar_pressed==false) {
    aguix->setFG(2);
  } else {
    aguix->setFG(1);
  }
  aguix->DrawLine(vbarwin,tx,ty+th-1,tx,ty);
  aguix->DrawLine(vbarwin,tx,ty,tx+tw-1,ty);
  if(vbar_pressed==false) {
    aguix->setFG(1);
  } else {
    aguix->setFG(2);
  }
  aguix->DrawLine(vbarwin,tx+1,ty+th-1,tx+tw-1,ty+th-1);
  aguix->DrawLine(vbarwin,tx+tw-1,ty+th-1,tx+tw-1,ty+1);  
}

void ListView::mredraw()
{
  if(bgset==false) {
//    aguix->SetWindowBG(win,parent->getBG());
    aguix->SetWindowBG(win,mbg);
    bgset=true;
  }
  aguix->ClearWin(win);
  aguix->setFG(2);
  aguix->DrawLine(win,0,h-1,0,0);
  aguix->DrawLine(win,0,0,w-1,0);
  aguix->setFG(1);
  aguix->DrawLine(win,1,h-1,w-1,h-1);
  aguix->DrawLine(win,w-1,h-1,w-1,1);
  if((hbar!=0)&&(vbar!=0)) {
    int tx,ty;
    tx=(vbar==1)?1:w-1-vbar_width;
    ty=(hbar==1)?1:h-1-hbar_height;
    aguix->setFG(0);
    aguix->FillRectangle(win,tx,ty,vbar_width,hbar_height);
  }
  aguix->Flush();
}

void ListView::recalcMAXValues()
{
  int th,tw;
  if(hbar!=0) {
    th=h-2-hbar_height;
  } else {
    th=h-2;
  }
  if(font==NULL) maxdisplayv=th/(aguix->getCharHeight());
  else maxdisplayv=th/(font->getCharHeight());
  if(vbar!=0) {
    tw=w-2-vbar_width;
  } else {
    tw=w-2;
  }
  if(font==NULL) maxdisplayh=tw/(aguix->getCharWidth());
  else maxdisplayh=tw/(font->getCharWidth());
}

ListViewContent *ListView::add(ListViewContent *lvc)
{
  lvcs->addElement(lvc);
  if(lvc->getActive()==true) activelvc=lvc;
  int len=strlen(lvc->getText());
  if(len>maxlen) maxlen=len;
  if(quickAdd==true) {
    if(qastartlvcpos<0) qastartlvcpos=getElements()-1;
  } else {
    lvc->setParent(this);
    int elements=getElements();
    if(((elements-1-yoffset)>=0)&&((elements-1-yoffset)<maxdisplayv)) {
      buildvisList();
      redrawalllvc();
    }
  }
  return lvc;
}

ListViewContent *ListView::add(ListViewContent *lvc,int pos)
{
  lvcs->addElementAt(pos,lvc);
  int len=strlen(lvc->getText());
  if(len>maxlen) maxlen=len;
  lvc->setParent(this);
  if(((pos-yoffset)>=0)&&((pos-yoffset)<maxdisplayv)) {
    buildvisList();
    redrawalllvc();
  }
  return lvc;
}

int ListView::remove(ListViewContent *lvc)
{
  int pos=lvcs->getIndex(lvc);
  return remove(pos);
}

int ListView::remove(int pos)
{
  int elements=getElements();
  if((pos<0)||(pos>=elements)) return -1;
  ListViewContent *tlvc=(ListViewContent*)lvcs->getElementAt(pos);
  if(activelvc==tlvc) {
    ListViewContent *tlvc2;
    if(pos>0) tlvc2=(ListViewContent*)lvcs->getElementAt(pos-1);
    else if(pos<(elements-1)) tlvc2=(ListViewContent*)lvcs->getElementAt(pos+1);
    else tlvc2=NULL;
    if(tlvc2==NULL) activelvc=NULL;
    else tlvc2->setActive(true);
  }
  lvcs->removeElementAt(pos);
  elements--;
  if((yoffset+maxdisplayv)>=elements) yoffset=elements-maxdisplayv;
  if(yoffset<0) yoffset=0;
  buildvisList();
  calcMaxLen();
  redrawalllvc();
  setupsubwins();
  delete tlvc;
  return 0;
}

void ListView::redrawlvc(int pos)
{
  if(pos>=getElements()) return;
  if((pos<yoffset)||(pos>=(yoffset+maxdisplayv))) return;
  // Eintrag ist sichtbar
  int i=pos-yoffset;
  ListViewContent *tlvc=vislvcs[i];
  if(tlvc!=NULL)
    winbgs[i]=tlvc->redraw(aguix,subwins[i],font,winbgs[i],maxdisplayh);
}

void ListView::redrawlvc(ListViewContent *tlvc)
{
  int i,j=-1,elements;
  elements=getElements();
  for(i=0;i<maxdisplayv;i++) {
    if((i+yoffset)>=elements) break;
    if(vislvcs[i]==tlvc) {
      j=i;
      break;
    }
  }
  if(j>=0) {
    redrawlvc(j+yoffset);
  }
}

void ListView::redrawalllvc()
{
  int elements=getElements();
  ListViewContent *lvc;
  for(int i=0;i<maxdisplayv;i++) {
    if((i+yoffset)>=elements) break;
    lvc=vislvcs[i];
    if(lvc!=NULL)
      winbgs[i]=lvc->redraw(aguix,subwins[i],font,winbgs[i],maxdisplayh);
  }
}

ListViewContent::ListViewContent()
{
  text=(char*)_allocsafe(1);
  text[0]=0;
  for(int i=0;i<4;i++) {
    fg[i]=1;
    bg[i]=0;
  }
  data=0;
  select=false;
  active=false;
  parent=NULL;
}

ListViewContent::ListViewContent(const char *ttext,int fg0,int fg1,int fg2,int fg3,
                                 int bg0,int bg1,int bg2, int bg3,int tdata)
{
  this->text=dupstring(ttext);
  fg[0]=fg0;
  fg[1]=fg1;
  fg[2]=fg2;
  fg[3]=fg3;
  bg[0]=bg0;
  bg[1]=bg1;
  bg[2]=bg2;
  bg[3]=bg3;
  this->data=tdata;
  select=false;
  active=false;
  parent=NULL;
}

void ListViewContent::setText(const char *newtext)
{
  setTextQ(newtext);
  if(parent!=NULL) parent->check4newlen(this);
}

void ListViewContent::useText(char *newtext)
{
  useTextQ(newtext);
  if(parent!=NULL) parent->check4newlen(this);
}

const char *ListViewContent::getText()
{
  return text;
}

void ListViewContent::setFG(int index2,int nfg)
{
  if((index2<0)||(index2>3)) return;
  setFGQ(index2,nfg);
  if(parent!=NULL) parent->redrawlvc(this);
}

int ListViewContent::getFG(int i)
{
  if((i<0)||(i>3)) return -1;
  return fg[i];
}

void ListViewContent::setBG(int index2,int nbg)
{
  if((index2<0)||(index2>3)) return;
  setBGQ(index2,nbg);
  if(parent!=NULL) parent->redrawlvc(this);
}

int ListViewContent::getBG(int i)
{
  if((i<0)||(i>3)) return -1;
  return bg[i];
}

void ListViewContent::setData(int tdata)
{
  this->data=tdata;
}

int ListViewContent::getData()
{
  return data;
}

void ListViewContent::setSelect(bool value)
{
  setSelectQ(value);
  if(parent!=NULL) parent->redrawlvc(this);
}

bool ListViewContent::getSelect()
{
  return select;
}

void ListViewContent::setActive(bool value)
{
  setActiveQ(value);
  if(parent!=NULL) parent->redrawlvc(this);
}

bool ListViewContent::getActive()
{
  return active;
}

int ListViewContent::redraw(AGUIX *aguix,Window win,AGUIXFont *font,int lastbg,int chars)
{
  int i,tbg,tfg;
  int xoffset=parent->getXOffset();
  char *tstr=dupstring(text);
  int len=strlen(text);
  i=(select==true)?1:0;
  i+=(active==true)?2:0;
  tbg=bg[i];
  tfg=fg[i];
  if(lastbg!=tbg) {
    aguix->SetWindowBG(win,tbg);
  }
  aguix->ClearWin(win);
  if(font==NULL) aguix->setFG(tfg);
  else aguix->setFG(font->getGC(),tfg);
  if(len>xoffset) {
    if((len-xoffset)>chars) tstr[xoffset+chars]=0;
    if(font==NULL) aguix->DrawText(win,tstr+xoffset,1,0);
    else aguix->DrawText(win,font,tstr+xoffset,1,0);
  }
  _freesafe(tstr);
  return tbg;
}

void ListViewContent::setParent(ListView *lv)
{
  this->parent=lv;
}

void ListViewContent::setFGQ(int index2,int nfg)
{
  if((index2<0)||(index2>3)) return;
  fg[index2]=nfg;
}

void ListViewContent::setBGQ(int index2,int nbg)
{
  if((index2<0)||(index2>3)) return;
  bg[index2]=nbg;
}

void ListViewContent::setSelectQ(bool value)
{
  select=value;
}

void ListViewContent::setActiveQ(bool value)
{
  if(value==active) return;
  if(value==false) {
    active=false;
    if(parent!=NULL) parent->setActiveLVC(NULL);
  } else {
    if(parent!=NULL) {
      ListViewContent *tlvc=parent->getActiveLVC();
      if(tlvc!=NULL) {
        tlvc->setActive(false);
      }
    }
    active=true;
    if(parent!=NULL) parent->setActiveLVC(this);
  }
}

void ListViewContent::setTextQ(const char *newtext)
{
  if(text!=NULL) _freesafe(text);
  text=dupstring(newtext);
}

void ListViewContent::useTextQ(char *newtext)
{
  if(text!=NULL) _freesafe(text);
  text=newtext;
}

void ListView::check4newlen(ListViewContent *lvc)
{
  int newlen=strlen(lvc->getText());
  if(newlen>maxlen) {
    maxlen=newlen;
    mredraw();
  } else if(newlen<maxlen) {
    calcMaxLen();
  }
}

void ListView::calcMaxLen()
{
  int nmlen=0,tl;
  int id=lvcs->initEnum();
  ListViewContent *tlvc=(ListViewContent*)lvcs->getFirstElement(id);
  while(tlvc!=NULL) {
    tl=strlen(tlvc->getText());
    if(tl>nmlen) nmlen=tl;
    tlvc=(ListViewContent*)lvcs->getNextElement(id);
  }
  lvcs->closeEnum(id);
  if(nmlen!=maxlen) {
    if(nmlen<maxlen) {
      if((xoffset+maxdisplayh)>nmlen) {
        xoffset=nmlen-maxdisplayh;
        if(xoffset<0) xoffset=0;
        redrawalllvc();
      }
    }
    maxlen=nmlen;
    mredraw();
  }
}

void ListView::buildvisList()
{
  int elements=getElements();
  for(int i=yoffset;(i<elements)&&((i-yoffset)<maxdisplayv);i++) {
    vislvcs[i-yoffset]=(ListViewContent*)lvcs->getElementAt(i);
  }
}

int ListView::getElements()
{
  return lvcs->size();
}

int ListView::getXOffset()
{
  return xoffset;
}

ListViewContent::~ListViewContent()
{
  if(text!=NULL) _freesafe(text);
}

void ListView::setXOffset(int new_offset)
{
  int no=new_offset;
  if((no+maxdisplayh)>maxlen) no=maxlen-maxdisplayh;
  if(no<0) no=0;
  if(xoffset==no) return;
  xoffset=no;
  redrawalllvc();
  hbarredraw();
}

void ListView::setYOffset(int new_offset)
{
  int no=new_offset;
  if((no+maxdisplayv)>getElements()) no=getElements()-maxdisplayv;
  if(no<0) no=0;
  if(yoffset==no) return; 
  yoffset=no;
  buildvisList();
  redrawalllvc();
  setupsubwins();
  vbarredraw();
}

int ListView::getYOffset()
{
  return yoffset;
}

int ListView::getMaxLen()
{
  return maxlen;
}

int ListView::getMAXdisplayv()
{
  return maxdisplayv;
}

int ListView::getMAXdisplayh()
{
  return maxdisplayh;
}

void ListView::setHBarState(int state)
{
  int oldmaxdv;
  oldmaxdv=maxdisplayv;
  if(hbar==state) return;
  if((state==0)&&(parent!=NULL)) parent->removeSubWin(hbarwin);
  if((hbar==0)&&(state>0)) hbarwin=parent->getSubWindow(win,0,0,5,5);
  hbar=state;
  recalcMAXValues();
  rebuildSubWins(oldmaxdv);
  reconfSubWins();
  setupBars();
}

void ListView::setVBarState(int state)
{
  if(state==vbar) return;
  if((state==0)&&(parent!=NULL)) parent->removeSubWin(vbarwin);
  if((vbar==0)&&(state>0)) vbarwin=parent->getSubWindow(win,0,0,5,5);
  vbar=state;
  recalcMAXValues();
  reconfSubWins();
  setupBars();
}

int ListView::getHBarState()
{
  return hbar;
}

int ListView::getVBarState()
{
  return vbar;
}

void ListView::setVBarWidth(int new_width)
{
  if((new_width<0)||((new_width+4)>=w)) return;
  vbar_width=new_width;
  recalcMAXValues();
  reconfSubWins();
  setupBars();
  vbarredraw();
}

int ListView::getVBarWidth()
{
  return vbar_width;
}

void ListView::setHBarHeight(int new_height)
{
  if((new_height<0)||((new_height+4)>=h)) return;
  int oldmaxdv;
  oldmaxdv=maxdisplayv;
  hbar_height=new_height;
  recalcMAXValues();
  rebuildSubWins(oldmaxdv);
  reconfSubWins();
  setupBars();
  hbarredraw();
}

int ListView::getHBarHeight()
{
  return hbar_height;
}

void ListView::setupsubwins()
{
  int elements=getElements();
  if((yoffset+maxdisplayv)>elements) {
    // Alle SubWindows, die kein LVC enthalten, auf parent->getBG() setzen
//    int tbg=parent->getBG();
    int tbg=mbg;
    for(int i=elements-yoffset;i<maxdisplayv;i++) {
      aguix->SetWindowBG(subwins[i],tbg);
      aguix->ClearWin(subwins[i]);
      winbgs[i]=tbg;
    }
  }
}

void ListView::setActiveLVC(ListViewContent *tlvc)
{
  activelvc=tlvc;
}

ListViewContent *ListView::getActiveLVC()
{
  return activelvc;
}

void ListView::rebuildSubWins(int oldsize)
{
  if(oldsize==maxdisplayv) return;
  Window *nsubwins;
  int *nwinbgs;
  int i;
  ListViewContent **nvislvcs;
  nsubwins=new Window[maxdisplayv];
  nwinbgs=new int[maxdisplayv];
  nvislvcs=new ListViewContent*[maxdisplayv];
  if(maxdisplayv<oldsize) {
    for(i=maxdisplayv;i<oldsize;i++) {
      parent->removeSubWin(subwins[i]);
    }
    for(i=0;i<maxdisplayv;i++) {
      nsubwins[i]=subwins[i];
      nwinbgs[i]=winbgs[i];
      nvislvcs[i]=vislvcs[i];
    }
    delete subwins;
    delete winbgs;
    delete vislvcs;
    subwins=nsubwins;
    winbgs=nwinbgs;
    vislvcs=nvislvcs;
  } else {
    for(i=0;i<oldsize;i++) {
      nsubwins[i]=subwins[i];
      nwinbgs[i]=winbgs[i];
    }
    int tx,ty,tw,th;
    tw=w-2;
    if(vbar!=0) tw-=vbar_width;
    if(font==NULL) th=aguix->getCharHeight();
    else th=font->getCharHeight();
    if(hbar==1) ty=1+hbar_height; else ty=1;
    if(vbar==1) tx=1+vbar_width; else tx=1;
//    int tbg=parent->getBG();
    int tbg=mbg;
    for(i=oldsize;i<maxdisplayv;i++) {
      nsubwins[i]=parent->getSubWindow(win,tx,ty+th*i,tw,th);
      aguix->SetWindowBG(nsubwins[i],tbg);
      aguix->ClearWin(nsubwins[i]);
      nwinbgs[i]=tbg;
    }
    delete subwins;
    delete winbgs;
    delete vislvcs;
    subwins=nsubwins;
    winbgs=nwinbgs;
    vislvcs=nvislvcs;
    buildvisList();
    int elements=getElements();
    if((yoffset+maxdisplayv)>elements) {
      int ny;
      ny=elements-maxdisplayv;
      if(ny<0) ny=0;
      setYOffset(ny);
    }
  }
}

void ListView::reconfSubWins()
{
  int tx,ty,tw,th;
  tw=w-2;
  if(vbar!=0) tw-=vbar_width;
  if(font==NULL) th=aguix->getCharHeight();
  else th=font->getCharHeight();
  if(hbar==1) ty=1+hbar_height; else ty=1;
  if(vbar==1) tx=1+vbar_width; else tx=1;
  for(int i=0;i<maxdisplayv;i++) {
    parent->resizeSubWin(subwins[i],tw,th);
    parent->moveSubWin(subwins[i],tx,ty+i*th);
  }
}

void ListView::handleSelect(Message *msg)
{
  int j=-1;
  int dely,i;
  int elements=getElements();
  bool ende;

  for(i=0;i<maxdisplayv;i++) {
    if(msg->window==subwins[i]) j=i;
  }
  if(j==-1) return;
  if((j+yoffset)>elements) return;

  if(quickAdd==true) return;

  int ch;
  if(font==NULL) ch=aguix->getCharHeight();
  else ch=font->getCharHeight();
  ListViewContent *tlvc,*firstselect;
  tlvc=vislvcs[j];
  bool state=tlvc->getSelect();
  state=(state==false)?true:false;
  tlvc->setSelect(state);
  tlvc->setActive(true);
  runSelectHandler(tlvc,j+yoffset);
  aguix->Flush();

  dely=j*ch;
  Message *tmsg;
  int newelement,element=j,my,elementh=ch;
  int dest,source;
  int selected=1;
  bool ignore4end = false;

  firstselect=tlvc;
  newelement=element;
  Time lasttime=0;
  for(ende=false;ende==false;) {
    tmsg=aguix->wait4messReact(MES_GET,false);
    if(tmsg!=NULL) {
      if(tmsg->type==ButtonRelease) {
        ende=true;
	lasttime=tmsg->time;

	my = tmsg->mousey;
	my += dely;
	newelement = my / elementh;
	if ( my < 0 ) newelement--;
	ignore4end = false;  // to allow apply new elementselection
      } else if((tmsg->type==MotionNotify)||(tmsg->type==EnterNotify)||(tmsg->type==LeaveNotify)) {
        Window root,child;
        int root_x,root_y,win_x,win_y;
        unsigned int keys_buttons;
        XQueryPointer(aguix->getDisplay(),msg->window,&root,&child,&root_x,&root_y,&win_x,
                      &win_y,&keys_buttons);
	if ( ( keys_buttons & Button1Mask ) == 0 ) {
	  // Release before Release-Msg from X
	  // this really happens
	  ignore4end = true;
	} else {
	  my=win_y;
	  my+=dely;
	  newelement=my/elementh;
	  if(my<0) newelement--;
	}
      }
      _freesafe(tmsg);
    } else {
      waittime(50);
      Window root,child;
      int root_x,root_y,win_x,win_y;
      unsigned int keys_buttons;
      XQueryPointer(aguix->getDisplay(),msg->window,&root,&child,&root_x,&root_y,&win_x,
                    &win_y,&keys_buttons);
      if ( ( keys_buttons & Button1Mask ) == 0 ) {
	// Release before Release-Msg from X
	// this really happens
	ignore4end = true;
      } else {
	my=win_y;
	my+=dely;
	newelement=my/elementh;
	if(my<0) newelement--;
      }
    }
    if ( ignore4end == false ) {
      if((element!=newelement)||(newelement<0)||(newelement>=maxdisplayv)) {
	dest=newelement;
	if(newelement<0) {
	  scrollV1(-1);
	  dest=0;
	  element++;
	} else if(newelement>=maxdisplayv) {
	  scrollV1(1);
	  dest=maxdisplayv-1;
	  element--;
	}
	source=element;
	if(source<0) source=0;
	if(source>=maxdisplayv) source=maxdisplayv-1;
	int starte,tende,dire;
	starte=source;
	tende=dest;
	if(starte<tende) {
	  starte++;
	  dire=1;
	} else if(starte>tende) {
	  starte--;
	  dire=-1;
	} else dire=0;
	// for i = starte to tende step dire
	for(i=starte;;i+=dire) {
	  if((i+yoffset)>=elements) break;
	  tlvc=vislvcs[i];
	  if(tlvc!=NULL) {
	    tlvc->setSelect(state);
	    runSelectHandler(tlvc,i+yoffset);
	    selected++;
	  }
	  if(i==tende) break;
	}
	element=dest;
	if((element+yoffset)>=elements) {
	  element=elements-yoffset-1;
	}
	tlvc=vislvcs[element];
	if(tlvc!=NULL) {
	  tlvc->setActive(true);
	  runSelectHandler(tlvc,element+yoffset);
	}
      }
    }
  }
  AGMessage *agmsg=(AGMessage*)_allocsafe(sizeof(AGMessage));
  agmsg->listview.lv=this;
  agmsg->listview.lvc=firstselect;
  agmsg->listview.time=lasttime;
  if(selected==1) {
    agmsg->type=AG_LISTVIEW_ONESELECT;
  } else {
    agmsg->type=AG_LISTVIEW_MULTISELECT;
  }
  aguix->putAGMsg(agmsg);
}

void ListView::handleThird(Message *msg)
{
  int midx,midy,dsx,dsy,xdir,ydir;
  int mx,my;
  int dely;
  bool ende;
  Window relwin;
  
  midx=w-2;
  if(vbar!=0) midx-=vbar_width;
  dsx=midx;
  midx/=2;
  midy=h-2;
  if(hbar!=0) midy-=hbar_height;
  dsy=midy;
  midy/=2;
  dsx/=6;
  dsy/=6;
  mx=msg->mousex;
  my=msg->mousey;
  Message *tmsg;
  int j=-1;
  for(int i=0;i<maxdisplayv;i++) {
    if(msg->window==subwins[i]) j=i;
  }
  if(j==-1) return;
  relwin=subwins[j];
  if(font==NULL) dely=j*aguix->getCharHeight();
  else dely=j*font->getCharHeight();
  my=my+dely;
  int waitt;
  for(ende=false;ende==false;) {
    if(mx<(midx-dsx)) xdir=-1;
    else if(mx>(midx+dsx)) xdir=1;
    else xdir=0;

    if(my<(midy-dsy)) ydir=-1;
    else if(my>(midy+dsy)) ydir=1;
    else ydir=0;

    if(xdir!=0) setXOffset(xoffset+xdir);
    if(ydir!=0) scrollV(ydir);

    aguix->Flush();

    if((my<dsy)||(my>(midy+2*dsy))) waitt=20;
    else if((mx<dsx)||(mx>(midx+2*dsx))) waitt=20; else waitt=50;
    if(ende==false) waittime(waitt);
    tmsg=aguix->wait4messReact(MES_GET,false);
    if(tmsg!=NULL) {
      if(tmsg->type==ButtonRelease) ende=true;
      else if(tmsg->type==MotionNotify) {
	aguix->queryPointer(relwin,&mx,&my);
        my=my+dely;
      } else {
/*        while((tmsg2=aguix->wait4mess(MES_GET,false))!=NULL) {
          if(tmsg2->type==ButtonRelease) ende=true;
          _freesafe(tmsg2);
        }*/
      }
      _freesafe(tmsg);
    }
  }
}

bool ListView::handleHBar(Message *msg)
{
  if(hbar==0) return false;
  int mx,my;
  int dir;
  mx=msg->mousex;
  my=msg->mousey;
  int hx,hy,hw,hh;
  bool ende;
  hw=w-2;
  if(vbar!=0) hw-=vbar_width;
  hh=hbar_height;
  hy=0;
  hx=0;
  dir=0;
  if((mx>hx)&&(mx<=(hx+hh))) {
    // left
    dir=-1;
    hbar_dir=1;
  }
  if((mx>(hx+hw-hh))&&(mx<=(hx+hw))) {
    // right
    dir=1;
    hbar_dir=2;
  }
  if(dir!=0) {
    hbarredraw();
    aguix->Flush();
    Message *tmsg,*tmsg2;
    for(ende=false;ende==false;) {
      setXOffset(xoffset+dir);
      aguix->Flush();
      if(ende==false) waittime(70);
      tmsg=aguix->wait4messReact(MES_GET,false);
      if(tmsg!=NULL) {
        if(tmsg->type==ButtonRelease) ende=true;
        else {
          while((tmsg2=aguix->wait4messReact(MES_GET,false))!=NULL) {
            if(tmsg2->type==ButtonRelease) ende=true;
            _freesafe(tmsg2);
          }
        }
        _freesafe(tmsg);
      }
    }
    hbar_dir=0;
    hbarredraw();
    aguix->Flush();
  } else {
    int tx,ty,tw,th;
    double b=0;
    tw=w-2;
    if(vbar!=0) tw-=vbar_width;
    th=hbar_height;
    ty=0;
    tx=0;
    tw-=2*th;  // Breite wenn alles anzeigbar ist
    th-=2;
    ty++;
    if(maxlen>maxdisplayh) {
      int dw=tw*maxdisplayh;
      dw/=maxlen;
      int a=maxlen-maxdisplayh;
      b=tw-dw;  // das was brig bleibt
      b/=a;            // verteilt sich auch die nichtsichtbaren Zeichen
      tx=tx+hbar_height+(int)(xoffset*b);
      tw=dw;
    } else {
      tx=tx+hbar_height;
    }
    if((mx>=tx)&&(mx<=(tx+tw))&&(my>=ty)&&(my<=(ty+th))) {
      // the hbar-scroller is pressed
      if(maxlen>maxdisplayh) {
        Display *dsp=aguix->getDisplay();
        int grabs=XGrabPointer(dsp,hbarwin,False,Button1MotionMask|ButtonReleaseMask,GrabModeAsync,GrabModeAsync,/*hbarwin*/None,None,CurrentTime);
	aguix->setCursor(hbarwin,AGUIX::SCROLLH_CURSOR);
        int dx=mx-tx;
	int newx=0;
	bool gotnewx=false;
/*        Window root,child;
        int root_x,root_y,win_x,win_y;
        unsigned int keys_buttons;*/
        hbar_pressed=true;
        hbarredraw();
        aguix->Flush();
        Message *tmsg,*tmsg2;
        for(ende=false;ende==false;gotnewx=false) {
          tmsg=aguix->wait4messReact(MES_GET,false);
          if(tmsg!=NULL) {
            if(tmsg->type==ButtonRelease) ende=true;
            else if(tmsg->type==MotionNotify) {
              newx=tmsg->mousex;
	      gotnewx=true;
            } else {
              while((tmsg2=aguix->wait4messReact(MES_GET,false))!=NULL) {
                if(tmsg2->type==ButtonRelease) ende=true;
                _freesafe(tmsg2);
              }
            }
            _freesafe(tmsg);
          } else {
//	    aguix->Flush();
	    waittime(20);
/*            XQueryPointer(aguix->getDisplay(),hbarwin,&root,&child,&root_x,&root_y,&win_x,
                          &win_y,&keys_buttons);
	    newx=win_x;
	    gotnewx=true;*/
	  }
	  if(gotnewx==true) {
            int tmx,pos;
            double f1;
            tmx=newx;
            tmx-=hbar_height;
            tmx-=dx;
            pos=tmx;
            f1=pos/b;
            setXOffset((int)f1);
	    aguix->Flush();
	  }
        }
	aguix->unsetCursor(hbarwin);
	if(grabs==GrabSuccess) XUngrabPointer(dsp,CurrentTime);
        hbar_pressed=false;
        hbarredraw();
        aguix->Flush();
      }
    } else {
      if(mx<tx) {
        setXOffset(xoffset-maxdisplayh+1);
      } else {
        setXOffset(xoffset+maxdisplayh-1);
      }
      aguix->Flush();
    }
  }
  return true;
}

bool ListView::handleVBar(Message *msg)
{
  if(vbar==0) return false;
  int mx,my;
  int dir;
  mx=msg->mousex;
  my=msg->mousey;
  int vx,vy,vw,vh;
  bool ende;
  vh=h-2;
  if(hbar!=0) vh-=hbar_height;
  vw=vbar_width;
  vx=0;
  vy=0;
  // vbar
  dir=0;
  if((my>vy)&&(my<=(vy+vw))) {
    // up
    dir=-1;
    vbar_dir=1;
  }
  if((my>(vy+vh-vw))&&(my<=(vy+vh))) {
    // down
    dir=1;
    vbar_dir=2;
  }
  if(dir!=0) {
    vbarredraw();
    aguix->Flush();
    Message *tmsg,*tmsg2;
    for(ende=false;ende==false;) {
      scrollV1(dir);
      aguix->Flush();
      if(ende==false) waittime(70);
      tmsg=aguix->wait4messReact(MES_GET,false);
      if(tmsg!=NULL) {
        if(tmsg->type==ButtonRelease) ende=true;
        else {
          while((tmsg2=aguix->wait4messReact(MES_GET,false))!=NULL) {
            if(tmsg2->type==ButtonRelease) ende=true;
            _freesafe(tmsg2);
          }
        }
        _freesafe(tmsg);
      }
    }
    vbar_dir=0;
    vbarredraw();
    aguix->Flush();
  } else {
    int tx,ty,tw,th;
    double b=0;
    int elements=getElements();
    th=h-2;
    if(hbar!=0) th-=hbar_height;
    tw=vbar_width;
    tx=0;
    ty=0;
    th-=2*tw;  // Hhe wenn alles anzeigbar ist
    tw-=2;
    tx++;
    if(elements>maxdisplayv) {
      int dh=th*maxdisplayv;
      dh/=elements;
      int a=elements-maxdisplayv;
      b=th-dh;  // das was brig bleibt
      b/=a;            // verteilt sich auch die nichtsichtbaren Zeichen
      ty=ty+vbar_width+(int)(yoffset*b);
      th=dh;
    } else {
      ty=ty+vbar_width;
    }
    ty++;
    if((mx>=tx)&&(mx<=(tx+tw))&&(my>=ty)&&(my<=(ty+th))) {
      // vbar-scroller pressed
      if(elements>maxdisplayv) {
        Display *dsp=aguix->getDisplay();
        int grabs=XGrabPointer(dsp,vbarwin,False,Button1MotionMask|ButtonReleaseMask,GrabModeAsync,GrabModeAsync,/*vbarwin*/None,None,CurrentTime);
	aguix->setCursor(vbarwin,AGUIX::SCROLLV_CURSOR);
        int dy=my-ty;
	int newy=0;
	bool gotnewy=false;
/*        Window root,child;
        int root_x,root_y,win_x,win_y;
        unsigned int keys_buttons;*/
        vbar_pressed=true;
        vbarredraw();
        aguix->Flush();
        Message *tmsg,*tmsg2;
        for(ende=false;ende==false;gotnewy=false) {
          tmsg=aguix->wait4messReact(MES_GET,false);
          if(tmsg!=NULL) {
            if(tmsg->type==ButtonRelease) ende=true;
            else if(tmsg->type==MotionNotify) {
              newy=tmsg->mousey;
	      gotnewy=true;
            } else {
              while((tmsg2=aguix->wait4messReact(MES_GET,false))!=NULL) {
                if(tmsg2->type==ButtonRelease) ende=true;
                _freesafe(tmsg2);
              }
            }
            _freesafe(tmsg);
          } else {
//	    aguix->Flush();
	    waittime(20);
/*            XQueryPointer(aguix->getDisplay(),vbarwin,&root,&child,&root_x,&root_y,&win_x,
                          &win_y,&keys_buttons);
	    newy=win_y;
	    gotnewy=true;*/
	  }
	  if(gotnewy==true) {
     	    int tmy,pos;
            double f1;
            tmy=newy;
            tmy-=vbar_width;
            tmy-=dy;
            pos=tmy;
            f1=pos/b;
            setYOffset((int)f1);
	    aguix->Flush();
	  }
        }
	aguix->unsetCursor(vbarwin);
        if(grabs==GrabSuccess) XUngrabPointer(dsp,CurrentTime);
        vbar_pressed=false;
        vbarredraw();
        aguix->Flush();
      }
    } else {
      if(my<ty) {
        scrollV(-maxdisplayv+1);
      } else {
        scrollV(maxdisplayv-1);
      }
      aguix->Flush();
    }
  }
  return true;
}

void ListView::scrollV(int delta)
{
  //TODO
  setYOffset(yoffset+delta);
}

void ListView::scrollV1(int dir)
{
  //TODO
  setYOffset(yoffset+dir);
}

void ListView::setSelectHandler(void (*nsh)(ListView*,ListViewContent*,int))
{
  selecthandler=nsh;
}

ListViewContent *ListView::get(int pos)
{
  return (ListViewContent*)lvcs->getElementAt(pos);
}

ListViewContent *ListView::getNext()
{
  return (ListViewContent*)lvcs->getNextElement();
}

/*
 * showActive zeigt den aktiven Eintrag
 * sie tut nichts, wenn Eintrag sichtbar
 * ist der oberhalb, wird er oben als erstes Element sichtbar
 * ist er unterhalb, wird er als letztes sichtbar
 */
void ListView::showActive()
{
  int id=lvcs->initEnum();
  ListViewContent *tlvc=(ListViewContent*)lvcs->getFirstElement(id);
  int actnr,i;
  actnr=-1;
  i=0;
  while(tlvc!=NULL) {
    if(tlvc->getActive()==true) {
      actnr=i;
      break;
    }
    tlvc=(ListViewContent*)lvcs->getNextElement(id);
    i++;
  }
  lvcs->closeEnum(id);
  if(actnr>=0) {
    if(actnr<yoffset) setYOffset(actnr);
    else if((actnr-yoffset)>=maxdisplayv) setYOffset(actnr-maxdisplayv+1);
  }
}

/*
 * centerActive zentriert Ansicht auf aktiven Eintrag
 * egal, ob sichtbar oder nicht
 */
void ListView::centerActive()
{
  int id=lvcs->initEnum();
  ListViewContent *tlvc=(ListViewContent*)lvcs->getFirstElement(id);
  int actnr,i;
  actnr=-1;
  i=0;
  while(tlvc!=NULL) {
    if(tlvc->getActive()==true) {
      actnr=i;
      break;
    }
    tlvc=(ListViewContent*)lvcs->getNextElement(id);
    i++;
  }
  lvcs->closeEnum(id);
  if(actnr>=0) {
    setYOffset(actnr-maxdisplayv/2);
  }
}

int ListView::removeAll()
{
  int elements=getElements();
  if(elements==0) return -1;
  int id=lvcs->initEnum();
  ListViewContent *tlvc=(ListViewContent*)lvcs->getFirstElement(id);
  while(tlvc!=NULL) {
    lvcs->removeElementAt(0);
    delete tlvc;
    tlvc=(ListViewContent*)lvcs->getFirstElement(id);
  }
  lvcs->closeEnum(id);
  activelvc=NULL;
  yoffset=0;
  buildvisList();
  maxlen=0;
  redrawalllvc();
  setupsubwins();
  return 0;
}

void ListView::setupBars()
{
  if(hbar!=0) {
    int tx,ty,tw,th;
    tw=w-2;
    if(vbar!=0) tw-=vbar_width;
    th=hbar_height;
    if(hbar==1) ty=1; else ty=h-1-th;
    if(vbar==1) tx=1+vbar_width; else tx=1;
    parent->resizeSubWin(hbarwin,tw,th);
    parent->moveSubWin(hbarwin,tx,ty);
  }
  if(vbar!=0) {
    int tx,ty,tw,th;
    th=h-2;
    if(hbar!=0) th-=hbar_height;
    tw=vbar_width;
    if(vbar==1) tx=1; else tx=w-1-tw;
    if(hbar==1) ty=1+hbar_height; else ty=1;
    parent->resizeSubWin(vbarwin,tw,th);
    parent->moveSubWin(vbarwin,tx,ty);
  }
}

int ListView::getPos(ListViewContent *lvc)
{
  return lvcs->getIndex(lvc);
}

int ListView::setFont(char *fontname)
{
  font=aguix->getFont(fontname);
  if(font==NULL) return -1;
  int oldmaxdv;
  oldmaxdv=maxdisplayv;
  recalcMAXValues();
  rebuildSubWins(oldmaxdv);
  reconfSubWins();
  setupBars();
  return 0;
}

const char *ListView::getType()
{
  return type;
}

bool ListView::isType(const char *qtype)
{
  if(strcmp(type,qtype)==0) return true;
  return false;
}

bool ListView::isParent(Window child)
{
  if(child==win) return true;
  if(child==hbarwin) return true;
  if(child==vbarwin) return true;
  for(int i=0;i<maxdisplayv;i++) {
    if(child==subwins[i]) return true;
  }
  return false;
}

void ListView::runSelectHandler(ListViewContent *lvc,int nr)
{
  if(selecthandler!=NULL) (selecthandler)(this,lvc,nr);
}

void ListView::startQuickAdd()
{
  qastartlvcpos=-1;
  quickAdd=true;
}

void ListView::endQuickAdd()
{
  if((qastartlvcpos<0)||(quickAdd==false)) return;
  int id=lvcs->initEnum();
  ListViewContent *tlvc=(ListViewContent*)lvcs->getElementAt(id,qastartlvcpos);
  while(tlvc!=NULL) {
    tlvc->setParent(this);
    tlvc=(ListViewContent*)lvcs->getNextElement(id);
  }
  buildvisList();
  redraw();
  lvcs->closeEnum(id);
  quickAdd=false;
  qastartlvcpos=-1;
}

void ListView::setSize(int ns)
{
  int elems=lvcs->size();
  ListViewContent *tlvc;
  while(elems<ns) {
    tlvc=new ListViewContent("",1,1,1,1,0,0,0,0,0);
    tlvc->setParent(this);
    lvcs->addElement(tlvc);
    elems++;
  }
  while(elems>ns) {
    tlvc=(ListViewContent*)lvcs->getElementAt(elems-1);
    if(activelvc==tlvc) activelvc=NULL;
    lvcs->removeElementAt(elems-1);
    elems--;
    delete tlvc;
  }
  if((yoffset+maxdisplayv)>elems) yoffset=elems-maxdisplayv;
  if(yoffset<0) yoffset=0;
}

void ListView::finish()
{
  buildvisList();
  calcMaxLen();  
  redraw();
  setupsubwins();
}

void ListView::showlvc(ListViewContent *lvc)
{
  int pos=-1,i,j;
  bool found;
  int elements=getElements();
  // first try to find lvc in vislvc
  found=false;
  for(i=yoffset;(i<elements)&&((i-yoffset)<maxdisplayv);i++) {
    if(lvc==vislvcs[i-yoffset]) {
      pos=i;
      found=true;
      break;
    }
  }
  if(found==false) {
    // now find it in lvcs
    i=yoffset-1;
    j=yoffset+maxdisplayv;
    while((i>=0)||(j<elements)) {
      if(i>=0) {
        if((ListViewContent*)lvcs->getElementAt(i)==lvc) {
	  pos=i;
	  found=true;
	  break;
	}
      }
      if(j<elements) {
        if((ListViewContent*)lvcs->getElementAt(j)==lvc) {
	  pos=j;
	  found=true;
	  break;
	}
      }
      i--;
      j++;
    }
  }
  if(pos>=0) {
    if(pos<yoffset) {
      if(abs(pos-yoffset)==1) scrollV1(-1);
      else scrollV(pos-yoffset);
    } else if((pos-yoffset)>=maxdisplayv) {
      if(abs(pos-yoffset-maxdisplayv+1)==1) scrollV1(1);
      else scrollV(pos-yoffset-maxdisplayv+1);
    }
  }
}

int ListView::initLVCEnum()
{
  return lvcs->initEnum();
}

void ListView::closeLVCEnum(int id)
{
  lvcs->closeEnum(id);
}

ListViewContent *ListView::get(int id,int pos)
{
  return (ListViewContent*)lvcs->getElementAt(id,pos);
}

ListViewContent *ListView::getNext(int id)
{
  return (ListViewContent*)lvcs->getNextElement(id);
}

int ListView::getLVCHeight()
{
  if(font==NULL) return aguix->getCharHeight();
  else return font->getCharHeight();
}

int ListView::getMBG()
{
  return mbg;
}

void ListView::setMBG(int color)
{
  mbg=color;
  if(parent!=NULL) {
    aguix->SetWindowBG(win,mbg);
    setupsubwins();
  }
}

int ListView::maximizeX()
{
  int elem_width,s;

  if(font==NULL) elem_width=aguix->getCharWidth();
  else elem_width=font->getCharWidth();
  s=elem_width*maxlen+((vbar!=0)?vbar_width:0)+2;
  resize(s,getHeight());
  return s;
}

int ListView::maximizeY()
{
  int elem_height,s;

  if(font==NULL) elem_height=aguix->getCharHeight();
  else elem_height=font->getCharHeight();
  s=elem_height*getElements()+((hbar!=0)?hbar_height:0)+2;
  resize(getWidth(),s);
  return s;
}

void ListView::handleMiddle(Message *msg)
{
  int j=-1;
  int dely,i;
  int elements=getElements();
  bool ende;
  for(i=0;i<maxdisplayv;i++) {
    if(msg->window==subwins[i]) j=i;
  }
  if(j==-1) return;
  if((j+yoffset)>elements) return;

  if(quickAdd==true) return;

  int ch;
  if(font==NULL) ch=aguix->getCharHeight();
  else ch=font->getCharHeight();
  ListViewContent *tlvc,*firstselect;
  tlvc=vislvcs[j];
  tlvc->setActive(true);
  runSelectHandler(tlvc,j+yoffset);
  aguix->Flush();
  
  Time lasttime=msg->time;

  dely=j*ch;
  Message *tmsg;
  int newelement,element=j,my,elementh=ch;
  int dest;
  firstselect=tlvc;
  newelement=element;
  for(ende=false;ende==false;) {
    tmsg=aguix->wait4messReact(MES_GET,false);
    if(tmsg!=NULL) {
      if(tmsg->type==ButtonRelease) {
        ende=true;
	lasttime=tmsg->time;
      } else if((tmsg->type==MotionNotify)||(tmsg->type==EnterNotify)||(tmsg->type==LeaveNotify)) {
        Window root,child;
        int root_x,root_y,win_x,win_y;
        unsigned int keys_buttons;
        XQueryPointer(aguix->getDisplay(),msg->window,&root,&child,&root_x,&root_y,&win_x,
                      &win_y,&keys_buttons);
        my=win_y;
        my+=dely;
        newelement=my/elementh;
        if(my<0) newelement--;
      }
      _freesafe(tmsg);
    } else {
      waittime(50);
      Window root,child;
      int root_x,root_y,win_x,win_y;
      unsigned int keys_buttons;
      XQueryPointer(aguix->getDisplay(),msg->window,&root,&child,&root_x,&root_y,&win_x,
                    &win_y,&keys_buttons);
      my=win_y;
      my+=dely;
      newelement=my/elementh;
      if(my<0) newelement--;
    }
    if((element!=newelement)||(newelement<0)||(newelement>=maxdisplayv)) {
      dest=newelement;
      if(newelement<0) {
        scrollV1(-1);
        dest=0;
        element++;
      } else if(newelement>=maxdisplayv) {
        scrollV1(1);
        dest=maxdisplayv-1;
        element--;
      }
      element=dest;
      if((element+yoffset)>=elements) {
        element=elements-yoffset-1;
      }
      tlvc=vislvcs[element];
      if(tlvc!=NULL) {
        tlvc->setActive(true);
        runSelectHandler(tlvc,element+yoffset);
      }
    }
  }
  //TODO: Muss hier eine Message verschickt werden
  // Erstmal mache ich das, dann der User bei Worker auch per mittleren Doppelklick Dateien starten kann
  AGMessage *agmsg=(AGMessage*)_allocsafe(sizeof(AGMessage));
  agmsg->listview.lv=this;
  agmsg->listview.lvc=firstselect;
  agmsg->listview.time=lasttime;
  agmsg->type=AG_LISTVIEW_ONESELECT;
  aguix->putAGMsg(agmsg);
}

bool ListView::handleKeys(Message *msg)
{
  bool returnvalue=false;
  int elements=getElements();

  if(msg!=NULL) {
    //TODO2:
    // most important features:
    // 1.up/down
    //   should react like handleMiddle
    // 2.return for simulate doubleclick
    //   problem: there is no doubleclick-feature at all at the moment
    // 3.space for selecting/unselecting
    //   just like handleSelect
    if(msg->type==KeyPress) {
      if((msg->key==XK_Up)||
	 (msg->key==XK_Down)||
	 (msg->key==XK_Home)||
	 (msg->key==XK_End)||
	 (msg->key==XK_Prior)||
	 (msg->key==XK_Next)) {
	int pos,activepos;
	ListViewContent *tlvc=getActiveLVC();
	if(tlvc!=NULL) {
	  activepos=getPos(tlvc);
	} else {
	  activepos=-1;
	}
	switch(msg->key) {
	case XK_Up:
	  if(activepos>0)
	    pos=activepos-1;
	  else
	    pos=0;
	  break;
	case XK_Down:
	  if(activepos<0)
	    pos=0;
	  else if(activepos>=(elements-1))
	    pos=elements-1;
	  else
	    pos=activepos+1;
	  break;
	case XK_Home:
	  pos=0;
	  break;
	case XK_End:
	  pos=elements-1;
	  break;
	case XK_Prior:
	  if(activepos<0)
	    pos=0;
	  else {
	    pos=activepos-maxdisplayv+1;
	    if(pos<0)
	      pos=0;
	  }
	  break;
	case XK_Next:
	  if(activepos<0)
	    pos=0;
	  else {
	    pos=activepos+maxdisplayv-1;
	    if(pos>=(elements-1))
	      pos=elements-1;
	  }
	  break;
	default:
	  pos=-1;
	  break;
	}
	if(pos>=0) {
	  // now activate entry pos
	  tlvc=get(pos);
	  if(tlvc!=NULL) {
	    tlvc->setActive(true);
	    if(((pos-yoffset)<0)||((pos-yoffset)>=maxdisplayv))
	      showActive();
	    runSelectHandler(tlvc,pos);
	    aguix->Flush();
	  }
	}
      } else if(msg->key==XK_space) {
	int activepos;
	ListViewContent *tlvc=getActiveLVC();
	if(tlvc!=NULL) {
	  activepos=getPos(tlvc);
	  bool state=tlvc->getSelect();
	  state=(state==false)?true:false;
	  tlvc->setSelect(state);
	  runSelectHandler(tlvc,activepos);
	  aguix->Flush();
	}
      } else if(msg->key==XK_Return) {
      }
    }
  }
  return returnvalue;
}
