/* -*-Mode: C++;-*-
 * $Id: dirc.cc 1.21 Tue, 15 May 2001 16:33:56 -0700 jmacd $
 *
 * Copyright (C) 1999, 2000, Joshua P. MacDonald <jmacd@CS.Berkeley.EDU>
 * and The Regents of the University of California.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 *    Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above
 *    copyright notice, this list of conditions and the following
 *    disclaimer in the documentation and/or other materials provided
 *    with the distribution.
 *
 *    Neither name of The University of California nor the names of
 *    its contributors may be used to endorse or promote products
 *    derived from this software without specific prior written
 *    permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * REGENTS OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "xdfs_cpp.h"
#include <errno.h>

//////////////////////////////////////////////////////////////////////
//		           NODEC OPERATIONS
//////////////////////////////////////////////////////////////////////

static DBFS_RMWFLAG
dbfs_link_rmw (int flags)
{
    if (flags & DBFS_LINK_RMW) {
	return DBFS_RMW;
    }

    return DBFS_NORMW;
}

int
NODEC::dir_open (DIRC &curs) const
{
    int   ret;
    DXN   dxn (txn (), ret);
    DBREF dbr;

    if (! (type () & FT_Container)) {
	ret = DBFS_NOTDIR;
	DBFS_ERROR (ret) (this);
	return ret;
    }

    if ((ret = dxn.get_index (*this, dbr))) {
	PROP_ERROR (ret) (this) ("get_directory");
	return ret;
    }

    if ((ret = dbr.cursor (txn (), curs._dbc))) {
	PROP_ERROR (ret) (this) ("db_cursor");
	return ret;
    }

    curs.set_open (*this);

    return 0;
}

int
NODEC::dir_link_lookup (const DKEY &key,
			MAJORC     &major,
			int         flags) const
{
    int  ret;
    DIRC curs;

    if ((ret = dir_open (curs))) {
	PROP_ERROR (ret) (this) ("dir_open");
	return ret;
    }

    if ((ret = curs.set_node (key, major, flags))) {
	if (ret != DBFS_NOTFOUND) {
	    PROP_ERROR (ret) (this) ("curs_set_node");
	}

	return ret;
    }

    if ((ret = curs.close ())) {
	PROP_ERROR (ret) (this) ("curs_close");
	return ret;
    }

    return 0;
}

int
NODEC::dir_link_lookup_seqno (const XSEQNO &key,
			      MAJORC       &major,
			      int          flags) const
{
    int  ret;
    DIRC curs;

    if ((ret = dir_open (curs))) {
	PROP_ERROR (ret) (this) ("dir_open");
	return ret;
    }

    if ((ret = curs.set_node_seqno (key, major, flags))) {
	PROP_ERROR (ret) (this) ("curs_set_node_seqno");
	return ret;
    }

    if ((ret = curs.close ())) {
	PROP_ERROR (ret) (this) ("curs_close");
	return ret;
    }

    return 0;
}

int
NODEC::dir_link_insert (const DKEY   &key,
			const MAJORC &major,
			int           flag) const
{
    int  ret;
    DIRC curs;

    if ((ret = dir_open (curs))) {
	PROP_ERROR (ret) (this) ("dir_open");
	return ret;
    }

    if ((ret = curs.insert_link (key, major, flag))) {
	if (ret != DBFS_EXISTS) {
	    PROP_ERROR (ret) (this) ("curs_insert_link");
	}

	return ret;
    }

    if ((ret = curs.close ())) {
	PROP_ERROR (ret) (this) ("curs_close");
	return ret;
    }

    return 0;
}

int
NODEC::dir_link_insert_seqno (const MAJORC &major,
			      XSEQNO       &seqno,
			      int           flags) const
{
    int   ret;
    DXN   dxn (txn (), ret);
    DBREF dbr;

    if (type () != FT_DirSeq) {
	ret = DBFS_NOTSEQ;
	DBFS_ERROR (ret) (this);
	return ret;
    }

    if ((ret = dxn.get_index (*this, dbr))) {
	PROP_ERROR (ret) (this) ("get_directory");
	return ret;
    }

    if ((ret = dbr.template append_T<XNUM> (txn (), seqno, major.number ()))) {
	PROP_ERROR (ret) (this) ("db_append");
	return ret;
    }

    if ((ret = major.dir_link_to (cont_id ().file_id (), RECDBT<XSEQNO> (seqno), flags))) {
	PROP_ERROR (ret) (this) ("dir_link_to");
	return ret;
    }

    return 0;
}

int
NODEC::dir_link_remove (const DKEY &key) const
{
    int  ret;
    DIRC curs;

    if ((ret = dir_open (curs))) {
	PROP_ERROR (ret) (this) ("dir_open");
	return ret;
    }

    if ((ret = curs.set_key (key, DBFS_LINK_RMW))) {
	if (ret != DBFS_NOTFOUND) {
	    PROP_ERROR (ret) (this) ("curs_set_key");
	}

	return ret;
    }

    if ((ret = curs.del_link ())) {
	PROP_ERROR (ret) ("curs_del_link");
	return ret;
    }

    if ((ret = curs.close ())) {
	PROP_ERROR (ret) (this) ("curs_close");
	return ret;
    }

    return 0;
}

int
NODEC::dir_link_remove_seqno (const XSEQNO &key) const
{
    int  ret;
    DIRC curs;

    if ((ret = dir_open (curs))) {
	PROP_ERROR (ret) (this) ("dir_open");
	return ret;
    }

    if ((ret = curs.set_seqno (key, DBFS_LINK_RMW))) {
	if (ret != DBFS_NOTFOUND) {
	    PROP_ERROR (ret) (this) ("curs_set_key");
	}

	return ret;
    }

    if ((ret = curs.del_link ())) {
	PROP_ERROR (ret) ("curs_del_link");
	return ret;
    }

    if ((ret = curs.close ())) {
	PROP_ERROR (ret) (this) ("curs_close");
	return ret;
    }

    return 0;
}

int
MAJORC::unlink () const
{
    int ret;

    CKEY cont_key;
    DKEY dkey;

    MAJOR_DESC &major = const_cast<MAJOR_DESC&> (ref ());

    cont_key.template set_rec<XSEQNO>   (major.num.cont_seq ());
    cont_key.template append_rec<XSTCK> (DBFS_INV_STACK);

    DBCREF scan;

    if ((ret = major.area->container_dbr.cursor (txn (), scan))) {
	PROP_ERROR (ret) ("scan_cont_cursor");
	return ret;
    }

    if ((ret = scan.set_range (cont_key, dkey)) && (ret != DBFS_NOTFOUND)) {
	PROP_ERROR (ret) ("unlink_set_range");
	return ret;
    }

    // Code is similar to MAJOR_DESC::minor_delete_present

    while (ret != DBFS_NOTFOUND) {

	const size_t seqno_prefix = (sizeof (XSEQNO));
	const size_t stack_prefix = (sizeof (XSEQNO) + sizeof (XSTCK));
	const size_t db_id_prefix = (sizeof (XSEQNO) + sizeof (XSTCK) + sizeof (XFID));

	if (cont_key.size () < db_id_prefix) {
	    break;
	}

	XSEQNO cont_seq;
	XSTCK  cont_stck;
	XFID   cont_db_id;

	cont_key.template get_rec_prefix<XSEQNO> (cont_seq);
	cont_key.template get_rec_at<XSTCK> (seqno_prefix, cont_stck);
	cont_key.template get_rec_at<XFID> (stack_prefix, cont_db_id);

	if (cont_seq  != major.num.cont_seq () ||
	    cont_stck != DBFS_INV_STACK) {
	    // Past the prefix
	    break;
	}

	incr_refcount (-1);

	DYNAMIC_DB dbr;

	if ((ret = dbr.open (txn ().dbfs (), cont_db_id, DB_UNKNOWN, -1, DBFS_NOFLAG))) {
	    PROP_ERROR (ret) ("unlink_dbr_open: ") () << dbr;
	    return ret;
	}

	// Okay without duplicate support
	if ((ret = dbr.delete_must_exist (txn (), dkey))) {
	    PROP_ERROR (ret) ("unlink_dbr_delete: ") () << dkey;
	    return ret;
	}

	if ((ret = scan.delete_current ())) {
	    PROP_ERROR (ret) ("final_delete_current");
	    return ret;
	}

	if ((ret = scan.move_pos (cont_key, dkey, DB_NEXT, DBFS_RMW)) && (ret != DBFS_NOTFOUND)) {
	    PROP_ERROR (ret) ("final_move_pos");
	    return ret;
	}
    }

    return 0;
}

//////////////////////////////////////////////////////////////////////
//		           LINK MANAGEMENT
//////////////////////////////////////////////////////////////////////

int
NODEC::dir_inverse_node (const MAJORC &node,
			 NODEC        &attr) const
{
    if (! is_container ()) {
	DBFS_ERROR ("Not a container");
	return DBFS_INVAL;
    }

    return node.node_mkey (REVERSE_LINK_MKEY (cont_id ().file_id ()), attr);
}

int
MAJORC::dir_link_to (const XFID    &db_id,
		     const COMDBT  &dkey_dbt,
		     int            flags) const
{
    int ret;

    incr_refcount (1);

    if (flags & DBFS_LINK_REVERSIBLE) {

	DEBUG_LINK ("Insert reverse link ") (this) (" to index ") () << db_id;

	if ((ret = locc ().put_no_overwrite (REVERSE_LINK_KEY (number (), db_id),
					     dkey_dbt))) {
	    if (ret == DBFS_LINK_INCONSISTENT) {
		PROP_ERROR (ret) (this) ("Need to support duplicate reverse links");
	    } else {
		PROP_ERROR (ret) (this) ("put_no_overwrite");
	    }
	    return ret;
	}
    }

    return 0;
}

int
MAJORC::del_link_to (const XFID   &db_id) const
{
    int ret;

    incr_refcount (-1);

    if ((ret = locc ().delete_or_notfound (REVERSE_LINK_KEY (number (), db_id)))) {

	if (ret == DBFS_NOTFOUND) {
	    // Allow reverse to be notfound, since creation is optional
	} else if (ret != 0) {
	    PROP_ERROR (ret) (this) ("delete_must_exist");
	    return ret;
	}
    }

    return 0;
}

int
NODEC::dir_link_invert (const MAJORC &major,
			DKEY         &key) const
{
    int    ret;

    if (! is_keyed ()) {
	ret = DBFS_NOTDIR;
	DBFS_ERROR (ret) (this);
	return ret;
    }

    if ((ret = major.locc ().get_or_notfound (REVERSE_LINK_KEY (major.number (), cont_id ().file_id ()),
					      key,
					      DBFS_NORMW))) { // @@ Should pass this flag
	if (ret != DBFS_NOTFOUND) {
	    PROP_ERROR (ret) (this) ("get_must_exist");
	}
	return ret;
    }

    return 0;
}

int
NODEC::dir_link_invert_seqno (const MAJORC &major,
			      XSEQNO       &seqno) const
{
    int ret;

    if (! is_sequence ()) {
	ret = DBFS_NOTSEQ;
	DBFS_ERROR (ret) (this);
	return ret;
    }

    RECDBT<XSEQNO> seqno_dbt (seqno);

    if ((ret = major.locc ().get_or_notfound (REVERSE_LINK_KEY (major.number (), cont_id ().file_id ()),
					      seqno_dbt,
					      DBFS_NORMW))) { // @@ Should pass this flag
	if (ret != DBFS_NOTFOUND) {
	    PROP_ERROR (ret) (this) ("get_must_exist");
	}
	return ret;
    }

    return 0;
}

//////////////////////////////////////////////////////////////////////
//		          CURSOR OPERATIONS
//////////////////////////////////////////////////////////////////////

DIRC::DIRC ()
    : _dnode     (NULL),
      _txn       (NULL),
      _close_ret (0),
      _get_ret   (DBFS_NOPOSITION)
{
}

int
DIRC::close ()
{
    int ret;

    if ((ret = _dbc.close ())) {
	PROP_ERROR (ret) ("dbc_close");
    }

    return _close_ret ? _close_ret : ret;
}

void
DIRC::set_open (const NODEC &node)
{
    _dnode = & node;
    _txn   = & node.txn ();
}

int
DIRC::last ()
{
    return set_curs (DB_LAST, DBFS_NORMW);
}

int
DIRC::first ()
{
    return set_curs (DB_FIRST, DBFS_NORMW);
}

bool
DIRC::next ()
{
    int ret;

    if ((ret = set_curs (DB_NEXT, DBFS_NORMW)) && (_close_ret == 0)) {

	if (ret != DBFS_NOTFOUND) {
	    PROP_ERROR (ret) ("set_curs");
	    _close_ret = ret;
	}
    }

    return ret == 0;
}

bool
DIRC::prev ()
{
    int ret;

    if ((ret = set_curs (DB_PREV, DBFS_NORMW)) && (_close_ret == 0)) {

	if (ret != DBFS_NOTFOUND) {
	    PROP_ERROR (ret) ("set_curs");
	    _close_ret = ret;
	}
    }

    return ret == 0;
}

int
DIRC::set_node (const DKEY   &key,
		MAJORC       &major,
		int           flags)
{
    int ret;

    if ((ret = set_key (key, flags))) {
	if (ret != DBFS_NOTFOUND) {
	    PROP_ERROR (ret) ("set_key");
	}

	return ret;
    }

    if ((ret = get_node (major))) {
	PROP_ERROR (ret) ("get_node");
	return ret;
    }

    return 0;
}

int
DIRC::set_node_seqno (const XSEQNO &key,
		      MAJORC       &major,
		      int           flags)
{
    int ret;

    if ((ret = set_seqno (key, flags))) {
	if (ret != DBFS_NOTFOUND) {
	    PROP_ERROR (ret) ("set_seqno");
	}

	return ret;
    }

    if ((ret = get_node (major))) {
	PROP_ERROR (ret) ("get_node");
	return ret;
    }

    return 0;
}

//////////////////////////////////////////////////////////////////////
//		          CURSOR GET OPERATIONS
//////////////////////////////////////////////////////////////////////

int
DIRC::get_key (DKEY &key)
{
    if (_get_ret) {
	DBFS_ERROR (_get_ret) ("cursor position");
	return _get_ret;
    }

    key = _get_key;

    return 0;
}

int
DIRC::get_seqno (XSEQNO &key)
{
    if (_get_ret) {
	DBFS_ERROR (_get_ret) ("cursor position");
	return _get_ret;
    }

    _get_key.template get_rec<XSEQNO> (key);

    return 0;
}

int
DIRC::get_node (MAJORC &major)
{
    int ret;
    DXN dxn (*_txn, ret);

    if (_get_ret) {
	ret = _get_ret;
	DBFS_ERROR (ret) ("cursor position");
	return ret;
    }

    if ((ret = dxn.get_major (_get_num, major))) {
	PROP_ERROR (ret) ("get_major");
	return ret;
    }

    return 0;
}

int
DIRC::get_number (XNUM &num)
{
    int ret;

    if (_get_ret) {
	ret = _get_ret;
	DBFS_ERROR (ret) ("cursor position");
	return ret;
    }

    num = _get_num;

    return 0;
}

//////////////////////////////////////////////////////////////////////
//		          CURSOR SET OPERATIONS
//////////////////////////////////////////////////////////////////////

int
DIRC::set_key (const DKEY &key,
	       int         flags)
{
    _set_key = key;

    return set_curs (DB_SET, dbfs_link_rmw (flags));
}

int
DIRC::set_seqno (const XSEQNO &key,
		 int           flags)
{
    _set_key.template set_rec<guint32> (key.key ());

    return set_curs (DB_SET, dbfs_link_rmw (flags));
}

int
DIRC::set_curs (int          dbflag,
		DBFS_RMWFLAG rmwflag)
{
    RECDBT<XNUM> num_dbt (_get_num);

    if (dbflag == DB_SET) {

	if ((_get_ret = _dbc.get_or_notfound (_set_key, num_dbt, rmwflag)) && (_get_ret != DBFS_NOTFOUND)) {
	    PROP_ERROR (_get_ret) ("get_or_notfound");
	    return _get_ret;
	}

	_get_key = _set_key;

    } else {

	if ((_get_ret = _dbc.move_pos (_get_key, num_dbt, dbflag, rmwflag))) {
	    if (_get_ret != DBFS_NOTFOUND) {
		PROP_ERROR (_get_ret) ("move_pos");
	    }

	    return _get_ret;
	}
    }

    return _get_ret;
}

int
DIRC::del_link ()
{
    int    ret;
    MAJORC major;

    if ((ret = get_node (major))) {
	PROP_ERROR (ret) ("get_node");
	return ret;
    }

    if ((ret = _dbc.delete_current ())) {
	PROP_ERROR (ret) ("delete_current");
	return ret;
    }

    if ((ret = major.del_link_to (_dnode->cont_id ().file_id ()))) {
	PROP_ERROR (ret) (major) ("del_link_to");
	return ret;
    }

    _get_ret = DBFS_NOTFOUND;

    return 0;
}

int
DIRC::repl_link (const MAJORC &major, int flags)
{
    int ret;

    if (_get_ret != 0) {
	DBFS_ERROR (_get_ret) ("repl_link requires current value");
	return _get_ret;
    }

    MAJORC old;

    if ((ret = get_node (old))) {
	PROP_ERROR (ret) ("get_node");
	return ret;
    }

    if ((ret = old.del_link_to (_dnode->cont_id ().file_id ()))) {
	PROP_ERROR (ret) (old) ("del_link_to");
	return ret;
    }

    if ((ret = _dbc.template put_current_T<XNUM> (major.number ()))) {
	PROP_ERROR (ret) ("put_current");
	return ret;
    }

    if ((ret = major.dir_link_to (_dnode->cont_id ().file_id (), _get_key, flags))) {
	PROP_ERROR (ret) (major) ("dir_link_to");
	return ret;
    }

    _get_ret = 0;
    _get_num = major.number ();

    return 0;
}

int
DIRC::insert_link (const DKEY   &key,
		   const MAJORC &major,
		   int           flags)
{
    int ret;

    if ((ret = dbfs_check_ow_flags (flags))) {
	PROP_ERROR (ret) ("dbfs_check_ow_flags");
	return ret;
    }

    if ((ret = set_key (key, DBFS_LINK_RMW)) && (ret != DBFS_NOTFOUND)) {
	PROP_ERROR (ret) ("set_key");
	return ret;
    }

    if ((ret == 0) && ! (flags & DBFS_OVERWRITE)) {
	return DBFS_EXISTS;
    }

    if (ret == DBFS_NOTFOUND) {
	// Can't call repl_link in this case, cursor is placed at the
	// last successful lookup's key

	RECDBT<XNUM> num_dbt (major.number ());

	if ((ret = _dbc.put_overwrite (_set_key, num_dbt))) {
	    return ret;
	}

	if ((ret = major.dir_link_to (_dnode->cont_id ().file_id (), _set_key, flags))) {
	    PROP_ERROR (ret) (major) ("dir_link_to");
	    return ret;
	}

	_get_num = major.number ();;
	_get_ret = 0;

    } else {

	if ((ret = repl_link (major, flags))) {
	    PROP_ERROR (ret) ("repl_link");
	    return ret;
	}
    }

    return 0;
}
