/* -*-Mode: C++;-*-
 * $Id: testdbfs.cc 1.10 Tue, 15 May 2001 16:33:56 -0700 jmacd $
 *
 * Copyright (C) 1999, 2000, Joshua P. MacDonald <jmacd@CS.Berkeley.EDU>
 * and The Regents of the University of California.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 *    Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above
 *    copyright notice, this list of conditions and the following
 *    disclaimer in the documentation and/or other materials provided
 *    with the distribution.
 *
 *    Neither name of The University of California nor the names of
 *    its contributors may be used to endorse or promote products
 *    derived from this software without specific prior written
 *    permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * REGENTS OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "xdfs_cpp.h"
#include "xdfs_comp.h"

typedef int (*TESTFUNC) (DBFS &dbfs);

static int
test_ckey (DBFS &dbfs)
{
    DKEY dk1 ("foo");
    DKEY dk2;

    dk2 = dk1;

    g_assert (dk1 == dk2);

    MKEY mk1 ("foo");
    MKEY mk2;

    mk2 = mk1;

    g_assert (mk1 == mk2);

    return 0;
}

static int
test_basic_directory (DBFS &dbfs)
{
    SAREA  area;
    MAJORC root;
    TXN    txn;
    int    ret;

    if ((ret = txn.begin_root (dbfs, DBFS_TXN_SYNC, root))) {
	TEST_ERROR (ret) ("txn_begin");
	return ret;
    }

    if ((ret = txn.create_area (area))) {
	TEST_ERROR (ret) ("txn_create_area");
	return ret;
    }

    // Allocate directory
    MAJORC test_dir;

    if ((ret = area.allocate (test_dir, 0))) {
	TEST_ERROR (ret) ("area_allocate");
	return ret;
    }

    if ((ret = test_dir.mk_directory (XTYPE_DIRBTREE, DBFS_NOOVERWRITE))) {
	TEST_ERROR (ret) ("mk_directory");
	return ret;
    }

    // Check inverse lookup
    DKEY orig_key ("testLinkName");
    DKEY inv_key;

    if ((ret = test_dir.dir_link_insert (orig_key, test_dir, DBFS_NOOVERWRITE | DBFS_LINK_REVERSIBLE))) {
	TEST_ERROR (ret) ("dir_link_insert");
	return ret;
    }

    if ((ret = test_dir.dir_link_invert (test_dir, inv_key))) {
	TEST_ERROR (ret) ("dir_link_invert");
	return ret;
    }

    if (inv_key != orig_key) {
	TEST_ERROR ("inverse key doesn't match");
	return -1;
    }

    // Check link nooverwrite
    if ((ret = test_dir.dir_link_insert (orig_key, test_dir, DBFS_NOOVERWRITE)) != DBFS_EXISTS) {
	TEST_ERROR (ret) ("link: no_overwrite");
	return ret;
    }

    // Check node nooverwrite
    if ((ret = test_dir.mk_directory (XTYPE_DIRHASH, DBFS_NOOVERWRITE)) != DBFS_EXISTS) {
	TEST_ERROR (ret) ("node: no_overwrite");
	return ret;
    }

    if ((ret = txn.commit ())) {
	TEST_ERROR (ret) ("txn_commit");
	return ret;
    }

    return 0;
}

#define BUFFER_SZ (1<<23)

static guint8 BUFFER[BUFFER_SZ];

class XTimer
{
public:

    XTimer ()
	: _timer (g_timer_new ())
    {
	g_timer_start (_timer);
    }

    ~XTimer ()
    {
	g_timer_destroy (_timer);
    }

    double elapsed ()
    {
	g_timer_stop (_timer);

	return g_timer_elapsed (_timer, NULL);
    }

private:

    GTimer *_timer;
};

static int
write_timing (DBFS &dbfs, int size, int nreps, int nreads)
{
    SAREA  area;
    MAJORC root;
    TXN    txn;
    int    ret;

    g_assert (size <= BUFFER_SZ);

    if ((ret = txn.begin_root (dbfs, DBFS_TXN_SYNC, root))) {
	TEST_ERROR (ret) ("txn_begin");
	return ret;
    }

    if ((ret = txn.create_area (area))) {
	TEST_ERROR (ret) ("txn_create_area");
	return ret;
    }

    XTimer timer;

    for (int i = 0; i < nreps; i += 1) {

	// Allocate directory
	MAJORC test_seg;

	if ((ret = area.allocate (test_seg, 0))) {
	    TEST_ERROR (ret) ("area_allocate");
	    return ret;
	}

	FileHandle *fh;

	if ((ret = test_seg.repl_segment (& fh, DBFS_OVERWRITE))) {
	    TEST_ERROR (ret) ("repl_segment");
	    return ret;
	}

	if (! handle_write (fh, BUFFER, size)) {
	    TEST_ERROR ("handle_write");
	    return -1;
	}

	if (! handle_close (fh)) {
	    TEST_ERROR ("handle_close");
	    return -1;
	}

	handle_free (fh);

	for (int j = 0; j < nreads; j += 1) {
	    if ((ret = test_seg.read_segment (& fh, DBFS_NOFLAG))) {
		TEST_ERROR (ret) ("read_segment");
		return ret;
	    }

	    g_assert (handle_length (fh) == size);

	    if (handle_read (fh, BUFFER, size) != size) {
		TEST_ERROR ("handle_read");
		return -1;
	    }

	    if (! handle_close (fh)) {
		TEST_ERROR ("handle_close");
		return -1;
	    }

	    handle_free (fh);
	}
    }

    if ((ret = txn.commit ())) {
	TEST_ERROR (ret) ("txn_commit");
	return ret;
    }

    INFO_TEST ("write_timing: write/%d read %5u bytes %f secs",
	       nreads,
	       size,
	       timer.elapsed () / (double) nreps);

    return 0;
}

static int
test_transient_readwrite (DBFS &dbfs)
{
    int ret;
    int NREPS  = 128;
    int NREADS = 2;

    // Note: Ignore TXN because of current descriptor limits

    if ((ret = write_timing (dbfs, 0, NREPS, NREADS))) { return ret; }

    if ((ret = write_timing (dbfs, 1, NREPS, NREADS))) { return ret; }

    if ((ret = write_timing (dbfs, DBFS_FS_SHORT_THRESH, NREPS, NREADS))) { return ret; }

    if ((ret = write_timing (dbfs, DBFS_FS_SHORT_THRESH + 1, NREPS, NREADS))) { return ret; }

    if ((ret = write_timing (dbfs, (1<<15), NREPS, NREADS))) { return ret; }

    if ((ret = write_timing (dbfs, (1<<20), NREPS, NREADS))) { return ret; }

    //if ((ret = write_timing (dbfs, (1<<23), NREPS, NREADS))) { return ret; }

    return 0;
}

static void
check (XdfsInvert &inv, int offset, int result_offset, int result_cstart, int result_len)
{
    XdfsInvert::iterator iter = inv.invsearch (offset);

    if (result_offset < 0) {
	g_assert (iter.end ());
    } else {
	g_assert (iter.offset () == (uint) result_offset);
	g_assert (iter.start  () == (uint) result_cstart);
	g_assert (iter.length () == (uint) result_len);
    }
}

static int
test_xdfs_invert (DBFS &dbfs)
{
    XdfsInvert invert;

    invert.insert (4, 2, 4);
    invert.insert (3, 2, 3);
    invert.insert (5, 2, 5);

    check (invert, 0, 3, 3, 1);
    check (invert, 1, 3, 3, 1);
    check (invert, 2, 3, 3, 1);
    check (invert, 3, 3, 3, 1);
    check (invert, 4, 4, 4, 2);
    check (invert, 5, 4, 4, 2);
    check (invert, 6, 6, 6, 1);
    check (invert, 7, -1, -1, -1);

    return 0;
}

static int
scribble (TXN &txn)
{
    MAJORC root = txn.root ();
    MAJORC one, two, three;
    XLNK   rootlnks;
    int    ret;

    rootlnks = root.nlinks ();

    if ((ret = root.sarea ().allocate (one, 0)) ||
	(ret = root.sarea ().allocate (two, 0)) ||
	(ret = root.sarea ().allocate (three, 0))) {
	TEST_ERROR (ret) ("area_allocate");
	return ret;
    }

    g_assert (one.nlinks   () == DBFS_ZERO_REFS &&
	      two.nlinks   () == DBFS_ZERO_REFS &&
	      three.nlinks () == DBFS_ZERO_REFS);

    if ((ret = one.mk_reflink (root, DBFS_NOOVERWRITE))) {
	TEST_ERROR (ret) ("one_mk_reflink");
	return ret;
    }

    if ((ret = two.mk_reflink (one, DBFS_NOOVERWRITE))) {
	TEST_ERROR (ret) ("two_mk_reflink");
	return ret;
    }

    if ((ret = three.mk_reflink (two, DBFS_NOOVERWRITE))) {
	TEST_ERROR (ret) ("three_mk_reflink");
	return ret;
    }

    g_assert (one.nlinks () == DBFS_ONE_REF &&
	      two.nlinks () == DBFS_ONE_REF &&
	      three.nlinks () == DBFS_ZERO_REFS);

    if ((ret = root.dir_link_insert ("scribble", three, DBFS_NOOVERWRITE))) {
	TEST_ERROR (ret) ("dir_link_insert");
	return ret;
    }

    g_assert (one.nlinks   () == DBFS_ONE_REF &&
	      two.nlinks   () == DBFS_ONE_REF &&
	      three.nlinks () == DBFS_ONE_REF &&
	      root.nlinks  () == (rootlnks + 1));

    return 0;
}

static int
unscribble (TXN &txn)
{
    int    ret;

    if ((ret = txn.root ().dir_link_remove ("scribble"))) {
	TEST_ERROR (ret) ("dir_link_remove");
	return ret;
    }

    return 0;
}

static int
test_precommit (DBFS &dbfs)
{
    int ret;
    TXN txn1, txn2, txn3;

    // Single txn
    if ((ret = txn1.begin (dbfs, DBFS_TXN_SYNC))) {
	TEST_ERROR (ret) ("txn_begin");
	return ret;
    }

    if ((ret = scribble (txn1))) {
	TEST_ERROR (ret) ("scribble");
	return ret;
    }

    if ((ret = unscribble (txn1))) {
	TEST_ERROR (ret) ("unscribble");
	return ret;
    }

    if ((ret = txn1.commit ())) {
	TEST_ERROR (ret) ("txn_commit");
	return ret;
    }

    // Separate txn
    if ((ret = txn2.begin (dbfs, DBFS_TXN_SYNC))) {
	TEST_ERROR (ret) ("txn_begin");
	return ret;
    }

    if ((ret = scribble (txn2))) {
	TEST_ERROR (ret) ("scribble");
	return ret;
    }

    if ((ret = txn2.commit ())) {
	TEST_ERROR (ret) ("txn_commit");
	return ret;
    }

    if ((ret = txn3.begin (dbfs, DBFS_TXN_SYNC))) {
	TEST_ERROR (ret) ("txn_begin");
	return ret;
    }

    if ((ret = unscribble (txn3))) {
	TEST_ERROR (ret) ("unscribble");
	return ret;
    }

    if ((ret = txn3.commit ())) {
	TEST_ERROR (ret) ("txn_commit");
	return ret;
    }

    return 0;
}

static int
test_invlink (DBFS& dbfs)
{
    int ret;
    TXN txn;
    MAJORC root;
    SAREA area;

    if ((ret = txn.begin_root (dbfs, DBFS_TXN_SYNC, root))) {
	TEST_ERROR (ret) ("txn_begin");
	return ret;
    }


    if ((ret = txn.create_area (area))) {
	TEST_ERROR (ret) ("create_area");
	return ret;
    }

    MAJORC targ, d1, d2, d3;

    if ((ret = area.allocate (targ, 0)) ||
	(ret = area.allocate (d1, 0)) ||
	(ret = area.allocate (d2, 0)) ||
	(ret = area.allocate (d3, 0))) {
	TEST_ERROR (ret) ("area_allocate");
	return ret;
    }

    if ((ret = targ.mk_reflink (root, DBFS_NOOVERWRITE)) ||
	(ret = d1.mk_directory (XTYPE_DIRHASH, DBFS_NOOVERWRITE)) ||
	(ret = d2.mk_directory (XTYPE_DIRBTREE, DBFS_NOOVERWRITE)) ||
	(ret = d3.mk_directory (XTYPE_DIRSEQ, DBFS_NOOVERWRITE))) {
	TEST_ERROR (ret) ("mk");
	return ret;
    }

    XSEQNO seqno;

    if ((ret = d1.dir_link_insert ("d1link1", targ, DBFS_NOOVERWRITE|DBFS_LINK_REVERSIBLE)) ||
	(ret = d2.dir_link_insert ("d2link1", targ, DBFS_NOOVERWRITE|DBFS_LINK_REVERSIBLE)) ||
	(ret = d3.dir_link_insert_seqno (targ, seqno, DBFS_LINK_REVERSIBLE))) {
	TEST_ERROR (ret) ("insert");
	return ret;
    }

    if ((ret = targ.unlink ())) {
	TEST_ERROR (ret) ("unlink");
	return ret;
    }

    if ((ret = targ.del_major ())) {
	TEST_ERROR (ret) ("del_major");
	return ret;
    }

    if ((ret = txn.commit ())) {
	TEST_ERROR (ret) ("txn_commit");
	return ret;
    }

    return 0;
}

static int
test_dbremove (DBFS& dbfs)
{
    int ret;
    TXN txn;
    MAJORC root;

    if ((ret = txn.begin_root (dbfs, DBFS_TXN_SYNC, root))) {
	TEST_ERROR (ret) ("txn_begin");
	return ret;
    }

    guint  NDBS = 128;
    MAJORC dbs[NDBS];

    XLNK root_lnks_orig = root.nlinks ();

    for (guint i = 0; i < NDBS; i += 1) {

	if ((ret = root.sarea ().allocate (dbs[i], 0))) {
	    TEST_ERROR (ret) ("area_allocate");
	    return ret;
	}

	if ((ret = dbs[i].mk_directory (XTYPE_DIRHASH, DBFS_NOOVERWRITE))) {
	    TEST_ERROR (ret) ("mk_directory");
	    return ret;
	}

	if ((ret = dbs[i].dir_link_insert ("testlink", root, DBFS_NOOVERWRITE))) {
	    TEST_ERROR (ret) ("dir_link");
	    return ret;
	}
    }

    g_assert ((root_lnks_orig + NDBS) == root.nlinks ());

    for (guint i = 0; i < NDBS; i += 1) {

	if ((ret = dbs[i].del_minor ())) {
	    TEST_ERROR (ret) ("dir_link");
	    return ret;
	}
    }

    g_assert (root_lnks_orig == root.nlinks ());

    if ((ret = txn.commit ())) {
	TEST_ERROR (ret) ("txn_commit");
	return ret;
    }

    return 0;
}

TESTFUNC tests[] =
{
    test_dbremove,
    test_ckey,
    test_basic_directory,
    test_xdfs_invert,
    test_precommit,
    test_invlink,
    test_transient_readwrite,
};

int
main (int argc, char **argv)
{
    DBFS dbfs (argc, argv);
    int  ret;

    if (argc != 3) {
	TEST_ERROR ("usage: %s DBFS_DIR DBFS_LOG", argv[0]);
	return 2;
    }

    if ((ret = dbfs.open (argv[1], argv[2], DBFS_CREATE | DBFS_CLEAR))) {
	TEST_ERROR (ret) ("dbfs_open");
	exit (1);
    }

    for (size_t i = 0; i < ARRAY_SIZE (tests); i += 1) {

	if ((ret = (* tests[i]) (dbfs))) {
	    TEST_ERROR (ret) ("test %u failed", i);
	    exit (1);
	}
    }

    return 0;
}
