/* -*-Mode: C++;-*-
 * $Id: txn.cc 1.16 Tue, 15 May 2001 16:33:56 -0700 jmacd $
 *
 * Copyright (C) 1999, 2000, Joshua P. MacDonald <jmacd@CS.Berkeley.EDU>
 * and The Regents of the University of California.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 *    Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above
 *    copyright notice, this list of conditions and the following
 *    disclaimer in the documentation and/or other materials provided
 *    with the distribution.
 *
 *    Neither name of The University of California nor the names of
 *    its contributors may be used to endorse or promote products
 *    derived from this software without specific prior written
 *    permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * REGENTS OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "xdfs_cpp.h"

static pthread_mutex_t TXNCOUNT_MUTEX = PTHREAD_MUTEX_INITIALIZER;
static int             TXNCOUNT;

int  STXN::pre_commit  () { return 0; }
int  STXN::post_commit () { return 0; }
void STXN::pre_abort   () { }
void STXN::post_abort  () { }

STXN::STXN ()
    : _dbfs  (NULL),
      _dbtxn (NULL)
{
}

TXN::TXN ()
    : _majors (DBFS_MAJOR_TABLE_SIZE),
      _minors (DBFS_MINOR_TABLE_SIZE),
      _areas  (DBFS_SAREA_TABLE_SIZE)
{
}

// Override the default so that post_commit has the right vtable.
TXN::~TXN ()
{
    abort ();
}

STXN::~STXN ()
{
    abort ();
}

int TXN::find_major (MAJORC &major, const XNUM &num)
{
    int ret;
    DXN dxn (*this, ret);

    if ((ret = dxn.get_major (num, major))) {
	PROP_ERROR (ret) ("get_major");
	return ret;
    }

    return 0;
}

int
TXN::create_area (SAREA &area)
{
    int ret;
    DXN dxn (*this, ret);

    if ((ret = dxn.create_area (area))) {
	PROP_ERROR (ret) ("create_area");
	return ret;
    }

    return 0;
}

int
TXN::begin (DBFS &dbfs, int flags)
{
    int ret;

    if ((ret = STXN::begin (dbfs, flags))) {
	return ret;
    }

    if ((flags & DBFS_NOROOT_SPECIAL) == 0 &&
	(ret = find_major (_root, DBFS_ROOT_DIRECTORY))) {
	PROP_ERROR (ret) ("find_root");
	return ret;
    }

    return 0;
}

int
TXN::begin_root (DBFS &dbfs, int flags, MAJORC &root)
{
    int ret;

    if ((ret = begin (dbfs, flags))) {
	return ret;
    }

    root = _root;

    return 0;
}

void
TXN::common_abort ()
{
    int ret;
    SAREA_DESC *adesc;

    for (adesc = _areas.inuse ();
	 adesc;
	 adesc = adesc->next ()) {

	if ((ret = adesc->close ())) {
	    PROP_ERROR (ret) ("pre_abort_area_close");
	}
    }
}

void
TXN::pre_abort ()
{
    DBFS_ERROR ("abort ") (*this) (" (%d active)", TXNCOUNT);

    common_abort ();
}

int
TXN::pre_commit ()
{
    int ret;
    MAJOR_DESC *mdesc;

    DEBUG_PRECOMMIT ("begin pre_commit");

    // This allows major nodes dirtied during finalization to
    // be re-finalized.
    while (_major_order.slp_remove_min (NULL, & mdesc)) {

	if ((ret = mdesc->final (*this))) {
	    return ret;
	}
    }

    common_abort ();

    DEBUG_PRECOMMIT ("end pre_commit");

    return 0;
}

int
STXN::begin (BASIC_DBENV &dbfs, int flags)
{
    int ret;
    int dbflags = 0;

    _dbfs = & dbfs;

    g_assert (_dbtxn == NULL);

    if ((flags & DBFS_TXN_NOSYNC_INTERNAL) || (flags & DBFS_TXN_ASYNC)) {
	dbflags |= DB_TXN_NOSYNC;
    } else {
	flags |= DBFS_TXN_SYNC;
    }

    if (flags & DBFS_TXN_NOWAIT) {
	dbflags |= DB_TXN_NOWAIT;
    }

    _flags = flags;

    if ((ret = dbfs_async_txn_chkflags (flags))) {
	PROP_ERROR (ret) ("dbfs_async_txn_chkflags");
	return ret;
    }

    if ((ret = _dbfs->_env->txn_begin (NULL, & _dbtxn, dbflags))) {

	DB_ERROR (ret) ("txn_begin");

    } else {

	MONITOR mon (& TXNCOUNT_MUTEX);
	int     count = ++TXNCOUNT;
	mon.release ();

	DEBUG_TXN ("begin ") (*this) (" (%d active)", count);
    }

    return ret;
}

int
STXN::commit ()
{
    DbTxn *t = _dbtxn;
    int    ret = 0;

    if (t) {

	if ((ret = pre_commit ())) {
	    PROP_ERROR (ret) ("pre_commit");
	    return ret;
	}

	MONITOR mon (& TXNCOUNT_MUTEX);
	int     count = --TXNCOUNT ;
	mon.release ();

	DEBUG_TXN ("commit ") (*this, t) (" (%d active)", count);

	_dbtxn = NULL;

	if ((ret = t->commit (0))) {
	    DB_ERROR (ret) ("txn_commit");
	}

	if ((ret == 0) &&
	    (ret = post_commit ())) {
	    PROP_ERROR (ret) ("post_commit");
	    return ret;
	}
    }

    return ret;
}

void
STXN::abort ()
{
    DbTxn *t = _dbtxn;
    int    ret;

    _dbtxn = NULL;

    if (t) {

	pre_abort ();

	MONITOR mon (& TXNCOUNT_MUTEX);
	/*int     count =*/ --TXNCOUNT;
	mon.release ();

	if ((ret = t->abort ())) {
	    DB_ERROR (ret) ("txn_abort");
	}

	post_abort ();
    }
}
